DROP PROCEDURE IF EXISTS `log_sms`;
DROP PROCEDURE IF EXISTS `update_cache`;

DROP TABLE IF EXISTS `soft_quota_report_sent`;
DROP TABLE IF EXISTS `sms_log`;
DROP TABLE IF EXISTS `user_month`;
DROP TABLE IF EXISTS `user`;
DROP TABLE IF EXISTS `customer`;

CREATE TABLE `customer` (
    id int unsigned NOT NULL AUTO_INCREMENT PRIMARY KEY,
    name varchar(255) NOT NULL UNIQUE,
    `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE `user` (
    id int unsigned NOT NULL AUTO_INCREMENT PRIMARY KEY,
    name varchar(255) NOT NULL UNIQUE,
    customer int unsigned NOT NULL REFERENCES `customer` (`id`),
    `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE `user_month` (
    id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
    `user` INT UNSIGNED NOT NULL REFERENCES `user` (id),
    month DATE NOT NULL,
    sum_cached INT UNSIGNED,
    UNIQUE(`user`, month)
);

CREATE TABLE `sms_log` (
    id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
    user_month INT UNSIGNED NOT NULL REFERENCES user_month (id),
    `count` INT UNSIGNED NOT NULL,
    `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE `soft_quota_report_sent` (
    id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
    month DATE NOT NULL,
    customer int unsigned NOT NULL REFERENCES `customer` (`id`),
    sent_to VARCHAR(512),
    `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE OR REPLACE VIEW sms_per_month_and_user (
        username,
        customer,
        month,
        `count`
    ) AS 
    SELECT
        `user`.name,
        customer.name,
        user_month.month,
        COALESCE(
            user_month.sum_cached,
            (SELECT SUM(sms_log.`count`) FROM sms_log WHERE sms_log.user_month = user_month.id)
        )
    FROM `user`
    JOIN user_month ON user_month.`user` = `user`.id
    JOIN customer   ON customer.id       = `user`.customer;

CREATE OR REPLACE VIEW sms_per_month ( customer, month, `count`)
    AS SELECT customer, month, SUM(`count`)
    FROM sms_per_month_and_user
    GROUP BY month, customer;


DELIMITER //
CREATE PROCEDURE `log_sms` (IN p_customer VARCHAR(255), IN p_user VARCHAR(255), IN p_count INT UNSIGNED)
LANGUAGE SQL
NOT DETERMINISTIC
SQL SECURITY INVOKER
COMMENT 'Log a number of SMS being sent right now'
BEGIN
    DECLARE user_id INT UNSIGNED;
    DECLARE customer_id INT UNSIGNED;
    DECLARE this_month DATE;
    DECLARE this_month_id INT UNSIGNED;

    START TRANSACTION;
    SELECT id INTO customer_id FROM customer WHERE name = p_customer;
    if (customer_id IS NULL) THEN BEGIN
        INSERT INTO customer (name) VALUES (p_customer);
        SELECT id INTO customer_id FROM customer WHERE name = p_customer;
    END; END IF;

    SELECT id INTO user_id FROM `user` WHERE name = p_user;
    IF (user_id IS NULL) THEN BEGIN
        INSERT INTO `user` (name, customer) VALUES (p_user, customer_id);
        SELECT id INTO user_id FROM `user` WHERE name = p_user;
    END; END IF;

    SELECT DATE(DATE_FORMAT(DATE(NOW()), '%Y-%m-00')) INTO this_month;

    SELECT id INTO this_month_id FROM user_month where `user` = user_id AND month = this_month;
    IF (this_month_id IS NULL) THEN BEGIN
        INSERT INTO user_month (`user`, month) VALUES (user_id, this_month);
        SELECT id INTO this_month_id FROM user_month where `user` = user_id AND month = this_month;
    END; END IF;

    INSERT INTO sms_log (user_month, count) VALUES (this_month_id, p_count);

    COMMIT;
END
//

DELIMITER //
CREATE PROCEDURE `update_cache`()
LANGUAGE SQL
NOT DETERMINISTIC
SQL SECURITY INVOKER
COMMENT 'update user_month.sum_cached for months older than the current one'
BEGIN
    DECLARE this_month DATE;

    SELECT DATE(DATE_FORMAT(DATE(NOW()), '%Y-%m-00')) INTO this_month;

    UPDATE user_month
       SET user_month.sum_cached = COALESCE(
               (SELECT SUM(sms_log.`count`) FROM sms_log WHERE sms_log.user_month = user_month.id),
               0
           )
     WHERE user_month.sum_cached IS NULL
       AND user_month.month < this_month;
END//

DELIMITER ;
