package noris::SMS;

use 5.014;
use utf8;
use warnings;
use Config::INI::Reader;

=encoding utf8

=head1 NAME

noris::SMS - versende SMs über HTTP(S)

=head1 SYNOPSE

    use noris::SMS;

    my $sms = new noris::SMS orig => 'fany';

    my $response   = $sms->send( data => 'Das ist der Rand von Ostermundigen.' );
    my $count_sent = $response->sent;

=head1 BESCHREIBUNG

Dieses Modul soll einen möglichst einfachen Versand von SMs (Short Messages)
über Anbieter, die eine HTTP(S)-Schnittstelle anbieten, bei der die SMs über
POST-Requests eingeliefert werden können, ermöglichen.

=cut

use Carp qw(croak);
use Encode qw(encode);
use HTTP::Request::Common qw(POST);
use LWP::UserAgent ();
use noris::Encode::OpenITGSM0338;
use Encode::GSM0338;
use Digest::MD5 qw(md5_hex);
use POSIX qw(ceil);

=head1 KONSTRUKTOR

C<-E<gt>new> erwartet (optional) eine Liste von Name-/Wert-Paaren, insbesondere
folgenden:

=over 4

=item debug

Wird die debug Variable auf L<1> angegeben, gibt L</->send> zu
Debugging-Zwecken Meldungen (auf das aktuelle L<select()ierte|perlfunc/select>
FileHandle) aus.

=item test

Damit keine SMs versendet werden, muss  der Parameter L<test> auf L<1> gesetzt
werden.

=item user

Benutzername beim SMS-Anbieter (vgl. L</params>)

=item max_sms

Wieviele SMs sollen pro Aufruf von L<-E<gt>send> maximal versandt werden?
(Default: 10)

=item re_ok

regulärer Ausdruck, auf den die vom POST-Request zurückgelieferte Seite (genau
dann) matcht, wenn der Versand erfolgreich war (Default: C</^+/>)

=item params

Welche Parameter sollen im POST-Request an den SMS-Anbieter übergeben werden?
(Default:
C<user>, C<pass>, C<project>, C<orig>, C<dest>, C<data>, C<maxparts>, C<enc>,
C<registered_delivery>.)
Mehrere Parameternamen können als Array-Referenz übergeben werden.

=item pass

Passwort zur Authentifikation beim SMS-Anbieter (vgl. L</params>)

=item orig

für die SMs zu verwendende Absenderkennung
(Ziffernfolge oder maximal elf alphanumerische Zeichen; Default: "noris.net")

=item dest

an welche Zielrnufnummer(n) soll(en) die SM(s) versandt werden?
Werden mehrere Zielrufnummern (als Array-Referenz) angegeben, so erfolgt der
Versand an alle Zielrnufnummern.

=item data

zu versendende(r) Text(e).
Mehrere Texte können als Array-Referenz übergeben werden; sie werden in jedem
Fall als getrennte SMs versandt.

=item enc

Das Encoding das beim übertragen der Daten an den Anbieter verwendet werden
soll.
(Default: gsm)

=item maxparts

Obergrenze der SMs, die das Gateway für größere Nachrichten ( E<gt> 160
 Zeichen ) erzeugen soll.
(Default: 10)

=item registered_delivery

Auslieferungsbestätigung anfordern?

Default: 1 = ja

=item timeout

Timeout für die HTTP(S)-Requests, der an L<LWP::UserAgent> übergeben wird

=item urls

URL(s), an die die POST-Requests abgesetzt werden sollen.
Bei der Angabe mehrerer URLs werden diese nacheinander durchprobiert, bis eine
positive Rückmeldung erfolgt.

=back

Weitere Parameter können jederzeit übergeben werden, werden aber ignoriert,
werden aber, außer im Zusammenhang mit L</params>, ignoriert.

=cut

use Memoize qw/memoize/;
memoize '_read_config';

sub _read_config {
    my $filename = '/etc/sms-perl/sms.conf';
    unless ($filename) {
        croak "No file $filename found!\n";
    }
    return Config::INI::Reader->read_file($filename)->{'_'};
}

sub new {
    croak "->new() called with odd number of arguments." unless @_ & 1;
    my $self;
    if ( ref( my $package = shift ) ) {
        croak "->new() called with foreign object."
          unless $package->isa(__PACKAGE__);
        $self = bless {%$package}, ref $package;
    }
    else {
        my $config = _read_config();
        for (qw/user pass/) {
            die qq[Missing entry for "$_" in the config file!]
                unless defined $config->{$_};
        }
        my $project = $config->{project};
        unless ($project) {
            local $ENV{PATH} = '/bin:/usr/bin';
            $project = `hostname -f`
              or eval {
                require Sys::Hostname;
                $project = Sys::Hostname::hostname();
              };
            $project //= 'unknown';
            chomp $project;
        }
        if (not $project =~ m?.*/.*?) {
            # wenn Project noch kein "/", dann "-/" für undefinierten Kunden voranstellen
            $project = "-/$project";
        }
        $self = bless {
            debug   => 0,
            user    => $config->{user},
            enc     => 'gsm',
            max_sms => 10,
            params  => [
                qw(
                  user
                  debug
                  project
                  orig
                  pass
                  maxparts
                  enc
                  registered_delivery
                  dest
                  data
                  )
            ],
            pass                => $config->{pass},
            orig                => 'noris.net',
            project             => $project,
            registered_delivery => 1,
            dest                => [],
            data                => [],
            re_ok               => qr/^REQUEST OK.*/,
            sent                => 0,
            maxparts            => 10,
            timeout             => undef,
            urls                => ["https://sms.openit.de/put.php"],
        }, $package;
    }

    # Parameter die new() übergeben würden werden in die $self Conf übernommen
    for ( my $i = 0 ; $i < $#_ ; $i += 2 ) {
        if ( ref $self->{ $_[$i] } ) {
            @{ $self->{ $_[$i] } } =
              ref $_[ $i + 1 ] ? @{ $_[ $i + 1 ] } : $_[ $i + 1 ];
        }
        else { $self->{ $_[$i] } = $_[ $i + 1 ] }
    }
    $self;
}

=head1 METHODEN

Zugriff auf alle Parameter ist über lvalue-sub-Routinen möglich, z. B. kann
eine neue C<orig> so gesetzt werden:

	$object->orig = 'noris.net';

Der Versand von SMs erfolgt durch Aufruf der Methode C<-E<gt>send>, die ein
neues noris::SMS-Objekt zurückliefert; außerdem können hier optional dieselben
Parameter übergeben werden wie bei L<-E<gt>new|/KONSTRUKTOR>, die dann in das
neu erzeugte Objekt übernommen werden.

Ging beim Versand etwas schief, liefert C<-E<gt>errstr> eine Fehlermeldung
zurück, ansonsten L<undef|perlfunc/undef>.

Mit C<-E<gt>sent> kann abgefragt werden, wie viele SMs erfolgreich versandt
wurden.

C<-E<gt>todo> liefert eine Referenz auf eine Liste von noch abzuarbeitenden
Parameter-Kombinationen.

Mit C<-E<gt>retry> kann ein erneuter Versuch gestartet werden, diese noch
abzuarbeiten.

=cut

sub send {
    my $self = shift;
    croak "->send() called with foreign object." unless $self->isa(__PACKAGE__);
    my $clone = $self->new(@_);
    my @params = @{ $clone->params };
    $clone->{todo} = [
        map {
            my %param = @$_;
            $param{data}  = encode( 'openit_gsm0338', $param{data} );
            if ($param{orig}) {
                use charnames qw/:short/;
                my %chars = (
                    "\xC4"  => 'Ae',
                    "\xe4"  => 'ae',
                    "\xD6"  => 'Oe',
                    "\xF6"  => 'oe',
                    "\xDC"  => 'Ue',
                    "\xFC"  => 'ue',
                    "\xDF"  => 'ss',
                );
                my $re = join '|', keys %chars;
                $param{orig} =~ s/($re)/$chars{$1}/g;
                $param{orig} =~ s/[^\0-\177]+//g;
                $param{orig} = substr $param{orig}, 0, 11;
            }
            $param{debug} = '1' if $clone->{test};
            \%param;
        }
        $clone->_combine(@params)
    ];
    $clone->retry;
}

sub _number_of_sms {
    my ($self, %param) = @_;

    # Laut Dokumentation https://sms.openit.de → Entwickler → Datentypen:
    #
    #                Übersicht maximale Textlängen
    #
    # Kodierung           Einfach/Mehrteilig  Maximale Länge pro SMS
    # --------------------------------------------------------------
    # 7-Bit-GSM-Alphabet  Einfach             160 Zeichen
    # 7-Bit-GSM-Alphabet  Mehrteilig          153 Zeichen
    my $n;
    if ( (my $length = length $param{data}) <= 160 ) {
        $n = 1;
    }
    else {
        $n = ceil( $length / 153 );
    }
    $n = $param{maxparts} if $n > $param{maxparts};

    print "No of SMS $n\n" if $self->debug;

    $n;
}

sub retry {
    my $self = shift;
    croak "->retry() called with foreign object." unless $self->isa(__PACKAGE__);
    croak "->retry() does not accept arguments." if @_;
    my $ua =
      new LWP::UserAgent $self->timeout ? ( timeout => $self->{timeout} ) : ()
      or return $self->error('cannot create LWP::UserAgent object');
  COMBINATION: while ( @{ $self->todo } ) {
        my %todo = %{ $self->todo->[0] };

        my $number_of_sms = $self->_number_of_sms(%todo);

        if ( defined $self->max_sms
            && $self->sent + $number_of_sms > $self->max_sms )
        {
            print 'Limit of '
              . $self->max_sms . ' SM'
              . ( $self->max_sms != 1 && 's' )
              . " would be exhausted.\n"
              if $self->debug;
            last COMBINATION;
        }
        my @errors;
        for my $url ( @{ $self->urls } ) {

            my $post_object = HTTP::Request::Common::POST( $url, \%todo );
            print "I am sending the following Request:\n" . $post_object->as_string()
              if $self->debug;
            my $response = $ua->request( $post_object, () )
              or return $self->error( "unknown error POST $url\n",
                pp($post_object) );
            my $error;
            if ( $response->is_success ) {
                my $content = $response->content;
                if ( $content =~ $self->re_ok ) {
                    $self->sent += $number_of_sms;
                    next COMBINATION;
                }
                $error = "request returned: $content";
            }
            else {
                $error =
                  'error during POST request: ' . $response->status_line;
            }
            push @errors, $error;
            print "$error\n" if defined $error && $self->debug;
        }
        return $self->error( join '; ', @errors ) if @errors;
    }
    continue { shift @{ $self->todo } }
    $self->error(undef);
    $self;
}

# Diese Funktion erstellt eine Liste aller Kombinationen der
# Werte der Attribute, die als params gelten, und somit eine
# TODO-Liste für den Aufruf des Gateways.
sub _combine {
    my $self = shift;
    croak "->_combine() called with foreign object."
      unless $self->isa(__PACKAGE__);
    defined( my $key = shift ) or return;
    my @combinations = @_ ? $self->_combine(@_) : [];
    my $values = $self->$key;
    map {
        my $value = $_;
        map [ $key, $value, @$_ ], @combinations;
    } ref $values ? @$values : $values;
}

sub error {
    my $self = shift;
    croak "->error() called with foreign object."
      unless $self->isa(__PACKAGE__);
    if (@_) {

        # Aufruf mit (undef) löscht errstr:
        if ( @_ == 1 && !defined $_[0] ) { $self->{errstr} = undef }
        else {

            # Ansonsten wird das erste Element als Fehlermeldung betrachtet...
            $self->{errstr} = shift;

            # ... an die ggf. ein Dump aller weiteren übergebenen
            # Datenstrukturen angehängt wird:
            $self->{errstr} .= pp(@_) if @_;
        }
    }
    $self;
}

sub pp {
    local $^W;
    require Data::Dump and Data::Dump->import('pp');
    goto &pp;
}

sub AUTOLOAD : lvalue {
    my $self = shift;
    my ($function) = our $AUTOLOAD =~ /([^:]+)$/;
    croak "->$function() called with foreign object."
      unless $self->isa(__PACKAGE__);
    $self->{$function};
}

=head1 OVERLOADING

In einem Wahrheitskontext liefert jedes noris::SMS-Objekt wahr, falls sich kein
Fehler ereignet hat (entspricht C<!defined($object-E<gt>errstr)>), als String
die auch mit C<-E<gt>errstr> abfragbare Fehlermeldung und in einem numerischen
Kontext die Anzahl der erfolgreich versendeten SMs (entspricht
C<$object-E<gt>sent>).

=cut

use overload
  'bool' => sub { !defined shift->errstr },
  '0+'   => sub { shift->sent },
  '""'   => sub { shift->errstr };

1;
