#!/usr/bin/perl

## 
##  Copyright (C) Fujitsu Technology Solutions 2012
##  All rights reserved
##

# version string
my $version = '1.00.02.04';
# Based on SNMP MIB up to 2012 (ServerView Suite V6.10)

use strict;
use warnings;
use Getopt::Long qw(:config no_ignore_case bundling);
use Pod::Usage;
use Net::SNMP;
use Time::Local 'timelocal';
#use Time::localtime 'ctime';
use utf8;

=head1 NAME

check_fujitsu_server.pl - Nagios-Check-Plugin for various Fujitsu servers

=head1 SYNOPSIS

check_fujitsu_server.pl 
	{   -H|--host=<host> 
	    {   [ -C|--community=<SNMP community string> ] | 
	        { --user=<username> [--authpassword=<pwd>] [--authkey=<key>] [--authprot=<prot>] 
		  [--privpassword=<pwd>] [--privkey=<key>] [--privprot=<prot>]
		}
	    }
	    {   [--chkuptime] | [--chkmemperf [-w<percent>] [-c<percent>] ] |
		{ { [--blade] } | { [--pq | --primequest] } 
		  { [--chksystem] 
		    { [--chkenv] | [--chkenv-fan] [--chkenv-temp] }
		    [--chkpower] 
		    { [--chkhardware] | [--chkcpu] [--chkvoltage] [--chkmemmodule] }
		    [--chkstorage] 
		    [--chkdrvmonitor]
		  }
		} |
		{ [--bladeinside] | [--bladesrv] [--bladeio] [--bladekvm] [--bladestore] }
	    }
            [-t|--timeout=<timeout in seconds>]
            [-v|--verbose=<verbose mode level>]
	} | [-h|--help] | [-V|--version] 

Checks a Fujitsu server using SNMP.

=head1 OPTIONS

=over 4

=item -H|--host=<name-or-ip>

Host address as DNS name or ip address of the server 

=item -C|--community=<SNMP community string>

SNMP community of the server - usable for SNMPv1 and SNMPv2. Default is 'public'.

=item --user=<username> [--authpassword=<pwd>] [--authkey=<key>] [--authprot=<prot>] 
			[--privpassword=<pwd>] [--privkey=<key>] [--privprot=<prot>]

SNMPv3 authentication data. Default of authprotocol is 'md5' - Default of
privprotocol is 'des'. More about this options see Perl Net::SNMP session options.

=item --chkuptime

Tool option to check the SNMP access by reading the SNMP uptime via RFC1213.
This option can not be combined with other check options

=item --chkmemperf [-w<percent>] [-c<percent>]

PRIMERGY server: Get memory usage performance data (in percent).
This option can not be combined with other check options

With options -w and -c the warning and critical threshold limit can be set.
<percent> should be a simple integer 0..100

=item --blade

Check management blade (MMB)

=item --pq or --primequest

Check management information for a PRIMEQUEST system

=item --chksystem 

=item --chkenv | [--chkenv-fan] [--chkenv-temp]

=item --chkpower

=item --chkhardware | [--chkcpu] [--chkvoltage] [--chkmemmodule]

Select range of system information: System meaning anything besides
Environment (Fan, Temperature) or Power (Supply units and consumption).

"Check Hardware" can be combined only with PRIMEQUEST option and returns voltage,
cpu and memory-module information.

Options chkenv and chkhardware can be splittet to select only parts of the above mentioned
ranges.

=item --chkhardware | [--chkcpu] [--chkvoltage] [--chkmemmodule]

=item --chkstorage

For PRIMERGY server these options can be used to monitor only "Hardware" or only "MassStorage" parts.
These areas are part of the Primergy System Information

=item --chkdrvmonitor

For PRIMERGY server these options can be used to monitor "DriverMonitor" parts.

=item --bladeinside

Check all sub blade status values in a PRIMERGY Blade server. This is a combination
of --bladesrv --bladeio --bladekvm --bladestore where messages that one part doesn't
exist were suppressed.
(Do not combine with --blade - Use only for PRIMERGY Blades)

=item --bladesrv

Check server blade status values in a PRIMERGY Blade server.
(Do not combine with --blade - Use only for PRIMERGY Blades)

=item --bladeio

Check io connection blade status values in a PRIMERGY Blade server. 
(Do not combine with --blade - Use only for PRIMERGY Blades)

=item --bladekvm

Check key/video/mouse blade status values in a PRIMERGY Blade server. 
(Do not combine with --blade - Use only for PRIMERGY Blades)

=item --bladestore

Check storage blade status values in a PRIMERGY Blade server. 
(Do not combine with --blade - Use only for PRIMERGY Blades)

=item -t|--timeout=<timeout in seconds>

Timeout for the script processing.

=item -v|--verbose=<verbose mode level>

Enable verbose mode (levels: 1,2).
Generates Multi-line output with verbose level 2.

=item -V|--version

Print version information and help text.

=item -h|--help

Print help text.

=cut

# global control definitions
our $skipInternalNamesForNotifies = 1;


# define states
#### TEXT LANGUAGE AWARENESS (Standard in Nagios-Plugin)
our @state = ('OK', 'WARNING', 'CRITICAL', 'UNKNOWN');

# init main options
my $argvCnt = $#ARGV + 1;
my $optHost = '';
my $optTimeout = 0;
my $optShowVersion = undef;
my $optHelp = undef;
my $optCommunity = undef; #SNMPv1, SNMPv2
my $optUserName = undef; #SNMPv3
my $optAuthKey = undef; #SNMPv3
my $optAuthPassword = undef; #SNMPv3
my $optAuthProt = undef; #SNMPv3
my $optPrivKey = undef; #SNMPv3
my $optPrivPassword = undef; #SNMPv3
my $optPrivProt = undef; #SNMPv3

# global option
$main::verbose = 0;
$main::verboseTable = 0;

# init additional options
our $optChkSystem = undef;
our $optChkEnvironment = undef;
our	$optChkEnv_Fan = undef;
our	$optChkEnv_Temp = undef;
our $optChkPower = undef;
our $optChkHardware = undef;
our	$optChkCPU = undef;
our	$optChkVoltage = undef;
our	$optChkMemMod = undef;
our $optChkStorage = undef;
our $optChkDrvMonitor = undef;
our $optChkCpuLoadPerformance = undef;
our $optChkMemoryPerformance = undef;

our $optBlade = undef;
our $optBladeContent = undef;
our $optBladeSrv = undef;
our $optBladeIO = undef;
our	$optBladeIO_Switch = undef;
our	$optBladeIO_FCPT = undef;
our	$optBladeIO_Phy = undef;
our	$optBladeIO_FCSwitch = undef;
our	$optBladeIO_IBSwitch = undef;
our	$optBladeIO_SASSwitch = undef;
our	$optBladeIO_FSIOM = undef;
our $optBladeKVM = undef;
our $optBladeStore = undef;

our $optPrimeQuest = undef;

our $optChkUpTime = undef;

# special sub options
our $optWarningLimit = undef;
our $optCriticalLimit = undef;

# init output data
our $error = '';
our $msg = '';
our $longMessage = '';
our $performanceData = '';
our $exitCode = 3;
our $variableVerboseMessage = '';

# init some multi used processing variables
our $session;
our $serverID = undef;

#----------- multi usable functions
sub finalize {
	my $exitCode = shift;
	print "@_" if (defined @_);
	print "\n";
	alarm(0); # stop timeout
	exit($exitCode);
}
sub simpleSNMPget {

	my $oid = shift;
	my $topic = shift;

	print '-> OID \'' . $oid . '\'... ' if ($main::verbose >= 10);
	my $response = $main::session->get_request($oid);
	#### TEXT LANGUAGE AWARENESS
	finalize(3, $state[3], 'SNMP::get_request(' . $topic . '): ' . $main::session->error) unless (defined $response);
	print '<- response: ' . $response->{$oid} . "\n" if ($main::verbose >= 10);
	return $response->{$oid};
}
sub trySNMPget {
	my $oid = shift;
	my $topic = shift;

	print '-> OID \'' . $oid . '\'... ' if ($main::verbose >= 10);
	my $response = $main::session->get_request($oid);
	print '<- response: ' . ($response?$response->{$oid}:'---') . "\n" 
		if ($main::verbose >= 10) ;
	return ($response?$response->{$oid}:undef);
}
sub RFC1213sysinfoUpTime {
	# RFC1213.mib
	my $snmpOidSystem = '.1.3.6.1.2.1.1.'; #system
	my $snmpOidUpTime	= $snmpOidSystem . '3.0'; #sysUpTime.0
	my $uptime = trySNMPget($snmpOidUpTime,"sysUpTime");
	if ($uptime) {
		$exitCode = 0;
		$msg .= "- SNMP access OK - SNMP UpTime = $uptime";
	} else {
		$msg .= "Unable to get SNMP information";
		$longMessage .= "SNMP::get_request: " . $main::session->error;
	}
}
sub RFC1213sysinfoToLong {
	# RFC1213.mib
	my $snmpOidSystem = '.1.3.6.1.2.1.1.'; #system
	my $snmpOidDescr	= $snmpOidSystem . '1.0'; #sysDescr.0
	my $snmpOidContact	= $snmpOidSystem . '4.0'; #sysContact.0
	my $snmpOidName		= $snmpOidSystem . '5.0'; #sysName.0
	my $snmpOidLocation	= $snmpOidSystem . '6.0'; #sysLocation.0

	my $descr = trySNMPget($snmpOidDescr,"sysDescr");
	my $name = trySNMPget($snmpOidName,"sysName");
	my $contact = trySNMPget($snmpOidContact,"sysContact");
	my $location = trySNMPget($snmpOidLocation,"sysLocation");

	$longMessage .= "Systemname=$name " if ($name);
	$longMessage .= "Description=\"$descr\"" if ($descr);
	addLocationContactToLong($location, $contact);
} #RFC1213sysinfoToLong
sub mibTestSNMPget {

	my $oid = shift;
	my $topic = shift;

	print '-> OID \'' . $oid . '\'... ' if ($main::verbose >= 10 and !$main::verboseTable);
	my $response = $main::session->get_request($oid);
	RFC1213sysinfoToLong() unless (defined $response);
	$longMessage .= "\n" unless (defined $response);
	#### TEXT LANGUAGE AWARENESS
	finalize(3, $state[3], "Unable to get SNMP " . $topic . " information for this host",
		($main::session->error?"\nSNMP::get_request: " . $main::session->error:''),
		($longMessage?"\n" . $longMessage:'')) 
		unless (defined $response);
	print '<- response: ' . $response->{$oid} . "\n" if ($main::verbose >= 10 and !$main::verboseTable);
	return $response->{$oid};
}
sub addExitCode {
	my $addCode = shift;
	if (defined $addCode) {
		if ($exitCode == 3) {
			$exitCode = $addCode;
		} else {
			$exitCode = $addCode if ($addCode < 3 and $addCode > $exitCode);
		}
	}
} #addExitCode

sub negativeValueCheck {
		my $val = shift;
		my $maxval = 0xFFFFFFFF;
		return undef if (!defined $val);
		return $val if ($val < 0x7FFFFFFF);

		my $diffval = $maxval - $val;
		my $newval = "-" . "$diffval";
		return $newval;
} #negativeValueCheck
#----------- performance data functions
sub addTemperatureToPerfdata {
	my $name = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	#my $min = shift;
	#my $max = shift;

	if (defined $name and $current) {
		$performanceData .= ' ' . $name . '=' if ($name);
		$performanceData .= $current . 'C';
		$performanceData .= ';' if ($warning or $critical);
		$performanceData .= "$warning" if ($warning);
		$performanceData .= ';' if ($critical);
		$performanceData .= "$critical" if ($critical);
	}
} #addTemperatureToPerfdata
sub addPowerConsumptionToPerfdata {
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;

	$current = undef if (defined $current and ($current == 0 or $current == -1));

	if ($current) {
		$current = negativeValueCheck($current);
		$warning = undef	if (defined $warning and ($warning == 0 or $warning ==-1));
		$critical = undef	if (defined $critical and ($critical == 0 or $critical ==-1));
		$min = undef		if (defined $min and ($min == 0 or $min == -1));
		$max = undef		if (defined $max and ($max == 0 or $max == -1));
		$warning = negativeValueCheck($warning);
		$critical = negativeValueCheck($critical);
		$min = negativeValueCheck($min);
		$max = negativeValueCheck($max);

		$warning = ''		if (!defined $warning and (defined $critical or defined $min or defined $max));
		$critical = ''		if (!defined $critical and (defined $min or defined $max));
		$min = ''		if (!defined $min and defined $max);

		$performanceData .= " PowerConsumption=$current" . "Watt";
		$performanceData .= ";$warning" if (defined $warning);
		$performanceData .= ";$critical" if (defined $critical);
		$performanceData .= ";$min" if (defined $min);
		$performanceData .= ";$max" if (defined $max);
	}
} #addPowerConsumptionToPerfdata
sub addPercentageToPerfdata {
	my $name = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;

	$performanceData .= ' ' . $name if ($name);
	$performanceData .= '=' . $current . '%' if (defined $current);
	$performanceData .= ';' if ($warning or $critical);
	$performanceData .= "$warning" if ($warning);
	$performanceData .= ';' if ($critical);
	$performanceData .= "$critical" if ($critical);
} #addPercentageToPerfdata
#----------- synchronize verbose output format functions
sub addStatusTopicToVerbose {
	my $status = shift;
	my $topic = shift;
	my $index = shift;
	$variableVerboseMessage .= "$status:" if (defined $status);
	$variableVerboseMessage .= " " if (defined $status and ($topic or $index));
	$variableVerboseMessage .= "$topic" if ($topic);
	$variableVerboseMessage .= "[$index]" if (defined $index);
	$variableVerboseMessage .= " -" if (!defined $status and ($topic or $index));
}
sub addStatusTopicToLong {
	my $status = shift;
	my $topic = shift;
	my $index = shift;
	$longMessage .= "$status:" if (defined $status);
	$longMessage .= " " if (defined $status and ($topic or $index));
	$longMessage .= "$topic" if ($topic);
	$longMessage .= "[$index]" if (defined $index);
	$longMessage .= " -" if (!defined $status and ($topic or $index));
}
sub endVariableVerboseMessageLine {
	$variableVerboseMessage .= "\n";
}
sub endLongMessageLine {
	$longMessage .= "\n";
}

sub addKeyValueToVerbose {
	my $key = shift;
	my $value = shift;
	$variableVerboseMessage .= " $key=$value" if ($value);
}
sub addKeyIntValueToVerbose {
	my $key = shift;
	my $value = shift;
	$variableVerboseMessage .= " $key=$value" if (defined $value);
}
sub addKeyUnsignedIntValueToVerbose {
	my $key = shift;
	my $value = shift;
	$variableVerboseMessage .= " $key=$value" if (defined $value and $value >= 0);
}
sub addKeyValueToLong {
	my $key = shift;
	my $value = shift;
	$longMessage .= " $key=$value" if ($value);
}
sub addKeyIntValueToLong {
	my $key = shift;
	my $value = shift;
	$longMessage .= " $key=$value" if (defined $value);
}
sub addKeyUnsignedIntValueToLong {
	my $key = shift;
	my $value = shift;
	$longMessage .= " $key=$value" if (defined $value and $value >= 0);
}
sub addKeyLongValueToVerbose {
	my $key = shift;
	my $value = shift;
	$variableVerboseMessage .= " $key=\"$value\"" if ($value);
}
sub addKeyLongValueToLong {
	my $key = shift;
	my $value = shift;
	$longMessage .= " $key=\"$value\"" if ($value);
}

sub addTableHeaderToVerbose {
	my $header = shift;
	$variableVerboseMessage .= "* $header:\n" if ($header);
}

sub addComponentStatusToVerbose {
	my $comp = shift;
	my $status = shift;
	$variableVerboseMessage .= " $comp($status)" if ($comp and defined $status);
}
sub addComponentStatusToLong {
	my $comp = shift;
	my $status = shift;
	$longMessage .= " $comp($status)" if ($comp and defined $status);
}

sub addCelsiusToVerbose {
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	if ($current) {
		$variableVerboseMessage .= " Temperature=$current" . "C";
		$variableVerboseMessage .= " Warning=$warning" . "C" if (defined $warning);
		$variableVerboseMessage .= " Critical=$critical" . "C" if (defined $critical);
	}
}
sub addCelsiusToLong {
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	if ($current) {
		$longMessage .= " Temperature=$current" . "C";
		$longMessage .= " Warning=$warning" . "C" if (defined $warning);
		$longMessage .= " Critical=$critical" . "C" if (defined $critical);
	}
}
sub addmVoltToVerbose {
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$current = negativeValueCheck($current);
	$warning = negativeValueCheck($warning);
	$critical = negativeValueCheck($critical);
	$min = negativeValueCheck($min);
	$max = negativeValueCheck($max);
	$variableVerboseMessage .= " Current=$current" . "mV" if (defined $current and $current != -1);
	$variableVerboseMessage .= " Warning=$warning" . "mV" if (defined $warning and $warning != -1);
	$variableVerboseMessage .= " Critical=$critical" . "mV" if (defined $critical and $critical != -1);
	$variableVerboseMessage .= " Min=$min" . "mV" if (defined $min and $min != -1);
	$variableVerboseMessage .= " Max=$max" . "mV" if (defined $max and $max != -1);
}
sub addmVoltToLong {
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$current = negativeValueCheck($current);
	$warning = negativeValueCheck($warning);
	$critical = negativeValueCheck($critical);
	$min = negativeValueCheck($min);
	$max = negativeValueCheck($max);
	$longMessage .= " Current=$current" . "mV" if (defined $current and $current != -1);
	$longMessage .= " Warning=$warning" . "mV" if (defined $warning and $warning != -1);
	$longMessage .= " Critical=$critical" . "mV" if (defined $critical and $critical != -1);
	$longMessage .= " Min=$min" . "mV" if (defined $min and $min != -1);
	$longMessage .= " Max=$max" . "mV" if (defined $max and $max != -1);
}
sub addKeyRpmToVerbose {
	my $key = shift;
	my $speed = shift;
	$speed = undef if (defined $speed and $speed == -1);
	$variableVerboseMessage .= " $key=$speed" . "rpm" if ($speed);
}
sub addKeyRpmToLong {
	my $key = shift;
	my $speed = shift;
	$speed = undef if (defined $speed and $speed == -1);
	$longMessage .= " $key=$speed" . "rpm" if ($speed);
}
sub addKeyMHzToVerbose {
	my $key = shift;
	my $speed = shift;
	$speed = undef if (defined $speed and $speed == -1);
	$variableVerboseMessage .= " $key=$speed" . "MHz" if ($speed);
}
sub addKeyMHzToLong {
	my $key = shift;
	my $speed = shift;
	$speed = undef if (defined $speed and $speed == -1);
	$longMessage .= " $key=$speed" . "MHz" if ($speed);
}
sub addKeyMBToVerbose {
	my $key = shift;
	my $mbytes = shift;
	$mbytes = undef if (defined $mbytes and $mbytes < 0);
	$variableVerboseMessage .= " $key=$mbytes" . "MB" if (defined $mbytes);
}
sub addKeyMBToLong {
	my $key = shift;
	my $mbytes = shift;
	$mbytes = undef if (defined $mbytes and $mbytes < 0);
	$longMessage .= " $key=$mbytes" . "MB" if (defined $mbytes);
}
sub addKeyGBToVerbose {
	my $key = shift;
	my $gbytes = shift;
	$gbytes = undef if (defined $gbytes and $gbytes < 0);
	$variableVerboseMessage .= " $key=$gbytes" . "GB" if (defined $gbytes);
}
sub addKeyGBToLong {
	my $key = shift;
	my $gbytes = shift;
	$gbytes = undef if (defined $gbytes and $gbytes < 0);
	$longMessage .= " $key=$gbytes" . "GB" if (defined $gbytes);
}
sub addKeyWattToVerbose {
	my $key = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$variableVerboseMessage .= " $key=$current" . "Watt" if ($current and $current != -1);
	$variableVerboseMessage .= " Warning=$warning" . "Watt" if ($warning and $warning != -1);
	$variableVerboseMessage .= " Critical=$critical" . "Watt" if ($critical and $critical != -1);
	$variableVerboseMessage .= " Min=$min" . "Watt" if ($min and $min != -1);
	$variableVerboseMessage .= " Max=$max" . "Watt" if ($max and $max != -1);
}
sub addKeyWattToLong {
	my $key = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$longMessage .= " $key=$current" . "Watt" if ($current and $current != -1);
	$longMessage .= " Warning=$warning" . "Watt" if ($warning and $warning != -1);
	$longMessage .= " Critical=$critical" . "Watt" if ($critical and $critical != -1);
	$longMessage .= " Min=$min" . "Watt" if ($min and $min != -1);
	$longMessage .= " Max=$max" . "Watt" if ($max and $max != -1);
}
sub addKeyPercentToVerbose {
	my $key = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$variableVerboseMessage .= " $key=$current" . "%" if (defined $current and $current != -1);
	$variableVerboseMessage .= " Warning=$warning" . "%" if (defined $warning and $warning != -1);
	$variableVerboseMessage .= " Critical=$critical" . "%" if (defined $critical and $critical != -1);
	$variableVerboseMessage .= " Min=$min" . "%" if (defined $min and $min != -1);
	$variableVerboseMessage .= " Max=$max" . "%" if (defined $max and $max != -1);
}
sub addKeyPercentToLong {
	my $key = shift;
	my $current = shift;
	my $warning = shift;
	my $critical = shift;
	my $min = shift;
	my $max = shift;
	$longMessage .= " $key=$current" . "%" if (defined $current and $current != -1);
	$longMessage .= " Warning=$warning" . "%" if (defined $warning and $warning != -1);
	$longMessage .= " Critical=$critical" . "%" if (defined $critical and $critical != -1);
	$longMessage .= " Min=$min" . "%" if (defined $min and $min != -1);
	$longMessage .= " Max=$max" . "%" if (defined $max and $max != -1);
}

sub addSerialIDsToVerbose {
	my $id = shift;
	my $id2 = shift;
	if ((defined $id) && ($id =~ m/00000000000/)) {
		$id = undef;
	}
	if ((defined $id) && ($id =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)) {
		$id = undef;
	}
	$variableVerboseMessage .= " ID=$id" if ($id);
	$variableVerboseMessage .= " ID2=$id2" if ($id2);
}
sub addSerialIDsToLong {
	my $id = shift;
	my $id2 = shift;
	if ((defined $id) && ($id =~ m/00000000000/)) {
		$id = undef;
	}
	if ((defined $id) && ($id =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)) {
		$id = undef;
	}
	$longMessage .= " ID=$id" if ($id);
	$longMessage .= " ID2=$id2" if ($id2);
}
sub addSlotIDToVerbose {
	my $slot = shift;
	$variableVerboseMessage .= " Slot=$slot" if (defined $slot);
}
sub addSlotIDToLong {
	my $slot = shift;
	$longMessage .= " Slot=$slot" if (defined $slot);
}
sub addIPToVerbose {
	my $ip = shift;
	$ip = undef if (($ip) and ($ip =~ m/0\.0\.0\.0/));

	$variableVerboseMessage .= " IP=$ip" if ($ip);
}
sub addIPToLong {
	my $ip = shift;
	$ip = undef if (($ip) and ($ip =~ m/0\.0\.0\.0/));
	$longMessage .= " IP=$ip" if ($ip);
}
sub addNameToVerbose {
	my $name = shift;
	$name = undef if (defined $name and $name eq '');
	$variableVerboseMessage .= " Name=$name" if ($name);
}
sub addNameToLong {
	my $name = shift;
	$name = undef if (defined $name and $name eq '');
	$longMessage .= " Name=$name" if ($name);
}
sub addHostNameToVerbose {
	my $hostname = shift;
	$variableVerboseMessage .= " Hostname=$hostname" if ($hostname);
}
sub addHostNameToLong {
	my $hostname = shift;
	$longMessage .= " Hostname=$hostname" if ($hostname);
}
sub addAdminURLToVerbose {
	my $admURL = shift;
	$admURL = undef if ($admURL and ($admURL !~ m/http/));
	$admURL = undef if ($admURL and ($admURL =~ m/0\.0\.0\.0/));
	$admURL = undef if ($admURL and ($admURL =~ m/255\.255\.255\.255/));
	$admURL = undef if ($admURL and ($admURL =~ m/\/\/127\./));
	$variableVerboseMessage .= " AdminURL=$admURL" if ($admURL);
}
sub addAdminURLToLong {
	my $admURL = shift;
	$admURL = undef if ($admURL and ($admURL !~ m/http/));
	$admURL = undef if ($admURL and ($admURL =~ m/0\.0\.0\.0/));
	$admURL = undef if ($admURL and ($admURL =~ m/255\.255\.255\.255/));
	$admURL = undef if ($admURL and ($admURL =~ m/\/\/127\./));
	$longMessage .= " AdminURL=$admURL" if ($admURL);
}
sub addProductModelToVerbose {
	my $product = shift;
	my $model = shift;
	if ((defined $product) 
	&& ($product =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)) {
		$product = undef;
	}
	if ((defined $model) 
	&& ($model =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)) {
		$model = undef;
	}
	$variableVerboseMessage .= " Product=\"$product\"" if ($product);
	$variableVerboseMessage .= " Model=\"$model\"" if ($model);
}
sub addProductModelToLong {
	my $product = shift;
	my $model = shift;
	if ((defined $product) 
	&& ($product =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)
	) {
		$product = undef;
	}
	if (defined $product and $skipInternalNamesForNotifies
	and ($product =~ m/^D\d{4}$/)
	) {
		$product = undef;
	}
	if ((defined $model) 
	&& ($model =~ m/0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff/)) {
		$model = undef;
	}
	if (defined $model and $skipInternalNamesForNotifies
	and ($model =~ m/A3C\d{8}/)) {
		$model = undef;
	}
	$longMessage .= " Product=\"$product\"" if ($product);
	$longMessage .= " Model=\"$model\"" if ($model);
}
sub addLocationContactToVerbose {
	my $location = shift;
	my $contact = shift;
	$location = undef if(defined $location and $location eq '');
	$contact = undef if(defined $contact and $contact eq '');
	$variableVerboseMessage .= " Location=\"$location\"" if ($location);
	$variableVerboseMessage .= " Contact=\"$contact\"" if ($contact);
}
sub addLocationContactToLong {
	my $location = shift;
	my $contact = shift;
	$location = undef if(defined $location and $location eq '');
	$contact = undef if(defined $contact and $contact eq '');
	$longMessage .= " Location=\"$location\"" if ($location);
	$longMessage .= " Contact=\"$contact\"" if ($contact);
}
#----------- primequest PSA-COM functions
our %psaStatusMap = (
                     0x00000000		=> 3,                           
                     0x10000000		=> 2,
                     0x01000000		=> 6,
                     0x00100000		=> 5,
                     0x00040000		=> 4,
                     0x00010000		=> 1,           
);
our @psaStatusText = (	undef,
	"unknown", "not-present", "ok", "degraded", "warning", "error",
);
sub primequestPsaComManagementInfo {
	my $partID = shift;
	#--	partitionManagementInfo :	.1.3.6.1.4.1.211.1.31.1.2.100.7.5
	my $snmpOidManagementInfoGroup = '.1.3.6.1.4.1.211.1.31.1.2.' . $partID . '.7.5.'; #partitionManagementInfo
	my $snmpOidName		= $snmpOidManagementInfoGroup . '2.0'; #partitionName
	my $snmpOidHostName	= $snmpOidManagementInfoGroup . '3.0'; #hostName
	my $snmpOidManAddress2	= $snmpOidManagementInfoGroup . '10.0'; #partitionManagementAddress2

	if ($main::verbose >= 1 and !$main::verboseTable) {
		my $name = trySNMPget($snmpOidName, "partitionManagementInfo");
		my $hostname = trySNMPget($snmpOidHostName, "partitionManagementInfo");
		my $manAddress2 = trySNMPget($snmpOidManAddress2, "partitionManagementInfo");

		if ($name or $hostname or $manAddress2) {
			addStatusTopicToVerbose(undef, "Partition", $partID);
			addKeyValueToVerbose("Name", $name);
			addHostNameToVerbose($hostname);
			addIPToVerbose($manAddress2);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequestPsaComManagementInfo
sub primequesPsaComPhysCompStatus {
	my $partID = shift;
	#--	physCompStatus :	.1.3.6.1.4.1.211.1.31.1.2.100.6.1
	my $snmpOidPhysCompStatusGroup = '.1.3.6.1.4.1.211.1.31.1.2.' . $partID . '.6.1.'; #physCompStatus
	my $snmpOidCpu		= $snmpOidPhysCompStatusGroup .	'1.0'; #cpuParTotalStatus
	my $snmpOidMem		= $snmpOidPhysCompStatusGroup .	'2.0'; #memoryParTotalStatus
	my $snmpOidPci		= $snmpOidPhysCompStatusGroup .	'3.0'; #pciFuncParTotalStatus
	my $snmpOidSysBoard	= $snmpOidPhysCompStatusGroup .	'4.0'; #systemBoardParTotalStatus
	my $snmpOidIoBoard	= $snmpOidPhysCompStatusGroup .	'5.0'; #ioBoardParTotalStatus
	my $snmpOidScsi		= $snmpOidPhysCompStatusGroup .	'6.0'; #scsiDevParTotalStatus

	# Remark: In tests only PCI and SCSI Status was printed

	if ($main::verbose >= 1 and !$main::verboseTable) {
		my $cpu = trySNMPget($snmpOidCpu, "partitionManagementInfo");
		my $mem = trySNMPget($snmpOidMem, "partitionManagementInfo");
		my $pci = trySNMPget($snmpOidPci, "partitionManagementInfo");
		my $sysBoard = trySNMPget($snmpOidSysBoard, "partitionManagementInfo");
		my $ioBoard = trySNMPget($snmpOidIoBoard, "partitionManagementInfo");
		my $scsi = trySNMPget($snmpOidScsi, "partitionManagementInfo");
		$sysBoard = undef if (defined $sysBoard and $sysBoard eq "NULL");
		if (defined $cpu or defined $mem or defined $pci or defined $sysBoard 
		or  defined $ioBoard or defined $scsi
		) {
			addStatusTopicToVerbose(undef, "Partition", $partID);
			addComponentStatusToVerbose("CPU", $psaStatusText[$psaStatusMap{$cpu}]) 
				if (defined $cpu);
			addComponentStatusToVerbose("Memory", $psaStatusText[$psaStatusMap{$mem}]) 
				if (defined $mem);
			addComponentStatusToVerbose("PCI", $psaStatusText[$psaStatusMap{$pci}]) 
				if (defined $pci);
			addComponentStatusToVerbose("SystemBoard", $psaStatusText[$psaStatusMap{$sysBoard}]) 
				if (defined $sysBoard);
			addComponentStatusToVerbose("IO-Board", $psaStatusText[$psaStatusMap{$ioBoard}]) 
				if (defined $ioBoard);
			addComponentStatusToVerbose("SCSI", $psaStatusText[$psaStatusMap{$scsi}]) 
				if (defined $scsi);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequesPsaComPhysCompStatus
sub primequestPsaComPhysicalCpuTable {
	my $partID = shift;
	#--	physicalCpuTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.1.1	
	my $snmpOidCpuTable = '.1.3.6.1.4.1.211.1.31.1.2.' . $partID . '.1.1.1.1.'; #physicalCpuTable (2 index)
	my $snmpOidName		= $snmpOidCpuTable . '3'; #phCpuPhysicalID
	my $snmpOidOldStatus	= $snmpOidCpuTable . '4'; #phCpuStatus - bits
	my $snmpOidStatus	= $snmpOidCpuTable . '6'; #phCpuPresentStatus - bits
		# ... in tests both status values were not present
	my @tableChecks = ( $snmpOidName, $snmpOidOldStatus, $snmpOidStatus);

	if ($main::verbose >= 2 and !$main::verboseTable) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidName.(\d+\.\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		$variableVerboseMessage .= "  CPU Table:\n" if ($#snmpIDs > 0);
		foreach my $snmpID (@snmpIDs) {
			my $name = $entries->{$snmpOidName . '.' . $snmpID};
			#my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			#my $oldstatus = $entries->{$snmpOidOldStatus . '.' . $snmpID};
			#my $bitStatus = undef;
			#$bitStatus = $oldstatus if (defined $oldstatus);
			#$bitStatus = $status if (!defined $bitStatus);
			#my $decimalstatus = undef;
			#$decimalstatus = $psaStatusMap{$bitStatus} if (defined $bitStatus);

			addStatusTopicToVerbose(undef, "    CPU", $snmpID);
			addNameToVerbose($name);
			#$variableVerboseMessage .= " Status=$psaStatusText[$decimalstatus]:" if (defined $decimalstatus);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequestPsaComPhysicalCpuTable
sub primequestPsaComSystemBoardTable {
	my $partID = shift;
	#--	systemBoardTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.6.1
	my $snmpOidSystemBoardTable = '.1.3.6.1.4.1.211.1.31.1.2.' . $partID . '.1.6.1.1.'; #systemBoardTable (1 index)
	my $snmpOidId		= $snmpOidSystemBoardTable . '2'; #sbPhysicalID
	my $snmpOidModel	= $snmpOidSystemBoardTable . '10'; #sbModel
	my $snmpOidSerialNo	= $snmpOidSystemBoardTable . '11'; #sbSerialNo
	my @tableChecks = ( 
		$snmpOidId, $snmpOidModel, $snmpOidSerialNo,
	);
	if ($main::verbose >= 2 and !$main::verboseTable) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidId.(\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		#print("COUNT=$#snmpIDs\n");
		$variableVerboseMessage .= "  System Board Table:\n" if ($#snmpIDs >= 0);
		foreach my $snmpID (@snmpIDs) {
			my $id = $entries->{$snmpOidId . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $serial = $entries->{$snmpOidSerialNo . '.' . $snmpID};

			addStatusTopicToVerbose(undef, "    SystemBoard", $snmpID);
			addSerialIDsToVerbose($serial, undef);
			addNameToVerbose($id);
			addProductModelToVerbose(undef, $model);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequestPsaComSystemBoardTable
sub primequestPSACOM {
	#### Hint of ServerView OperationsManager: use partitionID 0-4 instead of '100'

	#PSA-COM-MIB.mib
	my $i = 0;
	if ($optChkSystem and !$main::verboseTable) {
		addTableHeaderToVerbose("Partition Management Info");
		for ($i=0;$i<4;$i++) {
			primequestPsaComManagementInfo($i);
		}
		if ($main::verbose >= 3) { # in tests only PCI and SCSI status was shown
			addTableHeaderToVerbose("Partition Related Physical Component Status");
			for ($i=0;$i<4;$i++) {
				primequesPsaComPhysCompStatus($i);
			}
		}
		addTableHeaderToVerbose("Partition Related Physical Component Tables");
		for ($i=0;$i<4;$i++) {
			addStatusTopicToVerbose(undef, "Partition", $i);
			$variableVerboseMessage .= "\n";
			primequestPsaComSystemBoardTable($i);
			primequestPsaComPhysicalCpuTable($i);
			
			#--	memoryModuleTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.2.1 id status ?	
			#--	scsiDeviceTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.3.1 status ?
			#--	diskSliceTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.3.2 perf ?
			#--	nicTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.4.1 perf ?
			#--	pciFunctionTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.5.1
			#--	iobTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.7.1
			#--	iobSlotTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.7.2
			#--	sduTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.8.1
			#--	sduControllerTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.8.2
			#--	sduFanTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.8.3
			#--	sduPsuTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.8.4
			#--	pciboxTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.9.1	no status
			#--	pciboxSlotTable :	.1.3.6.1.4.1.211.1.31.1.2.100.1.9.2	no real status
			#--	gspbTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.9.3	no status
			#--	sasuTable :		.1.3.6.1.4.1.211.1.31.1.2.100.1.9.4	no status
		}
	} #System
} #primequestPSACOM

#----------- primequest functions
our $pqTempStatus = undef;
our $pqFanStatus = undef;
our $pqPowerSupplyStatus = undef;
our $pqVoltageStatus = undef;
our $pqCPUStatus = undef;
our $pqMemoryStatus = undef;

# UnitIds
# 1 - Chassis
# 2-5 Partition
# 18 free-pool,
# 19-22 sb, 43-44 iou, 51-54 liou-div, 67-68 pci-box ,
# 118-125 fan-tray, 136-137 mmb,
# 143-144 gspb, 145-148 lgspb-dev, 149-152 sas-unit,
# 153-156 psu, 157 dvdb, 158-161 lpci-box

our @pqClassText = ( "none",
	"chassis", "partition", "free-pool", "sb", undef,
	undef, "iou", "iou-divided", "iou-nodivided", "pci-box",
	undef, undef, undef, undef, undef,
	undef, "fan-tray", undef, undef, undef,
	"mmb", undef, undef, undef, undef,
	undef, undef, "gspb", "gspb-divided", "gspb-nodivided",
	"sas-unit", "psu", "dvdb", "lpci-box", "..unexpected..",
);
sub primequestStatusComponentTable {
	my $snmpOidOverallStatusArea = '.1.3.6.1.4.1.211.1.31.1.1.1.8.'; #status
	my %commonSystemCodeMap = (	1       =>      0,
					2       =>      1,
					3       =>      2,
					4       =>      2,
					5	=>	3,);
	my $snmpOidStatusTable = $snmpOidOverallStatusArea . '2.1.';#statusComponentTable
	my $snmpOidId			= $snmpOidStatusTable . '1'; #csUnitId
	my $snmpOidType			= $snmpOidStatusTable . '2'; #csType
	my $snmpOidValue		= $snmpOidStatusTable . '3'; #componentStatusValue
	my @tableChecks = ( 
		$snmpOidId, $snmpOidType, $snmpOidValue, 
	);
	my @cntCodes = ( -1, 0,0,0,0,0, );
	my @overallStatusText = ( "none", 
		"ok", "degraded", "error", "failed", "unknown", "..unexpected..",
	);
	my @typeText = ( "none", 
		"system-boot", "power", "temperatures", "fans", "power-supplies",
		"voltages", "cpus", "memory-modules", "total", "fan-redundancy", 
		"power-supply-redundancy", "..unexpected..",
	); 

	{ # Chassis:
		$msg .= " - Chassis:" if ($optChkSystem and $optChkSystem == 999);
		for (my $i=1;$i < 10 ;$i++) {
			my $value = trySNMPget($snmpOidValue . '.1.' . $i ,"StatusComponentTable");
			if (defined $value) {
				$value = 0 if ($value < 0);
				$value = 6 if ($value > 6);
				$msg .= " $typeText[$i]($overallStatusText[$value])" 
					if (($optChkSystem and $optChkSystem == 999)
					or  ($optChkEnvironment and ($i == 3 or $i == 4))
					or  ($optChkEnv_Fan and $i == 4)
					or  ($optChkEnv_Temp and $i == 3)
					or  ($optChkPower and ($i == 5))
					or  ($optChkHardware and ($i == 6 or $i == 7 or $i == 8))
					or  ($optChkVoltage and $i == 6)
					or  ($optChkCPU and $i == 7)
					or  ($optChkMemMod and $i == 8)
					);
				$pqTempStatus		= $commonSystemCodeMap{$value} if ($i == 3);
				$pqFanStatus		= $commonSystemCodeMap{$value} if ($i == 4);
				$pqPowerSupplyStatus	= $commonSystemCodeMap{$value} if ($i == 5);
				$pqVoltageStatus	= $commonSystemCodeMap{$value} if ($i == 6);
				$pqCPUStatus		= $commonSystemCodeMap{$value} if ($i == 7);
				$pqMemoryStatus		= $commonSystemCodeMap{$value} if ($i == 8);
			} # value
		} # for
	}
	if ($optChkSystem) { # partitions
		addTableHeaderToVerbose("StatusComponentTable - Partitions");
		for (my $unitID=2; $unitID < 6 ; $unitID++) { 
			my $nr = $unitID - 2;
			my $gotPartition = undef;
			for (my $i=1;$i < 10 ;$i++) { 
				my $value = trySNMPget($snmpOidValue . '.' . $unitID . '.' . $i ,"StatusComponentTable");
				if (defined $value and !defined $gotPartition) {
					$gotPartition = 1;
					addStatusTopicToVerbose(undef, "Partition", $nr);
				}
				addComponentStatusToVerbose($typeText[$i], $overallStatusText[$value]) 
					if (defined $value);
				$cntCodes[$value]++ if (defined $value and $i == 9);
			}
			$variableVerboseMessage .= "\n" if $gotPartition;
		}
		$msg .= " - Partitions:";
 		for (my $i=1;$i < 6;$i++) {
 			$msg .= "-$overallStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
	}
	if ($optChkSystem) { # system board
		@cntCodes = ( -1, 0,0,0,0,0, );
		addTableHeaderToVerbose("StatusComponentTable - SystemBoards");
		for (my $unitID=19; $unitID <= 22 ; $unitID++) { 
			my $nr = $unitID -19;
			my $gotSB = undef;
			
			for (my $i=1;$i < 10 ;$i++) { 
				my $value = trySNMPget($snmpOidValue . '.' . $unitID . '.' . $i ,"StatusComponentTable");
				if (defined $value and !defined $gotSB) {
					$gotSB = 1;
					addStatusTopicToVerbose(undef, "SystemBoard", $nr);
				}
				addComponentStatusToVerbose($typeText[$i], $overallStatusText[$value]) 
					if (defined $value);
				$cntCodes[$value]++ if (defined $value and $i == 9);
			}
			$variableVerboseMessage .= "\n" if ($gotSB);
		}
		$msg .= " - SystemBoard:";
 		for (my $i=1;$i < 6;$i++) {
 			$msg .= "-$overallStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
	}	
	if ($main::verboseTable == 821 and $optChkSystem) { # needs to much time:
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidType.(\d+\.\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		addTableHeaderToVerbose("StatusComponentTable - Other Units");
		foreach my $snmpID (@snmpIDs) {
			my $unit = $entries->{$snmpOidId . '.' . $snmpID};
			my $status = $entries->{$snmpOidValue . '.' . $snmpID};
			my $type = $entries->{$snmpOidType . '.' . $snmpID};
			$type = 0 if (!defined $type or $type < 0);
			$type = 12 if ($type > 12);
			if ($type and $status and ($unit == 18 or $unit > 22)) {
				addComponentStatusToVerbose("[$unit]-$typeText[$type]", 
					$overallStatusText[$status]);
			}
		}
		$variableVerboseMessage .= "\n";
	}
} #primequestStatusComponentTable
sub primequestUnitParentTable {
	my $snmpOidUnitParentTable = '.1.3.6.1.4.1.211.1.31.1.1.1.2.5.1.'; #unitParentTable
	my $snmpOidUnitId	= $snmpOidUnitParentTable . '1'; #pUnitId
	my $snmpOidParentNr	= $snmpOidUnitParentTable . '2'; #pParentNr
	my $snmpOidPClass	= $snmpOidUnitParentTable . '4'; #parentUnitClass
	my @tableChecks = (
		$snmpOidUnitId, $snmpOidParentNr, $snmpOidPClass, 
	);
	if ($main::verboseTable == 251) { #UnitParentTable
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Unit Parent Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidUnitId.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $unit = $entries->{$snmpOidUnitId . '.' . $snmpID};
			my $pnr = $entries->{$snmpOidParentNr . '.' . $snmpID};
			my $pclass = $entries->{$snmpOidPClass . '.' . $snmpID};
			$pclass = 0 if (!defined $pclass or $pclass <= 0);
			$pclass = 35 if ($pclass > 35);

			$variableVerboseMessage .= "$unit <--- $pnr";
			addKeyValueToVerbose("Class", $pqClassText[$pclass]);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequestUnitParentTable
sub primequestUnitChildTable {
	my $snmpOidUnitChildTable = '.1.3.6.1.4.1.211.1.31.1.1.1.2.6.1.'; #unitChildTable
	my $snmpOidUnitId	= $snmpOidUnitChildTable . '1'; #cUnitId
	my $snmpOidChildNr	= $snmpOidUnitChildTable . '2'; #cChildNr
	my $snmpOidCClass	= $snmpOidUnitChildTable . '4'; #childUnitClass
	my @tableChecks = (
		$snmpOidUnitId, $snmpOidChildNr, $snmpOidCClass, 
	);
	if ($main::verboseTable == 261) { #UnitChildTable
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Unit Child Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidUnitId.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $unit = $entries->{$snmpOidUnitId . '.' . $snmpID};
			my $cnr = $entries->{$snmpOidChildNr . '.' . $snmpID};
			my $cclass = $entries->{$snmpOidCClass . '.' . $snmpID};
			$cclass = 0 if (!defined $cclass or $cclass <= 0);
			$cclass = 35 if ($cclass > 35);

			$variableVerboseMessage .= "$unit ---> $cnr";
			addKeyValueToVerbose("Class", $pqClassText[$cclass]);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primequestUnitChildTable
sub primequestUnitTable {
	my $snmpOidUnitTable = '.1.3.6.1.4.1.211.1.31.1.1.1.2.3.1.'; #mmb sysinfo unitInformation unitTable
	my $snmpOidClass	= $snmpOidUnitTable .  '2'; #unitClass
	my $snmpOidDesignation	= $snmpOidUnitTable .  '4'; #unitDesignation
	my $snmpOidModel	= $snmpOidUnitTable .  '5'; #unitModelName
	my $snmpOidSerial	= $snmpOidUnitTable .  '7'; #unitSerialNumber
	my $snmpOidLocation	= $snmpOidUnitTable .  '8'; #unitLocation
	my $snmpOidContact	= $snmpOidUnitTable .  '9'; #unitContact
	my $snmpOidAdmURL	= $snmpOidUnitTable . '10'; #unitAdminURL
	my @tableChecks = (
		$snmpOidClass, $snmpOidDesignation, $snmpOidModel, $snmpOidSerial, 
		$snmpOidLocation, $snmpOidContact, $snmpOidAdmURL,
	);
	if ($main::verboseTable == 231) { # UnitTable - needs a lot of time
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Unit Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidClass.(\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $class = $entries->{$snmpOidClass . '.' . $snmpID};
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $serial = $entries->{$snmpOidSerial . '.' . $snmpID};
			my $location = $entries->{$snmpOidLocation . '.' . $snmpID};
			my $contact = $entries->{$snmpOidContact . '.' . $snmpID};
			my $admURL = $entries->{$snmpOidAdmURL . '.' . $snmpID};
			$class = 0 if (!defined $class or $class <= 0);
			$class = 35 if ($class > 35);

			addStatusTopicToVerbose(undef, undef, $snmpID);
			addSerialIDsToVerbose($serial, undef);
			addKeyValueToVerbose("Class", $pqClassText[$class]);
			addNameToVerbose($designation);
			addLocationContactToVerbose($location, $contact);
			addAdminURLToVerbose($admURL);
			addProductModelToVerbose(undef, $model);
			$variableVerboseMessage .= "\n";
		}
	}
} #primequestUnitTable
sub primequestUnitTableChassisSerialNumber {
	my $snmpOidUnitTable = '.1.3.6.1.4.1.211.1.31.1.1.1.2.3.1.'; #mmb sysinfo unitInformation unitTable
	my $snmpOidSerial	= $snmpOidUnitTable .  '7'; #unitSerialNumber
	if ($optChkSystem) { 
		my $serial = trySNMPget($snmpOidSerial . '.1' ,"unitSerialNumber-Chassis");
		if (defined $serial) {
			$msg .= "ID=$serial";
		}
	}
} #primequestUnitTableChassisSerialNumber
sub primequestUnitTableChassis {
	my $snmpOidUnitTable = '.1.3.6.1.4.1.211.1.31.1.1.1.2.3.1.'; #mmb sysinfo unitInformation unitTable
	my $snmpOidDesignation	= $snmpOidUnitTable .  '4'; #unitDesignation
	my $snmpOidModel	= $snmpOidUnitTable .  '5'; #unitModelName
	my $snmpOidSerial	= $snmpOidUnitTable .  '7'; #unitSerialNumber
	my $snmpOidLocation	= $snmpOidUnitTable .  '8'; #unitLocation
	my $snmpOidContact	= $snmpOidUnitTable .  '9'; #unitContact
	my $snmpOidAdmURL	= $snmpOidUnitTable . '10'; #unitAdminURL
	{
		my $getInfos = 0;
		my $verbose = 0;
		my $notify = 0;
		$verbose = 1 if ($main::verbose >= 1);
		$notify = 1 
			if (!$main::verbose and !$main::verboseTable and $exitCode and $exitCode < 3);
		$getInfos = 1 if ($verbose or $notify);
		if ($getInfos) {
			my $designation = trySNMPget($snmpOidDesignation . '.1' ,"unitDesignation-Chassis");
			my $model = trySNMPget($snmpOidModel . '.1' ,"unitModelName-Chassis");
			my $location = trySNMPget($snmpOidLocation . '.1' ,"unitLocation-Chassis");
			my $contact = trySNMPget($snmpOidContact . '.1' ,"unitContact-Chassis");
			my $admURL = trySNMPget($snmpOidAdmURL . '.1' ,"unitAdminURL-Chassis");
			if ($verbose) {
				RFC1213sysinfoToLong();
				$longMessage .= "\n";
				addNameToLong($designation);
				addLocationContactToLong($location, $contact);
				addAdminURLToLong($admURL);
				addProductModelToLong(undef, $model);
				$longMessage .= "\n";
			} elsif ($notify) {
				RFC1213sysinfoToLong();
				$longMessage .= "\n";
				addNameToLong($designation);
				#addLocationContactToLong($location, $contact);
				addAdminURLToLong($admURL);
				addProductModelToLong(undef, $model);
				$longMessage .= "\n";
			}
		}
	} # opt System
} #primequestUnitTableChassis
sub primequestTemperatureSensorTable {
	my $snmpOidTemperatureSensorTable = '.1.3.6.1.4.1.211.1.31.1.1.1.5.1.1.'; #temperatureSensorTable (2)
	#my $snmpOidUnitId	= $snmpOidTemperatureSensorTable . '1'; #tempUnitId
	#	... UnitId is SB SystemBoard number
	my $snmpOidDesignation	= $snmpOidTemperatureSensorTable . '3'; #tempSensorDesignation
	my $snmpOidIndentifier	= $snmpOidTemperatureSensorTable . '4'; #tempSensorIdentifier
	my $snmpOidStatus	= $snmpOidTemperatureSensorTable . '5'; #tempSensorStatus
	my $snmpOidCurrent	= $snmpOidTemperatureSensorTable . '6'; #tempCurrentTemperature
	my $snmpOidWarning	= $snmpOidTemperatureSensorTable . '7'; #tempWarningLevel
	my $snmpOidCritical	= $snmpOidTemperatureSensorTable . '8'; #tempCriticalLevel
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidIndentifier, 
		$snmpOidStatus, $snmpOidCurrent, $snmpOidWarning, $snmpOidCritical, 
	);
	my @statusText = ( "none",
		"unknown", "not-available", "ok", undef , "failed",
		"temperature-warning", "temperature-critical", "..unexpected..",
	);
	if (($optChkEnvironment or $optChkEnv_Temp) and !$main::verboseTable) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Temperature Sensors");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		my $saveUnitId = undef;
		foreach my $snmpID (@snmpIDs) {
			my $triggerLongMessage = 0;
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $identifier = $entries->{$snmpOidIndentifier . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $current = $entries->{$snmpOidCurrent . '.' . $snmpID};
			my $warning = $entries->{$snmpOidWarning . '.' . $snmpID};
			my $critical = $entries->{$snmpOidCritical . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			$designation =~ s/_Temp\.//;
			my $name = $designation;
			$name = $identifier if ($identifier);
			$status = 0 if (!defined $status or $status < 0); # seen for Chassis Unit
			$status = 8 if ($status > 8 or $status == 4);

			if (($main::verbose < 2) 
			&&  ($pqTempStatus == 1 or $pqTempStatus == 2)
			&&  ($status >= 5)
			) { 
				$triggerLongMessage = 1;
			}
			addTemperatureToPerfdata($name, $current, $warning, $critical) 
				if (!$main::verboseTable);

			if (!$main::verboseTable) {
				addStatusTopicToVerbose($statusText[$status],
					"Sensor", $snmpID);
				addNameToVerbose($name);
				addCelsiusToVerbose($current, $warning, $critical);
				$variableVerboseMessage .= "\n";
			}
			if ($triggerLongMessage) { 
				addStatusTopicToLong($statusText[$status],
					"Sensor", $snmpID);
				addNameToLong($name);
				addCelsiusToLong($current, $warning, $critical);
				$longMessage .= "\n";
			}
		} # each sensor
	} # optChkEnvironment
} #primequestTemperatureSensorTable
sub primequestFanTable {
	my $snmpOidFanTable = '.1.3.6.1.4.1.211.1.31.1.1.1.5.2.1.'; #fanTable (2)
	my $snmpOidDesignation	= $snmpOidFanTable . '3'; #fanDesignation
	my $snmpOidIdentifier	= $snmpOidFanTable . '4'; #fanIdentifier
	my $snmpOidStatus	= $snmpOidFanTable . '5'; #fanStatus
	my $snmpOidSpeed	= $snmpOidFanTable . '6'; #fanCurrentSpeed
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidIdentifier, 
		$snmpOidStatus, $snmpOidSpeed, 
	);
	my @statusText = ( "none",
		"unknown", "disabled", "ok", "failed", "prefailed-predicted",
		"redundant-fan-failed", "not-manageable", "not-present", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (!$main::verbose and !$main::verboseTable and $pqFanStatus and $pqFanStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkEnvironment or $optChkEnv_Fan) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Fans") if ($verbose);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $identifier = $entries->{$snmpOidIdentifier . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $speed = $entries->{$snmpOidSpeed . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			my $name = $designation;
			$name = $identifier if ($identifier);
			$status = 0 if (!defined $status or $status < 0);
			$status = 9 if ($status > 9);

			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status], "Fan", $snmpID =~ m/(\d+)\.\d+/);
				addNameToVerbose($name); 
				addKeyRpmToVerbose("Speed", $speed);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and  ($status == 2 or $status == 4 or $status == 5 or $status == 6)
			) { 
				addStatusTopicToLong($statusText[$status], "Fan", $snmpID =~ m/(\d+)\.\d+/);
				addNameToLong($name);
				addKeyRpmToLong("Speed", $speed);
				$longMessage .= "\n";
			}
		} # each fan
	} #optChkEnvironment and verbose or error
} #primequestFanTable
sub primequestPowerSupplyTable {
	my $snmpOidPowerSupplyTable = '.1.3.6.1.4.1.211.1.31.1.1.1.6.2.1.'; #powerSupplyTable
	my $snmpOidDesignation	= $snmpOidPowerSupplyTable . '3'; #powerSupplyDesignation
	my $snmpOidIdentifier	= $snmpOidPowerSupplyTable . '4'; #powerSupplyIdentifier
	my $snmpOidStatus	= $snmpOidPowerSupplyTable . '5'; #powerSupplyStatus
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidIdentifier, 
		$snmpOidStatus,
	);
	my @statusText = ( "none",
		"unknown", "not-present", "ok", "failed", "ac-fail", 
		"dc-fail", "critical-temperature", "not-manageable", "predictive-fail", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (!$main::verbose and !$main::verboseTable 
		and $pqPowerSupplyStatus and $pqPowerSupplyStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if ($optChkPower and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Power Supply Unit") if ($verbose);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $identifier = $entries->{$snmpOidIdentifier . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			my $name = $designation;
			$name = $identifier if ($identifier);
			$status = 0 if (!defined $status or $status < 0);
			$status = 10 if ($status > 10);

			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status], "PSU", $snmpID =~ m/\d+\.(\d+)/);
				addNameToVerbose($name);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and  ($status == 4 or $status == 5 or $status == 6 or $status == 7 or $status == 9)
			) { 
				addStatusTopicToLong($statusText[$status], "PSU", $snmpID =~ m/\d+\.(\d+)/);
				addNameToLong($name);
				$longMessage .= "\n";
			}
		} # each power supply
	} #optChkPower and verbose or error
} #primequestPowerSupplyTable
sub primequestPowerMonitoringTable {
	# --	powerMonitoringTable :	.1.3.6.1.4.1.211.1.31.1.1.1.4.5	 - perf
	my $snmpOidPowerMonitoringTable = '.1.3.6.1.4.1.211.1.31.1.1.1.4.5.1.'; #powerMonitoringTable
	my $snmpOidMax		= $snmpOidPowerMonitoringTable . '4.1'; #pmNominalPowerConsumption
	my $snmpOidCurrent	= $snmpOidPowerMonitoringTable . '5.1'; #pmCurrentPowerConsumption
	if ($optChkPower and !$main::verboseTable) {
		my $current = trySNMPget($snmpOidCurrent,"pmCurrentPowerConsumption");
		my $max = trySNMPget($snmpOidMax,"pmNominalPowerConsumption");

		#### TODO - enable -w, -c for PRIMEQUEST PowerConsumption if $max is defined
		if (defined $current and $current != -1 and !$main::verboseTable) {
			addPowerConsumptionToPerfdata($current, undef,undef, undef,$max);
		}
	} #optChkPower
} #primequestPowerMonitoringTable
sub primequestVoltageTable {
	# (1/100V)
	my $snmpOidVoltageTable = '.1.3.6.1.4.1.211.1.31.1.1.1.6.3.1.'; #voltageTable
	my $snmpOidDesignation	= $snmpOidVoltageTable . '3'; #voltageDesignation
	my $snmpOidStatus	= $snmpOidVoltageTable . '4'; #voltageStatus
	my $snmpOidCurrent	= $snmpOidVoltageTable . '5'; #voltageCurrentValue
	my $snmpOidMin		= $snmpOidVoltageTable . '7'; #voltageMinimumLevel
	my $snmpOidMax		= $snmpOidVoltageTable . '8'; #voltageMaximumLevel
	my @tableChecks = (
		$snmpOidDesignation, 
		$snmpOidStatus, $snmpOidCurrent, $snmpOidMin, $snmpOidMax,
	);
	my @statusText = ( "none", 
		"unknown", "not-available", "ok", "too-low", "too-high",
		"sensor-failed", "low-warning", "high-warning", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verboseTable == 631); # PQ VoltageTable
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (!$main::verbose and !$main::verboseTable and $pqVoltageStatus and $pqVoltageStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkHardware or $optChkVoltage) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Voltage Unit")	if ($verbose);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $current = $entries->{$snmpOidCurrent . '.' . $snmpID};
			my $min = $entries->{$snmpOidMin . '.' . $snmpID};
			my $max = $entries->{$snmpOidMax . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			my $name = $designation;
			$status = 0 if (!defined $status or $status < 0);
			$status = 9 if ($status > 9);

			#### TODO/QUESTION PRIMEQUEST Voltage and Performance -w -c ?

			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status], "Voltage", $snmpID);
				addNameToVerbose($name);
				addmVoltToVerbose($current,undef,undef,$min,$max);
				$variableVerboseMessage .= "\n";
			} 
			elsif ($notify and $status >= 4) { 
				addStatusTopicToLong($statusText[$status], "Voltage", $snmpID);
				addNameToLong($name);
				addmVoltToLong($current,undef,undef,$min,$max);
				$longMessage .= "\n";
			}
		} # each power supply
	} #optChkHardware - verbose or error
} #primequestVoltageTable
sub primequestCpuTable {
	my $snmpOidCpuTable = '.1.3.6.1.4.1.211.1.31.1.1.1.6.4.1.'; #cpuTable
	my $snmpOidDesignation	= $snmpOidCpuTable . '3'; #cpuDesignation
	my $snmpOidStatus	= $snmpOidCpuTable . '4'; #cpuStatus
	my $snmpOidModel	= $snmpOidCpuTable . '5'; #cpuModelName
	my $snmpOidSpeed	= $snmpOidCpuTable . '8'; #cpuCurrentSpeed
	my @tableChecks = (
		$snmpOidDesignation, 
		$snmpOidStatus, $snmpOidModel, $snmpOidSpeed,
	);
	my @statusText = ( "none",
		"unknown", "not-present", "ok", "disabled", "error",
		"failed", "missing-termination", "prefailed-warning", "..unexpected..",
	);

	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verboseTable == 641); # PQ CPUTable
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (!$main::verbose and !$main::verboseTable and $pqCPUStatus and $pqCPUStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkHardware or $optChkCPU) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("CPU Table") if ($verbose);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $speed = $entries->{$snmpOidSpeed . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			my $name = $designation;
			$status = 0 if (!defined $status or $status < 0);
			$status = 9 if ($status > 9);

			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status],"CPU",$snmpID);
				addNameToVerbose($name);
				addProductModelToVerbose(undef, $model); 
				addKeyMHzToVerbose("Speed", $speed);
				$variableVerboseMessage .= "\n";
			} elsif ($notify and $status >= 4) { 
				addStatusTopicToLong($statusText[$status],"CPU",$snmpID);
				addNameToLong($name);
				addProductModelToLong(undef, $model);
				addKeyMHzToLong("Speed", $speed);
				$longMessage .= "\n";
			}
		} # each
	} # hardware 
} #primequestCpuTable
sub primequestMemoryModuleTable {
	my $snmpOidMemoryModuleTable = '.1.3.6.1.4.1.211.1.31.1.1.1.6.5.1.'; #memoryModuleTable
	my $snmpOidDesignation	= $snmpOidMemoryModuleTable . '3'; #memModuleDesignation
	my $snmpOidStatus	= $snmpOidMemoryModuleTable . '4'; #memModuleStatus
	my $snmpOidCapacity	= $snmpOidMemoryModuleTable . '6'; #memModuleCapacity MB
	my $snmpOidConfig	= $snmpOidMemoryModuleTable . '13'; #memModuleConfiguration
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidStatus, 
		$snmpOidCapacity, $snmpOidConfig,
	);
	my @statusText = ( "none",
		"unknown", "not-present", "ok", "failed-disabled", "error",
		"..unexpected..", "warning", "hot-spare", "configuration-error", "..unexpected..",
	);
	my @configText = ( "none",
		"unknown", "normal", "disabled", "hotSpare", "mirror",
		"raid", "notUsable", "..unexpected..",
	); 
	if ($main::verboseTable == 651) { # PQ - MemoryModuleTable - needs toooooo much time
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Memory Module Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $capacity = $entries->{$snmpOidCapacity . '.' . $snmpID};
			my $config = $entries->{$snmpOidConfig . '.' . $snmpID};
			$designation =~ s/[ ,;=]/_/g;
			my $name = $designation;
			$status = 0 if (!defined $status or $status < 0);
			$status = 10 if ($status > 10);
			$config = 0 if (!defined $config or $config < 0);
			$config = 8 if ($config > 8);

			if ($status != 2 or $main::verbose >= 3) { # not "not-present"
				addStatusTopicToVerbose($statusText[$status],"MemMod", $snmpID);
				addNameToVerbose($name);
				addKeyValueToVerbose("Config", $configText[$config]) if ($config);
				addKeyMBToVerbose("Capacity", $capacity);
				$variableVerboseMessage .= "\n";
			}
		} # each
	} # hardware and this table
} #primequestMemoryModuleTable
sub primequestPerformanceTable {
	# in the test there was no data inside
	my $snmpOidPerformanceTable = '.1.3.6.1.4.1.211.1.31.1.1.1.4.3.1.'; #performanceTable (2)
	my $snmpOidType		= $snmpOidPerformanceTable . '3'; #perfType
	my $snmpOidValue	= $snmpOidPerformanceTable . '4'; #performanceValue
	my $snmpOidName		= $snmpOidPerformanceTable . '5'; #performanceName
	my @tableChecks = (
		$snmpOidType, $snmpOidValue, $snmpOidName,  
	);
	my @typeText = ( "none",
		"cpu", "cpu-overall", "pci-load", "pci-efficiency", "pci-transfer",
		"memory-physical", "memory-total", "memory-percent", "..unexpected..",
	);
        #            cpu:             Load of a single CPU in percent
        #            cpu-overall:     Overall CPU load in percent
        #            pci-load:        PCI bus load in percent
        #            pci-efficiency:  PCI bus efficiency in percent (100% is optimum)
        #            pci-transfer:    PCI bus transfer rate in MBytes/sec.
        #            memory-physical: Physical memory usage in MBytes
        #            memory-total:    Total memory usage (physical + virtual) in MBytes
        #            memory-percent:  Physical memory usage in percent"
	my $getInfos = 0;
	$getInfos = 1 if ($main::verbose >= 3 and $optChkHardware and !$main::verboseTable);
	$getInfos = 1 if ($optChkCpuLoadPerformance);
	$getInfos = 1 if ($optChkMemoryPerformance);
	my $totalCPU = undef;
	#my @cpu = (0,0,0,0,0, 0,0,0,0,0, -1,);
	my $virtualMemMBytes = undef;
	my $physMemPercent = undef;
	my $physMemMBytes = undef;
	if ($getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidType.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		$variableVerboseMessage .= "Hardware Performance Values:\n"
			if ($main::verbose >= 3 and $optChkHardware and !$main::verboseTable);
		my $maxInd = 0;
		foreach my $id (@snmpIdx) {
 			my $type  = $entries->{$snmpOidType . '.' . $id};
 			my $name = $entries->{$snmpOidName . '.' . $id};
			my $value = $entries->{$snmpOidValue . '.' . $id};
			$type = 0 if (!defined $type or $type < 0);
			$type = 9 if ($type > 9);
			if ($main::verbose >= 3 and !$main::verboseTable) {
				addStatusTopicToVerbose(undef,$typeText[$type],undef);
				addNameToVerbose($name);
				addKeyValueToVerbose("Value", $value);
				if ($type and $type >= 5 and $type <= 7) {
					$variableVerboseMessage .= "MB";
				} else {
					$variableVerboseMessage .= "%";
				}
				$variableVerboseMessage .= "\n";
			} 
			if ($optChkCpuLoadPerformance and defined $value) {
				$totalCPU = $value if ($type == 2);
				#$cpu[$objNr]=$value if ($type == 1);
				#$maxInd = $objNr if ($objNr > $maxInd);
			}
			if ($optChkMemoryPerformance and defined $value) {
				$physMemMBytes = $value if ($type == 6);
				$virtualMemMBytes = $value if ($type == 7 and ($name =~ m/virtual/));
				$physMemPercent = $value if ($type == 8);
			}
		} # each
		#if ($maxInd) {
		#	$maxInd++;
		#	$cpu[$maxInd] = -1;
		#}
	} #verbose
	if ($optChkCpuLoadPerformance) {
		$msg			.= "Total=$totalCPU" . "%" if ($totalCPU);
		addPercentageToPerfdata("Total", $totalCPU, undef, undef) if (!$main::verboseTable);
		#my $i = 0;
		#for ($i=0; $cpu[$i] != -1;$i++) {
		#	$msg			.= " CPU[$i]=$cpu[$i]" . "%" if (defined $cpu[$i]);
		#	addPercentageToPerfdata("CPU[$i]", $cpu[$i], undef, undef);
		#} #each
		$exitCode = 0;
	} #cpu load
	if ($optChkMemoryPerformance) {
		my $warn = ($optWarningLimit?$optWarningLimit:0);
		my $crit = ($optCriticalLimit?$optCriticalLimit:0);
		$warn = undef if ($warn == 0);
		$crit = undef if ($crit == 0);
		$msg .= "Physical-Memory=$physMemPercent" . "% " if ($physMemPercent);
		$msg .= "Physical-Memory=$physMemMBytes" . "MB " if ($physMemMBytes);
		$msg .= "Virtual-Memory=$virtualMemMBytes" . "MB " if ($virtualMemMBytes);
		addPercentageToPerfdata("Physical-Memory", $physMemPercent, $warn, $crit) 
			if (!$main::verboseTable);
		$exitCode = 0;
		$exitCode = 1 if ($warn and $physMemPercent > $warn);
		$exitCode = 2 if ($crit and $physMemPercent > $crit);
	} #memory
} #primequestPerformanceTable
sub primequestManagementNodeTable {
	my $snmpOidManagementNodeTable = '.1.3.6.1.4.1.211.1.31.1.1.1.3.1.1.'; #managementNodeTable
	my $snmpOidAddress	= $snmpOidManagementNodeTable . '4'; #unitNodeAddress
	my $snmpOidName		= $snmpOidManagementNodeTable . '7'; #unitNodeName
	my $snmpOidClass	= $snmpOidManagementNodeTable . '8'; #unitNodeClass
	my $snmpOidMAC		= $snmpOidManagementNodeTable . '9'; #unitNodeMacAddress
	my @tableChecks = (
		$snmpOidAddress, $snmpOidName, 
		$snmpOidClass, $snmpOidMAC,
	);
	my @classText = ( "none",
		"unknown", "primery", "secondary", "management-blade", "secondary-remote",
		"secondary-remote-backup", "baseboard-controller", "secondary-management-blade", "..unknown..",
	);
	if ($main::verbose == 3 or $main::verboseTable == 311) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Management Node Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidAddress.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $address = $entries->{$snmpOidAddress . '.' . $snmpID};
			my $name = $entries->{$snmpOidName . '.' . $snmpID};
			my $class = $entries->{$snmpOidClass . '.' . $snmpID};
			my $mac = $entries->{$snmpOidMAC . '.' . $snmpID};
			$class = 0 if (!defined $class or $class < 0);
			$class = 9 if ($class > 9);
			{ 
				addStatusTopicToVerbose(undef, "Node", $snmpID);
				addNameToVerbose($name);
				addKeyValueToVerbose("Address", $address);
				addKeyValueToVerbose("MAC", $mac);
				addKeyValueToVerbose("Class", $classText[$class]) if ($class);
				$variableVerboseMessage .= "\n";
			}
		} # each
	} # system and specific table
} #primequestManagementNodeTable
sub primequestServerTable {
	my $snmpOidServerTable = '.1.3.6.1.4.1.211.1.31.1.1.1.4.1.1.'; #serverTable
	my $snmpUnitID		= $snmpOidServerTable . '1'; #srvUnitId
	my $snmpOidBootStatus	= $snmpOidServerTable . '4'; #srvCurrentBootStatus
	my $snmpOidBootUUID	= $snmpOidServerTable . '7'; #srvUUID
	my $snmpOidManIP	= $snmpOidServerTable . '10'; #srvManagementIP
	my @tableChecks = (
		$snmpUnitID, $snmpOidBootStatus, $snmpOidBootUUID, 
		$snmpOidManIP,
	);
	my @bootText = ( "none",
		"unknown", "off", "no-boot-cpu", "self-test", "setup",
		"os-boot", "diagnostic-boot", "os-running", "diagnostic-running", "os-shutdown",
		"diagnostic-shutdown", "reset", "panic", "check-stop", "dumping", 
		"halt", "..unexpected..",
	);
	if ($main::verbose == 3 or $main::verboseTable == 411) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Server Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidBootStatus.(\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $unitid = $entries->{$snmpUnitID . '.' . $snmpID};
			my $bstatus = $entries->{$snmpOidBootStatus . '.' . $snmpID};
			my $uuid = $entries->{$snmpOidBootUUID . '.' . $snmpID};
			my $mip = $entries->{$snmpOidManIP . '.' . $snmpID};
			$bstatus = 0 if (!defined $bstatus or $bstatus < 0);
			$bstatus = 17 if ($bstatus > 17);
			{ 
				addStatusTopicToVerbose(undef, "Server", $snmpID);
				addKeyIntValueToVerbose("Partition",($unitid-2));
				addKeyValueToVerbose("BootStatus",$bootText[$bstatus]) if ($bstatus);
				addKeyValueToVerbose("ManagementIP",$mip);
				addKeyValueToVerbose("UUID",$uuid);
				$variableVerboseMessage .= "\n";
			}
		} # each
	}
} #primequestServerTable
sub primequest {
	# MMB-COM-MIB.mib
	my $snmpOidPrimeQuestMMBSysInfo = '.1.3.6.1.4.1.211.1.31.1.1.1.'; # primequest - mmb - sysinfo
	my $snmpOidUnitInfo = $snmpOidPrimeQuestMMBSysInfo . '2.'; #unitInformation
	my $snmpOidLocalID	= $snmpOidUnitInfo . '1.0'; #localServerUnitId
	#my $snmpOidNumberUnits	= $snmpOidUnitInfo . '2.0'; #numberUnits

	my $snmpOidOverallStatusArea = $snmpOidPrimeQuestMMBSysInfo . '8.'; #status
	my $snmpOidStatus = $snmpOidOverallStatusArea . '1.0'; #agentStatus
	my %commonSystemCodeMap = (	1       =>      0,
					2       =>      1,
					3       =>      2,
					4       =>      2,
					5	=>	3,);
	my @overallStatusText = ( "none", "ok", "degraded", "error", "failed", "unknown",);

	my $localID = mibTestSNMPget($snmpOidLocalID,"PRIMEQUEST");
	my $allStatus = simpleSNMPget($snmpOidStatus,"agentStatus");
	$allStatus = 0 if (!defined $allStatus or $allStatus < 0 or $allStatus > 5);
	$exitCode = $commonSystemCodeMap{$allStatus};

	primequestUnitTableChassisSerialNumber();
	$msg .= " - All=$overallStatusText[$allStatus]" if ($optChkSystem);
	primequestStatusComponentTable();

	$exitCode = $pqTempStatus if ($optChkEnv_Temp and !defined $optChkEnv_Fan);
	$exitCode = $pqFanStatus if ($optChkEnv_Fan and !defined $optChkEnv_Temp);
	addExitCode($pqFanStatus) if ($optChkEnv_Temp and $optChkEnv_Fan);
	if ((defined $optChkEnvironment) && (!defined $optChkPower) && (!defined $optChkSystem)) {
		$exitCode = 3 if (!defined $pqTempStatus);
		$exitCode = $pqTempStatus if (defined $pqTempStatus);
		addExitCode($pqFanStatus);
	}
	if ((defined $optChkPower) && (!defined $optChkEnvironment) && (!defined $optChkSystem)) { 
		$exitCode = 3 if (!defined $pqPowerSupplyStatus);
		$exitCode = $pqPowerSupplyStatus if (defined $pqPowerSupplyStatus);
	}
	$exitCode = $pqCPUStatus if ($optChkCPU and !defined $optChkVoltage and !defined $optChkMemMod);
	$exitCode = $pqVoltageStatus if ($optChkVoltage and !defined $optChkCPU and !defined $optChkMemMod);
	$exitCode = $pqMemoryStatus if ($optChkMemMod and !defined $optChkCPU and !defined $optChkVoltage);
	addExitCode($pqVoltageStatus) if ($optChkCPU and $optChkVoltage);
	addExitCode($pqMemoryStatus) if ($optChkMemMod and ($optChkCPU or $optChkVoltage));
	if ((defined $optChkHardware) 
	&& (!defined $optChkSystem) && (!defined $optChkEnvironment) && (!defined $optChkPower)) {
		$exitCode = 3 if (!defined $pqVoltageStatus);
		$exitCode = $pqVoltageStatus if (defined $pqVoltageStatus);
		addExitCode($pqCPUStatus);
		addExitCode($pqMemoryStatus);
	}
	if ((defined $optChkSystem) && (!defined $optChkEnvironment) && (!defined $optChkPower)) {
		$exitCode = $commonSystemCodeMap{$allStatus};
	}

	primequestUnitTableChassis(); # search System information like Contact ...
	primequestUnitTable();
	primequestUnitParentTable();
	primequestUnitChildTable();
	primequestManagementNodeTable();
	primequestServerTable();
	primequestPSACOM();

	primequestTemperatureSensorTable();
	primequestFanTable();

	primequestPowerSupplyTable();
	primequestPowerMonitoringTable();

	primequestVoltageTable();
	primequestCpuTable();
	primequestMemoryModuleTable();
	primequestPerformanceTable();
} #primequest
#----------- primergy management blade functions
our $overallFan = 0;
our $overallTemp = 0;
our $overallPS = 0;
our @s31OverallStatusText = ( "none",
	"unknown", "ok", "degraded", "critical",
);
sub primergyManagementBladePowerSupply {
	my %bladeErrorCodeMap = (	0 => 3,
					1 => 3,
					2 => 0,
					3 => 1,
					4 => 2);
	my $snmpOidBladePSOverall	= '.1.3.6.1.4.1.7244.1.1.1.3.2.1.0'; #s31SysPowerSupplyStatus.0
	my $snmpOidBladePSTable		= '.1.3.6.1.4.1.7244.1.1.1.3.2.4.1.';#s31SysPowerSupplyUnitTable
	my $snmpOidStatus		= $snmpOidBladePSTable . '2'; #s31SysPowerSupplyUnitStatus
	my $snmpOidProduct		= $snmpOidBladePSTable . '4'; #s31SysPowerSupplyUnitProductName
	my $snmpOidModel		= $snmpOidBladePSTable . '5'; #s31SysPowerSupplyUnitModelName
	my $snmpOidSerial		= $snmpOidBladePSTable . '7'; #s31SysPowerSupplyUnitSerialNumber
	my @bladePSChecks = (
		$snmpOidStatus,
		$snmpOidProduct,
		$snmpOidModel,
		$snmpOidSerial,
	);
	my @bladePSStatusText = ( "none", 
		"unknown", "ok", "not-present", "warning", "critical",
		"off", "dummy", "fanmodule", "..unexpected..",
	);	
	if (defined $optChkPower) { 
		my $oPS = simpleSNMPget($snmpOidBladePSOverall,"PowerSupplyOverall");
		$oPS = 0 if (!defined $oPS or $oPS < 0 or $oPS > 4);
		$overallPS = $bladeErrorCodeMap{$oPS};
		$msg .= ' - PowerSupply(' . $s31OverallStatusText[$oPS] . ')';

		if ((!defined $optChkEnvironment) && (!defined $optChkSystem)) {
		    $exitCode = $overallPS;
		}
		my $getinfos = 0;
		my $verbose = 0;
		my $notify = 0;
		$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
		$notify = 1 if (!$main::verbose and !$main::verboseTable and $overallPS and $overallPS < 3);
		$getinfos = 1 if ($verbose or $notify);
		if ($getinfos) {
			my $entries = $main::session->get_entries( -columns => \@bladePSChecks );
			my @snmpIDs = ();

			foreach my $snmpKey ( keys %{$entries} ) {
				#print "$snmpKey --- $entries->{$snmpKey}\n";
				push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
			}
			
			@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
			addTableHeaderToVerbose("Power Supply Unit") if ($verbose);
			foreach my $snmpID (@snmpIDs) {
				my $psStatus = $entries->{$snmpOidStatus . '.' . $snmpID};
				my $psSerial = $entries->{$snmpOidSerial . '.' . $snmpID};
				my $psProduct = $entries->{$snmpOidProduct . '.' . $snmpID};
				my $psModel = $entries->{$snmpOidModel . '.' . $snmpID};
				$psStatus = 0 if (!defined $psStatus or $psStatus < 0);
				$psStatus = 9 if ($psStatus > 9);
				if ($verbose) {
					addStatusTopicToVerbose($bladePSStatusText[$psStatus],
						"PowerSupplyUnit", $snmpID);
					addSerialIDsToVerbose($psSerial, undef);
					addProductModelToVerbose($psProduct, $psModel);
					endVariableVerboseMessageLine();
				}
				elsif ($notify and ($psStatus == 4) or ($psStatus == 5)) {
					addStatusTopicToLong($bladePSStatusText[$psStatus],
						"PowerSupplyUnit", $snmpID);
					addSerialIDsToLong($psSerial, undef);
					addProductModelToLong($psProduct, $psModel);
					endLongMessageLine();
				}
			} # all ids
		} # verbose or error
	}
	chomp($msg);
} #primergyManagementBladePowerSupply
sub primergyManagementBladeEnvironment {
	my %bladeErrorCodeMap = (	0 => 3,
					1 => 3,
					2 => 0,
					3 => 1,
					4 => 2);
	my $snmpOidBladeFanOverall = '.1.3.6.1.4.1.7244.1.1.1.3.3.4.0'; #s31SysFanOverallStatus.0
	my $snmpOidBladeTempOverall = '.1.3.6.1.4.1.7244.1.1.1.3.4.2.0'; #s31SysTemperatureStatus.0

	my $snmpOidBladeFanPrefix = '.1.3.6.1.4.1.7244.1.1.1.3.3.1.1.'; #s31SysFanTable
	my $snmpOidBladeFanStatus = $snmpOidBladeFanPrefix . '2'; #s31SysFanStatus
	my $snmpOidBladeFanDesc = $snmpOidBladeFanPrefix . '3'; #s31SysFanDesignation - verbose
	my $snmpOidBladeFanSpeed = $snmpOidBladeFanPrefix . '4'; #s31SysFanCurrentSpeed
	my $snmpOidBladeFanIdent = $snmpOidBladeFanPrefix . '8'; #s31SysFanIdentification
        my @bladeFanChecks = (
		$snmpOidBladeFanStatus,
		$snmpOidBladeFanSpeed,
		$snmpOidBladeFanDesc
	);
	my @bladeFanStatus = ( "none",
		"unknown", "disabled", "ok", "fail", "prefailure-predicted",
		"redundant-fan-failed", "not-manageable", "not-present", "not-available", "..unexpected..",
	);

	my $snmpOidBladeTempPrefix = '.1.3.6.1.4.1.7244.1.1.1.3.4.1.1.'; #s31SysTemperatureSensorTable
	my $snmpOidBladeTempStatus = $snmpOidBladeTempPrefix . '2'; #s31SysTempSensorStatus
	my $snmpOidBladeTempWarn = $snmpOidBladeTempPrefix . '4'; #s31SysTempUpperWarningLevel
	my $snmpOidBladeTempCrit = $snmpOidBladeTempPrefix . '5'; #s31SysTempUpperCriticalLevel
	my $snmpOidBladeTempValue = $snmpOidBladeTempPrefix . '6'; #s31SysTempCurrentValue
	my $snmpOidBladeTempDesc = $snmpOidBladeTempPrefix . '3'; #s31SysTempSensorDesignation
	my @bladeTempChecks = (
		$snmpOidBladeTempStatus,
		$snmpOidBladeTempValue,
		$snmpOidBladeTempWarn,
		$snmpOidBladeTempCrit,
		$snmpOidBladeTempDesc
	);
	my @bladeTempStatus = ( "none",
		"unknown", "disable", "ok", "failed", "warning", 
		"critical", "not-available", "..unexpected..",
	);
	# get subsystem information
	if (defined $optChkEnvironment or $optChkEnv_Fan or $optChkEnv_Temp) { 
		if (!defined $optChkEnv_Temp) {
			my $oFan = simpleSNMPget($snmpOidBladeFanOverall,"FanOverall");
			$oFan = 0 if (!defined $oFan or $oFan < 0 or $oFan > 4);
			$overallFan = $bladeErrorCodeMap{$oFan};
			$msg .= ' - Fans(' . $s31OverallStatusText[$oFan] . ')';
		}
		if (!defined $optChkEnv_Fan) {
			my $oTemp = simpleSNMPget($snmpOidBladeTempOverall,"TemperaturOverall");
			$oTemp = 0 if (!defined $oTemp or $oTemp < 0 or $oTemp > 4);
			$overallTemp = $bladeErrorCodeMap{$oTemp};
			$msg .= ' - Temperature(' . $s31OverallStatusText[$oTemp] . ')';
		}

		if ($optChkEnvironment and (!defined $optChkPower) and (!defined $optChkSystem)) {
		    $exitCode = 3;
		    addExitCode($overallFan);
		    addExitCode($overallTemp);
		}
		addExitCode($overallFan) if ($optChkEnv_Fan and (!defined $optChkSystem));
		addExitCode($overallTemp) if ($optChkEnv_Temp and (!defined $optChkSystem));
	}
	chomp($msg);
	# process fan information
	my $getinfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (!$main::verbose and !$main::verboseTable and $overallFan and $overallFan < 3);
	$getinfos = 1 if ($verbose or $notify);
	if ((defined $optChkEnvironment or $optChkEnv_Fan) and $getinfos) { 
		my @snmpIDs = ();
		my $entries = $main::session->get_entries( -columns => \@bladeFanChecks );
		
		foreach my $snmpkey ( keys %{$entries} ) {
			push(@snmpIDs, $1) if ($snmpkey =~ m/$snmpOidBladeFanStatus.(\d+)/);
		}
		
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		addTableHeaderToVerbose("Fans") if ($verbose);
		foreach my $fanID (@snmpIDs) {
			my $fanStatus  = $entries->{$snmpOidBladeFanStatus . '.' . $fanID};
			$fanStatus = 0 if (!defined $fanStatus or $fanStatus < 0);
			$fanStatus = 10 if ($fanStatus > 10);
			next if (($fanStatus eq '9' or $fanStatus eq '2' or $fanStatus eq '8' or $fanStatus eq '7') and $main::verbose < 3);
			my $fanDesc = $entries->{$snmpOidBladeFanDesc . '.' . $fanID};
			my $fanSpeed = $entries->{$snmpOidBladeFanSpeed . '.' . $fanID};
			my $fanIdent = trySNMPget($snmpOidBladeFanIdent . '.' . $fanID ,
				"FanIdentification");
			$fanDesc =~ s/[ ,;=]/_/g;
			if (! defined $fanIdent) {
				$fanIdent = $fanDesc;
			}
			if ($verbose) {
				addStatusTopicToVerbose($bladeFanStatus[$fanStatus], "Fan", $fanID);
				addNameToVerbose($fanIdent);
				addKeyRpmToVerbose("Speed", $fanSpeed);
				$variableVerboseMessage .= "\n";
			} elsif ($notify and $fanStatus >= 4 and $fanStatus <= 6) {
				addStatusTopicToLong($bladeFanStatus[$fanStatus], "Fan", $fanID);
				addNameToLong($fanIdent);
				addKeyRpmToLong("Speed", $fanSpeed);
				endLongMessageLine();
			}
		} #each
	} # fan

	# process temperature information
	$notify = 0;
	$notify = 1 if (!$main::verbose and !$main::verboseTable and $overallTemp and $overallTemp < 3);
	if (($optChkEnvironment or $optChkEnv_Temp) and !$main::verboseTable) { 
		my @snmpIDs = ();
		my $entries = $main::session->get_entries( -columns => \@bladeTempChecks );
		foreach my $snmpkey ( keys %{$entries} ) {
			push(@snmpIDs, $1) if ($snmpkey =~ m/$snmpOidBladeTempStatus.(\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		addTableHeaderToVerbose("Temperature Sensors");
		foreach my $tempID (@snmpIDs) {	
			my $tempStatus  = $entries->{$snmpOidBladeTempStatus . '.' . $tempID};
				# ... TemperaturSensorStatus !
			$tempStatus = 0 if (!defined $tempStatus or $tempStatus < 0);
			$tempStatus = 8 if ($tempStatus > 8);
			next if (($tempStatus eq '7' or $tempStatus eq '2') and $main::verbose < 3); 
			# skip if disabled or not available
			my $tempValue = $entries->{$snmpOidBladeTempValue . '.' . $tempID};
			my $tempWarn = $entries->{$snmpOidBladeTempWarn . '.' . $tempID};
			my $tempCrit = $entries->{$snmpOidBladeTempCrit . '.' . $tempID};
			my $tempDesc = $entries->{$snmpOidBladeTempDesc . '.' . $tempID};
			$tempDesc =~ s/[ ,;=]/_/g;
			
			addTemperatureToPerfdata($tempDesc, $tempValue, $tempWarn, $tempCrit)
				if (!$main::verboseTable);

			if ($verbose) {
				addStatusTopicToVerbose($bladeTempStatus[$tempStatus], "Sensor", $tempID);
				addNameToVerbose($tempDesc);
				addCelsiusToVerbose($tempValue, $tempWarn, $tempCrit);
				$variableVerboseMessage .= "\n";
			}
			if ($tempCrit > 0  and $tempValue >= $tempCrit) {
				$exitCode = 2;
			} elsif ($tempWarn > 0 and $tempValue >= $tempWarn) {
				$exitCode = 1 if ($exitCode != 2);
			} 
			if ($notify and ($tempStatus >=4 and $tempStatus <=6)) {
				addStatusTopicToLong($bladeTempStatus[$tempStatus], "Sensor", $tempID);
				addNameToLong($tempDesc);
				addCelsiusToLong($tempValue, $tempWarn, $tempCrit);
				endLongMessageLine();
			}
		} #each temp
	} # optChkEnvironment
} #primergyManagementBladeEnvironment
sub primergyManagementBladePowerConsumption {
	# ATTENTION - There are three index part in this table - one is a timestamp !
	my $snmpOidAgentDateTime	= '.1.3.6.1.4.1.7244.1.1.1.1.7.0'; #s31AgentDateTime
	my $snmpOidBladePowerConsumptionTable = '.1.3.6.1.4.1.7244.1.1.1.15.1.1.'; #s31UtilizationHistoryTable
	#my $snmpOidPowerUnitId		= $snmpOidBladePowerConsumptionTable . '1'; #s31uthUnitId
	#my $snmpOidPowerEntity		= $snmpOidBladePowerConsumptionTable . '2'; #s31uthEntity
	#my $snmpOidPowerTimeStamp	= $snmpOidBladePowerConsumptionTable . '3'; #s31uthTimeStamp
	#my $snmpOidPowerHwUUID		= $snmpOidBladePowerConsumptionTable . '4'; #s31uthHardwareUUID
	#	.1.3.6.1.4.1.7244.1.1.1.15.1.1.4.1.2.1110717600 --- XXXXXXXX
	my $snmpOidPowerAverage		= $snmpOidBladePowerConsumptionTable . '5'; #s31uthAverageValue
	my $snmpOidPowerMin		= $snmpOidBladePowerConsumptionTable . '6'; #s31uthMinValue
	my $snmpOidPowerMax		= $snmpOidBladePowerConsumptionTable . '7'; #s31uthMaxValue
	# Sample
	# .1.3.6.1.4.1.7244.1.1.1.15.1.1.3.1.1.1111363200 --- 1111363200

	my @bladePowerChecks = ( #verbose
		$snmpOidPowerAverage,
	); 
	# get power consumption if enabled
	if (defined $optChkPower) { 
		my $chkAgentDateTime = trySNMPget($snmpOidAgentDateTime, "AgentDateTime");
		my $timeStamp = 0;
		my $nodeTimeStamp = 0;
		my $localTimeStamp = time();
		$timeStamp = $localTimeStamp;
		if (defined $chkAgentDateTime) {
			$chkAgentDateTime =~ m|(..)/(..)/(....) (..):(..):(..).*|;
			my $tmon = $1; 
			my $tday = $2; 
			my $tyear = $3;
			my $thour = $4; 
			my $tmin = $5; 
			my $tsec = $6;
			#print "****** $chkAgentDateTime => $tsec,$tmin,$thour,$tday,$tmon,$tyear \n";
			$nodeTimeStamp = timelocal($tsec,$tmin,$thour,$tday,$tmon-1,$tyear);
			# about $tmon - 1 see timelocal / localtime  description
			$timeStamp = $nodeTimeStamp;
		}
		#printf "local=$localTimeStamp, node=$nodeTimeStamp\n";
		$timeStamp =~ m/(.*)([0-9]{3})/;
		my $preSeconds = $1;
		my $lastSeconds = $2;
		my $chkSeconds = ($lastSeconds > 800?800
			:	 ($lastSeconds > 600?600
			:	 ($lastSeconds > 400?400
			:	 ($lastSeconds > 200?200:'000') ) ) );
		my $timeChk = "$preSeconds" . "$chkSeconds";
		my @chkPowerKeys = ();

		for (my $pCount=0;$pCount<10;$pCount++) { # 200 seconds frequency	
			push(@chkPowerKeys, ($timeChk - ($pCount*200))) ;
			push(@chkPowerKeys, ($timeChk - ($pCount*200))) ;
			push(@chkPowerKeys, ($timeChk - ($pCount*200))) ;
		}

		if ($main::verbose > 98) { # for diagnostic only !!!
			my $pEntries = $main::session->get_entries( -columns => \@bladePowerChecks );
			foreach my $snmpPKey ( keys %{$pEntries} ) {
				print "$snmpPKey --- $pEntries->{$snmpPKey}\n";
			}
		}
		foreach my $pKey (@chkPowerKeys) {
			my $chkPowerAverage = undef;
			my $chkPowerAverage1 = trySNMPget($snmpOidPowerAverage . '.1.1.' . $pKey ,
				"PowerAverage");
			# ... should be available all 10 Minutes
			my $chkPowerAverage2 = trySNMPget($snmpOidPowerAverage . '.1.2.' . $pKey ,
				"PowerAverage");
			# ... should be available once per hour
			my $chkPowerAverage3 = trySNMPget($snmpOidPowerAverage . '.1.3.' . $pKey ,
				"PowerAverage");
			# ... should be available once per day
			my $powerEntity = undef;
			if ((defined $chkPowerAverage1) && ($chkPowerAverage1 != 0)) {
				$chkPowerAverage = $chkPowerAverage1 ;
				$powerEntity = '.1.1.';
			}
			if ((!defined $chkPowerAverage) 
			&& (defined $chkPowerAverage2) && ($chkPowerAverage2 != 0)) 
			{
				$chkPowerAverage = $chkPowerAverage2;
				$powerEntity = '.1.2.';	
			}
			if ((!defined $chkPowerAverage) 
			&& (defined $chkPowerAverage3) && ($chkPowerAverage2 != 0)) {
				$chkPowerAverage = $chkPowerAverage3;
				$powerEntity = '.1.3.';	
			}
			if ((defined $chkPowerAverage) && ($chkPowerAverage != 0)) {
				addPowerConsumptionToPerfdata($chkPowerAverage, undef,undef, undef,undef)
					if (!$main::verboseTable);
				if ($main::verbose >= 2) {
					my $chkPowerMin 
						= trySNMPget($snmpOidPowerMin . $powerEntity . $pKey ,
						"PowerMin");
					my $chkPowerMax 
						= trySNMPget($snmpOidPowerMax . $powerEntity . $pKey ,
						"PowerMax");
					addStatusTopicToVerbose(undef, "PowerConsumption", undef);
					addKeyValueToVerbose("TimeStamp", $pKey);
					addKeyWattToVerbose("Average", $chkPowerAverage,
					    undef,undef, $chkPowerMin, $chkPowerMax);
					$variableVerboseMessage .= "\n";
				}
				last;
			} #found
		} #try keys
	} #optChkPower
} #primergyManagementBladePowerConsumption
sub primergyManagementBlade_MgmtBladeTable {
	my $snmpOidMgmtBladeTable = '.1.3.6.1.4.1.7244.1.1.1.2.1.1.'; #s31MgmtBladeTable (1)
	my $snmpOidStatus	= $snmpOidMgmtBladeTable .  '2'; #s31MgmtBladeStatus
	my $snmpOidSerial	= $snmpOidMgmtBladeTable .  '5'; #s31MgmtBladeSerialNumber
	my $snmpOidProduct	= $snmpOidMgmtBladeTable .  '6'; #s31MgmtBladeProductName
	my $snmpOidModel	= $snmpOidMgmtBladeTable .  '7'; #s31MgmtBladeModelName
	my $snmpOidMAC		= $snmpOidMgmtBladeTable . '10'; #s31MgmtBladePhysicalAddress
	my $snmpOidRunMode	= $snmpOidMgmtBladeTable . '11'; #s31MgmtBladeRunMode
	my @tableChecks = (
		$snmpOidStatus, $snmpOidSerial, $snmpOidProduct, $snmpOidModel, 
		$snmpOidMAC, $snmpOidRunMode, 
	);
	my @statusText = ("none",
		"unknown", "ok", "not-present",	"error", "critical",
		"standby", "..unexpected..",
	);
	my @modeText = ( "none",
		"unknown", "master", "slave", "..unexpected..",
	);
	if ($main::verboseTable == 211) { # BLADE ManagementBlade
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Management Blade Table");
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $status = $entries->{$snmpOidStatus . '.' . $snmpID};
			my $serial = $entries->{$snmpOidSerial . '.' . $snmpID};
			my $product = $entries->{$snmpOidProduct . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $mac = $entries->{$snmpOidMAC . '.' . $snmpID};
			my $runmode = $entries->{$snmpOidRunMode . '.' . $snmpID};
			$status = 0 if (!defined $status or $status < 0);
			$status = 7 if ($status > 7);
			$runmode = 0 if (!defined $runmode or $runmode < 0);
			$runmode = 4 if ($runmode > 4);
			{ 
				addStatusTopicToVerbose($statusText[$status], "MMB", $snmpID);
				addSerialIDsToVerbose($serial, undef);
				addProductModelToVerbose($product,$model);
				addKeyValueToVerbose("MAC",$mac);
				addKeyValueToVerbose("RunMode",$modeText[$runmode]) if (defined $runmode);
				$variableVerboseMessage .= "\n";
			}
		} # each
	}
} #primergyManagementBlade_MgmtBladeTable
sub primergyManagementBlade {
	############# 
	my %bladeErrorCodeMap = (	0 => 3,
					1 => 3,
					2 => 0,
					3 => 1,
					4 => 2);

	my $snmpOidBladeStatus = '.1.3.6.1.4.1.7244.1.1.1.3.1.5.0'; #s31SysCtrlOverallStatus.0
	my $snmpOidBladeID = '.1.3.6.1.4.1.7244.1.1.1.3.5.1.0'; #s31SysChassisSerialNumber.0
	my $snmpOidS31Test = $snmpOidBladeID;

	my $snmpOidBladeCtrlOverall = '.1.3.6.1.4.1.7244.1.1.1.3.1.5.0'; #s31SysCtrlOverallStatus.0
	my $snmpOidAgentInfo = '.1.3.6.1.4.1.7244.1.1.1.1.'; #s31AgentInfo
	my $snmpOidBladeAdmURL	= $snmpOidAgentInfo . '5.0';#s31AgentAdministrativeUrl.0
	#my $snmpOidAgentName	= $snmpOidAgentInfo . '9.0';#s31AgentName.0 ... see RFC1213

	#------------------------------------------------------
	mibTestSNMPget($snmpOidS31Test,"PRIMERGY Blade");
	# get overall status
	my $bladestatus = simpleSNMPget($snmpOidBladeStatus,"BladeStatus");
	$bladestatus = 0 if (!defined $bladestatus or $bladestatus < 0 or $bladestatus > 4);
	$exitCode = $bladeErrorCodeMap{$bladestatus} if ($optChkSystem);

	# get ServerID
	if (defined $optChkSystem) { 
		$serverID = simpleSNMPget($snmpOidBladeID,"BladeID");
		$msg .= " - ID=" . $serverID . ' ';
	}
	# get power consumption if enabled
	primergyManagementBladePowerConsumption();
	
	#### TEXT LANGUAGE AWARENESS
	# get subsystem information
	primergyManagementBladeEnvironment();
	primergyManagementBladePowerSupply();
	if ($exitCode == 1 or $exitCode == 2 or $main::verbose >= 1) {
		my $admURL = trySNMPget($snmpOidBladeAdmURL,"AgentAdminURL");
		RFC1213sysinfoToLong();
		addAdminURLToLong($admURL);
		$longMessage .= "\n" if ($admURL);
	}
	if (defined $optChkSystem) { 
		if (($main::verbose >= 1)
		||  (  (($exitCode == 1) || ($exitCode == 2)) 
		    && ($overallFan < $exitCode) && ($overallTemp < $exitCode) && ($overallPS < $exitCode)
		    )
		) {
			my $oCtrl = simpleSNMPget($snmpOidBladeCtrlOverall,"ControlOverall");
			$oCtrl = 0 if (!defined $oCtrl or $oCtrl < 0 or $oCtrl > 4);
			my $overallCtrl = $bladeErrorCodeMap{$oCtrl};
			$msg .= ' - SystemControl(' . $s31OverallStatusText[$oCtrl] . ')';
			
			if ((($exitCode == 1) || ($exitCode == 2))
			&& ($overallCtrl == 0)
			) {
				$msg .= ' - BladesInside(' . $state[$exitCode] . ')'; # never reached point
			}
		} # verbose or search-not-ok
	} #optChkSystem

	primergyManagementBlade_MgmtBladeTable();
} # end primergyManagementBlade

#----------- primergy blade functions
sub primergyFSIOMBlade {
	my $snmpOidBladeFsiom		= '.1.3.6.1.4.1.7244.1.1.1.3.8.'; #s31SysFsiom
	my $snmpOidState		= $snmpOidBladeFsiom . '1.0'; #s31SysFsiomStatus.0 
	my $snmpOidProduct		= $snmpOidBladeFsiom . '3.0'; #s31SysFsiomProductName.0
	my $snmpOidSerial		= $snmpOidBladeFsiom . '5.0'; #s31SysFsiomSerialNumber.0
	my $snmpOidModel		= $snmpOidBladeFsiom . '9.0'; #s31SysFsiomModelName.0
	# '7.0' s31SysFsiomConnectionStatus
	# '8,0' s31SysFsiomConnectionTarget
	# unknown(1), ok(2), not-present(3), error(4), critical(5)
	my %bladeFsiomErrorCodeMap = (	0 => 3,
					1 => 3,
					2 => 0,
					3 => 3,
					4 => 1,
					5 => 2);
	my @statusText = ("none",
		"unknown", "ok", "not-present", "error", "critical",
		"..unexpected..",
	);
	my $overallFsiom = undef;
	my $tmpOverallFsiom = trySNMPget($snmpOidState,"FsiomOverall");
	$tmpOverallFsiom = 0 if (!defined $tmpOverallFsiom or $tmpOverallFsiom < 0 or $tmpOverallFsiom > 5);
	if ($tmpOverallFsiom) {
		$overallFsiom = $bladeFsiomErrorCodeMap{$tmpOverallFsiom};
	}
	if (($tmpOverallFsiom) && $overallFsiom != 3) { # not "not-present" or "unknown"
		$msg .= ' FSIOM(' . $statusText[$tmpOverallFsiom] . ')';
	}
	# return code
	if (defined $overallFsiom ) {
		my $serial = undef;
		my $product = undef;
		my $model = undef;
		my $getinfos = 0;
		my $verbose = 0;
		my $notify = 0;
		$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
		$notify = 1 if ($overallFsiom == 5 or $overallFsiom == 4);
		$getinfos = 1 if ($verbose or $notify);
		if ($getinfos) {
			$serial = trySNMPget($snmpOidSerial,"FsiomSerialNumber");
			$product = trySNMPget($snmpOidProduct,"FsiomProduct");
			$model = trySNMPget($snmpOidModel,"FsiomModel");
		}
		if ($verbose) {
			addStatusTopicToVerbose($statusText[$tmpOverallFsiom], "FSIOM", undef);
			addSerialIDsToVerbose($serial, undef);
			addProductModelToVerbose($product, $model);
			endVariableVerboseMessageLine();
		}
		elsif ($notify) {
			addStatusTopicToLong($statusText[$tmpOverallFsiom], "FSIOM", undef);
			addSerialIDsToLong($serial, undef);
			addProductModelToLong($product, $model);
			endLongMessageLine();
		}
		# return codes
		$exitCode = 0 if ($exitCode == 3);# reset default
		if (($overallFsiom == 4) && ($exitCode < 1)) { #nagios warning 
			$exitCode = 1;
		}
		if (($overallFsiom == 5) && ($exitCode < 2)) { #nagios critical 
			$exitCode = 2;
		}
	}
} #end primergyFSIOMBlade
sub primergyServerBlades {
	############# 
	my $snmpOidSrvBlade	= '.1.3.6.1.4.1.7244.1.1.1.4.'; #s31ServerBlade

	my $snmpOidSrvCtrlTable	= $snmpOidSrvBlade . '1.1.1.'; #s31SvrCtrlTable
	my $snmpOidAdmURL	= $snmpOidSrvCtrlTable . '4'; #s31SvrCtrlAdministrativeUrl.<id>

	my $snmpOidSrvBladeTable = $snmpOidSrvBlade . '2.1.1.'; #s31SvrBladeTable
	my $snmpOidStatus		= $snmpOidSrvBladeTable . '2'; #s31SvrBladeStatus
	my $snmpOidSerial		= $snmpOidSrvBladeTable . '5'; #s31SvrBladeSerialNumber
	my $snmpOidProduct		= $snmpOidSrvBladeTable . '6'; #s31SvrBladeProductName
	my $snmpOidModel		= $snmpOidSrvBladeTable . '7'; #s31SvrBladeModelName
	my $snmpOidIDSerial		= $snmpOidSrvBladeTable . '17'; #s31SvrBladeIdentSerialNumber
	my $snmpOidHostName		= $snmpOidSrvBladeTable . '21'; #s31SvrHostname
	#s31SvrBladeCustomerProductName 22
	my @bladeSrvBladeTableChecks = (
		$snmpOidStatus,
		$snmpOidSerial,
		$snmpOidProduct,
		$snmpOidModel,
		$snmpOidIDSerial,
		$snmpOidHostName,
	);
	my @cntSrvBladesCodes = ( -1, 0,0,0,0,0,0, 0 );
	my @verboseSrvBladeStatusText = ( "none", 
		"unknown","ok","not-present","error","critical",
		"standby", "..unexpected..",
	);

	my $snmpOidSrvNicInfoTable = $snmpOidSrvBlade . '7.1.1.'; #s31SvrNicInfoTable
	my $snmpOidNICPhyAddress	= $snmpOidSrvNicInfoTable . '3'; #s31SvrNicPhysicalAddress
	my $snmpOidNICIP		= $snmpOidSrvNicInfoTable . '4'; #s31SvrNicIpAddress
	my $snmpOidNICType		= $snmpOidSrvNicInfoTable . '6'; #s31SvrNicType
	my @nicInfoTableChecks = (
		$snmpOidNICPhyAddress, $snmpOidNICIP, $snmpOidNICType,
	);
	my @nicInfoTypeText = ( "none",
		"unknown", "on-board-lan-controller", "daughter-card", "baseboard-management-controller", "..unexpected..",
	);
		
	#------------------------------------------------------

	my $srvBladesData = $main::session->get_entries( -columns => \@bladeSrvBladeTableChecks );

	my $snmpKey = undef;
	my @listOfErrorServer = ();

	my $verbose = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	if (defined $srvBladesData) {
		my @srvBladesEntries = ();
		foreach $snmpKey ( keys %{$srvBladesData} ) {
			# print "$snmpKey --- $srvBladesEntries->{$snmpKey}\n";
			push(@srvBladesEntries, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
		}
		@srvBladesEntries = Net::SNMP::oid_lex_sort(@srvBladesEntries);

		addTableHeaderToVerbose("Server Blades") if ($verbose);
		foreach my $srvID (@srvBladesEntries) {
			my $srvStatus  = $srvBladesData->{$snmpOidStatus . '.' . $srvID};
			my $srvHostNm = $srvBladesData->{$snmpOidHostName . '.' . $srvID};
			my $serial = $srvBladesData->{$snmpOidSerial . '.' . $srvID};
			my $IDserial = $srvBladesData->{$snmpOidIDSerial . '.' . $srvID};
			my $product = $srvBladesData->{$snmpOidProduct . '.' . $srvID};
			my $model = $srvBladesData->{$snmpOidModel . '.' . $srvID};
			$srvStatus = 0 if (!defined $srvStatus or $srvStatus < 0);
			$srvStatus = 7 if ($srvStatus > 7);

			my $admURL = trySNMPget($snmpOidAdmURL . '.' . $srvID,"SrvBlade-AdminURL");
			if ((defined $admURL) && ($admURL !~ m/http/)) {
			    $admURL = undef;
			}
			if ($verbose) {
				addStatusTopicToVerbose($verboseSrvBladeStatusText[$srvStatus],"Server", $srvID);
				addSerialIDsToVerbose($serial, $IDserial);
				addHostNameToVerbose($srvHostNm);
				addAdminURLToVerbose($admURL);
				addProductModelToVerbose($product, $model);
				endVariableVerboseMessageLine();
			}
			$cntSrvBladesCodes[$srvStatus]++ if ($srvStatus and $srvStatus != 7);
			if (($srvStatus == 4 or $srvStatus == 5) and !$verbose ) {
				addStatusTopicToLong($verboseSrvBladeStatusText[$srvStatus], 
					"Server", $srvID);
				addSerialIDsToLong($serial, $IDserial);
				addHostNameToLong($srvHostNm);
				addAdminURLToLong($admURL);
				addProductModelToLong($product, $model);
				endLongMessageLine();
				push(@listOfErrorServer, $srvID);
			}
		}
		# output
		$msg .= " Server";
		for (my $i=1;$i < 7;$i++) {
			$msg .= "-$verboseSrvBladeStatusText[$i]($cntSrvBladesCodes[$i])"	
				if ($cntSrvBladesCodes[$i]);
		}
		$msg .= " ";
		# return code
		if (($exitCode == 3) 
		&& (  $cntSrvBladesCodes[2] || $cntSrvBladesCodes[3] || $cntSrvBladesCodes[4]
		   || $cntSrvBladesCodes[5] || $cntSrvBladesCodes[6])
		) { # reset default
		    $exitCode = 0;
		}
		if ($cntSrvBladesCodes[4] && $exitCode < 1) { #nagios warning 
			$exitCode = 1;
		}
		if ($cntSrvBladesCodes[5] && $exitCode < 2) { #nagios critical 
			$exitCode = 2;
		}
	} elsif (defined $optBladeSrv and $optBladeSrv != 999) {
		$msg .= " - no ServerBlade information - ";
	}
	# NIC
	if (defined $srvBladesData and ($verbose or $#listOfErrorServer >= 0)) {
		my $nicInfoEntries = $main::session->get_entries( -columns => \@nicInfoTableChecks );
		my @snmpKeys = ();
		foreach $snmpKey ( keys %{$nicInfoEntries} ) {
 			#print "$snmpKey --- $nicInfoEntries->{$snmpKey}\n";
			push(@snmpKeys, $1) if ($snmpKey =~ m/$snmpOidNICIP.(\d+\.\d+)/);
		}
		@snmpKeys = Net::SNMP::oid_lex_sort(@snmpKeys);
		my $saveSrvId = 0;
		my $foundSrvID = 0;
		addTableHeaderToVerbose("Server Blade NIC Table") if ($verbose);
		foreach my $snmpID (@snmpKeys) {
			my $phyAddress  = $nicInfoEntries->{$snmpOidNICPhyAddress . '.' . $snmpID};
			my $ip  = $nicInfoEntries->{$snmpOidNICIP . '.' . $snmpID};
			my $type  = $nicInfoEntries->{$snmpOidNICType . '.' . $snmpID};
			$ip = undef if ($ip and $ip =~ m/0\.0\.0\.0/);
			$snmpID =~ m/(.*)\.(.*)/;
			$type = 0 if (!defined $type or $type < 0);
			$type = 5 if ($type > 5);
			my $srvID = $1;
			my $nicIndex = $2;
			if ($saveSrvId != $srvID and $#listOfErrorServer >= 0) {
				$foundSrvID = 0;
				foreach my $chkid (@listOfErrorServer) {
					$foundSrvID = 1 if ($chkid == $srvID);
				}
			}
			if (defined $ip or $nicIndex == 1) {
				addStatusTopicToVerbose(undef,"ServerNicInfo", $snmpID);
				addKeyValueToVerbose("PhysicalAddress", $phyAddress);
				addIPToVerbose($ip);
				addKeyValueToVerbose("Type",$nicInfoTypeText[$type]) if ($type);
				$variableVerboseMessage .= "\n";
			}
			if (!$verbose and $foundSrvID and $ip) {
				addStatusTopicToLong(undef,"ServerNicInfo", $snmpID);
				addKeyValueToLong("PhysicalAddress", $phyAddress);
				addIPToLong($ip);
				addKeyValueToLong("Type",$nicInfoTypeText[$type]) if ($type);
				$longMessage .= "\n";
			}
			$saveSrvId = $srvID;
		}
	} #verbose
	chomp($msg);
} # end primergyServerBlades
sub primergySwitchBlades {
	############# 
	my $snmpOidSwBladeTable = '.1.3.6.1.4.1.7244.1.1.1.5.1.1.'; #s31SwitchBladeTable
	my $snmpOidStatus		= $snmpOidSwBladeTable . '2'; #s31SwitchBladeStatus
	my $snmpOidSerial		= $snmpOidSwBladeTable . '5'; #s31SwitchBladeSerialNumber
	my $snmpOidProduct		= $snmpOidSwBladeTable . '6'; #s31SwitchBladeProductName
	my $snmpOidModel		= $snmpOidSwBladeTable . '7'; #s31SwitchBladeModelName
	my $snmpOidIPAddr		= $snmpOidSwBladeTable . '10'; #s31SwitchBladeIpAddress
	#my $snmpOidAdmURL		= $snmpOidSwBladeTable . '11'; #s31SwitchBladeAdministrativeUrl
	# ... similar to IP
	my $snmpOidAssignedName		= $snmpOidSwBladeTable . '28'; #s31SwitchBladeUserAssignedName
	my $snmpOidIDSerial		= $snmpOidSwBladeTable . '29'; #s31SwitchBladeIdentSerialNumber
		# ATTENTION - IdentSerialNumber might be not available !
	my @bladeSwBladeTableChecks = (
		$snmpOidStatus,
		$snmpOidSerial,
		$snmpOidProduct,
		$snmpOidModel,
		$snmpOidIPAddr,
		$snmpOidAssignedName,
	);
	my @cntSwBladesCodes = ( -1, 0,0,0,0,0,0,0, 0,);
	my @verboseSwBladeStatusText = ( "none", 
		"unknown","ok","not-present","error","critical",
		"standby","present", "..unexpected..",
	);

	#------------------------------------------------------

	my $swBladesData = $main::session->get_entries( -columns => \@bladeSwBladeTableChecks );

	if (defined $swBladesData) {
		my @swBladesKeys = ();
		foreach my $snmpKey ( keys %{$swBladesData} ) {
			# print "$snmpKey --- $swBladesData->{$snmpKey}\n";
			push(@swBladesKeys, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
		}
		@swBladesKeys = Net::SNMP::oid_lex_sort(@swBladesKeys);

		my $verbose = 0;
		$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
		addTableHeaderToVerbose("Switch Blades") if ($verbose);
		foreach my $swID (@swBladesKeys) {
			my $swStatus  = $swBladesData->{$snmpOidStatus . '.' . $swID};
 			my $swIP = $swBladesData->{$snmpOidIPAddr . '.' . $swID};
 			my $serial = $swBladesData->{$snmpOidSerial . '.' . $swID};
 			my $product = $swBladesData->{$snmpOidProduct . '.' . $swID};
 			my $model = $swBladesData->{$snmpOidModel . '.' . $swID};
 			my $swUserName = $swBladesData->{$snmpOidAssignedName . '.' . $swID};
			my $IDserial = trySNMPget($snmpOidIDSerial . '.' . $swID,"IndentSerialNumber");
			$swStatus = 0 if (!defined $swStatus or $swStatus < 0);
			$swStatus = 8 if ($swStatus > 8);
			if ($verbose) {
				addStatusTopicToVerbose($verboseSwBladeStatusText[$swStatus], "Switch", $swID);
				addSerialIDsToVerbose($serial, $IDserial);
				addIPToVerbose($swIP);
				addNameToVerbose($swUserName);
				addProductModelToVerbose($product, $model);
				endVariableVerboseMessageLine();
			}
			$cntSwBladesCodes[$swStatus]++ if ($swStatus);
			if ((($swStatus == 4) || ($swStatus == 5)) &&  !$verbose ) {
				addStatusTopicToLong($verboseSwBladeStatusText[$swStatus], 
					"Switch", $swID);
				addSerialIDsToLong($serial, $IDserial);
				addIPToLong($swIP);
				addNameToLong($swUserName);
				addProductModelToLong($product, $model);
				endLongMessageLine();
			}
		}
		# output
		$msg .= " Switch";
		for (my $i=1;$i < 8;$i++) {
			$msg .= "-$verboseSwBladeStatusText[$i]($cntSwBladesCodes[$i])"	
				if ($cntSwBladesCodes[$i]);
		}
		$msg .= " ";
		# return code
		if (($exitCode == 3) 
		&& (  $cntSwBladesCodes[2] || $cntSwBladesCodes[3] || $cntSwBladesCodes[4]
		   || $cntSwBladesCodes[5] || $cntSwBladesCodes[6] || $cntSwBladesCodes[7])
		) { # reset default
		    $exitCode = 0;
		}
		if ($cntSwBladesCodes[4] && $exitCode < 1) { #nagios warning 
			$exitCode = 1;
		}
		if ($cntSwBladesCodes[5] && $exitCode < 2) { #nagios critical 
			$exitCode = 2;
		}
	} elsif (defined $optBladeIO_Switch) {
		$msg .= "-No Switch Blades-";
	}

	chomp($msg);
} # end primergySwitchBlades
sub primergyFiberChannelPassThroughBlades { # TODO - Blades with primergyFiberChannelPassThroughBlades
	############# 
	my $snmpOIDFcPTVoltageTable = '1.3.6.1.4.1.7244.1.1.1.8.1.1.1.'; #s31FcPassThroughBladeVoltageTable
	#### QUESTION s31FcPassThroughBladeVoltageTable
	my $snmpOIDFcPTPortsTable = '1.3.6.1.4.1.7244.1.1.1.8.1.3.1.'; #s31FcPassThroughBladePortsTable
	#### QUESTION s31FcPassThroughBladePortsTable
	my $snmpOIDFcPTInfoTable = '1.3.6.1.4.1.7244.1.1.1.8.1.2.1.'; #s31FcPassThroughBladeInfoTable
	my $snmpOIDInfoStatus = $snmpOIDFcPTInfoTable . '2'; #s31FcPassThroughBladeInfoStatus
	my $snmpOIDSerial	= $snmpOIDFcPTInfoTable . '5'; #s31FcPassThroughBladeInfoSerialNumber
	my $snmpOIDFcPTInfoWwnn = $snmpOIDFcPTInfoTable . '9'; #s31FcPassThroughBladeActiveWwnn
	my $snmpOIDFcPTInfoWwpn = $snmpOIDFcPTInfoTable . '10'; #s31FcPassThroughBladeActiveWwpn	
	my $snmpOIDSlot		= $snmpOIDFcPTInfoTable . '14'; #s31FcPassThroughBladeSlotId	
	my $snmpOIDIdSerial	= $snmpOIDFcPTInfoTable . '16'; #s31FcPassThroughBladeInfoIdentSerialNumber	
	my @bladeFcPTInfoTableChecks = (
		$snmpOIDInfoStatus,
		$snmpOIDFcPTInfoWwnn,
		$snmpOIDFcPTInfoWwpn,
		$snmpOIDSerial,
		$snmpOIDSlot,
 	);
	#### TODO Which Information is relevant/interesting for the customer ?
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0, );
	my @verboseStatusText = ( "none", 
		"unknown","ok","not-present","error","critical",
		"standby", "..unexpected..", 
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeFcPTInfoTableChecks );

	if (defined $entries) {
		#$msg .= "-FOUND FiberChannelPassThrough Blades-";
		my @bladesKeys = ();
		foreach my $snmpKey ( keys %{$entries} ) {
			# print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDInfoStatus.(\d+)/);
		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Fiber Channel Pass Through Switch Blades");
 		foreach my $bladeID (@bladesKeys) {
 			my $status  = $entries->{$snmpOIDInfoStatus . '.' . $bladeID};
			my $slotid = $entries->{$snmpOIDSlot . '.' . $bladeID};
 			my $serial = $entries->{$snmpOIDSerial . '.' . $bladeID};
			my $IDserial = trySNMPget($snmpOIDIdSerial . '.' . $bladeID,"IdentSerialNumber");
			$status = 0 if (!defined $status or $status < 0);
			$status = 7 if ($status > 7);
			addStatusTopicToVerbose($verboseStatusText[$status], "FCPT", $bladeID);
			addSerialIDsToVerbose($serial, $IDserial);
			addSlotIDToVerbose($slotid);
			endVariableVerboseMessageLine();
 			$cntCodes[$status]++;
 			if ((($status == 4) || ($status == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$status], "FCPT", $bladeID);
				addSerialIDsToLong($serial, $IDserial);
				addSlotIDToLong($slotid);
				endLongMessageLine();
 			}
 		}
		# output
 		$msg .= " Fiber Channel Pass Through Switch";
 		for (my $i=1;$i < 7;$i++) {
 			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
 		$msg .= " ";
		# return code
 		if (($exitCode == 3) 
 		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
 		   || $cntCodes[5] || $cntCodes[6])
 		) { # reset default
 		    $exitCode = 0;
 		}
 		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
 			$exitCode = 1;
 		}
 		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
 			$exitCode = 2;
 		}
	} elsif (defined $optBladeIO_FCPT) {
		$msg .= "-No FiberChannelPassThrough Blades-";
	}

	chomp($msg);
} # end primergyFiberChannelPassThroughBlades
sub primergyPhyBlades {
	############# Pass Through Blades
	my $snmpOIDPhyBladeTable = '1.3.6.1.4.1.7244.1.1.1.10.1.1.'; #s31PhyBladeTable
	my $snmpOIDSerial		= $snmpOIDPhyBladeTable . '4'; #s31PhyBladeSerialNumber
	my $snmpOIDProduct		= $snmpOIDPhyBladeTable . '5'; #s31PhyBladeProductName
	my $snmpOIDStatus		= $snmpOIDPhyBladeTable . '9'; #s31PhyBladeStatus
	my $snmpOIDIdSerial		= $snmpOIDPhyBladeTable . '11'; #s31PhyBladeIdentSerialNumber
	my @bladeTableChecks = (
		$snmpOIDSerial,
		$snmpOIDProduct,
		$snmpOIDStatus,
 	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0,);
	my @verboseStatusText = ( "none", 
		"unknown","ok","not-present","error","critical",
		"standby", "..unexpected..", 
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );

	if (defined $entries) {
 		my @swBladesKeys = ();
 		foreach my $snmpKey ( keys %{$entries} ) {
 			# print "$snmpKey --- $entries->{$snmpKey}\n";
 			push(@swBladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
 		}
 		@swBladesKeys = Net::SNMP::oid_lex_sort(@swBladesKeys);

		addTableHeaderToVerbose("LAN Pass Through Blades");
 		foreach my $swID (@swBladesKeys) {
 			my $swStatus  = $entries->{$snmpOIDStatus . '.' . $swID};
 			my $product = $entries->{$snmpOIDProduct . '.' . $swID};
 			my $serial = $entries->{$snmpOIDSerial . '.' . $swID};
 			my $IDserial = trySNMPget($snmpOIDIdSerial . '.' . $swID,"IndentSerialNumber");
			$swStatus = 0 if (!defined $swStatus or $swStatus < 0);
			$swStatus = 7 if ($swStatus > 7);
 			addStatusTopicToVerbose($verboseStatusText[$swStatus], "LANPT", $swID, );
			addSerialIDsToVerbose($serial, $IDserial);
			addProductModelToVerbose($product, undef);
			endVariableVerboseMessageLine();
			$cntCodes[$swStatus]++;
 			if ((($swStatus == 4) || ($swStatus == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$swStatus], "LANPT", $swID);
				addSerialIDsToLong($serial, $IDserial);
				addProductModelToLong($product, undef);
				endLongMessageLine();
 			}
		}
		# output
 		$msg .= " LAN Pass Through Blades";
 		for (my $i=1;$i < 8;$i++) {
 			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
 		$msg .= " ";
		# return code
 		if (($exitCode == 3) 
 		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
 		   || $cntCodes[5] || $cntCodes[6] )
 		) { # reset default
 		    $exitCode = 0;
 		}
 		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
 			$exitCode = 1;
 		}
 		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
 			$exitCode = 2;
 		}
	} elsif (defined $optBladeIO_Phy) {
		$msg .= "-No LAN Pass Through Blades-";
	}

	chomp($msg);
} # end primergyPhyBlades
sub primergyFCSwitchBlades {
	############# 
	my $snmpOIDFCSwitchBladeTable = '1.3.6.1.4.1.7244.1.1.1.12.1.1.'; #s31FCSwitchBladeTable
	my $snmpOIDSerial	= $snmpOIDFCSwitchBladeTable . '4'; #s31FCSwitchBladeSerialNumber
	my $snmpOIDProduct	= $snmpOIDFCSwitchBladeTable . '5'; #s31FCSwitchBladeProductName
	my $snmpOIDModel	= $snmpOIDFCSwitchBladeTable . '6'; #s31FCSwitchBladeModelName
	my $snmpOIDIpAddress	= $snmpOIDFCSwitchBladeTable . '8'; #s31FCSwitchBladeIpAddress
	#my $snmpOIDFcIpAddress	= $snmpOIDFCSwitchBladeTable . '11'; #s31FCSwitchBladeFcIpAddress
	#my $snmpOIDFcName	= $snmpOIDFCSwitchBladeTable . '13'; #s31FCSwitchBladeFcSwitchName
	my $snmpOIDSlotId	= $snmpOIDFCSwitchBladeTable . '15'; #s31FCSwitchBladeSlotId
	my $snmpOIDStatus	= $snmpOIDFCSwitchBladeTable . '17'; #s31FCSwitchBladeStatus
	#my $snmpOIDAdmURL	= $snmpOIDFCSwitchBladeTable . '18'; #s31FCSwitchBladeAdministrativeURL
	# ... see ipaddr
	my $snmpOIDIdSerial	= $snmpOIDFCSwitchBladeTable . '23'; #s31FCSwitchBladeIdentSerialNumber
		# ATTENTION - IdentSerialNumber might be not available !
	my @bladeTableChecks = (
		$snmpOIDSerial,
		$snmpOIDProduct,
		$snmpOIDModel,
		$snmpOIDIpAddress,
	#	$snmpOIDFcIpAddress,
	#	$snmpOIDFcName,
		$snmpOIDSlotId,
		$snmpOIDStatus,
	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0, );
	my @verboseStatusText = ( "none", 
		"unkown","ok","not-present","error","critical",
		"standby", "..unexpected..",
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );

	if (defined $entries) {
		my @bladesKeys = ();
		foreach my $snmpKey ( keys %{$entries} ) {
			# print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Fiber Channel Switch Blades");
 		foreach my $bladeID (@bladesKeys) {
 			my $status  = $entries->{$snmpOIDStatus . '.' . $bladeID};
 			my $ipaddr = $entries->{$snmpOIDIpAddress . '.' . $bladeID};
 			my $product = $entries->{$snmpOIDProduct . '.' . $bladeID};
 			my $model = $entries->{$snmpOIDModel . '.' . $bladeID};
			my $slotid = $entries->{$snmpOIDSlotId . '.' . $bladeID};
 			my $serial = $entries->{$snmpOIDSerial . '.' . $bladeID};
			my $IDserial = trySNMPget($snmpOIDIdSerial . '.' . $bladeID,"IdentSerialNumber");
 			$status = 0 if (!defined $status or $status < 0);
			$status = 7 if ($status > 7);
			addStatusTopicToVerbose($verboseStatusText[$status], "FCSwitch", $bladeID);
			addSerialIDsToVerbose($serial, $IDserial);
			addSlotIDToVerbose($slotid);
			addIPToVerbose($ipaddr);
			addProductModelToVerbose($product, $model);
			endVariableVerboseMessageLine();
 			$cntCodes[$status]++;
 			if ((($status == 4) || ($status == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$status], "FCSwitch", $bladeID);
				addSerialIDsToLong($serial, $IDserial);
				addSlotIDToLong($slotid);
				addIPToLong($ipaddr);
				addProductModelToLong($product, $model);
				endLongMessageLine();
 			}
 		}
		# output
 		$msg .= " Fibre Channel Switch";
 		for (my $i=1;$i < 7;$i++) {
 			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
 		$msg .= " ";
		# return code
 		if (($exitCode == 3) 
 		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
 		   || $cntCodes[5] || $cntCodes[6])
 		) { # reset default
 		    $exitCode = 0;
 		}
 		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
 			$exitCode = 1;
 		}
 		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
 			$exitCode = 2;
 		}
	} elsif (defined $optBladeIO_FCSwitch) {
		$msg .= " -No Fiber Channel Switch Blades- ";
	}

	chomp($msg);
} # end primergyFCSwitchBlades
sub primergyIBSwitchBlades { #TODO - Blades with primergyIBSwitchBlades
	############# Infiniband Switch Blades
	my $snmpOIDIBSwitchBladeTable = '1.3.6.1.4.1.7244.1.1.1.16.1.1.'; #s31IBSwitchBladeTable
	my $snmpOIDSerialNr	= $snmpOIDIBSwitchBladeTable . '4'; #s31IBSwitchBladeSerialNumber
	my $snmpOIDProduct	= $snmpOIDIBSwitchBladeTable . '5'; #s31IBSwitchBladeProductName
	my $snmpOIDModel	= $snmpOIDIBSwitchBladeTable . '6'; #s31IBSwitchBladeModelName
	my $snmpOIDIpAddress	= $snmpOIDIBSwitchBladeTable . '8'; #s31IBSwitchBladeIpAddress
	my $snmpOIDSlotId	= $snmpOIDIBSwitchBladeTable . '11'; #s31IBSwitchBladeSlotId
	my $snmpOIDStatus	= $snmpOIDIBSwitchBladeTable . '13'; #s31IBSwitchBladeStatus
	my $snmpOIDAdmURL	= $snmpOIDIBSwitchBladeTable . '14'; #s31IBSwitchBladeAdministrativeURL
	my $snmpOIDIdSerial	= $snmpOIDIBSwitchBladeTable . '19'; #s31IBSwitchBladeIdentSerialNumber
	my @bladeTableChecks = (
		$snmpOIDSerialNr,
		$snmpOIDProduct,
		$snmpOIDModel,
		$snmpOIDIpAddress,
		$snmpOIDSlotId,
		$snmpOIDStatus,
		$snmpOIDAdmURL,
	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0,);
	my @verboseStatusText = ( "none", 
		"unkown","ok","not-present","error","critical",
		"standby", "..unexpected..", 
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );

	if (defined $entries) {
		#$msg .= "-FOUND Infiniband Switch Blades- ";
		my @bladesKeys = ();
		foreach my $snmpKey ( keys %{$entries} ) {
			# print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Infiniband Switch Blades");
 		foreach my $bladeID (@bladesKeys) {
 			my $status  = $entries->{$snmpOIDStatus . '.' . $bladeID};
 			my $ipaddr = $entries->{$snmpOIDIpAddress . '.' . $bladeID};
 			my $product = $entries->{$snmpOIDProduct . '.' . $bladeID};
 			my $model = $entries->{$snmpOIDModel . '.' . $bladeID};
			my $slotid = $entries->{$snmpOIDSlotId . '.' . $bladeID};
 			my $serial = $entries->{$snmpOIDSerialNr . '.' . $bladeID};
			my $admURL = $entries->{$snmpOIDAdmURL . '.' . $bladeID};
			my $IDserial = trySNMPget($snmpOIDIdSerial . '.' . $bladeID,"IdentSerialNumber");
 			$status = 0 if (!defined $status or $status < 0);
			$status = 7 if ($status > 7);
 			addStatusTopicToVerbose($verboseStatusText[$status], "IBSwitch", $bladeID);
			addSerialIDsToVerbose($serial, $IDserial);
			addSlotIDToVerbose($slotid);
			addIPToVerbose($ipaddr);
			addAdminURLToVerbose($admURL);
			addProductModelToVerbose($product, $model);
			endVariableVerboseMessageLine();
 			$cntCodes[$status]++;
 			if ((($status == 4) || ($status == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$status], "IBSwitch", $bladeID);
				addSerialIDsToLong($serial, $IDserial);
				addSlotIDToLong($slotid);
				addIPToLong($ipaddr);
				addAdminURLToLong($admURL);
				addProductModelToLong($product, $model);
				endLongMessageLine();
 			}
 		}
		# output
 		$msg .= " Infiniband Switch";
 		for (my $i=1;$i < 7;$i++) {
 			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
 		$msg .= " ";
		# return code
 		if (($exitCode == 3) 
 		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
 		   || $cntCodes[5] || $cntCodes[6])
 		) { # reset default
 		    $exitCode = 0;
 		}
 		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
 			$exitCode = 1;
 		}
 		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
 			$exitCode = 2;
 		}
	} elsif (defined $optBladeIO_IBSwitch) {
		$msg .= "-No Infiniband Switch Blades- ";
	}

	chomp($msg);
} # end primergyIBSwitchBlades
sub primergySASSwitchBlades {
	############# Serial attached SCSI Switch Blade
	my $snmpOIDSASSwitchBladeTable = '1.3.6.1.4.1.7244.1.1.1.17.1.1.'; #s31SASSwitchBladeTable
	my $snmpOIDSerial	= $snmpOIDSASSwitchBladeTable . '4'; #s31SASSwitchBladeSerialNumber
	my $snmpOIDProduct	= $snmpOIDSASSwitchBladeTable . '5'; #s31SASSwitchBladeProductName
	my $snmpOIDModel	= $snmpOIDSASSwitchBladeTable . '6'; #s31SASSwitchBladeModelName
	my $snmpOIDIpAddress	= $snmpOIDSASSwitchBladeTable . '8'; #s31SASSwitchBladeIpAddress
	my $snmpOIDSlotId	= $snmpOIDSASSwitchBladeTable . '11'; #s31SASSwitchBladeSlotId
	my $snmpOIDStatus	= $snmpOIDSASSwitchBladeTable . '13'; #s31SASSwitchBladeStatus
	#my $snmpOIDAdmURL	= $snmpOIDSASSwitchBladeTable . '14'; #s31SASSwitchBladeAdministrativeURL
	my $snmpOIDIDSerial	= $snmpOIDSASSwitchBladeTable . '20'; #s31SASSwitchBladeIdentSerialNumber
		# ATTENTION - IdentSerialNumber might be not available !
	my @bladeTableChecks = (
		$snmpOIDSerial,
		$snmpOIDProduct,
		$snmpOIDModel,
		$snmpOIDIpAddress,
		$snmpOIDSlotId,
		$snmpOIDStatus,
	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0,);
	my @verboseStatusText = ( "none", 
		"unkown","ok","not-present","error","critical",
		"standby", "..unexpected..",
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );
	if (defined $entries) {
 		my @bladesKeys = ();
 		foreach my $snmpKey ( keys %{$entries} ) {
 			#print "$snmpKey --- $entries->{$snmpKey}\n";
 			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
 		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Serial Attached SCSI Switch Blades");
  		foreach my $bladeID (@bladesKeys) {
 			my $status  = $entries->{$snmpOIDStatus . '.' . $bladeID};
 			my $ipAddr = $entries->{$snmpOIDIpAddress . '.' . $bladeID};
 			my $slot = $entries->{$snmpOIDSlotId . '.' . $bladeID};
 			my $serial = $entries->{$snmpOIDSerial . '.' . $bladeID};
 			my $model = $entries->{$snmpOIDModel . '.' . $bladeID};
			my $product = $entries->{$snmpOIDProduct . '.' . $bladeID};
 			my $IDserial = trySNMPget($snmpOIDIDSerial . '.' . $bladeID,"IndentSerialNumber");
  			$status = 0 if (!defined $status or $status < 0);
			$status = 7 if ($status > 7);
			addStatusTopicToVerbose($verboseStatusText[$status], "SASSwitch", $bladeID);
			addSerialIDsToVerbose($serial, $IDserial);
			addSlotIDToVerbose($slot);
			addIPToVerbose($ipAddr);
			addProductModelToVerbose($product, $model);
			endVariableVerboseMessageLine();
 			$cntCodes[$status]++;
 			if ((($status == 4) || ($status == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$status], "SASSwitch", $bladeID);
				addSerialIDsToLong($serial, $IDserial);
				addSlotIDToLong($slot);
				addIPToLong($ipAddr);
				addProductModelToLong($product, $model);
				endLongMessageLine();
 			}
 		}
		# output
 		$msg .= " Serial Attached SCSI Switch";
 		for (my $i=1;$i < 7;$i++) {
 			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
 				if ($cntCodes[$i]);
 		}
 		$msg .= " ";
		# return code
 		if (($exitCode == 3) 
 		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
 		   || $cntCodes[5] || $cntCodes[6])
 		) { # reset default
 		    $exitCode = 0;
 		}
 		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
 			$exitCode = 1;
 		}
 		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
 			$exitCode = 2;
 		}
	} elsif (defined $optBladeIO_SASSwitch) {
		$msg .= " -No Serial Attached SCSI Switch Blades- ";
	}

	chomp($msg);
} # end primergySASSwitchBlades
sub primergyIOConnectionBlades {
	if ((defined $optBladeIO_FSIOM) || ($optBladeIO != 999)) {
		primergyFSIOMBlade();
	}
	if ((defined $optBladeIO_Switch) || ($optBladeIO != 999)) {
		primergySwitchBlades();
	}
	if ((defined $optBladeIO_FCPT) || ($optBladeIO != 999)) {
		primergyFiberChannelPassThroughBlades();
	}
	if ((defined $optBladeIO_Phy) || ($optBladeIO != 999)) {
		primergyPhyBlades();
	}
	if ((defined $optBladeIO_FCSwitch) || ($optBladeIO != 999)) {
		primergyFCSwitchBlades();
	}
	if ((defined $optBladeIO_IBSwitch) || ($optBladeIO != 999)) {
		primergyIBSwitchBlades();
	}
	if ((defined $optBladeIO_SASSwitch) || ($optBladeIO != 999)) {
		primergySASSwitchBlades();
	}
} # end primergyIOConnectionBlades
sub primergyKVMBlades {
	############# PRIMERGY BLADE SERVERBLADE OIDs 
	my $snmpOIDKvmBladeTable = '1.3.6.1.4.1.7244.1.1.1.11.1.1.'; #s31KvmBladeTable
	my $snmpOIDSerial	= $snmpOIDKvmBladeTable . '4'; #s31KvmBladeSerialNumber
	my $snmpOIDProduct	= $snmpOIDKvmBladeTable . '5'; #s31KvmBladeProductName
	my $snmpOIDModel	= $snmpOIDKvmBladeTable . '6'; #s31KvmBladeModelName
	my $snmpOIDIpAddress	= $snmpOIDKvmBladeTable . '9'; #s31KvmBladeIpAddress
	my $snmpOIDStatus	= $snmpOIDKvmBladeTable . '18'; #s31KvmBladeStatus
	#my $snmpOIDAdmURL	= $snmpOIDKvmBladeTable . '19'; #s31KvmBladeAdministrativeURL
	my @bladeTableChecks = (
		$snmpOIDSerial,
		$snmpOIDProduct,
		$snmpOIDModel,
		$snmpOIDIpAddress,
		$snmpOIDStatus,
	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0,);
	my @verboseStatusText = ( "none", 
		"unkown","ok","not-present","error","critical",
		"standby", "..unexpected..",
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );

	if (defined $entries) {
 		my @bladesKeys = ();
 		foreach my $snmpKey ( keys %{$entries} ) {
 			#print "$snmpKey --- $entries->{$snmpKey}\n";
 			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
 		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Key-Video-Mouse Blades");
 		foreach my $kvmID (@bladesKeys) {
 			my $kvmStatus  = $entries->{$snmpOIDStatus . '.' . $kvmID};
 			my $kvmIP = $entries->{$snmpOIDIpAddress . '.' . $kvmID};
			my $kvmProduct = $entries->{$snmpOIDProduct . '.' . $kvmID};
			my $model = $entries->{$snmpOIDModel . '.' . $kvmID};
			my $serial = $entries->{$snmpOIDSerial . '.' . $kvmID};
  			$kvmStatus = 0 if (!defined $kvmStatus or $kvmStatus < 0);
			$kvmStatus = 7 if ($kvmStatus > 7);
			addStatusTopicToVerbose($verboseStatusText[$kvmStatus], "KVM", $kvmID, );
			addSerialIDsToVerbose($serial, undef);
			addIPToVerbose($kvmIP);
			addProductModelToVerbose($kvmProduct, $model);
			endVariableVerboseMessageLine();
			$cntCodes[$kvmStatus]++;
			if ((($kvmStatus == 4) || ($kvmStatus == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$kvmStatus], "KVM", $kvmID);
				addSerialIDsToLong($serial, undef);
				addIPToLong($kvmIP);
				addProductModelToLong($kvmProduct, $model);
				endLongMessageLine();
			}
 		}
		# output
		$msg .= " KVM";
		for (my $i=1;$i < 7;$i++) {
			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
				if ($cntCodes[$i]);
		}
		$msg .= " ";
		# return code
		if (($exitCode == 3) 
		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
		   || $cntCodes[5] || $cntCodes[6])
		) { # reset default
		    $exitCode = 0;
		}
		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
			$exitCode = 1;
		}
		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
			$exitCode = 2;
		}
	} elsif (defined $optBladeKVM and $optBladeKVM != 999) {
		$msg .= "-No KVM Blades- ";
	}

	chomp($msg);
} # end primergyKVMBlades
sub primergyStorageBlades {
	############# PRIMERGY BLADE SERVERBLADE OIDs 
	my $snmpOIDStorageBladeTable = '1.3.6.1.4.1.7244.1.1.1.13.1.1.'; #s31StorageBladeTable
	my $snmpOIDSerialNR	= $snmpOIDStorageBladeTable . '4'; #s31StorageBladeSerialNumber
	my $snmpOIDProduct	= $snmpOIDStorageBladeTable . '5'; #s31StorageBladeProductName
	my $snmpOIDModel	= $snmpOIDStorageBladeTable . '6'; #s31StorageBladeModelName
	my $snmpOIDStatus	= $snmpOIDStorageBladeTable . '8'; #s31StorageBladeStatus
	my $snmpOIDIdSerialNR	= $snmpOIDStorageBladeTable . '10'; #s31StorageBladeIdentSerialNumber
	my $snmpOIDAdmURL	= $snmpOIDStorageBladeTable . '11'; #s31StorageBladeAdministrativeURL
	my @bladeTableChecks = (
		$snmpOIDSerialNR,
		$snmpOIDProduct,
		$snmpOIDModel,
		$snmpOIDStatus,
		$snmpOIDIdSerialNR,
		$snmpOIDAdmURL,
	);
	my @cntCodes = ( -1, 0,0,0,0,0,0, 0,);
	my @verboseStatusText = ( "none", 
		"unkown","ok","not-present","error","critical",
		"standby", "..unexpected..",
	);

	#------------------------------------------------------

	my $entries = $main::session->get_entries( -columns => \@bladeTableChecks );

	if (defined $entries) {
  		my @bladesKeys = ();
 		foreach my $snmpKey ( keys %{$entries} ) {
 			#print "$snmpKey --- $entries->{$snmpKey}\n";
 			push(@bladesKeys, $1) if ($snmpKey =~ m/$snmpOIDStatus.(\d+)/);
 		}
 		@bladesKeys = Net::SNMP::oid_lex_sort(@bladesKeys);

		addTableHeaderToVerbose("Storage Blades");
		foreach my $storeID (@bladesKeys) {
 			my $storeStatus  = $entries->{$snmpOIDStatus . '.' . $storeID};
 			my $storeSerial = $entries->{$snmpOIDSerialNR . '.' . $storeID};
 			my $storeIdSerial = $entries->{$snmpOIDIdSerialNR . '.' . $storeID};
			my $storeAdmURL = $entries->{$snmpOIDAdmURL . '.' . $storeID};
			my $product = $entries->{$snmpOIDProduct . '.' . $storeID};
			my $model = $entries->{$snmpOIDModel . '.' . $storeID};
   			$storeStatus = 0 if (!defined $storeStatus or $storeStatus < 0);
			$storeStatus = 7 if ($storeStatus > 7);
			addStatusTopicToVerbose($verboseStatusText[$storeStatus], "Storage", $storeID);
			addSerialIDsToVerbose($storeSerial, $storeIdSerial);
			addAdminURLToVerbose($storeAdmURL);
			addProductModelToVerbose($product, $model);
			endVariableVerboseMessageLine();
			$cntCodes[$storeStatus]++;
			if ((($storeStatus == 4) || ($storeStatus == 5)) &&  ($main::verbose < 2) ) {
				addStatusTopicToLong($verboseStatusText[$storeStatus], 
					"Storage", $storeID);
				addSerialIDsToLong($storeSerial, $storeIdSerial);
				addAdminURLToLong($storeAdmURL);
				addProductModelToLong($product, $model);
				endLongMessageLine();
			}
 		}
		# output
		$msg .= " Storage Blades";
		for (my $i=1;$i < 7;$i++) {
			$msg .= "-$verboseStatusText[$i]($cntCodes[$i])"	
				if ($cntCodes[$i]);
		}
		$msg .= " ";
 		# return code
		if (($exitCode == 3) 
		&& (  $cntCodes[2] || $cntCodes[3] || $cntCodes[4]
		   || $cntCodes[5] || $cntCodes[6])
		) { # reset default
		    $exitCode = 0;
		}
		if ($cntCodes[4] && $exitCode < 1) { #nagios warning 
			$exitCode = 1;
		}
		if ($cntCodes[5] && $exitCode < 2) { #nagios critical 
			$exitCode = 2;
		}
	} elsif (defined $optBladeStore and $optBladeStore != 999) {
		$msg .= "-No Storage Blades- ";
	}

	chomp($msg);
} # end primergyStorageBlades

#----------- primergy server functions
sub primergyServerFanTable {
	my $rcEnv = shift;

	my $snmpOidEnvFanPrefix = '.1.3.6.1.4.1.231.2.10.2.2.10.5.2.1.'; #sc2FanTable
	my $snmpOidEnvFanStatus		= $snmpOidEnvFanPrefix . '5'; #sc2fanStatus
	my $snmpOidEnvFanSpeed		= $snmpOidEnvFanPrefix . '6'; #sc2fanCurrentSpeed
	my $snmpOidEnvFanDesc		= $snmpOidEnvFanPrefix . '3'; #sc2fanDesignation
	my @envFanChecks = (
		$snmpOidEnvFanStatus,
		$snmpOidEnvFanSpeed,
		$snmpOidEnvFanDesc
	);

	#### TEXT LANGUAGE AWARENESS
	my @srvFanStatus = ( "none",
		"unknown", "disabled", "ok", "fail", "prefailure-predicted",
		"redundant-fan-failed", "not-manageable", "not-present", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if ($rcEnv and $rcEnv < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkEnvironment or $optChkEnv_Fan) and $getInfos) {	
		my @snmpIDs = ();
		my $entries = $main::session->get_entries( -columns => \@envFanChecks );
		
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidEnvFanStatus.(\d+\.\d+)/);
		}
		
		# sort fetched data
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		addTableHeaderToVerbose("Fans") if ($verbose);
		foreach my $fanID (@snmpIDs) {
			my $fanStatus  = $entries->{$snmpOidEnvFanStatus . '.' . $fanID};
			my $fanDesc = $entries->{$snmpOidEnvFanDesc . '.' . $fanID};
			my $fanSpeed = $entries->{$snmpOidEnvFanSpeed . '.' . $fanID};
			$fanDesc =~ s/[ ,;=]/_/g;
			$fanDesc =~ s/_$//;
			$fanStatus = 0 if (!defined $fanStatus or $fanStatus < 0);
			$fanStatus = 9 if ($fanStatus > 9);
			next if (($fanStatus eq '2' or $fanStatus eq '8') and $main::verbose < 3);
			
			if ($verbose) {
				addStatusTopicToVerbose($srvFanStatus[$fanStatus], "Fan", $fanID =~ m/\d+\.(\d+)/);
				addNameToVerbose($fanDesc);
				addKeyRpmToVerbose("Speed", $fanSpeed);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and   (($fanStatus == 2) || ($fanStatus == 4) || ($fanStatus == 5) ||($fanStatus == 6))
			) {
				addStatusTopicToLong($srvFanStatus[$fanStatus], "Fan", $fanID =~ m/\d+\.(\d+)/);
				addNameToLong($fanDesc);
				addKeyRpmToLong("Speed", $fanSpeed);
				$longMessage .= "\n";
			}
		} #for
	}
} #primergyServerFanTable
sub primergyServerTemperatureSensorTable {
	my $rcEnv = shift;

	my $snmpOidEnvTempPrefix = '1.3.6.1.4.1.231.2.10.2.2.10.5.1.1.'; #sc2TemperatureSensorTable
	my $snmpOidEnvTempStatus	= $snmpOidEnvTempPrefix . '5'; #sc2tempSensorStatus
	my $snmpOidEnvTempWarn		= $snmpOidEnvTempPrefix . '7'; #sc2tempWarningLevel
	my $snmpOidEnvTempCrit		= $snmpOidEnvTempPrefix . '8'; #sc2tempCriticalLevel
	my $snmpOidEnvTempValue		= $snmpOidEnvTempPrefix . '6'; #sc2tempCurrentTemperature
	my $snmpOidEnvTempDesc		= $snmpOidEnvTempPrefix . '3'; #sc2tempSensorDesignation
	my @envTempChecks = (
		$snmpOidEnvTempStatus,
		$snmpOidEnvTempValue,
		$snmpOidEnvTempWarn,
		$snmpOidEnvTempCrit,
		$snmpOidEnvTempDesc
	);
	my @srvTempStatus = ( "none",
		"unknown", "not-available", "ok", "sensor-failed", "failed", 
		"temperature-warning-toohot", "temperature-critical-toohot", "ok", "temperature-warning", "..unexpected..",
	); # 8 is "temperature-normal"
	# now check the temperatures
	my $getinfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 1 and !$main::verboseTable);
	$notify = 1 if ($rcEnv and $rcEnv < 3);
	$getinfos = 1; # get performance data always !
	if (($optChkEnvironment or $optChkEnv_Temp) and $getinfos) {
		my @snmpIDs = ();
		my $entries = $main::session->get_entries( -columns => \@envTempChecks );
		
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidEnvTempStatus.(\d+\.\d+)/);
		}		
		# sort fetched data
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		addTableHeaderToVerbose("Temperature Sensors") if ($verbose);
		foreach my $tempID (@snmpIDs) {	
			my $tempStatus  = $entries->{$snmpOidEnvTempStatus . '.' . $tempID};
			$tempStatus = 0 if (!defined $tempStatus or $tempStatus < 0);
			$tempStatus = 10 if ($tempStatus > 10);
			next if ($tempStatus eq '2' and $main::verbose < 3);

			my $tempValue = $entries->{$snmpOidEnvTempValue . '.' . $tempID};
			my $tempWarn = $entries->{$snmpOidEnvTempWarn . '.' . $tempID};
			my $tempCrit = $entries->{$snmpOidEnvTempCrit . '.' . $tempID};
			my $tempDesc = $entries->{$snmpOidEnvTempDesc . '.' . $tempID};
			#next if ($tempValue == 0);

			$tempDesc =~ s/[ ,;=]/_/g;
			$tempDesc =~ s/_$//;

			addTemperatureToPerfdata($tempDesc, $tempValue, $tempWarn, $tempCrit)
				if (!$main::verboseTable);

			if ($verbose) {
				addStatusTopicToVerbose($srvTempStatus[$tempStatus],"Sensor",$tempID=~m/\d+\.(\d+)/);
				addNameToVerbose($tempDesc);
				addCelsiusToVerbose($tempValue, $tempWarn,$tempCrit);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and   ($tempStatus>=4 and $tempStatus!=8) 
			) {
				addStatusTopicToLong($srvTempStatus[$tempStatus],"Sensor",$tempID=~m/\d+\.(\d+)/);
				addNameToLong($tempDesc);
				addCelsiusToLong($tempValue, $tempWarn,$tempCrit);
				$longMessage .= "\n";
			}
		} #each sensor
	} # optChkEnvironment
} #primergyServerTemperatureSensorTable

sub primergyServerPowerSupplyTable {
	my $rcPower = shift;

	my $snmpOidPowerSupplyTable = '1.3.6.1.4.1.231.2.10.2.2.10.6.2.1.'; #sc2PowerSupplyTable
	#my $snmpOidSupplyNr	= $snmpOidPowerSupplyTable . '2'; #sc2psPowerSupplyNr
	#my $snmpOidDesignation	= $snmpOidPowerSupplyTable . '3'; #sc2PowerSupplyDesignation
	my $snmpOidIdentifier	= $snmpOidPowerSupplyTable . '4'; #sc2PowerSupplyIdentifier
	my $snmpOidStatus	= $snmpOidPowerSupplyTable . '5'; #sc2PowerSupplyStatus
	my $snmpOidLoad		= $snmpOidPowerSupplyTable . '6';#sc2psPowerSupplyLoad
	my $snmpOidMax		= $snmpOidPowerSupplyTable . '7'; #sc2psPowerSupplyNominal
	my @tableChecks = (
		$snmpOidIdentifier,
		$snmpOidStatus,
		$snmpOidLoad,
		$snmpOidMax,
	);
	my @statusText = ( "none", 
	    "unknown", "not-present", "ok", "failed", "ac-fail", 
	    "dc-fail", "critical-temperature", "not-manageable", "an-failure-predicted", "fan-failure",
	    "power-safe-mode","non-redundant-dc-fail", "non-redundant-ac-fail", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if ($rcPower and $rcPower < 3);
	$getInfos = 1 if ($verbose or $notify);
	if ($optChkPower and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Power Supply Unit") if ($verbose);
		foreach my $id (@snmpIdx) {
 			my $pstatus  = $entries->{$snmpOidStatus . '.' . $id};
 			my $name = $entries->{$snmpOidIdentifier . '.' . $id};
 			my $load = $entries->{$snmpOidLoad . '.' . $id};
 			my $max = $entries->{$snmpOidMax . '.' . $id};
			$pstatus = 0 if (!defined $pstatus or $pstatus < 0);
			$pstatus = 14 if ($pstatus > 14);
			next if ($pstatus == 2 and $main::verbose < 3);
			if ($verbose) {
				addStatusTopicToVerbose($statusText[$pstatus],"PSU",$id=~m/\d+\.(\d+)/);
				addNameToVerbose($name);
				addKeyWattToVerbose("CurrentLoad", $load,
					undef,undef, undef,$max);
				$variableVerboseMessage .= "\n";
			} elsif ($notify and $pstatus >=4) {
				addStatusTopicToLong($statusText[$pstatus],"PSU",$id=~m/\d+\.(\d+)/);
				addNameToLong($name);
				addKeyWattToLong("CurrentLoad", $load,
					undef,undef, undef,$max);
				$longMessage .= "\n";
			}
		} # each
	}
	#### TODO / QUESTION PRIMERGY -w -c and performance limits for power supplies (multiple PSUs !)
} #primergyServerPowerSupplyTable
sub primergyServerPowerConsumption {
	my $powerConsumption = undef;
	# Power consumption OID
	my $snmpOidPowerMonitoringTable = '.1.3.6.1.4.1.231.2.10.2.2.10.4.5.1.'; #sc2PowerMonitoringTable
	my $snmpOidPowerConsumption	= $snmpOidPowerMonitoringTable . '5.1'; #sc2pmCurrentPowerConsumption.1
	my $snmpOidPowerStatus		= $snmpOidPowerMonitoringTable . '7.1'; #sc2pmPowerLimitStatus.1
	my $snmpOidPowerLimit		= $snmpOidPowerMonitoringTable . '8.1'; #sc2pmPowerLimitThreshold.1
	my $snmpOidPowerPercentWarning	= $snmpOidPowerMonitoringTable . '9.1'; #sc2pmPowerLimitWarning.1
	my $snmpOidPowerCriticLevel	= $snmpOidPowerMonitoringTable . '10.1'; #sc2pmRedundancyCritLevel.1 
	my @verboseText = ( "none", 
		"unknown", "ok", "warning", "error", "disabled", 
		"..unexpected..",
	);

	if (defined $optChkPower and !$main::verboseTable) { 
		$powerConsumption = trySNMPget($snmpOidPowerConsumption,"PowerConsumption");
		if (defined $powerConsumption) {
			my $status = trySNMPget($snmpOidPowerStatus,"PowerLimitStatus");
			my $limit = undef;
			my $lWarning = undef;
			my $lCritical = undef;
			my $warn = undef;
			my $notify = 0;
			if ((defined $status) && ($status > 1) && ($status < 5) ) {
				if ($status == 4) {
					$exitCode = 2;
					$notify = 1;
				}
				if ($status == 3) {
					$exitCode = 1 if ($exitCode == 0 or $exitCode == 3);
					$notify = 1;
				}
				$limit = trySNMPget($snmpOidPowerLimit,"PowerLimitThreshold");
				$lWarning = trySNMPget($snmpOidPowerPercentWarning,"PowerLimitWarning");
				$lCritical = trySNMPget($snmpOidPowerCriticLevel,"RedundancyCritLevel");
				if (defined $limit and defined $lWarning and $limit > 0) {
					$warn = ($limit / 100 ) * $lWarning;
				}
				if (defined $limit and defined $lCritical and ($lCritical <= 0)) {
					$lCritical = undef;
				}
				if ($main::verbose >= 2) {
					addStatusTopicToVerbose($verboseText[$status],"PowerConsumption",undef);
					addKeyWattToVerbose("Current", $powerConsumption,
						undef,$lCritical, undef, $limit);
					addKeyPercentToVerbose(undef,undef,
						$lWarning,undef, undef,undef);
					$variableVerboseMessage .= "\n"
				} elsif ($notify) {
					addStatusTopicToLong($verboseText[$status],"PowerConsumption",undef);
					addKeyWattToLong("Current", $powerConsumption,
						undef,$lCritical, undef, $limit);
					addKeyPercentToLong(undef,undef,
						$lWarning,undef, undef,undef);
					$longMessage .= "\n"
				}
			}

			addPowerConsumptionToPerfdata($powerConsumption, $warn,$lCritical, undef,undef)
				if (!$main::verboseTable);
		} # PowerConsumption available
	} #optChkPower
} #primergyServerPowerConsumption

sub primergyServer_ManagementNodeTable {
	my $snmpOidTable = '.1.3.6.1.4.1.231.2.10.2.2.10.3.1.1.'; #sc2ManagementNodeTable
	my $snmpOidAddress	= $snmpOidTable . '4'; #sc2UnitNodeAddress
	my $snmpOidName		= $snmpOidTable . '7'; #sc2UnitNodeName
 	my $snmpOidClass	= $snmpOidTable . '8'; #sc2UnitNodeClass
 	my $snmpOidMac		= $snmpOidTable . '9'; #sc2UnitNodeMacAddress
	#	... this mac is a string with 0x....
 	my $snmpOidModel	= $snmpOidTable . '12'; #sc2UnitNodeControllerModel
	my @tableChecks = (
		$snmpOidAddress,
		$snmpOidName,
		$snmpOidClass,
		$snmpOidMac,	
		$snmpOidModel,
	);
	my @classText = ( "none",
		 "unknown", "primary", "secondary", "management-blade",	"secondary-remote", "secondary-remote-backup", "baseboard-controller", "..unexpected..",
	);
	if (($optChkSystem and $main::verbose >=3) or $main::verboseTable == 311) { 
		#### QUESTION: is this relevant for somebody ?
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidAddress.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Management Nodes") if ($entries);
		foreach my $id (@snmpIdx) {
 			my $address  = $entries->{$snmpOidAddress . '.' . $id};
 			my $name = $entries->{$snmpOidName . '.' . $id};
 			my $classid = $entries->{$snmpOidClass . '.' . $id};
  			my $mac = $entries->{$snmpOidMac . '.' . $id};
			my $model = $entries->{$snmpOidModel . '.' . $id};
			$classid = 0 if (!defined $classid or $classid < 0);
			$classid = 8 if ($classid > 8);

			addStatusTopicToVerbose(undef, "Node", $id=~m/\d+\.(\d+)/);
			addIPToVerbose($address);
			addNameToVerbose($name) if ($name and !($name eq $address));
			addKeyLongValueToVerbose("ControllerType", $model);
			addKeyValueToVerbose("Class", $classText[$classid]) if ($classid > 2);
			addKeyValueToVerbose("MAC", $mac);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primergyServer_ManagementNodeTable
sub primergyServer_ServerTable {
	my $snmpOidTable = '.1.3.6.1.4.1.231.2.10.2.2.10.4.1.1.'; #sc2ServerTable
 	my $snmpOidMemory	= $snmpOidTable . '2'; #sc2srvPhysicalMemory
	my $snmpOidBootStatus	= $snmpOidTable . '4'; #sc2srvCurrentBootStatus
 	my $snmpOidUuid		= $snmpOidTable . '7'; #sc2srvUUID
	my @bootText = ( "none", 
		"unknown", "off", "no-boot-cpu", "self-test", "setup",
		"os-boot", "diagnostic-boot", "os-running", "diagnostic-running", "os-shutdown", 
		"diagnostic-shutdown", "reset", "..unexpected..",
	);
	my @tableChecks = (
		$snmpOidMemory,
		$snmpOidBootStatus,
		$snmpOidUuid,
	);
	if (($optChkSystem and $main::verbose >=3) or $main::verboseTable == 411) { 
		#### QUESTION: is this relevant for somebody ?
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidUuid.(\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Server Table") if ($entries);
		foreach my $id (@snmpIdx) {
 			my $memory  = $entries->{$snmpOidMemory . '.' . $id};
 			my $bstatus = $entries->{$snmpOidBootStatus . '.' . $id};
 			my $uuid = $entries->{$snmpOidUuid . '.' . $id};
			$bstatus = 0 if (!defined $bstatus or $bstatus < 0);
			$bstatus = 13 if ($bstatus > 13);

			addStatusTopicToVerbose(undef,"Server",$id);
			addKeyValueToVerbose("UUID", $uuid);
			addKeyMBToVerbose("Memory", $memory);
			addKeyValueToVerbose("BootStatus", $bootText[$bstatus]) if ($bstatus > 0);
			$variableVerboseMessage .= "\n";
		}
	} #verbose
} #primergyServer_ServerTable
our $cxChassisSerial	= undef;
our $cxChassisName	= undef;
our $cxChassisModel	= undef;
our $cxChassesLocation	= undef;
our $cxChassisContact	= undef;
sub primergyServerUnitTable {
	my $snmpOidUnitTable = '.1.3.6.1.4.1.231.2.10.2.2.10.2.3.1.'; #sc2UnitTable
	my $snmpOidUnitId	= $snmpOidUnitTable .'1' ;#sc2uUnitId
	my $snmpOidClass	= $snmpOidUnitTable .'2' ;#sc2UnitClass
	my $snmpOidCabNr	= $snmpOidUnitTable .'3' ;#sc2UnitCabinetNr
	my $snmpOidDesignation	= $snmpOidUnitTable .'4' ;#sc2UnitDesignation
	my $snmpOidModel	= $snmpOidUnitTable .'5' ;#sc2UnitModelName
	my $snmpOidSerial	= $snmpOidUnitTable .'7' ;#sc2UnitSerialNumber
	my $snmpOidLocation	= $snmpOidUnitTable .'8' ;#sc2UnitLocation
	my $snmpOidContact	= $snmpOidUnitTable .'9' ;#sc2UnitContact
	my $snmpOidAdmURL	= $snmpOidUnitTable .'10' ;#sc2UnitAdminURL
	my $snmpOidWWN		= $snmpOidUnitTable .'14' ;#sc2UnitWorldWideName
	my $snmpOidManIP	= $snmpOidUnitTable .'18' ;#sc2ManagementIpAddress

	# .1.3.6.1.4.1.231.2.10.2.2.10.2.3.1.4.1 Name of Server

	my @tableChecks = (
		$snmpOidUnitId, $snmpOidClass, $snmpOidCabNr, $snmpOidDesignation, $snmpOidModel,
		$snmpOidSerial, $snmpOidLocation, $snmpOidContact, $snmpOidAdmURL, $snmpOidWWN,
		$snmpOidManIP,
	);
	my @classText = ( "none",
		"unknown", "standardServer", "storageExtension", "bladeServerChassis", "bladeServer",
		"clusterNode", "multiCodeChassis", "multiNodeServer", "virtualServer", "virtualPartition",
		"systemboardInPartition", "unexpected12", "unexpected13", "unexpected14", "unexpected15", 
		"unexpected16", "unexpected17", "unexpected18", "unexpected19", "virtualServerVmware",
		"virtualServerHyperV", "virtualServerXen", "virtualServerPan", "..unexpected..", 
	);
	{ # SC2 UnitTable 2.3 => 231
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Unit Table") if ($main::verboseTable == 231);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidClass.(\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $unitid = $entries->{$snmpOidUnitId . '.' . $snmpID};
			my $class = $entries->{$snmpOidClass . '.' . $snmpID};
			my $cabnr = $entries->{$snmpOidCabNr . '.' . $snmpID};
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $serial = $entries->{$snmpOidSerial . '.' . $snmpID};
			my $location = $entries->{$snmpOidLocation . '.' . $snmpID};
			my $contact = $entries->{$snmpOidContact . '.' . $snmpID};
			my $admURL = $entries->{$snmpOidAdmURL . '.' . $snmpID};
			my $wwn = $entries->{$snmpOidWWN . '.' . $snmpID};
			my $manIp = $entries->{$snmpOidManIP . '.' . $snmpID};
			$class = 0 if (!defined $class or $class < 0);
			$class = 24 if ($class > 24);
			if ($class == 7 and $serial) { #multiCodeChassis
				$cxChassisSerial	= $serial;
				$cxChassisName		= $designation;
				$cxChassisModel		= $model;
				$cxChassesLocation	= $location;
				$cxChassisContact	= $contact;
			}
			if ($main::verboseTable == 231) { 
				addStatusTopicToVerbose(undef, undef, $unitid);
				addKeyIntValueToVerbose("CabinetNr", $cabnr);
				addSerialIDsToVerbose($serial, undef);
				addKeyValueToVerbose("Class",$classText[$class]) if ($class);
				addNameToVerbose($designation);
				addProductModelToVerbose(undef, $model);
				addLocationContactToVerbose($location, $contact);
				addAdminURLToVerbose($admURL);
				addKeyValueToVerbose("ManagementIP", $manIp);
				addKeyValueToVerbose("WWN", $wwn);
				$variableVerboseMessage .= "\n";
			}
		} # each
	} #
} #primergyServerUnitTable

sub primergyServerAgentInfo {
	my $snmpOidAgentInfoGroup = '.1.3.6.1.4.1.231.2.10.2.2.10.1.'; #sc2AgentInfo
	my $snmpOidAgtID	= $snmpOidAgentInfoGroup . '1.0'; #sc2AgentId
	my $snmpOidCompany	= $snmpOidAgentInfoGroup . '2.0'; #sc2AgentCompany
	my $snmpOidVersion	= $snmpOidAgentInfoGroup . '3.0'; #sc2AgentVersion
	if ($main::verbose >= 3) {
		my $id = trySNMPget($snmpOidAgtID,"sc2AgentInfo");
		my $company = trySNMPget($snmpOidCompany,"sc2AgentInfo");
		my $version = trySNMPget($snmpOidVersion,"sc2AgentInfo");
		if ($id || $company || $version) {
			addStatusTopicToVerbose(undef,"AgentInfo", undef);
			addKeyLongValueToVerbose("Ident",$id);
			addKeyValueToVerbose("Version",$version);
			addKeyLongValueToVerbose("Company",$company);
			$variableVerboseMessage .= "\n";
		} #found something
	} #verbose
} #primergyServerAgentInfo
sub primergyServerSystemInformation {
	#my $serial = shift;
	my $snmpOidUnitTable = '.1.3.6.1.4.1.231.2.10.2.2.10.2.3.1.'; #sc2UnitTable
	my $snmpOidName		= $snmpOidUnitTable . '4.1'; #sc2UnitDesignation.1
	my $snmpOidModel	= $snmpOidUnitTable . '5.1'; #sc2UnitModelName.1
	my $snmpOidLocation	= $snmpOidUnitTable . '8.1'; #sc2UnitLocation.1
	my $snmpOidContact	= $snmpOidUnitTable . '9.1'; #sc2UnitContact.1
	my $snmpOidAdminUrl	= $snmpOidUnitTable . '10.1'; #sc2UnitAdminURL.1
	my $snmpOidManIPAddress = $snmpOidUnitTable . '18.1'; #sc2ManagementIpAddress.1
	{
		if (($main::verbose) || ($exitCode==1) || ($exitCode==2)) {
			my $name = trySNMPget($snmpOidName,"sc2UnitTable");
			my $model = trySNMPget($snmpOidModel,"sc2UnitTable");
			my $location = trySNMPget($snmpOidLocation,"sc2UnitTable");
			my $contact = trySNMPget($snmpOidContact,"sc2UnitTable");
			my $admURL = trySNMPget($snmpOidAdminUrl,"sc2UnitTable");
			my $manageIP = trySNMPget($snmpOidManIPAddress,"sc2UnitTable");
			if ($location or $contact or $admURL or $manageIP or $model) {
				if ($cxChassisName or $cxChassisModel) {
					addStatusTopicToLong(undef,"Multi Node System", undef);
					addSerialIDsToLong($cxChassisSerial, undef);
					addNameToLong($cxChassisName);
					addLocationContactToLong($cxChassesLocation, $cxChassisContact);
					addProductModelToLong(undef, $cxChassisModel);
					$longMessage .= "\n";
				}
				if ($main::verbose) {
					RFC1213sysinfoToLong();
					$longMessage .= "\n";
					addStatusTopicToVerbose(undef, "SystemUnitInformation", undef);
					addNameToVerbose($name);
					addLocationContactToVerbose($location, $contact);
					addAdminURLToVerbose($admURL);
					addKeyValueToVerbose("ManagementIP",$manageIP);
					addProductModelToVerbose(undef, $model);
					if ($main::verbose == 1) {
						chop($longMessage); # remove previous \n
						addProductModelToLong(undef, $model);
						addAdminURLToLong($admURL);
						addKeyValueToLong("ManagementIP",$manageIP);
					}
					$variableVerboseMessage .= "\n";
				} elsif (($exitCode==1) || ($exitCode==2)) {
					RFC1213sysinfoToLong();
					addProductModelToLong(undef, $model);
					addAdminURLToLong($admURL);
					addKeyValueToLong("ManagementIP",$manageIP);
					$longMessage .= "\n";
				}
			}
		} # verbose or warning or error
	} 
} #primergyServerSystemInformation
our $psVoltageStatus = undef;
our $psCPUStatus = undef;
our $psMemoryStatus = undef;
our $psFanStatus = undef;
our $psTempStatus = undef;
our @psStatusText = ( "ok", "warning", "error", "unknown", ); #sc2Status in NAGIOS return values
sub primergyServerStatusComponentTable {
	my $snmpOidStatusGroup = '.1.3.6.1.4.1.231.2.10.2.2.10.8.'; #sc2Status
	my $snmpOidOverallStatus	= $snmpOidStatusGroup . '1.0'; #sc2AgentStatus
	my @statusText = ( "none",
		"ok", "warning", "error", undef, "unknown",
		"..unexpected..",
	);
	my %codeMap = (			1       =>      0,
					2       =>      1,
					3       =>      2,
					4       =>      2,
					5	=>	3,);
	my $snmpOidStatusComponentTable = $snmpOidStatusGroup . '2.1.'; #sc2StatusComponentTable
	my $snmpOidBoot		= $snmpOidStatusComponentTable . '3'; #sc2csStatusBoot
	my $snmpOidPower	= $snmpOidStatusComponentTable . '4'; #sc2csStatusPowerSupply
	my $snmpOidTemp		= $snmpOidStatusComponentTable . '5'; #sc2csStatusTemperature
	my $snmpOidFan		= $snmpOidStatusComponentTable . '6'; #sc2csStatusFans
	my $snmpOidVolt		= $snmpOidStatusComponentTable . '7'; #sc2csStatusVoltages
	my $snmpOidCpu		= $snmpOidStatusComponentTable . '8'; #sc2csStatusCpu
	my $snmpOidMem		= $snmpOidStatusComponentTable . '9'; #sc2csStatusMemoryModule
	my @tableChecks = (
		$snmpOidBoot, $snmpOidPower, $snmpOidTemp,
		$snmpOidFan, $snmpOidVolt, $snmpOidCpu, $snmpOidMem,
	);
	{
		#my $overallStatus = trySNMPget($snmpOidOverallStatus, "sc2AgentStatus");
		#$variableVerboseMessage .= "Overall: $statusText[$overallStatus]" if (defined $overallStatus);
		#$variableVerboseMessage .= "\n" if (defined $overallStatus);

		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("Status Component Table") if ($main::verbose >= 2);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidPower.(\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $boot = $entries->{$snmpOidBoot . '.' . $snmpID};
			my $power = $entries->{$snmpOidPower . '.' . $snmpID};
			my $temp = $entries->{$snmpOidTemp . '.' . $snmpID};
			my $fan = $entries->{$snmpOidFan . '.' . $snmpID};
			my $volt = $entries->{$snmpOidVolt . '.' . $snmpID};
			my $cpu = $entries->{$snmpOidCpu . '.' . $snmpID};
			my $mem = $entries->{$snmpOidMem . '.' . $snmpID};
			$boot = 0 if (!defined $boot or $boot < 0);
			$boot = 6 if ($boot > 6);
			if (($main::verbose >= 2 and $optChkSystem) or $main::verboseTable == 821) { 
				addStatusTopicToVerbose(undef,undef, $snmpID);
				addComponentStatusToVerbose("Boot", $statusText[$boot]) if (defined $boot); 
				addComponentStatusToVerbose("PowerSupplies", $statusText[$power]) 
					if (defined $power); 
				addComponentStatusToVerbose("Temperatures", $statusText[$temp]) 
					if (defined $temp); 
				addComponentStatusToVerbose("Fans", $statusText[$fan]) if (defined $fan); 
				addComponentStatusToVerbose("Voltages", $statusText[$volt]) if (defined $volt); 
				addComponentStatusToVerbose("CPU", $statusText[$cpu]) if (defined $cpu); 
				addComponentStatusToVerbose("MemoryModules", $statusText[$mem]) 
					if (defined $mem); 
				$variableVerboseMessage .= "\n";
			}
			$volt = 5 if ($volt and $volt > 5);
			$cpu = 5 if ($cpu and $cpu > 5);
			$mem = 5 if ($mem and $mem > 5);
			$fan = 5 if ($fan and $fan > 5);
			$temp = 5 if ($temp and $temp > 5);
			$psVoltageStatus = $codeMap{$volt} if ($volt and !defined $psVoltageStatus);
			$psCPUStatus = $codeMap{$cpu} if ($cpu and !defined $psCPUStatus);
			$psMemoryStatus = $codeMap{$mem} if ($mem and !defined $psMemoryStatus);
			$psFanStatus = $codeMap{$fan} if ($fan and !defined $psFanStatus);
			$psTempStatus = $codeMap{$temp} if ($temp and !defined $psTempStatus);

			$msg .= " Fan($statusText[$fan])" if ($optChkEnv_Fan and defined $psFanStatus);
			$msg .= " TemperatureSensor($statusText[$temp])" if ($optChkEnv_Temp and defined $psTempStatus);
		} # each
	}
} #primergyServerStatusComponentTable

sub primergyServerSystemBoardTable {
	my $snmpOidSystemBoardTable = '.1.3.6.1.4.1.231.2.10.2.2.10.6.1.1.'; #sc2SystemBoardTable
	my $snmpOidModel	= $snmpOidSystemBoardTable . '3'; #sc2SystemBoardModelName
	my $snmpOidProduct	= $snmpOidSystemBoardTable . '4'; #sc2SystemBoardProductNumber
	my $snmpOidSerial	= $snmpOidSystemBoardTable . '6'; #sc2SystemBoardSerialNumber
	my $snmpOidDesignation	= $snmpOidSystemBoardTable . '7'; #sc2SystemBoardDesignation
	my @tableChecks = (
		$snmpOidModel, $snmpOidProduct, $snmpOidSerial, $snmpOidDesignation,
	);
	if ($main::verboseTable == 611) { #PS SystemBoardTable
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIDs = ();

		addTableHeaderToVerbose("System Board Table") if ($entries);
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIDs, $1) if ($snmpKey =~ m/$snmpOidSerial.(\d+\.\d+)/);
		}
		@snmpIDs = Net::SNMP::oid_lex_sort(@snmpIDs);
		foreach my $snmpID (@snmpIDs) {
			my $designation = $entries->{$snmpOidDesignation . '.' . $snmpID};
			my $model = $entries->{$snmpOidModel . '.' . $snmpID};
			my $product = $entries->{$snmpOidProduct . '.' . $snmpID};
			my $serial = $entries->{$snmpOidSerial . '.' . $snmpID};
			{ 
				addStatusTopicToVerbose(undef,"SystemBoard", $snmpID);
				addSerialIDsToVerbose($serial, undef);
				addNameToVerbose($designation);
				addProductModelToVerbose($product, $model);
				$variableVerboseMessage .= "\n";
			}
		} # each
	}
} #primergyServerSystemBoardTable
sub primergyServerVoltageTable {
	my $snmpOidVoltageTable = '.1.3.6.1.4.1.231.2.10.2.2.10.6.3.1.'; #sc2VoltageTable
	my $snmpOidDesignation		= $snmpOidVoltageTable . '3'; #sc2VoltageDesignation
	my $snmpOidStatus		= $snmpOidVoltageTable . '4'; #sc2VoltageStatus
	my $snmpOidCurrent		= $snmpOidVoltageTable . '5'; #sc2VoltageCurrentValue mV
	my $snmpOidMin			= $snmpOidVoltageTable . '7'; #sc2VoltageMinimumLevel
	my $snmpOidMax			= $snmpOidVoltageTable . '8'; #sc2VoltageMaximumLevel
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidStatus, $snmpOidCurrent, $snmpOidMin, $snmpOidMax,
	);
	my @statusText = ( "none",
		"unknown", "not-available", "ok", "too-low", "too-high",
		"out-of-range",	"warning", "..unexpected..",
	);
	if ($optChkHardware or $optChkVoltage) {
		$msg .= " Voltage($psStatusText[$psVoltageStatus])";
	}
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if ($psVoltageStatus and $psVoltageStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if ( ($optChkSystem or $optChkHardware or $optChkVoltage) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Voltage Table") if ($entries and $verbose);
		foreach my $id (@snmpIdx) {
 			my $status  = $entries->{$snmpOidStatus . '.' . $id};
 			my $name = $entries->{$snmpOidDesignation . '.' . $id};
			my $current = $entries->{$snmpOidCurrent . '.' . $id};
  			my $min = $entries->{$snmpOidMin . '.' . $id};
 			my $max = $entries->{$snmpOidMax . '.' . $id};
			$name =~ s/[ ,;=]/_/g;
			$status = 0 if (!defined $status or $status < 0);
			$status = 8 if ($status > 8);
			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status],"Voltage",$id=~m/\d+\.(\d+)/);
				addNameToVerbose($name);
				addmVoltToVerbose($current, undef,undef, $min,$max);
				$variableVerboseMessage .= "\n";
			} 
			elsif ($notify and ($status >= 4)
			) {
				addStatusTopicToLong($statusText[$status],"Voltage",$id=~m/\d+\.(\d+)/);
				addNameToLong($name);
				addmVoltToLong($current, undef,undef, $min,$max);
				$longMessage .= "\n";
			}
		} #each
	}
} #primergyServerVoltageTable
sub primergyServerCPUTable {
	my $snmpOidCPUTable = '.1.3.6.1.4.1.231.2.10.2.2.10.6.4.1.'; #sc2CPUTable
	my $snmpOidDesignation	= $snmpOidCPUTable . '3'; #sc2cpuDesignation
	my $snmpOidStatus	= $snmpOidCPUTable . '4'; #sc2cpuStatus
	my $snmpOidModel	= $snmpOidCPUTable . '3'; #sc2cpuModelName
	my $snmpOidSpeed	= $snmpOidCPUTable . '8'; #sc2cpuCurrentSpeed MHz
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidStatus, $snmpOidModel, $snmpOidSpeed, 
	);
	my @statusText = ( "none",
		"unknown", "not-present", "ok", "disabled", "error",
		"failed", "missing-termination", "prefailure-warning", "..unexpected..",
	);
	if ($optChkHardware or $optChkCPU) {
		$msg .= " CPU($psStatusText[$psCPUStatus])";
	}
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if ($psCPUStatus and $psCPUStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkHardware or $optChkSystem or $optChkCPU) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("CPU Table") if ($entries and $verbose);
		foreach my $id (@snmpIdx) {
 			my $status  = $entries->{$snmpOidStatus . '.' . $id};
 			my $name = $entries->{$snmpOidDesignation . '.' . $id};
			my $model = $entries->{$snmpOidModel . '.' . $id};
			my $speed = $entries->{$snmpOidSpeed . '.' . $id};
			$model = undef if ($model and $name and $model eq $name);
 			$name =~ s/[ ,;=]/_/g;
			$status = 0 if (!defined $status or $status < 0);
			$status = 9 if ($status > 9);
			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status],"CPU",$id=~m/\d+\.(\d+)/);
				addNameToVerbose($name);
				addProductModelToVerbose(undef, $model);
				addKeyMHzToVerbose("Speed", $speed);
				$variableVerboseMessage .= "\n";
			} 
			elsif ($notify and ($status >= 5)
			) {
 				addStatusTopicToLong($statusText[$status],"CPU",$id=~m/\d+\.(\d+)/);
				addNameToLong($name);
				addProductModelToLong(undef, $model);
				addKeyMHzToLong("Speed", $speed);
				$longMessage .= "\n";
			}
		} #each
	}
} #primergyServerCPUTable
sub primergyServerMemoryModuleTable {
	my $snmpOidMemoryModuleTable = '.1.3.6.1.4.1.231.2.10.2.2.10.6.5.1.'; #sc2MemoryModuleTable
	my $snmpOidDesignation	= $snmpOidMemoryModuleTable . '3'; #sc2memModuleDesignation
	my $snmpOidStatus	= $snmpOidMemoryModuleTable . '4'; #sc2memModuleStatus
	my $snmpOidCapacity	= $snmpOidMemoryModuleTable . '6'; #sc2memModuleCapacity MB
	my $snmpOidType		= $snmpOidMemoryModuleTable . '9'; #sc2memModuleType
	my $snmpOidFrequency	= $snmpOidMemoryModuleTable . '14'; #sc2memModuleFrequency MHz
	my $snmpOidMaxFrequency	= $snmpOidMemoryModuleTable . '15'; #sc2memModuleMaxFrequency
	my @tableChecks = (
		$snmpOidDesignation, $snmpOidStatus, $snmpOidCapacity, $snmpOidType, 
		$snmpOidFrequency, $snmpOidMaxFrequency, 
	);
	my @statusText = ( "none",
		"unknown", "not-present", "ok", "disabled", "error",
		"failed", "prefailure-predicted", "hot-spare", "mirror", "raid",
		"hidden", "..unexpected..",
	);
	if ($optChkHardware or $optChkMemMod) {
		$msg .= " MemoryModules($psStatusText[$psMemoryStatus])";
	}
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if ($psMemoryStatus and $psMemoryStatus < 3);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkHardware or $optChkSystem or $optChkMemMod) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Memory Memory") if ($entries and $verbose);
		foreach my $id (@snmpIdx) {
 			my $status  = $entries->{$snmpOidStatus . '.' . $id};
 			my $name = $entries->{$snmpOidDesignation . '.' . $id};
			my $capacity = $entries->{$snmpOidCapacity . '.' . $id};
			my $type = $entries->{$snmpOidType . '.' . $id};
			my $frequency = $entries->{$snmpOidFrequency . '.' . $id};
			my $max = $entries->{$snmpOidMaxFrequency . '.' . $id};
			$status = 0 if (!defined $status or $status < 0);
			$status = 12 if ($status > 12);
 			$name =~ s/[ ,;=]/_/g;
			if ($verbose and $status != 2) {
				addStatusTopicToVerbose($statusText[$status],"Memory",$id=~m/\d+\.(\d+)/);
				addNameToVerbose($name);
				addKeyLongValueToVerbose("Type", $type);
				addKeyMBToVerbose("Capacity", $capacity);
				addKeyMHzToVerbose("Frequency", $frequency);
				addKeyMHzToVerbose("Frequency-Max", $max);
				$variableVerboseMessage .= "\n";
			} 
			elsif ($notify and ($status >= 5)
			) {
				addStatusTopicToLong($statusText[$status],"Memory",$id=~m/\d+\.(\d+)/);
				addNameToLong($name);
				addKeyLongValueToLong("Type", $type);
				addKeyMBToLong("Capacity", $capacity);
				addKeyMHzToLong("Frequency", $frequency);
				addKeyMHzToLong("Frequency-Max", $max);
				$longMessage .= "\n";
			}
		} #each
	}
} #primergyServerMemoryModuleTable

sub sieStComponentTable {
	# --      sieStComponentTable: 1.3.6.1.4.1.231.2.10.2.11.4.1
	my $snmpOidStComponentTable = '.1.3.6.1.4.1.231.2.10.2.11.4.1.1.'; #sieStComponentTable
	my $snmpOidStatus	= $snmpOidStComponentTable . '3'; #sieStComponentStatusValue
	my $snmpOidLastMessage	= $snmpOidStComponentTable . '4'; #sieStComponentLastErrorMessage
	my $snmpOidName		= $snmpOidStComponentTable . '5'; #sieStComponentDisplayName
	my @tableChecks = (
		$snmpOidStatus, $snmpOidLastMessage, $snmpOidName, 
	);
	my @statusText = ( "none",
		"ok", "degraded", "error", "failed", "unknown",	
		"notPresent", "notManageable", "..unexpected..",
	);
	if ($main::verboseTable == 1411) { #sieStComponentTable
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Status Component Table - Status.mib");
		foreach my $id (@snmpIdx) {
			my $name = $entries->{$snmpOidName . '.' . $id};
			my $status = $entries->{$snmpOidStatus . '.' . $id};
			my $message = $entries->{$snmpOidLastMessage . '.' . $id};
			$status = 0 if (!defined $status or $status < 0);
			$status = 8 if ($status > 8);
			next if ($status == 5);
			{
				addStatusTopicToVerbose($statusText[$status],"StatusComponent",$id);
				addKeyLongValueToVerbose("Name",$name);
				$message = undef if ($message =~ m/<<not supported>>/);
				addKeyLongValueToVerbose("LastError",$message);
				$variableVerboseMessage .= "\n";
			} 
		} # each
	} # table verbose
} #sieStComponentTable
our $resultEnv = 0;  
our $resultPower = 0;  
our $resultSystem = 0; 
our $resultMassStorage = undef;
our $resultDrvMonitor = undef;
sub sieStatusAgent {
	# Status.mib
	my $snmpOidPrefix = '1.3.6.1.4.1.231.2.10.2.11.'; #sieStatusAgent
	my $snmpOidSysStat		= $snmpOidPrefix . '2.1.0'; #sieStSystemStatusValue.0
	#my $snmpOidStSystemMessage	= $snmpOidPrefix . '2.2.0'; #sieStSystemLastErrorMessage.0
	my $snmpOidSubSysCnt	= $snmpOidPrefix . '3.2.0'; #sieStNumberSubsystems.0
	my $snmpOidSubSys	= $snmpOidPrefix . '3.1.1.'; #sieStSubsystemTable
	my $snmpOidSubSysName		= $snmpOidSubSys . '2'; #sieStSubsystemName
	my $snmpOidSubSysValue		= $snmpOidSubSys . '3'; #sieStSubsystemStatusValue
	#my $snmpOidSubSysMessage	= $snmpOidSubSys . '4' ;#sieStSubsystemLastErrorMessag
	#	... in tests LastErrorMessage was NOT set !
	my @subSysStatusText = ( 'none', 
		'ok', 'degraded', 'error', 'failed', 'unknown' );
	#my @subSystemNameText = ( "none",
	#	'Environment', 'PowerSupply', 'MassStorage', 'SystemBoard', 'Network',
	#	'DrvMonitor',
	#); # not always in this order !
	my %commonSystemCodeMap = (	1       =>      0,
					2       =>      1,
					3       =>      2,
					4       =>      2,
					5	=>	3,);
	my $srvSubSystem_cnt = undef;
	my $srvCommonSystemStatus = undef;
	my $result = undef;
	# fetch central system state
	$srvCommonSystemStatus = simpleSNMPget($snmpOidSysStat,"SystemStatus");
	#my $lastMessage = trySNMPget($snmpOidStSystemMessage,"sieStSystemLastErrorMessage");
	#$longMessage .= "LastErrorMessage=$lastMessage\n" if ($lastMessage and $lastMessage !~ m/<<not supported>>/);

	# set exit value
	$srvCommonSystemStatus = 5 if (!$srvCommonSystemStatus or $srvCommonSystemStatus > 5);
	$exitCode = $commonSystemCodeMap{$srvCommonSystemStatus};
	# get subsystem information
	$srvSubSystem_cnt = simpleSNMPget($snmpOidSubSysCnt,"SubSystemCount");
	
	for (my $x = 1; $x <= $srvSubSystem_cnt; $x++) {	
		$result = simpleSNMPget($snmpOidSubSysValue . '.' . $x,"SubsystemStatusValue"); #sieStSubsystemStatusValue	
		my $subSystemName = simpleSNMPget($snmpOidSubSysName . '.' . $x,"SubsystemName"); #sieStSubsystemName	
		next if ($result >= 5 or !defined $result or !$result);

		if ($x < 5) {
		    if (((defined $optChkEnvironment) && ($x == 1))
		    ||  ((defined $optChkPower) && ($x == 2)) 
		    ||  ((defined $optChkSystem) && ($x > 2)) 
		    ||  ((defined $optChkStorage) && ($x == 3)) 
		    ) {  
			$msg .= " $subSystemName($subSysStatusText[$result])";
			$result = 5 if ($result > 5);
			if ($x == 1) {
				$resultEnv = $commonSystemCodeMap{$result};
			} elsif ($x == 2) {
				$resultPower = $commonSystemCodeMap{$result};
			} elsif ($x > 2) {
				my $tmp_result = 0;
				$tmp_result = $commonSystemCodeMap{$result};
				$resultSystem = $tmp_result if ($resultSystem <= $tmp_result);
				if ($x == 3) {
					$resultMassStorage = $commonSystemCodeMap{$result};
				}
			}
		    }
		} # displayname
		elsif ((($x > 4) && (defined $optChkSystem) && ($main::verbose >= 1))
		|| ($result > 1 and $exitCode > 0)
		) {	# ... for verbose or on warning or error
			$msg .= " $subSystemName";
			$msg .= "($subSysStatusText[$result])";
		}	
		if ($x >= 6 and ($subSystemName eq 'DrvMonitor')) {
			$resultDrvMonitor = $commonSystemCodeMap{$result};
			if (!$main::verbose and $optChkDrvMonitor) {
				$msg .= " $subSystemName";
				$msg .= "($subSysStatusText[$result])";
			}
		}
		if ($result > $srvCommonSystemStatus) {
			my $tmp_result = 0;
			$tmp_result = $commonSystemCodeMap{$result};
			$exitCode = $tmp_result if ($exitCode <= $tmp_result);
		}
	} # for subsystems
	sieStComponentTable();
} #sieStatusAgent

sub primergyServerPerformanceTable {
	my $snmpOidPerformanceTable = '.1.3.6.1.4.1.231.2.10.2.2.10.4.3.1.'; #sc2PerformanceTable (2)
	my $snmpOidType		= $snmpOidPerformanceTable . '3'; #sc2PerformanceType
	my $snmpOidObjNr	= $snmpOidPerformanceTable . '4'; #sc2PerformanceObjectNr
	my $snmpOidName		= $snmpOidPerformanceTable . '5'; #sc2PerformanceName
	my $snmpOidValue	= $snmpOidPerformanceTable . '6'; #sc2PerformanceName
	my @tableChecks = (
		$snmpOidType, $snmpOidObjNr, $snmpOidName, $snmpOidValue, 
	);
	my @typeText = ( "none",
		"cpu", "cpu-overall", "pci-load", "pci-efficiency", "pci-transfer",
		"memory-physical", "memory-total", "memory-percent", "..unexpected..",
	);
        #            cpu:             Load of a single CPU in percent
        #            cpu-overall:     Overall CPU load in percent
        #            pci-load:        PCI bus load in percent
        #            pci-efficiency:  PCI bus efficiency in percent (100% is optimum)
        #            pci-transfer:    PCI bus transfer rate in MBytes/sec.
        #            memory-physical: Physical memory usage in MBytes
        #            memory-total:    Total memory usage (physical + virtual) in MBytes
        #            memory-percent:  Physical memory usage in percent"
	my $getInfos = 0;
	$getInfos = 1 if ($main::verbose >= 3 and $optChkHardware and !$main::verboseTable);
	$getInfos = 1 if ($optChkCpuLoadPerformance);
	$getInfos = 1 if ($optChkMemoryPerformance);
	my $totalCPU = undef;
	my @cpu = (0,0,0,0,0, 0,0,0,0,0, -1,);
	my $virtualMemMBytes = undef;
	my $physMemPercent = undef;
	my $physMemMBytes = undef;
	if ($getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidType.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Hardware Performance Values") 
			if ($entries and $main::verbose >= 3 and $optChkHardware and !$main::verboseTable);
		my $maxInd = 0;
		foreach my $id (@snmpIdx) {
 			my $type  = $entries->{$snmpOidType . '.' . $id};
 			my $name = $entries->{$snmpOidName . '.' . $id};
			my $objNr = $entries->{$snmpOidObjNr . '.' . $id};
			my $value = $entries->{$snmpOidValue . '.' . $id};
			$type = 0 if (!defined $type or $type < 0);
			$type = 9 if ($type > 9);
			if ($main::verbose >= 3 and !$main::verboseTable) {
				addStatusTopicToVerbose(undef,$typeText[$type],undef);
				addKeyLongValueToVerbose("Data","$name" . "[$objNr]") if ($type and $type == 1);
				addKeyLongValueToVerbose("Data",$name) if ($type and $type != 1);
				if ($type == 5) {
					addKeyMBToVerbose("Value", $value);
					$variableVerboseMessage .= "/sec";
				} elsif ($type and $type > 5 and $type <= 7) {
					addKeyMBToVerbose("Value", $value);
				} else {
					addKeyPercentToVerbose("Value", $value);
				}
				$variableVerboseMessage .= "\n";
			} 
			if ($optChkCpuLoadPerformance and defined $value) {
				$totalCPU = $value if ($type == 2);
				$cpu[$objNr]=$value if ($type == 1);
				$maxInd = $objNr if ($objNr > $maxInd);
			}
			if ($optChkMemoryPerformance and defined $value) {
				$physMemMBytes = $value if ($type == 6);
				$virtualMemMBytes = $value if ($type == 7 and ($name =~ m/virtual/));
				$physMemPercent = $value if ($type == 8);
			}
		} # each
		if ($maxInd) {
			$maxInd++;
			$cpu[$maxInd] = -1;
		}
	} #verbose
	if ($optChkCpuLoadPerformance) {
		$msg			.= "Total=$totalCPU" . "%" if ($totalCPU);
		addPercentageToPerfdata("Total", $totalCPU, undef, undef)
			if (!$main::verboseTable);
		my $i = 0;
		for ($i=0; $cpu[$i] != -1;$i++) {
			$msg			.= " CPU[$i]=$cpu[$i]" . "%" if (defined $cpu[$i]);
			addPercentageToPerfdata("CPU[$i]", $cpu[$i], undef, undef)
				if (!$main::verboseTable);
		} #each
		$exitCode = 0;
	} #cpu load
	if ($optChkMemoryPerformance) {
		my $warn = ($optWarningLimit?$optWarningLimit:0);
		my $crit = ($optCriticalLimit?$optCriticalLimit:0);
		$warn = undef if ($warn == 0);
		$crit = undef if ($crit == 0);
		$msg .= "Physical-Memory=$physMemPercent" . "% " if ($physMemPercent);
		$msg .= "Physical-Memory=$physMemMBytes" . "MB " if ($physMemMBytes);
		$msg .= "Virtual-Memory=$virtualMemMBytes" . "MB " if ($virtualMemMBytes);
		addPercentageToPerfdata("Physical-Memory", $physMemPercent, $warn, $crit)
			if (!$main::verboseTable);
		$exitCode = 0;
		$exitCode = 1 if ($warn and $physMemPercent > $warn);
		$exitCode = 2 if ($crit and $physMemPercent > $crit);
	} #memory
} #primergyServerPerformanceTable
sub primergyServerDriverMonitoring {
	#--      sc2DriverMonitorComponentTable:	1.3.6.1.4.1.231.2.10.2.2.10.11.1
	#--      sc2DriverMonitorMessageTable:		1.3.6.1.4.1.231.2.10.2.2.10.11.2
	my $snmpOidDrvMonCompTable = '.1.3.6.1.4.1.231.2.10.2.2.10.11.1.1.'; #sc2DriverMonitorComponentTable(3)
	my $snmpOidClass	= $snmpOidDrvMonCompTable . '1'; #sc2drvmonCompClass
	my $snmpOidName		= $snmpOidDrvMonCompTable . '3'; #sc2drvmonCompName
	my $snmpOidType		= $snmpOidDrvMonCompTable . '4'; #sc2drvmonCompType
	my $snmpOidDrvName	= $snmpOidDrvMonCompTable . '5'; #sc2drvmonCompDriverName
	my $snmpOidLocation	= $snmpOidDrvMonCompTable . '6'; #sc2drvmonCompLocationDesignation
	my $snmpOidStatus	= $snmpOidDrvMonCompTable . '8'; #sc2drvmonCompStatus
	my @tableChecks = (
		$snmpOidClass, $snmpOidName, $snmpOidType, $snmpOidDrvName, $snmpOidLocation, 
		$snmpOidStatus,
	);
	my @classText = ("none",
		"other", "software", "network",	"storage", "..unexpected..",
	);
	my @typeText = ("none",
		"other", "pci", "usb",  "..unexpected..",
	);
	my @statusText = ("none",
		"ok", "warning", "error", undef, "unknown",
		 "..unexpected..",
	);
	my $getInfos = 0;
	my $verboseInfo = 0;
	my $notokInfo = 0;
	$verboseInfo = 1 
		if ($main::verbose >= 2 or $main::verboseTable == 1111);
	$notokInfo = 1
		if (!$main::verboseTable and $resultDrvMonitor 
		and ($resultDrvMonitor == 1 or $resultDrvMonitor == 2));
	$getInfos = 1 if ($verboseInfo or $notokInfo);
	if ($getInfos) { #sc2DriverMonitorComponentTable
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("Driver Monitor Component Table");
		foreach my $id (@snmpIdx) {
			my $name = $entries->{$snmpOidName . '.' . $id};
			my $status = $entries->{$snmpOidStatus . '.' . $id};
			my $class = $entries->{$snmpOidClass . '.' . $id};
			my $type = $entries->{$snmpOidType . '.' . $id};
			my $drvName = $entries->{$snmpOidDrvName . '.' . $id};
			my $location = $entries->{$snmpOidLocation . '.' . $id};
			$status = 0 if (!defined $status or $status < 0);
			$status = 6 if ($status > 6);
			$class = 0 if (!defined $class or $class < 0);
			$class = 5 if ($class > 5);
			$type = 0 if (!defined $type or $type < 0);
			$type = 4 if ($type > 4);
			next if (($status == 4 or $status > 5) and $main::verbose < 3);
			$id =~ m/(\d+\.)\d+\.(\d+)/;
			my $shortid = $1 . $2;
			if ($verboseInfo) {
				addStatusTopicToVerbose($statusText[$status],"DrvMon",
					$shortid);
				addKeyLongValueToVerbose("Name", $name);
				addKeyValueToVerbose("Class",$classText[$class]) if ($class);
				addKeyValueToVerbose("Type",$typeText[$type]) if ($type);
				addKeyValueToVerbose("Driver",$drvName);
				addKeyLongValueToVerbose("Location",$location);
				$variableVerboseMessage .= "\n";
			} elsif ($notokInfo and ($status == 2 or $status == 3)) {
				addStatusTopicToLong($statusText[$status],"DrvMon",
					$shortid);
				addKeyLongValueToLong("Name", $name);
				addKeyValueToLong("Class",$classText[$class]) if ($class);
				addKeyValueToLong("Type",$typeText[$type]) if ($type);
				addKeyValueToLong("Driver",$drvName);
				addKeyLongValueToLong("Location",$location);
				$longMessage .= "\n";
			}
		} # each
	}
} #primergyServerDriverMonitoring

sub primergyServer {
	# Server identification (via serial number)
	my $snmpOidServerID = '.1.3.6.1.4.1.231.2.10.2.2.10.2.3.1.7.1'; #sc2UnitSerialNumber.1
	my $snmpOidSc2Test = $snmpOidServerID;

	#--------------------------------------------
	mibTestSNMPget($snmpOidSc2Test,"SV-Agent") 
		if ($optChkSystem or $optChkEnvironment or $optChkPower);
	# fetch ServerID
	primergyServerUnitTable(); # for CX400 search $serverID
	$msg = '';
	if (defined $optChkSystem and $cxChassisSerial) {
		$msg .= "- Chassis-ID=" . $cxChassisSerial . ' ';
	}
	if (defined $optChkSystem) { 
		$serverID = simpleSNMPget($snmpOidServerID,"ServerID");
		$msg .= "- ID=" . $serverID . ' -';
	}
	primergyServerStatusComponentTable();
	# fetch central system state
	sieStatusAgent();
	
	if ((defined $optChkEnvironment) 
	&& (!defined $optChkPower) && (!defined $optChkSystem) && (!defined $optChkStorage)
	and (!defined $optChkDrvMonitor)
	) {  
		$exitCode = $resultEnv;
	}
	elsif ((defined $optChkPower) 
	&& (!defined $optChkEnvironment) && (!defined $optChkSystem) && (!defined $optChkStorage)
	and (!defined $optChkDrvMonitor)
	) { 
		$exitCode = $resultPower;
	}
	elsif ((defined $optChkSystem) 
	&& (!defined $optChkEnvironment) && (!defined $optChkPower) && (!defined $optChkStorage)
	and (!defined $optChkDrvMonitor)
	) {
		$exitCode = $resultSystem;
	}
	elsif ((defined $optChkStorage)
	and (!defined $optChkEnvironment) and (!defined $optChkPower) and (!defined $optChkSystem)
	and (!defined $optChkDrvMonitor)
	) {
		if (defined $resultMassStorage) {
			$exitCode = $resultMassStorage;
		} elsif ($optChkStorage != 999) {
			$exitCode = 3;
			$msg .= "- No MassStorage information";
		}
	}
	elsif ((defined $optChkDrvMonitor)
	and (!defined $optChkEnvironment) and (!defined $optChkPower) and (!defined $optChkSystem)
	and (!defined $optChkStorage)
	) {
		if (defined $resultDrvMonitor) {
			$exitCode = $resultDrvMonitor;
		} elsif ($optChkDrvMonitor != 999) {
			$exitCode = 3;
			$msg .= "- No Driver Monitor information";
		}
	}
	if ($optChkHardware and (!defined $optChkDrvMonitor)
	and (!defined $optChkEnvironment) and (!defined $optChkPower) and (!defined $optChkSystem)
	and (!defined $optChkStorage)) {
		$exitCode = $psCPUStatus;
		addExitCode($psVoltageStatus);
		addExitCode($psMemoryStatus);
	}
	$exitCode = $psFanStatus if ($optChkEnv_Fan and defined $psFanStatus and !defined $optChkEnv_Temp);
	$exitCode = $psTempStatus if ($optChkEnv_Temp and defined $psTempStatus and !defined $optChkEnv_Fan);
	addExitCode($psTempStatus) if (defined $psFanStatus and $optChkEnv_Fan and $optChkEnv_Temp);
	# $optChkHardware
	$exitCode = $psCPUStatus if ($optChkCPU and !defined $optChkVoltage and !defined $optChkMemMod);
	$exitCode = $psVoltageStatus if ($optChkVoltage and !defined $optChkCPU and !defined $optChkMemMod);
	$exitCode = $psMemoryStatus if ($optChkMemMod and !defined $optChkCPU and !defined $optChkVoltage);
	addExitCode($psVoltageStatus) if ($optChkCPU and $optChkVoltage);
	addExitCode($psMemoryStatus) if ($optChkMemMod and ($optChkCPU or $optChkVoltage));

	primergyServerSystemInformation();
	primergyServerAgentInfo();
	primergyServer_ManagementNodeTable();
	primergyServer_ServerTable();
	
	chomp($msg);
	
	# get Powerconsumption
	primergyServerPowerConsumption();

	# process fan information
	primergyServerFanTable($resultEnv);
	
	# now check the temperatures
	primergyServerTemperatureSensorTable($resultEnv);

	# power supply
	primergyServerPowerSupplyTable($resultPower);

	#hardware ... part of SystemBoard status
	primergyServerSystemBoardTable();
	primergyServerVoltageTable();
	primergyServerCPUTable();
	primergyServerMemoryModuleTable();
	primergyServerPerformanceTable();

	#MassStorage -> RAID.mib
	if (defined $resultMassStorage) { # component MassStorage exist
		RAID();
	} # MassStorage
	if (defined $resultDrvMonitor and $optChkDrvMonitor) {
		primergyServerDriverMonitoring();
	}
} # end primergyServer

#----------- RAID functions
our $raidLDrive = undef;
our $raidPDevice = undef;
our $raidCtrl = undef;
our @raidCompStatusText = ( "none",	"ok", "prefailure", "failure", "..unexpected..",);
our @raidInterfaceText = ( "none",
	"other", "scsi", "ide",	"ieee1394", "sata",
	"sas", "fc", "..unexpected..",
);
sub RAIDsvrStatus {
	my $snmpOidSrvStatusGroup = '.1.3.6.1.4.1.231.2.49.1.3.'; #svrStatus
	my $snmpOidLogicDrive	= $snmpOidSrvStatusGroup . '1.0'; #svrStatusLogicalDrives
	my $snmpOidPhysDevice	= $snmpOidSrvStatusGroup . '2.0'; #svrStatusPhysicalDevices
	my $snmpOidController	= $snmpOidSrvStatusGroup . '3.0'; #svrStatusControllers
	my $snmpOidOverall	= $snmpOidSrvStatusGroup . '4.0'; #svrStatusOverall
	if ($main::verbose >=2 or $resultMassStorage != 0) {
		my $logicDrive = trySNMPget($snmpOidLogicDrive, "svrStatus");
		my $phyDevice = trySNMPget($snmpOidPhysDevice, "svrStatus");
		my $controller = trySNMPget($snmpOidController, "svrStatus");
		my $overall = trySNMPget($snmpOidOverall, "svrStatus");
		my $found = 0;
		if ($logicDrive or $phyDevice or $controller or $overall) {
			$found = 1;
			$raidLDrive = $logicDrive;
			$raidPDevice = $phyDevice;
			$raidCtrl = $controller;
		}
		if (defined $overall) {
			$overall = 0 if ($overall < 0);
			$overall = 4 if ($overall > 4);
		}
		if (defined $controller) {
			$controller = 0 if ($controller < 0);
			$controller = 4 if ($controller > 4);
		}
		if (defined $phyDevice) {
			$phyDevice = 0 if ($phyDevice < 0);
			$phyDevice = 4 if ($phyDevice > 4);
		}
		if (defined $logicDrive) {
			$logicDrive = 0 if ($logicDrive < 0);
			$logicDrive = 4 if ($logicDrive > 4);
		}
		if ($found and $main::verbose >=2) {
			$variableVerboseMessage .= "\n";
			addStatusTopicToVerbose($raidCompStatusText[$overall],
				"RAID -", undef);
			addComponentStatusToVerbose("Controller", $raidCompStatusText[$controller])
				if (defined $controller);
			addComponentStatusToVerbose("PhysicalDevice", $raidCompStatusText[$phyDevice])
				if (defined $phyDevice);
			addComponentStatusToVerbose("LogicalDrive", $raidCompStatusText[$logicDrive])
				if (defined $logicDrive);
			$variableVerboseMessage .= "\n";
		} elsif ($found and $resultMassStorage != 0) {
			addStatusTopicToLong($raidCompStatusText[$overall],
				"RAID -", undef);
			addComponentStatusToLong("Controller", $raidCompStatusText[$controller])
				if (defined $controller);
			addComponentStatusToLong("PhysicalDevice", $raidCompStatusText[$phyDevice])
				if (defined $phyDevice);
			addComponentStatusToLong("LogicalDrive", $raidCompStatusText[$logicDrive])
				if (defined $logicDrive);
			$longMessage .= "\n";
		}
	}
} #RAIDsvrStatus
sub RAIDsvrCtrlTable {
	#--		:		
	my $snmpOidSvrCtrlTable = '.1.3.6.1.4.1.231.2.49.1.4.2.1.'; #svrCtrlTable (1)
	my $snmpOidModel	= $snmpOidSvrCtrlTable .  '2'; #svrCtrlModelName
	my $snmpOidDescription	= $snmpOidSvrCtrlTable .  '5'; #svrCtrlBusLocationText
	my $snmpOidCache	= $snmpOidSvrCtrlTable . '13'; #svrCtrlCacheSize MB
	my $snmpOidBbuStatus	= $snmpOidSvrCtrlTable . '14'; #svrCtrlBBUStatus
	my $snmpOidStatus	= $snmpOidSvrCtrlTable . '15'; #svrCtrlStatus
	my $snmpOidInterface	= $snmpOidSvrCtrlTable . '16'; #svrCtrlInterface
	my $snmpOidSerial	= $snmpOidSvrCtrlTable . '21'; #svrCtrlSerialNo
	my $snmpOidDriverName	= $snmpOidSvrCtrlTable . '22'; #svrCtrlDriverName
	my $snmpOidDisplayName	= $snmpOidSvrCtrlTable . '25'; #svrCtrlDisplayName
	my $snmpOidHostName	= $snmpOidSvrCtrlTable . '26'; #svrCtrlHostName
	my @tableChecks = (
		$snmpOidModel, $snmpOidDescription, $snmpOidCache, $snmpOidBbuStatus, $snmpOidStatus, 
		$snmpOidInterface, $snmpOidSerial, $snmpOidDriverName, $snmpOidDisplayName, $snmpOidHostName, 
	);
	my @bbuStatusText = ( "none",
		"notAvailable",	"onLine", "onBattery", "onBatteryLow", "charging",
		"discharging", "failed", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (defined $raidCtrl and $raidCtrl > 1);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkSystem or $optChkStorage) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("RAID Controller") if ($verbose and $entries);
		foreach my $id (@snmpIdx) {
 			my $model  = $entries->{$snmpOidModel . '.' . $id};
 			my $descr = $entries->{$snmpOidDescription . '.' . $id};
			my $cache = $entries->{$snmpOidCache . '.' . $id};
			my $bbuStatus = $entries->{$snmpOidBbuStatus . '.' . $id};
			my $status = $entries->{$snmpOidStatus . '.' . $id};
			my $interface  = $entries->{$snmpOidInterface . '.' . $id};
 			my $serial = $entries->{$snmpOidSerial . '.' . $id};
			my $driver = $entries->{$snmpOidDriverName . '.' . $id};
			my $display = $entries->{$snmpOidDisplayName . '.' . $id};
			my $hostname = $entries->{$snmpOidHostName . '.' . $id};
			$status = 0 if (!defined $status or $status < 0);
			$status = 4 if ($status > 4);
			$interface = 0 if (!defined $interface or $interface < 0);
			$interface = 8 if ($interface > 8);
			$bbuStatus = 8 if ($bbuStatus and $bbuStatus > 8);

			if ($verbose) {
				addStatusTopicToVerbose($raidCompStatusText[$status],
				"RAIDCtrl", $id);
				addSerialIDsToVerbose($serial, undef);
				addKeyLongValueToVerbose("Name", $display);
				addKeyLongValueToVerbose("Description", $descr);
				addHostNameToVerbose($hostname);
				addKeyMBToVerbose("Cache", $cache);
				addKeyValueToVerbose("BBU", $bbuStatusText[$bbuStatus])
					if (defined $bbuStatus and $bbuStatus > 1);
				addKeyValueToVerbose("Interface", $raidInterfaceText[$interface])
					if (defined $interface);
				addKeyValueToVerbose("Driver", $driver);
				addProductModelToVerbose(undef, $model) if (!defined $display);
				$variableVerboseMessage .= "\n";
			} elsif ($notify and $status > 1) {
				addStatusTopicToLong($raidCompStatusText[$status],
				"RAIDCtrl", $id);
				addSerialIDsToLong($serial, undef);
				addKeyLongValueToLong("Name", $display);
				addKeyLongValueToLong("Description", $descr);
				addHostNameToLong($hostname);
				addKeyValueToLong("BBU", $bbuStatusText[$bbuStatus])
					if (defined $bbuStatus and $bbuStatus > 1);
				addProductModelToLong(undef, $model) if (!defined $display);
				$longMessage .= "\n";
			}
		} # each
	} #getInfos
} #RAIDsvrCtrlTable
sub RAIDsvrPhysicalDeviceTable {
	my $snmpOidPhysicalDeviceTable = '.1.3.6.1.4.1.231.2.49.1.5.2.1.'; #svrPhysicalDeviceTable (4)
	my $snmpOidNodel	= $snmpOidPhysicalDeviceTable .  '5'; #svrPhysicalDeviceModelName
	my $snmpOidGB		= $snmpOidPhysicalDeviceTable .  '7'; #svrPhysicalDeviceCapacity GB
	my $snmpOidType		= $snmpOidPhysicalDeviceTable .  '9'; #svrPhysicalDeviceType
	my $snmpOidInterface	= $snmpOidPhysicalDeviceTable . '11'; #svrPhysicalDeviceInterface
	my $snmpOidErrors	= $snmpOidPhysicalDeviceTable . '12'; #svrPhysicalDeviceErrors
	my $snmpOidBadBlocks	= $snmpOidPhysicalDeviceTable . '13'; #svrPhysicalDeviceNrBadBlocks
	my $snmpOidSmartStatus	= $snmpOidPhysicalDeviceTable . '14'; #svrPhysicalDeviceSmartStatus
	my $snmpOidStatus	= $snmpOidPhysicalDeviceTable . '15'; #svrPhysicalDeviceStatus
	my $snmpOidSerial	= $snmpOidPhysicalDeviceTable . '17'; #svrPhysicalDeviceSerialNumber
	#my $snmpOidExStatus	= $snmpOidPhysicalDeviceTable . '20'; #svrPhysicalDeviceStatusEx
	my $snmpOidMB		= $snmpOidPhysicalDeviceTable . '21'; #svrPhysicalDeviceCapacityMB MB
	my $snmpOidEnclosure	= $snmpOidPhysicalDeviceTable . '22'; #svrPhysicalDeviceEnclosureNumber
	my $snmpOidSlot		= $snmpOidPhysicalDeviceTable . '23'; #svrPhysicalDeviceSlot
	my $snmpOidDisplay	= $snmpOidPhysicalDeviceTable . '24'; #svrPhysicalDeviceDisplayName
	my $snmpOidPower	= $snmpOidPhysicalDeviceTable . '26'; #svrPhysicalDevicePowerStatus
	my @tableChecks = (
		$snmpOidNodel, $snmpOidGB, $snmpOidType, $snmpOidInterface, $snmpOidErrors, 
		$snmpOidBadBlocks, $snmpOidSmartStatus, $snmpOidStatus, $snmpOidSerial, 
		$snmpOidMB, $snmpOidEnclosure, $snmpOidSlot, $snmpOidDisplay, $snmpOidPower, 
	);
	my @typeText = ( "none",
		"other", "disk", "tape", "printer", "processor",
		"writeOnce", "cdRomDvd", "scanner", "optical", "jukebox",
		"communicationDevice", undef, undef, undef, undef,
		undef, undef, "host", "..unexpected..",
	); # host(98) -> 18
	my @smartStatusText = ( "none",
		"ok", "failurePredicted", "smartNotAvailable", "smartMonitoringDisabled",
		"..unexpected..",
	);
	my @statusText = ( "none",
		"unknown", "noDisk", "online", "ready",	"failed",
		"rebuilding", "hotspareGlobal",	"hotspareDedicated", "offline",	"unconfiguredFailed",
		"formatting", "dead", "..unexpected..",
	);
	my @powerText = ( "none",
		"active", "stopped", "transition", "..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (defined $raidPDevice and $raidPDevice > 1);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkSystem or $optChkStorage) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+\.\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("RAID Physical Device")	if ($verbose and $entries);
		foreach my $id (@snmpIdx) {
			my $model  = $entries->{$snmpOidNodel . '.' . $id};
 			my $gb = $entries->{$snmpOidGB . '.' . $id};
			my $type = $entries->{$snmpOidType . '.' . $id};
			my $interface  = $entries->{$snmpOidInterface . '.' . $id};
			my $errors = $entries->{$snmpOidErrors . '.' . $id};
			my $badblocks = $entries->{$snmpOidBadBlocks . '.' . $id};
			my $smart = $entries->{$snmpOidSmartStatus . '.' . $id};
			my $status = $entries->{$snmpOidStatus . '.' . $id};
 			my $serial = $entries->{$snmpOidSerial . '.' . $id};
			my $mb = $entries->{$snmpOidMB . '.' . $id};
			my $enclosure = $entries->{$snmpOidEnclosure . '.' . $id};
			my $slot = $entries->{$snmpOidSlot . '.' . $id};
			my $display = $entries->{$snmpOidDisplay . '.' . $id};
			my $power = $entries->{$snmpOidPower . '.' . $id};
			$interface = 0 if (!defined $interface or $interface < 0);
			$interface = 8 if ($interface > 8);

			$type = 0 if (!defined $type or $type < 0);
			$type = $type - 80 if ($type and $type > 90);
			$type = 19 if (($type >= 12 and $type <= 17) or $type > 19);
			$status = 0 if (!defined $status or $status < 0);
			$status = 13 if ($status > 13);
			$power = 0 if (!defined $power or $power < 0);
			$power = 4 if ($power > 4);

			if ($verbose) {
				if (defined $smart and ($smart == 1 or $smart == 2)) {
					addStatusTopicToVerbose($smartStatusText[$smart], undef, undef);
				}
				addStatusTopicToVerbose($statusText[$status], "PhysicalDevice", '');
				addKeyLongValueToVerbose("Name", $display);
				addSerialIDsToVerbose($serial, undef);
				addKeyValueToVerbose("CntErrors",$errors);
				addKeyValueToVerbose("CntBadBlocks",$badblocks);
				addKeyValueToVerbose("SmartStatus",$smartStatusText[$smart])
					if ($smart and $smart > 2 and $smart < 6);
				addKeyValueToVerbose("PowerStatus",$powerText[$power]) if ($power);
				addKeyUnsignedIntValueToVerbose("EnclosureNr",$enclosure);
				addKeyUnsignedIntValueToVerbose("SlotNr",$slot);
				if ($gb) {
					addKeyGBToVerbose("Capacity", $gb);
				} elsif ($mb) {
					addKeyMBToVerbose("Capacity", $mb);
				}
				addKeyValueToVerbose("Interface",$raidInterfaceText[$interface]) if ($interface);
				addKeyValueToVerbose("Type",$typeText[$type]) if ($type);
				addProductModelToVerbose(undef, $model) if (!defined $display);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and (      (defined $smart and $smart == 2) 
				or (defined $status and $status > 4 and (!defined $smart or $smart != 1)) )
			) {
				if (defined $smart and ($smart == 1 or $smart == 2)) {
					addStatusTopicToLong($smartStatusText[$smart], undef, undef);
				}
				addStatusTopicToLong($statusText[$status], "PhysicalDevice", '');
				addKeyLongValueToLong("Name", $display);
				addSerialIDsToLong($serial, undef);
				addKeyValueToLong("CntErrors",$errors);
				addKeyValueToLong("CntBadBlocks",$badblocks);
				addKeyUnsignedIntValueToLong("EnclosureNr",$enclosure);
				addKeyUnsignedIntValueToLong("SlotNr",$slot);
				addKeyValueToLong("Interface",$raidInterfaceText[$interface]) if ($interface);
				addKeyValueToLong("Type",$typeText[$type]) if ($type);
				addProductModelToLong(undef, $model) if (!defined $display);
				$longMessage .= "\n";
			}
		} # each
	} #getInfos
} #RAIDsvrPhysicalDeviceTable
sub RAIDsvrLogicalDriveTable {
	my $snmpOidLogicalDriveTable = '.1.3.6.1.4.1.231.2.49.1.6.2.1.'; #svrLogicalDriveTable (2)
	my $snmpOidSize		= $snmpOidLogicalDriveTable .  '4'; #svrLogicalDriveTotalSize GB
	my $snmpOidLevel	= $snmpOidLogicalDriveTable .  '5'; #svrLogicalDriveRaidLevel
	my $snmpOidStatus	= $snmpOidLogicalDriveTable . '10'; #svrLogicalDriveStatus
	my $snmpOidName		= $snmpOidLogicalDriveTable . '11'; #svrLogicalDriveName
	my $snmpOidOSDev	= $snmpOidLogicalDriveTable . '14'; #svrLogicalDriveOSDeviceName
	my $snmpOidDisplay	= $snmpOidLogicalDriveTable . '20'; #svrLogicalDriveDisplayName (try)
	my @tableChecks = (
		$snmpOidSize, $snmpOidLevel, $snmpOidStatus, $snmpOidName, $snmpOidOSDev, 
	);
	my @levelText = ( "none",
		"unknown", "raid0", "raid01", "raid1", "raid1e",
		"raid10", "raid3", "raid4", "raid5", "raid50",
		"raid5e", "raid5ee", "raid6", "concat",	"single",
		"raid60", "raid1e0", "..unexpected..",		
	);
	my @statusText = ( "none",
		"unknown", "online", "degraded", "offline", "rebuilding",
		"verifying", "initializing", "morphing", "partialDegraded",
		"..unexpected..",
	);
	my $getInfos = 0;
	my $verbose = 0;
	my $notify = 0;
	$verbose = 1 if ($main::verbose >= 2 and !$main::verboseTable);
	$notify = 1 if (defined $raidLDrive and $raidLDrive > 1);
	$getInfos = 1 if ($verbose or $notify);
	if (($optChkSystem or $optChkStorage) and $getInfos) {
		my $entries = $main::session->get_entries( -columns => \@tableChecks );
		my @snmpIdx = ();
		# store fetched data
		foreach my $snmpKey ( keys %{$entries} ) {
			#print "$snmpKey --- $entries->{$snmpKey}\n";
			push(@snmpIdx, $1) if ($snmpKey =~ m/$snmpOidStatus.(\d+\.\d+)/);
		}		
		@snmpIdx = Net::SNMP::oid_lex_sort(@snmpIdx);
		addTableHeaderToVerbose("RAID Logical Drive") if ($verbose and $entries);
		foreach my $id (@snmpIdx) {
			my $status = $entries->{$snmpOidStatus . '.' . $id};
			my $size = $entries->{$snmpOidSize . '.' . $id};
			my $level = $entries->{$snmpOidLevel . '.' . $id};
			my $name = $entries->{$snmpOidName . '.' . $id};
			my $osdev = $entries->{$snmpOidOSDev . '.' . $id};
			my $display = trySNMPget($snmpOidDisplay . '.' . $id,"svrLogicalDriveTable");
			$level = 0 if (!defined $level or $level < 0);
			$level = 18 if ($level > 18);
			$status = 0 if (!defined $status or $status < 0);
			$status = 11 if ($status > 11);

			if ($verbose) {
				addStatusTopicToVerbose($statusText[$status], 
					"LogicalDrive", $id);
				addKeyLongValueToVerbose("Name", $display) if ($display);
				addKeyLongValueToVerbose("Name", $name) if ($name and !defined $display);
				addKeyLongValueToVerbose("OSDeviceName", $osdev);
				addKeyValueToVerbose("Level", $levelText[$level]) if ($level);
				addKeyGBToVerbose("TotalSize", $size);
				$variableVerboseMessage .= "\n";
			} elsif ($notify
			and (defined $status and $status > 2)
			) {
				addStatusTopicToLong($statusText[$status], 
					"LogicalDrive", $id);
				addKeyLongValueToLong("Name", $display) if ($display);
				addKeyLongValueToLong("Name", $name) if ($name and !defined $display);
				addKeyLongValueToLong("OSDeviceName", $osdev);
				addKeyValueToLong("Level", $levelText[$level]) if ($level);
				addKeyGBToLong("TotalSize", $size);
				$longMessage .= "\n";
			}
		} # each
	} #getInfos
} #RAIDsvrLogicalDriveTable
sub RAID {
	# RAID.mib
	#my $snmpOidRaidMib = '.1.3.6.1.4.1.231.2.49.'; #fscRAIDMIB
	if ($optChkSystem or $optChkStorage) {
		RAIDsvrStatus();
		RAIDsvrCtrlTable();
		RAIDsvrPhysicalDeviceTable();
		RAIDsvrLogicalDriveTable();
	} #optChkSystem
} # RAID

#------------ MAIN PART
# get command-line parameters
#sub evaluateOptions { ---- did'nt work with icinga as function
	GetOptions(
	   "H|host=s"		=> \$optHost,
	   "C|community=s"	=> \$optCommunity,
	   "t|timeout=i"	=> \$optTimeout,
	   "v|verbose=i"	=> \$main::verbose,
	   "w|warning=i"	=> \$optWarningLimit,
	   "c|critical=i"	=> \$optCriticalLimit,
	   "chkuptime"		=> \$optChkUpTime,
	   "pq"			=> \$optPrimeQuest,
	   "primequest"		=> \$optPrimeQuest,
	   "blade"		=> \$optBlade,
	   "bladeinside"	=> \$optBladeContent,
	   "bladesrv"		=> \$optBladeSrv,
	   "bladeio"		=> \$optBladeIO,
	   "bladeio-switch"	=> \$optBladeIO_Switch,
	   "bladeio-fcpt"	=> \$optBladeIO_FCPT,
	   "bladeio-phy"	=> \$optBladeIO_Phy,
	   "bladeio-fcswitch"	=> \$optBladeIO_FCSwitch,
	   "bladeio-ibswitch"	=> \$optBladeIO_IBSwitch,
	   "bladeio-sasswitch"	=> \$optBladeIO_SASSwitch,
	   "bladeio-fsiom"	=> \$optBladeIO_FSIOM,
	   "bladekvm"		=> \$optBladeKVM,
	   "bladestore"		=> \$optBladeStore,
	   "chksystem"		=> \$optChkSystem,
	   "chkenv"		=> \$optChkEnvironment,
	   "chkenv-fan"		=> \$optChkEnv_Fan,
	   "chkenv-temp"	=> \$optChkEnv_Temp,
	   "chkpower"		=> \$optChkPower,
	   "chkhardware"	=> \$optChkHardware,
	   "chkcpu"		=> \$optChkCPU,
	   "chkvoltage"		=> \$optChkVoltage,
	   "chkmemmodule"	=> \$optChkMemMod,
	   "chkdrvmonitor"	=> \$optChkDrvMonitor,
	   "chkstorage"		=> \$optChkStorage,
	   "chkcpuload"		=> \$optChkCpuLoadPerformance,
	   "chkmemperf"		=> \$optChkMemoryPerformance,
	   "V|version"		=> \$optShowVersion,
	   "h|help"		=> \$optHelp,
	   "user=s"		=> \$optUserName,
	   "authkey=s"		=> \$optAuthKey,
	   "authpassword=s"	=> \$optAuthPassword,
	   "authprot=s"		=> \$optAuthProt,
	   "privkey=s"		=> \$optPrivKey,
	   "privpassword=s"	=> \$optPrivPassword,
	   "privprot=s"		=> \$optPrivProt,
	) or pod2usage({
		-msg     => "\n" . 'Invalid argument!' . "\n",
		-verbose => 1,
		-exitval => 3
	});

	# check command-line parameters
	pod2usage(
		-verbose => 2,
		-exitval => 0,
	) if ((defined $optHelp) || !$argvCnt);

	pod2usage(
		-msg		=> "\n$0" . ' - version: ' . $version . "\n",
		-verbose	=> 0,
		-exitval	=> 0,
	) if (defined $optShowVersion);

	pod2usage(
		-msg		=> "\n" . 'Missing host address !' . "\n",
		-verbose	=> 1,
		-exitval	=> 3
	) if ($optHost eq '');

	pod2usage(
		-msg		=> "\n" . 'Unable to handle IPv6 host address !' . "\n",
		-verbose	=> 0,
		-exitval	=> 3
	) if ($optHost =~ m/.*:.*:.*/); # one ':' might be portnumber - two hints on IPv6

	if (!defined $optUserName and !defined $optCommunity) {
		$optCommunity = 'public'; # same default as other snmp nagios plugins
	}

	# after readin of options
	if ((!defined $optChkSystem) and (!defined $optChkEnvironment) and (!defined $optChkPower)
	and  (!defined $optChkHardware) and (!defined $optChkStorage) and (!defined $optChkDrvMonitor)
	and (!defined $optChkCpuLoadPerformance) and (!defined $optChkMemoryPerformance)
	and (!defined $optChkEnv_Fan) and (!defined $optChkEnv_Temp) 
	and (!defined $optChkCPU) and (!defined $optChkVoltage) and (!defined $optChkMemMod)
	) {
		$optChkSystem = 999;
		$optChkEnvironment = 999;
		$optChkPower = 999;
		$optChkHardware = 999 if ($optPrimeQuest);
		# exotic values if somebody needs to see if an optchk was explizit set via argv or if this 
		# is default
	}
	if (  (defined $optBladeIO_Switch)   || (defined $optBladeIO_FCPT)     || (defined $optBladeIO_Phy) 
	   || (defined $optBladeIO_FCSwitch) || (defined $optBladeIO_IBSwitch) || (defined $optBladeIO_SASSwitch)
	   || (defined $optBladeIO_FSIOM)
	   ) 
	{
		$optBladeIO = 999;
	}
	if ((defined $optBladeSrv) || (defined $optBladeIO) || (defined $optBladeKVM) || (defined $optBladeStore)) {
		$optBladeContent = 999 if (!defined $optBladeContent);
	}
	if (defined $optBladeContent and $optBladeContent != 999) { # all blades inside
		$optBladeSrv = 999 if (!defined $optBladeSrv);
		$optBladeIO = 888 if (!defined $optBladeIO); # ! not 999 !
		$optBladeKVM = 999 if (!defined $optBladeKVM);
		$optBladeStore = 999 if (!defined $optBladeStore);
	}
	if ($main::verbose > 100) {
		$main::verboseTable = $main::verbose;
		$main::verbose = 0;
	}
#} #evaluateOptions

# set timeout
local $SIG{ALRM} = sub {
	#### TEXT LANGUAGE AWARENESS
	print 'UNKNOWN: Timeout' . "\n";
	exit(3);
};
alarm($optTimeout);

# connect to SNMP host
if (defined $optUserName) {
	$variableVerboseMessage .= "--- SNMPv3 ---\n" if ($main::verbose >= 4);
	($main::session, $error) = Net::SNMP->session(
		-hostname	=> $optHost,
		-version        => 'snmpv3',
		-username	=> $optUserName,
		-authkey	=> $optAuthKey,
		-authpassword	=> $optAuthPassword,
		-authprotocol	=> $optAuthProt || 'md5',
		-privkey	=> $optPrivKey,
		-privpassword	=> $optPrivPassword,
		-privprotocol	=> $optPrivProt || 'des',
	);
	$variableVerboseMessage .= "--- SNMPv3 Error=$error\n" 
		if ((!defined $main::session) && ($main::verbose >= 4 and $main::verbose <= 20));
	#### QUESTION: Should I switch automaticaly to SNMP1 in case of errors ???
} # else {
if (!defined $main::session) {
	$variableVerboseMessage .= "--- SNMPv1/v2 ---\n" if ($main::verbose >= 4 and $main::verbose <= 20);
	($main::session, $error) = Net::SNMP->session(
		Hostname	=> $optHost,
		Community	=> $optCommunity || 'public',
	);
}
finalize(3, $state[3], "SNMP::session(): $error") unless $main::session;

if (defined $optChkUpTime) {
	RFC1213sysinfoUpTime();
} elsif (defined $optPrimeQuest) {
	primequest();
} elsif (defined $optBlade) {
	primergyManagementBlade();
} elsif (defined $optBladeContent) {
	if (defined $optBladeSrv) {
		primergyServerBlades();
	}
	if (defined $optBladeIO) {
		primergyIOConnectionBlades();
	}
	if (defined $optBladeKVM) {
		primergyKVMBlades();
	}
	if (defined $optBladeStore) {
		primergyStorageBlades();
	}
} elsif ($optChkCpuLoadPerformance or $optChkMemoryPerformance) {
	my $snmpOidServerID = '.1.3.6.1.4.1.231.2.10.2.2.10.2.3.1.7.1'; #sc2UnitSerialNumber.1
	my $snmpOidSc2Test = $snmpOidServerID;
	mibTestSNMPget($snmpOidSc2Test,"SV-Agent");
	primergyServerPerformanceTable();
	primergyServerSystemInformation();
} else {
	primergyServer();
}

# close SNMP session
$main::session->close;
chop($error);

# output to nagios
finalize(
	$exitCode, 
	$state[$exitCode], 
	$msg,
	($longMessage eq '' ? '' : "\n" . $longMessage),
	($main::verbose >= 2 or $main::verboseTable) ? "\n" . $variableVerboseMessage: '',
	($performanceData ? "\n |" . $performanceData : ''),
);
################ EOSCRIPT


