#!/usr/bin/perl

# author: Hubert Faulhaber <hubert.faulhaber@noris.de>
# adapted from IBM Bladecenter plugin from Eric Schultz
# what:    monitor various aspects of an FSC Bladecenter
# license: GPL - http://www.fsf.org/licenses/gpl.txt
#

use strict;
require 5.6.0;
use lib qw( /usr/lib/nagios/plugins );
use utils qw(%ERRORS $TIMEOUT &print_revision &support &usage);
use Net::SNMP;
use Getopt::Long;
use vars qw/$exit $opt_version $opt_timeout $opt_help $opt_host $opt_community $opt_port $opt_mountpoint $snmp_session $PROGNAME $TIMEOUT/;

$PROGNAME      = "check_snmp_FSC_Bladecenter.pl";
$opt_host      = undef;
$opt_community = 'public';
$opt_port      = 161;
$exit          = 'OK';

# =========================================================================== #
# =====> MAIN
# =========================================================================== #
process_options();

alarm( $TIMEOUT ); # make sure we don't hang Nagios

my $snmp_error;
($snmp_session,$snmp_error) = Net::SNMP->session(
		-version => 'snmpv1',
		-hostname => $opt_host,
		-community => $opt_community,
		-port => $opt_port,
		);

my $res = "OK";
my $err = "";
my $ok = "";
my $oid;
my $mod_res;
my @ok_states;
my $count;
my $oid_prefix = ".1.3.6.1.4.1."; #Enterprises
$oid_prefix .= "7244.1.1.1."; #FSC Bladecenter

$|=1;

# get Overall Status
@ok_states = (2);
&getState("3.1.5.0", "Overall");
&getState("3.2.1.0", "Power");
&getState("3.3.4.0", "Fans");
&getState("3.4.2.0", "Temp");

# get Server Blade(s) Status
# 1 - unknown  2 - ok       3 - not-present
# 4 - error    5 - critical 6 - standby
@ok_states = (2,3,6);
#$count = &getSubsysCount("4.2.1.1.2");
for (my $i = 1; $i <= 10; $i++) {
  &getState("4.2.1.1.2.".$i, "Srv$i");
}

# get Management Blade(s) Status
# 1 - unknown  2 - ok       3 - not-present
# 4 - error    5 - critical 6 - standby
@ok_states = (2,6);
$count = &getSubsysCount("2.1.1.2");
for (my $i = 1; $i <= $count; $i++) {
  &getState("2.1.1.2.".$i, "Mgmt$i");
}

# get Switch Blade(s) Status
# 1 - unknown  2 - ok      3 - not-present 4 - error
# 5 - critical 6 - standby 7 - present
@ok_states = (2,6,7);
$count = &getSubsysCount("5.1.1.2");
for (my $i = 1; $i <= $count; $i++) {
  &getState("5.1.1.2.".$i, "Sw$i");
}

$snmp_session->close;
alarm( 0 ); # we're not going to hang after this.

if ($res eq "CRITICAL") { print "$res: $err\n"; }
else { print "$res: $ok\n"; }
exit $ERRORS{$res};


# =========================================================================== #
# =====> Sub-Routines
# =========================================================================== #

sub process_options {
	Getopt::Long::Configure( 'bundling' );
	GetOptions(
			'V'     => \$opt_version,       'version'     => \$opt_version,
			'h'     => \$opt_help,          'help'        => \$opt_help,
			'H:s'   => \$opt_host,          'hostname:s'  => \$opt_host,
			'C:s'   => \$opt_community,     'community:s' => \$opt_community,
			'p:i'   => \$opt_port,          'port:i'      => \$opt_port,
			'o:i'   => \$TIMEOUT,           'timeout:i'   => \$TIMEOUT,
		  );
	if ( defined($opt_version) ) { local_print_revision(); }
	if ( !defined($opt_host) || defined($opt_help) ) {
		print_help();
		exit $ERRORS{UNKNOWN};
		}
	}

sub local_print_revision { print_revision( $PROGNAME, '$Revision: 1.0 $ ' ); }

sub print_usage { print "Usage: $PROGNAME -H <host> -C <snmp_community>\n"; }

sub print_help {
	local_print_revision();
	print "SNMP FSC Bladecenter plugin for Nagios\n\n";
	print_usage();
print <<EOT;
	-h, --help
		print this help message
	-H, --hostname=HOST
		name or IP address of host to check
	-C, --community=COMMUNITY NAME
		community name for the host's SNMP agent
EOT
}

# params: oid, ok states, module name
sub getState {
	my $mod_res = SNMP_getvalue($snmp_session, $oid_prefix.$_[0]);
	if (grep $_ == $mod_res, @ok_states) { $ok .= "$_[1] "; }
	else { $res = "CRITICAL"; $err .= "$_[1] "; }
}

sub getSubsysCount {
	my $result = `snmpwalk -v2c -c $opt_community $opt_host $oid_prefix$_[0]`;
	if ($? != 0) { 
		print "UNKNOWN: Error executing snmpwalk!\n";
		exit $ERRORS{"UNKNOWN"};
	}
	$result =~ /.*\.(.+)\s=.*$/;
	return($1);
}

sub SNMP_getvalue{
	my ($snmp_session,$oid) = @_;

	my $res = $snmp_session->get_request(
			-varbindlist => [$oid]);
	
	if(!defined($res)){
		print "ERROR: ".$snmp_session->error."\n";
		exit $ERRORS{"UNKNOWN"};
	}
	
	return($res->{$oid});
}
