package noris::ManageUsers::Dovecot;

use 5.010;
use strict;
use utf8;
use Data::Dump qw(pp);     # depends: libdata-dump-perl
use Log::Log4perl qw(get_logger :levels); # depends: liblog-log4perl-perl

use open IO => ':locale';

use Moose;                         # depends libmoose-perl
use Moose::Util::TypeConstraints;  # depends libmoose-perl
use File::Path qw/remove_tree/;
use File::Basename qw/basename/;

has 'root_folder'  => ( is => 'ro', isa => 'Str', default => '/var/mail/mailboxes/' );

my $logger;

sub BUILD {
    my $self = shift;

    $logger = get_logger(__PACKAGE__);

    $logger->debug( "Script started " );
}

# Get User List
sub list_users {
    my $self = shift;

    $logger->debug("List Users called");

    # ls aufs VZ und alle User holen
    my @user = map basename($_), glob $self->root_folder . '/*/*';

    $logger->debug("List Users: User Mailboxes: " . pp(@user) );
    return @user;
}

# Create a new User
sub add_user {
    my ( $self, $user ) = @_;

    $logger->info("Creation of user '$user' requested; this is not implemented, so not doing anything.");
    # NOT IMPLEMENTED
    #$logger->debug("Add User called with user '$user'");
    #$logger->debug("Trying to create User '$user'");

    return 1;
}

# Delete an existing User
sub delete_user {
    my ( $self, $user ) = @_;

    my $errors = 0;
    $logger->debug("Delete User called with user '$user'");
    unless ( length $user ) {
        $logger->error('Delete User called with empty user name');
        return;
    }
    if ( $user =~ m{([\$\s/'"\0])}) {
        $logger->error("Potentially dangerous character '$1' found in user name '$user'");
        return;
    }
    my $path = join '/', $self->root_folder, substr($user, 0, 1), $user;
    unless (-e $path) {
        $logger->error("Path '$path' does not exist");
        return;
    }
    unless (-d $path) {
        $logger->error("Path '$path' is not a directory");
        return;
    }

    $logger->debug("Delete User '$user' causes deletion of path '$path'");
    remove_tree($path, { error => \my $err });
    if (@$err) {
        for my $diag (@$err) {
            $errors++;
            my ($file, $message) = %$diag;
            if ($file eq '') {
                $logger->error("Error while deleting user '$user': $message");
            }
            else {
                $logger->error("Error while deleting user '$user', file '$file': $message");
            }
        }
    }

    if (-e $path) {
        $logger->error("Cannot delete user '$user': Directory '$path' still exists");
        $errors++;
    }
    else {
        $logger->info("User '$user' deleted!");
    }
    return !$errors;
}

no Moose;
__PACKAGE__->meta->make_immutable;

1;
__END__

=encoding UTF-8

=head1 NAME

noris::ManageUsers::Dovecot - Dovecot Modul für manage-users

=head1 BESCHREIBUNG

Dieses Modul ist Teil des manage-users Pakets und ist zuständig für die 
Administration der Benutzer in einem Dovecot-Server.

=head1 OPTIONEN

=over 4

=item root_folder String

Der Root Folder wo die User-Verzeichnisse liegen.

=back

=head1 PUBLIC METHODEN

=over 4

=item new()

Diese Funktion wird als erstes aufgerufen und der werden alle notwendigen
Parameter übergeben.

    use noris::ManageUsers::Dovecot;

    my %BackendArg = ('username'   => 'cyradm',
                      'password'   => '123456',
                      'host'       => 'localhost',
                      'subfolders' => 'Drafts,Sent,Trash',
                     );
    my $BackendObject = noris::ManageUsers::Dovecot->new(%BackendArg);

=item list_users()

list_users liefert eine Liste allen vorhandenen Usern im Dovecot Server zurück.

    my @user_list = $BackendObject->list_users()

=item add_user()

MOMENTAN NICHT IMPLEMENTIERT

add_user wird der Username des zu generierenden Benutzers übergeben.

MOMENTAN NICHT IMPLEMENTIERT

    $BackendObject->add_user('username');

=item delete_user()

delete_user wird der zu löschende User übergeben und löscht alle rekursiv
alle Verzeichnisse des jeweiligen Users.

    $BackendObject->delete_user('username');

=back

=head1 PRIVATE METHODEN

=over 4

=item BUILD

Mosse Framework Function.

=back

=head1 AUTOR

 Stelios Gikas <stelios.gikas@noris.net>

