<?php
if (!defined('MEDIAWIKI'))
    die();
/**
 * A hook to allow specify an initial number for <h1>, together with an optional title prefix.
 * This allows subsequently consistent numbering <h1> through <h5> (for if it's supported in a future),
 * regardless autonumbering is enabled or not.
 * Usage: <H1_CSS chapter="5" prefix="Chapter" /> (remember that "prefix" is optional),
 * which starts numbering from "5", and prefix it with the word "Chapter",
 * so a wiki code of "=My custom chapter=" would be shown as "Chapter 5.
 * My custom chapter", and the table of contents will show the numbering starting from 5.
 * The example is exactly the same that <H1_CSS full="Chapter 5" />, which means that
 * the parameter "full" is parsed, assigning the last word to the chapter, and the rest, to the prefix.
 * As an extra option, you can enclose the chapter title, so <H1_CSS full="Chapter 5">My custom chapter</H1_CSS>
 * allows to do it all in one line.
 *
 * This is based in CSS 2 counters, and has been proved successfully in both Firefox and konqueror.
 * IE shows no numbering at all, though the chapter prefix is shown ok.
 *
 * @author Oscar Manuel Gómez Senovilla <omgs@escomposlinux.org>
 * @copyright Copyright © 2007, Oscar Manuel Gómez Senovilla
 * @license http://www.gnu.org/copyleft/gpl.html GNU General Public License 2.0 or later
 */

$GLOBALS['wgExtensionFunctions'][]             = array(
    'H1Hook',
    'setup'
);
$GLOBALS['wgExtensionCredits']['parserhook'][] = array(
    'name' => 'H1 CSS',
    'description' => 'Parser hook to init H1 tags at a custom number using CSS 2',
    'author' => 'Oscar Manuel Gómez Senovilla'
);

class H1Hook {

    public static function setup() {
        $GLOBALS['wgParser']->setHook('H1_CSS', array(
            'H1Hook',
            'parse'
        ));
    }

    /**
     * @param string $content wiki content, maybe not necessary
     * @param array $args, being processed "chapter", "prefix" and "full".
     */
    public static function parse($content, array $args, Parser &$parser) {
        $h1 = htmlspecialchars(trim(Sanitizer::checkCss($content)));

        // Force disable numbering, to use our numbering
        $parser->mOptions->setNumberHeadings(false);

        if (!$args['full']) {
            $chapter = ($args['chapter'] ? $args['chapter'] - 1 : "0");
            $prefix  = ($args['prefix'] ? trim($args['prefix']) . " " : "");
        } else {
            // We strip the string, to assign the last word to $chapter,
            // and the rest, to $prefix
            $full = explode(" ", $args['full']);
            if ($numargs = count($full)) {
                // Let's preserve explicit chapter and prefix parameters, for backwards compatibility
                if ($args['chapter']) {
                    $chapter = $args['chapter'] - 1;
                } else {
                    $chapter = $full[$numargs - 1] - 1;
                }
                if ($args['prefix']) {
                    trim($args['prefix']) . " ";
                } else {
                    unset($_prefix);
                    for ($c = 0; $c < ($numargs - 1); $c++) {
                        $_prefix .= " " . $full[$c];
                    }
                    $prefix = trim($_prefix) . " ";
                }
            }
        }

        // Add inline style to head secion
        $parser->mOutput->addHeadItem(<<<EOT
<style type="text/css">
/*<![CDATA[*/

#bodyContent {
    counter-reset: chapter {$chapter} toc_chapter {$chapter};
}

#bodyContent h1:before{
    content: "{$prefix}" counter(chapter) ". ";
}

#bodyContent h1 {
    counter-reset: sub_h2;
    counter-increment: chapter;
}

#bodyContent h2 span.mw-headline:before {
    content: counter(chapter) "." counter(sub_h2) " ";
}

#bodyContent h2 {
    counter-increment: sub_h2;
    counter-reset: sub_h3;
}

#bodyContent h3 span.mw-headline:before{
    content: counter(chapter) "." counter(sub_h2) "." counter(sub_h3) " ";
}

#bodyContent h3  {
    counter-increment: sub_h3;
    counter-reset: sub_h4;
}

#bodyContent h4 span.mw-headline:before{
    content: counter(chapter) "." counter(sub_h2) "." counter(sub_h3) "." counter(sub_h4) " ";
}

#bodyContent h4  {
    counter-increment: sub_h4;
    counter-reset: sub_h5;
}

/* While there's no support for "=====", no span here */
#bodyContent h5:before{
    content: counter(chapter) "." counter(sub_h2) "." counter(sub_h3) "." counter(sub_h4) "." counter(sub_h5) " ";
}

#bodyContent h5  {
    counter-increment: sub_h5;
}

.tocnumber { display: none; }

#bodyContent li.toclevel-1 .toctext:before {
    content: counter(toc_chapter) ". ";
}

#bodyContent li.toclevel-1  {
    counter-increment: toc_chapter;
    counter-reset: toc_h2;
}

#bodyContent li.toclevel-2 .toctext:before {
    content: counter(toc_chapter) "." counter(toc_h2) " ";
}

#bodyContent li.toclevel-2  {
    counter-increment: toc_h2;
    counter-reset: toc_h3;
}

#bodyContent li.toclevel-3 .toctext:before {
    content: counter(toc_chapter) "." counter(toc_h2) "." counter(toc_h3) " ";
    counter-increment: toc_h3;
}

#bodyContent li.toclevel-3 {
    counter-reset: toc_h4;
}

#bodyContent li.toclevel-4 .toctext:before {
    content: counter(toc_chapter) "." counter(toc_h2) "." counter(toc_h3) "." counter(toc_h4) " ";
    counter-increment: toc_h4;
}

#bodyContent li.toclevel-4 {
    counter-reset: toc_h5;
}

#bodyContent li.toclevel-5 .toctext:before {
    content: counter(toc_chapter) "." counter(toc_h2) "." counter(toc_h3) "." counter(toc_h4) "." counter(toc_h5) " ";
    counter-increment: toc_h5;
}

/*]]>*/
</style>
EOT
            );

        // If there's a content in the call, return it as h1,
        if (!empty($h1)) {
            return "<h1>$h1</h1>";
        }
    }
}
