package noris::IO::AutoEncoding;

use strict;
use warnings;
use utf8;

use base 'Exporter';

our %EXPORT_TAGS = (
    'all' => [
        qw(
          decode_auto
          encode_auto
          encoding
          set_encoding
          )
    ]
);
our @EXPORT_OK   = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT      = qw( );

use constant TAG2HANDLES => {
    ':auto'   => [ \*STDIN, \*STDOUT, \*STDERR ],
    ':err'    => [ \*STDERR ],
    ':in'     => [ \*STDIN ],
    ':out'    => [ \*STDOUT, \*STDERR ],
    ':stderr' => [ \*STDERR ],
    ':stdin'  => [ \*STDIN ],
    ':stdout' => [ \*STDOUT ],
};
our $VERSION = '1.0';

sub _decode_argv() { @ARGV = map decode_auto($_), @ARGV }

sub import {
    my $package = shift;
    if (@_) {
        my @imports = grep {
            if ( defined( my $handles = TAG2HANDLES->{$_} ) )
            {
                set_encoding(@$handles);
                _decode_argv() if $_ eq ':auto';
                '';
            }
            elsif ( $_ eq '@ARGV' ) { _decode_argv(); '' }
            else                    { 1 }
        } @_;
        $package->export_to_level( 1, $package, @imports ) if @imports;
    }
    else { set_encoding( @{ TAG2HANDLES->{':out'} } ) }
}

{
    my $encoding;

    sub encoding() {

        #10618338 — korrektes Encoding für die aktuelle Umgebung ermitteln
        #$encoding ||= defined $ENV{LC_ALL} && $ENV{LC_ALL} =~ /utf-?8/i
        #  || !defined $ENV{LC_ALL}
        #  && defined $ENV{LANG}

        #  # Hier sollte latin9 stehen. Grund noris::NetSaint Modul im Destroy
        #  # Modus erkennt ISO-8859-15 nicht. Siehe auch #10074868
        #  && $ENV{LANG} =~ /utf-?8/i ? 'UTF-8' : 'latin9';

        require encoding;
        return encoding::_get_locale_encoding();

    }
}

sub decode_auto {
    require Encode;
    Encode::decode( encoding(), shift );
}

sub encode_auto {
    require Encode;
    local *Encode::find_alias =
      \&Encode::Alias::find_alias;    # Workaround für #10045028
    Encode::encode( encoding(), shift );
}

sub set_encoding {
    require Encode;
    my $encoding = encoding();
    local *Encode::find_alias =
      \&Encode::Alias::find_alias;    # Workaround für #10045028
    binmode( $_, ":raw :encoding($encoding)" ) for @_;
}

1;

__END__

=encoding utf8

=head1 NAME

noris::IO::AutoEncoding - Autoencoding IO messages

=head1 SYNOPSIS

    use noris::IO::AutoEncoding;

=head1 DESCRIPTION

Das Skript versucht anhand der eingestellten Locale herauszufinden, welches das
passende Encoding für die jeweilige Umgebung ist.
Wenn UTF-8 vorgesehen ist, wird auch dieses angewendet, ansonsten von
ISO-8859-15 ausgegangen.

=head1 EXPORTIERBARE FUNKTIONEN

=over 4

=item decode_auto( Daten )

Gibt eine passend zur Umgebung dekodierte Version der übergebenen Daten zurück.

=item encode_auto( Zeichenkette )

Gibt eine zur Umgebung passend kodierte Version der übergebenen Zeichenkette
zurück.

=item encoding()

Gibt den Namen des zur Umgebung passenden Encodings zurück.

=item set_encoding( <Kanal>+ )

Schaltet automatische Kodierung für die angegebenen Kanäle an.

=back

=head1 WEITERE IMPORT-FUNKTIONALITÄT

=over 4

=item @ARGV

Die in C<@ARGV> enthaltenen Elemente werden in-place
L<auto-dekodiert|/decode_auto>.

=item :auto

Standardeingabe, Standardausgabe und StandardError werden auf automatische
Dekodierung bzw Kodierung geschaltet und C<@ARGV> dekodiert.

=item :err

=item :stderr

StandardError wird auf automatische Kodierung geschaltet.

=item :in

=item :stdin

Standardeingabe wird auf automatische Dekodierung geschaltet.

=item :out

Standardausgabe und StandardError werden auf automatische Kodierung geschaltet.

Das ist das Default-Verhalten, wenn das Modul ohne explizite Import-Liste
L<eingebunden|perlfunc/use> wird.

=item :stdout

Standardausgabe wird auf automatische Kodierung geschaltet.

=back

