#!/bin/sh
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
# This software is supplied under the terms of a license agreement or
# nondisclosure agreement with Dell Inc. and may not
# be copied or disclosed except in accordance with the terms of that
# agreement.
#
# Copyright (c) 2000-2005 Dell Inc. All Rights Reserved.
#
# Module Name:
#
#   uninstall.sh
#
#
# Abstract/Purpose:
#
#   This script uninstalls all the installed Systems Management
#   rpm packages from the system
#
#
# Environment:
#
#   Linux
#
###############################################################################

# ensure sbin utilities are available in path, so that su will also work
export PATH=/usr/kerberos/sbin:/usr/local/sbin:/sbin:/usr/sbin:$PATH

# global variables
DELL_OPENMANAGE_PREFIX="srvadmin"
FORCE_UNINSTALL=0


###############################################################################
# Function : Uninstall
#
# Uninstall all the installed packages
#
###############################################################################
function Uninstall {
    # get all the Server Administrator packages, last one first
    # based on time stamp
  
   #if only instsvc-drivers is installed,the below statement fails & instsvc-drivers will not be uninstalled
   # SM_PKGS=`rpm -qa --last | grep "^$DELL_OPENMANAGE_PREFIX" | sed "s/ .*$//"`
   
    SM_PKGS=`rpm -qa --last | egrep "^$DELL_OPENMANAGE_PREFIX|^instsvc" | sed "s/ .*$//"`

    UNINSTALL_STATUS=0

    if [ -n "$SM_PKGS" ] ; then
        # check for forced uninstall
        userinput="y"
        if [ ${FORCE_UNINSTALL} == 0 ];
        then
            # This is not forced, ask user if ok to uninstall
            echo "Are you sure you want to uninstall?"
            # prompt and get response
            echo -n "   Press ('y' for yes | 'Enter' to exit): " 1>&2
            read RESP
            # remove leading/trailing spaces
            userinput=`echo "${RESP}" | sed 's#^[[:space:]]*##; s#[[:space:]]*$##'`
        fi

        # test for ok to uninstall
        #fix for DF 39092 - added a '$' in the regexp - BK
        if echo "${userinput}" | grep -iq "^y$" ; then

            echo "   Uninstalling Server Administrator."
            LIMIT_ATTEMPTS=30
            NUM_PKGS=`echo "${SM_PKGS}" | wc -l | sed "s/ //g" | egrep "^[0-9]+$"`
            [ $? == 0 ] && LIMIT_ATTEMPTS=${NUM_PKGS}

            # erase the packages in the correct order, exit after a finite number of attempts
            # if ever get stuck because of bad scripts, etc.,
            NUM_ATTEMPTS=0
            while [ -n "$SM_PKGS" -a ${NUM_ATTEMPTS} -lt ${LIMIT_ATTEMPTS} ] ; do
                NUM_ATTEMPTS=`expr ${NUM_ATTEMPTS} + 1`
                for PKG in $SM_PKGS ; do
                    # time stamp can be wrong sometimes, so test it
                    rpm -e $PKG --test >/dev/null 2>&1
                    if [ $? == 0 ] ; then
                        rpm -e $PKG || UNINSTALL_STATUS=1
                        SM_PKGS=`echo "$SM_PKGS" | grep -v $PKG`
		    else 
			UNINSTALL_STATUS=1
                    fi
		
                done
            done
            # legacy hapi support for DUP 1.0
            rpm -q instsvc-drivers >/dev/null 2>&1
            if [ $? == 0 ] ; then
              rpm -e instsvc-drivers >/dev/null 2>&1
            fi
	    # If time stamp is wrong, uninstall status might have been set incorrectly.
            if [ ${UNINSTALL_STATUS} == 1 ] && (rpm -qa |grep "$DELL_OPENMANAGE_PREFIX" > /dev/null 2>&1) ; then
                echo "Failed to uninstall the following packages:"
                rpm -qa | grep "$DELL_OPENMANAGE_PREFIX"
            else
                echo "   Uninstalled Successfully."
            fi
        fi
    else
        echo "No Server Administrator RPM packages are installed."
    fi
}

###############################################################################
# Function : ErrorMsg
#
# Sends an error message to stderr
###############################################################################
ErrorMsg()
{
  MSG=${1}
  echo "${MSG}" 1>&2
}

###############################################################################
# Function : GetTemp
#
# returns a safe temporary filename (respecting any $TMP directory given)
###############################################################################
GetTemp()
{
  GETTEMPFILE=`mktemp ${TMP:-/var/tmp}/ominstall.XXXXXXX`
  [ $? -ne 0 ] && ErrorMsg "error: cannot create temp file, exiting..." && exit 1
  chmod 600 ${GETTEMPFILE} && chown root.root ${GETTEMPFILE}
  echo "${GETTEMPFILE}"
  return 0
}

###############################################################################
# Function : GetRegVal
#
# Retrieve the value from a filename and registry key
###############################################################################
GetRegVal()
{
  FILE="${1}"
  KEY="${2}"
  [ ! -f "${FILE}" ] && echo "" && return 1
  GetVal "`grep -i "^[[:space:]]*${KEY}[[:space:]]*=" ${FILE}`"
  return 0
}

###############################################################################
# Function : GetRegVal
#
# Retrieve the value portion from a key=value pair
###############################################################################
GetVal()
{
  PAIR="${1}"
  echo "${PAIR}" | sed 's#^[^=]*=##; s#^[[:space:]]*##; s#[[:space:]]*$##'
}

###############################################################################
# Function : GetKey
#
# Retrieve the key portion from a key=value pair
###############################################################################
GetKey()
{
  PAIR="${1}"
  echo "${PAIR}" | sed 's#=.*$##; s#^[[:space:]]*##; s#[[:space:]]*$##'
}

###############################################################################
# Function : RemoveRegPair
#
# Remove a key from registry
# if a pair is provided, the value is stripped and the key alone is used
# if the file is empty after the deletion, the file is removed as well
###############################################################################
RemoveRegPair()
{
  FILE="${1}"
  shift

  [ ! -f "${FILE}" ] && return 0

  for PAIR in ${*} ;
  do
    TMP_KEY=`GetKey "${PAIR}"`

    # strip old from the regentry
    TEMPFILE=`GetTemp`
    grep -iv "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} > ${TEMPFILE}
    sort ${TEMPFILE} > ${FILE} ||
        ErrorMsg "unable to update ${FILE}"
    rm -f ${TEMPFILE}
  done

  # if now empty, remove the file
  [ ! -s "${FILE}" ] && rm -f ${FILE}
  return 0
}


###############################################################################
# Function : Main
#
# This is the starting point for the script, this
# function invokes other functions in the required
# order
###############################################################################

function Main {
    # check for the presence of Server Administrator packages
    # Only packages with prefix srvadmin was getting listed	
    #if rpm -qa | grep "$DELL_OPENMANAGE_PREFIX" ;
    if rpm -qa | egrep "^$DELL_OPENMANAGE_PREFIX|^instsvc" ;
    then
        # clear the /etc/omreg.cfg entry: upgrade.relocation if set to "bad"
        UPGRADE_CHECK=`GetRegVal /etc/omreg.cfg upgrade.relocation`
        if [ -n "${UPGRADE_CHECK}" -a "${UPGRADE_CHECK}" == "bad" ];
        then
            RemoveRegPair /etc/omreg.cfg "upgrade.relocation=bad"
        fi
        Uninstall
    else
        echo "No Server Administrator RPM packages are installed."
    fi
}

###############################################################################
#
#  Function : Usage
#
###############################################################################

function Usage
{
echo "Usage: ${0} [OPTION]"
echo "Server Administrator Custom Uninstall Utility."
echo "This Utility will run in the interactive mode if no option"
echo "is given and runs silently if a valid option is given."
echo " "
echo "Option: "
echo "  [-f, --force]  do not prompt before uninstall"
echo "  [-h, --help]   Displays this help text."
echo " "
    exit 1
}

###############################################################################
#
#  Script execution begins here with the invocation
#  of Main function
#
###############################################################################
# check for root privileges
if [ ${UID} != 0 ]; then
    echo "Uninstall requires root privileges"
    exit 1
fi

if [ $# -gt 0 ]; then
    if [ "${1}" == "-f" -o "${1}" == "--force" ]; then
        FORCE_UNINSTALL=1
    elif [ "${1}" == "-h" -o "${1}" == "--help" ]; then
        Usage
    else
        echo "Invalid option '${1}', please see the usage below"
        echo " "
        Usage
    fi
fi
Main
exit 0
