#!/bin/sh
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
#
#  This software is supplied under the terms of a license agreement or
#  nondisclosure agreement with Dell Inc. and may not
#  be copied or disclosed except in accordance with the terms of that
#  agreement.
#
#  Copyright (c) 2000-2005 Dell Inc. All Rights Reserved.
#
#
#  Revision:
#    $Revision: 1.4 $
#
#  Last Modified By / On:
#    $Author: root $ / $Date: 2006/11/07 11:44:11 $
#
#  Author:
#    OpenManage Install
#
#  Environment:
#    Linux
#
#  User/Kernel Mode:
#    User
#
#  Abstract/Purpose:
#    Helper functions used by serveradministrator spec scriptlets
#
#  See Also:
#
#
#  Notes:
#
#
###############################################################################
[ -n "${OMIDEBUG}" ] && set -x

# set default access rights to limit potential security lapses
umask 077
# limit all commands to /bin directory unless a full is specified
PATH=/sbin:/bin:/usr/sbin:/usr/bin

###############################################################################
###############################################################################
# "Global" (i.e., "GBL_") variables that are expected to be used by script
# files that "source" this script file, in addition to being used internally
# by functions within this script file.
###############################################################################
###############################################################################

###############################################################################
# "Global" (i.e., "GBL_") variables:
#
# Various version-related variables.
###############################################################################
#Sourcing this text file will populate all the GBL_ variables.
#The file is of the format GBL_SOMETHING=23 
#This file should be in the same folder as openipmifuncs.sh on the CD

 if [ -e "$LINUXROOT/supportscripts/prereqcheck/openipmiver.sh" ]; then
	. "$LINUXROOT/supportscripts/prereqcheck/openipmiver.sh"

#This is for DUP. They call from the current directory
  elif [ -e "./openipmiver.sh" ]; then
    .  "./openipmiver.sh"

#This is being called from hapi preinstall script
#So you cant find the file in this folder
#but omilcore should already have put the entries down in omreg
  elif [ -s "/etc/omreg.cfg" ]; then
    OPENIPMIVERFILEPATH=$(grep -i "^openmanage.openipmi.openipmifuncs=" /etc/omreg.cfg |sed s#^[^=]*=##)
    TEMP=$( dirname  $OPENIPMIVERFILEPATH)
    OPENIPMIVERFILEPATH=$TEMP
    . "$OPENIPMIVERFILEPATH/openipmiver.sh"

#This is the default case - Do a "find"
   else 
     _OPENIPMIVER_SCRIPT=$(find "${LINUXROOT:-.}" -name "openipmiver.sh")
   if [ -n "$_OPENIPMIVER_SCRIPT" -a -e "$_OPENIPMIVER_SCRIPT" ]; then
     .  "$_OPENIPMIVER_SCRIPT"
   fi
fi

# Set consolidated version info for RHEL3.
GBL_PACKAGED_DKMS_VERSION_RHEL3="${GBL_PACKAGED_DKMS_VERSION_MAJOR_RHEL3}.${GBL_PACKAGED_DKMS_VERSION_MINOR_RHEL3}.${GBL_PACKAGED_DKMS_VERSION_MAINT_RHEL3}-${GBL_PACKAGED_DKMS_VERSION_BUILD_RHEL3}"
GBL_PACKAGED_DRIVER_VERSION_RHEL3="${GBL_PACKAGED_DRIVER_VERSION_MAJOR_RHEL3}.${GBL_PACKAGED_DRIVER_VERSION_MINOR_RHEL3}"
GBL_DRIVER_VER_MIN_RHEL3="${GBL_DRIVER_VER_MIN_MAJOR_RHEL3}.${GBL_DRIVER_VER_MIN_MINOR_RHEL3}"

GBL_PACKAGED_DKMS_VERSION_RHEL4="${GBL_PACKAGED_DKMS_VERSION_MAJOR_RHEL4}.${GBL_PACKAGED_DKMS_VERSION_MINOR_RHEL4}.${GBL_PACKAGED_DKMS_VERSION_MAINT_RHEL4}-${GBL_PACKAGED_DKMS_VERSION_BUILD_RHEL4}"
GBL_PACKAGED_DRIVER_VERSION_RHEL4="${GBL_PACKAGED_DRIVER_VERSION_MAJOR_RHEL4}.${GBL_PACKAGED_DRIVER_VERSION_MINOR_RHEL4}"
GBL_DRIVER_VER_MIN_RHEL4="${GBL_DRIVER_VER_MIN_MAJOR_RHEL4}.${GBL_DRIVER_VER_MIN_MINOR_RHEL4}"

GBL_PACKAGED_DKMS_VERSION_SLES9="${GBL_PACKAGED_DKMS_VERSION_MAJOR_SLES9}.${GBL_PACKAGED_DKMS_VERSION_MINOR_SLES9}.${GBL_PACKAGED_DKMS_VERSION_MAINT_SLES9}-${GBL_PACKAGED_DKMS_VERSION_BUILD_SLES9}"
GBL_PACKAGED_DRIVER_VERSION_SLES9="${GBL_PACKAGED_DRIVER_VERSION_MAJOR_SLES9}.${GBL_PACKAGED_DRIVER_VERSION_MINOR_SLES9}"
GBL_DRIVER_VER_MIN_SLES9="${GBL_DRIVER_VER_MIN_MAJOR_SLES9}.${GBL_DRIVER_VER_MIN_MINOR_SLES9}"

GBL_KERNEL_VER_MIN="${GBL_KERNEL_VER_MIN_MAJOR}.${GBL_KERNEL_VER_MIN_MINOR}.${GBL_KERNEL_VER_MIN_PATCH}"

###############################################################################
# "Global" (i.e., "GBL_") variables:
#
# Various "enum" type variables.
###############################################################################

# OpenIPMI driver module that is used for version checking.
GBL_OPENIPMI_DRIVER_MODULE="ipmi_msghandler"

# OpenIPMI driver RPM that is used for version checking.
GBL_OPENIPMI_DRIVER_RPM="openipmi"

# Operating system ("OS") types.
# These are the return values for the following function:
#     GetOSType
#
GBL_OS_TYPE_ERROR=0
GBL_OS_TYPE_UKNOWN=1
GBL_OS_TYPE_RHEL3=2
GBL_OS_TYPE_RHEL4=3
GBL_OS_TYPE_RHEL5=6
GBL_OS_TYPE_SLES9=4
GBL_OS_TYPE_SLES10=5

# System types.
# These are the return values for the following functions:
#     GetSupportedSystemTypes
#     GetActiveSystemType
# And they are the values used to set the global variables:
#     GBL_SUPPORTED_SYSTEM_TYPES
#     GBL_ACTIVE_SYSTEM_TYPE
#
GBL_SYSTEM_TYPE_ERROR=0
GBL_SYSTEM_TYPE_UNKNOWN=1
GBL_SYSTEM_TYPE_ESM2_ONLY=2
GBL_SYSTEM_TYPE_TVM_ONLY=3
GBL_SYSTEM_TYPE_IPMI_ONLY=4
GBL_SYSTEM_TYPE_TVM_AND_IPMI=5
GBL_SYSTEM_TYPE_ESM2_OR_TVM=6

# Status values for the OpenIPMI driver module (i.e., "ipmi_msghandler.").
GBL_DRIVER_MODULE_STATUS_ERROR=0
GBL_DRIVER_MODULE_STATUS_NOT_INSTALLED=1
GBL_DRIVER_MODULE_STATUS_VERSION_UNKNOWN=2
GBL_DRIVER_MODULE_STATUS_VERSION_INSUFFICIENT=3
GBL_DRIVER_MODULE_STATUS_VERSION_SUFFICIENT=4

# Status values for the OpenIPMI driver RPM (i.e., "openipmi").
GBL_DRIVER_RPM_STATUS_ERROR=0
GBL_DRIVER_RPM_STATUS_NOT_INSTALLED=1
GBL_DRIVER_RPM_STATUS_VERSION_UNKNOWN=2
GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT=3
GBL_DRIVER_RPM_STATUS_VERSION_SUFFICIENT=4

###############################################################################
# "Global" (i.e., "GBL_") variables:
#
# Recommended actions with respect to the OpenIPMI driver.
# These are the return values for the following functions:
#     DetermineOpenIPMIDriverAction
#     CheckOpenIPMIDriverStatus
###############################################################################

# The following four recommended actions are being deprecated.
# One should use one of the four
# "GBL_ACTION_[INSTALL|UPGRADE]_DRIVER_BY_INSTALLING_RPM_ON_[IPMI|TVM]" values
# that follow these four values instead.
GBL_ACTION_INSTALL_DRIVER_ON_IPMI=2
GBL_ACTION_UPGRADE_DRIVER_ON_IPMI=3
GBL_ACTION_INSTALL_DRIVER_ON_TVM=4
GBL_ACTION_UPGRADE_DRIVER_ON_TVM=5

# Error while trying to determine recommended action.
GBL_ACTION_ERROR=0

# Do nothing. Installed OpenIPMI driver is sufficient.
GBL_ACTION_DO_NOTHING=1

# Install/upgrade the OpenIPMI driver by installing the "openipmi" RPM (on
# IPMI/TVM).
GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_IPMI=${GBL_ACTION_INSTALL_DRIVER_ON_IPMI}
GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_IPMI=${GBL_ACTION_UPGRADE_DRIVER_ON_IPMI}
GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_TVM=${GBL_ACTION_INSTALL_DRIVER_ON_TVM}
GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_TVM=${GBL_ACTION_UPGRADE_DRIVER_ON_TVM}

# Install/upgrade the OpenIPMI driver by upgrading the "openipmi" RPM (on
# IPMI/TVM).
GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_IPMI=6
GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_TVM=7
GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_IPMI=8
GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_TVM=9

# Install/upgrade the OpenIPMI driver by installing kernel source and building
# and installing OpenIPMI driver modules using DKMS (on IPMI/TVM).
GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI=10
GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM=11
GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI=12
GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM=13

# Install/upgrade the OpenIPMI driver by building and installing OpenIPMI
# driver modules using DKMS (on IPMI/TVM).
GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_IPMI=14
GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_TVM=15
GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_IPMI=16
GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_TVM=17

###############################################################################
##
## Function:    GetOSType
##
## Decription:  Determines the operating system type.
##
## Returns:     0 = No error
##
## Return Vars: GBL_OS_TYPE=[GBL_OS_TYPE_ERROR|
##                           GBL_OS_TYPE_UKNOWN|
##                           GBL_OS_TYPE_RHEL3|
##                           GBL_OS_TYPE_RHEL4|
##                           GBL_OS_TYPE_RHEL5|
##                           GBL_OS_TYPE_SLES9|
##                           GBL_OS_TYPE_SLES10]
##              GBL_OS_TYPE_STRING=[RHEL3|RHEL4|RHEL5|SLES9|SLES10|UNKNOWN]
##
###############################################################################
GetOSType()
{
    # Set default values for return variables.
    GBL_OS_TYPE=${GBL_OS_TYPE_UKNOWN}
    GBL_OS_TYPE_STRING="UKNOWN"

    # Local variable.
    local LOC_VERSION=""

    # Check if operating system is RHEL3.
    if [ -f /etc/redhat-release ] && [ `grep -c "Taroon" /etc/redhat-release` -gt 0 ]; then
        GBL_OS_TYPE=${GBL_OS_TYPE_RHEL3}
        GBL_OS_TYPE_STRING="RHEL3"

    # Else check if operating system is RHEL4.
    elif [ -f /etc/redhat-release ] && [ `grep -c "Nahant" /etc/redhat-release` -gt 0 ]; then
        GBL_OS_TYPE=${GBL_OS_TYPE_RHEL4}
        GBL_OS_TYPE_STRING="RHEL4"

    # Else check if operating system is RHEL5.
    elif [ -f /etc/redhat-release ] && [ `grep -c "Tikanga" /etc/redhat-release` -gt 0 ]; then
        GBL_OS_TYPE=${GBL_OS_TYPE_RHEL5}
        GBL_OS_TYPE_STRING="RHEL5"

    # Else check if operating system is SLES.
    elif [ -f /etc/SuSE-release ]; then
        LOC_VERSION=`cat /etc/SuSE-release | grep "VERSION" | sed  -e 's#[^0-9]##g'`
        # Check if operating system is SLES9 or SLES10.
        if [ "${LOC_VERSION}" = "9" ]; then
            GBL_OS_TYPE=${GBL_OS_TYPE_SLES9}
            GBL_OS_TYPE_STRING="SLES9"

        elif [ "${LOC_VERSION}" = "10" ]; then
            GBL_OS_TYPE=${GBL_OS_TYPE_SLES10}
            GBL_OS_TYPE_STRING="SLES10"
        fi
    fi

    return 0
}

###############################################################################
##
## Function:    IsRAC3HardwarePresent
#
## Decription:  Check for RAC3 hardware.
##
## Inputs:      ${1} = Path-filename for the syslist.txt file.
##
## Returns:     0 = False. Not present.
##              1 = True. Is present.
##
###############################################################################
IsRAC3HardwarePresent()
{
    # Input parameter to the function.
    LOC_SYSIDFILEPATH="${1}"

    # Set default return value.
    LOC_DRAC3PRESENT=0

    for X in `lspci -n | awk -F' ' '{print $4}' 2>/dev/null`;
    do
        # DRAC3 found?
        grep $X=DRAC3 "${SYSIDFILEPATH}" > /dev/null;
        if [ $? = 0 ];
        then
            LOC_DRAC3PRESENT=1
        fi
    done

    return ${LOC_DRAC3PRESENT}
}

###############################################################################
##
## Function:    GetSupportedSystemTypes
##
## Decription:  Determines the type(s) of instrumentation supported by the
##              system. Note that a few systems support two types: a few
##              systems that are TVM by default, but become IPMI if a DRAC3
##              card is present.
##
##              Note that this function does not attempt to distinguish
##              between ESM2 verse TVM for systems older than 6G.
##
##              Note that for 6G and newer systems, this function relies on
##              the syslisttypes.txt to identify any systems that support
##              "TVM only" or support "TVM and IPMI". Any 6G or newer system
##              not in the file will be considered to support IPMI.
##
## Inputs:      ${1} = Path-filename for the sysreport program file.
##              ${2} = Path-filename for the syslisttypes.txt file.
##
## Returns:     One of the following:
##
##                  0 = GBL_SYSTEM_TYPE_ERROR
##                  1 = GBL_SYSTEM_TYPE_UNKNOWN
##                  2 = GBL_SYSTEM_TYPE_ESM2_ONLY
##                  3 = GBL_SYSTEM_TYPE_TVM_ONLY
##                  4 = GBL_SYSTEM_TYPE_IPMI_ONLY
##                  5 = GBL_SYSTEM_TYPE_TVM_AND_IPMI
##                  6 = GBL_SYSTEM_TYPE_ESM2_OR_TVM
##
###############################################################################
GetSupportedSystemTypes()
{
    # Input parameters to the function.
    LOC_SYSREPORTPATH="${1}"
    LOC_SYSLISTTYPESFILEPATH="${2}"

    # Set default return value.
    LOC_SUPPORTED_SYSTEM_TYPES=${GBL_SYSTEM_TYPE_UNKNOWN}

    # Get the system id.
    LOC_SYSID=`"${LOC_SYSREPORTPATH}" | grep "System ID" | awk -F0x '{print $2}'`
    if [ ! -n "${LOC_SYSID}" ]; then
        # System id not found/determined.
        return ${GBL_SYSTEM_TYPE_UNKNOWN}
    fi

    #
    # Check if the system is less than a 6G system. (i.e., system id is
    # If so, generalize it as supporting ESM2 or TVM.
    # We do not really case which one specifically.
    # We check for less than 6G by checking for a system id that is less than
    # system id 0x0101 (which is decimal 257).
    #

    # Convert system id from hex to decimal.
    let "LOC_SYSID_DECIMAL=0x${LOC_SYSID}"
    LOC_SYSID_6GL=257 # 0x0101
    if [ "${LOC_SYSID_DECIMAL}" -lt "${LOC_SYSID_6GL}" ]; then
        # The system supports either ESM2 or TVM.
        LOC_SUPPORTED_SYSTEM_TYPES=${GBL_SYSTEM_TYPE_ESM2_OR_TVM}
    else
        # Check if the system supports TVM only by checking the system id
        # verse the list in the syslisttype.txt file.
        grep "${LOC_SYSID}" "${LOC_SYSLISTTYPESFILEPATH}" | grep "TVM_ONLY" > /dev/null;
        if [ "$?" = "0" ]; then
            LOC_SUPPORTED_SYSTEM_TYPES=${GBL_SYSTEM_TYPE_TVM_ONLY}
        else
            # Check if the system supports TVM and IPMI by checking the system
            # id verse the list in the syslisttype.txt file.
            grep "${LOC_SYSID}" "${LOC_SYSLISTTYPESFILEPATH}" | grep "TVM_AND_IPMI" > /dev/null;
            if [ "$?" = "0" ]; then
                LOC_SUPPORTED_SYSTEM_TYPES=${GBL_SYSTEM_TYPE_TVM_AND_IPMI}
            else
                # At this point we assume the system supports IPMI.
                LOC_SUPPORTED_SYSTEM_TYPES=${GBL_SYSTEM_TYPE_IPMI_ONLY}
            fi
        fi
    fi

    return ${LOC_SUPPORTED_SYSTEM_TYPES}
}

###############################################################################
##
## Function:    GetActiveSystemType
##
## Decription:  Determines the active type of instrumentation supported by the
##              system. Note that a few systems support two types, though only
##              one is active at any time: a few systems that are TVM by
##              default, but become IPMI if a DRAC3 card is present.
##
##              Note that this function (by virtue of calling the
##              GetSupportedSystemTypes function) does not attempt to
##              distinguish between ESM2 verse TVM for systems older than 6G.
##
##              Note that for 6G and newer systems, this function (by virtue
##              of calling the GetSupportedSystemTypes function) relies on
##              the syslisttypes.txt to identify any systems that support
##              "TVM only" or support "TVM and IPMI". Any 6G or newer system
##              not in the file will be considered to be actively running
##              IPMI.
##
## Inputs:      ${1} = Path-filename for the sysreport program file.
##              ${2} = Path-filename for the syslisttypes.txt file.
##              ${3} = Path-filename for the syslist.txt file.
##
## Returns:     One of the following:
##
##                  0 = GBL_SYSTEM_TYPE_ERROR
##                  1 = GBL_SYSTEM_TYPE_UNKNOWN
##                  2 = GBL_SYSTEM_TYPE_ESM2_ONLY
##                  3 = GBL_SYSTEM_TYPE_TVM_ONLY
##                  4 = GBL_SYSTEM_TYPE_IPMI_ONLY
##                  6 = GBL_SYSTEM_TYPE_ESM2_OR_TVM
##
###############################################################################
GetActiveSystemType()
{
    # Input parameters to the function.
    LOC_SYSREPORTPATH="${1}"
    LOC_SYSLISTTYPESFILEPATH="${2}"
    LOC_SYSIDFILEPATH="${3}"

    # Set default value.
    LOC_ACTIVE_SYSTEM_TYPE=${GBL_SYSTEM_TYPE_UNKNOWN}

    # Get the supported system type(s) for the system.
    GetSupportedSystemTypes "${LOC_SYSREPORTPATH}" "${LOC_SYSLISTTYPESFILEPATH}"
    LOC_SUPPORTED_SYSTEM_TYPES=$?

    # At this point set the active system type to the supported system types
    # value. We will adjust it below, if necessary.
    LOC_ACTIVE_SYSTEM_TYPE=${LOC_SUPPORTED_SYSTEM_TYPES}

    # Check if the system is one of the special few system that support two
    # types (TVM and IPMI), though only one can be active at any time.
    # A few systems that are TVM by default become IPMI if a DRAC3 card is
    # present.
    if [ "${LOC_SUPPORTED_SYSTEM_TYPES}" = "${GBL_SYSTEM_TYPE_TVM_AND_IPMI}" ]; then
        # Check if a DRAC3 card is present.
        IsRAC3HardwarePresent "${LOC_SYSIDFILEPATH}"
        if [ "$?" = "1" ]; then
            # A DRAC3 card is present. IPMI should be active.
            LOC_ACTIVE_SYSTEM_TYPE=${GBL_SYSTEM_TYPE_IPMI_ONLY}
        else
            # A DRAC3 card is not present. TVM should be active.
            LOC_ACTIVE_SYSTEM_TYPE=${GBL_SYSTEM_TYPE_TVM_ONLY}
        fi
    fi

    return ${LOC_ACTIVE_SYSTEM_TYPE}
}

###############################################################################
##
## Function:    CheckForIPMISystem
##
## Description: Check if the system supports IPMI 1.0 or greater system.
##
##              Also returns (through global vartables) info on the supported
##              system types and active system type for the system.
##
##              Note that a few systems support two types: a few systems that
##              are TVM by default, but become IPMI if a DRAC3 card is
##              installed. Even when such systems do not have a DRAC3 card
##              installed, and are currently running TVM, we want to treat
##              these systems as supporting IPMI and want to install (or
##              recommended install) OpenIPMI on such systems, in case a DRAC3
##              card is later installed. Thus for such systems we return true.
##
## Inputs:      ${1} = Path-filename for the sysreport program file.
##              ${2} = Path-filename for the syslisttypes.txt file.
##              ${3} = Path-filename for the syslist.txt file.
##
## Returns:     0 = False. System does not support IPMI 1.0 or greater.
##              1 = True.  System does support IPMI 1.0 or greater.
##
## Return vars: GBL_SUPPORTED_SYSTEM_TYPES
##              GBL_ACTIVE_SYSTEM_TYPE
##
###############################################################################
CheckForIPMISystem()
{
    # Input parameters to the function.
    LOC_SYSREPORTPATH="${1}"
    LOC_SYSLISTTYPESFILEPATH="${2}"
    LOC_SYSIDFILEPATH="${3}"

    # Set default return value.
    LOC_IPMI_SYSTEM=0

    # Set default values for return variables.
    GBL_SUPPORTED_SYSTEM_TYPES=$GBL_SYSTEM_TYPE_UNKNOWN
    GBL_ACTIVE_SYSTEM_TYPE=$GBL_SYSTEM_TYPE_UNKNOWN

    # Get the supported system type(s) for the system.
    GetSupportedSystemTypes "${LOC_SYSREPORTPATH}" "${LOC_SYSLISTTYPESFILEPATH}"
    GBL_SUPPORTED_SYSTEM_TYPES=$?

    if [ "${GBL_SUPPORTED_SYSTEM_TYPES}" = "${GBL_SYSTEM_TYPE_IPMI_ONLY}" ]; then
        # The system supports IPMI.
        LOC_IPMI_SYSTEM=1
    elif [ "${GBL_SUPPORTED_SYSTEM_TYPES}" = "${GBL_SYSTEM_TYPE_TVM_AND_IPMI}" ]; then
        # The system supports IPMI.
        LOC_IPMI_SYSTEM=1

        # Additionally get the active system type for the system.
        GetActiveSystemType "${LOC_SYSREPORTPATH}" \
            "${LOC_SYSLISTTYPESFILEPATH}" \
            "${LOC_SYSIDFILEPATH}"
        GBL_ACTIVE_SYSTEM_TYPE=$?
    fi

    return ${LOC_IPMI_SYSTEM}
}

###############################################################################
##
## Function:    GetInstalledOpenIPMIDriverModuleInfo
##
## Description: Checks if the OpenIPMI driver module (i.e., "ipmi_msghandler")
##              is currently installed. And if so, tries to get the version of
##              the installed driver module.
##
## Returns:     0 = No error
##
## Return Vars: GBL_DRIVER_MODULE_INSTALLED=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_MODULE_VERSION_FOUND=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_MODULE_VERSION_COMPLETE=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_MODULE_VERSION=<version|unknown|number.unknown>
##              GBL_DRIVER_MODULE_VERSION_MAJOR=<number>
##              GBL_DRIVER_MODULE_VERSION_MINOR=<number|0 if unknown>
##
###############################################################################
GetInstalledOpenIPMIDriverModuleInfo()
{
    # Set defaults for global variables.
    GBL_DRIVER_MODULE_INSTALLED=0
    GBL_DRIVER_MODULE_VERSION_FOUND=0
    GBL_DRIVER_MODULE_VERSION_COMPLETE=0
    GBL_DRIVER_MODULE_VERSION="unknown"
    GBL_DRIVER_MODULE_VERSION_MAJOR=0
    GBL_DRIVER_MODULE_VERSION_MINOR=0

    # Local variables.
    local LOC_RUNNING_KERNEL=$(uname -r)
    local LOC_OPENIPMI_FILEVER="ipmi_msghandler"
    local LOC_OPENIPMI_GETVER_MOD="modinfo ${LOC_OPENIPMI_FILEVER}"
    local LOC_OPENIPMI_GETVER_FILE="/lib/modules/${LOC_RUNNING_KERNEL}/kernel/drivers/char/ipmi/${LOC_OPENIPMI_FILEVER}"
    local loc_modext=""
    local loc_major=0
    local loc_major=0

    # determine module extenstion based on kernel versions
    # default to ".ko"
    loc_modext=".ko"
    loc_major=$(uname -r | cut -d. -f1-2)
    if [ "${loc_major}" == "2.4" ]; then
        ## kernel version 2.4.x
        loc_modext=".o"
    fi

    # Check if the OpenIPMI driver is installed.
    if [ -f "${LOC_OPENIPMI_GETVER_FILE}${loc_modext}" ]; then
        GBL_DRIVER_MODULE_INSTALLED=1
    else
        GBL_DRIVER_MODULE_INSTALLED=0

        # Simply return now.
        return 0
    fi

    # reset variables to be reused
    loc_major=0
    loc_minor=0

    # check version using modprobe
    loc_major=$(${LOC_OPENIPMI_GETVER_MOD} 2>/dev/null \
        | grep 'version:' | awk -F'version:' '{ print $2 }' \
        | cut -d. -f1 | sed -e 's/[ \t]*//g' -e 's/v//g')
    if [ ! -z "${loc_major}" ]; then
        # major version found
        GBL_DRIVER_MODULE_VERSION_MAJOR="${loc_major}"

        # version found at this point
        GBL_DRIVER_MODULE_VERSION_FOUND=1

        # check for minor version
        loc_minor=$(${LOC_OPENIPMI_GETVER_MOD} 2>/dev/null \
            | grep 'version:' | awk -F'version:' '{ print $2 }' \
            | awk -F'.' '{ print $2 }' | sed -e 's/\([0-9]*\).*/\1/')
        if [ ! -z "${loc_minor}" ]; then
            # minor version found
            GBL_DRIVER_MODULE_VERSION_MINOR="${loc_minor}"

            # complete version found at this point
            GBL_DRIVER_MODULE_VERSION_COMPLETE=1
        fi
    fi

    # if check of version version using modprobe did not find a version,
    # then check version using strings in the kernel module file
    if [ "${GBL_DRIVER_MODULE_VERSION_FOUND}" = "0" ]; then
        loc_major=$(strings ${LOC_OPENIPMI_GETVER_FILE}${loc_modext} 2>/dev/null \
            | grep ' version ' | awk -F' version ' '{ print $2 }' \
            | cut -d. -f1 | sed -e 's/[ \t]*//g' -e 's/v//g')
        if [ ! -z "${loc_major}" ]; then
            # major version found
            GBL_DRIVER_MODULE_VERSION_MAJOR="${loc_major}"

            # version found at this point
            GBL_DRIVER_MODULE_VERSION_FOUND=1

            # check for minor version
            loc_minor=$(strings ${LOC_OPENIPMI_GETVER_FILE}${loc_modext} 2>/dev/null \
                | grep ' version ' | awk -F' version ' '{ print $2 }' \
                | awk -F'.' '{ print $2 }' | sed -e 's/\([0-9]*\).*/\1/')
            if [ ! -z "${loc_minor}" ]; then
                # minor version found
                GBL_DRIVER_MODULE_VERSION_MINOR="${loc_minor}"

                # complete version found at this point
                GBL_DRIVER_MODULE_VERSION_COMPLETE=1
            fi
        fi
    fi

    # Set the GBL_DRIVER_MODULE_VERSION_FOUND return var.
    if [ "${GBL_DRIVER_MODULE_VERSION_FOUND}" = "1" ]; then
        if [ "${GBL_DRIVER_MODULE_VERSION_COMPLETE}" = "1" ]; then
            GBL_DRIVER_MODULE_VERSION="${GBL_DRIVER_MODULE_VERSION_MAJOR}.${GBL_DRIVER_MODULE_VERSION_MINOR}"
        else
            GBL_DRIVER_MODULE_VERSION="${GBL_DRIVER_MODULE_VERSION_MAJOR}.unknown"
        fi
    fi

    return 0
}

###############################################################################
##
## Function:    DetermineInstalledOpenIPMIDriverModuleStatus
##
## Description: Based on comparing the version of an installed OpenIPMI
##              driver module (i.e., "ipmi_msghandler") (and whether or not the
##              OpenIPMI driver module is even currently installed) against a
##              minimum version of the OpenIPMI driver that is required by
##              Systems Management, determines the "status" of the OpenIPMI
##              driver RPM module with respect to the min version.
##
## Inputs:      ${1} = Driver module installed (boolean)
##              ${2} = Driver module version found (boolean)
##              ${3} = Driver mdoule version major
##              ${4} = Driver module version minor
##              ${5} = Minimum driver version major
##              ${6} = Minumum driver version minor
##
## Returns:     One of the following:
##
##                  0 = GBL_DRIVER_MODULE_STATUS_ERROR
##                  1 = GBL_DRIVER_MODULE_STATUS_NOT_INSTALLED
##                  2 = GBL_DRIVER_MODULE_STATUS_VERSION_UNKNOWN
##                  3 = GBL_DRIVER_MODULE_STATUS_VERSION_INSUFFICIENT
##                  4 = GBL_DRIVER_MODULE_STATUS_VERSION_SUFFICIENT
##
###############################################################################
DetermineInstalledOpenIPMIDriverModuleStatus()
{
    # Input parameters to the function. Local variables.
    local LOC_DRIVER_MODULE_INSTALLED="${1}"
    local LOC_DRIVER_MODULE_VERSION_FOUND="${2}"
    local LOC_DRIVER_MODULE_VERSION_MAJOR="${3}"
    local LOC_DRIVER_MODULE_VERSION_MINOR="${4}"
    local LOC_MIN_DRIVER_VERSION_MAJOR="${5}"
    local LOC_MIN_DRIVER_VERSION_MINOR="${6}"

    # Local variables.
    local LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_ERROR}"

    # Determines the "status" of the OpenIPMI driver module with respect to the
    # minimum version of the OpenIPMI driver that is required by Systems
    # Management.
    if [ "${LOC_DRIVER_MODULE_INSTALLED}" = "0" ]; then
        # The OpenIPMI driver module is not installed.
        LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_NOT_INSTALLED}"

    elif [ "${LOC_DRIVER_MODULE_VERSION_FOUND}" = "0" ]; then
        # The OpenIPMI driver module is installed, but the version of the
        # module could not be found or determined.
        LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_VERSION_UNKNOWN}"

    elif [ "${LOC_DRIVER_MODULE_VERSION_MAJOR}" -lt "${LOC_MIN_DRIVER_VERSION_MAJOR}" ]; then
        # The version of the installed OpenIPMI driver module is less than the
        # minimum version of the OpenIPMI driver that is required by Systems
        # Management.
        LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_VERSION_INSUFFICIENT}"

    elif [ "${LOC_DRIVER_MODULE_VERSION_MAJOR}" -eq  "${LOC_MIN_DRIVER_VERSION_MAJOR}" ] && [ "${LOC_DRIVER_MODULE_VERSION_MINOR}" -lt  "${LOC_MIN_DRIVER_VERSION_MINOR}" ]; then
        # The version of the installed OpenIPMI driver module is less than the
        # minimum version of the OPenIPMI driver that is required by Systems
        # Management.
        LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_VERSION_INSUFFICIENT}"

    else
        # The version of the installed OpenIPMI driver module is equal to or
        # greater than the minimum version of the OpenIPMI driver that is
        # required by Systems Management.
        LOC_DRIVER_MODULE_STATUS="${GBL_DRIVER_MODULE_STATUS_VERSION_SUFFICIENT}"
    fi

    return "${LOC_DRIVER_MODULE_STATUS}"
}

###############################################################################
##
## Function:    GetInstalledOpenIPMIDriverRPMInfo
##
## Description: Checks if the OpenIPMI driver RPM (i.e., "openipmi") is
##              currently installed. And if so, tries to get the version of the
##              installed driver RPM.
##
## Inputs:      ${1} = OpenIPMI driver RPM package name.
##
## Returns:     0 = No error
##
## Return Vars: GBL_DRIVER_RPM_INSTALLED=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_RPM_VERSION_FOUND=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_RPM_VERSION_COMPLETE=[0|1] (where 0=false, 1=true)
##              GBL_DRIVER_RPM_VERSION=<version|unknown>
##              GBL_DRIVER_RPM_VERSION_MAJOR=<number>
##              GBL_DRIVER_RPM_VERSION_MINOR=<number>
##
###############################################################################
GetInstalledOpenIPMIDriverRPMInfo()
{
    # Set defaults for global variables.
    GBL_DRIVER_RPM_INSTALLED=0
    GBL_DRIVER_RPM_VERSION_FOUND=0
    GBL_DRIVER_RPM_VERSION_COMPLETE=0
    GBL_DRIVER_RPM_VERSION="unknown"
    GBL_DRIVER_RPM_VERSION_MAJOR=0
    GBL_DRIVER_RPM_VERSION_MINOR=0

    # Input parameters to the function. Local variable.
    local LOC_DRIVER_RPM="${1}"

    # Local variables.
    local LOC_TEMP=""
    local LOC_MAJOR=0
    local LOC_MINOR=0

    # Check if the OpenIPMI driver RPM is installed.
    # (Note: We add "-" to the end of the grep string to help avoid inadvertant
    # hits. We do not expect any. But just in case.)
    if [ `rpm -qa | grep -c "${LOC_DRIVER_RPM}-"` -gt 0 ]; then
        GBL_DRIVER_RPM_INSTALLED=1
    else
        GBL_DRIVER_RPM_INSTALLED=0

        # Simply return now.
        return 0
    fi

    #
    # Note: We expect openipmi RPM package names like the following:
    #
    #    openipmi-<major>.<minor>.<OS>-<n>dkms
    #
    # where:
    #
    #    <OS> = [RHEL3|RHEL4|SLES9]
    #    <n>  = Integer value we do not care about.
    #

    # Get the "<major>.<minor>.<os>" portion of the RPM name.
    LOC_TEMP=`rpm -qa | grep "${LOC_DRIVER_RPM}-" | cut -d- -f2`

    # Get the "<major>" portion (and verify that it is an integer).
    LOC_MAJOR=`echo ${LOC_TEMP} | cut -d. -f1 | sed -e 's/\([0-9]*\).*/\1/'`
    if [ -z "${LOC_MAJOR}" ]; then
        # Major version is either not found, or is not an integer, in which
        # case we also treat it as not found (i.e., unknown).
        # Simply return now.
        return 0
    fi

    # Get the "<minor>" portion (and verify that it is an integer).
    LOC_MINOR=`echo ${LOC_TEMP} | cut -d. -f2 | sed -e 's/\([0-9]*\).*/\1/'`
    if [ -z "${LOC_MINOR}" ]; then
        # Minor version is either not found, or is not an integer, in which
        # case we also treat it as not found (i.e., unknown).
        # Simply return now.
        return 0
    fi

    # Complete version found at this point.
    # Set the rest of the global return variables.
    GBL_DRIVER_RPM_VERSION_FOUND=1
    GBL_DRIVER_RPM_VERSION_COMPLETE=1
    GBL_DRIVER_RPM_VERSION="${LOC_MAJOR}.${LOC_MINOR}"
    GBL_DRIVER_RPM_VERSION_MAJOR="${LOC_MAJOR}"
    GBL_DRIVER_RPM_VERSION_MINOR="${LOC_MINOR}"

    return 0
}

###############################################################################
##
## Function:    DetermineInstalledOpenIPMIDriverRPMStatus
##
## Description: Based on comparing the version of the installed OpenIPMI
##              driver RPM (and whether or not an OpenIPMI driver RPM is even
##              currently installed) against a minimum version of the OpenIPMI
##              driver that is required by Systems Management, determines the
##              "status" of the OpenIPMI driver RPM with respect to the min
##              version.
##
## Inputs:      ${1} = Driver RPM installed (boolean)
##              ${2} = Driver RPM version found (boolean)
##              ${3} = Driver RPM version major
##              ${4} = Driver RPM version minor
##              ${5} = Minimum driver version major
##              ${6} = Minumum driver version minor
##
## Returns:     One of the following:
##
##                  0 = GBL_DRIVER_RPM_STATUS_ERROR
##                  1 = GBL_DRIVER_RPM_STATUS_NOT_INSTALLED
##                  2 = GBL_DRIVER_RPM_STATUS_VERSION_UNKNOWN
##                  3 = GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT
##                  4 = GBL_DRIVER_RPM_STATUS_VERSION_SUFFICIENT
##
###############################################################################
DetermineInstalledOpenIPMIDriverRPMStatus()
{
    # Input parameters to the function. Local variables.
    local LOC_DRIVER_RPM_INSTALLED="${1}"
    local LOC_DRIVER_RPM_VERSION_FOUND="${2}"
    local LOC_DRIVER_RPM_VERSION_MAJOR="${3}"
    local LOC_DRIVER_RPM_VERSION_MINOR="${4}"
    local LOC_MIN_DRIVER_VERSION_MAJOR="${5}"
    local LOC_MIN_DRIVER_VERSION_MINOR="${6}"

    # Local variables.
    local LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_ERROR}"

    # Determines the "status" of the OpenIPMI driver RPM with respect to the
    # minimum version of the OpenIPMI driver that is required by Systems
    # Management.
    if [ "${LOC_DRIVER_RPM_INSTALLED}" = "0" ]; then
        # The OpenIPMI driver RPM is not installed.
        LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_NOT_INSTALLED}"

    elif [ "${LOC_DRIVER_RPM_VERSION_FOUND}" = "0" ]; then
        # The OpenIPMI driver RPM is installed, but the version of the RPM
        # could not be found or determined.
        LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_VERSION_UNKNOWN}"

    elif [ "${LOC_DRIVER_RPM_VERSION_MAJOR}" -lt "${LOC_MIN_DRIVER_VERSION_MAJOR}" ]; then
        # The version of the installed OpenIPMI driver RPM is less than the
        # minimum version of the OpenIPMI driver that is required by Systems
        # Management.
        LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT}"

    elif [ "${LOC_DRIVER_RPM_VERSION_MAJOR}" -eq  "${LOC_MIN_DRIVER_VERSION_MAJOR}" ] && [ "${LOC_DRIVER_RPM_VERSION_MINOR}" -lt  "${LOC_MIN_DRIVER_VERSION_MINOR}" ]; then
        # The version of the installed OpenIPMI driver RPM is less than the
        # minimum version of the OPenIPMI driver that is required by Systems
        # Management.
        LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT}"

    else
        # The version of the installed OpenIPMI driver RPM is equal to or
        # greater than the minimum version of the OpenIPMI driver that is
        # required by Systems Management.
        LOC_DRIVER_RPM_STATUS="${GBL_DRIVER_RPM_STATUS_VERSION_SUFFICIENT}"
    fi

    return "${LOC_DRIVER_RPM_STATUS}"
}

###############################################################################
##
## Function:    GetKernelSourceInfo
##
## Description: Checks if kernel source appears to currently be installed.
##              Also sets the base name (minus the kernel version info) of the
##              kernel source related RPM that is applicable to the given
##              operating system type and running kernel.
##
## Inputs:      ${1} = Operating system type.
##
## Returns:     0 = No error
##
## Return Vars: GBL_KERNEL_SOURCE_INSTALLED=[0|1] (where 0=false, 1=true)
##              GBL_KERNEL_SOURCE_RPM_NAME=<kernel-source|kernel-devel|
##                  kernel-smp-devel|kernel-hugemem-devel>
##
###############################################################################
GetKernelSourceInfo()
{
    # Set defaults for global variables.
    GBL_KERNEL_SOURCE_INSTALLED=0
    GBL_KERNEL_SOURCE_RPM_NAME="unknown"

    # Input parameters to the function. Local variables.
    local LOC_OS_TYPE="${1}"

    # Local variables.
    local LOC_RUNNING_KERNEL=`uname -r`

    # Do checks of kernel source based on the operating system type.
    if [ "${LOC_OS_TYPE}" = "${GBL_OS_TYPE_RHEL3}" ]; then

        # Check if kernel source is installed by checking if the following
        # directory exists:
        #
        #     /lib/modules/<running-kernel>/build/kernel
        #
        # Note that on RHEL3, checking for the
        # "/lib/modules/<running-kernel>/build/include" directory does not
        # appear to be a valid check. We have found with testing that after
        # installing, and then uninstalling, the kernel-source RPM, the
        # directory "/lib/modules/<running-kernel>/build/include/config" was
        # left installed. However, the
        # "/lib/modules/<running-kernel>/build/kernel" was not left installed.
        # Thus we have chosen to use the "../build/kernel" directory to do our
        # check.
        #
        if [ -e "/lib/modules/`uname -r`/build/kernel" ]; then
            GBL_KERNEL_SOURCE_INSTALLED=1
        else
            GBL_KERNEL_SOURCE_INSTALLED=0
        fi

        # Set the expected name of the kernel source related RPM (less any
        # kernel version info) based on the OS being RHEL3.
        GBL_KERNEL_SOURCE_RPM_NAME="kernel-source"

    elif [ "${LOC_OS_TYPE}" = "${GBL_OS_TYPE_RHEL4}" ]; then

        # Check if kernel source is installed by checking if the following
        # directory exists:
        #
        #     /lib/modules/<running-kernel>/build/include
        #
        if [ -e "/lib/modules/`uname -r`/build/include" ]; then
            GBL_KERNEL_SOURCE_INSTALLED=1
        else
            GBL_KERNEL_SOURCE_INSTALLED=0
        fi

        # Set the expected name of the kernel source related RPM (less any
        # kernel version info) based on the assumption that the name of kernel
        # source related RPM will be like one of the following, depending of
        # the type (i.e., "hugemem", "smp", etc) of the running kernel:
        #
        #     kernel-devel-<running kernel>
        #     kernel-hugemem-devel-<running kernel>
        #     kernel-smp-devel-<running kernel>
        #
        if [ `echo ${LOC_RUNNING_KERNEL} | grep -c "hugemem"` -gt 0 ]; then
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-hugemem-devel"
        elif [ `echo ${LOC_RUNNING_KERNEL} | grep -c "smp"` -gt 0 ]; then
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-smp-devel"
        else
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-devel"
        fi

    elif [ "${LOC_OS_TYPE}" = "${GBL_OS_TYPE_SLES9}" ]; then

        # Check if kernel source is installed by checking if the following
        # directory exists:
        #
        #     /lib/modules/<running-kernel>/build/include
        #
        if [ -e "/lib/modules/`uname -r`/build/include" ]; then
            GBL_KERNEL_SOURCE_INSTALLED=1
        else
            GBL_KERNEL_SOURCE_INSTALLED=0
        fi

        # Set the expected name of the kernel source related RPM (less any
        # kernel version info) based on the OS being RHEL3.
        GBL_KERNEL_SOURCE_RPM_NAME="kernel-source"

    # For now provide a default case, though we do not expect this case
    # to get executed.
    # For now the default case is set to the equivalent of the RHEL4 case.
    else

        # Check if kernel source is installed by checking if the following
        # directory exists:
        #
        #     /lib/modules/<running-kernel>/build/include
        #
        if [ -e "/lib/modules/`uname -r`/build/include" ]; then
            GBL_KERNEL_SOURCE_INSTALLED=1
        else
            GBL_KERNEL_SOURCE_INSTALLED=0
        fi

        # Set the expected name of the kernel source related RPM (less any
        # kernel version info) based on the assumption that the name of kernel
        # source related RPM will be like one of the following, depending of
        # the type (i.e., "hugemem", "smp", etc) of the running kernel:
        #
        #     kernel-devel-<running kernel>
        #     kernel-hugemem-devel-<running kernel>
        #     kernel-smp-devel-<running kernel>
        #
        if [ `echo ${LOC_RUNNING_KERNEL} | grep -c "hugemem"` -gt 0 ]; then
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-hugemem-devel"
        elif [ `echo ${LOC_RUNNING_KERNEL} | grep -c "smp"` -gt 0 ]; then
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-smp-devel"
        else
            GBL_KERNEL_SOURCE_RPM_NAME="kernel-devel"
        fi
    fi

    return 0
}

###############################################################################
##
## Function:    DetermineOpenIPMIDriverAction
##
## Description: Based on comparing the version of the installed OpenIPMI
##              driver (and whether or not an OpenIPMI driver is even
##              currently installed) against a minimum version of the OpenIPMI
##              driver that is required by Systems Management, makes a final
##              determination of recommendation of whether or not the OpenIPMI
##              driver should be installed or upgraded, or whether nothing
##              needs to be done.
##
## Inputs:      ${1} = Driver module status
##              ${2} = Driver RPM status
##              ${3} = Active system type (i.e., IPMI vs TVM)
##              ${4} = Kernel source installed (boolean)
##
## Returns:     One of the following:
##
##                  0  = GBL_ACTION_ERROR
##                  1  = GBL_ACTION_DO_NOTHING
##                  2  = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_IPMI
##                  3  = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_IPMI
##                  4  = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_TVM
##                  5  = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_TVM
##                  6  = GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_IPMI
##                  7  = GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_TVM
##                  8  = GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_IPMI
##                  9  = GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_TVM
##                  10 = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI
##                  11 = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM
##                  12 = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI
##                  13 = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM
##                  14 = GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_IPMI
##                  15 = GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_TVM
##                  16 = GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_IPMI
##                  17 = GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_TVM
##
###############################################################################
DetermineOpenIPMIDriverAction()
{
    # Input parameters to the function.
    LOC_DRIVER_MODULE_STATUS="${1}"
    LOC_DRIVER_RPM_STATUS="${2}"
    LOC_ACTIVE_SYSTEM_TYPE="${3}"
    LOC_KERNEL_SOURCE_INSTALLED="${4}"

    if [ "${LOC_DRIVER_MODULE_STATUS}" = "${GBL_DRIVER_MODULE_STATUS_NOT_INSTALLED}" ]; then

        if [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_NOT_INSTALLED}" ]; then
            # (1) The OpenIPMI driver module is not installed.
            # (2) The OpenIPMI driver RPM is not installed.
            # Action: Install the OpenIPMI driver by installing the OpenIPMI
            #         driver RPM that is packaged with Systems Management.

            # Also distiguish between whether IPMI verse TVM is active.
            if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_TVM}
            else
                LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_IPMI}
            fi

        elif [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_VERSION_UNKNOWN}" ] || [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT}" ]; then
            # (1) The OpenIPMI driver module is not installed.
            # (2) The OpenIPMI driver RPM is installed.
            # (3) The version of the installed OpenIPMI driver RPM is either:
            #     (a) unknown, or
            #     (b) less than the minimum version of the OpenIPMI driver that
            #         is required by Systems Management.
            # Action: Install the OpenIPMI driver by upgrading the OpenIPMI
            #         driver RPM with the OpenIPMI driver RPM that is packaged
            #         with Systems Management.

            # Also distiguish between whether IPMI verse TVM is active.
            if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_TVM}
            else
                LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_IPMI}
            fi

        else
            #
            # The version of the installed OpenIPMI RPM is equal to or greater
            # than the minimum version of the OpenIPMI driver that is required
            # by Systems Management.
            #

            if [ "${LOC_KERNEL_SOURCE_INSTALLED}" = "0" ]; then
                # (1) The OpenIPMI driver module is not installed.
                # (2) The OpenIPMI driver RPM is installed.
                # (3) The version of the installed OpenIPMI driver RPM is equal
                #     to or greater than the minimum version of the OpenIPMI
                #     driver that is required by Systems Management.
                # (4) Kernel source is not installed.
                # Action: Install the OpenIPMI driver by installing kernel
                #         source and building and installing driver modules
                #         using DKMS.

                # Also distiguish between whether IPMI verse TVM is active.
                if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                    LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM}
                else
                    LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI}
                fi

            else
                # (1) The OpenIPMI driver module is not installed.
                # (2) The OpenIPMI driver RPM is installed.
                # (3) The version of the installed OpenIPMI driver RPM is equal
                #     to or greater than the minimum version of the OpenIPMI
                #     driver that is required by Systems Management.
                # (4) Kernel source is installed.
                # Action: Install the OpenIPMI driver by building and
                #         installing driver modules using DKMS.

                # Also distiguish between whether IPMI verse TVM is active.
                if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                    LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_TVM}
                else
                    LOC_ACTION=${GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_IPMI}
                fi
            fi
        fi

    elif [ "${LOC_DRIVER_MODULE_STATUS}" = "${GBL_DRIVER_MODULE_STATUS_VERSION_UNKNOWN}" ] || [ "${LOC_DRIVER_MODULE_STATUS}" = "${GBL_DRIVER_MODULE_STATUS_VERSION_INSUFFICIENT}" ]; then

        if [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_NOT_INSTALLED}" ]; then
            # (1) The OpenIPMI driver module is installed.
            # (2) The version of the installed OpenIPMI driver module is
            #     either:
            #     (a) unknown, or
            #     (b) less than the minimum version of the OpenIPMI driver that
            #         is required by Systems Management.
            # (3) The OpenIPMI driver RPM is not installed.
            # Action: Upgrade the OpenIPMI driver by installing the OpenIPMI
            #         driver RPM that is packaged with Systems Management.

            # Also distiguish between whether IPMI verse TVM is active.
            if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_TVM}
            else
                LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_IPMI}
            fi

        elif [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_VERSION_UNKNOWN}" ] || [ "${LOC_DRIVER_RPM_STATUS}" = "${GBL_DRIVER_RPM_STATUS_VERSION_INSUFFICIENT}" ]; then
            # (1) The OpenIPMI driver module is installed.
            # (2) The version of the installed OpenIPMI driver module is
            #     either:
            #     (a) unknown, or
            #     (b) less than the minimum version of the OpenIPMI driver that
            #         is required by Systems Management.
            # (3) The OpenIPMI driver RPM is installed.
            # (4) The version of the installed OpenIPMI driver RPM is either:
            #     (a) unknown, or
            #     (b) less than the minimum version of the OpenIPMI driver that
            #         is required by Systems Management.
            # Action: Upgrade the OpenIPMI driver by upgrading the OpenIPMI
            #         driver RPM with the OpenIPMI driver RPM that is packaged
            #         with Systems Management.

            # Also distiguish between whether IPMI verse TVM is active.
            if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_TVM}
            else
                LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_IPMI}
            fi

        else
            #
            # The version of the installed OpenIPMI RPM is equal to or greater
            # than the minimum version of the OpenIPMI driver that is required
            # by Systems Management.
            #

            if [ "${LOC_KERNEL_SOURCE_INSTALLED}" = "0" ]; then
                # (1) The OpenIPMI driver module is installed.
                # (2) The version of the installed OpenIPMI driver module is
                #     either:
                #     (a) unknown, or
                #     (b) less than the minimum version of the OpenIPMI driver
                #         that is required by Systems Management.
                # (3) The OpenIPMI driver RPM is installed.
                # (4) The version of the installed OpenIPMI driver RPM is equal
                #     to or greater than the minimum version of the OpenIPMI
                #     driver that is required by Systems Management.
                # (5) Kernel source is not installed.
                # Action: Upgrade the OpenIPMI driver by installing kernel
                #         source and building and installing driver modules
                #         using DKMS.

                # Also distiguish between whether IPMI verse TVM is active.
                if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                    LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM}
                else
                    LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI}
                fi

            else
                # (1) The OpenIPMI driver module is installed.
                # (2) The version of the installed OpenIPMI driver module is
                #     either:
                #     (a) unknown, or
                #     (b) less than the minimum version of the OpenIPMI driver
                #         that is required by Systems Management.
                # (3) The OpenIPMI driver RPM is installed.
                # (4) The version of the installed OpenIPMI driver RPM is equal
                #     to or greater than the minimum version of the OpenIPMI
                #     driver that is required by Systems Management.
                # (5) Kernel source is installed.
                # Action: Upgrade the OpenIPMI driver by building and
                #         installing driver modules using DKMS.

                # Also distiguish between whether IPMI verse TVM is active.
                if [ "${LOC_ACTIVE_SYSTEM_TYPE}" = "${GBL_SYSTEM_TYPE_TVM_ONLY}" ]; then
                    LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_TVM}
                else
                    LOC_ACTION=${GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_IPMI}
                fi
            fi
        fi

    else
        # (1) The OpenIPMI driver module is installed.
        # (2) The version of the installed OpenIPMI driver module is equal to
        #     or greater than the minimum version of the OpenIPMI driver that
        #     is required by Systems Management.
        # Action: Do nothing. (Installed driver is sufficient. Simply leave the
        #         installed driver alone.)
        LOC_ACTION=${GBL_ACTION_DO_NOTHING}
    fi

    return ${LOC_ACTION}
}

###############################################################################
##
## Function:    CheckOpenIPMIDriverStatus
##
## Description: Makes overall determination of recommendation of whether or
##              not the the OpenIPMI driver should be installed or upgraded,
##              or whether nothing needs to be done.
##
## Inputs:      ${1} = Path-filename for the sysreport program file.
##              ${2} = Path-filename for the syslisttypes.txt file.
##              ${3} = Path-filename for the syslist.txt file.
##
## Returns:     One of the following:
##
##                  0  = GBL_ACTION_ERROR
##                  1  = GBL_ACTION_DO_NOTHING
##                  2  = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_IPMI
##                  3  = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_IPMI
##                  4  = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_RPM_ON_TVM
##                  5  = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_RPM_ON_TVM
##                  6  = GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_IPMI
##                  7  = GBL_ACTION_INSTALL_DRIVER_BY_UPGRADING_RPM_ON_TVM
##                  8  = GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_IPMI
##                  9  = GBL_ACTION_UPGRADE_DRIVER_BY_UPGRADING_RPM_ON_TVM
##                  10 = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI
##                  11 = GBL_ACTION_INSTALL_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM
##                  12 = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_IPMI
##                  13 = GBL_ACTION_UPGRADE_DRIVER_BY_INSTALLING_SOURCE_AND_USING_DKMS_ON_TVM
##                  14 = GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_IPMI
##                  15 = GBL_ACTION_INSTALL_DRIVER_BY_USING_DKMS_ON_TVM
##                  16 = GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_IPMI
##                  17 = GBL_ACTION_UPGRADE_DRIVER_BY_USING_DKMS_ON_TVM
#3
## Return vars: GBL_SUPPORTED_SYSTEM_TYPES
##              GBL_ACTIVE_SYSTEM_TYPE
##              GBL_VER_MIN
##
###############################################################################
CheckOpenIPMIDriverStatus()
{
    # Input parameters to the function. Local variables.
    local LOC_SYSREPORTPATH="${1}"
    local LOC_SYSLISTTYPESFILEPATH="${2}"
    local LOC_SYSIDFILEPATH="${3}"

    # Other local variables.
    local LOC_SYSTEM_SUPPORTS_IPMI=0
    local LOC_DRIVER_VER_MIN_MAJOR=0
    local LOC_DRIVER_VER_MIN_MINOR=0
    local LOC_DRIVER_MODULE_STATUS=${GBL_DRIVER_MODULE_STATUS_ERROR}
    local LOC_DRIVER_RPM_STATUS=${GBL_DRIVER_RPM_STATUS_ERROR}
    local LOC_OPENIPMI_STATUS=${GBL_ACTION_ERROR}

    # Check if the system supports IPMI 1.0 or greater system.
    # Also sets info on the supported system types and active system type.
    CheckForIPMISystem "${LOC_SYSREPORTPATH}" "${LOC_SYSLISTTYPESFILEPATH}" "${LOC_SYSIDFILEPATH}"
    LOC_SYSTEM_SUPPORTS_IPMI=$?
    if [ "${LOC_SYSTEM_SUPPORTS_IPMI}" = "0" ]; then
        # System does not support IPMI 1.0 or greater.
        # OpenIPMI is only applicable on systems that support IPMI 1.0 or
        # greater. No need for OpenIPMI driver.
        return ${GBL_ACTION_DO_NOTHING}
    fi

    # Get the operating system type.
    GetOSType
    if [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_UKNOWN}" ] || [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_ERROR}" ]; then
        # Operating system type is unknown, or an error occurred trying to
        # determine the operating sysstem type. Do nothing.
        return ${GBL_ACTION_DO_NOTHING}
    fi

    # Set some "OS-independent" local variables based on the OS type so that
    # any subsequent code in this function does not need to know the OS type.
    # Also set an "OS-independent" global variable, named "GBL_VER_MIN", that
    # tracks the mininum version, so that calling functions can use
    # "GBL_VER_MIN" without needing to know the OS type.
    if [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_RHEL3}" ]; then
        LOC_DRIVER_VER_MIN_MAJOR=${GBL_DRIVER_VER_MIN_MAJOR_RHEL3}
        LOC_DRIVER_VER_MIN_MINOR=${GBL_DRIVER_VER_MIN_MINOR_RHEL3}
        GBL_VER_MIN=${GBL_DRIVER_VER_MIN_RHEL3}

    elif [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_RHEL4}" ]; then
        LOC_DRIVER_VER_MIN_MAJOR=${GBL_DRIVER_VER_MIN_MAJOR_RHEL4}
        LOC_DRIVER_VER_MIN_MINOR=${GBL_DRIVER_VER_MIN_MINOR_RHEL4}
        GBL_VER_MIN=${GBL_DRIVER_VER_MIN_RHEL4}

    elif [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_SLES9}" ]; then
        LOC_DRIVER_VER_MIN_MAJOR=${GBL_DRIVER_VER_MIN_MAJOR_SLES9}
        LOC_DRIVER_VER_MIN_MINOR=${GBL_DRIVER_VER_MIN_MINOR_SLES9}
        GBL_VER_MIN=${GBL_DRIVER_VER_MIN_SLES9}

    elif [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_SLES10}" ]; then
        return ${GBL_ACTION_DO_NOTHING}

    elif [ "${GBL_OS_TYPE}" = "${GBL_OS_TYPE_RHEL5}" ]; then
        return ${GBL_ACTION_DO_NOTHING}
    fi

    # Get info about the currently installed OpenIPMI driver module, if one is
    # currently installed. And then determine the status of the currently
    # installed OpenIPMI driver module (if one), relative to the minimum
    # version of the OpenIPMI driver required by Systems Management.
    GetInstalledOpenIPMIDriverModuleInfo
    DetermineInstalledOpenIPMIDriverModuleStatus \
        "${GBL_DRIVER_MODULE_INSTALLED}" \
        "${GBL_DRIVER_MODULE_VERSION_FOUND}" \
        "${GBL_DRIVER_MODULE_VERSION_MAJOR}" \
        "${GBL_DRIVER_MODULE_VERSION_MINOR}" \
        "${LOC_DRIVER_VER_MIN_MAJOR}" \
        "${LOC_DRIVER_VER_MIN_MINOR}"
    LOC_DRIVER_MODULE_STATUS=$?

    # Do a preliminary check at this point to see if the OpenIPMI driver is
    # installed and the version is sufficient.
    if [ "${LOC_DRIVER_MODULE_STATUS}" = "${GBL_DRIVER_MODULE_STATUS_VERSION_SUFFICIENT}" ]; then
        # Do nothing. (Installed driver is sufficient. Simply leave the
        # installed driver alone.)
        return ${GBL_ACTION_DO_NOTHING}
    fi

    # Get info about the currently installed OpenIPMI driver RPM, if one is
    # currently installed. And then determine the status of the currently
    # installed OpenIPMI driver RPM (if one), relative to the minimum
    # version of the OpenIPMI driver required by Systems Management.
    GetInstalledOpenIPMIDriverRPMInfo "${GBL_OPENIPMI_DRIVER_RPM}"
    DetermineInstalledOpenIPMIDriverRPMStatus \
        "${GBL_DRIVER_RPM_INSTALLED}" \
        "${GBL_DRIVER_RPM_VERSION_FOUND}" \
        "${GBL_DRIVER_RPM_VERSION_MAJOR}" \
        "${GBL_DRIVER_RPM_VERSION_MINOR}" \
        "${LOC_DRIVER_VER_MIN_MAJOR}" \
        "${LOC_DRIVER_VER_MIN_MINOR}"
    LOC_DRIVER_RPM_STATUS=$?

    # Get info about whether or not kernel source is currently installed and
    # get the base name (minus the kernel version info) of the kernel source
    # related RPM that is applicable to the given operating system type and
    # running kernel.
    GetKernelSourceInfo "${GBL_OS_TYPE}"

    # Determine what action is recommended.
    DetermineOpenIPMIDriverAction \
        "${LOC_DRIVER_MODULE_STATUS}" \
        "${LOC_DRIVER_RPM_STATUS}" \
        "${GBL_ACTIVE_SYSTEM_TYPE}" \
        "${GBL_KERNEL_SOURCE_INSTALLED}"
    LOC_OPENIPMI_STATUS=$?

    return $LOC_OPENIPMI_STATUS
}

###############################################################################

##
## test code, pass 'test_os' or 'test_status' as arg1 to execute
##
if [ "${1}" = "test_os" ] ; then

    GetOSType
    echo "Test: OS Type = $GBL_OS_TYPE"

elif [ "${1}" = "test_status" ] ; then

    CheckOpenIPMIDriverStatus \
        "./sysreport" \
        "./syslisttypes.txt" \
        "./syslist.txt"
    DRIVER_STATUS=$?
    echo "Test: Recommended action = $DRIVER_STATUS"

fi
