use strict;
use warnings;
use utf8;
use Test::More tests => 26;
use Data::Dump qw(pp);
use NorisOtrsReport::QuelleProcessor;


my $ErrorTagDE = 'Beschreibung der Störung';
my $ReasonTagDE = 'Auswirkungen/Ursache';
my $ActionsTagDE = 'Eingeleitete Maßnahmen';
my $HistoryTagDE = 'Ticket-Verlauf';
my $ErrorTagEN = 'Incident description';
my $ReasonTagEN = 'Impact/Cause';
my $ActionsTagEN = 'Actions taken';
my $HistoryTagEN = 'Ticket history';

my $DefaultHistoryText = <<"EOT";
$HistoryTagDE:
2009-01-01 00:00 Feuerwerk
EOT
my $DefaultHistoryTextEN = <<"EOT" ;
$HistoryTagEN:
2009-01-01 00:00 Feuerwerk
EOT
my $DefaultHistoryResult = [ ['2009-01-01 00:00',  'Feuerwerk' ] ];

my $RealWorld01_body = <<'EOT';
Dear Sir or Madam,
 
please take care of the following incident:

Malfunctioning service: Jacada (Desktop UAD)
Priority: Q3_P3
Affected customer(s): Quelle Contact
Client: Quelle
Location/no. of employees/of total: Magdeburg: 4/300
Start of incident: 2009-02-09 10:02
End of incident: -
Incident description: Several of our co-workers are experiencing increasing difficulties
logging in to Jacada.
When typing in the username/password combination and confirming by
clicking "Login", the input data disappears, the user has to start
over again and is only logged in after the second attempt; but even
then, the system states "Not connected to CTI" and the favorites on
the left-hand side are not displayed, which poses a big problem for
the agents.
If the affected co-worker logs in at another workstation (B),
everything works fine; a new attempt logging in at workstation (A)
yields the same results described above.
If any other agent logs in at workstation (A), everything works just
fine.
Currently this affects the following agents:
Carola.Schmidt2 - workstation SGCMD5292
Janine.Boege - workstation SGCMD5302
Anika.Saborowski - workstation SGCMD5258
--Beispiel:
PC-Name: SGCMD5292
Login: carola.schmidt2
Business impact: -
Customer perception: (yes/no/partially)no
Prospective duration (if known yet):
Business impact: <OTRS_TICKET_TicketFreeText14>
Customer perception: (yes/no/partially)
Prospective duration (if known yet):
Incident was announced (to / at): Jacada / 2009-02-09 10:02
ID of incident (at external service provider): -
        
Ticket history:
2009-02-09 10:11 Noris ITO informiert
2009-02-09 10:17 RM Noris ITO nicht zuständig
2009-02-09 10:43 RM an Clearingstelle zur Prüfung
2009-02-09 11:31 RM von Clearingstelle, Jacada Support zuweissen
2009-02-09 11:53 Störungsbild wird für Supporter übersetzt
2009-02-09 12:00 Clearingstelle über zeitverzögerung informiert
2009-02-09 12:03 Übersetzung beendet
2009-02-09 12:13 Mail an Supporter

Actions taken:
        
No particular action taken.

Impact/Cause:

No particular cause known.

Yours sincerely

Pippilotta Victualia Ephraimstochter (...) Langstrumpf

-- 
Signature
EOT

my $RealWorld01_error = <<'EOT';
Several of our co-workers are experiencing increasing difficulties
logging in to Jacada.
When typing in the username/password combination and confirming by
clicking "Login", the input data disappears, the user has to start
over again and is only logged in after the second attempt; but even
then, the system states "Not connected to CTI" and the favorites on
the left-hand side are not displayed, which poses a big problem for
the agents.
If the affected co-worker logs in at another workstation (B),
everything works fine; a new attempt logging in at workstation (A)
yields the same results described above.
If any other agent logs in at workstation (A), everything works just
fine.
Currently this affects the following agents:
Carola.Schmidt2 - workstation SGCMD5292
Janine.Boege - workstation SGCMD5302
Anika.Saborowski - workstation SGCMD5258
--Beispiel:
PC-Name: SGCMD5292
Login: carola.schmidt2
EOT

chomp $RealWorld01_error;

my @tests_einfach = (
    # Leerer String, nichts zurück
    ['', {}],
    
    # ohne Ticket-Verlauf nichts zurück
    [ "$ErrorTagDE: Fehlerbeschreibung\n\n\n",
      { }],
      
    # Komplett-Test aller Tags auf deutsch    
    [ "$ErrorTagDE: Fehlerbeschreibung\n"
      . "$ReasonTagDE\nUrsache\n"
      . "$ActionsTagDE\nMaßnahmen\n"
      . "$DefaultHistoryText",
      { error => 'Fehlerbeschreibung',
        reason => 'Ursache',
        actions => 'Maßnahmen',
        history => $DefaultHistoryResult } ],
    
    # Komplett-Test aller Tags auf englisch
    [ "$ErrorTagEN: Fehlerbeschreibung\n"
      . "$ReasonTagEN\nUrsache\n\n"
      . "$ActionsTagEN\nMaßnahmen\n\n"
      . "$DefaultHistoryTextEN",
      { error => 'Fehlerbeschreibung',
        reason => 'Ursache',
        actions => 'Maßnahmen',
        history => $DefaultHistoryResult } ],
    
    # Detail-Tests für Tags, bei denen der Wert nach ':' folgt
    # alle anhand des Tags error:
     
    # Mehrzeilig
    [ "$ErrorTagDE: Fehlerbeschreibung\nzweite Zeile.\n$DefaultHistoryText", 
      { error => "Fehlerbeschreibung\nzweite Zeile.",
        history => $DefaultHistoryResult }],
    
    # Mit anderem Merkmal dahinter
    [ "$ErrorTagDE: Fehlerbeschreibung\nzweite Zeile.\nPriorität: Bla\n$DefaultHistoryText", 
      { error => "Fehlerbeschreibung\nzweite Zeile.",
        history => $DefaultHistoryResult }],
      
    # Mit englischem anderen Merkmal dahinter
    [ "$ErrorTagDE: Fehlerbeschreibung\nzweite Zeile.\nPriority: Bla\n$DefaultHistoryText", 
      { error => "Fehlerbeschreibung\nzweite Zeile.",
        history => $DefaultHistoryResult }],
    
    # Mit : in der zweiten Zeile, danach anderes Merkmal
    [ "$ErrorTagDE: Fehlerbeschreibung\nzweite: Zeile.\nPriority: Bla\n$DefaultHistoryText", 
      { error => "Fehlerbeschreibung\nzweite: Zeile.",
        history => $DefaultHistoryResult }],
      
    # Davor und danach anderes Merkmal
    [ "Business impact: Blabla\n"
      . "$ErrorTagDE: Fehlerbeschreibung\n"
      . "Priority: Bla\n"
      . $DefaultHistoryText,
      { error => 'Fehlerbeschreibung',
        history => $DefaultHistoryResult }],
        
    # Danach ein mehrzeiliges Merkmal
    [ "$ErrorTagDE: Fehlerbeschreibung\n\n"
      . "$ReasonTagDE\nBla\n"
      . "$DefaultHistoryText",
      { error => 'Fehlerbeschreibung',
        reason => 'Bla',
        history => $DefaultHistoryResult }],
    
    # Whitespace davor und danach entfernen
    [ "$ErrorTagDE:   Fehlerbeschreibung  \n$DefaultHistoryText", 
      { error => 'Fehlerbeschreibung',
        history => $DefaultHistoryResult }],
    
    # Umgang mit überflüssigen Zeilenumbrüchen
    [ "$ErrorTagDE:\n  Fehlerbeschreibung  \n\n\n$DefaultHistoryText", 
      { error => 'Fehlerbeschreibung',
        history => $DefaultHistoryResult }],
        
    # Mehrzeile Felder, am Beispiel von Reason
    [ "$ReasonTagDE\nUrsache\n$DefaultHistoryText",
      { reason => 'Ursache',
        history => $DefaultHistoryResult }],
        
    # Whitespace und Leerzeichen entfernen
    [ "  $ReasonTagDE \n\n  Ursache  \n  $DefaultHistoryText  ",
      { reason => 'Ursache',
        history => $DefaultHistoryResult }],
    
    # Nun zu den Tests des Verlaufs
    # Zusätzliches ':' nach Datum
    [ "$HistoryTagDE\n\n2009-01-01 00:00: Feuerwerk",
      { history => $DefaultHistoryResult } ],
      
    # Mehrere Verlaufszeilen, teilweise mehrzeilig
    [ "$HistoryTagDE\n\n"
      . "2009-01-01 00:00 Feuerwerk\n"
      . "2009-01-01 01:00 Ein Uhr, was um 2009-01-01 01:00 kein besonderes\n"
      . "Ereignis ist.\n"
      . "2009-01-01 02:00 Zwei Uhr.\n"
      . "2009 ist jetzt 2 Stunden alt.\n"
      . "2009-01-01 03:00 Jetzt ist es drei Uhr. Leerzeile folgt.\n\n"
      . "2009-01-01 04:00 Vier Uhr. Nicht mehr Teil des Verlaufs, wg. Leerzeile.\n\n"
      . "Mit freundlichen Grüßen\n"
      . "Karla Reporter\n"
      . "Jetzt ist der Verlauf beendet.\n"
      . "2009-01-01 05:00 Nicht mehr Teil des Verlaufs\n",
      { history => [
            [ "2009-01-01 00:00", "Feuerwerk"],
            [ "2009-01-01 01:00", "Ein Uhr, was um 2009-01-01 01:00 kein besonderes\nEreignis ist."],
            [ "2009-01-01 02:00", "Zwei Uhr.\n2009 ist jetzt 2 Stunden alt."],
            [ "2009-01-01 03:00", "Jetzt ist es drei Uhr. Leerzeile folgt."],
        ]
      }
    ],
    
    # Whitespace entfernen (außer innerhalb des Datums)
    [ "$HistoryTagDE\n\n"
      . "  2009-01-01   00:00    Feuerwerk   \n",
      { history => 
        [ [ "2009-01-01   00:00", "Feuerwerk"] ],
      }
    ],
    
    # Ein Beispiel aus dem echten Leben
    [ $RealWorld01_body,
      { reason => 'No particular cause known.',
        actions => 'No particular action taken.',
        error => $RealWorld01_error,
        history =>  [
            ['2009-02-09 10:11', 'Noris ITO informiert'],
            ['2009-02-09 10:17', 'RM Noris ITO nicht zuständig'],
            ['2009-02-09 10:43', 'RM an Clearingstelle zur Prüfung'],
            ['2009-02-09 11:31', 'RM von Clearingstelle, Jacada Support zuweissen'],
            ['2009-02-09 11:53', 'Störungsbild wird für Supporter übersetzt'],
            ['2009-02-09 12:00', 'Clearingstelle über zeitverzögerung informiert'],
            ['2009-02-09 12:03', 'Übersetzung beendet'],
            ['2009-02-09 12:13', 'Mail an Supporter'] ],
        'standort/ma' => 'Magdeburg: 4/300',
      } ],
); # Ende von @tests_einfach


my @tests_komplex = (
    # letzter passender Artikel hat Vorrang, 
    # note-external wird akzeptiert (#10007075)
    [ [ { ArticleType => 'email-external',
            SenderType => 'agent',
            Body => $DefaultHistoryText },
        { ArticleType => 'note-external',
            SenderType => 'agent',
            Body => $DefaultHistoryText. "2009-02-01 00:00 Februaranfang" }
        ], 
      { history => [ ['2009-01-01 00:00',  'Feuerwerk' ],
                     ['2009-02-01 00:00',  'Februaranfang' ] ] },
      1
    ],
    
    # email-external akzeptieren
    [ [ { ArticleType => 'email-external',
            SenderType => 'agent',
            Body => $DefaultHistoryText }, ],
      { history => $DefaultHistoryResult },
      0 ],
    
    # ... aber nicht vom Kunden!
    [ [ { ArticleType => 'email-external',
            SenderType => 'customer',
            Body => $DefaultHistoryText }, ],
      { },
      undef ],
    
    # note-internal nicht akzeptieren
    [ [ { ArticleType => 'note-internal',
            SenderType => 'agent',
            Body => $DefaultHistoryText }, ],
      { },
      undef ],
      
    # email-internal nicht akzeptieren
    [ [ { ArticleType => 'email-internal',
            SenderType => 'agent',
            Body => $DefaultHistoryText }, ],
      { },
      undef ],
    
    # Vorrang von opcall-external
    [ [ { ArticleType => 'note-external-opcall',
            SenderType => 'agent',
            Body => $DefaultHistoryText },
        { ArticleType => 'email-external',
            SenderType => 'agent',
            Body => $DefaultHistoryText. "2009-02-01 Februaranfang" }
        ], 
      { history => $DefaultHistoryResult },
      0 ],
); # Ende von @tests_komplex    
    



# Einfache Tests

my $i = 0;
for my $testcase (@tests_einfach) {
    my ($body, $expected) = @$testcase;
    my $article = {
        ArticleType => 'email-external',
        SenderType => 'agent',
        Body => $body 
    };
    my $computed = NorisOtrsReport::QuelleProcessor::parse_verlauf( [ $article ] );
    # warn "got: " . pp($computed). "\n";
    $expected->{article} = $article if %$expected;
    is_deeply $computed, $expected, "tests_einfach[$i]:\n$body -> " . pp($expected);
    ++$i;
}

# Komplexe Tests

$i = 0;
for my $testcase (@tests_komplex) {
    my ($articles, $expected, $article_index) = @$testcase;
    my $computed = NorisOtrsReport::QuelleProcessor::parse_verlauf($articles);
    $expected->{article} = $articles->[$article_index] if defined $article_index;
    is_deeply $computed, $expected, "tests_komplex[$i]:\n" . pp($articles) . " -> " . pp($expected);
    ++$i;
}


# Probleme aus der realen Welt


{ # aus #10005499, Artikel 33
    my $body = << "EOT";
$HistoryTagDE

2009-02-03 16:00 Rückmeldung von Siemens: Information durch das BLS
und dem Systemspezialist.:Die Erkennung der bekannten Wörter ist im Wörterbuch
hinterlegt. Dabei wird es zwischen System und Persönlichen Wörterbüchern
unterschieden. Die persönliche Wörterbücher ( vocabulary ) des Agenten sind in
Contact Datenbank abgespeichert. In der Tabelle "SPELLDIRTIONARY" gibt es ein
Eintrag pro Agent. Bei der Migration wurden persönlicher Wörterbücher nicht
umgezogen.Das hat den Effekt, dass die Agenten im neuen System ihre alten Einträge
nicht mehr sehen. Um das zu beseitigen musste die Tabelle SPELLDIRCTIONARY
exportiert und in die neue Datenbank importiert werden. Das
wurde ausgeführt und in die neue Datenbank eingebracht. Das haben wir am
20.01.2009 ausgeführt. Die Agenten, die vor der Migration ein persönliches
Wörterbuch gehabt haben, haben dieses wieder. Neue Mitarbeiter oder Mitarbeiter,
die jetzt keine Einträge in ihren Wörterbüchern haben, müssen ihre Wörterbücher
neu ausfüllen. Dieses ist die Lösung zu diesem Problem.
EOT
    my $result = << 'EOT';
Rückmeldung von Siemens: Information durch das BLS
und dem Systemspezialist.:Die Erkennung der bekannten Wörter ist im Wörterbuch
hinterlegt. Dabei wird es zwischen System und Persönlichen Wörterbüchern
unterschieden. Die persönliche Wörterbücher ( vocabulary ) des Agenten sind in
Contact Datenbank abgespeichert. In der Tabelle "SPELLDIRTIONARY" gibt es ein
Eintrag pro Agent. Bei der Migration wurden persönlicher Wörterbücher nicht
umgezogen.Das hat den Effekt, dass die Agenten im neuen System ihre alten Einträge
nicht mehr sehen. Um das zu beseitigen musste die Tabelle SPELLDIRCTIONARY
exportiert und in die neue Datenbank importiert werden. Das
wurde ausgeführt und in die neue Datenbank eingebracht. Das haben wir am
20.01.2009 ausgeführt. Die Agenten, die vor der Migration ein persönliches
Wörterbuch gehabt haben, haben dieses wieder. Neue Mitarbeiter oder Mitarbeiter,
die jetzt keine Einträge in ihren Wörterbüchern haben, müssen ihre Wörterbücher
neu ausfüllen. Dieses ist die Lösung zu diesem Problem.
EOT
    chomp $result;
    my $Article = { 
            ArticleType => 'email-external',
            SenderType => 'agent',
            Body => $body 
        };
    is_deeply 
        NorisOtrsReport::QuelleProcessor::parse_verlauf( [$Article] ),
        { history => [ [ "2009-02-03 16:00", $result ] ],
          article => $Article },
        'R01: Zufälliges Datum am Zeilenanfang nicht als neues Ereignis interpretieren';
 }

{ # aus #10005907, Artikel 13
    my $body = <<"EOT";
$HistoryTagDE:
2009-02-10 14:00 OpCall: Herr Oberg klärt intern 

Details:
Berlin 10/10
Alle Webanwendungen 
EOT
    my $Article = { 
            ArticleType => 'email-external',
            SenderType => 'agent',
            Body => $body
        };
    is_deeply 
        NorisOtrsReport::QuelleProcessor::parse_verlauf( [ $Article ] ),
        { history => [ [ "2009-02-10 14:00", "OpCall: Herr Oberg klärt intern"] ],
          article => $Article },
        'R02: Verlauf nach doppelter Leerzeile beenden.';
}

 
    
    
