#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use lib '/opt/noris-otrs-reports';
use lib '/usr/share/otrs/Kernel/cpan-lib';

use NorisOtrsReport::Framework qw(:data :convert generate);
use NorisOtrsReport::Renderer::Template;
use NorisOtrsReport::Renderer::SimpleXML;
use NorisOtrsReport::Renderer::Excel;
use NorisOtrsReport::Renderer::CSV;
use NorisOtrsReport::Processor::ALNO;
use NorisOtrsReport::Processor::Debug;
use DateTime::Duration;
use DateTime::Format::Strptime;
use Date::Manip;
use Data::Dump qw(pp);
use Getopt::Long qw(GetOptions :config no_ignore_case);
use Pod::Usage qw(pod2usage);

my $DateParser = new DateTime::Format::Strptime(
    pattern   => '%Y%m%d%T',
    locale    => 'de_DE',
    time_zone => 'local',
    on_error  => 'croak'
);

GetOptions(
    'debug|d'    => \my $opt_debug,     # no 'evaluate' step
    'von=s'      => \my $opt_von,
    'bis=s'      => \my $opt_bis,
    'monat|m'    => \my $opt_monat,
    'woche|w'    => \my $opt_woche,
    'tag|t'      => \my $opt_tag,
    'format|f=s' => \my $opt_format,    # html, xml, json
    'filename=s' => \my $opt_filename,     # no 'evaluate' step
    'queue|q=s'  => \my @opt_queues,
    'raw'        => \my $opt_raw,
    'load=s'     => \my $opt_load,
    'help|h'     => \my $opt_help,
) or pod2usage( -verbose => 0, -exitval => 2 );

pod2usage( -verbose => 2, -exitval => 0 ) if $opt_help;

my @TicketNumbers = @ARGV;
my ( $Von, $Bis, $Duration );

if ( $opt_monat ) {
    $Duration = DateTime::Duration->new( months => 1 );
}
elsif ( $opt_woche ) {
    $Duration = DateTime::Duration->new( weeks => 1 );
}
elsif ( $opt_tag ) {
    $Duration = DateTime::Duration->new( days  => 1 );
}

if ( defined $Duration && !$opt_von && !$opt_bis ) {
    $opt_bis = "now";
}
if ($opt_von) {
    $Von = $DateParser->parse_datetime( ParseDate($opt_von) )
      or die "Cannot parse --von $opt_von";
    $Bis = $Von + $Duration if defined $Duration;
}
if ($opt_bis) {
    $Bis = $DateParser->parse_datetime( ParseDate($opt_bis) )
      or die "Cannot parse --bis $opt_bis";
    $Von = $Bis - $Duration if defined $Duration;
}
if ( !defined $Von && !defined $Bis && !@TicketNumbers ) {
    die "Insufficient date specification!" unless defined $Von and defined $Bis;
}

@opt_queues = ( 'sd::alno', ) unless @opt_queues;

# Renderer
if ( !$opt_format ) {
    if ($opt_debug) {
        $opt_format = 'json';
    }
    elsif ($opt_raw) {
        $opt_format = 'xml';
    }
    else {
        $opt_format = 'xls';
    }
}

my $Fields = [
    'Ticketnummer',
    'Melder',
    'Melder betroffen',
    'Priorität',
    'e-Mail-Adresse',
    'Telefonnummer',
    'Heimat Standort',
    'derzeitiger Standort',
    'Tickettyp',
    'Kategorie',
    'Fehlerbeschreibung',
    'Dienstleister',
    'Ticket-ID des Dienstleisters',
    'Bearbeitungsbegin',
    'Bearbeitungsende',
    'Bearbeitungszeit',
    'Ticketverlauf',
    'Eingeleitete Maßnahmen',
    'Bearbeitungszeit nn',
    'Fehlerursache',
    'Status',
    'Termin',
];

my $Renderer;
my $ExistingTickets = {};
if ( $opt_format =~ m/^json$/i ) {
    $Renderer = NorisOtrsReport::Renderer::JSON->new();
}
elsif ( $opt_format =~ m/^xml$/i ) {
    $Renderer = NorisOtrsReport::Renderer::SimpleXML->new(
        AdditionalData => {
            'report-start' => { date => convert_epoch_to_date_hash( $Von->epoch() ) },
            'report-end'   => { date => convert_epoch_to_date_hash( $Bis->epoch() ) },
            type       => 'monthly',
            version    => 1,
            subversion => 0,
            created    => { date => convert_epoch_to_date_hash( time() ) },
        },
        SimpleXMLAttributes => {
            NoAttr     => 0,
            RootName   => 'ticketlist',
            KeyAttr    => { customtext => 'key', customdate => 'key' },
            ContentKey => 'value',
            ValueAttr  => { 'created' => 'date' },
            NoSort     => 0,
            GroupTags  => { 'state-history' => 'state-event', 'history' => 'item' },
            XMLDecl    => 1,
        },
    );
}
elsif ( $opt_format =~ m/^xls$/i ) {
    my %Config = $opt_filename ? ( Filename => $opt_filename ) : ();
    $Renderer = NorisOtrsReport::Renderer::Excel->new( Fields => $Fields, %Config );
}
elsif ( $opt_format =~ m/^csv$/i ) {
    $Renderer = NorisOtrsReport::Renderer::CSV->new( Fields => $Fields, );
}
else {
    die "format '$opt_format' not implemented";
}

my $Processors;
if ($opt_debug) {
    $Processors = [ NorisOtrsReport::Processor::Debug->new( Renderer => $Renderer ) ];
}
else {
    $Processors = [
        NorisOtrsReport::Processor::ALNO->new(
            Renderer        => $Renderer,
            Typ             => $opt_raw ? 'raw' : 'monthly',
            ExistingTickets => $ExistingTickets,
            Start           => $Von->epoch(),
        )
    ];
}

my $Search;
if (@TicketNumbers) {
    $Search = [ map { TicketNumber => $_ }, @TicketNumbers ];
}
else {
    $Search = [
        {
            Queues                    => \@opt_queues,
            TicketCreateTimeNewerDate => $Von->strftime('%F %T'),
            TicketCreateTimeolderDate => $Bis->strftime('%F %T'),
        },
    ];
}

my $Collectors;
if ($opt_load) {
    $Collectors = undef;
    $Search     = undef;
}
else {
    $Collectors = [
        \&data_otrs_ticket,
        \&data_customtext,
        \&data_customdate,
        \&data_workedtime,
        \&data_state_history,
        make_data_articles(
            qw(
email-external email-notification-ext
email-internal
phone webrequest
note-external subticket-external
            ),
        ),
    ];
}
generate (
    UserID      => 1,
    Search      => $Search,
    LoadFixture => $opt_load,
    Conversions => $opt_debug ? [] : [
        [ Created             => \&convert_isodate_to_date_hash, 'created' ],
        [ CreateTime          => \&convert_isodate_to_date_hash, 'created' ],
        [ RealTillTimeNotUsed => \&convert_epoch_to_date_hash,   'pending-until' ],
    ],
    Collectors => $Collectors,
    Processors => $Processors,
);

=head1 alno-report

alno-report - Erstellen von Berichten über Tickets für den Kunden alno

=head1 SYNOPSIS

alno-report [options]

 Optionen:
        
        --debug, -d      Zum Debuggen, es werden nur die Rohdaten geholt, ohne "eval"
        --incidents      Einen OP-Call-Bericht erstellen ("opcall" auf "ja")
        --von DATUM      Beginn des Berichts-Zeitraums setzen
        --bis DATUM      Ende des Berichts-Zeitraums setzen
        --monat          Berichtszeitraum ist ein Monat. Wenn weder --von noch --bis
                         angegeben ist, ist es der Monat bis gerade eben.
        --woche          Berichtszeitraum ist eine Woche. Wenn weder --von noch --bis
                         angegeben ist, ist es die Woche bis gerade eben.
        --tag            Berichtszeitraum ist ein Tag. Wenn weder --von noch --bis
                         angegeben ist, sind es die 24 h bis gerade eben.
        --format, -f     Setzt das Ausgabeformat. Möglich ist xls, csv, xml oder json.
        --filename       Der Name der Datei in der das Ergebniss gespeichert wird.
                         Funktioniert nur mit der Option '--format xls'
        --queue, -q      Welche Queues durchsucht werden. Optional.
        --raw            Der Kunde wollte auch "Rohdaten" geliefert bekommen. --raw
                         macht das.
        --load DATEI     Lädt Tickets aus einer Fixture-Datei und erstellt dafür einen
                         Bericht. Zum Testen gedacht. Fixture-Dateien kann man einfach
                         mit "--debug --format=json" erstellen.
        --help, -h       Anzeige dieser Gebrauchsanleitung

=cut
