package NorisOtrsReport::Processor::Quelle;

use strict;
use warnings;
use utf8;

use Moose;
use Moose::Util::TypeConstraints;
use NorisOtrsReport::Framework qw(:convert data_customtext);

extends 'NorisOtrsReport::Processor::List';
has 'Typ' => (isa => 'Str', is => 'ro', required => 1);   # 'opcall' | 'daily' | 'knowledge'
has 'ExistingTickets' => (isa => 'HashRef', is => 'ro', default => sub { return {} } );
has 'Start' => (is => 'ro', required => 0, isa => 'Int');

=head1 NAME

NorisOtrsReport::Processor::Quelle - Berichte für Quelle aus dem Service Desk

=head1 SYNOPSIS

=for example begin

 use NorisOtrsReport::Processor::Quelle;
 my ($RendererObj, $TypString, $ExistingTicketsHashRef,
     $EpochInt);
 
 NorisOtrsReport::Processor::Quelle->new(
                Renderer => $RendererObj,
                Typ => $TypString,
                ExistingTickets => $ExistingTicketsHashRef,
                Start => $EpochInt,
            );

=for example end

Parameter:
    
=over

=item Renderer

Ein Renderer-Objekt, das das Formatieren der Ausgabe übernimmt. Siehe NorisOtrsReport::Renderer.

=item Typ

Bestimmt die Art des Berichts. Erlaubt sind:
    
=over

=item 'daily'

Tages- oder auch Wochenbericht.

=item 'opcall'

OPC-Liste

=item 'knowledge'

Knowledge-Bericht

=item 'raw'

"Rohformat" für Quelle

=item 'Start'

Beginn des Berichtszeitraums. Es werden nur Tickets berücksichtigt, die nach
Berichtsbeginn noch einen neuen Artikel mit (relevantem) Verlauf erhalten haben.
Relevant heißt, daß der neue Verlauf auch letztlich in den Bericht aufgenommen
wurde.

=back

=item ExistingTickets

Referenz auf ein zu Beginn leeres Hash, in das Processor::Quelle 
einen Eintrag für jede Ticket-Nummer anlegt, die im Bericht auftaucht.
Die Ticket-Nummer steht im Key, der Value ist immer 1.

Das Hash kann dann auch im Renderer z.B. als C<AdditionalData> verwendet werden,
wenn z.B. die Darstellung verlinkter Tickets davon abhängt, ob das andere Ticket 
auch im Bericht enthalten ist.

=back
        
=head1 DESCRIPTION

Dies Klasse übernimmt es, aus den Ticket-Daten die für das
Reporting an Quelle nötigen Daten zu destillieren.

Siehe auch die Dokumentation zu NorisOtrsReport::Framework.

Siehe C<parse_verlauf> für eine Liste der Felder, die aus dem Ticket-Verlauf geparst werden (und nicht aus TicketFreeText).

=cut


my %QuelleStatusMap = (
    'closed' => 'Geschlossen',
    'closed successful' => 'Geschlossen',
    'closed unconfirmed' => 'Gelöst',
    'closed unsuccessful' => 'Geschlossen',
    'closed at first call' => 'Geschlossen',
    'needinfo' => 'Gelöst',
    'new' => 'in Bearbeitung',
    'open' => 'in Bearbeitung',
    'parking' => 'in Bearbeitung',
    'pending auto close' => 'in Bearbeitung',
    'pending auto close+' => 'in Bearbeitung',
    'pending auto close-' => 'in Bearbeitung',
    'pending auto close?' => 'in Bearbeitung',
    'pending reminder' => 'Wiedervorlage',
    'pending service provider' => 'in Bearbeitung',
    'remind' => 'in Bearbeitung',
    'retry' => 'in Bearbeitung',
    'retry service provider' => 'in Bearbeitung',
    'when back' => 'in Bearbeitung',
);

my %QuelleResolvedStatus = (
    'closed' => 'Geschlossen',
    'closed successful' => 'Geschlossen',
    'closed unconfirmed' => 'Gelöst',
    'closed unsuccessful' => 'Geschlossen',
    'closed at first call' => 'Geschlossen',
    'needinfo' => 'Gelöst',
);

=head4 date_hash_to_quelle_format()

Umwandeln eines Hashes in das Berichtsformat.

=for example begin

use NorisOtrsReport::Processor::Quelle qw(date_hash_to_quelle_format);
my $date = 
    {   
        day=>"5", epoch=>"1233836217", hour=>"12",
        minute=>"16", month=>"2", second=>"57", year=>"2009" 
    };

=for example end

=begin testing

use NorisOtrsReport::Processor::Quelle ;
my $date = 
    {   
        day=>"5", epoch=>"1233836217", hour=>"12",
        minute=>"16", month=>"2", second=>"57", year=>"2009" 
    };
is NorisOtrsReport::Processor::Quelle::date_hash_to_quelle_format($date), 
    '2009-02-05 12:16';

=end testing

=cut

sub date_hash_to_quelle_format {
    my ($DateHashRef) = @_;
    if (defined %$DateHashRef && %$DateHashRef) {
        return sprintf "%4d-%02d-%02d %02d:%02d",
                ($DateHashRef->{year}, $DateHashRef->{month}, $DateHashRef->{day},
                 $DateHashRef->{hour}, $DateHashRef->{minute});
    }
    else {
        return '';
    }
}


=begin testing

use NorisOtrsReport::Processor::Quelle ;
is NorisOtrsReport::Processor::Quelle::epoch_diff_to_quelle_format(900), '00:15';
is NorisOtrsReport::Processor::Quelle::epoch_diff_to_quelle_format(
        2*24*60*60 + 15*60*60 + 31*60), 
    '2d 15:31';

=end testing

=cut

sub epoch_diff_to_quelle_format {
    my ($EpochDiff) = @_;
    return undef unless defined $EpochDiff;
    my $minutes = int($EpochDiff / 60);
    my $hours = int($minutes / 60);
    my $days = int($hours / 24);
    return
        ($days ? $days . "d " : "")
        . sprintf "%02d:%02d", ($hours - 24 * $days, $minutes - 60 * $hours)
}

sub get_last_article {
    my ($Articles, $Type) = @_;
    for (my $i=$#$Articles; $i >= 0; --$i) {
        return $Articles->[$i]->{Body} if $Articles->[$i]->{ArticleType} eq $Type;
    }
    return '';
}    

=begin testing

use NorisOtrsReport::Processor::Quelle ;
is_deeply 
    NorisOtrsReport::Processor::Quelle::parse_prio('Prio 3 QC / Prio 4 Primondo'),
    [3,4];
is_deeply
    NorisOtrsReport::Processor::Quelle::parse_prio('Prio 3 QC / Prio X Primondo'),
    [3,'X'];
is_deeply 
    NorisOtrsReport::Processor::Quelle::parse_prio(undef), 
    ['',''];
is_deeply 
    NorisOtrsReport::Processor::Quelle::parse_prio('Unfug'), 
    ['Unfug', 'Unfug'];
is_deeply 
    NorisOtrsReport::Processor::Quelle::parse_prio('Q1_P2'),
    [1,2];

=end testing

=cut

sub parse_prio {
    my ($CombinedPrio) = @_;
    return ['', ''] unless $CombinedPrio;
    if ($CombinedPrio =~ m{^Prio (\w) QC / Prio (\w) Primondo$}
        || $CombinedPrio =~ m{^Q(\w)_P(\w)$})
    {
        return [$1,$2];
    }
    return [$CombinedPrio, $CombinedPrio];
}

=head4 parse_standort_ma()

Einziger Parameter ist der String, wie er im Freitextfeld steht. Erkannt werden z.B.:

=over

=item *
C<'Usedom 50/50, Westerauge 20/40'>

=item *
C<'Usedom ?; Westerauge 20/50'>

=item *

C<'Usedom, Westerauge ?/?'>

=back

Zurückgegeben werden zwei Listen-Referenzen:
    
    ( [ 'Standort1', 'Standort2', ...],
      [ 'XX/YY1', 'XX/YY2' ] )

Wenn die Mitarbeiteranzahl zu einem Standort nicht angegeben ist, wird
stattdessen C<undef> übergeben.

=cut

sub parse_standort_ma {
    my ($StandortMA) = @_;
    return ([], []) unless $StandortMA;
    my @entries = split /\s*[,;]\s*/, $StandortMA;
    if (! @entries) {
        @entries = ($StandortMA);
    }
    my @parsed = map $_ =~ /^\s*([^:?]+?)\s*[:\s]\s*([-?\d]+(?:\s*\/\s*[-?\d]*)?)\s*$/ ? [$1, $2] : [ $_, undef], @entries;
    return ([map $_->[0], @parsed], [map $_->[1], @parsed])
}

sub find_resolving_article {
    my ($Articles) = @_;
    for my $Article (reverse @{$Articles}) {
        if (exists $QuelleResolvedStatus{$Article->{State}}) {
            return $Article;
        }
    }
    return undef;
}

sub find_resolving_event {
    my ($Events) = @_;
    for my $Event (reverse @{$Events}) {
        if (exists $Event->{NewState}
            && exists $QuelleResolvedStatus{$Event->{NewState}}
            && ! exists $QuelleResolvedStatus{$Event->{OldState}})
        {
            return $Event;
        }
    }
    return undef;
}

=head4 parse_verlauf()

Sucht sich aus einer Artikelliste den letzten Artikel mit
einem Ticket-Verlauf heraus und parst die darin enthaltenen
Daten.

Dabei wird die E-Mail als eine Folge von Abschnitten interpretiert.
Ein neuer Abschnitt beginnt:
        
=over

=item *

Nach einer Leerzeile, außer an Anfang eines Abschnittes ODER

=item *

Wenn eine Zeile mit einem der Tags in C<@AlleTags> beginnt

=back

Am Beginn eines Abschnittes wird zuerst ein "Tag" aus C<@AlleTags> erwartet,
gefolgt von einem ":" oder einem Zeilenende. Der Rest des Abschnittes gehört
dann zum Wert des Tags. Beginnt ein Abschnitt mit keinem bekannten Tag, wird
er ignoriert.

Zurückgegeben werden aber stets nur die bekannten, in C<%TagMap> beschriebenen Tags.

=for example begin

my @ArticleList = ( );
my $ResultHashRef = parse_verlauf(\@ArticleList);

=for example end

Parameter:

=over

=item ArticleList

Liste der relevanten Artikel.

=back

Zurückgegeben wird eine Referenz auf ein Hash mit folgenden Einträgen:
        
=over

=item history

Enthält die Daten aus dem Ticket-Verlauf in der Form einer Listen-Referenz. Jedes Listenelemen entspricht einer Verlaufszeile und besteht wiederum aus einer Referenz auf eine 2-elementige Liste, C<[$DatumString, $BeschreibungString]>.

=item reason

Die Ursache, als String

=item error

Das Fehlerbild, als String

=item actions

Ergriffene Maßnahmen, als String

=item article

Die OTRS-Daten zum Artikel, der den benutzten Verlauf enthält.

=item standort/ma

Standort/MA als Text. Kann mit C<parse_verlauf()> weiterverarbeitet werden.

=back


=begin testing

use utf8;
my $Body = <<'EOT';
Beschreibung der Störung: Irgendwas kaputt

Ticket-Verlauf:
2009-02-09 14:59 Irgendwas gemeldet

Eingeleitete Maßnahmen:
        
Irgendwas getan.

Auswirkungen/Ursache:

Hat irgendeinen Grund.
EOT

my @Articles = ( {
    ArticleType => 'note-external',
    SenderType => 'agent',
    Body => $Body,
});

my $Result = NorisOtrsReport::Processor::Quelle::parse_verlauf(\@Articles);
is ref $Result, 'HASH';
is $Result->{error}, 'Irgendwas kaputt';
is $Result->{reason}, 'Hat irgendeinen Grund.';
is $Result->{actions}, 'Irgendwas getan.';
is_deeply $Result->{history}, [ ['2009-02-09 14:59', 'Irgendwas gemeldet'] ];
is $Result->{article}->{Body}, $Body;

=end testing

=cut

# $VerlaufRE Erkennt den Verlauf innerhalb eines Tickets
my $VerlaufRE = qr/(?:^|\n)\s*(?:Ticket[-\s]?(?:Verlauf|History)):?\s*\n[\n\s]*20[-\d :.]{7,17}\d/i;
        
# $VerlaufTrennerRE trennt die Ereignisse in einem Verlauf voneinander
my $VerlaufTrennerRE = qr/\n(?=20\d\d-\d\d-\d\d\s+\d\d[-:.]\d\d(?:[-:.]\d\d)?[: ]+)/;
        
# $VerlaufTeileRE erkennt Datum und Text in einem Ereignis
my $VerlaufTeileRE = qr/^(20[\d]{2}-[\d]{2}-[\d]{2}\s+[\d]{2}[-:.][\d]{2}(?:[-:.][\d]{2})?)([:\s]+)(|\S.*)$/s;

my @AlleTags = ('gestörter Service', 'Priorität', 'betroffene Kunde(n)', 'Mandant', 'Standort/Anz. MA/von Gesamt', 
               'Störungsbeginn', 'Störungsende', 'Beschreibung der Störung', 'Auswirkung(en) (Business-Impact)',
               'Kundenwahrnehmung', 'voraussichtliche Dauer (falls zu diesem Zeitpunkt bekannt)',
               'Störung wurde gemeldet (an/um)', 'Incident-Nummer (externer DL)', 'Eingeleitete Maßnahmen',
               'Auswirkungen/Ursache',
               'Malfunctioning service', 'Priority', 'Affected customer(s)', 'Client', 
               'Location/no. of employees/of total', 'Start of incident',
               'End of incident', 'Incident description', 'Business impact',
               'Customer perception: (yes/no/partially)',
               'Prospective duration (if known yet)',
               'Incident was announced (to / at)',
               'ID of incident (at external service provider)',
               'Actions taken', 'Impact/Cause',
               'Yours sincerely', '-- '
              );
my $AlleTagsForRE = join("|", map quotemeta, @AlleTags);

my $TagStartRE = qr/^(ticket[- ]?verlauf|ticket[- ]?history|Mit freundlichen Grüßen|$AlleTagsForRE)\s*(?=:|$)(?::\s*)?(.*)$/i;

my %TagMap = (
    'ticket-verlauf' => 'history',
    'ticket verlauf' => 'history',
    'ticketverlauf' => 'history',
    'ticket-history' => 'history',
    'ticket history' => 'history',
    'tickethistory' => 'history',
    'auswirkungen/ursache' => 'reason',
    'impact/cause' => 'reason',
    'eingeleitete maßnahmen' => 'actions',
    'actions taken' => 'actions',
    'beschreibung der störung' => 'error',
    'incident description' => 'error',
    'standort/anz. ma/von gesamt' => 'standort/ma',
    'location/no. of employees/of total' => 'standort/ma'
);
    

sub parse_verlauf {
    my ($Articles) = @_;
    my $SelectedArticle;
    # Gibt es einen Artikel-Typ 'note-external-opcall'? Dann diesen nehmen.
    for my $Article (reverse @{$Articles}) {
        if ($Article->{ArticleType} eq 'note-external-opcall') {
            $SelectedArticle = $Article;
            last;
        }
    }
    if (! defined $SelectedArticle) {
        # Dann schauen wir uns die Bodies von externer E-Mail der Agenten an.
        for my $Article (reverse @{$Articles}) {
            if ($Article->{SenderType} eq 'agent'
                && $Article->{ArticleType} =~ /(email|note)-external/
                && $Article->{Body} =~ $VerlaufRE) {
                $SelectedArticle = $Article;
                last;
            }
        }
    }
    return {} unless defined $SelectedArticle;   # kein Verlauf im Ticket
    
    my ($Tag, $NextTag);
    my $Body = $SelectedArticle->{Body};
    my %Result = (article => $SelectedArticle);
    my @Buffer = ();
    for my $Line (split /\n/, $Body) {
        $Line =~ s/^\s*//;
        $Line =~ s/\s*$//;
        if ($Line =~ $TagStartRE
            || ($Line eq '' && @Buffer && defined $Tag)) 
        {
            ($NextTag, $Line) = $Line eq '' ? (undef, '') : ($1, $2);
            $Result{$Tag} = join("\n", @Buffer) if defined $Tag;
            $Tag = $TagMap{lc $NextTag};
            @Buffer = $Line ? ($Line) : ();
        }
        else {
            push @Buffer, $Line if $Line ne '';
        }
    }
    $Result{$Tag} = join("\n", @Buffer) if defined $Tag;
    
    if (exists $Result{history}) {
        my @Items = split $VerlaufTrennerRE, $Result{history};
        $Result{history} = [ map $_ =~ $VerlaufTeileRE ? [$1, $3] : $_, @Items ]
    }
    return \%Result;
}
            
sub verlauf_roh {
    my ($zeile) = @_;
    return { 
        item => {
            date => convert_isodate_to_date_hash($zeile->[0]),
            value => $zeile->[1],
        }
    };
}
            

sub evaluate {
    my ($Self, $Data) = @_;
    my $ExistingTickets = $Self->{ExistingTickets};
    $ExistingTickets->{$Data->{TicketNumber}} = 1;
    my $Customtext = $Data->{customtext};
    my $Customdate = $Data->{customdate};
    my $Typ = $Self->{Typ};
#     if ($Typ eq 'opcall' && $Customtext->{'Op-Call'}->{value} ne 'Ja') {
#         # Schnellabbruch, da Ticket nicht Opcall-relevant
#         return undef;
#     }
    my $QuelleStatus = $QuelleStatusMap{$Data->{State}};
    my $IncidentStart = 
            exists $Customdate->{'Incident-Start'}
                    ? $Customdate->{'Incident-Start'}->{date}
                    : $Data->{created};
    my %LinkResult = ( Source => [], Target => [] );
    if (exists $Data->{'linked-tickets'}) {
        my $LinkList = $Data->{'linked-tickets'};
        for my $LinkType (qw(MainSub ParentChild)) {
            if (exists $LinkList->{$LinkType}) {
                for my $Direction (keys %{$LinkList->{MainSub}}) {
                    my $Links = $LinkList->{$LinkType}->{$Direction};
                    for my $OtherTicket (values %$Links) {
                        my %OtherData = %$OtherTicket;
                        data_customtext(1, [], $OtherData{TicketNumber}, \%OtherData);
                        push @{$LinkResult{$Direction}}, {
                            TicketNumber => $OtherData{TicketNumber},
                            Kategorie => $OtherData{customtext}->{'Kategorie/Anwendung'}->{value},
                        };
                    }
                }
            }
        }
    }     
    
    # Incident-Resolved, SolutionTime
    my ($IncidentResolved, $SolutionTime, $Verlauf);
    if (exists $Customdate->{'Incident-Resolved'}) {
        $IncidentResolved = $Customdate->{'Incident-Resolved'}->{date};
    }
    elsif (exists $QuelleResolvedStatus{$Data->{State}}) {
        my $ResolveEvent = find_resolving_event($Data->{history});
        if (defined $ResolveEvent) {
            $IncidentResolved = $ResolveEvent->{created};
        }
    }
    if (defined $IncidentResolved) {
        $SolutionTime = $IncidentResolved->{epoch} - $IncidentStart->{epoch};
    }
    # Daily: Weder Ticket noch relevanter Verlauf nach Berichtsbeginn?
    my $VerlaufInfo = parse_verlauf($Data->{articles});
    if (($Self->{Typ} eq 'daily' || $Self->{Typ} eq 'knowledge')
        && defined $Self->{Start}
        && $Data->{created}->{epoch} < $Self->{Start}
        && ( ! %$VerlaufInfo 
             || $VerlaufInfo->{article}->{created}->{epoch} 
                < $Self->{Start}) ) 
    {
        return undef;  # Njet, weglassen
    }
    my ($Standorte, $MA) = exists $VerlaufInfo->{'standort/ma'} ? parse_standort_ma($VerlaufInfo->{'standort/ma'}) : ([],[]);
    
    if ($Typ eq 'raw') {
        # Quelle-Roh-Format
        my $ResultState = {
                type => $Data->{StateType},
                id => $Data->{StateID},
                name => $Data->{State},
            };
        $ResultState->{until} =  {date => $Data->{'pending-until'}} if defined $Data->{'pending-until'};
        my $Result = {
            TicketNumber => $Data->{TicketNumber},
            state => $ResultState,
            article => [map 
                    $_->{ArticleType} =~ /-external$/ 
                        ? {
                            id => $_->{ArticleID},
                            type_id => $_->{ArticleTypeID},
                            type => $_->{ArticleType},
                            value => $_->{Body},
                          } 
                        : (), 
                    @{$Data->{articles}}],
            'state-history' => [map
                    $_->{HistoryType} eq 'StateUpdate'
                        ? {
                            date => $_->{created},
                            value => $_->{Name},
                          }
                        : (),
                    @{$Data->{history}}],
            'sub-ticket' => $LinkResult{Target},
            'main-ticket' => $LinkResult{Source},
            'incident-period' => [$SolutionTime],
            actions => [$VerlaufInfo->{actions}],
            reasons => [$VerlaufInfo->{reason}],
            'error-description' => [$VerlaufInfo->{error}],
            history => [map {verlauf_roh($_)} @{$Verlauf->{history}}],
            'standort/ma' => $VerlaufInfo->{'standort/ma'},
        };
        $Result->{customtext} = $Data->{customtext} if %{$Data->{customtext}};
        $Result->{customdate} = $Data->{customdate} if %{$Data->{customdate}};
        return $Result;
    }
    
    my $Prios = parse_prio($Customtext->{'Prio QC/Primondo'}->{value});
    
    return {
        TicketNumber => $Data->{TicketNumber},
        IncidentStart => date_hash_to_quelle_format($IncidentStart),
        IncidentStartEpoch => $IncidentStart->{epoch},  # fuers Sortieren
        IncidentResolved => date_hash_to_quelle_format($IncidentResolved),
        SolutionTime => epoch_diff_to_quelle_format($SolutionTime),
        Mandant => $Customtext->{Mandant}->{value},
        PrioQuelle => $Prios->[0],
        PrioPrimondo => $Prios->[1],
        Kategorie => $Customtext->{'Kategorie/Anwendung'}->{value},
        Standorte => $Standorte,
        MA => $MA,
        Impact => $Customtext->{'Business-Impact'}->{value},
        Verursacher => $Customtext->{'externe Servicestelle'}->{value},
        Fehlerbild => $VerlaufInfo->{error},
        Massnahmen => $VerlaufInfo->{actions},
        Ursache => $VerlaufInfo->{reason},
        Status => $QuelleStatus,
        Termin => date_hash_to_quelle_format($Customdate->{'Dienstleister-Termin'}->{date}),
        SubTickets => $LinkResult{Target},
        MainTickets => $LinkResult{Source},
        ExterneCallID => $Customtext->{'Externe-ID'}->{value},
        ChangeNummer => $Customtext->{'Störung nach Change'}->{value},
        Verlauf => $VerlaufInfo->{history},
    };
}

sub opcall_cmp {
    return
        (($a->{PrioQuelle}||99) <=> ($b->{PrioQuelle}||99))
        || (($a->{Verursacher}||'zzz') cmp ($b->{Verursacher}||'zzz'))
        || (($b->{IncidentStartEpoch}||1) <=> ($a->{IncidentStartEpoch}||1));
}
    
override 'sort' => sub {
    my ($Self) = @_;
    if ($Self->{Typ} eq 'opcall') {
        $Self->{_List} = [ sort opcall_cmp @{$Self->{_List}} ];
    }
    elsif ($Self->{Typ} eq 'knowledge') {
        $Self->{_List} = [
            sort {
                $a->{TicketNumber} <=> $b->{TicketNumber}
            }
            @{$Self->{_List}}
        ]
    }
    else {
        $Self->{_List} = [
            sort {
                $a->{IncidentStartEpoch} <=> $b->{IncidentStartEpoch}
            } 
            @{$Self->{_List}}
        ];
    }
};

1;
