package NorisOtrsReport::Processor::ALNO;

use strict;
use warnings;
use utf8;

use Moose;
use Moose::Util::TypeConstraints;
use NorisOtrsReport::Framework qw(:convert data_customtext);
use Data::Dump qw(pp);

extends 'NorisOtrsReport::Processor::List';
has 'Typ' => ( isa => 'Str', is => 'ro', required => 1 );    # 'opcall' | 'daily' | 'knowledge'
has 'ExistingTickets' => ( isa => 'HashRef', is => 'ro', default => sub { return {} } );
has 'Start' => ( is => 'ro', required => 0, isa => 'Int' );

=head1 NAME

NorisOtrsReport::Processer::ALNO - Berichte für ALNO aus dem Service Desk

=head1 SYNOPSIS

=for example begin

 use NorisOtrsReport::Processer::ALNO;
 my ($RendererObj, $TypString, $ExistingTicketsHashRef,
     $EpochInt);
 
 NorisOtrsReport::Processer::ALNO->new(
                Renderer => $RendererObj,
                ExistingTickets => $ExistingTicketsHashRef,
                Start => $EpochInt,
            );

=for example end

Parameter:
    
=over

=item Renderer

Ein Renderer-Objekt, das das Formatieren der Ausgabe übernimmt. Siehe NorisOtrsReport::Renderer.

=item ExistingTickets

Referenz auf ein zu Beginn leeres Hash, in das Processer::ALNO 
einen Eintrag für jede Ticket-Nummer anlegt, die im Bericht auftaucht.
Die Ticket-Nummer steht im Key, der Value ist immer 1.

Das Hash kann dann auch im Renderer z.B. als C<AdditionalData> verwendet werden,
wenn z.B. die Darstellung verlinkter Tickets davon abhängt, ob das andere Ticket 
auch im Bericht enthalten ist.

=back
        
=head1 DESCRIPTION

Dies Klasse übernimmt es, aus den Ticket-Daten die für das
Reporting an ALNO nötigen Daten zu destillieren.

Siehe auch die Dokumentation zu NorisOtrsReport::Framework.

Siehe C<parse_verlauf> für eine Liste der Felder, die aus dem Ticket-Verlauf
geparst werden (und nicht aus TicketFreeText).

=cut


my %ALNOStatusMap = (
    'closed'                   => 'Geschlossen',
    'closed at first call'     => 'Geschlossen (Erstlösung)',
    'needinfo'                 => 'in Bearbeitung',
    'new'                      => 'in Bearbeitung',
    'open'                     => 'in Bearbeitung',
    'parking'                  => 'in Bearbeitung',
    'pending auto close'       => 'in Bearbeitung',
    'pending reminder'         => 'in Bearbeitung',
    'pending service provider' => 'in Bearbeitung',
    'remind'                   => 'in Bearbeitung',
    'retry'                    => 'in Bearbeitung',
    'retry service provider'   => 'in Bearbeitung',
    'when back'                => 'in Bearbeitung',
);

my %ALNOResolvedStatus = (
    'closed'               => 'Geschlossen',
    'closed successful'    => 'Geschlossen',
    'closed unconfirmed'   => 'Geschlossen',
    'closed unsuccessful'  => 'Geschlossen',
    'closed at first call' => 'Geschlossen',
);

=head4 date_hash_to_alno_format()

Umwandeln eines Hashes in das Berichtsformat.

=for example begin

use NorisOtrsReport::Processer::ALNO qw(date_hash_to_alno_format);
my $date = 
    {   
        day=>"5", epoch=>"1233836217", hour=>"12",
        minute=>"16", month=>"2", second=>"57", year=>"2009" 
    };

=for example end

=begin testing

use NorisOtrsReport::Processer::ALNO ;
my $date = 
    {   
        day=>"5", epoch=>"1233836217", hour=>"12",
        minute=>"16", month=>"2", second=>"57", year=>"2009" 
    };
is NorisOtrsReport::Processer::ALNO::date_hash_to_alno_format($date), 
    '2009-02-05 12:16';

=end testing

=cut

sub date_hash_to_alno_format {
    my ($DateHashRef) = @_;
    if ( defined %$DateHashRef && %$DateHashRef ) {
        return sprintf "%4d-%02d-%02d %02d:%02d",
          (
            $DateHashRef->{year}, $DateHashRef->{month},
            $DateHashRef->{day},  $DateHashRef->{hour},
            $DateHashRef->{minute}
          );
    }
    else {
        return '';
    }
}

=begin testing

use NorisOtrsReport::Processer::ALNO ;
is NorisOtrsReport::Processer::ALNO::epoch_diff_to_alno_format(900), '00:15';
is NorisOtrsReport::Processer::ALNO::epoch_diff_to_alno_format(
        2*24*60*60 + 15*60*60 + 31*60), 
    '2d 15:31';

=end testing

=cut

sub epoch_diff_to_alno_format {
    my ($EpochDiff) = @_;
    return undef unless defined $EpochDiff;
    my $minutes = int( $EpochDiff / 60 );
    my $hours   = int( $minutes / 60 );
    my $days    = int( $hours / 24 );
    return ( $days ? $days . "d " : "" ) . sprintf "%02d:%02d",
      ( $hours - 24 * $days, $minutes - 60 * $hours );
}

=begin testing

use NorisOtrsReport::Processer::ALNO ;
is_deeply 
    NorisOtrsReport::Processer::ALNO::parse_prio('Prio 3 QC / Prio 4 Primondo'),
    [3,4];
is_deeply
    NorisOtrsReport::Processer::ALNO::parse_prio('Prio 3 QC / Prio X Primondo'),
    [3,'X'];
is_deeply 
    NorisOtrsReport::Processer::ALNO::parse_prio(undef), 
    ['',''];
is_deeply 
    NorisOtrsReport::Processer::ALNO::parse_prio('Unfug'), 
    ['Unfug', 'Unfug'];
is_deeply 
    NorisOtrsReport::Processer::ALNO::parse_prio('Q1_P2'),
    [1,2];

=end testing

=cut

sub find_resolving_article {
    my ($Articles) = @_;
    for my $Article ( reverse @{$Articles} ) {
        if ( exists $ALNOResolvedStatus{ $Article->{State} } ) {
            return $Article;
        }
    }
    return undef;
}

sub find_resolving_event {
    my ($Events) = @_;
    for my $Event ( reverse @{$Events} ) {
        if (   exists $Event->{NewState}
            && exists $ALNOResolvedStatus{ $Event->{NewState} }
            && !exists $ALNOResolvedStatus{ $Event->{OldState} } )
        {
            return $Event;
        }
    }
    return undef;
}

=head4 parse_verlauf()

Sucht sich aus einer Artikelliste den letzten Artikel mit
einem Ticket-Verlauf heraus und parst die darin enthaltenen
Daten.

Dabei wird die E-Mail als eine Folge von Abschnitten interpretiert.
Ein neuer Abschnitt beginnt:
        
=over

=item *

Nach einer Leerzeile, außer an Anfang eines Abschnittes ODER

=item *

Wenn eine Zeile mit einem der Tags in C<@AlleTags> beginnt

=back

Am Beginn eines Abschnittes wird zuerst ein "Tag" aus C<@AlleTags> erwartet,
gefolgt von einem ":" oder einem Zeilenende. Der Rest des Abschnittes gehört
dann zum Wert des Tags. Beginnt ein Abschnitt mit keinem bekannten Tag, wird
er ignoriert.

Zurückgegeben werden aber stets nur die bekannten, in C<%TagMap> beschriebenen Tags.

=for example begin

my @ArticleList = ( );
my $ResultHashRef = parse_verlauf(\@ArticleList);

=for example end

Parameter:

=over

=item ArticleList

Liste der relevanten Artikel.

=back

Zurückgegeben wird eine Referenz auf ein Hash mit folgenden Einträgen:
        
=over

=item history

Enthält die Daten aus dem Ticket-Verlauf in der Form einer Listen-Referenz. Jedes Listenelemen entspricht einer Verlaufszeile und besteht wiederum aus einer Referenz auf eine 2-elementige Liste, C<[$DatumString, $BeschreibungString]>.

=item reason

Die Ursache, als String

=item error

Das Fehlerbild, als String

=item actions

Ergriffene Maßnahmen, als String

=item article

Die OTRS-Daten zum Artikel, der den benutzten Verlauf enthält.

=back


=begin testing

use utf8;
my $Body = <<'EOT';
Beschreibung der Störung: Irgendwas kaputt

Ticket-Verlauf:
2009-02-09 14:59 Irgendwas gemeldet

Eingeleitete Maßnahmen:
        
Irgendwas getan.

Auswirkungen/Ursache:

Hat irgendeinen Grund.
EOT

my @Articles = ( {
    ArticleType => 'note-external',
    SenderType => 'agent',
    Body => $Body,
});

my $Result = NorisOtrsReport::Processer::ALNO::parse_verlauf(\@Articles);
is ref $Result, 'HASH';
is $Result->{error}, 'Irgendwas kaputt';
is $Result->{reason}, 'Hat irgendeinen Grund.';
is $Result->{actions}, 'Irgendwas getan.';
is_deeply $Result->{history}, [ ['2009-02-09 14:59', 'Irgendwas gemeldet'] ];
is $Result->{article}->{Body}, $Body;

=end testing

=cut

# $VerlaufRE Erkennt den Verlauf innerhalb eines Tickets
my $VerlaufRE = qr/(?:^|\n)\s*(?:Ticket[-\s]?(?:Verlauf|History)):?\s*\n[\n\s]*20[-\d :.]{7,17}\d/i;
        
# $VerlaufTrennerRE trennt die Ereignisse in einem Verlauf voneinander
my $VerlaufTrennerRE = qr/\n(?=20\d\d-\d\d-\d\d\s+\d\d[-:.]\d\d(?:[-:.]\d\d)?[: ]+)/;
        
# $VerlaufTeileRE erkennt Datum und Text in einem Ereignis
my $VerlaufTeileRE = qr/^(20[\d]{2}-[\d]{2}-[\d]{2}\s+[\d]{2}[-:.][\d]{2}(?:[-:.][\d]{2})?)([:\s]+)(|\S.*)$/s;

my @AlleTags = (
    'Melder',
    'Melder betroffen',
    'Priorität',
    'e-Mail-Adresse',
    'Telefonnummer',
    'Heimat Standort',
    'derzeitiger Standort',
    'Tickettyp',
    'Kategorie',
    'Fehlerbeschreibung',
    'Dienstleister',
    'Ticket-ID des Dienstleisters',
    'Bearbeitungsbeginn',
    'Bearbeitungsende',
    'Bearbeitungszeit',
    'Eingeleitete Maßnahmen',
    'Fehlerursache',
    'Ticket-Verlauf',
);
my $AlleTagsForRE = join( "|", map quotemeta, @AlleTags );

my $TagStartRE =
qr/^(ticket[- ]?verlauf|ticket[- ]?history|Mit freundlichen Grüßen|$AlleTagsForRE)\s*(?=:|$)(?::\s*)?(.*)$/i;

my %TagMap = (
    'ticket-verlauf'         => 'history',
    'ticket verlauf'         => 'history',
    'ticketverlauf'          => 'history',
    'ticket-history'         => 'history',
    'ticket history'         => 'history',
    'tickethistory'          => 'history',

    'eingeleitete maßnahmen' => 'actions',

    'fehlerursache'          => 'reason',

    'fehlerbild'             => 'error',
    'fehlerbeschreibung'     => 'error',
);
    

sub parse_verlauf {
    my ($Articles) = @_;
    my $SelectedArticle;

    # Dann schauen wir uns die Bodies von externer E-Mail der Agenten an.
    for my $Article ( reverse @{$Articles} ) {
        if (
            $Article->{SenderType} eq 'agent'
            && $Article->{Body} =~ $VerlaufRE
          )
        {
            $SelectedArticle = $Article;
            last;
        }
    }
    return {} unless defined $SelectedArticle;    # kein Verlauf im Ticket

    my $Body   = $SelectedArticle->{Body};
    my %Result = ( article => $SelectedArticle );
    my @Buffer = ();
    my $OldTag;
    for my $Line ( split /\n/, $Body ) {
        $Line =~ s/^\s*//;
        $Line =~ s/\s*$//;

        if ( $Line eq '' ) {
            if ( @Buffer && $OldTag ) {
                $Result{ $TagMap{ lc $OldTag } } = [@Buffer]
                  if $TagMap{ lc $OldTag };
                $Result{Fields}->{$OldTag} = join( "\n", @Buffer );
            }
            @Buffer = ();
            $OldTag = '';
        }

        if ( $Line =~ $TagStartRE ) {
            my ( $Tag, $Value ) = $Line eq '' ? ( undef, '' ) : ( $1, $2 );

            if ($Tag) {
                $OldTag = $Tag;
                @Buffer = ();
            }
            push @Buffer, $Value if $Value;
            $Result{Fields}->{$Tag} = join( "\n", @Buffer ) if $Tag;
            if ( my $TM = $TagMap{ lc $Tag } ) {
                $Result{$TM} = [@Buffer];
            }
        }
        elsif ( $Line ne '' ) {
            push @Buffer, $Line;
        }
    }

    if ( exists $Result{history} ) {
        my @Items = split $VerlaufTrennerRE, $Result{history};
        $Result{Ticketverlauf} = join "\n", @{$Result{history}};
        $Result{history} =
          [ map $_ =~ $VerlaufTeileRE ? [ $1, $3 ] : $_, @Items ];
    }
    return \%Result;
}

sub verlauf_roh {
    my ($zeile) = @_;
    return {
        item => {
            date  => convert_isodate_to_date_hash( $zeile->[0] ),
            value => $zeile->[1],
        }
    };
}
            
sub evaluate {
    my ( $Self, $Data ) = @_;
    my $ExistingTickets = $Self->{ExistingTickets};
    $ExistingTickets->{ $Data->{TicketNumber} } = 1;
    my $Customtext = $Data->{customtext};
    my $Customdate = $Data->{customdate};
    my $Typ        = $Self->{Typ};

    my $ALNOStatus = $ALNOStatusMap{ $Data->{State} };
    my $IncidentStart =
      exists $Customdate->{'Incident-Start'}
      ? $Customdate->{'Incident-Start'}->{date}
      : $Data->{created};
    
    # Incident-Resolved, SolutionTime
    my ( $IncidentResolved, $SolutionTime, $Verlauf );
    if ( exists $Customdate->{'Incident-Resolved'} ) {
        $IncidentResolved = $Customdate->{'Incident-Resolved'}->{date};
    }
    elsif ( exists $ALNOResolvedStatus{ $Data->{State} } ) {
        my $ResolveEvent = find_resolving_event( $Data->{history} );
        if ( defined $ResolveEvent ) {
            $IncidentResolved = $ResolveEvent->{created};
        }
    }
    if ( defined $IncidentResolved ) {
        $SolutionTime = $IncidentResolved->{epoch} - $IncidentStart->{epoch};
    }

    # Daily: Weder Ticket noch relevanter Verlauf nach Berichtsbeginn?
    my $VerlaufInfo = parse_verlauf( $Data->{articles} );

    if ( $Typ eq 'raw' ) {

        # ALNO-Roh-Format
        my $ResultState = {
            type => $Data->{StateType},
            id   => $Data->{StateID},
            name => $Data->{State},
        };
        $ResultState->{until} = { date => $Data->{'pending-until'} }
          if defined $Data->{'pending-until'};
        my $Result = {
            TicketNumber => $Data->{TicketNumber},
            state        => $ResultState,
            article      => [
                map $_->{ArticleType} =~ /-external$/
                ? {
                    id      => $_->{ArticleID},
                    type_id => $_->{ArticleTypeID},
                    type    => $_->{ArticleType},
                    value   => $_->{Body},
                  }
                : (),
                @{ $Data->{articles} }
            ],
            'state-history' => [
                map $_->{HistoryType} eq 'StateUpdate'
                ? {
                    date  => $_->{created},
                    value => $_->{Name},
                  }
                : (),
                @{ $Data->{history} }
            ],
            'incident-period' => [$SolutionTime],
            Fields            => $VerlaufInfo->{Fields},
            actions           => [ $VerlaufInfo->{actions} ],
            reasons           => [ $VerlaufInfo->{reason}  ],
            error             => [ $VerlaufInfo->{error}   ],
            history => [ map { verlauf_roh($_) } @{ $Verlauf->{history} } ],
        };
        $Result->{customtext} = $Data->{customtext} if %{ $Data->{customtext} };
        $Result->{customdate} = $Data->{customdate} if %{ $Data->{customdate} };
        $Result->{workedtime} = $Data->{workedtime} if %{ $Data->{workedtime} };
        return $Result;
    }

    return {
        Ticketnummer => $Data->{TicketNumber},

        Bearbeitungsbegin  => date_hash_to_alno_format($IncidentStart),
        Bearbeitungsende   => date_hash_to_alno_format($IncidentResolved),
        Bearbeitungszeit   => epoch_diff_to_alno_format($SolutionTime),
        IncidentStartEpoch => $IncidentStart->{epoch},    # fuers Sortieren

        'Ticket-ID des Dienstleisters' => $VerlaufInfo->{Fields}->{'Ticket-ID des Dienstleisters'},
        Dienstleister          => $VerlaufInfo->{Fields}->{'Dienstleister'},
        'e-Mail-Adresse'       => $VerlaufInfo->{Fields}->{'e-Mail-Adresse'},
        'Heimat Standort'      => $VerlaufInfo->{Fields}->{'Heimat Standort'},
        Kategorie              => $VerlaufInfo->{Fields}->{'Kategorie'},
        'Melder betroffen'     => $VerlaufInfo->{Fields}->{'Melder betroffen'},
        Melder                 => $VerlaufInfo->{Fields}->{'Melder'},
        'Priorität'            => $VerlaufInfo->{Fields}->{'Priorität'},
        'derzeitiger Standort' => $VerlaufInfo->{Fields}->{'derzeitiger Standort'},
        Telefonnummer          => $VerlaufInfo->{Fields}->{'Telefonnummer'},

        Fehlerbeschreibung       => $VerlaufInfo->{Fields}->{Fehlerbeschreibung},
        Fehlerursache            => $VerlaufInfo->{Fields}->{Fehlerursache},
        'Eingeleitete Maßnahmen' => $VerlaufInfo->{Fields}->{'Eingeleitete Maßnahmen'},

        'Bearbeitungszeit nn' => ( $Data->{workedtime}->{simple_sum} / 60 ) . ' min',

        Tickettyp        => $Data->{'Type'},
        Ticketverlauf    => $VerlaufInfo->{Ticketverlauf},
        Status           => $ALNOStatus,
        Termin           => date_hash_to_alno_format(
            $Customdate->{'Due-Date'}->{date}
        ),

        #error            => $VerlaufInfo->{error},
        #actions          => $VerlaufInfo->{actions},
        #reason           => $VerlaufInfo->{reason},
        #history          => $VerlaufInfo->{history},
    };
}

override 'sort' => sub {
    my ($Self) = @_;
    $Self->{_List} =
      [ sort { $a->{IncidentStartEpoch} <=> $b->{IncidentStartEpoch} }
          @{ $Self->{_List} } ];
};

1;
