# (c) 1999 Matthias Urlichs, smurf@noris.de

package RT::support::utils;

use utf8;
use warnings; no warnings qw(once redefine uninitialized);
use strict;
use Carp;
require Exporter;
use vars qw(@EXPORT_OK @ISA);
@ISA = qw(Exporter);
@EXPORT_OK=qw(parse_time normalize_sn date_diff_l date_diff get_effective_sn
	can_display_queue can_manipulate_request can_manipulate_queue
	can_create_request list_sn template_read template_replace_tokens
	is_owner quote_content is_not_a_requestor can_display_request
	can_admin_queue booleanize depending_stalled can_add_time
	date_diff_ll adr_filter is_rt_system);

use vars qw(%queues %users %req $time);

use Dbase::Help qw(DoFn DoT DoSelect qquote);
use Dbase::Globals qw(find_descr get_person mpersinfo);
use Cf qw($MAILDOM $RT_DOMAIN $WDESCR);
use RT::database;
use RT::MIME qw(mime_parse);
use UTFkram qw(fix_mime_encoding decode_anything);
use Dbase::OTRS qw(TicketID);

# The following is necessary because sometimes recursive inclusion causes
# the setup code not to get executed
sub req_in($;$);
sub transaction_in($$);

BEGIN {
	no warnings 'prototype';
    *queues = \%RT::queues;
    *users = \%RT::users;
    *req = \%RT::req;
    *time = \$RT::time;
	
	*req_in = \&RT::database::req_in;
	*transaction_in = \&RT::database::transaction_in;
}

sub untaint($) {
	my $data = shift;
    if ($data =~ /^([-\@\w.]+)$/) {
    	$data = $1;                     # $data now untainted
    } else {
    	die "Bad data in $data";        # log this somewhere
    }
	return $data;
}

sub is_rt_system($) {
	my($user) = @_;
	$user eq '_rt_system' || $user =~ /^\d+\@\Q$RT_DOMAIN\E\z/;
}

#
#can the named user add work times
sub can_add_time($){
    my ($in_user) = @_;
	return 0 unless defined $in_user;

    return 1 if $users{$in_user}{pop_rt};
    return 0;
}

#
#can the named user modify the named queue
sub can_manipulate_request($$){
    my ($in_serial_num, $in_user) = @_;

	return 0 if $in_serial_num !~ /^\d+$/;
	return 0 if TicketID($in_serial_num);
	return 0 unless defined $in_user;

    req_in($in_serial_num,$in_user);

    return 1 if defined $req{$in_serial_num}{owner}
	        and $in_user eq $req{$in_serial_num}{owner};
    return 0 if $req{$in_serial_num}{queue_id} eq "";
    return 1 if can_manipulate_queue($req{$in_serial_num}{queue_id},$in_user,1);
    return 0;
}

sub can_display_request($$) {
	my ($in_serial_num, $in_user) = @_;
	return 0 if $in_serial_num !~ /^\d+$/;
	return 0 unless defined $in_user;

    req_in($in_serial_num,$in_user);
    return 1 if defined $req{$in_serial_num}{owner}
	        and $in_user eq $req{$in_serial_num}{owner};
    return 1 if can_display_queue($req{$in_serial_num}{queue_id},$in_user);
	return 0;
}

sub can_create_request($$;$){
    my ($in_queue, $in_user, $skip_otrs_test) =@_;
  
	return 0 if not $skip_otrs_test and $queues{$in_queue}{otrs};
	if(defined $in_user) {
    	return 1 if $queues{$in_queue}{acls}{$in_user}{manipulate};
		return 1 if can_admin_queue($in_queue, $in_user);
	}
    return 2 if $queues{$in_queue}{allow_user_create};
	return 0;
}

sub can_manipulate_queue($$;$) {
    my ($in_queue, $in_user, $skip_otrs_test) =@_;

	# Sonderfall: wenn eine Queue schon im OTRS ist, ein Ticket jedoch
	# noch nicht (oder umgekehrt), dann beachte den OTRS-Status der
	# Queue _nicht_

	return 0 if not $skip_otrs_test and $queues{$in_queue}{otrs};
	return 0 unless defined $in_user;
	return 3 if is_rt_system($in_user);
	return 1 if $queues{$in_queue}{acls}{$in_user}{manipulate};
	return -2 if $users{$in_user}{admin_rt};
	return 0;
}

sub can_display_queue($$) {
    my ($in_queue, $in_user) =@_;
	return 0 unless defined $in_user;
	return 3 if is_rt_system($in_user);
	return 1 if $queues{$in_queue}{acls}{$in_user}{display};
	return -2 if $users{$in_user}{admin_rt};
	return 0;
}

sub can_admin_queue($$) {
    my ($in_queue, $in_user) =@_;
	return 0 if $queues{$in_queue}{otrs};
	return 0 unless defined $in_user;
	return 3 if is_rt_system($in_user);
	return 1 if $queues{$in_queue}{acls}{$in_user}{admin};
	return -2 if $users{$in_user}{admin_rt};
	return 0;
}

sub is_not_a_requestor($$){
    my($address,$serial_num) =@_;
	return 0 if defined $req{$serial_num}{requestors} && defined $address && $req{$serial_num}{requestors} =~ /(^|\s|,)$address(,|\s|\b)/;
	return 1;
}

sub is_owner($$){
    my($serial_num,$user) =@_;
	return 0 unless defined $user;
	return 0 unless defined $req{$serial_num}{owner};
	return 1 if $req{$serial_num}{owner} eq $user;
	return 0;
}


#normalize_sn takes care of opersations on reqs which have been merged
##actually, it's just an alias into get_effective_sn
sub normalize_sn($){
    my ($in_serial_num)=@_;
	return undef unless defined $in_serial_num;
	get_effective_sn($in_serial_num);
}

sub get_effective_sn($) {
    my ($in_serial_num) =@_;
    return unless defined $in_serial_num;
    $in_serial_num =~ s/\D+//g;
    return unless $in_serial_num;
    return $RT::effcache{$in_serial_num} if defined $RT::effcache{$in_serial_num};

    my $effective_sn = DoFn("select ticket from ticket where id = $in_serial_num");;
	## ticket.ticket is supposed to be equal to id,
	## which is why no errors are checked for
    $RT::effcache{$in_serial_num} = $effective_sn;
}

sub list_sn($){
    my ($serial)=@_;
	my $res = DoT("select id from ticket where ticket = $serial");
	return undef unless ref $res;
    map { $_->[0] } @$res;
}

#
# return something's boolean value
#

sub booleanize($) {
	my ($in_val)=@_;
	return 1 if $in_val;
	return 0;
}

BEGIN {
	package TWrapper;
	sub new { my $pack = shift; bless { @_ }, $pack;}
	sub wrap {
		 my($self,$txt) = @_;
		 $txt =~ s/^/$self->{par_start}/mg;
		 # use Data::Dumper;
		 # "CONV:|".Dumper($self)."|".$txt."|";
		 $txt;
	 }
}

#
# quote_content
# will generate the content of a transaction...prefixed.
#

sub conv_mime($$$);
sub conv_mime($$$) {
	my($header_mode,$msg,$level) = @_;
	return "(unknown)\n" unless $msg;
	return $msg unless ref $msg;
	$level++;

	$msg->head->unfold;
	my $headers = $msg->head->original_text;

	my $type = $msg->mime_type;
	if($type =~ m#^message/rfc822#i) {
		$type = 3;
	} elsif($type =~ m#^application/(?:x-)?pgp-signature#i) {
		return ""; # ignore
	} elsif($type =~ m#^multipart/signed#i) {
		$type = 5;
	} elsif($type =~ m#^text/html#i) {
		$type = 2;
	} elsif($type =~ m#^message/#i or $type =~ m#^text($|/)#i) {
		$type = 1;
	} else {
		$type = 0;
	}
	my $wr = new TWrapper(body_start => '>'x$level.' ', par_start => ('>'x$level).' ', columns => 76);

	$headers =~ s/\n\s+/ /g;
	my $res = "";

	if($type != 3 and $type != 5) {
		if($header_mode ne 'none') {
			foreach my $line (split (/\n/, $headers)) {
				next if $line =~ /^content-type:/i;
				my ($field, $value)= split (/:\s+/,$line, 2);
				$res .= '>'x$level."_$field: $value\n" if $header_mode eq 'all' or $field =~ /$header_mode/i;
			}
			chop $res;
		}
		$res .= "\n\n";
	}

	# $body =~ s/(.{76})(.{3})/$1\n $2/g;

	if($type == 2) {
		use Data::Dumper;
		use HTMLfilter qw($filter);
		$res .= $filter->scrub($msg->bodyhandle->as_string);
	} else {
		if($msg->bodyhandle) {
			if($type == 0) {
				$res .= '>'x$level." (".length($msg->bodyhandle->as_string)." bytes)\n";
			} elsif($type == 3) {
				$res .= conv_mime($header_mode,$msg->{SubMessage},$level) if $msg->{SubMessage};
				return $res;
			} else {
				$res .= $wr->wrap($msg->bodyhandle->as_string);
			}
		}

		$res .= $wr->wrap(join("\n",@{$msg->preamble}))."\n" if $msg->preamble;
		my $num=0;

		if($type == 3 or $type == 5) {
			foreach my $part ($msg->parts) {
				$num++;
				$res .= conv_mime($header_mode,$part,$level-1);
			}
		} else {
			foreach my $part ($msg->parts) {
				$num++;
				$res .= ">"x$level." (BEGIN Part $num)\n";
				$res .= conv_mime($header_mode,$part,$level)."\n";
			}
			$res .= ">"x$level." (END Part $num)\n" if $num;
		}
		$res .= $wr->wrap(join("\n",@{$msg->epilogue}))."\n" if $msg->epilogue;
	}
	$res;
}

# Adressfilter, wirft RT-Mailadressen raus
use Mail::Address;
sub adr_filter($) {
	my($adr) = @_;
	my @res;
	foreach my $a(Mail::Address->parse($adr)) {
		next if (my $address = $a->address) =~ /^\d+\@\Q$RT_DOMAIN\E$/si;
		push @res, $address;
	}
	if (wantarray) { @res }
	elsif (@res) { join ', ', @res }
	else { undef }
}

sub quote_content($$$) {
    my($transaction, $current_user, $form) = @_;
    my($quoted_content, $body, $headers);

	my $serial_num = $transaction->[0];
    my $trans=transaction_in($transaction,$current_user);

	my ( $msg, $state ) =
	  eval { mime_parse( \$trans->{content}, decode_headers => 1 ) };

	my $from = $msg->head->get('from');
	my($to,$gto,$cc,$subject);
	if($@ or not ref $msg or ($state ne "" and $state ne "EOF")) {
		$msg = "...bad MIME state...";
	} else {
		if(wantarray) {
			$to = adr_filter($msg->head->get("reply-to"));
			$to = adr_filter($from) unless defined $to;
			$gto = adr_filter($msg->head->get("to"));
			$cc = adr_filter($msg->head->get("cc"));
			if ( defined( $subject = $msg->head->get('subject') ) ) {
				$subject =~ y/\cM\cJ//d;
				$subject =~ s/^Re:\s*//;
			}
		}
		$msg = conv_mime(qr/^(Content-Type|From|Subject|Date|To|Cc|Bcc|X-.*)$/i,$msg,0);
	}
	(($headers, $body) = split ('\n\n',$msg,2)) or
    $body = $msg;

	my $id;
	if ( $trans->{actor} eq '<unbekannt>' && $from ) {
		$id = $from;
	} elsif ( ( $id = get_person($trans->{actor},'?email',1) ) > 0 ) {
		$id = mpersinfo($id);
	} else {
		$id = $trans->{actor} || $to;
	}
    chomp($id);
	$body = "$id wrote ($trans->{text_time}):\n".decode_anything($body);
	return $body if $form eq "";
	$form = template_read($form,$req{$serial_num}{queue_id});
    my $res = template_replace_tokens($form,$serial_num,$transaction, $body, $current_user);
	# print STDERR "Data: to $to gto $gto cc $cc\n";
	wantarray ? ($res,$to,$gto,$cc,$subject): $res;
}

my @DoW = ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');
my @MoY = ('Jan','Feb','Mar','Apr','May','Jun',
    'Jul','Aug','Sep','Oct','Nov','Dec');

sub parse_time($) {
	use Time::localtime;
	my ($tm) = @_;
	return "never" unless $tm;
	$tm = localtime $tm;

    ($DoW[$tm->wday], $MoY[$tm->mon], $tm->mday, $tm->hour, $tm->min, $tm->sec, "ZonE", $tm->year+1900);
}
	
#
# Totally redone by Matthias Urlichs to report all steps
#	
sub date_diff_ll($$) {
	use integer;
    my($old, $new) = @_;
    my($diff, $minute, $hour, $day, $week, $month, $year, $s, $string, $negative);
	my @res;

    $diff = $new - $old;
    if ($diff < 0) {
		$negative = "-";
		$diff = -$diff;
    } else {
		$negative = "";
	}

    $minute = 60;
    $hour   = 60 * $minute;
    $day    = 24 * $hour;
    $week   = 7 * $day;
    $month  = 30 * $day;
    $year   = 365 * $day;
    
	$s = $diff/$hour;
	if ($s) { push(@res, "$s hr"); } else { push(@res,"") if @res; }
	$diff %= $hour;

	$s = $diff/$minute;
	if ($s) { push(@res, "$s min") } else { push(@res,"") if @res; }
	$diff %= $minute;

	if ($diff or not @res) { push(@res, "$diff sec"); }

    return $negative.join(" ",@res)
}

sub date_diff_l($$) {
	use integer;
    my($old, $new) = @_;
    my($diff, $minute, $hour, $day, $week, $month, $year, $s, $string, $negative);
	my @res;

    $diff = $new - $old;
    if ($diff < 0) {
		$negative = "-";
		$diff = -$diff;
    } else {
		$negative = "";
	}

    $minute = 60;
    $hour   = 60 * $minute;
    $day    = 24 * $hour;
    $week   = 7 * $day;
    $month  = 30 * $day;
    $year   = 365 * $day;
    
	$s = $diff/$year;
	if ($s) { push(@res, "$s yr"); } else { push(@res,"") if @res; }
	$diff %= $year;

	$s = $diff/$month;
	if ($s) { push(@res, "$s mth"); } else { push(@res,"") if @res; }
	$diff %= $month;

	$s = $diff/$week;
	if ($s) { push(@res, "$s wk"); } else { push(@res,"") if @res; }
	$diff %= $week;

	$s = $diff/$day;
	if ($s) { push(@res, "$s day"); } else { push(@res,"") if @res; }
	$diff %= $day;

	$s = $diff/$hour;
	if ($s) { push(@res, "$s hr"); } else { push(@res,"") if @res; }
	$diff %= $hour;

	$s = $diff/$minute;
	if ($s) { push(@res, "$s min") } else { push(@res,"") if @res; }
	$diff %= $minute;

	if ($diff or not @res) { push(@res, "$diff sec"); }

    return $negative.($res[1] ? $res[0]." ".$res[1] : $res[0]);
}

sub date_diff($$) {
	use integer;
    my($old, $new) = @_;
    my($diff, $minute, $hour, $day, $week, $month, $year, $s, $string, $negative);
	my $res;

    $diff = $new - $old;
    if ($diff < 0) {
		$negative = "-";
		$diff = -$diff;
    } else {
		$negative = "";
	}

    $minute = 60;
    $hour   = 60 * $minute;
    $day    = 24 * $hour;
    $week   = 7 * $day;
    $month  = 30 * $day;
    $year   = 365 * $day;
    
	$s = $diff/$year;
	if ($s) { $res = "$s yr"; } else {

	$s = $diff/$month;
	if ($s) { $res = "$s mth"; } else {
	$diff %= $month;

	$s = $diff/$week;
	if ($s) { $res = "$s wk"; } else {
	$diff %= $week;

	$s = $diff/$day;
	if ($s) { $res = "$s day"; } else {
	$diff %= $day;

	$s = $diff/$hour;
	if ($s) { $res = "$s hr"; } else {
	$diff %= $hour;

	$s = $diff/$minute;
	if ($s) { $res = "$s min" } else {
	$diff %= $minute;

	if ($diff or not $res) { $res = "$diff sec"; }
	}}}}}}

    return $negative.$res;
}


sub depending_stalled($) {
	my($ticket) = @_;
	$ticket = get_effective_sn($ticket);
	my @res;
	my $typ = find_descr("tickett","queue_ptr");
	my $sta = find_descr("tickets","stalled");
	DoSelect {
		my($id) = @_;
		push(@res,$id) if $id != $ticket;
	} "select distinct tc.id from ticket as ta,ticketid,ticket as tb, ticket as tc where ta.id = ticketid.inhalt and ticketid.typ = $typ and tb.ticket = $ticket and ticketid.ticket = tb.id and tc.status = $sta and tc.id = ta.ticket";
	@res;
}


sub template_read($$) {
    my ($in_template, $in_queue) =@_;
    my $template_content="";
    
	my $text;

	$text = DoFn("select inhalt from rt_template where name=${\qquote $in_template}");
	return $text if defined $text;

   	return <<_;
The specified template '$in_template' for queue '$in_queue'
is missing or inaccessable.
However, the custom content which was supposed to fill the template was:
%content%
_
}


sub date_parse($) {
    my ($date_string) = shift;
	return str2time($_[0]);
}


my $seq = 0;

sub template_replace_tokens($$$$$) {
    my ($template,$serial,$in_id, $in_custom_content, $in_current_user) = @_;

	my $unique  = sprintf("%x-%x-%x\@%s",++$seq,$time,$$,$MAILDOM);
	my $unique1 = sprintf("%x-%x-%x\@%s",++$seq,$time,$$,$MAILDOM);
	my $unique2 = sprintf("%x-%x-%x\@%s",++$seq,$time,$$,$MAILDOM);
	my $unique3 = sprintf("%x-%x-%x\@%s",++$seq,$time,$$,$MAILDOM);

	# print STDERR "Reading request $serial\n";
	req_in($serial,'_rt_system') if $serial;

	my $trans = transaction_in($in_id,'_rt_system');
	my $trc;
	if($template =~/%trans:content%/) {

		my ( $msg, $state ) = mime_parse( \$trans->{content} );
		
		if (not ref $msg or (defined $state and $state ne "" and $state ne "EOF")) {
			$trc = $trans->{content};
		} else {
			fix_mime_encoding($msg);
			$trc = $msg->as_string;
			$msg->purge;
		}
	}


    $template =~ s/%rtname%/$WDESCR/g;
    $template =~ s/%unique%/$unique/g;
    $template =~ s/%unique1%/$unique1/g;
    $template =~ s/%unique2%/$unique2/g;
    $template =~ s/%unique3%/$unique3/g;
    $template =~ s/%rtversion%/$RT::rtversion/g;
    $template =~ s/%actor%/ $in_current_user . ( exists $users{$in_current_user} && defined $users{$in_current_user}{short_name} && " ($users{$in_current_user}{short_name})" ) /eg if defined $in_current_user;
    $template =~ s/%owner%/$users{$req{$serial}{owner}}{real_name} ($users{$req{$serial}{owner}}{email})/g if $serial;
    # $template =~ s/%subject%/$in_subject/g;
    # $template =~ s/%info%/$in_info/g;
    $template =~ s/%serial_num%/$serial/g if $serial;
    # $template =~ s/%mailalias%/$mail_alias/g;
    $template =~ s/%mailalias%/$queues{$req{$serial}{queue_id}}{mail_alias}/g;
    $template =~ s/%content%/$in_custom_content\n/g;
    $template =~ s/%req:(\w+)%/$serial && $req{$serial}{$1} || "<unbekannt>"/eg;
    $template =~ s/%trans:content%/$trc/g;
    $template =~ s/%trans:(\w+)%/$trans->{$1} || "<unbekannt>"/eg;
    $template =~ s/%queue:(\w+)%/$queues{$req{$serial}{queue_id}}{$1}/g if $serial;

    if ( defined $in_current_user ) {
        $template =~ s/%user:email%/ exists $users{$in_current_user} && $users{$in_current_user}{email} || "$in_current_user\@$MAILDOM"/eg;
        $template =~ s/%user:(${\ join '|', grep defined $users{$in_current_user}{$_}, keys %{$users{$in_current_user}} })%/$users{$in_current_user}{$1}/g
	  if exists $users{$in_current_user};
    }

	req_in($serial,$in_current_user) if $serial;

    return ($template);
}


1;
