package RT::AddOn::Incidents;

use utf8;
use warnings; no warnings qw(once redefine uninitialized);
use strict;
use base 'RT::AddOn';

use Date::Parse qw(str2time);
use Date::Format qw(time2str);
use Dbase::Globals qw(bignum find_descr info_descr map_descr);
use Dbase::Help qw(Do DoFn DoTime qquote text_only);
use RT::support::utils qw(normalize_sn);

use constant DATE_FORMAT => '%Y-%m-%d %T';

our %Flag;
our %Flag_Info;
unless ( $ENV{TESTING} ) {
	%Flag = reverse map_descr 'rt_incidents_flags';
	$Flag_Info{$_} = info_descr rt_incidents_flags => $_ for keys %Flag;
}

our $flag_rt_incidents = find_descr kunde => rt_incidents => 1;

sub init_ticket($$;$$$) {
	my ( $package, $serial, $date_created, $kunde, $subject ) = @_;
	$serial = normalize_sn($serial);
	unless ( defined $kunde ) {
		require RT::database and RT::database->import('req_in') unless defined &req_in;
		my $req = req_in($serial);
		$date_created = $req->{date_created}
		  unless defined $date_created;
		$kunde   = $req->{kunde_id} unless defined $kunde;
		$subject = $req->{subject}  unless defined $subject;
	}
	return
	  unless $kunde =~ /^\d+\z/
	  and DoFn
	  "SELECT flags & ( 1 << $flag_rt_incidents ) FROM kunde WHERE id = $kunde";
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	$date_created ||= DoTime;
	for ( undef, undef ) {
		my @values = DoFn <<_;
	SELECT art, ursache, t_meldung, t_reaktion, t_entstoerung, flags
	FROM   rt_incidents
	WHERE  ticket = $serial
_
		return wantarray
		  ? @values
		  : {
			art           => shift @values,
			ursache       => shift @values,
			t_meldung     => shift @values,
			t_reaktion    => shift @values,
			t_entstoerung => shift @values,
			flags         => shift @values
		  }
		  if @values;
		my $flags = bignum 0;
		Do <<_;
	INSERT INTO rt_incidents (ticket, t_meldung, flags)
	                  VALUES ($serial, $date_created, $flags)
_
	}
	die;
}

sub _format_date($) {
	my ($time) = @_;
	if ($time) { time2str DATE_FORMAT, $time }
	else { '' }
}

sub reply($$) {
	my ( $package, $serial ) = @_;
	my $value = $package->init_ticket($serial) or return;
	return if $value->{t_reaktion};
	Do <<_;
	UPDATE rt_incidents
	SET    t_reaktion = UNIX_TIMESTAMP(NOW())
	WHERE  ticket = $serial
_
}

sub resolvable($$) {
	my ( $package, $req ) = @_;
	my $value = $package->init_ticket( $req->{serial_num} ) or return;
	my @missing;
	$value->{ $_->[0] }
	  or push @missing, $_->[1]
	  for [ art => 'die Störungsart' ], [ ursache => 'die Störungsursache' ],
	  [ t_meldung     => 'der Zeitpunkt der ersten Störungsmeldung' ],
	  [ t_reaktion    => 'der Zeitpunkt der ersten Rückmeldung an den Kunden' ],
	  [ t_entstoerung => 'der Zeitpunkt der Entstörung' ];
	return unless @missing;
	'Das Ticket darf erst resolved werden, wenn '
	  . join( ' und ', @missing )
	  . ' eingegeben '
	  . ( @missing == 1 ? 'ist' : 'sind' ) . '.';
}

sub show_form($$) {
	my ( $package, $req ) = @_;
	( my $serial = $req->{serial_num} ) =~ /^\d+\z/ or die;
	my $value =
	  $package->init_ticket( $serial, $req->{date_created}, $req->{kunde_id},
		$req->{subject} )
	  or return;
	require RT::CGI4AddOn;
	my $cgi = RT::CGI4AddOn->new;
	require RT::database::manipulate;    # Probleme bei use, vgl. RT#160135-4
	print $cgi->table(
		{
			bgcolor => RT::database::manipulate::is_visible( $serial, $req )
			? '#ffcccc'
			: '#ccccff',
			cellpadding => 16
		},
		$cgi->Tr(
			$cgi->td(
				$cgi->start_form,
				$cgi->table(
					$cgi->Tr(
						[
							map ( $cgi->td(
									[
										"Zeitpunkt der $_->[1]",
										$cgi->textfield(
											-name  => "_incidents_t_$_->[0]",
											-value => _format_date(
												$value->{"t_$_->[0]"}
											),
											-size => 42
										)
									]
								),
								[ meldung  => 'Störungsmeldung' ],
								[ reaktion => '1. Rückmeldung an den Kunden' ],
								[ entstoerung => 'Entstörung' ] ),
							$cgi->td(
								[
									'Störungsart:',
									$cgi->textfield(
										-name  => '_incidents_art',
										-size  => 42,
										-value => $value->{art}
									)
								]
							),
							$cgi->td(
								{ colspan => 2 },
								'Störungsursache:',
								$cgi->br,
								$cgi->textarea(
									-name  => '_incidents_ursache',
									-cols  => 80,
									-rows  => 8,
									-value => $value->{ursache}
								)
							),
							$cgi->td(
								{ colspan => 2 },
								$cgi->checkbox_group(
									-name     => '_incidents_flags',
									-values   => [ keys %Flag ],
									-defaults => [
										grep $value->{flags} & bignum(1) << $_,
										keys %Flag
									],
									-labels => \%Flag_Info
								)
							),
							$cgi->td(
								{ align => 'center', colspan => 2 },
								$cgi->reset . ' ' . $cgi->save_data($serial)
							)
						]
					)
				)
			)
		)
	  ),
	  $cgi->end_form;
}

sub save_data {
	my ( $package, $cgi ) = @_;
	my ( @errors, @messages, @update );
	( my $serial = $cgi->param('serial_num') ) =~ /^\d+\z/ or die;
	$serial = normalize_sn($serial);
	my $value = $package->init_ticket($serial) or return;
	for (
		[ meldung     => 'Störungsmeldung' ],
		[ reaktion    => 'ersten Rückmeldung an den Kunden' ],
		[ entstoerung => 'Entstörung' ]
	  )
	{
		my $zeitpunkt = "Zeitpunkt der $_->[1]";
		unless ( defined( my $p = $cgi->param("_incidents_t_$_->[0]") ) ) {
			push @errors, qq(Parameter "_incidents_t_$_->[0]" nicht gefunden);
		}
		elsif ( $p !~ /\S/ ) {
			if ( $value->{"t_$_->[0]"} ) {
				push @update,   "t_$_->[0]=NULL";
				push @messages, "$zeitpunkt gelöscht";
			}
		}
		elsif ( !( my $t = str2time $p) ) {
			push @errors, qq(eingegebener $zeitpunkt nicht parsbar);
		}
		elsif ( $t != $value->{"t_$_->[0]"} ) {
			push @update,   "t_$_->[0]=$t";
			push @messages, "$zeitpunkt auf " . _format_date($t) . ' gesetzt';
		}
	}
	for (qw(art ursache)) {
		next unless defined( my $p = $cgi->param("_incidents_$_") );
		next if $p eq $value->{$_};
		push @update,   "$_=" . qquote($p);
		push @messages, qq(Störungs$_ auf "$p" gesetzt);
	}
	{
		my %flag;
		$flag{ $cgi->param('_incidents_flags') } = ();
		my $flags = $value->{flags};
		while ( my ( $bit, $flag ) = each %Flag ) {
			my $old = $value->{flags} & bignum(1) << $bit;
			my $new = exists $flag{$bit};
			if ( $old && !$new ) {
				$flags &= ~( bignum(1) << $bit );
				push @messages, qq(Flag "$flag" gelöscht);
			}
			elsif ( !$old && $new ) {
				$flags |= bignum(1) << $bit;
				push @messages, qq(Flag "$flag" gesetzt);
			}
		}
		push @update, "flags=$flags" if $value->{flags} != $flags;
	}
	Do 'UPDATE rt_incidents SET '
	  . join( ', ', @update )
	  . " WHERE ticket=$serial"
	  if @update;
	$package->add_transaction( $serial, undef, text_only($_) ) for @messages;
	join '; ', @errors, @messages or '(keine Änderungen)';
}

1;
