# -*- coding: utf-8 -*-

"""\
This module is a generic database interface.

Some databases don't have nice cursors, so if we nest commands we need
to open a second connection.

Databases have multiple variants for putting parameters into statements,
so use a common interface and adapt it.

This module does not try to be asynchronous, fork-safe, or whatever.

"""

from __future__ import generators
from noris import Cf
from time import time,sleep
import string
import MySQLdb
import re
from array import array
from sys import exc_info

def fixup_error(cmd):
	"""Append the full command to the error message"""
	e1,e2,e3 = exc_info()
	k=list(e2.args)
	k.append(cmd)
	e2.args=tuple(k)

class db_data(object):
	def __init__(self, p="",**kwargs):
		"""host,port,dbase_name,user,password"""

		try: self.host = kwargs["host"]
		except KeyError:
			try: self.host = getattr(Cf,"DATAHOST"+p)
			except KeyError: self.host = Cf.DATAHOST
			try: self.host=self.host[:self.host.index(" ")]
			except ValueError: pass

		try: self.port = int(kwargs["port"])
		except KeyError:
			try: self.port = int(getattr(Cf,"DATAPORT"+p))
			except KeyError:
				try: self.port = int(Cf.DATAPORT)
				except KeyError: self.port = None

		try: self.dbname = kwargs["database"]
		except KeyError:
			try: self.dbname = getattr(Cf,"DBDATABASE"+p)
			except KeyError: self.dbname = Cf.DBDATABASE

		try: self.username = kwargs["username"]
		except KeyError:
			try: self.username = getattr(Cf,"DATAUSER"+p)
			except KeyError: self.username = Cf.DATAUSER

		try: self.password = kwargs["password"]
		except KeyError:
			try: self.password = getattr(Cf,"DATAPASS"+p)
			except KeyError: self.password = Cf.DATAPASS


class _db_mysql(db_data):
	SINGLE=0
	def __init__(self,p,**kwargs):
		self.DB = __import__("MySQLdb")
		db_data.__init__(self,p,**kwargs)
##		if self.port:
##			self.host = self.host+":"+str(self.port)
##			self.port=None

	def conn(self,p=None,**kwargs):
		args={}
		if self.port: args["port"]=self.port
		return self.DB.connect(db=self.dbname, host=self.host, user=self.username, passwd=self.password, **args)

class _db_odbc(db_data):
	SINGLE=1
	def __init__(self,p,**kwargs):
		self.DB = __import__("mx.ODBC.iODBC")
		db_data.__init__(self,p,**kwargs)

	def conn(self,p=None,**kwargs):
		if port: host = host+":"+str(port)

		return self.DB.connect(database=self.dbname, host=self.host, user=self.username, password=self.password)

class _db_postgres(db_data):
	SINGLE=1
	def __init__(self,p,**kwargs):
		self.DB = __import__("pgdb")
		db_data.__init__(self,p,**kwargs)

	def conn(self,p=None,**kwargs):
		if self.port:
			self.host = self.host+":"+str(self.port)
			self.port=None

		return self.DB.connect(database=self.dbname,host=self.host, user=self.username, password=self.password)



## possible parsers

## qmark: '?'
def _init_qmark():
	return []
def _do_qmark(name, arg, params):
	p = params[name]
	if isinstance(p,array):
		p = p.tostring()
	arg.append(p)
	return "?"
def _done_qmark(cmd,args):
	return (cmd,tuple(args))

## numeric: ':n'
def _init_numeric():
	return []
def _do_numeric(name, arg, params):
	p = params[name]
	if isinstance(p,array):
		p = p.tostring()
	arg.append(p)
	return ":" + str(len(arg))
def _done_numeric(cmd,args):
	return (cmd,args)

## named: ':name'
def _init_named():
	return {}
def _do_named(name, arg, params):
	p = params[name]
	if isinstance(p,array):
		p = p.tostring()
	arg[name] = p
	return ":" + name
def _done_named(cmd,args):
	return (cmd,args)

## format: '%s'
def _init_format():
	return []
def _do_format(name, arg, params):
	p = params[name]
	if isinstance(p,array):
		p = p.tostring()
	arg.append(p)
	return "%s"
def _done_format(cmd,args):
	return (cmd,args)

## pyformat: '%(name)s'
def _init_pyformat():
	return {}
def _do_pyformat(name, arg, params):
	p = params[name]
	if isinstance(p,array):
		p = p.tostring()
	arg[name] = p
	return "%(" + name + ")s"
def _done_pyformat(cmd,args):
	return (cmd,args)

_parsers = {
		"qmark"   : (_init_qmark,_do_qmark,_done_qmark),
		"numeric" : (_init_numeric,_do_numeric,_done_numeric),
		"named"   : (_init_named,_do_named,_done_named),
		"format"  : (_init_format,_do_format,_done_format),
		"pyformat": (_init_pyformat,_do_pyformat,_done_pyformat),
	}


## these might be based on StopIteration instead.  Too dangerous ???

class NoData(Exception):
	pass
class ManyData(Exception):
	pass
#class NoDatabase(Exception):
#	pass

class Db:
	"""\
		Implement a nice database-independent and Python-ish database interface.
		Database statements are implemented as strings. Values that are
		to be interpolated into statements are always written as
		${name}; the value is a named parameter "name". Precise rules
		for quoting et al. are left to the database; so is quoting.
		Null variables are given as "None".

		Examples:
			db=Db()
			db.Do("insert into foo(bar) values(${baz})", baz="Hi there!\n")
			# assuming that foo is a table with an auto-increment column...
			id = db.Do("insert into foo(bar) values(${baz})", baz="Hi there!\n")
			# DoFn always returns a list, so the syntax is a bit strange if there's just one value
			val, = db.DoFn("select bar from foo where id=${id}", id=id)
			assert val=="Hi there!\n"
			# ditto for DoSelect
			for val, in DoSelect("select distinct bar from foo"):
				print val
		"""

	def __str__(self):
		return "<"+self.DB.host+":"+self.DB.dbname+">"
	def __repr__(self):
		return "Db"+self.__str__()

	def __init__(self, prefix=None, **kwargs):
		""" Initialize connection collection """

		if prefix is None:
			prefix=""
		else:
			prefix="_"+prefix.upper()
		self.prefix=prefix
		try: dbtype = kwargs["dbtype"]
		except KeyError:
			try: dbtype=getattr(Cf,"DATABASE"+prefix)
			except KeyError: dbtype=Cf.DATABASE
		self.DB = globals()["_db_"+dbtype](prefix, **kwargs)
		self.DB.dbtype=dbtype

		if self.DB.SINGLE:
			self._conn=None
		else:
			self.conns=[]
		self._trace=None
		self.in_transaction = False

		(self.arg_init, self.arg_do, self.arg_done) \
			 = _parsers[self.DB.DB.paramstyle]
		
	def conn(self):
		"""Allocate a database connection (internal method)"""
		
		if self.DB.SINGLE or self.in_transaction:
			if self._conn is not None:
				return self._conn

		else:
			while self.conns:
				r,rtm=self.conns.pop()
				if rtm > time()-300:
					return r

		Ex=None
#		for i in range(3):
#			try:
		r = self.DB.conn(self.prefix)
		try: r.setconnectoption(self.DB.DB.SQL.AUTOCOMMIT, self.DB.DB.SQL.AUTOCOMMIT_OFF)
		except AttributeError:
			try: r.cursor().execute("SET AUTOCOMMIT=0")
			except: pass
		
		try: r.cursor().execute("SET WAIT_TIMEOUT=7200") # 2h
		except: pass
		
#		try:
#			r.stringformat = self.DB.DB.UNICODE_STRINGFORMAT
#			r.encoding = 'utf-8'
#		except AttributeError:
#			pass

		if self.DB.SINGLE or self.in_transaction:
			r.commit()
			self._conn = r
		return r

	def deconn(self,conn):
		"""'Free' a database connection (internal method)"""
		if conn is None:
			raise ValueError
		if self.DB.SINGLE or self.in_transaction:
			if self._conn is None:
				self._conn = conn
			elif self._conn != conn:
				raise ValueError,("Conn diff",self._conn,conn)
		else:
			self.conns.append((conn,time()))

	def commit(self):
		"""Commit the current transaction"""
		if self._trace:
			self._trace("Commit","","")
		if self.DB.SINGLE or self.in_transaction:
			self.conn().commit()
		else:
			# Assume that only one is actually doing something...
			for c in self.conns:
				c[0].commit()


	def rollback(self):
		"""Rollback the current transaction"""
		if self._trace:
			self._trace("RollBack","","")
		if self.DB.SINGLE:
			self.conn().rollback()

	def prep(self,_cmd,**kwargs):
		"""Prepare a statement. (internal method)"""
		args = self.arg_init()
		def _prep(name):
			return self.arg_do(name.group(1), args, kwargs)
		
		_cmd = re.sub(r"\$\{([a-zA-Z][a-zA-Z_0-9]*)\}",_prep,_cmd)
		return self.arg_done(_cmd,args)

	def DoFn(self, _cmd, **params):
		"""Execute a database query which is expected to return exactly one row."""
		conn=self.conn()

		try:
			curs=conn.cursor()

			_cmd = self.prep(_cmd, **params)
			try:
				apply(curs.execute, _cmd)
			except:
				fixup_error(_cmd)
				raise
			val=curs.fetchone()

			if self._trace:
				self._trace("DoFn",_cmd,val)
		except:
			if not self.DB.SINGLE:
				conn.close()
			raise

		if not val:
			curs.close()
			self.deconn(conn)
			raise NoData,_cmd
		if curs.fetchone():
			if not self.DB.SINGLE:
				conn.close()
			raise ManyData,_cmd

		curs.close()
		self.deconn(conn)
		if params.get("_dict",False):
			return dict(zip([x[0] for x in curs.description],val))
		else:
			return val

	def Do(self, _cmd, **params):
		"""Execute a database statement."""
		conn=self.conn()
		curs=conn.cursor()

		_cmd = self.prep(_cmd, **params)
		try:
			apply(curs.execute, _cmd)
		except:
			fixup_error(_cmd)
			raise
		try:
			r = conn.insert_id()
		except AttributeError:
			r = None
		if not r:
			r = curs.lastrowid
		if not r:
			r = curs.rowcount
		curs.close()
		self.deconn(conn)

		if self._trace:
			self._trace("DoFn",_cmd,r)
		if r == 0 and not params.has_key("_empty"):
			raise NoData,_cmd
		return r

	def DoN(self,*a,**k):
		try:
			self.Do(*a,**k)
		except MySQLdb.OperationalError:
			pass

	def DoSelect(self, _cmd, *params, **keys):
		"""\
			Execute a database query which may return multiple rows.
			Returns an iterator for it.
		
		'_store' is 0: force save on server
			'_store' is 1: force save on client, may do nested calls

		'_head' is 1: first return is headers as text
		'_head' is 2: first return is DB header tuples

			'_dict' is 1: return entries as dictionary
			'_empty' is 1: don't throw an error when no data are returned
		"""

		conn=self.conn()

		try:
			store=keys.get("_store",False)
			if store:
				curs=conn.cursor()
			elif self.DB.dbtype == "mysql":
				curs=conn.cursor(self.DB.DB.cursors.SSCursor)
			else:
				curs=conn.cursor()

			_cmd = self.prep(_cmd, **keys)
			try:
				apply(curs.execute, _cmd)
			except:
				fixup_error(_cmd)
				raise
		
			head=keys.get("_head",False)
			if head:
				if head>1:
					yield curs.description
				else:
					yield [x[0] for x in curs.description]

			as_dict=keys.get("_dict",False)
			if as_dict:
				as_dict = map(first, curs.description)

			val=curs.fetchone()
			if not val:
				if self._trace:
					self._trace("DoSelect",_cmd,None)

				if not keys.has_key("_empty"):
					raise NoData,_cmd

			n=0
			while val != None:
				if as_dict:
					yield dict(zip(as_dict,val))
				else:
					yield val[:]

				val=curs.fetchone()

		except:
			if not self.DB.SINGLE:
				conn.close()
			raise

		if self._trace:
			self._trace("DoSelect",_cmd,n)
		curs.close()
		self.deconn(conn)

	def transact(self,proc,*a,**k):
		"""\
			Run a database transaction.
			May be called recursively.
			"""
		if self.in_transaction:
			return proc(*a,**k)

		# Yes, this order is correct; we want to grab one of the
		# existing connections if SINGLE is false.
		conn = self.conn()
		self.in_transaction = True
		if not self.DB.SINGLE:
			self._conn = conn

		# Python <2.5 does not support try: ... except: ... finally:
		try:
			try:
				res = proc(*a,**k)
			except:
				a,b,c = exc_info()
				try:
					self.rollback()
				except:
					pass
	
				raise a,b,c
			else:
				self.commit()
		finally:	
			self.in_transaction = False
			self.deconn(self._conn)
			if not self.DB.SINGLE:
				self._conn = None
		return res

	def transact_iter(self,proc,*a,**k):
		"""\
			Run an iterator with a database transaction.
			May be called recursively.
			"""
		if self.in_transaction:
			for f in proc(*a,**k):
				yield r
			return

		# Yes, this order is correct; we want to grab one of the
		# existing connections if SINGLE is false.
		conn = self.conn()
		self.in_transaction = True
		if not self.DB.SINGLE:
			self._conn = conn

		# Python <2.5 does not allow try: ... finally: in an iterator.
		try:
			for r in proc(*a,**k):
				yield r
		except:
			a,b,c = exc_info()
			try:
				self.rollback()
				self.in_transaction = False
				self.deconn(self._conn)
				if not self.DB.SINGLE:
					self._conn = None
			except:
				pass

			raise a,b,c
		else:
			self.commit()

			self.in_transaction = False
			self.deconn(self._conn)
			if not self.DB.SINGLE:
				self._conn = None
		return

