package noris::Person;

=head1 NAME

noris::Person - Funktionen um mit Personen zu arbeiten.

=head1 SYNOPSIS

  use noris::Person (qw(personid_get_by_kundeid));

  # Liste aller Personen des `POP'-Kunden:
  my @personen = personid_get_by_kundeid (1);

=cut

use utf8;
use strict;
use warnings;

BEGIN {
	unshift @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib'
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp (qw(carp cluck croak confess));
use Exporter;

use Dbase::Help;
use Dbase::Globals;

@noris::Person::EXPORT_OK = (qw(personid_get_by_kundeid));
@noris::Person::ISA = ('Exporter');

=head1 ATTRIBUTE

Alle Attribute sind zur Zeit "read-only", das heisst sie koennen nur gelesen
(C<get>) werden; schreiben (C<set>) funktioniert nicht.

=over 4

=item id I<(Integer)>

ID der Person

=item kunde I<(Integer)>

Kunde-ID des Haupt-Kunden der Person

=item name I<(String)>

Name der Person

=item email I<(String)>

E-Mail der Person

=item user I<(String)>

Username der Person

=back

=cut

our $ValidFields =
{
	id	=> {get => \&_get_int},
	kunde	=> {get => \&_get_int},
	name	=> {get => \&_get_string},
	email	=> {get => \&_get_string},
	user	=> {get => \&_get_string}
};

=head1 KONSTRUKTOREN

=over 4

=item noris::Person-E<gt>B<new> (I<[%args]>)

Generiert ein neues Personen-Objekt. I<%args> kann entsprechende Attribute
setzen (siehe oben).

=cut

sub new
{
	my $pkg = shift;
	my %args = @_;
	my $obj;

	for (keys %args)
	{
		my $key = $_;

		if (!defined ($ValidFields->{$key}))
		{
			cluck "Not a valid field: $key";
			delete ($args{$key});
			next;
		}
	}
	
	$obj = \%args;

	return (bless $obj, $pkg);
}

=item noris::Person-E<gt>B<load> (I<%args>)

Laedt ein Personen-Objekt aus der Datenbank. Die Person wird mittels I<%args>
ausgewaehlt. Gueltige Schluessel sind B<id>, B<name> und B<user> die, falls
mehrere gegeben sind, mit I<AND> verknuepft werden.

=cut

sub load
{
	my $pkg = shift;
	my %args = @_;

	my %obj = ();
	my @where = ();

	if ($args{'id'})
	{
		push (@where, 'p.id = ' . qquote ($args{'id'}));
	}
	if ($args{'name'})
	{
		push (@where, 'p.name = ' . qquote ($args{'name'}));
	}
	if ($args{'user'})
	{
		push (@where, 'p.user = ' . qquote ($args{'user'}));
	}

	return if (!@where);

	my $sql = <<SQL;
	SELECT p.id, p.kunde, p.name, p.email, p.user
	FROM person p
	WHERE ${\join (' AND ', @where)}
SQL

	@obj{qw(id kunde name email user)} = DoFn ($sql);

	return (new ($pkg, %obj));
}

=back

=head1 METHODEN

=over 4

=item I<$obj>-E<gt>B<set> (I<$key>, I<$value>)

Setzt das Attribut I<$key> auf den Wert I<$value>. Zur Zeit sind noch alle
Attribute "read-only".

=cut

sub set
{
	my $obj = shift;
	my $key = shift;

	if (!defined ($ValidFields->{$key}) or !defined ($ValidFields->{$key}{'set'}))
	{
		cluck "Not a valid key: $key";
		return;
	}

	return ($ValidFields->{$key}{'set'}->($obj, $key, @_));
}

=item I<$value> = I<$obj>-E<gt>B<get> (I<$key>)

Liefert den Wert das Attributes I<$key>.

=cut

sub get
{
	my $obj = shift;
	my $key = shift;

	if (!defined ($ValidFields->{$key}) or !defined ($ValidFields->{$key}{'get'}))
	{
		cluck "Not a valid key: $key";
		return;
	}

	return ($ValidFields->{$key}{'get'}->($obj, $key, @_));
}

sub _get_int
{
	my $obj = shift;
	my $key = shift;

	return ($obj->{$key});
}

sub _get_string
{
	my $obj = shift;
	my $key = shift;

	return ($obj->{$key});
}

=back

=head1 STATISCHE FUNKTIONEN

=over 4

=item I<@personids> = B<personid_get_by_kundeid> (I<$kundeid>)

Liefert eine Liste von Personen-IDs die diesem Kunden zugeordnet sind. Im
skalaren Kontext wird eine Array-Referenz zurueckgegeben.

=cut

sub personid_get_by_kundeid
{
	my $kundeid = shift;
	my @ret = ();

	DoSelect (sub {
		my $id = shift;
		push (@ret, $id);
	}, <<SQL);
	SELECT p.id
	FROM person p
	WHERE p.kunde = ${\qquote ($kundeid)}
SQL

	return (@ret) if (wantarray ());
	return (\@ret);
}

=head1 AUTOR

Florian octo Forster E<lt>octo@noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut

# vim:background=light:hlsearch:incsearch
