package noris::Kunde;

=head1 NAME

noris::Kunde - Funktionen um mit Kunden bzw. Kunden-Objekten zu arbeiten.

=head1 SYNOPSIS

  use noris::Kunde (qw(kundeid_get_by_flags));

  # Alle Kunden die eine RZ3-Karte haben oder bei denen mindestens eine Person
  # das `service'-Flag hat.
  my @kundeids = kundeid_get_by_flags (kunde => [qw(rz3karte)], person => [qw(service)]);

  # Alle Kunden die _keine_ Trouble-Tickets gemailt bekommen.
  @kundeids = kundeid_get_by_flags (person => [qw(tt)], invert => 1);

=cut

use utf8;
use strict;
use warnings;

BEGIN {
	unshift @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib'
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp (qw(carp cluck croak confess));
use Exporter;

use Dbase::Help;
use Dbase::Globals;

@noris::Kunde::EXPORT_OK = (qw(kundeid_get_all kundeid_get_unterkunden kundeid_get_by_flags));
@noris::Kunde::ISA = ('Exporter');

=head1 STATISCHE FUNKTIONEN

=over 4

=item I<@kundenids> = B<kundeid_get_all> ()

Liefert eine Liste aller (aktiven, also nicht beendeten) KundenIDs. Im skalaren
Kontext wird eine Hash-Referenz der Form C<$kundeid =E<gt> $kundename>
zurueckgegeben.

Falls Hauptkunden gesetzt ist dann werden nur Hauptkunden angezeigt.

=cut

sub kundeid_get_all
{
	my ($hauptkunden) = @_;
	my $ret = {};

	my $HK_SQL = ($hauptkunden)?' AND k.kunde IS NULL ':'';

	DoSelect (sub {
		my $id = shift;
		my $name = shift;
		$ret->{$id} = $name;
	}, <<SQL);
	SELECT k.id, k.name
	FROM kunde k
	WHERE (k.ende IS NULL or k.ende = 0 OR k.ende > UNIX_TIMESTAMP(NOW()))
		$HK_SQL
SQL

	return wantarray ? (keys %$ret) : ($ret);
}

=item I<@kundenids> = B<kundeid_get_unterkunden> ()

Liefert eine Liste aller (aktiven, also nicht beendeten) UnterkundenIDs, eines
Kunden. Im skalaren Kontext wird eine Hash-Referenz der Form
C<$kundeid =E<gt> $kundename> zurueckgegeben.

=cut

sub kundeid_get_unterkunden
{
	my ($kid) = @_;
	my $ret = {};

	DoSelect (sub {
		my $id = shift;
		my $name = shift;
		$ret->{$id} = $name;
	}, <<SQL);
	SELECT k.id, k.name
	FROM kunde k
	WHERE (k.ende IS NULL or k.ende = 0 OR k.ende > UNIX_TIMESTAMP(NOW()))
		AND k.kunde = $kid
SQL

	return wantarray ? (keys %$ret) : ($ret);
}

=item I<@kundenids> = B<kundeid_get_by_flags> (I<%options>)

Liefert eine Liste aller Kunden(IDs) zurueck, bei denen mindestens eines der
gegebenen Flags gesetzt ist. Der optionale Optionen-Hash ist wie folgt
aufgebaut:

  %options =
  (
    kunde  => [$kunde_flag0, $kunde_flag1, ...],
    person => [$person_flag0, $person_flag1, ...],
    invert => 1
    hauptkunden => 0
  );

Die einzelnen Flags in den Array-Referenzen sind die B<Namen> der Deskriptoren.
Im Listenkontext wird eine (unsortierte!) Liste von KundenIDs zurueckgegeben,
im skalaren Kontext eine Hash-Referenz der Form C<$kundeid =E<gt> $kundename>.

Ist das Flag B<invert> gesetzt, werden alle Kunden zurueckgegeben, fuer die
keines der angegebenen Flags gesetzt ist. Ein Aufruf ohne und ein Aufruf mit
B<invert> liefern also zwei disjunkte Mengen die vereinigt eine Liste aller
Kunden ergibt. (Kohaerenzprobleme mal ausser Acht gelassen ;)

Ist das Flag B<hauptkunden> gesetzt, werden nur Hauptkunden zurueckgegeben.

Diese Funktion wurde fuer das Trouble-Ticket-System geschrieben.

=cut

sub kundeid_get_by_flags
{
	my %opts = @_;
	my $ret = {};

	my $kunde_flags  = $opts{'kunde'}  || [];
	my $person_flags = $opts{'person'} || [];
	my $hauptkunden  = $opts{'hauptkunden'} || 0;
	my $invert = $opts{'invert'} ? 'NOT' : '';

	my $sql;

	if (!@$kunde_flags and !@$person_flags)
	{
		if (!$invert)
		{
			return (qw()) if (wantarray ());
			return ({});
		}

		$ret = kundeid_get_all ($opts{hauptkunden});
		return (keys %$ret) if (wantarray ());
		return ($ret);
	}

	my $kunde_mask  = bignum (0);
	my $person_mask = bignum (0);

	for (@$kunde_flags)
	{
		my $name = $_;
		my $id   = find_descr ('kunde', $name, 0x01);

		$kunde_mask |= bignum (1) << $id;
	}

	for (@$person_flags)
	{
		my $name = $_;
		my $id   = find_descr ('pwdomain', $name, 0x01);

		$person_mask |= bignum (1) << $id;
	}

	my $HK_SQL = ($opts{hauptkunden})?' AND k.kunde IS NULL ':'';
	$sql = <<SQL;
	SELECT k.id, k.name, BIT_OR(k.flags) AS kunde_flags, BIT_OR(p.pwuse) AS person_flags
	FROM person p
	LEFT JOIN kunde k ON p.kunde = k.id
	WHERE (k.ende IS NULL OR k.ende = 0 OR k.ende > UNIX_TIMESTAMP(NOW()))
		$HK_SQL
	GROUP BY k.id, k.name
SQL
	$sql .= "\tHAVING ((kunde_flags & $kunde_mask != 0) OR (person_flags & $person_mask != 0))\n" if (!$invert);
	$sql .= "\tHAVING ((kunde_flags & $kunde_mask = 0) AND (person_flags & $person_mask = 0))\n" if ($invert);

	{
		my $tmp = $sql;
		$tmp =~ s/\n/ /g;
		$tmp =~ s/\s+/ /g;
		print STDERR "[noris::Kunde::kundeid_get_by_flags] $tmp\n"
			unless $ENV{TESTING3};
	}
	DoSelect (sub {
		my $id = shift;
		my $name = shift;
		$ret->{$id} = $name;
	}, $sql);

	return (keys %$ret) if (wantarray ());
	return ($ret);
}

=head1 AUTOR

Florian octo Forster E<lt>octo@noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut

# vim:background=light:hlsearch:incsearch
