### Laderoutine

package Loader;

=head1 Loader -- lädt Funktionen des Kundeprogramms nach

Zwecks schnellerem Programmstart besteht das Kundeprogramm aus einem
Riesenhaufen Einzelfunktionen, die alle in einer eigenen Datei liegen,
nämlich @POPHOME@/lib/kunde/NAME.

Dieses Modul lädt die betreffenden Funktionen nach, wenn sie benötigt
werden.

Damit die Verzeichnisse nicht überlaufen, werden in den Dateinamen
(von links) Unterstriche beim Öffnen sukzessive durch Schrägstriche
ersetzt, bis die Datei gefunden wird.

=head2 Neues Interface: Prozeduren einzeln laden

	use Loader qw(FUNKTION); ## FUNKTION sofort laden

Die Funktion wird hier in ihr eigenes Modul "verbannt" und muss somit
alle anderen Funktionen, die sie benötigt, selbst laden.

	use Loader; ## alles nachladen, beim ersten Aufruf

Die Funktion wird in die Package B<main> geladen und hat somit direkt
Zugriff auf alle anderen, von irgendeinem anderen Modul geladenen Funktionen.

Die zweite Methode ist strenggenommen eine Altlast, die noch für das
Kunde-Programm selbst verwendet wird.

=head2 Spezial-Importstatements

=over 4

=item :preload

Hiermit wird dem Loader gesagt, dass alle weiteren nachzuladenden Prozeduren
sofort geladen werden müssen. B<Vorsicht>: Bei rekursiven C<use Loader>-
Aufrufen wird ein Fehler gemeldet. Diese Funktion ist hauptsächlich zum
Debuggen gedacht.

=item :postload

Hiermit wird der Grundzustand wiederhergestellt: Prozeduren werden beim
ersten Aufruf nachgeladen.

Diese beiden Statements werden momentan nicht verwendet.

=back

=cut

use utf8;
use strict;
use warnings;
no warnings qw(redefine);
our $AUTOLOAD;
our $debug;
use Fehler qw(fehler);

my %used;
my %prototype;
my $kf;
our $global;

use Cf qw($POPHOME);
use Dbase qw(db_handle);
my $db = $ENV{'TESTING'} ? undef : Dbase->new();

BEGIN { 
	$kf = $ENV{'KUNDE'};
	if(defined $kf) {
		$kf =~ /(.*)/; # $kf = "$1/../obj";
	} else {
		$kf = "$POPHOME/lib/kunde";
	}

	if (open(my $proto, "$kf/kunde_funcs")) {
		while(defined(my $line = <$proto>)) {
			$prototype{$2} = $1 if $line =~ /^(sub\s+(\w+).*)/;
		}
		close $proto;
	};
}

sub import {
	my $callpkg = caller(0);
	my $pkg = shift;
	my $preload = 0;
	my $xglobal;

	my @syms = @_;
	if(@syms and $syms[0] eq ":global") {
		$xglobal = $callpkg;
		fehler "Import: either globally, or single functions" if @syms != 1;
		@syms = ();
	}
	@syms = keys %prototype unless @syms;
	print STDERR "$callpkg: Lade $pkg '@_'\n" if $debug;
	
	foreach my $sym(@syms) {
		next if defined &{"${callpkg}::$sym"};
		$sym =~ s/^\&//;
		if($sym eq ":preload") {
			$preload = 1;
			next;
		}
		if($sym eq ":postload") {
			$preload = 0;
			next;
		}
		if($sym eq '$base') {
			no strict 'refs';
			*{"${callpkg}::base"} = \$kf;
		} elsif($global) { # alles global laden
			no strict 'refs';
			*{"${callpkg}::$sym"} = \&{"${global}::$sym"};
		} elsif($preload) { # Funktionen jetzt laden
			my $sub = _findload($pkg,$sym);
			next unless $sub;
			*{"${callpkg}::$sym"} = $sub;
		} else { # Funktionen laden, wenn sie gebraucht werden
			defined $prototype{$sym} or fehler "$pkg doesn't export '$sym'";
			eval $prototype{$sym};

			{	no strict 'refs';
				*{"${callpkg}::$sym"} = \&{"${pkg}::$sym"};
			}
		}
	}
	$global = $xglobal if $xglobal;
}

sub _findload($$) {
	my($pkg,$rwhat) = @_;

	my $what = $rwhat;
	my $gwhat = $global ? $global : (__PACKAGE__.'::'.$rwhat);
	$what =~ s/.*:://;

	print STDERR "Loading $global :: $gwhat :: $what\n" if $debug;

	# Wenn dies der zweite Ladeversuch ist, gab es beim ersten Mal einen Fehler.
	if($used{$gwhat."::".$what}++) {
		no strict 'refs';

		fehler "Autoload von '$what' schlug bereits fehl!\n"
			unless defined &{"${gwhat}::$rwhat"};
		return \&{"${gwhat}::$rwhat"};
	}
	
	# Suche bla_fasel_laber auch in bla/fasel_laber und bla/fasel/laber.
	findload: {
		do {
	    	last findload if -s "$kf/$what";
		} while $what =~ s/_/\//;
    	fehler "Kein Autoload von $rwhat (via '$kf'): $!\n";
    	return undef;
	}

	# Der Code muss in die richtige Package...
	unless(eval "package $gwhat; require \"$kf/$what\"; " ) {
		fehler "Autoload von '$rwhat' (via '$kf') schlug fehl: $@\n";
		return undef;
	}

	# Zurückgeliefert wird eine Referenz auf die neue Funktion
	{	no strict 'refs';
		\&{"${gwhat}::$rwhat"};
	}
}

use Dbase::Help qw(Do DoFn qquote);

=for future use

erstmal deaktiviert wegen anhaltender Probleme mit Prototypes, s. RT#359763:

{
	my %cached_id;

	sub get_id($$) {
		return if db_handle->{no_write};

		my ( $tabelle, $name ) = @_;

# Cachen ist problematisch wegen Datenbank-ROLLBACKs!
#		return $cached_id{$tabelle}{$name}
#		  if exists $cached_id{$tabelle}{$name};

		my $q_name = qquote($name);
		$db->DoTrans(sub {
			$cached_id{$tabelle}{$name} =
				$db->DoFn("SELECT id FROM $tabelle WHERE name = $q_name FOR UPDATE") ||
		        	$db->Do("INSERT INTO $tabelle SET name = $q_name");
		});
	}
}

{
	my ( $applikation, $person );

	sub log_call {
		$applikation ||= get_id( applikation => $0 );
		$person      ||= current_user();
		*log_call = \&_log_call;
		goto &_log_call;
	}

	sub _log_call {
		my ( $from, $to ) = @_;

		print STDERR "$from() called $to.\n" if $debug;

		eval {
			my $funktion    = get_id( funktion => $to );
			my $tag         = $db->DoFn('SELECT TO_DAYS(NOW())');

			my $id = $db->DoFn(<<_);
	SELECT id
	FROM   aufrufe
	WHERE  applikation = $applikation
	   AND funktion    = $funktion
	   AND person      = $person
	   AND tag         = $tag
	FOR UPDATE
_
	   		$db->Do( $id ? <<U : <<I );
	UPDATE aufrufe SET anzahl = anzahl + 1 WHERE id = $id
U
	INSERT INTO aufrufe SET
		applikation = $applikation,
		funktion    = $funktion,
		person      = $person,
		tag         = $tag,
		anzahl      = 1
I
		};

		warn $@ if $@;
	}
}

# Funktionsaufrufe sollen für folgende Applikationen nicht geloggt werden:
use constant IGNORE_CALLS_FROM_APPLICATION => { map +($_=>undef), qw(
    /dev/null
    /usr/pop/bin/domainrobot
  ) };

# Aufrufe folgender Funktionen sollen nicht geloggt werden(, weil die sehr oft
# aufgerufen werden und das die Datenbank unnötig belasten würde):
use constant IGNORE_CALLS_TO => { map +($_=>undef), qw(
    current_user
    check_perm
    line_in
    line_init
    line_print_end
    line_printer
    strip_kn
    valid_kunde
    valid_person
    warn_arbeit
  ) };

use Hook::LexWrap qw(wrap);

=cut

sub AUTOLOAD {
	my $what = $AUTOLOAD;
	my($res,@res);
	$what =~ s/.*:://;

	my $callpkg = caller(0);
	print STDERR "For $callpkg: " if $debug;
	my $sub = _findload($callpkg,$what);
	return wantarray ? () : undef unless ref $sub; # warned already

	# Der Aufrufer bekommt einen Eintrag auf die Funktion in seine
	# Symboltabelle geknallt. Das ersetzt die durch den Import
	# eingetragene alte Referenz auf Loader::NAME und verhindert, dass
	# Loader::AUTOLOAD beim nächsten Aufruf wieder aufgerufen wird.
	{	no strict 'refs';
		*{"${callpkg}::$what"} = \&$sub;
	}

=for future use

	unless ( exists IGNORE_CALLS_FROM_APPLICATION->{$0} || db_handle->{no_write} || exists IGNORE_CALLS_TO->{$what} ) {
		( my $function = $callpkg ) =~ s/^${\__PACKAGE__}:://;
		log_call( $function => $what );
		wrap "${callpkg}::$what", pre => sub { log_call( $function => $what ) };
	}

=cut

	# Mit dem GOTO tun wir so, als hätte es den Autoload nie gegeben.
	goto &$sub;
}

1;
