package HousingDB::RZ;

use utf8;
use strict;
use warnings;

BEGIN {
	unshift(@INC, ($ENV{'POPHOME'}||'@POPHOME@').'/lib')
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Dbase::Help;
use Dbase::Globals;
use Loader qw(log_update print_adresse);

=head1 NAME

HousingDB::RZ

=head1 ATTRIBUTE

=over 4

=item B<id> I<(int2)>

=item B<name> I<(char255)>

=back

=cut

our %ValidFields =
(
	id	=> 1,
	name	=> 2
);

return (1);

=head1 METHODEN

=over 4

=item I<$obj> = HousingDB::RZ-E<gt>B<retrieve> (I<$rz_id>)

Laed das entsprechende RZ-Objekt und gibt es zurueck.

=cut

sub retrieve
{
	my $pkg = shift;
	my $id  = shift;
	my $obj = {};

	die unless (defined ($id));
	$id =~ s/\D//g;
	die unless ($id);

	@$obj{qw(id name standort)} =
	DoFn ("SELECT id, name, standort FROM rz WHERE id = $id");

	return (bless ($obj, $pkg));
}

=item HousingDB::RZ-E<gt>B<retrieve_all> ()

Gibt eine Liste aller in der Datenbank gefundenen Objekte zurueck.

=cut

sub retrieve_all
{
	my $pkg = shift;
	my @retval = ();

	DoSelect (sub
		{
			my $id = shift;
			my $obj = retrieve ($pkg, $id);
			push (@retval, $obj);
		}, 'SELECT id FROM rz'
	);

	return (@retval);
}

sub create
{
	my $pkg = shift;
	my $name = shift or die;
	my $standort = shift or die;
	my $obj = {};
	my $id;

	$obj->{'name'} = $name;
	$obj->{'standort'} = get_person ($standort);
	$name = qquote ($name);

	return (undef) unless (defined ($obj->{'standort'}));
	$standort = $obj->{'standort'};

	$obj->{'id'} = Do ("INSERT INTO rz (name, standort) VALUES ($name, $standort)");
	log_update ('rz', 'id', $obj->{'id'}, undef, '*', undef);

	return (bless ($obj, $pkg));
}

sub extinguish
{
	my $obj = shift;
	die if (!$obj->{'id'});

	log_update ('rz', 'id', $obj->{'id'}, undef, '-', undef);
	Do ('DELETE FROM rz WHERE id = ' . $obj->{'id'});
}

=item HousingDB::RZ-E<gt>B<search> ([I<%where>], [I<$opts>])

Sucht Eintraege in der Datenbank. Die Suche kann durch I<%where> eingeschraenkt
werden. I<$opts> kann weitere Optionen angeben. Liefert eine Liste von
HousingDB::RZ-Objekten.

  %where =
  (
    field0 => $value0,
    field1 => $value1,
    ...
  );
  $opts =
  {
    order_by => 'field1'
  };

=cut

sub search
{
	my $pkg = shift;
	my $opts = pop if (scalar (@_) % 2);
	my %fields = @_;
	my $found_invalid = 0;

	for (keys %fields)
	{
		my $key = $_;
		my $val = $fields{$key};

		if (!defined ($ValidFields{$key}))
		{
			warn ("Not a valid field: $key");
			$found_invalid++;
			::delete ($fields{$key});
			next;
		}

		if (!defined ($val))
		{
			$val = 'NULL';
		}
		elsif ($key eq 'name')
		{
			$val =~ s/"/""/g;
			$val = qq("$val");
		}
		elsif ($key eq 'id' or $key eq 'standort')
		{
			$val =~ s/\D//g;
			$val ||= '0';
		}
		else
		{
			die ("You should not get here");
		}

		$fields{$key} = $val;
	}

	return (qw()) if (!%fields and $found_invalid);

	my $where = '';
	my $order = '';

	if (%fields)
	{
		$where = 'WHERE ' . join (' AND ', map { $fields{$_} eq 'NULL' ? $_ . ' IS NULL' : $_ . ' = ' . $fields{$_} } (keys %fields));
	}
	
	if (defined ($opts->{'order_by'}))
	{
		my $tmp = $opts->{'order_by'};
		if (defined ($ValidFields{$tmp}))
		{
			$order = " ORDER BY $tmp";
		}
		else
		{
			warn ("Not a valid field: $tmp");
		}
	}

	my @retval = ();

	DoSelect (sub
		{
			my $id = shift;
			my $obj = retrieve ($pkg, $id);
			push (@retval, $obj);
		}, "SELECT id FROM rz $where $order"
	);

	return (@retval);
}

=item $obj-E<gt>B<id> ()

Liefert die ID des RZs zurueck.

=cut

sub id
{
	my $obj = shift;
	return ($obj->{'id'});
}

=item $obj-E<gt>B<name> ([I<$name>])

Liefer den Namen des RZs zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub name
{
	my $obj = shift;

	if (@_)
	{
		my $new_val = shift;
		my $old_val = $obj->{'name'} || '-';

		if ($new_val ne $old_val)
		{
			my $id = $obj->{'id'};
			
			$obj->{'name'} = $new_val;
			$new_val = qquote ($obj->{'name'});

			log_update ('rz', 'id', $id, undef, 'name', undef, $old_val);
			Do ("UPDATE rz SET name = $new_val WHERE id = $id");
		}
	}
	
	return ($obj->{'name'});
}

=item I<$string> = $obj-E<gt>B<standort> ([I<$person>])

=item I<@array>  = $obj-E<gt>B<standort> ([I<$person>])

Liefert den Standort als String oder als Array mit den Feldern I<Kundenname
(Kundennummer)>, I<Personenname>, I<Adresse>. Falls I<$person> uebergeben wird
und ein zugehoeriges Personen-Objekt gefunden wird, wird der Standort
entsprechend gesetzt.

=cut

sub standort
{
	my $obj = shift;

	if (@_)
	{
		my $new_val = get_person (shift);
		my $old_val = $obj->{'standort'};
		my $id = $obj->{'id'};

		if ($new_val)
		{
			$obj->{'standort'} = $new_val;
			$new_val = qquote ($obj->{'standort'});

			log_update ('rz', 'id', $id, undef, 'standort', undef, $old_val);
			Do ("UPDATE rz SET standort = $new_val WHERE id = $id");
		}
	}

	my $id = $obj->{'standort'};
	my ($pers, $addr, $knm, $knr) = DoFn (<<SQL);
SELECT ps.name, ps.adresse, k.name, ps.kunde
FROM person ps, kunde k
WHERE ps.id = $id and k.id = ps.kunde
SQL
	
	$addr = print_adresse $addr if $addr;
	if (wantarray ())
	{
		return ("$knm ($knr)", $pers, $addr);
	}
	else
	{
		return ("$knm ($knr), $pers, $addr");
	}
}

=back

=head1 SIEHE AUCH

L<HousingDB::Hardware>, L<HousingDB::Rack>

=head1 AUTOR

Florian Forster E<lt>octo at noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut
