package Dbase::_Help;
require Exporter;

use utf8;
use Dbase qw();
use Time::Local qw(timelocal_nocheck);
use Fehler qw(fehler report_fehler ffehler @fehler);
use Date::Calc qw(Add_Delta_YMD);
use Cf qw($DATAHOST $DBDATABASE $MODE);
use UTFkram qw(decode_anything);

use strict;
use warnings;
our @ISA;
BEGIN { @ISA = qw(Dbase Exporter); }
our @EXPORT = qw(Do DoN DoT DoFn DoSeq isotime unixtime in_test date_add_ymd
		sisodate isodate unixdate dbtime hash DelSeq qquote quote 
		DoSelect DoSel DoCopy DoFree DoAtCommit DoTrans DoPing );
our @EXPORT_OK = qw(
  DoBinary
  DoNonFatal
  DoTime
  DoTransExit
  DoTransFail
  DoTransSuspend
  DoReadOnly
  in_list
  in_range
  like_list
  text_only
);
###
### Neue Exporte müssen auch in Dbase::Help hinzugefügt werden!

sub quote($);
sub qquote($;$);

sub import {
	my @arg;
	foreach my $arg(@_) {
		if($arg eq ":readonly") {
			Dbase::db_handle(no_write => 1) unless $ENV{'TESTING'};
		} else {
			push(@arg,$arg);
		}
	}
	Dbase::_Help->export_to_level(1, @arg);
}


=pod

Dbase::Help -- Zentrale Funktionen für den Datenbankzugriff

=head1 Import

Das Import-Flag C<:readonly> schaltet die gesamte Datenbank auf read-only
und verhält sich somit wie ein globales DoReadOnly{}.

Die Verwendung außerhalb von C<tools/> ist verboten.

Neuer (oder anderweitig signifikant geänderter) Code sollte DoReadOnly{} verwenden.

=head1 Datenbankfunktionen


=head2 Do( SqlString )

Führt den SQL-Befehl aus (B<kein> C<select>-Befehl). Abbruch im Fehlerfall.
Liefert nach einem C<INSERT> in eine Tabelle mit einem C<auto_increment>-Feld
dessen Wert, ansonsten die Anzahl der modifizierten Datensätze.


=head2 DoN( SqlString )

Führt den SQL-Befehl aus (B<kein> C<select>-Befehl). Im Fehlerfall wird
die Fehlermeldung zurückgeliefert, ansonsten siehe C<Do>.


=head2 DoCopy( Tabelle WHERE Feld=>Wert... )

Liest alle durch WHERE beschriebenen Einträge der Tabelle ein und
fügt sie mit den angegebenen Ersetzungen neu in die Datenbank ein.

Ist ein Wert eine Code-Referenz, wird dieser Code mit dem alten Inhalt
als Parameter aufgerufen und der zurückgegebene Wert im INSERT-Statement
verwendet.

Wurde nur ein Datensatz angelegt und hat dieser einen
Auto-Inkrement-Index, so liefert DoCopy dessen ID. Ansonsten wird die
Anzahl der neuen Datensätze zurückgeliefert.


=head2 DoSelect( {Sub} SqlBefehl )

Führt den SQL-Befehl aus (immer C<select> o.ä.).

Die Subroutine, normalerweise ein einfacher Codeblock, wird für jede
Ergebniszeile mit den entsprechenden Werten aufgerufen. C<NULL>-Werte
werden als C<undef> übergeben.

Das Ergebnis kann (momentan) sein

=over 4

=item eine Zahl > 0 (Zahl der resultierenden Zeilen),

=item "empty" (kein Ergebnis gefunden)

=item ein String, der mit "-" anfängt (sonstige Fehlermeldung).

=back

Demnächst wird das geändert zu

=over 4

=item eine Zahl >= 0

Zahl der resultierenden Zeilen

=item ein fehler()-Aufruf

z.B. C<fehler("keine Datenbank erreichbar")>.

=back

	$res = DoSelect {
		my($id,$str) = @_;
		print "$id: '$str'\n";
	} "select id,name from bla";
	if($res =~ /^-/) {
		print STDERR "Fehler: $res\n";
	} else {
		print "... insgesamt $res Einträge.\n";
	}

Momentan gibt der obige Code unter C<-w> im Fall, daß das Ergebnis eine
leere Tabelle ist, einen Fehler aus, weil "empty" keine Zahl ist.
Das wird sich aber demnächst ändern.

Soll ein Unterprogramm direkt übergeben werden, sieht der Aufruf so aus:

	DoSelect (\&SubName, "select bla from fasel");

oder z.B. eine Prozedur, die genauso wie C<DoSelect> aufrufbar ist:

	sub Selectiere(&$) {
		my($code,$select) = @_;
		
		DoSelect (\&$code, $select);
	}


=head2 DoSel( SqlBefehl )

Liefert ein Objekt zurück, dem mittels C<nextrow>-Methode je eine Zeile
des Ergebnisses entlockt werden kann. Im Arraykontext werden die Werte
direkt als Array zurückgeliefert, ansonsten eine Referenz darauf.
Am Ende der Tabelle wird C<undef> bzw. eine leere Array zurückgeliefert.

Werden nicht alle Zeilen abgerufen, so wird die Verbindung zum
SQL-Server destruktiv beendet. Das kann bei mehrfacher Verwendung dazu
führen, daß der Server die Verbindung sperrt.

=head2 DoT( SqlBefehl [{Sub}] )

Führt den SQL-Befehl aus (immer C<select> o.ä.).

Mit einem Argument wird eine Referenz auf eine Liste der Ergebnisse
zurückgeliefert. Wird eine Subroutine übergeben, dann wird diese
einzeln mit einer Referenz auf die einzelnen Resultate aufgerufen.

Im Fehlerfall wird ein String mit der Fehlermeldung zurückgeliefert
("empty", wenn die Query keine Resultate liefert), ansonsten die
Zahl der Zeilen, bzw. das Ergebis selbst.


	$res = DoT("select id,name from bla", sub {
		my($id,$str) = @{$_[0]};
		print "$id: '$str'\n";
	});

Dieses Interface ist alt; einfacher zu verwenden ist C<DoSelect()>.


=head2 DoFn( SqlBefehl [Default] [Uniq?] )

Führt den Befehl aus, liefert den ersten Wert (oder alle, wenn
Arraykontext) in der ersten Zeile des Ergebnisses.

Wenn der Befehl im skalaren Kontext aufgerufen wird, der Befehl keine Zeilen
zurückliefert und der Defaultwert angegeben ist, wird dieser zurückgeliefert
anstatt C<undef>.

Werden mehrere Datensätze selektiert (d. h. ist das SELECT-Statement nicht
eindeutig genug), so wird ein L<Fehler|Fehler/fehler> erzeugt.
Dieses Verhalten kann modifiziert werden, indem als drittes Argument ein
wahrer Wert übergeben wird; in diesem Fall wird ggf. so getan, als wäre gar
kein Datensatz gefunden worden, d. h. gar nichts bzw. der Default-Wert
zurückgegeben,

Für das folgende Beispiel existiert bereits eine Funktion (C<name_kunde>):

	$name = DoFn("select name from kunde where id = $id");
	($name,id) = DoFn("select name,id from kunde where id = $id");

=head2 DoSeq( SeqName )

Erhöht den Wert der unter I<SeqName> gespeicherten Folgenummer um 1.

Der Anfangswert ist 1.

Die Folgenummern sind in der Tabelle I<nextid> gespeichert.


=head2 DoBinary( {SUB} )

Normalerweise arbeiten alle Datenbankzugriffe mit UTF8 als
Basis-Zeichensatz. Das funktioniert aber nicht, wenn Binärdaten
(z.B. aus der Passworttabelle) gelesen werden sollen.

Für diesen Fall stellt DoBinary in diesem Block auf Binärmodus um
und stellt dadurch sicher, dass die Daten von der Datenbank nicht 
in irgendeiner Weise modifiziert werden.


=head2 DoReadOnly( {SUB} )

Während der Laufzeit dieses Codeblocks sind Schreibzugriffe grundsätzlich
verboten.


=head2 DoNonFatal( {SUB} )

Normalerweise sind Warnungen der Datenbank fatale Fehler.
Innerhalb dieses Codeblocks wird stattdessen eine Warnung
generiert.


=head2 DoTrans( {SUB} [repeat])

Liefert den Wert von C<{SUB}> zurück. Definiert eine Transaktion.

Beim nichtlokalen Verlassen der Transaktion (goto / fehler() / die())
oder nach einem Aufruf von DoTransFail() wird ein Roll-Back ausgeführt.

Der optionale Wert C<repeat> gibt an, wie oft die Transaktion wiederholt
werden soll, wenn sie auf die Nase fällt. Eine durch
DoTransFail/-Exit/-Suspend bereits unterbrochene Transaktion kann natürlich
nicht wiederholt werden.

=head2 DoTransFail()

Kann nur innerhalb einer Transaktion aufgerufen werden; verursacht nach
deren Ende einen roll-back statt eines commit. Dasselbe geschieht
beim Verlassen der Transaktion mit goto / fehler() / die().


=head2 DoTransExit( [Flag] )

Kann nur innerhalb einer Transaktion aufgerufen werden; verursacht
einen sofortigen commit (Flag TRUE) oder roll-back (Flag FALSE --
Default). Danach ausgeführte Befehle laufen in einer neuen Transaktion!

Flags:

=over 4

=item 1

Es wird ein COMMIT anstelle eines ROLLBACKs durchgeführt.

=item 2

Die Datenbankverbindung wird weggeworfen (sonst: Cache).

=item 4

Die Transaktion wird nicht beendet, sondern nur als nicht-wiederholbar
markiert. Nicht mit den anderen Bits kombinierbar.

=back

=head2 DoTransSuspend( code )

Kann nur innerhalb einer Transaktion aufgerufen werden; verursacht
einen sofortigen commit, führt den Codeblock ohne Transaktion aus,
und öffnet danach eine neue transaktion.


=head2 DoPing( {SUB} )

Innerhalb einer Transaktion: Schicke eine Nachricht an die Datenbank,
damit die Verbindung nicht wegen Timeout zusammenbricht.


=head2 DoAtCommit( {SUB} )

Führt C<SUB> aus, sobald die aktuelle Transaktion beendet ist, bzw.
sofort, falls keine Transaktion offen ist.

Liefert nichts zurück.


=head1 Datumsangaben

Die Datenbank verwendet intern Unix-Zeitangaben (Sekunden seit 1970).

Extern werden ISO-Zeitangaben (C<JJJJ-MM-TT>) verwendet.

Alle von diesen Funktionen ausgegebenen ISO-Zeiten beziehen sich auf die
lokale Zeitzone, I<nicht> auf GMT.


=head2 unixtime( "JJJJ-MM-TT HH:MM:SS" )

Liefert die Unixzeit (Sekunden) für den angegebenen Zeitpunkt.

Ebenfalls möglich sind

	$time = unixtime( "JJJJ-MM-TT","HH:MM:SS" );
	$time = unixtime( JJJJ,MM,TT,HH,MM,SS );
	$time = unixtime( "JJJJ-MM" );
	$time = unixtime( "JJJJ" );



=head2 unixdate( "JJJJ-MM-TT" )

Liefert die Unixzeit (Sekunden) von Mitternacht (0 Uhr) des angegebenen Tags.

Ein Aufruf ist auch in der Form 

	unixdate(JJJJ,MM,TT)

oder

	unixdate(JJJJ,MM)

(erster Tag des Monats) möglich.


=head2 dbtime()

Liefert die aktuelle Zeit auf dem Server als Unix-Timestamp.


=head2 isotime( Sekunden [Flags] )

Liefert einen ISO-Zeitangabe (C<1999-01-01 12:34:56>).

Die Sekunden werden nur mit angegeben, wenn Bit 1 des Flags gesetzt ist.

Für Mitternacht wird die Zeit nicht mit ausgegeben.

Im Arraykontext werden die Daten als Array übergeben.

Es ist okay, wenn C<Sekunden> L<undef|perlfunc/undef> ist; in diesem Fall wird
ebenfalls L<undef|perlfunc/undef> bzw. eine leere Liste zurückgeliefert.

Flags:

=over 4

=item 1

Sekunden mitliefern.

=item 2

Zeitangabe immer mitliefern.

=item 4

Zeitzone mitliefern.

=back

=head2 isodate( Sekunden )

Liefert eine ISO-Datumsangabe (C<1999-01-01>).

Die Uhrzeit wird nicht mit ausgegeben.

Im Arraykontext werden die Daten als Array übergeben.


=head2 add_date_ymd( Sekunden Jahre [Monate [Tage ]] )

Addiert zu einem Unix-Timestamp die angegeben Zahl Jahre/Monate/Tage.

Negative Werte sind erlaubt.


=head2 sisodate( Sekunden [Endflag] )

Liefert eine ISO-Datumsangabe (C<1999-01-12>).

Ist der Tag der erste des Monats, so wird er weggelassen (C<199-02>),
wenn I<EndFlag> = 1 ist. Dasselbe gilt für den letzten Tag des Monats,
wenn I<EndFlag> = 2 ist.

Ist der angegebene Tag der erste des Monats und I<EndFlag> = 2, dann
wird der Vormonat ausgegeben.

Diese Funktion ist vor allem eine Hilfsfunktion für C<daterange>.


=head2 dbtime( Sekunden )

Liefert eine Zeitangabe nach Art des MySQL-Timestamps.


=head2 hash( Zahlen... )

Diese simple Hashfunktion wird unter anderem im Accounting verwendet,
um für die Accountingrecords einen einigermaßen eindeutigen Index zu
generieren.

Programme sollten sich nicht darauf verlassen, daß diese Funktion immer
denselben Wert zurückliefert. Für das Accounting z.B. ist es unerheblich,
ob zwei Werte eines Tages addiert oder als zwei Datensätze gespeichert
werden -- außer daß ersteres effizienter ist.


=head1 Sonstiges


=head2 in_test()

Diese Funktion meldet >0 zurück, wenn gerade getestet wird.

=over 4

Genauer:

=item 0

produktives System

=item 1

QSU-Umgebung

=item 2

Entwicklungssystem

=item 3

unbekannte Umgebung

=item 4

Testlauf

=back

Sie wird vorrangig für das Blockieren oder Umleiten externer Effekte (Mails,
Nameserver) verwendet.


=head2 DelSeq( Name )

Löscht die Sequenznummer namens I<Name>.

Diese Funktion wird nur intern verwendet.


=head2 quote( String )

Quotet I<String> zwecks Verwendung in SQL-Befehlen.

	my $qtext = quote($text);
	Do("update bla set fasel = '$text' where laber");

Die von der Datenbank in Resultaten gelieferten Strings sind
I<nicht> gequotet.

Vorsicht: Nicht doppelt quoten!


=head2 qquote( String [Flag] )

Quotet I<String> zwecks Verwendung in SQL-Befehlen. Der String ist
danach komplett SQL-gequotet, ein undef-Wert liefert als Resultat "NULL".

	my $qtext = qquote($text);
	Do("update bla set fasel = $text where laber");

Die von der Datenbank in Resultaten gelieferten Strings sind
I<nicht> gequotet.

Ist Flag&1, so wird auch eine Eingabe von "-" als NULL gewertet.
Bei Flag&2 gilt dasselbe für einen Leerstring.


=cut


sub DoReadOnly (&;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($job) = @_;
	$self->DbReadOnly($job);
}
sub DoBinary (&;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($job) = @_;
	$self->DbBinary($job);
}
sub DoNonFatal (&;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($job) = @_;
	$self->DbNonFatal($job);
}
sub DoTrans (&;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($job,$rep) = @_;
	$self->DbTrans($job,$rep);
}
sub DoTransFail () {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	$self->DbTransFail();
}
sub DoTransExit (;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	$self->DbTransExit(@_);
}
sub DoTransSuspend (&) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	$self->DbTransSuspend(@_);
}

sub DoPing () {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	$self->DbPing();
}

sub DoAtCommit (&) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($job) = @_;
	$self->DoAtCommit($job);
}

sub DoSeq ($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query) = @_;

	# nicht in Transaktion, wegen Kompatibilität zu altem Code
	$self->DoTrans (sub{
		$self->DbSeq($query);
	});
}

sub DelSeq ($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query) = @_;
	$self->Do("update nextid set id = 0 where name = '$query'");
}

sub DoN ($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query) = @_;

	ffehler {
    	return $self->Do($query);
	} sub {
		return "-".join(" ",@fehler);
	}
}

sub Do ($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my $query = $_[0];
	$self->DbCmd($query);
}

sub DoT ($;&) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query,$cmd) = @_;
	my @data;
	my $again = 0;
	my $res = $self->DoSelect(sub {
		if (ref $cmd) {
			&$cmd( [ @_ ] );
		} else {
			push(@data, [ @_ ] );
		}
	}, $query);

	@data ? \@data : $res;
}

sub DoSelect (&$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($cmd,$query) = @_;
	  ($cmd,$query) = ($query,$cmd) if ref $query eq "CODE";

	my $lines = 0;
	my $res = $self->DbSelect($query);
	use Carp qw(confess); confess "NULL" unless ref $res;

	my @res;
	while(@res = $res->nextrow) {
		$lines++;
		&$cmd(@res);
	}
	$lines;
}

sub DoSel ($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query) = @_;

	$self->DbSelect($query);
}

sub gen_startlog_string($$) {
	my($self,$str) = @_;
	return ("NULL","NULL") if not defined $str or $str eq "";
	$str = $self->quote(substr($str,0,250));
	my $slid;
	unless($slid = $self->DoFn("select id from startlog_kram where daten = '$str'")) {
		$self->DoTrans(sub {
			$slid = $self->DoFn("select id from startlog_kram where daten = '$str'");
			$slid = $self->Do("insert into startlog_kram set daten='$str'")
				unless $slid;
		});
	}
	return $slid;
}

{
	# Cwd::getcwd() fällt auf die Nase, vgl. RT#117302:
	my ($start_name, @start_args, $start_host, $start_dir);

	BEGIN {
		$start_name = $0;
		@start_args = @ARGV;
		chomp($start_host = `hostname -f`);
		chomp($start_dir = `pwd`);
		$start_dir = '?' unless defined $start_dir; # vgl. RT#241020
	}
	sub log_me() {
		my $self = shift;
		return if $self->{'DATAHOST'} ne $DATAHOST;
		return if $self->{'DBDATABASE'} ne $DBDATABASE;

		ffehler {
			local $self->{'no_write'} = 0;
			# Teil-Kopie von current_user()
			my $uid;
			{
				my $uname = $ENV{'USER'} || $ENV{'LOGNAME'};
				my $unuid = getpwuid($<);

				return 1 if not $uname and not $unuid;
				$uname=$unuid unless $uname;
				die "Autorisierungsproblem. Sorry.\n" unless $uname eq $unuid || in_test() || exists $ENV{HTTP_HOST};
			
				$uid = $self->DoFn("select person.id from person,kunde where person.user = ${\ $self->qquote($uname)} and kunde.id = person.kunde and ( kunde.ende is NULL or kunde.ende > UNIX_TIMESTAMP(NOW()) ) ") || "NULL";
			}
			my $sl_name = $self->gen_startlog_string($start_name);
			my $sl_args = $self->gen_startlog_string(join("|",@start_args));
			my $sl_host = $self->gen_startlog_string($start_host);
			my $sl_dir  = $self->gen_startlog_string($start_dir);
			$self->Do("insert writeonce into startlog set person=$uid,programmname=$sl_name, argumente=$sl_args, hostname=$sl_host, verzeichnis=$sl_dir");
		} sub {
			# ignoriere das Problem
			print "Fehler beim Loggen des Programmstarts:\n", map "$_\n", @_ if -t STDOUT and not $ENV{'TESTING2'}; # vgl. RT#241020
			0;
		};

		$self->DbFree(0) if $self->{'no_write'};
	}
}

sub init_handler () {
    $SIG{'HUP'}  = 'Dbase::_Help::handler';
    $SIG{'INT'}  = 'Dbase::_Help::handler';
    $SIG{'QUIT'} = 'Dbase::_Help::handler';
}


sub handler ($) {  # 1st argument is signal name
    my($sig) = @_;
    exit(2);
}

sub DoFree () {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	$self->DbFree();
}

sub DoFn ($;$$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($query,$default,$uniq) = @_;
	my @ret = ();
	local $self->{'FieldHeader'} = 0;
	$self->DoSelect (sub { push @ret, \@_ }, $query);
	if ( @ret > 1 ) {
		if ($uniq) { undef @ret }
		else {
			require Data::Dumper;
			my $dumper = new Data::Dumper [\@ret];
			$dumper->Terse(1)->Indent(0);
			fehler qq("$query" returned ) . @ret . " rows (instead of one):\n" . $dumper->Dump;
		}
	} elsif (@ret) {
		@ret = @{$ret[0]};
	}
	wantarray ? @ret : @ret ? $ret[0] : $default;
}

sub DoCopy($$@) {
	my $old = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my $new = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($tabelle,$sel,%repl) = @_;

	local $old->{'FieldHeader'} = 1;
	my $res = $old->DoSel("select * from $tabelle where $sel");
	my @hdr = $res->nextrow;
	my $count=0;
	my $ret;
	while(1) {
		my @val = $res->nextrow;
		unless(@val) {
			return undef unless $count;
			return $count if $count != 1;
			return $ret;
		}
	
		my $set = "";
		foreach my $hdr(@hdr) {
			my $xval = shift @val;
			next if $hdr eq "timestamp";
	
			my $val = (exists $repl{$hdr}) ? $repl{$hdr} : $xval;
			$val = &$val($xval) if UNIVERSAL::isa($val,"CODE");
			fehler "DoCopy mit Referenz?" if ref $val;

			if(not defined $val) {
				$val = "NULL";
			} elsif($val !~ /\A\d+\z/) {
				$val = "'${\quote $val}'";
			}
			$set .= ", " if $set ne "";
			$set .= "$hdr=$val";
		}
		$ret = $new->Do("insert into $tabelle set $set");
		$count++;
	}
}

sub quote($) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($res) = @_;
	my $ro = 1;
	my($what,$tm) = $self->dbget($ro);
	if($what) {
		if($self->{'binary'}) {
			$res = $what->quote($res);
		} else {
			$res = decode_anything($what->quote(decode_anything($res)));
		}
		$res = substr($res,1,-1);
		$self->dbput($what,$tm,$ro,"quote");
	} else {
		$res =~ s/\\/\\\\/g;
		$res =~ s/'/\\'/g;
		$res =~ s/"/\\"/g;
		$res =~ s/\n/\\n/g;
		$res =~ s/\t/\\t/g;
		$res =~ s/\010/\\b/g;
		$res =~ s/\000/\\000/g;
	}
	return $res;
}

sub qquote($;$) {
	my $self = UNIVERSAL::isa($_[0],__PACKAGE__) ? shift : Dbase::db_handle();
	my($res,$flag) = @_;
	$flag=0 if not defined $flag;
	return "NULL" unless defined $res;
	return "NULL" if $res eq "-" and $flag&1;
	return "NULL" if $res eq "" and $flag&2;

	my $ro = 1; ## WAR: 2
	my($what,$tm) = $self->dbget($ro);
	if($what) {
		if($self->{'binary'}) {
			$res = $what->quote($res);
		} else {
			$res = decode_anything($what->quote(decode_anything($res)));
		}
		$self->dbput($what,$tm,$ro,"qquote");
	} else {
		$res =~ s/\\/\\\\/g;
		$res =~ s/'/\\'/g;
		$res =~ s/"/\\"/g;
		$res =~ s/\n/\\n/g;
		$res =~ s/\t/\\t/g;
		$res =~ s/\010/\\b/g;
		$res =~ s/\000/\\000/g;
		$res = "'$res'";
	}
	return $res;
}

=head2 text_only( String )

Entfernt alle Steuerzeichen aus dem String.
(Vgl. "!%ctrl" in POP-Datenbank.)
Ist String L<C<undef>|perlfunc/undef>, gibt die Funktion ebendies zurück, ohne
dass dies als Fehler betrachtet wird.

=cut

sub text_only($) {
	my ($text) = @_;
	$text =~ s/[[:cntrl:]]//g if defined $text;
	$text;
}

###
### die folgenden Funktionen gehören eigentlich nicht hierher...
###

sub isotime ($;$) {
	my($time,$flag) = @_;
	my($res);
	if(wantarray) {
		return () unless $time;
	} else {
		return undef unless defined $time;
		no warnings 'numeric';
		return "* Datum unbekannt *" if $time == 0 || $time == 1;
	}
	$flag ||= 0;
	my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst);
	if($time =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d) ?(\d\d)\:?(\d\d)\:?(\d\d)$/) {
		($year,$mon,$mday,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
	} else {
		($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($time);
		$mon += 1; $year += 1900;
	}
	if(wantarray) {
		return ($year,$mon,$mday,$hour,$min,$sec,($isdst+1)*3600) if $flag&4;
		return ($year,$mon,$mday,$hour,$min,$sec);
	} 
	$res = sprintf("%04d-%02d-%02d",$year,$mon,$mday);
	if($flag & 1) {
		$res.= sprintf(" %02d:%02d:%02d",$hour,$min,$sec) if $hour || $min || $sec || ($flag & 2);
	} else {
		$res.= sprintf(" %02d:%02d",$hour,$min) if $hour || $min || $sec || ($flag & 2);
	}
	$res.= sprintf(" +0%1d00",$isdst+1) if $flag&4;
	$res;
}

sub unixtime ($;@) {
	my($yy,$mm,$dd,$h,$m,$s) = @_;
	return undef unless defined $yy;

	if($yy =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d) ?(\d\d)\:?(\d\d)\:?(\d\d)$/) {
		($yy,$mm,$dd,$h,$m,$s) = ($1,$2,$3,$4,$5,$6);
	} elsif($yy !~ /^\d{4}$/) { # textuelle Angaben
		my $datum = $yy;
		return undef unless $datum =~ s/^(\d\d\d\d)(?:\-?)(\d\d?)(?:\-(\d\d?))?\b//;
		$datum .= " $mm" if $mm;
		($yy,$mm,$dd) = ($1,$2,$3);
		if ($datum =~ /^\s*(\d\d)(?:\:(\d\d)(?:\:(\d\d))?)?\s*$/) {
			$h = $1; $m = $2; $s = $3;
		}
	}
	$yy -= 1900;
	if($mm) { $mm--; } else { $mm = 0; }
	$dd = 1 unless $dd;
	$h = 0 unless $h;
	$m = 0 unless $m;
	$s = 0 unless $s;

	$dd = 1 if $dd < 1;
	use Carp "confess";
	confess "day" if $dd>31;
	timelocal_nocheck($s,$m,$h,$dd,$mm,$yy);
}

sub isodate ($) {
	my($time) = @_;
	if(wantarray) {
		return () unless defined $time;
	} else {
		return undef unless defined $time;
		return "* Datum unbekannt *" if $time == 0 || $time == 1;
	}

	my($res);
	my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst);
	if($time =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d) ?(\d\d)\:?(\d\d)\:?(\d\d)$/) {
		($year,$mon,$mday,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
	} else {
		($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($time);
		$mon += 1; $year += 1900;
	}
	return ($year,$mon,$mday) if wantarray;
	$res = sprintf("%04d-%02d-%02d",$year,$mon,$mday);
	# $res.= sprintf(" %02d:%02d +0%1d00",$hour,$min, $isdst+1) if $hour || $min || $sec;
	$res;
}

sub sisodate ($;$) {
	my($time,$isend) = @_; ## 1: Monatsbeginn; 2: Monatsende
	return undef unless defined $time;
	return "* Datum unbekannt *" if $time == 0 || $time == 1;

	my($res);
	my($year,$mon,$mday) = isodate $time;
	$isend=0 if $isend==2 and (isodate(date_add_ymd($time,0,0,1)))[2] != 1;

	if($mday == 1 and $isend == 1) {
		$res = sprintf("%04d-%02d",$year,$mon);
	} elsif($mday == 1 and $isend == 2) {
		if($mon == 1) {
			$res = sprintf("%04d-%02d",$year-1,12);
		} else {
			$res = sprintf("%04d-%02d",$year, $mon-1);
		}
	} elsif($isend == 2) {
		$res = sprintf("%04d-%02d",$year, $mon);
	} else {
		$res = sprintf("%04d-%02d-%02d",$year,$mon,$mday);
	}
	# $res.= sprintf(" %02d:%02d +0%1d00",$hour,$min, $isdst+1) if $hour || $min || $sec;
	$res;
}

sub unixdate ($;@) {
	my($yy,$mm,$dd) = @_;
	return undef unless defined $yy;

	unless(defined $mm) {
		return undef unless $yy =~ s/(\d\d\d\d)(?:\-?)(\d\d?)(?:\-(\d\d?))?//;
		($yy,$mm,$dd) = ($1,$2,$3);
	}
	$mm --; $yy -= 1900;
	$dd = 1 unless $dd;
	timelocal_nocheck(0,0,0,$dd,$mm,$yy);
}

sub hash (@) {
	my($res,$i) = 0;
	use integer;

	foreach $i(@_) {
		next if not defined $i or $i !~ /^-?\d+$/s;
		$res %= 0x7edcba9f/57;
		$res = $res + $res * 57 + $res / 5;
		$res += $i;
	}
	$res & 0x7FFFFFFF;
}

sub dbtime ($) {
	my($time) = @_;
	return undef unless defined $time;
	return "00000000000000" if $time == 0;

	my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst);
	if($time =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)$/) {
		($year,$mon,$mday,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
	} else {
		($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($time);
		$mon += 1; $year += 1900;
	}
	sprintf("%04d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec);
}

sub in_test() {
	return 4 if $ENV{'TESTING2'};
	return 1 if lc($MODE) eq "qsu";
	return 2 if lc($MODE) eq "dev";
	return 0 if lc($MODE) eq "production";
	return 3;
}

sub date_add_ymd($;$$$) {
	my($date,$dy,$dm,$dd) = @_;
	$dy ||= 0; $dm ||= 0; $dd ||= 0;

	my($y,$m,$d,$hh,$mm,$ss) = isotime $date;
	($y,$m,$d) = Add_Delta_YMD($y,$m,$d, $dy,$dm,$dd);
	return unixtime($y,$m,$d,$hh,$mm,$ss);
}

my $last_local;
my $last_server;

sub DoTime() {
	my $cur = time;
	if(not defined $last_local or $cur-$last_local>60){
		$last_local=$cur;
		$last_server=DoFn("select UNIX_TIMESTAMP(NOW())");
	}
	return $cur-$last_local+$last_server;
}

=head2 in_list( Datenfeld Negation [Werte] )

liefert einen SQL-Schnipsel, um alle Datensätze auszuwählen, die im
I<Datenfeld> einen der angegebenen I<Werte> haben, wobei L<undef|perlfunc/undef>
für C<NULL>-Werte steht.
Wird im Parameter I<Negation> ein wahrer Wert übergeben, so wird ein
SQL-Schnipsel generiert, der dazu dient, diejenigen Datensätze auszuwählen,
die im I<Datenfeld> I<keinen> der angegebenen Werte haben.
Die Werte werden automagisch gequotet.

Beispiele:

    in_list( 'foo', '', qw/bar baz/ )
        # liefert "foo IN ('bar','baz')"

    in_list( 'foo', NOT => qw/bar baz/ )
        # liefert "( foo IN ('bar,'baz') OR foo IS NULL )"

    in_list( 'foo', 'NOT' )
        # liefert '' im Skalar- und () im Listen-Kontext

=cut

sub in_list($$;@) {
    my ( $field, $not, @values ) = @_;

    return wantarray ? () : 1 unless @values;

    $not = $not ? ' NOT' : '';

    my ( $nulls, @sql );

    push @sql, "$field$not IN (" . join( ',', map qquote($_), @values ) . ')'
        if @values = grep !( !defined && ++$nulls ), @values;

    if ( !@sql && $not ) { push @sql, "$field IS NOT NULL" }
    elsif ( !@sql or ( $not xor $nulls ) ) { push @sql, "$field IS NULL" }

    if   ( @sql == 1 ) { "@sql" }
    else               { '( ' . join( ' OR ', @sql ) . ' )' }
}

=head2 in_range( Datenfeld Mindestwert Maximalwert )

liefert einen SQL-Schnipsel, um alle Datensätze auszuwählen, die im
I<Datenfeld> einen Wert zwischen (jeweils einschließlich) dem
C<Mindestwert> und C<Maximalwert> haben.
Falls einer der Werte L<undef|perlfunc/undef> ist, gilt nur die jeweils andere
Grenze; sind beide L<undef|perlfunc/undef>, gibt die Funktion inchts zurück.

=cut

sub in_range($$$) {
    my ( $field, $lower, $upper ) = @_;
    defined $lower
    ? defined $upper
    ? "$field BETWEEN $lower AND $upper"
    : "$field >= $lower"
    : defined $upper
    ? "$field <= $upper"
    : ();
}

=head2 like_list( Datenfeld Negation [SQL-Wildcards] )

liefert einen SQL-Schnipsel, um alle Datensätze auszuwählen, bei denen das
I<Datenfeld> einen zu einer der übergebenen SQL-Wildcards passenden Werte
aufweist.
Wird im Parameter I<Negation> ein wahrer Wert übergeben, so wird ein
SQL-Schnipsel generiert, der dazu dient, diejenigen Datensätze auszuwählen,
die im I<Datenfeld> I<keinen> passenden Wert haben.

=cut

sub like_list($;$@) {
    my ( $field, $not, @wildcards ) = @_;

    return wantarray ? () : '' unless @wildcards;

    $not = $not ? ' NOT' : '';

    if ( ( my @sql = map "$field$not LIKE " . qquote($_), @wildcards ) == 1 ) {
        "@sql";
    }
    else {
        '( ' . join( ' OR ', @sql ) . ' )';
    }
}

1;
