package Dbase::WebUser;

use utf8;
use warnings;
use strict;
use Dbase::Help qw(DoFn quote qquote);
use Dbase::Globals qw(bignum find_descr get_kunde get_person ist_unterkunde);
use Apache::Constants qw(:common);

=head1 Dbase::WebUser -- überprüft einen Benutzer nach Kriterien

Überprüft einen via Dbase::WebLogin angemeldeten Benutzer, ob er
zugangsberechtigt ist.

	<Location "/">
		AuthName Noris-Intranet
		AuthType Basic
		PerlAuthenHandler Dbase::WebLogin
		PerlAuthzHandler Dbase::WebUser
		AuthAuthoritative off
		require flag pop
	</Location>

Die folgenden C<require>-Einträge werden unterstützt:

=over 4

=item C<kunde A B C>

Der Benutzer muß einem der angegebenen Kunden (oder seinen Unterkunden,
wenn hinter dem Kundennamen ein Stern folgt) zugeordnet sein.

=item flag A B C

Der Benutzer muß eines dieser Flags gesetzt haben. Siehe C<kunde d
pwdomain l>.

=item user A B C

Der Benutzer muß eine der angegebenen Personen sein.

=item queue A B C

Der Benutzer benötigt Leseberechtigung für eine der angegebenen RT-Queues.
C<queue *> bezeichnet alle Queues.


=back

Ein Benutzer is autorisiert, wenn I<alle> C<require>-Zeilen auf ihn
zutreffen.

=cut


my %user;
my %userchk;

sub handler {
	my $time = time;
	my $r = shift;
	my $requires = $r->requires;
	return DECLINED unless $requires;
	my $user = $r->connection->user;
	return DECLINED unless $user;

	return OK unless $r->is_initial_req;
	# Damit werden übergeordnete Verzeichnisse gar nicht erst geprüft

#	main: foreach my $entry (split(/\s+/,$r->dir_config('Auth'))) {
#		my($e,@e) = split(/:/,$entry,2);
#		@e = split(/,/,$e[0]);
#
	foreach my $entry (@$requires) {
		my($e,@e) = split(/\s+/,$entry->{'requirement'});
		next if $e eq "valid-user";

		my $ok = 0;
		if(lc($e) eq "flag") {
			my($fl) = DoFn("select pwuse from person where user = ${\qquote $user}");
			if ($fl) {
				foreach my $xfl(@e) {
					$xfl = find_descr('pwdomain',$xfl);
					next unless $xfl;
					$ok++,last if $fl & (bignum(1)<<$xfl);
				}
			}
		}
		if(lc($e) eq "kunde") {
			my($kd) = DoFn("select kunde from person where user = ${\qquote $user}");
			if($kd) {
				foreach my $xfl(@e) {
					my $uk = ($xfl =~ s/\*$//);
					$xfl = get_kunde($xfl);
					next unless $xfl;
					$ok++,last if $uk ? ist_unterkunde($kd, $xfl) : ($kd == $xfl);
				}
			}
		}
		if(lc($e) eq "user") {
			my($pn) = get_person($user);
			if($pn) {
				foreach my $xfl(@e) {
					$xfl = get_person($xfl);
					next unless $xfl;
					$ok++,last if $kd == $xfl;
				}
			}
		}
		if(lc($e) eq "queue") {
			my($pn) = DoFn("select id from person where user = ${\qquote $user}");
			if($pn) {
				foreach my $xfl(@e) {
					if($xfl eq "*") {
						$ok++,last if DoFn("select distinct person from queue_acl where queue_acl.person = $pn");
					} else {
						$xfl = quote($xfl);
						$ok++,last if $xfl eq "*" or DoFn("select queue_acl.person from queue_acl,queue where queue.name = ${\qquote $xfl} and queue.id = queue_acl.queue and queue_acl.person = $pn");
					}
				}
			}
		}

		unless($ok) {
			print STDERR "NixDa '$user'\n";
			$r->note_basic_auth_failure;
			$r->log_reason("User '$user' not allowed");
			return AUTH_REQUIRED;
		}
	}
	print STDERR "OK '$user'\n";
	return OK;
}
1;
