package Dbase::Getopt;

use utf8;
use warnings;
use strict;
use Umlaut qw(latinmodus textmodus);

use base 'Exporter';

use Getopt::Long ();

use constant DEFAULT_EXPORTS => qw(GetOptions -help);

my %option;

sub getopt_ap {
    my $typ    = shift;
    my $person = &getopt_person;
    my @kunden_ids;
    _load( 'Dbase::Help' => 'DoSelect' );
    DoSelect( sub { push @kunden_ids, @_ },
        "SELECT id FROM kunde WHERE ap_$typ = $person" );
    @kunden_ids;
}

sub push_if_new(\@@) {
    my $arrayref = shift;
    my %seen;
    @seen{@$arrayref} = ();
    push @$arrayref, grep !exists $seen{$_}, @_;
}

our ( @Kunden, @OhneKunden );

use constant EXPORT => {

    ':DEFAULT' => sub { __PACKAGE__->_import( 4, DEFAULT_EXPORTS ) },

    ':kunden' => sub {
        _load( 'Dbase::Globals' => 'unterkunden' );
        $option{'ap-technik=s'} =
          sub { push_if_new( @Kunden, getopt_ap( technik => @_ ) ) };
        $option{'ap-vertrieb=s'} =
          sub { push_if_new( @Kunden, getopt_ap( vertrieb => @_ ) ) };
        $option{'kunde=s'} = sub { push_if_new( @Kunden, &getopt_kunde ) };
        $option{'kunde-und-unterkunden=s'} =
          sub { push_if_new( @Kunden, map unterkunden($_), &getopt_kunde ) };
        $option{'ohne-ap-technik=s'} =
          sub { push_if_new( @OhneKunden, getopt_ap( technik => @_ ) ) };
        $option{'ohne-ap-vertrieb=s'} =
          sub { push_if_new( @OhneKunden, getopt_ap( vertrieb => @_ ) ) };
        $option{'ohne-kunde=s'} =
          sub { push_if_new( @OhneKunden, &getopt_kunde ) };
        $option{'ohne-kunde-und-unterkunden=s'} =
          sub { push_if_new( @OhneKunden, map unterkunden($_), &getopt_kunde ) };

		__PACKAGE__->_import( 4, qw(@Kunden @OhneKunden) );
      },

    '-help' => sub {
        $option{'help|?'} =
          sub {
			$ENV{PAGER} = "less" unless defined $ENV{PAGER};
			open(DATA,"-|", "iconv -f utf8 -t latin1 < $0 | pod2text -o");
			open(PAGER,"|-", $ENV{PAGER});
			latinmodus(\*DATA);
			textmodus(\*PAGER);
			print PAGER $_ while <DATA>;
			close(PAGER);
			close(DATA);
			exit(0);
		  }
    },
};

our @EXPORT_OK = qw(
  GetOptions
  getopt_abt_like
  getopt_ap
  getopt_date
  getopt_descr
  getopt_dienst
  getopt_flags
  getopt_kunde
  getopt_net
  getopt_person
  @Kunden
  @OhneKunden
);

sub import {
    my $package = shift;
    $package->_import( 2, @_ ? @_ : DEFAULT_EXPORTS );
}

sub _import {
    my $package = shift;
    my $level   = shift;

    my ( @errors, @functions2export );
    for my $symbol (@_) {
        if ( grep $_ eq $symbol, @EXPORT_OK ) {
            push @functions2export, $symbol;
        }
        elsif ( defined( my $handler = EXPORT->{$symbol} ) ) {
            $handler->(caller);
        }
        else {
            push @errors, qq("$symbol" wird von $package nicht exportiert.\n);
        }
    }
    $package->export_to_level( $level, $package, @functions2export )
      if @functions2export;

    _load( Carp => 'croak' );
    croak( @errors, 'Abbruch wegen Importfehlern' ) if @errors;
}

sub _load($;@) {
    my $module = shift;
    my @missing = grep !defined &$_, @_ or return;
    eval "require $module";
    die $@ if length $@;
    $module->import(@missing);
}

sub GetOptions {
    Getopt::Long::GetOptions( @_, %option ) or exit 1;

    return @ARGV if defined wantarray || !@ARGV;

    _load( 'Data::Dump' => 'pp' );
    die(  'Unnötige'
          . ( @ARGV == 1 ? 's Argument' : ' Argumente' )
          . ' auf der Kommandozeile: '
          . pp(@ARGV)
          . "\n" );
}

sub getopt_abt_like {
    my ( $option, $value ) = @_;
    _load( 'Dbase::Globals' => 'find_descr' );
    _load( 'Dbase::Help' => qw(DoSelect qquote) );
    my @personen;
    DoSelect( sub { push @personen, shift }, <<_ );
	SELECT person.id
	FROM   kunde, kundemail, person
	WHERE  kunde.id = kundemail.kunde
	   AND kundemail.dienst = ${\ find_descr( dienst => contact => 1 ) }
	   AND kundemail.person = person.id
	   AND person.abt LIKE ${\ qquote($value) }
_
    @personen;
}

sub getopt_date {
    my ( $option, $value ) = @_;
    _load( 'Time::ParseDate' => 'parsedate' );
    defined( my $date = parsedate( $value, PREFER_PAST => 1, WHOLE => 1 ) )
      or die <<_;
Das bei -$option übergebene Datum "$value" verstehe ich nicht.
_
    $date;
}

sub getopt_descr {
    my ( $descr, $option, $value ) = @_;
    _load( 'Dbase::Globals' => 'find_descr' );
    defined( my $n = find_descr( $descr, $value ) )
      or die <<_;
Den bei -$option angegebenen $descr-Deskriptor "$value" kenne ich nicht.
_
    $n;
}

sub getopt_dienst {
    my ( $option, $value ) = @_;
    _load( 'Dbase::Globals' => 'find_dienst' );
    defined( my $dienst_id = find_dienst($value) ) or die <<_;
Den bei -$option angegebene Dienst "$value" kenne ich nicht.
_
    $dienst_id;
}

sub getopt_flags {
    my ( $descr, $field, $option, $value ) = @_;
    _load( 'Dbase::Globals' => 'get_gruppen' );
    my ( $flag_grs, $flag_grc ) = get_gruppen( $descr => $value, 1 );
    my @flagspec;
    push @flagspec, "$field & $flag_grs = $flag_grs" if $flag_grs;
    push @flagspec, "$field & $flag_grc = 0"         if $flag_grc;

    # Kann passieren, denn der strict-Mode von get_gruppen() greift nicht für
    # nicht existente Ausschlusskriterien.
    return unless @flagspec;
    join ' AND ', @flagspec;
}

sub getopt_kunde {
    my ( $option, $value ) = @_;
    _load( 'Dbase::Globals' => 'get_kunde' );
    defined( my $kunde = get_kunde($value) ) or die <<_;
Den bei -$option angegebenen Kunden "$value" kenne ich nicht.
_
    $kunde;
}

sub getopt_net {
    my ( $option, $value ) = @_;
    require Dbase::IP;
    Dbase::IP->new($value) or die <<_;
Das bei -$option angegebene Netz "$value" verstehe ich nicht.
_
}

sub getopt_person {
    my ( $option, $value ) = @_;
    return undef if $value eq '' || $value eq '-';
    _load( 'Dbase::Globals' => 'get_person' );
    defined( my $person = get_person($value) ) or die <<_;
Die bei -$option angegebene Person "$value" kenne ich nicht.
_
    $person;
}

1;

__END__

=head1 NAME

Dbase::Getopt - Auswertung von Kommandozeilenoptionen

=head1 SYNOPSE

    use Dbase::Getopt qw(
      :DEFAULT
      getopt_date
      getopt_descr
      getopt_dienst
      getopt_flags
      getopt_kunde
      getopt_net
      getopt_person
    );

    my ( $Datum, @Dienste, @FlagSpec, @Kunde, $Netz, @Rollen, @User );
    GetOptions(
        'datum=s' => sub { $Datum = &getopt_date },
        'rolle=s'  => sub { push @Rollen,  getopt_descr( dienst => @_ ) },
        'dienst=s' => sub { push @Dienste, &getopt_dienst },
        'flags=s'  => sub {
            push @FlagSpec, getopt_flags( pwdomain => 'person.pwuse', @_ );
        },
        'kunde=s' => sub { push @Kunden, &getopt_kunde },
        'netz=s' => sub { $Netz = &getopt_net },
        'user=s' => sub { push @User, &getopt_person },
    );

    my $sql;
    $sql = '( ' . join( ' OR ', @FlagSpec ) . ' )' if @FlagSpec;

=head1 BESCHREIBUNG

Dieses Modul dient zur vereinfachten Auswertung von Optionen auf der
Kommandozeile.
Es greift auf L<Getopt::Long> zurück, bietet jedoch diverse zusätzliche
Funktionalität.

=head1 FUNKTIONEN

=head2 per Default exportiert

Wird keine Import-Liste angegeben oder explizit C<:DEFAULT> importiert, so
stellt das Modul folgende Funktionalität zur Verfügung:

=over 4

=item GetOptions()

Diese Funktion wird exportiert und funktioniert prinzipiell wie
L<Getopt::Long/Getoptions>, jedoch mit folgenden Abweichungen:

=over 8

=item *

Die L</-help>-Option wird ggf. automagisch berücksichtigt.

=item *

Tritt bei der Auswertung der Optionen ein Fehler auf, wird das Programm mit
Exit-Status 1 beendet.

=item *

Wird die Funktion in einem Void-Kontext aufgerufen, bricht sie die
Programmausführung mit einer entsprechenden Fehlermeldung ab, sofern nach der
Auswertung der Optionen auf der Kommandozeile noch Argumente übrig sind.
Andernfalls liefert die Funktion diese Argumente zurück.

Programme, die solche zusätzlichen Argumente unterstützen möchten, sollten die
Funktion daher so aufrufen:

	my @Args = GetOptions( ... );

Oder, falls man z. B. C<E<lt>E<gt>> nutzen möchte, eben

	@ARGV = GetOptions( ... );

=back

=item -help

L</GetOptions()> implementiert automagisch eine Funktion

    'help|?' => sub { exec perldoc => -F => $0 or die ... }

=back

=head2 exportierbare Optionengruppen

=head3 C<:kunden>

Eine Angabe dieses Tags in der Importliste führt automagisch zur Unterstützung
der folgenden Optionen:

=over 4

=item -kunde Kunde

Selektion nach Kunde(n); bitte die Option mehrfach verwenden,
um mehrere Kunden auszuwählen!
Per Default werden Tickets aller Kunden ausgewertet.

=item -kunde-und-unterkunden Kunde

Wie L</-kunde>, jedoch werden auch Unterkunden (nicht: Oberkunden) mit
selektiert.

=item -ap-technik Person

=item -ap-vertrieb Person

(auch) Kunden, die den angegebenen technischen bzw. vertrieblichen
Ansprechpartner haben, selektieren
(Kann mehrfach verwendet werden, um zusätzliche Kunden zu selektieren.)

=item -ohne-kunde Kunde

Ausschluss von Kunde(n); bitte die Option mehrfach verwenden,
um mehrere Kunden auszuschließen!
Per Default werden Tickets aller Kunden ausgewertet.

=item -ohne-kunde-und-unterkunden Kunde

Wie L</-ohne-kunde>, jedoch werden auch Unterkunden (nicht: Oberkunden)
ausgeschlossen.

=item -ohne-ap-technik Person

=item -ohne-ap-vertrieb Person

(auch) Kunden ausschließen, die den angegebenen technischen
Ansprechpartner haben.
(Kann mehrfach verwendet werden, um zusätzliche Kunden auszuschließen.)

=back

Es werden außerdem die Arrays C<@Kunden> und C<@OhneKunden> exportiert, über
die das Hauptprogramm dann auf die selektierten Kunden zugreifen kann.

=head2 optional exportierbar

Hierbei handelt es sich um Funktionen, die beim L</GetOptions>-Aufruf zur
Bearbeitung einzelner Optionen verwendet werden können, vgl. L</SYNOPSE>.
Dazu müssen jeweils die Argumente übergeben werden, die
L<Getopt::Long/GetOptions> an die benutzerdefinierte Sub-Routine übergeben hat;
manche Funktionen benötigen außerdem zusätzliche Argumente, die grundsätzlich
vorangestellt werden.
Alle Funktionen lösen mittels L<die()|perlfunc/die> Fehler aus, falls etwas
schiefgeht.

=over 4

=item getopt_abt_like( SQL-Wildcard )

liefert eine Liste der IDs aller Mitarbeiter, deren Abteilung einer der
angegebenen SQL-Wildcards entspricht.

=item getopt_date( ... )

wandelt ein Datum mittels L<Time::Parsedate/parsedate> in einen Unix-Timestamp

=item getopt_descr( Deskriptorenliste ... )

ermittelt mittels L<Dbase::Globals/find_descr> den numerischen Wert eines
Deskriptors.
Als erstes Argument muss dazu zusätzlich der Name der Deskriptorenliste
übergeben werden, z. B.

	getopt_descr( quelle => @_ )

=item getopt_dienst( ... )

ermittelt mittels L<Dbase::Globals/find_dienst> den numerischen Wert eines
Dienstes

=item getopt_flags( Deskriptorenliste Datenbankfeld ... )

liefert einen SQL-Baustein zur Selektion von Datensätzen, die zu den angegebenen
Flags passen.
Verwendet intern L<Dbase::Globals/get_gruppen>.

=item getopt_kunde( ... )

ermittelt mittels L<Dbase::Globals/get_kunde> eine Kundennummer

=item getopt_net( ... )

erzeugt ein L<Dbase::IP>-Objekt für den angegebenen Netzbereich

=item getopt_person( ... )

ermittelt mittels L<Dbase::Globals/get_person> die ID einer Person

=back

