package Dbase;

=head1 Datenbankinterface, Low-Level

Dieses Modul implementiert die Anbindung an die Datenbank, inkl.
Wiederholung bei Datenbankausfall, redundante Datenbankanbindung,
Unterscheidung zwischen read-only- und read/write-Datenbank, etc.pp.

=head2 Allgemeine Funktionen

=item db_handle()

Liefert "den" Datenbank-Handle zurück, den die Funktionen aus Dbase::Help
verwenden, wenn sie direkt und nicht objektorientiert aufgerufen werden.

Parameter in den Argumenten führen zum Setzen der entsprechenden
Variablen.

=head2 Variablen

=over 4

=item C<no_write>

Das Setzen dieser Variable erzwingt die Verwendung der
read-only-Datenbank. Alle Schreibversuche werden blockiert.

=item C<queue_result>

Das Setzen dieser Variable erzwingt, daß Daten vor der Verarbeitung
vollständig eingelesen werden, um während der Verarbeitung anfallende
Updates nicht zu blockieren.

Innerhalb von Transaktionen wird diese Variable ignoriert; Daten werden
immer vollständig eingelesen, um Blockieren zu verhindern.
Dasselbe gilt, wenn C<DATAHOST> und C<DATAHOST2> identisch sind.

=item C<binary>

Das Setzen dieser Variable erzwingt, dass eingelesene Daten nicht als
UTF8 interpretiert, sondern in Ruhe gelassen werden.

=item C<nonfatal>

Das Setzen dieser Variable erzwingt, dass MySQL-Warnungen (keine Fehler!)
als Warnung ausgegeben werden, anstatt einen C<fehler()> zu werfen.

=item C<FieldHeader>

Wenn diese Variable I<true> ist, liefern alle C<select>-Aufrufe in der
ersten Zeile die Feldüberschriften.

=item C<fast_error>

Wenn diese Variable I<true> ist, führen Verbindungsfehler zu einer
sofortigen Fehlermeldung.

=back


=head2 lokale Variablen im Dbase-Object

=over 4

=item C<$db-E<gt>cache_timeout>

Cache-Timeout. Verbindungen, die länger als diese Zeit aufgehoben
wurden, werden abgebrochen und nicht recycelt. Sollte etwas niedriger
sein als der Timeout der Datenbank.

=item C<$db-E<gt>cmd_timeout>

Befehls-Timeout. Transaktionen, in denen für längere Zeit, als dieser
Parameter angibt, nichts passiert, werden abgebrochen.

Dieser Wert, wenn gesetzt, überschreibt die Timeout-Parameter in der
Datenbank. Wichtig ist das insbesondere beim Accounting, weil beim
Zählen der Datensätze möglicherweise längere Zeit nichts passiert.

=item C<$db-E<gt>last_stmt>

Der letzte Befehl auf der Leitung.

=item C<$db-E<gt>last_time>

Der Zeitpunkt des letzten Befehl auf der Leitung.

=back

=head2 Speziell interpretierte SQL-Statements

=over 4

=item select readonly

Verwendet eine Verbindung zur read-only-Datenbank.

=item select writeonly

Verwendet eine Verbindung zur write-only-Datenbank.

=item insert/replace/update/delete now

Normalerweise werden Updatebefehle solange verzögert, bis niemand mehr auf
die Daten zugreift, um ein Blockieren zu verhindern. Der Nachteil dieser
Vorgehensweise ist, daß der Update beliebig lang hinausgezögert werden kann.

Wird dieses Schlüsselwort verwendet, dann werden Updates priorisiert und 
blockieren dadurch andere Lesezugriffe.

=head2 Interne Funktionen

Diese Funktionen werden nur von C<Dbase::Help> verwendet.

=over 4

=item C<DbCmd>

=item C<DbSelect>

=item C<DbSeq>

=item C<DbFunc>

=item C<DbTrans>

=item C<DbTransFail>

=item C<DbTransExit>

=item C<DbTransSuspend>

=item C<DbAtCommit>

=back

=for privat

"$readonly" ist:
-1 'select writeonly'
0  write-DB
1  read-DB
2  'select readonly'

=cut

BEGIN { $Cf::delay_read = 1 unless defined $Cf::delay_read; } # siehe ganz unten
use utf8;
use strict;
use warnings;

use Cf qw($DATAHOST $DATAUSER $DATAPASS $DATAHOST2 $DBDATABASE $DATABASE
	$DBCMD_TIMEOUT $DBCACHE_TIMEOUT);
use Carp;
use DBI;
use IO::Socket;
use Time::HiRes qw(time);
use Fehler qw(fehler ffehler warnung nfehler re_fehler add_to_fehler);
use IO::File;
use UTFkram qw(decode_anything safe_encode_utf8);
use Encode qw();
use Encode::Alias qw(define_alias);
define_alias( 'unicode-1-1-utf-7' => 'utf7' );
use Umlaut qw(binmodus);

our $TheDB;

require Exporter;
require DynaLoader;

my $pid = $$; ## aktuelle PID für Datenbankverbindungen nach fork()

my $WDELAY=5;  ## Testintervall für DB-Replikation.

if($ENV{'DB_DEBUG'}) {
	if($ENV{'DB_DEBUG'} eq "STDERR") {
		$Db::LogFile = \*STDERR;
	} else {
		$Db::LogFile = IO::File->new("/tmp/db.$$","w");
		print STDERR "\r"," "x70,"\rLogging => /tmp/db.$$\n";
	}
}

our @ISA = qw(Exporter);

our @EXPORT_OK = qw(
	db_handle
);

sub DbOpen1 {
	my $self = shift;
	my($db,$timeout) = @_;
	my $pop = $self->{'DBDATABASE'} || "pop";
	my $dsn;
	$timeout=1 if $self->{'fast_error'};

	if($DATABASE eq "mysql") {
		$dsn = "DBI:mysql:database=$pop;host=$db;mysql_connect_timeout=$timeout";
	} elsif($DATABASE eq "postgresql") {
		$dsn = "DBI:Pg:dbname=$pop;host=$db";
	} else {
		return "-bad database type '$DATABASE'";
	}
	my $res = DBI->connect($dsn,$self->{'DATAUSER'},$self->{'DATAPASS'}, {RaiseError=>0,PrintError=>0,AutoCommit=>(defined $ENV{'NO_AC'})?0:1,auto_reconnect=>0,mysql_auto_reconnect=>0});
	if (not defined $res or $res eq "") {
		$res = DBI->errstr;
	} elsif(ref $res) {
		my $isol = $self->{'isolation'} || "REPEATABLE READ";
		my $sth = $self->statement($res,time(), undef,"SET SESSION TRANSACTION ISOLATION LEVEL $isol");
		#print $Db::LogFile "#$sth fin\n" if $Db::LogFile;
		$sth->finish();
		print $Db::LogFile "#$res open\n" if $Db::LogFile;
	}
	$res;
}

sub DbOpen {
	my $self = shift;
	my($readonly) = @_;
	my $sock;
	
	main:
	foreach my $to(1,1,2,3,5,8,13) { # ,3
		my $lost = 0;
		my $tries = 0;
		if($readonly > 0) {
			foreach my $db(split(/\s+/,$self->{'DATAHOST2'})) {
				$tries++;
				$sock = $self->DbOpen1($db,$to);
				if(ref $sock) {
					return $sock;
				}
				last main if $sock =~ /access denied/i
					or $sock =~ /Unknown database/
					or $sock =~ /Client does not support authentication protocol/i;
				$lost++ if $sock =~ /error: 111\z/;
			}
		} else {
			foreach my $db(split(/\s+/,$self->{'DATAHOST'})) {
				$tries++;
				$sock = $self->DbOpen1($db,$to);
				return $sock if ref $sock;
				last main if $sock =~ /access denied/i
					or $sock =~ /Unknown database/
					or $sock =~ /Client does not support authentication protocol/i;
				$lost++ if $sock =~ /error: 111\z/;
			}
		}
		last main if $lost == $tries;
	} continue {
		last if $self->{'fast_error'} or $ENV{'TESTING'};
		sleep $to;
	}
	fehler $sock||$DBI::errstr||"no connection";
}

#$hard:
#0 normal
#1 Timeout
#2 NewPID
#3 toomanyConn
#4 error
sub DbClose {
	my $self = shift;
	my($fh,$hard) = @_;
	return unless defined $fh; # Ahem ..?
	fehler "DbClose: fh '$fh'" unless ref $fh;

	if($hard and $hard == 2) {
		print $Db::LogFile "#$fh drop\n" if $Db::LogFile;
		$fh->{'InactiveDestroy'} = 1;
	} else {
		print $Db::LogFile "#$fh close\n" if $Db::LogFile;
		$fh->disconnect();
	}
	1;
}

# Verbindungsparameter setzen.
sub _setup {
	my($self,$fh) = @_;
	my $timeout = $self->{'cmd_timeout'};

	my $tm = time;
	if($timeout) {
		foreach my $var(qw(interactive_timeout wait_timeout net_read_timeout)) {
			my $sth = $fh->prepare("set $var=$timeout");
			unless(ref $sth) {
				return $self->dbfehler($fh,$tm,undef,0,$var);
			}
			print $Db::LogFile "#$sth set $var\n" if $Db::LogFile;

			my $res = $self->exec($sth,$fh,1);
			unless($res) {
				my $lfd = $self->{'log'};
				print $lfd " ERROR:".$sth->errstr."\n" if $lfd;

				return $self->dbfehler($fh,$tm,$sth,$0,$var);
			}
		}
	}
	DbCharset($self,$fh,"utf8");
}

sub DbCharset($$$) {
	my($self,$fh,$chs) = @_;

	foreach my $var(qw(client connection results database server)) {
		my $sth = $fh->prepare("set character_set_$var=$chs");
		unless(ref $sth) {
			return $self->dbfehler($fh,0,undef,0,$var);
		}
		print $Db::LogFile "#$sth set $var\n" if $Db::LogFile;

		my $res = $self->exec($sth,$fh,1);
		unless($res) {
			my $lfd = $self->{'log'};
			print $lfd " ERROR:".$sth->errstr."\n" if $lfd;

			return $self->dbfehler($fh,0,$sth,$0,$var);
		}
	}
}

# Hole einen DB-Handle.
# Alte Handles werden weggeworfen, wenn:
# - Zeit abgelaufen
# - andere PID, also nach einem fork()
sub _dbget {
	my($self,$readonly,$autocommit) = @_;
	my $fh;
	my $free = ($readonly <= 0) ? $self->{'wfree'} : $self->{'rfree'};
	my $tm = time;
	$autocommit = 1 unless defined $autocommit;

	again: {
		if($pid != $$) { ## We can NOT use these. Our parent owns them.
			while($fh = pop @{$self->{'rfree'}}) {
				$self->DbClose($fh->[0],2);
			}
			while($fh = pop @{$self->{'wfree'}}) {
				$self->DbClose($fh->[0],2);
			}
			$pid = $$;
			print $Db::LogFile "# Clear noPID; NewConn:$readonly\n" if
			$Db::LogFile;
			$fh = $self->DbOpen($readonly);
			$self->_setup($fh);
		} elsif(@$free) {
			$fh = pop @$free;

			if($fh->[1]+$self->{'cache_timeout'} < $tm) {
				print $Db::LogFile "#$fh->[0] ReDo: has been idle too long: $fh->[1] $tm\n" if $Db::LogFile;
				$self->DbClose($fh->[0],1);
				redo again;
			}

			$self->{'last_stmt'}=$fh->[2];
			$self->{'last_time'}=$fh->[1];
			print $Db::LogFile "#$fh reuse ".($tm-$self->{'last_time'})."\n" if $Db::LogFile;
			$tm = $fh->[1];
			$fh = $fh->[0];
		} else {
			print $Db::LogFile "# NewConn:$readonly\n" if $Db::LogFile;
			$fh = $self->DbOpen($readonly);
			$self->_setup($fh);
			$self->{'last_stmt'}="<open>";
			$self->{'last_time'}=$tm;
		}
	}
	eval {
		$fh->{'AutoCommit'} = $autocommit
			if $fh->{'AutoCommit'} != $autocommit;
	};
	return($fh,$tm);
}
sub dbget {
	my $self = shift;
	my($readonly,$force_new,$autocommit) = @_;
	my $tm = time;

	return $self->{'transact'}->db() if ref $self->{'transact'} and not $force_new;

	fehler "Du darfst nicht schreiben\n" if $self->{'no_write'} and $readonly <= 0;

	# verwende per Default die r/w-Datenbank, RT#257066
	$readonly = 0 if $readonly==1 and not $self->{'no_write'};

	# verwende nur eine Liste möglicher Verbindungen, wenn es nur einen Server gibt
	$readonly = 0 if $self->{'DATAHOST'} eq $self->{'DATAHOST2'};

	my $fh;
	($fh,$tm) = $self->_dbget($readonly,$autocommit);
	$_[0]=$readonly; # modifiziert den Aufrufer!
	wantarray ? ($fh,$tm) : $fh;
}

sub dbput {
	my $self = shift;
	my($fh,$tm,$readonly,$stmt) = @_;
	confess "dbput ohne filehandle" unless $fh;
	$stmt="<???>" unless defined $stmt;

	if(ref $self->{'transact'}) {
		my $rfh = $self->{'transact'}->db();
		return $self->{'transact'}->put($tm) if $rfh == $fh;
	}

	if(defined $readonly) {
		my $free = ( $readonly <= 0 ) ? $self->{'wfree'} : $self->{'rfree'};
		push(@$free, [ $fh, $tm, $stmt ] );
		print $Db::LogFile "#$fh Put:$readonly $stmt\n" if $Db::LogFile;
		if(@$free > (($readonly <= 0) ? 2 : 3)) {
			my $fh = shift(@$free);
			$self->DbClose($fh->[0],3);
		}
	} else {
		print $Db::LogFile "# Put:undef $stmt\n" if $Db::LogFile;
		$self->DbClose($fh,4);
	}
}

my $perr="";
sub dbfehler {
	my $self = shift;
	my($fh,$tm,$sth,$rw,@msg) = @_;
	my $err = $perr||$sth->errstr||$fh->errstr||"no DB error message available";
	my $nerr = $sth->err||$fh->err||"(#?)";
	my $hi = $fh->{'mysql_hostinfo'};
	print $Db::LogFile "#$sth error\n" if $Db::LogFile;
	eval { $sth->finish; } if $sth;

	my $lfd = $self->{'log'};
	print $lfd " ERROR:".$err."\n" if $lfd;

	if($err =~ /lost conn/i or not defined $rw) {
		$self->DbClose($fh,4);
	} else {
		$self->dbput($fh,$tm,$rw,"ERR:".$err." :: ".$msg[0]);
	}

	if($err =~ /Lost connection to MySQL server during query/) {
		nfehler "Das dauerte zu lang. $nerr","Die Datenbankverbindung wurde getrennt!";
	}
	fehler "[$$] Datenbankfehler $nerr \@ $hi: $err",@msg,"* last:".$self->{'last_time'},$self->{'last_stmt'}
}

sub piperr {
	print $Db::LogFile "# SIGPIPE -- connection timed out\n" if
	$Db::LogFile;
	$perr="Timeout (EPIPE)";
}

sub exec {
	my($self,$sth,$dbh,$no_warn_check) = @_;
	local $SIG{'PIPE'} = \&piperr;

#	# RT#275070: wenn ein Befehl zu lang dauert: frage die DB was da Sache ist
#	local $SIG{'ALRM'} = sub {
#		eval {
#			use IO::Socket::INET;
#			IO::Socket::INET->new(PeerAddr=>"$DATAHOST:27507")->print("$ENV{REMOTE_USER}\cM\cJ");
#		};
#		sleep(5); # Zeit, damit die andere Seite das auch tun kann
#	};
#	alarm(20) if defined $Apache::{request};
#
	$perr="";
	my $res = $sth->execute;
	fehler "MYSQL error ".$dbh->{'mysql_errno'}.": ".$dbh->{'mysql_error'} if $dbh->{'mysql_errno'};

	#if ($dbh->{mysql_warning_count})
	$no_warn_check=1 if $ENV{'DB_IGNORE_WARNINGS'};
	unless($no_warn_check) {
		my $warning_query = $dbh->prepare("SHOW WARNINGS");
		$warning_query->execute;
		while (my $warning = $warning_query->fetchrow_hashref) {
			next if $warning->{'Level'} eq "Note";
			next if $warning->{'Code'} == 1364; # INSERT: nicht erwähntes Feld ohne Defaultwert
			#
			my $w = $warning->{'Level'}." ".$warning->{'Code'}.": ".$warning->{'Message'}."\n";
#			while(my($k,$v) = each %$warning) {
#				$w .= "$k: $v\n";
#			}
			if($self->{'nonfatal'}) {
				warnung $w;
			} else {
				fehler $w;
			}
		}
	}

#	alarm(0) if defined $Apache::{request};
	fehler $perr if $perr;
	return $res;
}

=head1 Logging zu Debugging-Zwecken

Zu Debugging-Zwecken kann für alle ausgeführten SQL-Statements die ausführende
Funktion, der Ausführungszeitpunkt und das Statement mitgeloggt werden.

=cut

my(@sql, $log_statements);

=head2 log_statements([BOOL])

schaltet dieses Logging ein (falls C<BOOL> wahr ist) bzw. aus,
bzw. liefert, wenn kein Wert übergeben wird, einen Wahrheitswert,
der angibt, ob es grade an oder aus ist.

=cut

sub log_statements(;$) {
	$log_statements = shift if @_;
	$log_statements;
}

=head2 clear_statement_log()

löscht den aktuellen Inhalt des Logs

=cut

sub clear_statement_log() { undef @sql }

=head2 show_statement_log()

liefert (in Listenkontext) eine Liste von Hash-Referenzen, wobei jede Referenz
für ein ausgeführtes SQL-Statement steht und jeder Hash folgende Elemente
enthält:

=over 4

=item time

Ausführzeitpunkt

=item db

Datenbank (read/write)

=item function

Funktion, die das Statement ausgeführt hat

=item sql

SQL-Statement

=back

=cut

sub show_statement_log() {
	if (wantarray) { @sql }
	else { \@sql }
}

sub statement {
	my($self,$fh,$tm,$ro,$cmd)=@_;
	my $sth=undef;
	print $Db::LogFile "# ${\(defined $ro ? $ro : '-')} <$cmd>\n" if $Db::LogFile;

	ffehler {
		push @sql, { db => $ro, function => 'statement', time => time, sql => $cmd } if $log_statements;
		if($self->{'binary'}) {
			$cmd = Encode::encode("latin1",$cmd);
		} else {
			$cmd = safe_encode_utf8($cmd);
		}
		$sth = $fh->prepare($cmd, $self->{'queue_result'} ? undef : { "mysql_use_result" => 1});
		ref $sth or fehler "statement";
		#print $Db::LogFile "#$sth prep\n" if $Db::LogFile;
		my $res = $self->exec($sth,$fh,$cmd =~ /\A\s*(?:select|show|set)\b/i);
		unless($res) {
			my $err="exec: $cmd";
			# Beim Testen wird der sub-Teil nicht aufgerufen und somit
			# die Datenbank-Fehlermeldung ansonsten nicht ausgegeben
			$err .= ":".$sth->errstr if $ENV{"TESTING2"};
			fehler $err;
		}
		return $sth;
	} sub {
		$self->dbfehler($fh,$tm,$sth,$ro,@_,$cmd);
	};
}

{
	# Automatisches Freigeben der Verbindung und des Statementhandles

	package Dbase::Ex;
	use Data::Dumper;
	use UTFkram qw(decode_anything);
	sub new { my $w = shift; bless([@_,[],0],$w); }
	sub nextrow {
		my $me = shift;
		my($self,$db,$tm,$sth,$rd,$fh,$stmt,$stored,$rows) = @$me;
		#print $Db::LogFile "#$sth next\n" if $Db::LogFile;

		if($fh) { # FieldHeader: Kopfzeilen mitschicken?
			$me->[5] = 0;
			return wantarray ? @{$sth->{'NAME'}} : [ @{$sth->{'NAME'}} ];
		}

		my $ref;
		if($stored and @$stored) { 
			$ref = shift @$stored;
		} else {
			$ref = $sth->fetchrow_arrayref;
			$me->[8]++;
		}
		if(not ref $ref) {
			print $Db::LogFile "#$sth end\n" if $Db::LogFile;
			$self->dbput($db,$tm,$rd,$stmt);
			bless($me,"Dbase::nix");
			$me->[0] = $rows;
			return wantarray ? () : undef;
		}
		if( $Db::LogFile) {
			local $Data::Dumper::Terse=1;
			local $Data::Dumper::Indent=0;
			print $Db::LogFile "#$sth next: ".Dumper($ref)."\n";
		}
		unless($self->{'binary'}) {
			my @re;
			foreach my $r(@$ref) { 
				push(@re,decode_anything($r));
			}
			$ref = \@re;
		}
		wantarray ? @$ref : $ref;
	}

	sub rows { 
		my $me = shift;
		my($self,$db,$tm,$sth,$rd,$fh,$stmt,$stored,$rows) = @$me;
		while(my $ref = $sth->fetchrow_arrayref) {
			push(@$stored,$ref);
			$rows++;
		}
		$me->[8] = $rows;
		$rows;
	}

	sub DESTROY {
		my $me = shift;
		my($self,$db,$tm,$sth,$rd,$fh,$stmt) = @$me;
		my $bla = $sth->fetchrow_arrayref;
		if($bla) { ## more data than expected?
			## Destruct the connection: don't transfer the rest of the data
			## Warning: The settings below have been derived experimentally.
			print $Db::LogFile "#$sth fin KILL\n" if $Db::LogFile;
			$sth->{'InactiveDestroy'} = 1;
			# $db->{'InactiveDestroy'} = 1;
			$sth->{'Warn'}=0;
			$db->{'Warn'}=0;
			# $db->disconnect();
		} else {
			#print $Db::LogFile "#$sth fin\n" if $Db::LogFile;
			$sth->finish; # may or may not be a no-op
			$self->dbput($db,$tm,$rd,$stmt);
		}
	}
}
{
	# Ditto, nur wurde (wegen Transaktionsdingens) alles schon geholt

	package Dbase::Ex2;
	use UTFkram qw(decode_anything);
    sub new { my $w = shift; bless( [ @_, scalar @{ $_[3] || [] } ], $w ); }
	sub nextrow {
		my $me = shift;
		my($self,$stn,$fh,$dat) = @$me;

		if($fh) { # FieldHeader: Kopfzeilen mitschicken?
			$me->[2] = 0;
			return wantarray ? @$stn : $stn;
		}

		my $ref = shift @$dat;
		if(not ref $ref) {
			return wantarray ? () : undef;
		}
		unless($self->{'binary'}) {
			my @re;
			foreach my $r(@$ref) { 
				push(@re,decode_anything($r));
			}
			$ref = \@re;
		}
		wantarray ? @$ref : $ref;
	}

	sub rows { shift->[4] }

	sub DESTROY {
	}
}
{
	package Dbase::nix;
	sub nextrow { wantarray ? () : undef }
	sub rows { $_[0]->[0]; }
	sub DESTROY { }
}

sub DbSelect {
	my $self = shift;
	my($cmd) = @_;

	my $has_read = 1;
	$has_read = 2 if $cmd =~ s/^\s*select\s+readonly\b/select/is;
	$has_read = -1 if $cmd =~ s/^\s*select\s+writeonly\b/select/is;
	my $read = $has_read;
	my $fh = $self->dbget($read);
	my $tm = time;

	fehler $fh,$self->{'last_stmt'},$self->{'last_time'} unless ref $fh;

	my $sth = $self->statement($fh,$tm,$read,$cmd);
	fehler "undef result" if not ref $sth or not ref $sth->{'NAME'};

	if($self->{'transact'}) {
		my @names = @{$sth->{'NAME'}};
		my $data = $sth->fetchall_arrayref();
		#print $Db::LogFile "#$sth fin\n" if $Db::LogFile;
		$sth->finish(); $sth=undef;
		return Dbase::Ex2->new($self,\@names,$self->{'FieldHeader'},$data);
	} else {
		return Dbase::Ex->new($self,$fh,$tm,$sth,$read,$self->{'FieldHeader'},$cmd);
	}
}

sub DbFunc {
	my $self = shift;
	local $self->{'FieldHeader'} = 0;
	my($cmd) = @_;
	my $res = $self->DbSelect($cmd)->nextrow;
	return wantarray ? () : undef unless $res;
	wantarray ? @$res : $res->[0];
}

{
	# Automatisches Freigeben einer Transaktion
	# 0:zugehöriges DB-Handle
	# 1:DB-Backend-Verbindung
	# 2:Statzeitpunkt
	# 3:verzögerte Befehle
	# 4:read-only-Flag
	# 5:Abbruch-Flag
	# 6:Transaktion wurde unterbrochen-Flag

	package Dbase::ExT;
	use Fehler qw(fehler);

	sub new {
		my $w = shift;
		my($self) = @_;
		my $rd=0;
		$rd=1 if $self->{'no_write'};
		my($db,$tm) = $self->dbget($rd,undef,0);
		print $Db::LogFile "## TRANS $?\n" if $@ and $Db::LogFile;
		print $Db::LogFile "## TRANS start\n" if not $@ and $Db::LogFile;
		bless([$self,$db,$tm,[],$rd,0,0],$w);
	}
	sub db { # Statementhandle holen
		my $me = shift;
		my($self,$db,$tm) = @$me;
		fehler "Transaktion: Nochwas, nach DONE"
			unless ref $db;
		if(wantarray) {
			return ($me->[1],$me->[2]);
		} else {
			return $me->[1];
		}
	}
	sub put {
		# print $Db::LogFile "## TRANS put\n" if $Db::LogFile;
		my $me = shift;
		my $tm = shift;
		$me->[2] = $tm;
	}
	sub delay {
		print $Db::LogFile "## TRANS delay\n" if $Db::LogFile;
		my $me = shift;
		my($self,$db,$tm,$run) = @$me;
		my($job) = @_;
		push(@$run,$job);
	}
	sub fail {
		print $Db::LogFile "## TRANS fail\n" if $Db::LogFile;
		my $me = shift;
		$me->[5]=1;
	}
	sub ping { # Datenbankhandle refreshen
		print $Db::LogFile "## TRANS ping\n" if $Db::LogFile;
		my $me = shift;
		my $tm = time;
		return if $tm < $me->[2] + $me->[0]->{'cache_timeout'}/3;
		$me->[0]->DbCmd('set @ping = 1');
		$me->[2] = $tm;
	}

	sub done { # Transaktion beenden
		my $me = shift;
		my($flag) = @_;
		$flag ||= 0;
		my($self,$db,$tm,$run,$rd,$fail) = @$me;
		$me->[2] = undef;

		foreach my $m(@$run) {
			&$m();
		}

		$self->{transact} = undef;
		unless($db->{'AutoCommit'}) {
			if($fail) {
				print $Db::LogFile "## TRANS rollback\n" if $Db::LogFile;
				$db->rollback();
			} else {
				print $Db::LogFile "## TRANS commit\n" if $Db::LogFile;
				$db->commit();
			}
		}
		$self->dbput($db,$tm,($flag&2) ? undef : $rd,"transaction end");

		$me->[1]=undef;
		#$db->{'mysql_auto_reconnect'} = 1;
	}
	sub not_original {
		my $me = shift;
		my($flag) = @_;
		if(defined $flag) {
			print $Db::LogFile "## TRANS not_orig $flag\n" if $Db::LogFile;
			$me->[6] = $flag;
		} else {
			return $me->[6];
		}
	}

	sub DESTROY {
		my $me = shift;
		my($self,$db,$tm) = @$me;
		return unless ref $db; # schon vorbei
		if(defined $self->{transact} and $self->{transact} == $me) {
			print $Db::LogFile "## TRANS rollback destroy\n" if $Db::LogFile;
			$db->rollback();

			$self->{transact} = undef;
			$me->[1]=undef;

			# It's an error, undefined state -- kill the connection.
			#$db->{'AutoCommit'} = 1;
			##$db->{'mysql_auto_reconnect'} = 1;
		}
		$self->dbput($db,$tm,undef,"transaction rollback");
	}
}


sub DbPing {
	my $self = shift;
	my $tr = $self->{transact};
	$tr->ping() if $tr;
}

sub DbBinary {
	my $self = shift;
	my($job) = @_;
	my($res,@res);
	if($self->{'binary'}) {
		if(wantarray) {
			@res = &$job();
		} else {
			$res = &$job();
		}
	} elsif($self->{transact}) {
		local $self->{'binary'} = 1;
		ffehler {
			DbCharset($self,$self->{transact}->[1],"binary");
			if(wantarray) {
				@res = &$job();
			} else {
				$res = &$job();
			}
			DbCharset($self,$self->{transact}->[1],"utf8");
		} sub {
			DbCharset($self,$self->{transact}->[1],"utf8");
			fehler();
		};
	} else {
		if(wantarray) {
			@res = DbTrans($self, sub { DbBinary($self, $job); });
		} else {
			$res = DbTrans($self, sub { DbBinary($self, $job); });
		}
	}
	wantarray ? @res : $res;
}


sub DbReadOnly {
	my $self = shift;
	my($job) = @_;
	my($res,@res);
	if($self->{'no_write'}) {
		if(wantarray) {
			@res = &$job();
		} else {
			$res = &$job();
		}
	} elsif($self->{transact}) {
		die "DoReadOnly kann nicht in einer Transaktion aufgerufen werden.\n";
	} else {
		local $self->{'no_write'} = 1;
		if(wantarray) {
			@res = DbTrans($self, sub { &$job(); });
		} else {
			$res = DbTrans($self, sub { &$job(); });
		}
	}
	wantarray ? @res : $res;
}


sub DbNonFatal {
	my $self = shift;
	my($job) = @_;
	my($res,@res);
	local $self->{'nonfatal'} = 1;
	if(wantarray) {
		@res = &$job();
	} else {
		$res = &$job();
	}
	wantarray ? @res : $res;
}


sub DbTrans {
	my $self = shift;
	my($job,$rep) = @_;
	my($res,@res);

	if($self->{transact}) {
		if(wantarray) {
			@res = &$job();
		} else {
			$res = &$job();
		}
	} else {
		my $again;
		local $self->{transact} = Dbase::ExT->new($self);
		do {
			$again=0;
			ffehler {
				if(wantarray) {
					@res = &$job();
				} else {
					$res = &$job();
				}
				$self->{transact}->done;
			} sub {
				$rep=0 unless "@_" =~ /^FeHlEr!/m
				   or "@_" =~ /^MYSQL error (\d):/
					and ($1 == 1150 or $1 == 1192 or $1 == 1205
					  or $1 == 1206 or $1 == 1213);
				if($rep and not $self->{transact}->not_original()) {
					$rep--;
					$again=1;
					$self->DbTransExit();
					$self->{transact}->not_original(0);
				} else {
					$self->{transact}->fail;
					$self->{transact}->done;
					fehler();
				}
			}
		} while($again);
	}
	wantarray ? @res : $res;
}

sub DbTransFail {
	my $self = shift;
	my $trans = $self->{transact};
	print $Db::LogFile "## TRANS FAIL\n" if $Db::LogFile;
	fehler "Not in a transaction" unless $trans;
	$trans->fail();
}

sub DbTransExit {
	my $self = shift;
	my $ok = shift;
	$ok ||= 0;
	print $Db::LogFile "## TRANS EXIT $ok\n" if $Db::LogFile;
	my $trans = $self->{transact};
	#fehler "Not in a transaction" unless $trans;
	return unless $trans;

	$trans->fail() unless $ok&5;
	unless($ok & 4) {
		$trans->done($ok & 2);

		print "## TRANS RESTART\n" if $ENV{'TESTING2'};
		$self->{transact} = Dbase::ExT->new($self);
	}
	$self->{transact}->not_original(1);
}

sub DbTransSuspend {
	my $self = shift;
	my $proc = shift;
	my $trans = $self->{transact};
	#fehler "Not in a transaction" unless $trans;
	return &$proc() unless $trans;
	print "# TRANSACTION SUSPEND\n" if $ENV{'TESTING2'};
	print $Db::LogFile "## TRANS SUSPEND\n" if $Db::LogFile;
	my(@res,$res);

	ffehler {
		$trans->done();

		if(wantarray) { @res = $proc->(); } else { $res = $proc->(); }

		print "# TRANSACTION RESUME\n" if $ENV{'TESTING2'};
		print $Db::LogFile "## TRANS RESUME\n" if $Db::LogFile;
		$self->{transact} = Dbase::ExT->new($self);
		$self->{transact}->not_original(1);

	} sub {
		print "# TRANSACTION ERROR RESUME\n" if $ENV{'TESTING2'};
		print $Db::LogFile "## TRANS ERROR RESUME\n" if $Db::LogFile;
		$self->{transact} = Dbase::ExT->new($self);
		$self->{transact}->not_original(1);
  		
		re_fehler();
	};

	if(wantarray) { return @res; } else { return $res; }
}

sub DbAtCommit {
	my $self = shift;
	my($job) = @_;
	if($self->{transact}) {
		$self->{transact}->delay($job);
	} else {
		&$job();
	}
}

sub DbCmd {
	my $self = shift;
	my($cmd) = @_;

	my $res;
	local $self->{'FieldHeader'} = 0;
	my $rw = 0;
	$rw = 1 if $cmd =~ m#^\s*(SET)\b#si;
	$rw = -1 if $cmd =~ s#^\s*(INSERT|REPLACE|UPDATE|DELETE)\s+writeonce\b#$1 #si;
	print $Db::LogFile "# $rw <$cmd>\n" if $Db::LogFile;

	fehler "Du darfst nicht schreiben\n" if $self->{'no_write'} and $rw <= 0;

	return 1 if $cmd eq "begin" or $cmd eq "end";
	my $fh = $self->dbget($rw);
	my $tm = time;

	$cmd =~ s#^\s*(INSERT|REPLACE|UPDATE|DELETE)\s+now\b#$1 #si or
	$cmd =~ s#^\s*(INSERT|REPLACE|UPDATE|DELETE)(?!\s+(?:low_priority|delayed)\b)#$1 low_priority #si if $DATABASE eq "mysql";
	$cmd =~ s#^\s*(INSERT|REPLACE|UPDATE|DELETE)(?:\s+(?:low_priority|delayed)\b)#$1 #si if $DATABASE eq "postgresql";

	push @sql, { db => $rw, function => 'DbCmd', time => time, sql => $cmd } if $log_statements;

	if($self->{'binary'}) {
		$cmd = Encode::encode("latin1",$cmd);
	} else {
		$cmd = safe_encode_utf8($cmd);
	}

	if($self->{'log'}) {
		my $lcmd = $cmd;
		$lcmd =~ s/[\s\n]+/ /g;
		Encode::_utf8_off($lcmd);
		my $lfd = $self->{'log'};
		print $lfd "< $lcmd\n";
	}

	my $sth = $fh->prepare($cmd, $self->{'queue_result'} ? undef : { "mysql_use_result" => 1});
	unless(ref $sth) {
		$self->dbfehler($fh,$tm,undef,$rw,$cmd);
	}
	#print $Db::LogFile "#$sth prep\n" if $Db::LogFile;

	ffehler {
		$res = $self->exec($sth,$fh,$cmd =~ /\A\s*(?:select|show|set)\b/i);
	} sub {
		add_to_fehler($cmd);
		re_fehler();
	};
	unless($res) {
		my $lfd = $self->{'log'};
		print $lfd " ERROR:".$sth->errstr."\n" if $lfd;

		$self->dbfehler($fh,$tm,$sth,$rw,$cmd);
	}
	if($cmd =~ m#^\s*INSERT\b#i) {
		$res = $sth->{'mysql_insertid'}||1 if $DATABASE eq "mysql";
		$res = $sth->{'pg_oid_status'} if $DATABASE eq "postgresql";
	}

	my $lfd = $self->{'log'};
	print $lfd "> $res\n" if $lfd;
	flush $lfd if $lfd;

	#print $Db::LogFile "#$sth fin\n" if $Db::LogFile;
	$sth->finish;
	$self->dbput($fh,$tm,$rw,"2 ".$cmd);
	return 0 unless defined $res;
	return $res;
}

my $step = 0;


{
	# Automatisches Freigeben des Locks
	package Dbase::LockEx;
	use Fehler qw(warnung);

	sub new {
		my $w = shift;
		bless([@_],$w);
	}
	sub kill {
		my $me = shift;
		my($fh,$tm,$lock,$self,$ro) = @$me;
		return unless defined $lock;
		$me->[2] = undef;
		my $sth = $self->statement($fh,$tm,$ro,"select release_lock($lock)");
		my($rres) = $sth->fetchrow_array;
		print $Db::LogFile "#$sth fin2 $tm $lock\n" if $Db::LogFile;
		$sth->finish;
		$self->dbput($fh,$tm,$ro,"unlock $lock");

		warnung "Die Sperre für '$lock' konnte nicht freigegeben werden ??\n" unless $rres;
	}
	sub DESTROY {
		my $me = shift;
		$me->kill();
	}
}

sub DbSeq {
	my $self = shift;
	my($seq) = @_;

	my $res = $self->DbFunc("select id from nextid where name = '$seq' for update");
	unless(defined $res and $res =~ /^\d+$/) {
		return 1 if $self->DbCmd("insert into nextid set id=1, name='$seq'") == 1;
		croak "Seq-Update-Problem";
	}
	$res++;
	my $ret = $self->DbCmd("update nextid set id = $res where name = '$seq' and id = ".($res-1));
	croak "Seq-Update-Problem" unless $ret == 1;
	$res;
}

use IO::File;
my $logged_me; # Saved call setup data?
sub new {
	my $self = shift;
	my $pack = ref($self) || $self;

	# in Dbase::Help sind ein Haufen Abfragen auf isa(__PACKAGE__),
	# und dort ist eh das öffentliche Interface; deshalb gleich
	# umsetzen.
	if($pack eq "Dbase") {
		require Dbase::_Help unless defined &Dbase::_Help::new;
		$pack = "Dbase::_Help";
	}
	$self = bless { ref $self ? %$self : (), @_ }, $pack;
	if(defined $self->{'DB'}) {
		my $db = $self->{'DB'};
		no strict 'refs';

		$self->{'DATAHOST'} = ${"Cf::DATAHOST_$db"};
		$self->{'DATAUSER'} = ${"Cf::DATAUSER_$db"};
		$self->{'DATAPASS'} = ${"Cf::DATAPASS_$db"};
		$self->{'DATAHOST2'} = ${"Cf::DATAHOST2_$db"} || ${"Cf::DATAHOST_$db"};
		$self->{'DBDATABASE'} = ${"Cf::DBDATABASE_$db"};
	}
	$self->{'DATAHOST'} = $DATAHOST unless defined $self->{'DATAHOST'};
	$self->{'DATAUSER'} = $DATAUSER unless defined $self->{'DATAUSER'};
	$self->{'DATAPASS'} = $DATAPASS unless defined $self->{'DATAPASS'};
	$self->{'DATAHOST2'} = $DATAHOST2 unless defined $self->{'DATAHOST2'};
	$self->{'DBDATABASE'} = $DBDATABASE unless defined $self->{'DBDATABASE'};
	$self->{'DBLOG'} = $ENV{'DBLOG'} unless exists $self->{'DBLOG'};
	$self->{'cache_timeout'} = $DBCACHE_TIMEOUT||10 unless exists $self->{'cache_timeout'};
	$self->{'cmd_timeout'} = $DBCMD_TIMEOUT unless exists $self->{'cmd_timeout'};
	$self->{'queue_result'} = 0 unless exists $self->{'queue_result'};
	$self->{'fast_error'} = 0 unless exists $self->{'fast_error'};
	$self->{'FieldHeader'} = 0 unless exists $self->{'FieldHeader'};
	$self->{'binary'} = 0;
	$self->{'nonfatal'} = 0;
	$self->{'rfree'} = [];
	$self->{'wfree'} = [];
	$self->{'last_stmt'} = "<new>";
	$self->{'last_time'} = 0;

	$self->{'queue_result'} = 1 if $self->{'DATAHOST'} eq $self->{'DATAHOST2'};
	
	my $ro = 1;
	#my $fh = $self->dbget($ro);
	my $tm = time;
	
	if(defined $self->{'DBLOG'}) {
		$self->{'log'} = new IO::File $self->{'DBLOG'}, O_WRONLY|O_APPEND|O_CREAT;
		binmodus($self->{'log'});
	}
	#$self->dbput($fh,$tm,$ro,"Log");

	unless($logged_me) {
		$logged_me=1;
		$self->log_me();
	}
	$self;
}

sub db_handle(@) {
	unless($TheDB) {
		$TheDB = Dbase->new(@_);

		fehler "No Database\n" unless ref $TheDB;

	}
	if(@_) {
		while(@_) {
			my $a = shift;
			my $b = shift;
			$TheDB->{$a} = $b;
		}
        if(defined $TheDB->{'DB'}) {
			my $db = $TheDB->{'DB'};
			no strict 'refs';

			$TheDB->{'DATAHOST'} = ${"Cf::DATAHOST_$db"} || ${"Cf::DATAHOST"};
			$TheDB->{'DATAUSER'} = ${"Cf::DATAUSER_$db"} || ${"Cf::DATAUSER"};
			$TheDB->{'DATAPASS'} = ${"Cf::DATAPASS_$db"} || ${"Cf::DATAPASS"};
			$TheDB->{'DATAHOST2'} = ${"Cf::DATAHOST2_$db"} || ${"Cf::DATAHOST_$db"} || ${"Cf::DATAHOST2"} || ${"Cf::DATAHOST"};
			$TheDB->{'DBDATABASE'} = ${"Cf::DBDATABASE_$db"} || ${"Cf::DBDATABASE"};
        }

		$TheDB->DbFree(0);
	}
	$TheDB;
}

sub DbFree {
	my $self = shift;
	my($rw) = @_;
	my $fh;
	if(not defined $rw or $rw == 1) {
		while($fh = pop @{$self->{'rfree'}}) {
			$self->DbClose($fh->[0],($pid != $$)?2:0);
		}
	}
	if(not defined $rw or $rw == 0) {
		while($fh = pop @{$self->{'wfree'}}) {
			$self->DbClose($fh->[0],($pid != $$)?2:0);
		}
	}
}

sub DESTROY {
	my $self = shift;
	use Scalar::Util qw(reftype);
	return unless reftype($self) eq 'HASH'; # Workaround, s. RT#244083
	$self->DbFree();
}
END {
	$TheDB = undef;
}

# Wenn Dbase.pm vor Cf.pm importiert wird, dann muss das Lesen der Konfig
# in Cf.pm verzögert werden, bis Dbase.pm fertig geladen ist.
# Das erledigt dieser Code.
if($Cf::delay_read > 1) {
	$Cf::delay_read = 0;
	Cf::read_db();
} else {
	$Cf::delay_read = 0;
}

1;

__END__

