package Acct;

use utf8;
use warnings;
use strict;
use Cf qw($CACHEDIR $LOGDIR $LOGSUB $POPHOME $WDESCR $DEFAULTKLASSE);
use Dbase::Help;
use Dbase::Globals;
use Dbase::Globals qw(find_dienst);
use Dbase::IP;
use Getopt::Std;
use Time::Local;
use File::Path qw(mkpath);
use File::Basename qw(dirname);
use IO::File;
use IO::Pipe;
use Compress::Zlib;
use File::ShLock;
use Fehler qw(fehler);
use Umlaut qw(utf8modus binmodus);
use Socket qw(inet_ntoa);

my %acctcache;

# Dienste, bei denen das Accounting für unterschiedliche Tage parallel
# verarbeitet werden darf:
use constant DO_PARALLEL =>
  { map +( lc() => undef ), qw(cache cache-hit ftp ip isdn sms www) };
# Gegenbeispiele: Radius- und exim-Accounting, bei denen die
# Verarbeitungsreihenfolge relevant ist.

=head1 Datenbankfunktionen für die Accountinganalyse

=head2 Variablen

Implementiert als Hash im C<Acct>-Objekt, normalerweise gesetzt via
C<$acct = Acct->new(Name => "Wert", ...)>.

=over 4

=item C<this>

Die aktuelle Zeile des analysierten Logs, gesetzt von C<acct_head()> und
geloggt von C<acct_tail()>. Kann verändert werden.

=item C<do_reverse>

Normalerweise versucht das System, lokales und entferntes Accounting
zu vertauschen, damit die Daten bidirektional korrekt gezählt werden.

Soll dem nicht so sein, ist diese Variable auf Null zu setzen.

Default ist 1.

=item want

Ein Array (oder eine mit Kommas getrennter String) von Datenquellen, die
das aktuelle Accountingskript benötigt.

=over 4

=item kunde

Mapping der Kundennamen in der Datenbank.

=item uucp

Mapping der UUCP-Tabelle in der Datenbank (= Kunden-Aliasnamen).

=item person

Mapping der Personennamen in der Datenbank.

=item mail

Mapping der Mailadressen in der Datenbank.

B<Nicht implementiert.> Das Mail-Backend macht diese Lookups selbst.

=item domain

Mapping der Domainnamen in der Datenbank.

=item ip

Mapping der IP-Adressen in der Datenbank.

=item ip_NAME

Liest die mit I<gzip> komprimierte Datei C</var/cache/pop/net.I<NAME>.I<YYYYMM>>
ein und merkt sich die dort gefundenen Netze.

=back

Im Fall von IP-Adressen muß die spezifischste Quelle zuerst angegeben werden.

=item C<line>

Zeilennummer; wird mit jedem C<line()>-Aufruf hochgezählt.

=item C<verbose>

Die aktuelle Aktion (analysierte Zeilennummer etc.) wird auf Konsole
mitgeloggt.

=item C<do_tarife>

Wenn gesetzt: Aktualisiere die Tarife für die genutzten Dienste.

Default ist 1.

=item C<zip>

Wenn gesetzt: Komprimiere die ausgegebenen Logs.

Realisiert via Perl-Modul, d.h. ohne externe Aufrufe von C<gzip>.

=item C<logdir>

Subverzeichnis für die zu loggenden Daten.
Der resultierende Pfad steht in @POPCONF@, als C<LOGDIR/LOGSUB>.
Das Subverzeichnis ist dort als C<\X> kodiert.

Ohne diesen Wert wird nichts geloggt.

=item C<timestamp>

Zeitstempel der zu verarbeitenden Accountingdaten.

=item C<dienst>

Dienst, dessen Daten geloggt werden. Siehe C<kunde d dienst>.

=item C<quelle>

Quelle, aus der die Daten stammen. Siehe C<kunde d quelle>.

=item C<kunde>

Nummer des Kunden, für den die Datensätze geloggt werden. Wird gesetzt,
wenn die aktuelle Datei nur einen Kunden betrifft.

=item C<idienst>

Dienst (als Zahl), generiert aus C<dienst>.

=item C<hash_uucp>

Hash: Mapping Kundenname o.ä. auf Kundennummer.

Die übrigen C<hash_*>-Variablen sollten nicht direkt verwendet werden.

=back

=head2 Methoden

=over 4

=item C<new( Arg => val, ... )>

Initialisierung. Die Argumente C<timestamp want dienst quelle>
müssen angegeben werden.

Bitte in Accountingskripts nicht direkt benutzen, sondern stattdesen
C<do_start(...)> aufrufen.

=item C<line( Zeile )>

Aufruf nach dem Einlesen einer Logzeile. Liefert I<true> zurück,
wenn die Zeile ein Merker aus einer alten Logdatei ist und übersprungen
werden sollte.

Bitte in Accountingskripts nicht direkt benutzen, sondern C<line_in($acct)>
aufrufen; diese Prozedur liefert die zu analysierende Zeile oder C<undef>
bei Dateiende / Fehler.

=item C<acct( Kunde kDienst Remote remDienst Pakete Bytes [Zeile] )>

Merkt sich die eingelesene Zeile, schreibt die Zeile (oder das Argument
des letzten Aufrufs von C<line>) in temporäre Logdateien.

I<Kunde> und I<Remote> bestehen aus einer Kundennummer und/oder einem Ziel.
Vor Letzterem steht zwecks Unterscheidung ein '>'-Zeichen; sind beide
angegeben, sind die durch eine Leerstelle voneinander getrennt.

I<Kunde>, I<kDienst> und I<remDienst> können durch C<undef> ersetzt werden;
in diesem Fall werden die im C<new()>-Aufruf angegebenen Werte verwendet.

I<remDienst> wird verwendet, wenn I<do_reverse> an ist und I<Remote> eine
Kundennummer ist.

=item C<finish()>

Beendet den Accountingvorgang, schreibt Daten in die Datenbank,
friert die Logdaten ein. Ohne diesen Aufruf werden die Daten weggeworfen.

In Accountingskripts ist stattdessen C<do_end($acct)> aufzurufen.

=item C<who_dom( Domain [Suche] )>

Mappt Domainname auf Kundennummer.

Ein optionales zweites Argument besagt, wenn C<true>, daß bei Mißlingen
der direkten Suche B<nicht> versucht werden soll, den Domainnamen zu einer
IP-Adresse aufzulösen und nach der weiterzusuchen.

=item C<who_ip( Adresse )>

Mappt IP-Adresse auf Kundennummer.

=back

=head2 Usage

Beispiel (C</usr/pop/lib/acct/ip>):

	use Dbase::Acct;

	sub acct_ip {
		my($file) = @_;
		my $acct = do_start(want=>[qw(kunde ip domain ip_de services)],
							timestamp=>time, logdir=>"Test",
							dienst=>"ip", quelle=>"test");
		my $line;
		while(defined($line = line_in($acct))) {
			my($locadr,$remadr,$numpack,$volpack,$prot,$locserv,$remserv) = split;
			my($kunde,$kdienst,$rem,$rdienst);
			if($locadr =~ s/^_//) {
				($kunde,$kdienst) = $acct->who_ip($locadr);
			} else {
				($kunde,$kdienst) = $acct->who_dom($locadr);
			}
			if($remadr =~ s/^_//) {
				($rem,$rdienst) = $acct->who_ip($remadr);
			} else {
				($rem,$rdienst) = $acct->who_dom($remadr);
			}
			$acct->acct($kunde,$kdienst, $rem,$rdienst, $numpack,$volpack);
		}
		do_end($acct);
	}
	
Die Aufrufe C<do_start>, C<line_in> und C<do_end> sind Wrapperprozeduren
für C<acct->new>, C<line> und C<finish> aus C<acct/acctrun>, die zusätzlich
globale Parameter (z.B. für Tests) einstellen.

=cut

sub gen_filename {
	my $self = shift;
	my($cust) = @_;
	my $fn = $LOGSUB;

	$fn =~ s/\\C/$cust/g;
	$fn =~ s/\\c/name_kunde($cust)/eg;
	$fn =~ s/\\T/$self->{tt}/g;
	$fn =~ s/\\M/$self->{mm}/g;
	$fn =~ s/\\J/$self->{jj}/g;
	$fn =~ s/\\X/$self->{logdir}/g;
	$fn =~ s/\\S/$self->{iquelle}/g;
	$fn =~ s/\\D/$self->{idienst}/g;
	$fn =~ s/\\s/$self->{quelle}/g;
	$fn =~ s/\\d/$self->{dienst}/g;

	$fn = $LOGDIR."/".$fn;
	mkpath(dirname($fn),0,0770);
	$fn;
}

sub read_kunde() {
	my $self = shift;
	$0=$self->{'zb'}." kunde";
	DoSelect {
		my ($cust, $name) = @_;
		$self->{'hash_uucp'}{$name} = $cust
			unless exists $self->{'hash_uucp'}{$name};
		# unlink($self->gen_filename($cust).".x");
	} "select id,name from kunde where $self->{'tmcmd1'}";
}

sub read_perskunde() {
	my $self = shift;
	$0=$self->{'zb'}." person";
	DoSelect {
		my ($name, $cust) = @_;
		$self->{'hash_uucp'}{$name} = $cust 
			unless exists $self->{'hash_uucp'}{$name};
	} "select LOWER(person.user), person.kunde from person,kunde where 
			person.kunde = kunde.id and person.user is not NULL and $self->{'tmcmd1'}";
}

sub read_ipkunde() {
	my $self = shift;
	my $tmc = $self->{'tmcmd'}; $tmc =~ s/DEST/ipkunde/g;
	$0=$self->{'zb'}." ipkunde";
	my $no_accounting = find_descr( ipflags => no_accounting => 1 );
	DoSelect {
		my ($cust,$ip,$bits,$tarif,$ziel) = @_;
		$ip = Dbase::IP->new_db($ip,$bits); # ohne Bitmaske
		$self->{hash_ip}[$bits]{$ip->addr} = "$cust >$ziel";
		$self->{hash_tarif}[$bits]{$ip->addr} = $tarif;
		if($ip->is_v4) {
			$self->{'hash_ip4'}[$bits]{$ip->old_ip4} = "$cust >$ziel";
			$self->{'hash_tarif4'}[$bits]{$ip->old_ip4} = $tarif;
		}
	} <<_;
	SELECT kunde.id, ipkunde.ip6,ipkunde.bits, ipkunde.tarif, ipkunde.dest
	FROM   ipkunde, kunde
	WHERE  $tmc AND ipkunde.flags & ( 1 << $no_accounting ) = 0
_
}

sub read_domainkunde() {
	my $self = shift;
	my $tmc = $self->{'tmcmd'}; $tmc =~ s/DEST/domainkunde/g;
	$0=$self->{'zb'}." domainkunde";
	DoSelect {
		my ($domid,$cust,$domain) = @_;
		$domain =~ tr/A-Z/a-z/;
		$self->{'hash_dom'}{$domain} = $cust;
	} "select domainkunde.id,kunde.id,domainkunde.domain from domainkunde,kunde where $tmc";
}

sub read_uucpkunde() {
	my $self = shift;
	$0=$self->{'zb'}." uucpkunde";
	DoSelect {
		my ($cust,$domain) = @_;
		$self->{'hash_uucp'}{$domain} = $cust
			unless exists $self->{'hash_uucp'}{$domain};
	} "select kunde,name from uucpkunde";
}

# IP-Ziele, d.h. externe IP-Adressen die nicht "welt" sind, stehen in
# gzip-komprimierten externen Listen (Format: 10.2.3.4/30) unter
# $CACHEDIR/net.TYP.YYYYMM. 
sub read_ip_sub($) {
	my $self = shift;
	my $line = 0;
	my($dom) = @_;

	my $de=$self->{'hash_dom'}{$dom} or fehler("No DOMAIN $dom");
	my $yyyymm = do {
		my @localtime = localtime $self->{timestamp};
		sprintf '%d%02d', $localtime[5]+1900, $localtime[4]+1;
	  };
	my $cf = "$CACHEDIR/net.$dom.$yyyymm";
	my $did = 0;

	$0=$self->{'zb'}." ip_$dom $yyyymm";
	# TODO: folgenden Abschnitt vereinfachen, vgl. RT#272938:
	if(! -f $cf or -s $cf < 30) {
		if(in_test() > 2) {
			fehler "keine Daten für '$dom' bekannt",$cf unless -f $cf;
			return 1; ## leer
		}
		fehler "No cache: $cf"; # OTRS#10027149
	}

	my $gz = gzopen($cf,"r") or fehler "No cache '$cf'";
	my $data;
	while($gz->gzreadline($data)) {
		chop $data;
		$0=$self->{'zb'}." $line" if !($line % 100);
		$line++;
		my $ip = Dbase::IP->new($data);
		next unless ref $ip;
		my $mask = $ip->db_bits;
		my $ipa = $ip->addr;
		unless(exists $self->{'hash_ip'}[$mask]{$ipa}) {
			$self->{'hash_ip'}[$mask]{$ipa} = $de;
			$self->{'hash_tarif'}[$mask]{$ipa} = $self->{'idienst'};
			if ($ip->is_v4) {
				$self->{'hash_ip4'}[$mask]{$ip->old_ip4} = $de;
				$self->{'hash_tarif4'}[$mask]{$ip->old_ip4} = $self->{'idienst'};
			}
		}
		$did++;
	}
	$gz->gzclose;
	fehler "No COUNTRY found ($dom $yyyymm)" unless $did;
	$line;
}

sub read_services() {
	my $self = shift;
	if (open(FOO,"/etc/services")) {
		while(<FOO>) {
			chop;
			s/#.*//;
			s/\s+$//;
			next if /^$/;
			my ($name,$nrprot,$rest) = split(/\s+/,$_,3);
			my ($nr,$prot) = split(/\//,$nrprot,2);
			$self->{'service'}{$prot,$name} = $nr;

		}
		close(FOO);
	}
}


### tested_name -> class
sub who_dom {
	my $self = shift;
	my($i_name,$i_skipip) = @_;
	my($i_result);
	my($i_domain);
	$i_name =~ tr/A-Z/a-z/;
	return $i_result if defined($i_result = $self->{'hash_dom'}{$i_name});
	{
		my($i_domain) = $i_name;
		while($i_domain =~ s/^[^\.]*(\.|$)//) {
			last if($i_domain eq "");
			if(defined($i_result = $self->{'hash_dom'}{"*.".$i_domain}) or 
		   	defined($i_result = $self->{'hash_dom'}{$i_domain})) {
				$self->{'hash_dom'}{$i_name} = $i_result;
				return $i_result;
			}
			next;
		}
	}
	unless($i_skipip) {
		$0 = $self->{'zb'}." lookup $i_name          ";
		print STDERR "  ".substr($0,0,75)."\r" if $self->{'verbose'};

		my($hname,$haliases,$haddrtype,$hlength,@haddrs) =
			gethostbyname($i_name);
		if($#haddrs >= 0) {
			$i_result = $self->{'hash_dom'}{$i_name} = $self->who_ip4(inet_ntoa($haddrs[0]));
			return $i_result if $i_result;
		}
	}
	$i_result = $self->{'hash_dom'}{"*"};
	$self->{'hash_dom'}{$i_name} = $i_result;
	$i_result;
}


### tested_ipaddr -> Kunde(numerisch) oder Ziel(char)
sub who_ip {
	my $self = shift;
	my($ip) = @_;
	my $ret = 0;
	my $net = $ip;
	my $hip = $self->{'hash_ip'};

	$ip = $ip->str if ref $ip;

	if(exists($self->{'hash_ip_'}{$ip})) {
		$ret = $self->{'hash_ip_'}{$ip};
		return (wantarray ? ($ret,$self->{'hash_tarif_'}{$ip}) : $ret);
	}

	$net = Dbase::IP->new($net) unless ref $net;
	my $snet = $net->str;

	while(1) {
		my $bits = $net->db_bits;
		my $mnet = $net->bitmask(0)->addr;
		if(exists($hip->[$bits]{$mnet})) {
			$ret = $hip->[$bits]{$mnet};
			$self->{'hash_ip_'}{$ip} = $ret;
			$self->{'hash_tarif_'}{$ip} = $self->{'hash_tarif'}[$bits]{$mnet};
			return(wantarray ? ($ret,$self->{'hash_tarif_'}{$ip}) : $ret);
		}
		last if $bits == $net->bits;
		$net = $net->bitmask(++$bits,1);
    }

	# return (wantarray ? ($ret,$self->{'idienst'}) : $ret) if defined($ret = $self->{'hash_ip_'}{$net});
	$ret = $self->{'hash_dom'}{"*"};

	$self->{'hash_ip_'}{$ip} = $ret;
	$self->{'hash_tarif_'}{$ip} = $self->{'idienst'};
	wantarray ? ($ret,$self->{'idienst'}) : $ret;
}


# wie who_ip, aber nur IPv4-Adressen; "alter" Code mit Bit-Arithmetik,
# aus Geschwindigkeitsgründen
sub who_ip4 {
	my $self = shift;
	my($ip) = @_;
	my $ret = 0;
	my $net = $ip;

	if(exists($self->{'hash_ip_'}{$net})) {
		$ret = $self->{'hash_ip_'}{$net};
		return (wantarray ? ($ret,$self->{'hash_tarif_'}{$net}) : $ret);
	}

	my $hip = $self->{'hash_ip4'};

	if($net =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/) {
		my($count,$ip,$mask);

		$ip = ($1 << 24)|($2 << 16)|($3 << 8)|$4;
		$mask = 0xFFFFFFFF;
		foreach $count(0..32) {
			if(exists($hip->[$count]{$ip & $mask})) {
				$ret = $hip->[$count]{$ip & $mask};
				$self->{'hash_ip_'}{$net} = $ret;
				$self->{'hash_tarif_'}{$net} = $self->{'hash_tarif4'}[$count]{$ip & $mask};
				return(wantarray ? ($ret,$self->{'hash_tarif_'}{$net}) : $ret);
			}
			$mask <<= 1;
		}
		
    }

	# return (wantarray ? ($ret,$self->{'idienst'}) : $ret) if defined($ret = $self->{'hash_ip_'}{$net});
	$ret = $self->{'hash_dom'}{"*"};
	$self->{'hash_ip_'}{$net} = $ret;
	$self->{'hash_tarif_'}{$net} = $self->{'idienst'};
	wantarray ? ($ret,$self->{'idienst'}) : $ret;
}


sub line {
	my $self = shift;
	my($data,$len) = @_;
	if(!($self->{'line'} % 1000)) {
		$0 = $self->{'zb'}="acct $self->{'jj'}-$self->{'mm'}-$self->{'tt'} $self->{'dienst'}/$self->{'quelle'} $self->{'line'}";
		print STDERR "  ".substr($0,0,75)."\r" if $self->{'verbose'};
	}
	$self->{'line'}++;
	if($len) {
		0;
	} elsif ($data =~ /^\#/) {
		1;
	} else {
		$self->{'this'} = $data;
		0;
	}
}

sub xprint($@) {
	my($fh,@args) = @_;
	if((ref $fh) =~ /^Compress::Zlib/) {
		my $buf = join("",@args);
		$fh->gzwrite($buf) or fehler "gzwrite('$buf'): ".$fh->gzerror()."\n" if length $buf;
	} else {
		print $fh @args;
	}
}

sub kunde2ziel {
	my($self,$rem) = @_;
	if (defined $rem and $rem =~ /^(?:\d+ )?>(\d+)/) {
		$rem = $1;
	} else {
		$rem = find_descr("ziel",$DEFAULTKLASSE);
	}
	$rem;
}

sub acct {
	# $acct->acct($kunde,$kdienst, $rem,$rdienst, $numpack,$volpack);
	my $self = shift;
	my($kunde,$kdienst, $rem,$rdienst, $numpack,$volpack, $line) = @_;
	if(defined $line) {
		$line .= "\n" if $line !~ /\n\z/;
	} else {
		$line = $self->{'this'};
	}

	$kunde = $self->{'kunde'} unless defined $kunde;
	$kdienst = $self->{'idienst'} unless defined $kdienst;
	$rdienst = $self->{'idienst'} unless defined $rdienst;

	no warnings 'numeric';
    if(defined $rem and $rem > 0 and $kunde == 0 and $self->{'do_reverse'}) {
        ( $kunde, $rem, $kdienst, $rdienst ) = ( $rem, $kunde, $rdienst, $kdienst );
    } elsif($kunde == 0) {
		print $_ if defined $_;
		return;
    } elsif (not defined $rem or $rem eq "") {
        $rem = undef;
    } elsif(defined $rem and $rem > 0) {
		if($self->{'do_reverse'} and ($rem != $kunde or $self->{'do_same_customer'} )) {
			$self->{'do_reverse'}=0;
			$self->acct($rem,$rdienst, $kunde,$kdienst, $numpack,$volpack, $line);
			$self->{'do_reverse'}=1;
		}
	}
	$kunde = 0+$kunde;
	return if $self->{'kunde'} and $kunde != $self->{'kunde'};
	$rem = $self->kunde2ziel($rem);

	if($self->{'do_logging'} and $self->{'logdir'}) {
		my $log = $self->{'logfile'}{$kunde};
		unless($log) {
			my $fn  = $self->gen_filename($kunde);
			my $fh;
			if($self->{'zip'}) {
				$fn .= ".gz";
				$fh = gzopen("$fn.x","w");
			} else {
				$fh = new IO::File("$fn.x",O_WRONLY|O_CREAT|O_TRUNC);
			}
			utf8modus($fh);
			fehler "No file to write: $!\n" unless $fn;
			$self->{'dofiles'}{$fn}++;
			$self->{'logfile'}{$kunde} = $log = $fh;
			xprint($log,"# $self->{'jjmm'}.$self->{'tt'}\n");
		}
		xprint($log,$line);
	}

	$rem = '?' if $rem eq ''; # unbekannt
	$volpack = 0 unless $volpack =~ /^\d+$/;

	print "# dienst=$kdienst kunde=$kunde remote=$rem num=$numpack vol=$volpack\n" if $ENV{'TESTING4'};

	$self->{'pkt'}{$kdienst}{$kunde}{$rem} += $numpack;
	my($bytes) = \$self->{'bytes'}{$kdienst}{$kunde}{$rem};
	if(defined $$bytes) {
		$$bytes = bignum($$bytes) if not ref $$bytes and length $$bytes > 7;
		$$bytes += $volpack;
	} else {
		$$bytes = bignum($volpack);
	}

}

sub finish {
	my $self = shift;
	print STDERR "  Closing files\r" if $self->{'verbose'};
	foreach my $log (values %{$self->{'logfile'}}) {
		if((ref $log) =~ /^Compress::Zlib/) {
			$log->gzclose;
		} else {
			close $log;
		}
	}
	$self->{'logfile'} = {};
	$self->database();

	print STDERR "  Rename/Copy files       \r" if $self->{'verbose'};
	foreach my $file (keys %{$self->{'dofiles'}}) {
		$0="acct copy $file";
		next unless -s "$file.x";
		if(-s "$file") {
		    print STDERR "  Copy file: $file       \r" if $self->{'verbose'};
			open(IN,"$file.x") or fehler "No file read '$file.x': $!\n";
			open(OUT,">>$file") or fehler "No file write '$file': $!\n";
			binmodus(\*IN);
			binmodus(\*OUT);
			while(<IN>) {
				print OUT or fehler "No file print '$file': $!\n";
			}
			close(IN) or fehler "No file close '$file.x': $!\n";
			close(OUT) or fehler "No file close '$file': $!\n";
			unlink("$file.x") or fehler "No file remove '$file.x': $!\n";
		} else {
			rename("$file.x",$file) or fehler "No rename '$file': $!\n";
		}
	}
	$self->{'dofiles'} = {};
}

sub database {
	my $self = shift;
	my($cust,$sub,$key,$bytes,$pkt,$hibytes,$res,$dien,$suba,$sdien,$ndien,$sub2);
	foreach $dien (keys %{$self->{'bytes'}}) {
		$suba = $self->{'bytes'}{$dien};
		foreach $cust (keys %$suba) {
			$sub = $suba->{$cust};
			$sub2 = $self->{'pkt'}{$dien}{$cust};

			if($self->{'do_tarife'}) {
				my $etime = 0;
				my $nid = DoFn("select id from tarifkunde where kunde = $cust and dienst = $dien and $self->{'tmcmd1'} order by beginn desc limit 1");
				if(not $nid and ($nid = DoFn("select id from tarifkunde where kunde = $cust and dienst = $dien and beginn > $self->{'timestamp'} and (ende is NULL or ende > $self->{'timestamp'}) and anzahl = 0 order by beginn limit 1"))) {
					# Wenn es einen Pseudotarif in der Zukunft gibt,
					# lasse ihn eher beginnen. Siehe auch #10051463.
					Do("update tarifkunde set beginn = $self->{'timestamp'} where id = $nid");
				} elsif($nid) {
					Do("update tarifkunde set notiz = $self->{'timestamp'} where id = $nid and notiz < $self->{'timestamp'}");
				} else { ### kein Pseudotarif => lege einen an.
					my $ndien = $dien;
					my $tnid;
					my $gend = find_dienst "general";
					while(1) {
						($tnid,$etime) = DoFn("select tarifname,ende from tarifkunde where kunde = $cust and dienst = $ndien and $self->{'tmcmd1'} order by beginn desc limit 1");
						last if $tnid or $ndien == $gend;
						$ndien = DoFn("select berechne from tarifeq where dienst = $ndien");
						$ndien = $gend if not defined $ndien;
					}
					unless($tnid) {
						$tnid = DoFn("select id from tarifname where name='unbekannt'") || Do("insert into tarifname set name='unbekannt'");
						$etime="NULL";
					}
					Do("insert now into tarifkunde set kunde=$cust, tarifname=$tnid, dienst=$dien, beginn=$self->{'timestamp'}, ende=".($etime||"NULL").", notiz=$self->{'timestamp'}");
					# XXX TODO ist das so ???
					# Do("update kunde set geaendert = UNIX_TIMESTAMP(NOW()) where id = $cust");
				}
			}
			foreach $key(keys %$sub) {
				$bytes = $sub->{$key};
				$pkt = $sub2->{$key};
				$0="acct $self->{'jj'}-$self->{'mm'}-$self->{'tt'} $self->{'dienst'}/$self->{'quelle'} set $cust.$key";
				print STDERR "  Set $cust $dien $key $bytes,$pkt         \r" if $self->{'verbose'};
				add_acct($cust,$dien,$key,$self->{'jjmm'},$self->{'tt'},$self->{'iquelle'},$bytes,$pkt);
			}
		}
	}
	$self->{'pkt'} = {};
	$self->{'bytes'} = {};
}

sub new {
	my($name,@opt) = @_;
	my $self = bless { @opt }, $name;

	foreach my $i(qw(timestamp want dienst quelle)) {
		fehler "Es fehlt: '$i'\n" unless defined $self->{$i};
	}

	$0 = $self->{'zb'}="acct init";
	$self->{'line'} = 0;

	$self->{'hash_ip'} = [];
	$self->{'hash_ip4'} = [];
	$self->{'hash_ip_'} = {};
	$self->{'hash_tarif'} = [];
	$self->{'hash_tarif4'} = [];
	$self->{'hash_tarif_'} = {};
	$self->{'hash_dom'} = {};
	$self->{'hash_uucp'} = {};
	$self->{'no_cache'} = 1;

	list_descr("acziel",1,"",sub {
		my(undef,$id,$tag)=@_;
		$self->{'hash_dom'}{$tag} = ">$id";
	});

	$self->{'logfile'} = {};
	$self->{'dofiles'} = {};
	$self->{'pkt'} = {};
	$self->{'bytes'} = {};

	$self->{'idienst'} = find_dienst $self->{'dienst'};
	fehler "Dienst '".$self->{'dienst'}."' nicht bekannt.\n" unless defined $self->{'idienst'};
	$self->{'idienst2'} = find_dienst $self->{'dienst'}."-hit";

	$self->{'iquelle'} = find_descr("quelle",$self->{'quelle'});
	fehler "quelle '".$self->{'quelle'}."' nicht bekannt.\n" unless defined $self->{'iquelle'};

	$self->{'do_reverse'} = 1 unless exists $self->{'do_reverse'};
	$self->{'do_tarife'} = 1 unless exists $self->{'do_tarife'};
	$self->{'do_logging'} = 1 unless exists $self->{'do_logging'};

	($self->{'jj'},$self->{'mm'},$self->{'tt'}) = isodate($self->{'timestamp'});
	$self->{'jjmm'} = $self->{'jj'}*100 + $self->{'mm'};
	$self->{'mm'} = "0".$self->{'mm'} if $self->{'mm'} < 10;
	$self->{'tt'} = "0".$self->{'tt'} if $self->{'tt'} < 10;

	unless($self->{'no_lock'}) {
		my $lock;
		if( exists DO_PARALLEL->{ lc $self->{dienst} } ) {
			$lock = "Acct.$self->{'idienst'}.$self->{'iquelle'}.$self->{'jj'}-$self->{'mm'}-$self->{'tt'}";
		} else {
			$lock = "Acct.$self->{'idienst'}";
		}
		$self->{'Lock'} = new File::ShLock(name => $lock, basedir => ($ENV{'TESTING2'} ? "/tmp" : "/var/lock/kunde"));
		fehler "Busy" unless ref $self->{'Lock'};
	}

	$0=$self->{'zb'}="acct $self->{'dienst'}/$self->{'quelle'} init $self->{'jj'}-$self->{'mm'}-$self->{'tt'}";

	my $time2 = $self->{'timestamp'} - (24*3600-3);
	$self->{'tmcmd1'} = "beginn <= $self->{'timestamp'} and (ende is NULL or ende > $time2)";
	$self->{'tmcmd'} = "kunde.id = DEST.kunde
		and  DEST.beginn <= $self->{'timestamp'} and ( DEST.ende is NULL or DEST.ende > $time2 )";

	$self->{'want'} = [ split(/,\s*/,$self->{'want'}) ] unless ref $self->{'want'};

	my $old_want = $acctcache{'want'};
	if(defined $old_want and $old_want eq join(",",@{$self->{'want'}})) {
		my $nt = isodate $self->{'timestamp'};
		my $ot = isodate $acctcache{'timestamp'};
		if ($ot eq $nt) {
			while(my($k,$v) = each %acctcache) {
				next unless $k =~/^hash_/;
				$self->{$k} = $v;
			}
			return $self;
		}
	}
	foreach my $req(@{$self->{'want'}}) {
		my $def = undef;
		$def = $self->read_kunde if $req eq "kunde";
		$def = $self->read_uucpkunde if $req eq "uucp" or $req eq "uucpkunde";
		$def = $self->read_perskunde if $req eq "pers" or $req eq "person" or $req eq "perskunde";
		$def = $self->read_domainkunde if $req eq "domain" or $req eq "domainkunde";
		$def = $self->read_ipkunde if $req eq "ip" or $req eq "ipkunde";
		$def = $self->read_ip_sub($1) if $req =~ /^ip_(.+)/;
		fehler "Unbekannter Key: '$req'\n" unless defined $def;
	}

	$self;
}

sub DESTROY {
	my $self = shift;

	foreach my $fh (values %{$self->{'logfile'}}) {
		if((ref $fh) =~ /^Compress::Zlib/) {
			$fh->gzclose;
		} else {
			close $fh;
		}
	}
	foreach my $file (keys %{$self->{'dofiles'}}) {
		unlink "$file.x";
	}
	return if $self->{'no_cache'};
	%acctcache = (
		want => ref $self->{want}
		? join( ',', @{ $self->{want} } )
		: $self->{want},
		timestamp => $self->{timestamp}
	);
	while(my ($k,$v) = each %$self) {
		next unless $k =~ /^hash_/;
		$acctcache{$k} = $v;
	}
}

1;
