#!/usr/bin/perl

=head1 NAME

snmp-collect.pl

Collects SNMP data from routers and switches and writes data to RRD files. The
script reads a simple config file which defines all switches and routers to be
queried. It then tries to get as much information as needed from these devices
and writes it to one RRD file per interface. The default locations are
/etc/rrdhosts for the config file and /var/rrd for the RRD files.

=head1 USAGE

There are no command line arguments. However, you can set the ``DEBUG''
environment variable to enable debugging.

=cut

use strict;
use warnings;

use Date::Format qw(time2str);
use DBI;
use Fcntl qw(LOCK_EX LOCK_SH SEEK_SET);
use Getopt::Long;
use POSIX qw(:sys_wait_h);    # for waitpid
use Regexp::Common qw(balanced);
use RRDs;
use SNMP;
use Getopt::Long;
use Net::SNMP;

#	Version 1.0 Thomas Gericke
#		- initial Version
#	2001-12-21 Version 1.1 Ingo Kraupa
#		- Verwendung eines externes Host-File /etc/rddhosts
#		- Support fuer eine Community pro Host
#		- Support fuer Bintec-Router
#	2001-12-28 Version 1.2 Ingo Kraupa
#		- SNMP Timeouts verursachen keinen Haenger mehr
#		- Debug-Ausgabe gekuerzt, $output=1 liefert sinnvoll
#		  verwertbaren Output
#		- $Num-Bug repariert
#	2002-02-07 Version 1.3 Thomas Gericke
#		- applied no-update-patch for interfaces without
#		  traffic (no update into database if octets dit
#		  _not_ change)
#		- debug-support fixed
#	2002-02-08 Version 1.4 Thomas Gericke
#		- fixed no-update-patch
#	2003-07-31 Version 1.5 Ingo Kraupa
#		- Support fr Maximalwerte (speed) im Zusammenhang mit
#		  dem Tool "rrdcheckrange" fr saubere Statistiken
#	2003-09-10 Version 1.5.1 Ingo Kraupa
#		- Hack fr DSL-Leitungen mit zu groer Speed
#	2003-09-15 Version 1.5.2 Ingo Kraupa
#		- Hack fr Catalyst-Switche mit seltsamer ifDescr
#	2004-08-13 Version 2 Florian Forster
#		- Almost complete rewrite
#	2004-09-01 Version 2.1 Florian Forster
#		- Statt SNMP::Util + snmpwalk's werden SNMP, RRDs und
#		  snmpget's verwendet..
#	2004-09-22 Version 2.2 Florian Forster
#		- Das Script forkt und fragt mehrere Router (ca. 10)
#		  gleichzeitig ab. Dadurch beeintraechtigen laengere
#		  Antwortzeiten das Laufzeitverhalten nicht mehr so
#		  nachhaeltig.
#	2005-01-21 Version 2.2.1 Florian Forster
#		- Sleep-1 nach dem flock um Fehlermeldungen von rrdtool zu
#		  vermeiden.. (RT#187875)
#	2010-09-22 Version 2.2.2 Florian Forster
#		- Sonderbehandlungen fuer die Kunden "noris" und "norisbank"
#		  eingebaut. (OTRS#10245109)
#	2012-07-13 Version 2.2.3 Stefan Pfab
#		- automagisches Abfragen von Juniper VSYS eingebaut
#		  (Ticket 12199189)
#	2013-06-06 Version 2.3.0 Stelios Gikas
#		- Kunden Aliase sollen automatisch greifen.
#		  Es gibt ein Kunden Mapping File das eingelesen wird
#		  (Ticket 14474473)
# 2015-03-31 Version 2.4.0 Klaus Franken
#   - automagisches Abfragen von Juniper VSYS ausgebaut
#   - Umstellung auf Net::SNMP.pm statt SNMP.pm fuer ipv6

my $time = localtime;
# Konfigurierbare Variablen
GetOptions(
    'datadir=s'         => \(our $DataDir         = '/var/lib/rrd/traffic/'),
    'configfile=s'      => \(our $Config          = 'rrdhosts'),
    'customermapping=s' => \(our $CustomerMapping = '/var/lib/snmp/customer_mapping'),
    'logfile=s'         => \(our $LogFile         = '/var/log/snmp-collect.log'),
    'dry-run!'          => \(our $DryRun          = 0),
);


our $dbase   = 'techinfosnmpcollect';
our $dbuser  = 'snmpcollect';
our $dbpass  = 'Ahque4to';
our $dbhost  = 'db.noris.net';
our $dbtable = 'interfaces';

# ACHTUNG: Pfad MUSS mit einem Slash enden!
#our $DataDir = '/var/rrd/2004-10-19/';
our $Debug            = defined( $ENV{'DEBUG'} ) ? $ENV{'DEBUG'} : 0;
our $DebugLog         = defined( $ENV{'DEBUGLOG'} ) ? $ENV{'DEBUGLOG'} : 0;
our $ConcurrencyLevel = 25;
our $DefaultCustomer  = 'noris';
our $FlockTimeout     = 20;

our @AdminStatus = qw(unknown up down testing);

# Irgendwie muessen wir dieses directory zu den mibs angeben
# ... ein paar weitere Variablen geben wir auch noch an
#$ENV{'MIBDIRS'}       = '/var/lib/mibs/ietf;/var/lib/mibs/iana';
$ENV{'MAX_LOG_LEVEL'} = $Debug ? 'debug' : 'none';

&SNMP::addMibDirs('/var/lib/mibs/ietf', '/var/lib/mibs/iana');
&SNMP::loadModules('NETSCREEN-INTERFACE-MIB');


our $TimeStamp = time2str( '%Y%m%d%H%M%S', time );    # pro Durchlauf identisch

write_to_log ("--- BEGIN ---\n");

GetOptions(
    "mapping|m=s"     => \$CustomerMapping,
    "concurrency|c=i" => \$ConcurrencyLevel,
    "debug|v+"        => \$Debug,
    "debuglog+"       => \$DebugLog,
    "datadir|d=s"     => \$DataDir,
    "logfile|l=s"     => \$LogFile,
    "config|f=s"      => \$Config,
    "dbase=s"         => \$dbase,
    "dbuser=s"        => \$dbuser,
    "dbpass=s"        => \$dbpass,
    "dbhost=s"        => \$dbhost,
    "dbtable=s"       => \$dbtable,
    'customer=s'      => \$DefaultCustomer,
    "help|h|?"        => sub {
        print <<"EOF";
Usage: $0 [options]

Valid options are:
    --debug             -v              Print debug messages.
    --datadir <dir>     -d <dir>        Sets the directory where to store the RRD files.
    --logfile <file>    -l <file>       Sets the logfile to use
                                        (default: /var/log/snmp-collect.log)
    --config <file>     -f <file>       Sets the config file to read.
    --mapping <file>    -m <file>       Sets the mappings for Customer
                                        (default: /var/lib/snmp/customer_mapping)
    --concurrency <num> -c <num>        The number of processes to start.
    FIXME: Document the rest of the options
EOF
        exit(0);
    }
);

my %CustomerMappings;
if ( $CustomerMapping && -f $CustomerMapping ) {

    open( CUSTOMER, '<', $CustomerMapping ) or die "open ($CustomerMapping): $!";
    while (<CUSTOMER>) {
        chomp;
        s/#.*//;
        next unless ($_);
        my ( $kunde, $aliase ) = split( /\s+/, $_ );
        for my $alias ( split( /,/, $aliase ) ) {
            $CustomerMappings{$alias} = $kunde
              if $alias ne $kunde;
        }
    }
    close(CUSTOMER) or die "close ($CustomerMapping): $!";
}

open (HOSTS, '<', $Config) or die ("open ($Config): $!");
flock (HOSTS, LOCK_SH) or die ("flock ($Config): $!");

sub sigchld_handler {
    while ( waitpid( -1, WNOHANG ) > 0 ) {
        $ConcurrencyLevel++;
    }
    $SIG{'CHLD'} = \&sigchld_handler;
}
$SIG{'CHLD'} = \&sigchld_handler;

while (<HOSTS>) {
    chomp;
    s/#.*//;
    next unless ($_);

    my @data;
    my ( $host, $comm, $ip ) = split( /\s+/, $_ );
    my $num_if;
    my $i = 0;
    my $data = {};
    my @interfaces = ();
    my $result; # for SNMP-result

    next unless ( defined($host) and defined($comm) );

    $host = lc ($host);

    # Ugly spin lock. I don't think I can wait on signals in perl, so this
    # will have to do for now..
    while ( $ConcurrencyLevel <= 0 ) {
        sleep(1);
    }

    my $pid = fork();

    if ( !defined($pid) ) {
        die("fork() failed: $!");
    }
    elsif ( $pid != 0 ) {

	write_to_log ("forked new child with PID $pid to handle host `$host'\n");

        # XXX race condition possible! ConcurrencyLevel is not locked!
        $ConcurrencyLevel--;
        next;
    }

    print "Host: $host $ip (in $comm)\n" if ($Debug);

    # wir legen erstmal fuer jeden Router ein File an, worin wir eine
    # Uebersicht der Interfaces speichern. Diese Datei dient auch der
    # serialisierung verschiedener Sammel-Prozesse.
    if ( !-d "$DataDir/$host" ) {
        mkdir("$DataDir/$host") or die("mkdir $DataDir/$host: $!");
    }
    open( HOSTFILE, ">> $DataDir/$host/interfaces" )
      or die("open $DataDir/$host/interfaces : $!");
    {
	$SIG{'ALRM'} = sub { die ("flock ($DataDir/$host/interfaces) timed out"); };
	alarm ($FlockTimeout);
	if ( !flock( HOSTFILE, LOCK_EX ) ) {
	    print STDERR "Unable to get lock on ``$DataDir/$host/interfaces''\n";
	    exit (1);
	}
	alarm (0);
	$SIG{'ALRM'} = 'DEFAULT';
    }
    truncate( HOSTFILE, 0 );
    seek( HOSTFILE, 0, SEEK_SET );

    my $session;
    my $error;

    my $domain = 'udp/ipv4';
    if ($ip =~ m/:/) { # FIXME: ist das ein gutes Kriterium?
        $domain = 'udp/ipv6';
        print "switching to ipv6\n" if ($Debug);
    }
    ($session, $error) = Net::SNMP->session(
        -hostname  => $ip || $host,
        -community => $comm,
        -domain    => $domain,
        -version   => 2,
    );
    if (! $session) {
        print STDERR "Error: cannot connect to host $host: $error\n";
        exit (1);
    }

    my $ifNumber='1.3.6.1.2.1.2.1';
    #my $ifNumber='ifNumber';
    $result = $session->get_request(-varbindlist => [$ifNumber.'.0']);
    $num_if = $$result{$ifNumber.'.0'};
    if ( !defined($num_if) or !$num_if ) {
        print STDERR "Unable to get number of interfaces from $host\n";
        exit (1);
    }
    if ($Debug) {
        use Data::Dumper;
	print "ifNumber: " . Dumper($result);
    }

    # IfIndex fr Juniper Firewalls
    # die Werte in der Juniper-spezififschen MIB sind um eins neben dem default ifIndex...
    # deswegen: fr Juniper Netscreens diesen Index etc. verwenden, sonst wie gehabt.

    # for Juniper Firewalls: .1.3.6.1.4.1.3224.9.3.1.3 inBytes; .1.3.6.1.4.1.3224.9.3.1.5 outBytes
    ##my @fields = (qw(ifDescr ifType ifSpeed ifAdminStatus ifInOctets ifOutOctets
           ##ifName ifHCInOctets ifHCOutOctets ifAlias nsIfDescr nsIfName .1.3.6.1.4.1.3224.9.3.1.3 .1.3.6.1.4.1.3224.9.3.1.5));
    # ohne Juniper Besoderheiten:
    my @fields = (qw(ifDescr ifType ifSpeed ifAdminStatus ifInOctets ifOutOctets
           ifName ifHCInOctets ifHCOutOctets ifAlias));

    # manuell ermittelt, z.b: snmpwalk -On -c wirepubcis -v 2c 172.17.17.151 ifDescr.4: 
    my %fields_oids = (
        ifDescr                     => '.1.3.6.1.2.1.2.2.1.2',
        ifType                      => '.1.3.6.1.2.1.2.2.1.3',
        ifSpeed                     => '.1.3.6.1.2.1.2.2.1.5',
        ifAdminStatus               => '.1.3.6.1.2.1.2.2.1.7',
        ifInOctets                  => '.1.3.6.1.2.1.2.2.1.10',
        ifOutOctets                 => '.1.3.6.1.2.1.2.2.1.16',
        ifName                      => '.1.3.6.1.2.1.31.1.1.1.1',
        ifHCInOctets                => '.1.3.6.1.2.1.31.1.1.1.6',
        ifHCOutOctets               => '.1.3.6.1.2.1.31.1.1.1.10',
        ifAlias                     => '.1.3.6.1.2.1.31.1.1.1.18',
        nsIfDescr                   => 'FIXME',
        nsIfName                    => 'FIXME',
        '.1.3.6.1.4.1.3224.9.3.1.3' => '.1.3.6.1.4.1.3224.9.3.1.3',
        '.1.3.6.1.4.1.3224.9.3.1.5' => '.1.3.6.1.4.1.3224.9.3.1.5',
    );

    my $ifindex_vars;

    $result = $session->get_request(-varbindlist => ['.1.3.6.1.4.1.3224.9.1.1.1.0']);
    my $num_juniper = $$result{'.1.3.6.1.4.1.3224.9.1.1.1.0'};
    if ( 0 and defined($num_juniper) and $num_juniper ne 'NOSUCHOBJECT' ) {
        # FIXME: solche Junipers gibt es nicht mehr
        warn ("num_juniper: $num_juniper");

            $ifindex_vars = SNMP::Varbind->new (['nsIfIndex', 0]);

            while ($session->getnext ($ifindex_vars))
            {
                printf "\%s.\%i = \%s\n", $ifindex_vars->[0], $ifindex_vars->[1], '' . $ifindex_vars->[2]
                   if ($Debug);
                last if ($ifindex_vars->[0] ne 'nsIfIndex');
                push (@interfaces, $ifindex_vars->[2]);
                $data->{$ifindex_vars->[2]} = {nsIfIndex => $ifindex_vars->[2]};
            }

    } else {
        my $ifIndex='1.3.6.1.2.1.2.2.1.1';
        my $Interfaces = $session->get_table( Baseoid => $ifIndex);
        if ($Debug) {
		use Data::Dumper;
		print "Interfaces: " . Dumper($Interfaces);
	}
        for my $index (keys($Interfaces)) {
            $index =~ m/\.(\d+)$/;
            my $interface = $1;
            my $interface2 = $$Interfaces{$index};
            printf "\%s.\%i = \%s\n", 'ifIndex', $interface, '' . $interface2
                if ($Debug);
            push (@interfaces, $interface2);
            $data->{$interface2} = {ifIndex => $interface2};
        }
    }
    for (my $f = 0; $f < @fields; $f++) {
	for (my $i = 0; $i < @interfaces; $i++) {
            my $oid = $fields_oids{$fields[$f]} . '.' .$interfaces[$i];
            $result = $session->get_request(-varbindlist => [$oid]);
            my $d = $$result{$oid};
	    next if (!defined ($d));
            print "field $f: $oid => $d, " . $interfaces[$i] . "," . $fields[$f] . "\n";
            next if ("$d" eq 'NOSUCHINSTANCE');
            next if ("$d" eq 'noSuchInstance');
            next if ("$d" eq 'noSuchObject');
            next if ("$d" eq 'NOSUCHOBJECT');
            $data->{$interfaces[$i]}{$fields[$f]} = $d;
        }
    }

    # First, iterate over all interfaces and write RRD files.
    # Hack fr Juniper Netscreen VSYS: 
    # es gibt dort keinen ifSpeed und ifOctets, aber unter zwei anderen OIDs
    # ifOctets fr die VSYS-Interfaces, .1.3.6.1.4.1.3224.9.3.1.3 und .1.3.6.1.4.1.3224.9.3.1.5
    # => wenn kein ifSpeed und ifOctects vorhanden sind, aber unter den anderen OIDs Werte kommen ist das ok.
    INTERFACE: for (@interfaces)
    {
        my $if = $_;

        if ( (!defined( $data->{$if}{'ifIndex'} ) and !defined( $data->{$if}{'nsIfIndex'}))
            or !defined( $data->{$if}{'ifAdminStatus'} )
            or (!defined( $data->{$if}{'ifDescr'} ) and !defined( $data->{$if}{'nsIfDescr'}))
	    or (!defined( $data->{$if}{'ifSpeed'} ) and !defined( $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.3'} ) and !defined( $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.5'} ))
            or (!defined( $data->{$if}{'ifInOctets'} ) and !defined( $data->{$if}{'ifOutOctets'} ) and !defined( $data->{$if}{'ifType'} ))  )
        {
	    if ($Debug)
	    {
		print "\tInterface ``$if'' is missing fields: "
		. join (', ',
			grep { !defined ($data->{$if}{$_}) } (qw(ifIndex ifAdminStatus ifDescr ifSpeed ifType ifInOctets ifOutOctets)))
		. ". Skipping.\n";
	    }

	    delete ($data->{$if});
            next INTERFACE;
        }

        if (defined($data->{$if}{'nsIfName'})){
                $data->{$if}{'ifName'} = $data->{$if}{'nsIfName'}
        } else {
                $data->{$if}{'ifName'} = $if if (!defined ($data->{$if}{'ifName'}));
        }

        my $index;
        if (defined( $data->{$if}{'nsIfIndex'})) {
                $index  = $data->{$if}{'nsIfIndex'};
                $data->{$if}{'ifIndex'} = $data->{$if}{'nsIfIndex'};
        } else {
                $index  = $data->{$if}{'ifIndex'};

        }
        my $adminstatus = $data->{$if}{'ifAdminStatus'};

        my $descr;
        if (defined( $data->{$if}{'nsIfDescr'})) {	# FIXME: hier kann man aufraeumen
                $descr  = $data->{$if}{'nsIfDescr'};
                $data->{$if}{'ifDescr'} = $data->{$if}{'nsIfDescr'};
        } else {
                $descr  = $data->{$if}{'ifDescr'};

        }

        $data->{$if}{'ifName'} = $if if (!defined ($data->{$if}{'ifName'}));
        $data->{$if}{'ifAlias'} = $descr if (!$data->{$if}{'ifAlias'});

        my $speed       = $data->{$if}{'ifSpeed'};
        my $incoming=0;
        my $outgoing=0;
        my $type = $data->{$if}{'ifType'};
        my $name = $data->{$if}{'ifName'};
        my $alias;
        if (defined $data->{$if}{'nsIfDescr'}) {
                $alias = $data->{$if}{'nsIfDescr'};
        } else {
                $alias = $data->{$if}{'ifAlias'};
        }


        # "... (noris: ...) ..."-Syntax, vgl. RT#471813:
        if ( $alias =~ / \(noris:
                         (
                             (?:
                                 [^(]+
                                 |
                                 $RE{balanced}{-parens=>'()'}
                             )+
                         )
                         \)
                       /x
        ) {
            print "\tReplacing alias [$alias] by [$1].\n" if $Debug;
            $alias = $1;
        }

        my $customer;

        print "\tInterface: $index ($alias)\n" if ($Debug);

        $data->{$if}{'queryType'} = '32bit';

	if (defined ($data->{$if}{'ifHCInOctets'})
			&& defined ($data->{$if}{'ifHCOutOctets'})
			&& (($data->{$if}{'ifHCInOctets'} != 0) || $data->{$if}{'ifHCOutOctets'} != 0))
	{
	    $data->{$if}{'queryType'} = '64bit';
	}

        if (defined($data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.3'}) and defined($data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.5'})) {
		            # FIXME: Juniper Zeug
                $incoming = $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.3'};
                $outgoing = $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.5'};
        } else {
                $incoming = ($data->{$if}{'queryType'} eq '64bit') ? $data->{$if}{'ifHCInOctets'} : $data->{$if}{'ifInOctets'};
                $outgoing = ($data->{$if}{'queryType'} eq '64bit') ? $data->{$if}{'ifHCOutOctets'} : $data->{$if}{'ifOutOctets'};
        }

        $adminstatus = $AdminStatus[ int($adminstatus) ];

        if (($DefaultCustomer eq 'noris')
		&& !( $alias =~ m#^([\w\-\+]+)/(\d+)# ) ) {
            print "\t\tAlias does not look as expected\n" if ($Debug);
	    delete ($data->{$if});
            next INTERFACE;
        }

	# RT#308891: If the speed is unknown for an interface, use 100 GBit/s
	#   just to be save. Also use a higher number for very fast interfaces.
	#   In this case the speed, which is a 32bit value, is set to 2^32-1 =
	#   4294967295. -octo
	if (!$speed || ($speed == 4294967295))
	{
	    $data->{$if}{'ifSpeed'} = $speed = 1.0E11;
	}

        #ik - atm subinterfaces special speed hack: DSL lines are 2.3
        # Mbit max!
	# RT#308891: DS-Lines are 10MBit/s max. -octo
        if ( $type == 49 and $speed == 149760000 )
	#if ($type eq 'aal5' and $speed == 149760000)
        {
            print "\t\t$alias: changed speed to 10000000\n" if ($Debug);
            $data->{$if}{'ifSpeed'} = $speed =  1.0E7;
        }

        #ik - special hack for Catalyst OS 6k Switche that don't have
        # any useful interface descriptions - use ifName instead
        if (   $descr =~ 'ethernet \('
            || $descr =~ /fiber gigabit ethernet/
            || $descr eq 'gigabit ethernet without GBIC installed'
            || $descr eq 'aggregated interface'
            || $descr eq 'Ethernet Interface' )
        {
            my $old_descr = $descr;
            $data->{$if}{'ifDescr'} = $descr = $name || 'undef';
            print "\t\tifDescr changed to ifName: $old_descr -> $descr\n" if ($Debug);
        }

        if ( $alias =~ m#^(Fast|Gigabit)Ethernet\d+/# ) {
            print "\t\tIgnoring generic named interface\n" if ($Debug);
	    delete ($data->{$if});
            next INTERFACE;
        }

        if ( $type == 134 ) {
            print "\t\tIgnoring interface: Type is ATM Subinterface\n"
              if ($Debug);
	    delete ($data->{$if});
            next INTERFACE;
        }

        # das alles schreiben wir nun in das File rein
        print HOSTFILE "$index $adminstatus '$descr' $speed $type '$alias' $incoming $outgoing\n";

	$customer = $DefaultCustomer;
	if ($DefaultCustomer eq 'noris')
	{
	    my $customer_num = -1;

	    # hier filtern wir aus der Interface-Description den Kundennamen raus
	    if ($alias =~ m#([^\s/]+)(?:/([1-9][0-9]*))?#)
	    {
		$customer = $1;
		if ($2)
		{
		    $customer_num = 0 + $2;
		}
	    }

	    # Spezialbehandlung fuer Kunden-Aliase
	    if ( exists $CustomerMappings{$customer} ) {
		print "CustomerMapping: $customer -> $CustomerMappings{$customer}\n"
		  if ($Debug);
		$customer = $CustomerMappings{$customer};
	    }

	}
	$data->{$if}{'customer'} = $customer;
        print "\t\tParsed customer: `$customer'\n" if ($Debug);

        # wir schmeissen die Slashes, Leerzeichen und Doppelpunkte aus den
        # Interfaces raus, denn das stoert beim speichern
        $descr =~ s#[/ :]+#_#g;
	$data->{$if}{'ifDescr'} = $descr;

        if ($Debug) {
            open( CUSTFILE, ">> $DataDir$customer.cust" )
              or die("open $DataDir$customer.cust: $!");
            print CUSTFILE "$host/$descr $alias\n";
            close(CUSTFILE);
        }

	$data->{$if}{'RRDFile'} = "$DataDir$host/$descr.rrd";

        # falls wir Daten auf den Interfaces haben...
        if ($incoming ne 'undef' || $outgoing ne 'undef')
	{
            # jetzt gucken wir, ob es schon eine rrd gibt
            if ( !-f $data->{$if}{'RRDFile'})
	    {
                my $bps = ($speed / 8) || 'U';

                if ($Debug) {
                    print "\t\tCreating " . $data->{$if}{'RRDFile'} . " (with $bps Bytes/s)\n";
                }
                else {
			$time = localtime;
                    print "$time: Creating graphs for new interface ``$descr'' on ``$host''\n";
                }

				unless ($DryRun) {
                    RRDs::create ($data->{$if}{'RRDFile'},
                        "DS:input:COUNTER:600:0:$bps",
                        "DS:output:COUNTER:600:0:$bps",
                        "RRA:AVERAGE:0.5:1:600",
                        "RRA:AVERAGE:0.5:6:700",
                        "RRA:AVERAGE:0.5:24:775",
                        "RRA:AVERAGE:0.5:288:797",
                        "RRA:MAX:0.5:1:54720",
                        "RRA:MAX:0.5:6:700",
                        "RRA:MAX:0.5:24:775",
                        "RRA:MAX:0.5:288:797");
                }
            }

            if ($DryRun) {
                print STDERR "dryrun: UPDATE $data->{$if}{'RRDFile'}: 'N:$incoming:$outgoing'\n";
            }
            else {
                # dann schieben wir die aktuellen Daten in die rrd
                # (auch pro Interface)
                RRDs::update($data->{$if}{'RRDFile'}, "N:$incoming:$outgoing" );
                print STDERR "RRD-UPDATE $data->{$if}{'RRDFile'}: 'N:$incoming:$outgoing'\n" if ($Debug);

                if ( RRDs::error() ) {
                    my $msg = RRDs::error();
                    print STDERR "'RRDs::update " . $data->{$if}{'RRDFile'}
                      . " N:$incoming:$outgoing': $msg\n";
                }
            }
        } # if ( $incoming ne 'undef' || $outgoing ne 'undef' )
    } # for (@interfaces)

    close(HOSTFILE);

    # Now that all the RRD files are written, connect to the database and
    # update the records.
    my $dbh = DBI->connect ("DBI:mysql:database=$dbase;host=$dbhost",
	    $dbuser, $dbpass);
    if (!$dbh)
    {
	print STDERR "Unable to connect to MySQL database on $dbhost: " . DBI->errstr () . "\n";
	write_to_log ("Unable to connect to MySQL database on $dbhost: " . DBI->errstr () . "\n");
	exit (1);
    }

    my $sth_get_counter = $dbh->prepare (<<SQL);
	SELECT ID, input_counter, output_counter, speed
	FROM $dbtable
	WHERE routerindex = ? AND interface = ?
SQL
    if (!$sth_get_counter)
    {
	print STDERR "Unable to prepare SQL statement 1: " . $dbh->errstr() . "\n";
	write_to_log ("Unable to prepare SQL statement 1: " . $dbh->errstr() . "\n");
	exit (1);
    }

    my $sth_update_interface = $dbh->prepare(<<SQL);
	UPDATE $dbtable SET
	customer = ?, description = ?, timestamp = ?, input_counter = ?, output_counter = ?, speed = ?
	WHERE ID = ?
SQL
    if (!$sth_update_interface)
    {
	print STDERR "Unable to prepare SQL statement 2: " . $dbh->errstr() . "\n";
	write_to_log ("Unable to prepare SQL statement 2: " . $dbh->errstr() . "\n");
	exit (1);
    }

    my $sth_create_interface = $dbh->prepare (<<SQL);
	INSERT INTO $dbtable
	(routerindex, interface, customer, description, input_counter, output_counter, speed) 
	VALUES (?, ?, ?, ?, ?, ?, ?)
SQL
    if (!$sth_create_interface)
    {
	print STDERR "Unable to prepare SQL statement 3: " . $dbh->errstr() . "\n";
	write_to_log ("Unable to prepare SQL statement 3: " . $dbh->errstr() . "\n");
	exit (1);
    }

    for (@interfaces)
    {
    	my $if = $_;

	if (!defined ($data->{$if})
		|| !defined ($data->{$if}{'queryType'})
		|| !defined ($data->{$if}{'customer'}))
	{
	    if ($Debug)
	    {
		require Data::Dumper;
		print "$host/$if has no queryType or customer, so no database update can be done.\n" if ($Debug);
		print STDOUT Data::Dumper->Dump ([$data->{$if}], ["data->{$if}"]);
	    }
	    next;
	}

	my $incoming;
	my $outgoing;

        if (defined($data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.3'}) and defined($data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.5'})) {
                $incoming = $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.3'};
                $outgoing = $data->{$if}{'.1.3.6.1.4.1.3224.9.3.1.5'};
        } else {
		$incoming = ($data->{$if}{'queryType'} eq '64bit') ? $data->{$if}{'ifHCInOctets'} : $data->{$if}{'ifInOctets'};
		$outgoing = ($data->{$if}{'queryType'} eq '64bit') ? $data->{$if}{'ifHCOutOctets'} : $data->{$if}{'ifOutOctets'};
	}

        if (!defined($incoming)) {
                $incoming = 0;
        }

        if (!defined($outgoing)) {
                $outgoing = 0;
        }
	
	$sth_get_counter->execute ($host, $data->{$if}{'ifDescr'})
	    or die ("execute: " . $sth_get_counter->errstr());
	my ($id, $old_incoming, $old_outgoing, $old_speed) = $sth_get_counter->fetchrow_array ();

	# There are already some counters in the database..
	if ($id)
	{
#                if ($DebugLog) {
#                    my $sth_dl =
#                      $dbh->prepare(
#"INSERT INTO debug (interface, direction, octets, code) "
#                          . "VALUES (?, ?, ?, ?)" )
#                      or die( "prepare: " . $dbh->errstr() );
#                    $sth_dl->execute( $id, 'in',  $incoming, $data->{$if}{'queryType'} );
#                    $sth_dl->execute( $id, 'out', $outgoing, $data->{$if}{'queryType'} );
#                }

	    if (($incoming != $old_incoming) || ($outgoing != $old_outgoing))
	    {
		$sth_update_interface->execute ($data->{$if}{'customer'},
			$data->{$if}{'ifAlias'}, $TimeStamp,
			$incoming, $outgoing,
			$data->{$if}{'ifSpeed'}, $id)
		    or write_to_log ("execute: " . $sth_update_interface->errstr () . "\n");

		if (($data->{$if}{'ifSpeed'} != 0) && ($data->{$if}{'ifSpeed'} != $old_speed))
		{
		    my $bps = ($data->{$if}{'ifSpeed'} / 8);

		    $time = localtime;
		    print "INFO: Tuning interface ``" . $data->{$if}{'ifDescr'} . "'' on ``$host'': "
			. "$old_speed -> " . $data->{$if}{'ifSpeed'} . " ($bps Byte/s)\n";
		    RRDs::tune ($data->{$if}{'RRDFile'},
			    '-a', "input:$bps",
			    '-a', "output:$bps") unless $DryRun;
		}
	    }
	    elsif ($Debug)
	    {
		print "\t\tCounters didn't change for " . $data->{$if}{'ifAlias'} . "\n";
	    }
	}
	else # if (!$id)
	{
	    print "\t\tCreating DB entry for " . $data->{$if}{'ifAlias'} . "\n" if ($Debug);

            unless ($DryRun) {
                $sth_create_interface->execute ($host,
                        $data->{$if}{'ifDescr'},
                        $data->{$if}{'customer'},
                        $data->{$if}{'ifAlias'}, 
                        $incoming, $outgoing,
                        $data->{$if}{'ifSpeed'})
                    or die ("execute: " . $sth_create_interface->errstr());
            }
        }
    } # for (@interfaces)

    $sth_create_interface->finish ();
    $sth_update_interface->finish ();
    $sth_get_counter->finish ();
    $dbh->disconnect ();

    # Exit the child and let the parent fork a new one
    exit(0);
}
close(HOSTS);

while ( waitpid( -1, 0 ) != -1 ) {

    # there are more children
}

write_to_log ("--- END ---\n");

exit;

sub write_to_log
{
	my $fh;
	my $status;
	my $prefix = sprintf ('[%5i] %s ', $$, scalar (localtime ()));

	unshift (@_, $prefix);

	if (!open ($fh, ">> $LogFile"))
	{
		warn ("open ($LogFile): $!");
		return;
	}

	if (!flock ($fh, LOCK_EX))
	{
		warn ("flock ($LogFile): $!");
		return;
	}

	$status = print $fh @_;
	print STDOUT @_ if ($Debug);

	close ($fh);

	return ($status);
}

=head1 AUTHOR

Florian octo Forster <octo@noris.net>

=cut

# vim:background=light:softtabstop=4:shiftwidth=4:hlsearch:incsearch:noignorecase
# vim: sw=4 expandtab
