#!/bin/sh 
# #############################################################################
#     (c) Copyright 1993, Hewlett-Packard NSMD, all rights reserved.
#
#     PACKAGE    HP OpenView OmniBackII
#     FILE       omni_chk_vers.sh
#     RCS        $Header: /src/shell/inst/remote/sup/omni_chk_vers.sh /main/43 2003-05-30 13:07:44 uxbuild $
#
#     USAGE
#        omni_chk_vers.sh <packet> <omni_homedir> <omni_vers> <series>
#                         <mgmt_sv> INSTALL|REMOVE
#
#     DESCRIPTION
#        Checks currently installed  version of OmniBack II packets
#
#     INSTALL
#
#        If OmniBack II is not yet installed on the current system, then 
#        OmniBack II HOME directory will be established. Otherwise the currently
#        installed version will be checked and if the system belongs to the 
#        OmniBack II installation of Cell Server <mgmt_sv>.
# 
#     REMOVE
#
#        Check, if current system belongs to the OmniBackII installation of 
#        Management Server <mgmt_sv>.
#
# #############################################################################
#     RETURNS
# 
# Exit value:                                                         stdout*
# ############                                                        ######
#
#    * if called by maintenance control script omni_inst.sh
#    W just a WARNING
#
# a) METHOD=INSTALL
#
#    0   OmniBackII HOME directory already exists and is empty
#        or
#        OmniBackII HOME directory has been succ. created
#    1   WILL NEVER BE SET EXPLICTLY
#    2   WILL NEVER BE SET EXPLICTLY
#    3   latest version of OmniBackII already installed
#    4   OmniBackII already installed, but not up-to-date            curr vers
#    5   Cannot change access rights of OmniBackII HOME directory
#    6   Cannot create OmniBackII HOME directory
#    7   Usage error
#    8   programming error when reaching this line
#    9   system belongs to a different OmniBackII installation       Mgmt.Server
#   12   OmniBackII HOME directory is not writeable
#   13 W OmniBackII HOME directory exists, 
#        but contains unexpected info
#   14   Calling user does not equal root
#   19   Omniback II supports only OmniStorage A.02.00
#
# b) METHOD=REMOVE
#
#    7   Usage error
#    8   programming error when reaching this line
#    9   system belongs to a different OmniBackII installation       Mgmt.Server
#   10   a certain version of OmniBackII is installed (okay)
#   11   OmniBackII not installed at all (or not for the desired architecture)
#   12   OmniBackII HOME directory is not writeable
#   13 W OmniBackII HOME directory exists, 
#        but contains unexpected info
#   14   Calling user does not equal root
#
# #############################################################################
PATH=$PATH:/bin:/usr/bin

PACKET=$1
OMNIHOME=$2         # OmniBackII HOME directory
OMNIVERS=$3         # latest OmniBackII version-ID; don't care if REMOVE method
SERIES=$4           # hp/s800/hp-ux, hp/s800/hp-ux-10,ibm/rs6000/aix, ...
MGMT_SV=$5          # name of Cell Server system
METHOD=$6           # INSTALL or REMOVE

# exists only if script is called by omni_inst.sh
# ###############################################
TOMNI=/tmp/omni_tmp

# file containing the name of the Cell Server
# this file does only exist, if this script is called by omni_inst.sh 
# ###################################################################
MGMT_SV_F=${TOMNI}/mgmt_sv.dat      

case "${SERIES}" in
  sun/sparc/solaris-7)
    if [ "${PACKET}" = "Core" ]
    then
                             # Check if the old version CORE has installed.
      $TOMNI/omni_chk_vers.sh Core /usr/omni/ ${OMNIVERS} sun/sparc/solaris-26 ${MGMT_SV} REMOVE
      RC_SOLARIS_OLD=$?
      case "${RC_SOLARIS_OLD}" in
        0 | 11 | 13)         # The old version CORE is NOT installed.
          ;;
        10)                  # The old version CORE is installed.
          exit ${RC_SOLARIS_OLD}
          ;;
        *)                   # Erros
          exit ${RC_SOLARIS_OLD}
          ;;
      esac
    fi
    OMNICONFIG=/etc/${OMNIHOME}
    CELL_SERVER=${OMNICONFIG}/cell/cell_server
    OMNI_INFO=${OMNICONFIG}/cell/omni_info
    ;;
  hp/s800/hp-ux-1* | ncr/i386/mp-ras | hp/sia64/hp-ux-11*)
    OMNICONFIG=/etc/${OMNIHOME}
    CELL_SERVER=${OMNICONFIG}/cell/cell_server
    OMNI_INFO=${OMNICONFIG}/cell/omni_info
    ;;
  *)
    OMNICONFIG=${OMNIHOME}/config
    CELL_SERVER=${OMNICONFIG}/cell/cell_server
    OMNI_INFO=${OMNIHOME}/bin/install/omni_info
    ;;
esac

case "${SERIES}" in
  hp/s800/hp-ux*|gpl/i386/linux*|gpl/ia64/linux*)
    HOSTNAME=`hostname`
    FULLNAME=`nslookup ${HOSTNAME} | /bin/grep Name: | /bin/sed 's/Name:[ ]*//'`
    ;;
  *)
    FULLNAME=""
    ;;
esac

# #############################################################################
#
# FUNCTION     standalone_run_msg 
#
# DESCRIPTION
#
#  Generate the corresponding error message on stderr, if running standalone; 
#
# ############################################################################# 
standalone_run_msg()
{
	MSG="$*"
	echo ${MSG} 1>&2
}

# #############################################################################
# script body
# #############################################################################

if [ $# -lt 6 ]
then
  # usage error
  # ###########
  standalone_run_msg \
    "Usage: omni_chk_vers.sh <packet> <omni_homedir> <omni_vers> <series> <mgmt_sv>\n \
     INSTALL|REMOVE"
  exit 7
fi

if [ -f ${CELL_SERVER} ]
then
  # ----------------------------------------------------------
  # now check, if related Cell Server is a different one
  # ----------------------------------------------------------
  CURR_MGMT_SV=`awk '{print $1}' ${CELL_SERVER}`
  if [ -n "${CURR_MGMT_SV}" ] && [ "${MGMT_SV}" != "${CURR_MGMT_SV}" ]
  then
    # #############################################################
    # current system belongs to a different OmniBackII installation
    # #############################################################
	echo "${CURR_MGMT_SV}" 1>&2
    exit 9
  fi
fi
case ${PACKET} in
  Core)
	STRING="CORE_VERSION"
        # check if omni_info has old format
        # ######################################################################
        if [ -f "${OMNI_INFO}" ]; then
          NO=`head -n 1 "${OMNI_INFO}" | grep "^-key " | wc -l`
          if [ $NO = 0 ]; then
            rm -f "${OMNI_INFO}"
          fi
        fi
	;;
  DiskAgent)
    STRING="DISK_AGENT_VERSION"
    ;;
  MediaAgent)
    STRING="MEDIA_AGENT_VERSION"
    ;;
  CellConsole)
    STRING="CELL_CONSOLE_VERSION"
    ;;
  OmniBackI)
    STRING="OMNIBACKI_VERSION"
    ;;
  SAP_Integration)
    STRING="SAP_INTEGRATION_VERSION"
    ;;
  OmniStorage_Integration)
    STRING="OST_INTEGRATION_VERSION"
    if [ "${SERIES}" = "hp/s800/hp-ux" ]; then
        V=OK 
    fi
    ;;
  ITO_Integration)
    STRING="OPC_INTEGRATION_VERSION"
    case ${SERIES} in
        hp/s800/hp-ux)
            OPCINFO=/usr/OV/bin/OpC/install/opcsvinfo
            ;;
        hp/s800/hp-ux-10*)
            OPCINFO=/opt/OV/bin/OpC/install/opcsvinfo
            ;;
    esac

    if [ -f "${OPCINFO}" ]
    then
      OPCHOST=`grep OPC_MGMT_SERVER ${OPCINFO}| awk '{print $2}'`
      HOST_IP=`nslookup ${OPCHOST} | \
                awk 'BEGIN{s=0}
                    {if ($1 == "Name:")
                        {s=1}
                     if (s==1)
                     {
                        if ($1 == "Address:")
                            {print $2}
                     }
                    }
                END{}'`
      INTERFACE_IP=`netstat -in |grep lan |awk '{if ($4 != "none") {print $4}}'`

      if [ "${HOST_IP}" != "${INTERFACE_IP}" ] 
      then
          standalone_run_msg \
             "OpC Management Server is not installed on the host."
          exit 20
      fi
    else
      standalone_run_msg \
             "OpC management Server is not installed on the host."
      exit 20
    fi
    ;;
  Oracle_Integration)
    STRING="ORACLE_INTEGRATION_VERSION"
    ;;
  Oracle8_Integration)
    STRING="ORACLE8_INTEGRATION_VERSION"
    ;;
  Informix_Integration)
    STRING="INFORMIX_INTEGRATION_VERSION"
    ;;
  Sybase_Integration)
    STRING="SYBASE_INTEGRATION_VERSION"
    ;;
  Enabler_Integration)
    STRING="ENABLER_INTEGRATION_VERSION"
    ;;
  EMC_Integration)
    STRING="EMC_INTEGRATION_VERSION"
    ;;
  Fastrax_MediaAgent)
    STRING="FASTRAX_MEDIA_AGENT_VERSION"
    ;;
  Fastrax_Integration)
    STRING="FASTRAX_INTEGRATION_VERSION"
    ;;
  ACS_Integration)
    STRING="ACS_INTEGRATION_VERSION"
    ;;
  DAS_Integration)
    STRING="DAS_INTEGRATION_VERSION"
    ;;
  SSEA_Integration)
    STRING="SSEA_INTEGRATION_VERSION"
    ;;
  *)
	standalone_run_msg "Unknown component ${PACKET}"
	exit 8
esac

# ###############################################################
# check, if directory already exists
# if not, create it - if METHOD equals INSTALL
# if yes, check if latest OmniBackII version is already installed
# ###############################################################
if [ -d ${OMNIHOME} ] 
then
  if [ ! -w ${OMNIHOME} ]
  then
    # ################################################
    # ${OMNIHOME} already exists, but is not writeable
    # ################################################
    standalone_run_msg \
     "Data Protector home directory ${OMNIHOME} already exists,\n \
      but is not writable"
    exit 12
  else
    if [ ! -f ${OMNI_INFO} -o ! -r ${OMNI_INFO} ]
    then
      # ############################
      # check, if directory is empty
      # ############################
      if [ "`ls ${OMNIHOME}`" = "" ]
      then
        if [ "${METHOD}" = "INSTALL" ]
        then
          # ${OMNIHOME} already exists, but it's empty - okay
          standalone_run_msg \
            "Data Protector home directory ${OMNIHOME} already exists,\n \
             but it's empty - OK"
          exit 0
        else 
          # ########################################################
          # REMOVE
          # Ooops, OMNIHOME exists, but OmniBackII is not installed
          # maybe OmniBackII has been removed before
          # --> go on, as if OmniBackII is still installed
          # ########################################################
          if [ -f "${OMNICONFIG}/cell/cell_info" -a "${PACKET}" = "CellConsole" ]
          then
            standalone_run_msg \
             "Removing CellConsole from Cell Server is not a valid operation"
            exit 15    
          fi 

          standalone_run_msg \
            "Data Protector home directory ${OMNIHOME} already exists,\n \
             but it's empty"
          standalone_run_msg "Maybe Data Protector has been removed manually - okay"
          exit 10
        fi
      else
        exit 13
      fi
    else

      # Omni_info exists
      # ########################
      case ${SERIES} in
      hp/s800/hp-ux|gpl/i386/linux*|gpl/ia64/linux*)
        NUMBER=`/bin/grep -c "${STRING}[ ]*${FULLNAME}" ${OMNI_INFO}`
        ;;
      *) 
        NUMBER=`/usr/bin/grep -c ${STRING} ${OMNI_INFO}`
        ;;
      esac

      if [ ${NUMBER} -eq 0 ]
      then
        exit 0
      else
        case ${SERIES} in
        hp/s800/hp-ux)
          CURR_VERS=`awk '{ if ($1 == STR && $2 == FN) {print $(NF)}}' \
                            STR=${STRING} FN=${FULLNAME} ${OMNI_INFO}`
          ;; 
        *)
          CURR_VERS=`awk '{ if ($1 == STR) {print $(NF)}}' \
                            STR=${STRING} ${OMNI_INFO}`
          ;; 
        esac
      fi

      if [ "${METHOD}" = "INSTALL" ]
      then
        # ######################################################
        # check, which OmniBackII version is currently installed
        # (not of interest for REMOVE option)
        # ######################################################
        if [ "${OMNIVERS}" = "${CURR_VERS}" ]
        then
          # ##############################################################
          # ==> latest version ${OMNIVERS} of OmniBackII already installed
          # ##############################################################
          if [ -z "${CURR_MGMT_SV}" ]
          then
            # Node will be added to the cell
            # ##############################
            if [ ! -d "${OMNICONFIG}/cell" ]
            then
              mkdir -p ${OMNICONFIG}/cell
            fi
            echo ${MGMT_SV} > ${CELL_SERVER} 
            standalone_run_msg "Latest version ${OMNIVERS} already installed"
            exit 17
          fi
          standalone_run_msg "Latest version ${OMNIVERS} already installed"
          exit 3
        else
          # checking if ${OMNIVERS} is older or newer than ${CURR_VERS}
          # ###########################################################
          NEWER=`echo ${OMNIVERS} ${CURR_VERS} | \
				awk '{	split($1, NEW, ".")
					split($2, CURR, ".")
                                        split(NEW[3], ITEM1, "%")
                                        split(CURR[3], ITEM2, "%")
				        if(NEW[2] < CURR[2])
                                            print $2
				        else if(NEW[2] > CURR[2]) 
                                            print $1
                                        if(ITEM1[1] < ITEM2[1])
                                            print $2
                                        else if(ITEM1[1] > ITEM2[1])
                                            print $1
                                        else
                                            if(ITEM1[2]=="")
                                                if(ITEM2[2]=="")
                                                    print $2 
					        else 
                                                    print $1
                                            else
                                                print $1}'`


          if [ "${NEWER}" != "${OMNIVERS}" ]; then
            # we don't allow to install an older version than the version 
            # that is currently installed 
            # ###########################################################
            standalone_run_msg \
              "The installation of an older version ${OMNIVERS} than the currently"
            standalone_run_msg \
              "installed version ${CURR_VERS} on the remote host isn't permitted."
            standalone_run_msg \
               "If you really want to proceed with this operation, deinstall\
                the software first."
            if [ -f ${MGMT_SV_F} ]
            then
              echo ${CURR_VERS}
            fi
            exit 18
          else
            # version is OK, ${OMNIVERS} will be installed
            # ############################################
            if [ -f ${MGMT_SV_F} ]
            then
              echo ${CURR_VERS}
            fi
            standalone_run_msg \
              "Version ${CURR_VERS} instead of ${OMNIVERS} currently installed"
            if [ -z "${CURR_MGMT_SV}" ]
            then
              # Node will be added to the cell
              # ##############################
              if [ ! -d "${OMNICONFIG}/cell" ]
              then
                mkdir -p ${OMNICONFIG}/cell
              fi
              echo ${MGMT_SV} > ${CELL_SERVER} 
            fi
            exit 4
          fi 
        fi
      else 
        # -----------------------------------------
        # METHOD = REMOVE
        # system belongs to OmniBackII installation 
        # that's okay for REMOVE option
        # -----------------------------------------
        if [ -f ${OMNICONFIG}/cell/cell_info -a "${PACKET}" = "CellConsole" ]
        then
            standalone_run_msg \
            "Removing CellConsole from Cell Server is not a valid operation"
            exit 15    
        fi
        exit 10
      fi
    fi
  fi 
else  
  # #########################################
  # OmniBack II HOME directory does not exist
  # #########################################
  if [ ${METHOD} = "INSTALL" ]
  then
    # ##############################################
    # OmniBack II HOME directory does not yet exist, 
    # therefore it will be created
    # ##############################################
    mkdir -p ${OMNIHOME}
    if [ $? -eq 0 ]
    then
      chmod 755 ${OMNIHOME}
      if [ $? -eq 0 ]
      then
        # ####################################
        # ==> ${OMNIHOME} successfully created
        # ####################################
        standalone_run_msg "${OMNIHOME} successfully created"
        exit 0
      else
        # ##############################################
        # ==> cannot change access rights of ${OMNIHOME}
        # ##############################################
        standalone_run_msg "Cannot change access rights of ${OMNIHOME}"
        exit 5
      fi
    else 
      # #############################
      # ==> cannot create ${OMNIHOME}
      # #############################
      standalone_run_msg "Cannot create ${OMNIHOME}"
      exit 6
    fi
  else 
    # ############################################
    # METHOD = REMOVE
    # OmniBackII HOME directory does not exist; 
    # therefore OmniBackII is not installed at all
    # ############################################
    standalone_run_msg "Data Protector not installed at all"
    exit 11
  fi
fi
# #################################################################
# ==> programming error in omni_chk_vers.sh when reaching this line
# #################################################################

standalone_run_msg "Programming error"
exit 8

# #############################################################################
# End of file: omni_chk_vers.sh
# #############################################################################
