#!/bin/sh
###############################################################################
#
#     (c) Copyright 1993, Hewlett-Packard NSMD, all rights reserved.
#
#     PACKAGE      HP OpenView OmniBackII
#     FILE         omni_chk_ds.sh
#     RCS          $Header: /src/shell/inst/remote/sup/omni_chk_ds.sh /main/hsl_dp61/hsl_hpit2_2/1 2009-07-01 13:17:08 seidh $
#
#     USAGE
#        CMD <loc1> <size1> <series> 
#
#     DESCRIPTION
#        Check diskspace on a UN*X system, called by omni_inst.sh
#
###############################################################################
#     RETURNS
#
# exit value:                                              value on stdout
# ------------                                             ---------------
#  0  okay (enough disk space for <location>);             free disk space
#  3  programming error: unexpected architecture/series          ---
#  4  same file system for <location> and <loc2>
#     and enough disk space                                free disk space
#  6  too less free disk space for <location>;             free disk space
#  7  usage error                                                ---
#  8  Calling user does not equal root
###############################################################################

PATH=$PATH:/bin:/usr/bin

# exists only if script is called by omni_inst.sh
# ###############################################
TOMNI=/tmp/omni_tmp

# file containing result 
# ######################
TMP_FILE=${TOMNI}/omni_chk_ds.tmp

# script call parameters
# ######################

CMD=$0
LOCATION=$1     # path where required disk space must be available
                # (<homedir> or /tmp/omni_tmp)
SIZE=$2         # required disk space in kilo bytes
SERIES=$3       # architecture where this script runs on
                # hp/s700/hp-ux, ibm/rs6000/aix, sun/sparc/sunos, 
                # hp/s700/hp-ux-10, gpl/i386/linux, sun/sparc/solaris, 
                # dec/alpha/osf1, sgi/mips/irix, /sco/i386/sco_sv
                # dec/alpha/osf1-4, sgi/mips/irix-62, siemens/r400/sinix-543
                # ibm/rs6000/aix-42, gpl/i386/linux-60, gpl/ia64/linux

# #############################################################################


# #############################################################################
#
# FUNCTION     standalone_run_msg 
#
# DESCRIPTION
#
#   Generates the corresponding error message on stderr, if running standalone 
#
# #############################################################################
standalone_run_msg()
{
	MSG="$*"
	echo ${MSG} 1>&2
}

# #############################################################################
#
#     FUNCTION    upd_res_symlink 
#
#     ARGUMENTS
#         $1    absolute file path (e.g.: /users/guest)
#
#     DESCRIPTION
#         resolves the symbolic link in the specified absolute file path
#
# #############################################################################
upd_res_symlink()
{
  PATHCOMP=`echo ${1} | awk 'BEGIN{FS="/"}
                              $0 != "/" {
                                          for (i=1; i<=NF; i++)
                                          {
                                            if (i == 1)
                                              {printf "/ "}
                                            else
                                              {printf "/%s ", $i}
                                          }                              
                                        }
                              $0 == "/" {print $0}'`

  RETFLAG=0
  RESPATH=""
 
  for comp in ${PATHCOMP}
  do
    if [ "${RESPATH}" = "/" ]
    then
      RESPATH=${comp}
    else
      RESPATH="${RESPATH}${comp}"
    fi

    if [ -h ${RESPATH} ]
    then
      RESPATH=`ls -la ${RESPATH} | awk '{ print $NF }'`
      RETFLAG=1
    fi
  done

  echo ${RESPATH}

  return ${RETFLAG}
}

# #############################################################################
#
#     FUNCTION    get_pathname 
#         
#     DESCRIPTION
#         determine absolute file path without symbolic links
#
# #############################################################################
get_pathname()
{
  search_dir=`upd_res_symlink $1`

  while [ $? -ne 0 ]
  do
    search_dir=`upd_res_symlink ${search_dir}`
  done

  # prepare physical home directory and mount output for matching procedure
  # #######################################################################

  search_dir=`echo ${search_dir} | sed -e 's/\//:/g' -e 's/^://'`
 
  ${MOUNT} 2>/dev/null | awk '{ print $1 }' > ${TMP_FILE}
 
  # try to match the physical home directory of OmniBack II 
  # with a filesystem mounted on the Cell Server
  # #######################################################

  file_sys=`awk 'BEGIN {finalmatch=""; finallevel=0; FS="/"}
		    {
		      # initialize variables
		      # --------------------

		      split(filepath,patternlist,":");
		      pattern=patternlist[1]

		      matchp=""
		      matchlevel=0

		      # loop over all record fields
		      # ---------------------------

		      for (i=2; i<=NF; i++)
		      {
			# test if the pattern matches with 
			# the record field $i
			# --------------------------------

			if ($i == pattern)
			{
			  matchlevel=matchlevel+1
			  matchp=matchp"/"pattern
			  pattern=patternlist[i]
			  continue
			}
			break
		      }

		      # test if a better match than before 
		      # was made
		      # ----------------------------------

		      if (i == (NF+1) && finallevel < matchlevel)
		      {
			finalmatch=matchp
			finallevel=matchlevel
		      }
		    }
	       END  {
		      if (finallevel == 0)
		      {
			# no other match found than /
			# ---------------------------

			finalmatch="/"
		      }

		      # print matched file system
		      # -------------------------

		      print finalmatch

		    }' filepath=${search_dir} ${TMP_FILE}`

  echo $file_sys
  return 0
}

# #############################################################################
#
#     FUNCTION    create_dir 
#         
#     DESCRIPTION
#         Creates the /opt/omni directory with the necessary permission 
#         and ownership.
#
# #############################################################################
create_dir()
{
  if [ ! -d $LOCATION ]
  then
      mkdir $LOCATION
      chmod 755 $LOCATION
      chown 0:3 $LOCATION
  fi
}

# #############################################################################
# script body
# #############################################################################

if [ $# -lt 3 ]
then
  standalone_run_msg "usage: ${CMD} <loc1> <size1> <series>"
  exit 7
fi


# determine still available disk space
#-------------------------------------

case "${SERIES}" in
  hp/s[78]00/hp-ux)
    # problem HP-UX needs name of corresponding file system, because it
    # does not resolve it by itself !!!
    MOUNT=/etc/mount
    FS=`get_pathname ${LOCATION}`
    # Problems with HP-UX 9.01
    #
    # /          (/dev/dsk/c201d6s0   ):     12232 blocks ...
    # instead of
    # /          (/dev/dsk/c201d6s0):     12232 blocks ...
    DSK_SPACE=`/bin/df ${FS} | awk 'BEGIN{s=0}
                                               {if ($3 != "):" )
                                                  {s=$3}
                                                else
                                                  {s=$4}
                                               }
                                             END {print s}'`

    DSK_SPACE=`expr ${DSK_SPACE} \/ 2`  # df displays disk space in 512 Byte Blocks
    /bin/rm -f ${TMP_FILE}
    ;;
  hp/*/hp-ux-11*)
    # problem HP-UX needs name of corresponding file system, because it
    # does not resolve it by itself !!!
    MOUNT=/sbin/mount
    FS=`get_pathname ${LOCATION}`
    
    DSK_SPACE=`/usr/bin/df ${FS} | /sbin/awk 'BEGIN{s=0}
                                               {if ($3 != "):" )
                                                  {s=$3}
                                                else
                                                  {s=$4}
                                               }
                                             END {print s}'`
    DSK_SPACE=`expr ${DSK_SPACE} \/ 2` 
    DSK_SPACE=`expr ${DSK_SPACE} \+ 2000`
    # df displays disk space in 512 Byte Blocks
    /sbin/rm -f ${TMP_FILE}
    ;;

  sun/*/solaris*)
    create_dir
    DSK_SPACE=`/bin/df -k ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`

    # just determine free disk space for <location>
    if [ ${DSK_SPACE} -lt ${SIZE} ]
    then
        echo ${DSK_SPACE}
        exit 6
    else
        echo ${DSK_SPACE}
        exit 0
    fi
    ;;

  sgi/mips/irix*)
    create_dir
    DSK_SPACE=`/bin/df -k ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$5}
                                                   END {print s}'`
    ;;

  siemens/r400/sinix*)
    create_dir
	if [ -x /usr/ucb/df ]
	then
		DSK_SPACE=`/usr/ucb/df ${LOCATION} | /usr/bin/awk ' BEGIN{s=0}
													{s=$4}
												END {print s}'`
	else
		DSK_SPACE=`/bin/df -b ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$NF}
                                                   END {print s}'`
	fi
    ;;

  dec/alpha/osf1*)
    create_dir
    DSK_SPACE=`/bin/df ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`
    # df displays disk space in 512 Byte Blocks
    DSK_SPACE=`expr ${DSK_SPACE} \/ 2`
    ;;

  ibm/rs6000/aix*)
    create_dir
    DSK_SPACE=`/bin/df ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$3}
                                                   END {print s}'`
    if [ "`/bin/df | grep 'Total KB'`" = "" ]; then
    # df displays disk space in 512 Byte Blocks
         DSK_SPACE=`expr ${DSK_SPACE} \/ 2`
    fi
    ;;

  sco/i386/sco_sv)
    create_dir
    DSK_SPACE=`/bin/df -B ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`
    # df displays disk space in 512 Byte Blocks
    DSK_SPACE=`expr ${DSK_SPACE} \/ 2`
    ;;

  sco/i386/unixware)
    create_dir
    DSK_SPACE=`/bin/df -k ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`
    # df displays disk space in 512 Byte Blocks
    DSK_SPACE=`expr ${DSK_SPACE} \/ 2`
    ;;

  ncr/i386/mp-ras)
    create_dir
    DSK_SPACE=`/usr/ucb/df ${LOCATION} | /usr/bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`
    ;;

  gpl/*/linux*)
    create_dir
    DSK_SPACE=`/bin/df -P ${LOCATION} | /bin/awk 'BEGIN{s=0}
                                                     {s=$4}
                                                   END {print s}'`
    ;;
  *)
    # Programming error: unexpected architecture/series
    exit 3
    ;;
esac

# check, if required disk space is available
# ##########################################

# just determine free disk space for <location>
if [ ${DSK_SPACE} -lt ${SIZE} ]
then
	echo ${DSK_SPACE}
    exit 6
else
	echo ${DSK_SPACE}
    exit 0
fi

# #############################################################################
# End of file: omni_chk_ds.sh
# #############################################################################
