#!/bin/sh
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
# This software is supplied under the terms of a license agreement or
# nondisclosure agreement with Dell Inc. and may not
# be copied or disclosed except in accordance with the terms of that
# agreement.
#
# Copyright (c) 2000-2004 Dell Inc. All Rights Reserved.
#
# Module Name:
#
#   cli_ini_modifier.sh
#
# Abstract/Purpose:
#
#   This Script is used by OpenManage rpm packages to dynamically
#   register and unregister with OpenManage Command Line Interface 
#   at %post and %preun install sections of rpm packages.
#
# Environment:
#
#   Linux
#
###############################################################################

###############################################################################
#
#  Global Variables
#
###############################################################################

# omsa cli ini file path
OMSA_CLI_INI_FILEPATH=oma/ini/omcli32.ini

# omreg keys
OMREG_FILE_PATH=/etc/omreg.cfg
OMREG_OM_OMCLIREGISTRATION_KEY=openmanage.omacore.clireg
OMREG_OM_OMACORE_INSTPATH_KEY=openmanage.omacore.installpath
OMREG_SYSLISTFILE=openmanage.syslistfile
OM_ARCHTYPE_KEY=openmanage.archtype
ADDCLITOKEN="AddOMSACLI="
ADDCLITOKENBLADE="AddOMSACLIOnBlade="
DELCLITOKEN="DeleteOMSACLI="
DELCLITOKENBLADE="DeleteOMSACLIOnBlade="
REPLACESPACEWITH="REPLACINGSPACE"
FEATURE_INI_FILEPATH=
EXECUTE_INI_SECTION=
PLATFORMID=

###############################################################################
#
#  Function : AddToOMSACLI
#
#  Adds the given cli entry to omcli32.ini
#
###############################################################################

#copied from funcs.sh.
#gets value from ini file for a particular key
#$1 is the ini file name, and $2 is the key

function GetINIVal() 
{
  FILE="${1}"
  KEY="${2}"

  [ ! -f "${FILE}" ] && echo "" && return 1

  GetValue "`grep -i "^[[:space:]]*${KEY}[[:space:]]*=" ${FILE}`"
  return 0
}

#copied from funcs.sh.
#needed by function GetINIVal

function GetValue() 
{
  PAIR="${1}"

  echo "${PAIR}" | sed 's#^[^=]*=##; s#^[[:space:]]*##; s#[[:space:]]*$##'
}

#checks whether a sysid belongs to a blade server
#$1 is the sysid
#prints a 1 if blade, else a null

function CheckBlade 
{
    pattern=$1
   
    if [ "${pattern}" = "" ]
    then    
      return 1
    else
        SYSLIST_PATH=`GetINIVal "${OMREG_FILE_PATH}" "${OMREG_SYSLISTFILE}"`

        awk -v found=0 -v pat="$pattern" -F= '
        /\[BladeSystems\]/ {found = 1} 
        { if((found == "1") && ($1 ~ pat)) { print "1";found=0 }}
        ' "${SYSLIST_PATH}"
    fi
    return 0
}

function AddToOMSACLI {
  # $1 : cli string

  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      inside_section = 0
      added_new_kv = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
      numkvtoks = split(secarr[2], kvarr, "=")
      key_str = kvarr[1]
      value_str = kvarr[2]
    }
    {
      if (found_section == 1 && inside_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # check if key found
          if (added_new_kv == 0) {
            # key not found in the section
            # create for the first time.
            printf("%s=%s\n", key_str, value_str) >> tmpfile
            added_new_kv = 1;
          }

          # this line is a new section
          inside_section = 0
          printf("%s\n", $0) >> tmpfile
        }
        else {
          # this line is a key-value pair
          # search for the correct key
          if (index($0, key_str) == 1) {
            # found the key
            added_new_kv = 1

            # split the string using delimiter (=)
            arrlen = split($0, tokenarr, "=")
            if (arrlen > 0) {
              printf("%s=", tokenarr[1]) >> tmpfile
              printf("%s%s\n", tokenarr[2], value_str) >> tmpfile
            }
          }
          else {
            printf("%s\n", $0) >> tmpfile
          }
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1
          inside_section = 1
        }
        printf("%s\n", $0) >> tmpfile
      }
    }
    END{
      if (found_section == 0) {
        # create section and add key-value
        printf("%s\n", section_str) >> tmpfile
        printf("%s=%s\n", key_str, value_str) >> tmpfile
      }
      else {
        if (added_new_kv == 0) {
          # key not found in the section
          # create for the first time.
          printf("%s=%s\n", key_str, value_str) >> tmpfile
        }
      }
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}

###############################################################################
#
#  Function : IsComponentRegistered
#
#  Reads /etc/omreg file for confirmation
#
###############################################################################

function IsComponentRegisteredWithOmreg {
  if [ -e $OMREG_FILE_PATH ];
  then
      CURRENT_CLI_VALUES=`grep "^$OMREG_OM_OMCLIREGISTRATION_KEY=" $OMREG_FILE_PATH`
      echo "$CURRENT_CLI_VALUES" | grep $FEATURE_INI_FILEPATH >/dev/null
      if [ $? == 0 ];
      then
          exit 0
      fi
  else
      # don't need to prompt
      exit 0
  fi
}

###############################################################################
#
#  Function : RegisterOMSACLI
#
#  Reads all the add cli entries from the feature ini file and invokes
#  AddToOMSACLI for each cli entry
#
###############################################################################

function RegisterOMSACLI {
  # $1 : path to feature ini file

  # Any cli entry with a space should be replaced with some unique string to
  # prevent it from being split into 2 tokens by "for" loop, if not done this
  # way, a tmp file is required. The Unique string used here is "REPLACINGSPACE" but
  # it will be changed.

  # do not register if already registered
  IsComponentRegisteredWithOmreg

  FEATURE_INI_FILE=$1
  SYSID=$2

  for CLIENTRY in `grep "$ADDCLITOKEN" $FEATURE_INI_FILE | sed "s/^$ADDCLITOKEN//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    AddToOMSACLI "$CLIENTRY"
  done

  IS_BLADE=`CheckBlade "$SYSID"`
  if [ "$IS_BLADE" = "1" ]
  then
     for CLIENTRY in `grep "$ADDCLITOKENBLADE" $FEATURE_INI_FILE | sed "s/^$ADDCLITOKENBLADE//g" | sed "s/ /$REPLACESPACEWITH/g"`;
     do
        AddToOMSACLI "$CLIENTRY"
     done
 fi

  # register with /etc/omreg file
  echo "$OMREG_OM_OMCLIREGISTRATION_KEY=$FEATURE_INI_FILEPATH" >> $OMREG_FILE_PATH
}

###############################################################################
#
#  Function : RemoveEmptySection
#
#  For any given section, if sections is empty, remove it
#
###############################################################################

function RemoveEmptySection {
  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      is_section_empty = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
    }
    {
      if (found_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # this line is a new section
          printf("%s\n", $0) >> tmpfile
          found_section = 0
        }
        else {
          # section is not empty,
          if ( is_section_empty == 0 ){
            printf("%s\n", section_str) >> tmpfile
            is_section_empty = 1
          }
          printf("%s\n", $0) >> tmpfile
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1          
        }
        else {
          printf("%s\n", $0) >> tmpfile
        }
      }
    }
    END{
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}


###############################################################################
#
#  Function : DelFromOMSACLI
#
#  Deletes the given cli entry from omcli32.ini
#
###############################################################################

function DelFromOMSACLI {

  # replace the REPLACESPACEWITH varibale with space
  CLIENTRY=`echo "$1" | sed "s/$REPLACESPACEWITH/ /"`
  TMP_OMSA_FILE=$OMSA_CLI_INI_FILEPATH".tmp"

  awk -v val_str="$CLIENTRY" -v tfile=$TMP_OMSA_FILE '
    BEGIN{
      # initialize variables
      tmpfile = tfile
      found_section = 0
      added_new_kv = 0

      # split the clientry string into section, key and value strings
      # clientry string is formatted as: "section;key=value"
      # example: omreport storage;controllers=sngl,cdll,ssclp

      numsectoks = split(val_str, secarr, ";")
      section_str = "["secarr[1]"]"
      numkvtoks = split(secarr[2], kvarr, "=")
      key_str = kvarr[1]
      value_str = kvarr[2]
    }
    {
      if (found_section == 1) {
        # we are in the correct section
        # there are 2 possibilites, the next line can be a
        # key-value pair or a new section.

        if (index($0, "[") == 1) {
          # this line is a new section, so key not present
          # reset section to not found
          found_section = 0
          printf("%s\n", $0) >> tmpfile
        }
        else {
          # this line is a key-value pair, search for the correct key
          if (index($0, key_str) == 1) {
            # found the key, split the entry using delimiter (=)
            arrlen = split($0, tokenarr, "=")
            if (arrlen > 0) {
              keyentry = tokenarr[1]
              valentry = tokenarr[2]

              if (index (valentry, value_str) != 0) {
                # current value contains value_str
                # remove value_str from current value
                newvaluehead=substr(valentry, 0, index (valentry, value_str) - 1)
                newvaluetail=substr(valentry, index (valentry, value_str) + length(value_str) , length(valentry))
                newvaluestring=sprintf("%s%s",newvaluehead, newvaluetail)
                if (length(newvaluestring) > 0) {
                  # otherwise, write the updated key=value to the file
                  printf("%s=%s\n", keyentry, newvaluestring) >> tmpfile
                }
              }
              else {
                printf("%s\n", $0) >> tmpfile
              }
            }
          }
          else {
            printf("%s\n", $0) >> tmpfile
          }
        }
      }
      else {
        if (index ($0, section_str) != 0) {
          found_section = 1
        }
        printf("%s\n", $0) >> tmpfile
      }
    }
    END{
      close(tmpfile)
    }
  ' $OMSA_CLI_INI_FILEPATH

  # rename the tmp file to the original file name
  mv $TMP_OMSA_FILE $OMSA_CLI_INI_FILEPATH
  chmod 644 ${OMSA_CLI_INI_FILEPATH}
}

###############################################################################
#
#  Function : UnRegisterOMSACLI
#
#  Reads all the delete cli entries from the feature ini file and invokes
#  DelFromOMSACLI for each cli entry.
#
###############################################################################

function UnRegisterOMSACLI {
  # $1 : path to feature ini file

  FEATURE_INI_FILE=$1
  for CLIENTRY in `grep "$DELCLITOKEN" $FEATURE_INI_FILE | sed "s/^$DELCLITOKEN//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    DelFromOMSACLI "$CLIENTRY"
    RemoveEmptySection "$CLIENTRY"
  done

  #delete if any blade specific entries
  for CLIENTRY in `grep "$DELCLITOKENBLADE" $FEATURE_INI_FILE | sed "s/^$DELCLITOKENBLADE//g" | sed "s/ /$REPLACESPACEWITH/g"`;
  do
    DelFromOMSACLI "$CLIENTRY"
    RemoveEmptySection "$CLIENTRY"
  done

  egrep -iv "$OMREG_OM_OMCLIREGISTRATION_KEY=$FEATURE_INI_FILEPATH" $OMREG_FILE_PATH >$OMREG_FILE_PATH.tmp
  mv $OMREG_FILE_PATH.tmp $OMREG_FILE_PATH
  chmod 644 ${OMREG_FILE_PATH}
}

###############################################################################
#
#  Function : GetOMSAInstallPath
#
#  Reads the OpenManage install path from omreg.cfg and populates 
#  the global variable OMSA_CLI_INI_FILEPATH
#
###############################################################################

function GetOMSAInstallPath {
  OM_ARCH_TYPE=`grep "^$OM_ARCHTYPE_KEY=" $OMREG_FILE_PATH | sed "s/^.*=//"`


  OM_INST_PATH_ENTRY=`grep "^$OMREG_OM_OMACORE_INSTPATH_KEY=" $OMREG_FILE_PATH`
  OM_INST_PATH=`echo "$OM_INST_PATH_ENTRY" | sed "s/^.*=//"`
 
  if [ "$OM_ARCH_TYPE" != "" ];
  then 
      OMSA_CLI_INI_FILEPATH=$OM_INST_PATH/oma/ini/omcli"$OM_ARCH_TYPE".ini
  else
      OMSA_CLI_INI_FILEPATH=$OM_INST_PATH/$OMSA_CLI_INI_FILEPATH
  fi

  # check for the existence of file
  if [ ! -e $OMSA_CLI_INI_FILEPATH ];
  then
      # file doesn't exist, react differntly for "add" and "del"
      if [ "$EXECUTE_INI_SECTION" = "add" ];
      then
          echo "OMSA CLI registration file doesn't exist"
          echo "OMSA CLI registration failed"
          exit 1
      elif  [ "$EXECUTE_INI_SECTION" = "del" ];
      then
          egrep -iv "$OMREG_OM_OMCLIREGISTRATION_KEY=$FEATURE_INI_FILEPATH" $OMREG_FILE_PATH >$OMREG_FILE_PATH.tmp
          mv $OMREG_FILE_PATH.tmp $OMREG_FILE_PATH
          exit 1
      else
          exit 1
      fi
  fi
}

###############################################################################
#
#  Function : Main
#
#  This is the starting point for the script, it invokes other functions
#  in the required order
#
###############################################################################

function Main {
  # get omsa install path
  GetOMSAInstallPath

  # Add to OMSA CLI or Delete from OMSA CLI depending on the request
  if [ "$EXECUTE_INI_SECTION" = "add" ];
  then
      RegisterOMSACLI $FEATURE_INI_FILEPATH $PLATFORMID
  elif [ "$EXECUTE_INI_SECTION" = "del" ];
  then
      UnRegisterOMSACLI $FEATURE_INI_FILEPATH $PLATFORMID
  else
      echo "Invalid Parameters to OMSA CLI Registration"
      echo "OMSA CLI Registration failed"
      exit 1
  fi
}

###############################################################################
#
#  Script execution begins here with the invocation
#  of Main function
#
###############################################################################
if [ $# -lt 2 ];
then
    echo "Invalid Parameters to OMSA CLI Registration"
    echo "OMSA CLI Registration failed!"
    exit 1
else
    FEATURE_INI_FILEPATH=$1
    FEATURE_INI=`basename "$FEATURE_INI_FILEPATH"`
    OMREG_OM_OMCLIREGISTRATION_KEY="${OMREG_OM_OMCLIREGISTRATION_KEY}.${FEATURE_INI}"

    EXECUTE_INI_SECTION=$2
    #Note: $3 is optional. current ussage is only for blades.added in om5.3
    PLATFORMID=$3 
fi

Main
