#!/bin/sh
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
#
#  This software is supplied under the terms of a license agreement or
#  nondisclosure agreement with Dell Inc. and may not
#  be copied or disclosed except in accordance with the terms of that
#  agreement.
#
#  Copyright (c) 2000-2005 Dell Inc. All Rights Reserved.
#
#
#  Revision:
#    $Revision: $
#
#  Last Modified By / On:
#    $Author: $ / $Date: $
#
#  Author:
#    OpenManage Install
#
#  Environment:
#    Linux
#
#  User/Kernel Mode:
#    User
#
#  Abstract/Purpose:
#    Helper functions used by serveradministrator spec scriptlets
#
#  See Also:
#
#
#  Notes:
#
#
###############################################################################
[ -n "${OMIDEBUG}" ] && set -x

# set default access rights to limit potential security lapses
umask 077
# limit all commands to /bin directory unless a full is specified
PATH=/sbin:/bin:/usr/sbin:/usr/bin

REPLACESPACEWITH="_REPLACINGSPACE_"

##
## returns a safe temporary filename (respecting any $TMP directory given)
##
GetTemp() {
  GETTEMPFILE=`mktemp ${TMP:-/var/tmp}/ominstall.XXXXXXX`
  [ $? -ne 0 ] && ErrorMsg "error: cannot create temp file, exiting..." && exit 1
  chmod 600 ${GETTEMPFILE} && chown root.root ${GETTEMPFILE}
  echo "${GETTEMPFILE}"
  return 0
}

MakeFile() {
  MAKEFILE="${1}"
  [ ! -f "${MAKEFILE}" ] && touch ${MAKEFILE} && chmod 664 ${MAKEFILE} && chown root.root ${MAKEFILE}
}


##
## Sends an error message to stderr
##
ErrorMsg() {
  MSG=${1}

  echo "${MSG}" 1>&2
}

##
## Cleans up omreg.cfg of unneeded entries
## This should be ONLY called while omilcore is being uninstalled.
##
CleanOMReg(){

# Now that omilcore is getting uninstalled,
# the CLIREGUTIL will go
# which means that all the regutil entries cannot be
#cleaned up by the relevant rpms. Clean them now.
 OMINSTPATHVALUE=`GetRegVal /etc/omreg.cfg openmanage.omilcore.installpath`
 CLIREGUTIL=${OMINSTPATHVALUE}/omil/tools/cli_ini_modifier.sh
 for i in $(ls ${OMINSTPATHVALUE}/omil/supportfiles/ini/) ; do
        [ -f $CLIREGUTIL ] && ($CLIREGUTIL ${OMINSTPATHVALUE}/omil/supportfiles/ini/$i del)
 done
 
# Now clean-up extra reference counted values which will not be called
# if CheckForRACInstall Function fails.
# Check for RacInstall uses 8gsyslist and 9gsyslist both of which are
# provided by omilcore
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac3/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac3/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/oma/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/oma/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac4/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac5/bin"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac3/libs"
        RemoveEntry /etc/ld.so.conf "${RPM_INSTALL_PREFIX}/rac3/libs"
	

}

##
## Sends an error message to stderr
##
ErrorBlock() {

  cat 1>&2
}


##
## Retrieve the text value from a filename and registry key
##
GetRegValText()
{
	FILE="${1}"
  	KEY="${2}"

  	[ ! -f "${FILE}" ] && echo "" && return 1

	# get the key value line in the file
  	GETVALUETEXT="`grep -i "^[[:space:]]*${KEY}[[:space:]]*=" ${FILE}`"
	# get the actual key value
  	VALUETEXT=`echo "${GETVALUETEXT}" | sed 's#^[^=]*=##; s#^[[:space:]]*##; s#[[:space:]]*$##'`
	# replace any _REPLACINGSPACE_ with spaces/blanks
	# as "for PAIR" in UpdateRegPair parses on spaces
    RPL_SPACE_ENTRY=`echo "${VALUETEXT}" | sed "s/$REPLACESPACEWITH/ /g"`
	echo "${RPL_SPACE_ENTRY}"
  return 0
}

##
## Retrieve the value from a filename and registry key
##
GetRegVal() {
  FILE="${1}"
  KEY="${2}"

  [ ! -f "${FILE}" ] && echo "" && return 1

  GetVal "`grep -i "^[[:space:]]*${KEY}[[:space:]]*=" ${FILE}`"
  return 0
}


##
## Retrieve the value portion from a key=value pair
##
GetVal() {
  PAIR="${1}"

  echo "${PAIR}" | sed 's#^[^=]*=##; s#^[[:space:]]*##; s#[[:space:]]*$##'
}


##
## Retrieve the key portion from a key=value pair
##
GetKey() {
  PAIR="${1}"

  echo "${PAIR}" | sed 's#=.*$##; s#^[[:space:]]*##; s#[[:space:]]*$##'
}


##
## Update a file with a key=value pair.
## adds the pair if it doesnt exist
## if the key already exists, append the value to the end of already existing 
## values (with blank space in between)
## updates the value with the current registry prefix if provided
##
UpdateRegSvcList() {
  FILE="${1}"
  PREFIX="${2}"
  shift
  shift

  svc_present="FALSE"

  # if the file doesnt exist, create it
  MakeFile "${FILE}"

  for PAIR in ${*} ;
  do
    TMP_KEY=`GetKey "${PAIR}"`
    TMP_VAL=`GetVal "${PAIR}"`

    if [ -n "${TMP_VAL}" -a -n "${PREFIX}" ];
    then
        NEW_VALUE="${PREFIX}/${TMP_VAL}"
    else
        NEW_VALUE="${PREFIX}${TMP_VAL}"
    fi

    #Check whether the service is already in the list.
    #If already present, do nothing.
    grep -qi "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE}
    key_present=$?
    if [[ $key_present == 0 ]]; then

       pattern=`grep -i "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE}`
       ret=`echo $pattern | awk -F"=" '{print $2}'`
       echo " $ret " | grep -qi " ${NEW_VALUE} "

       if [[ $? == 0 ]]; then
          svc_present="TRUE"
       fi
    fi

    if [[ $svc_present == "FALSE" ]]; then
       # strip old from the regentry, then add new
       TEMPFILE=`GetTemp`

       grep -iv "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} > ${TEMPFILE}
       if [[ $key_present == 0 ]]; then
           echo "`grep -i "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE}` $NEW_VALUE" >> ${TEMPFILE} &&
           sort ${TEMPFILE} > ${FILE} ||
           ErrorMsg "unable to update ${FILE}"
       else
           echo "${TMP_KEY}=${NEW_VALUE}" >> ${TEMPFILE} &&
           sort ${TEMPFILE} > ${FILE} ||
           ErrorMsg "unable to update ${FILE}"
       fi

       rm -f ${TEMPFILE}
    fi

  done
  return 0
}

##
## Update a file with a key=value pair.
## adds the pair if it doesnt exist
## replaces the pair if it does exist
## updates the value with the current registry prefix if provided
##
UpdateRegPair() {
  FILE="${1}"
  PREFIX="${2}"
  shift
  shift

  # if the file doesnt exist, create it
  MakeFile "${FILE}"

  for PAIR in ${*} ;
  do
    TMP_KEY=`GetKey "${PAIR}"`
    TMP_VAL=`GetVal "${PAIR}"`

    if [ -n "${TMP_VAL}" -a -n "${PREFIX}" ];
    then
        NEW_VALUE="${PREFIX}/${TMP_VAL}"
    else
        NEW_VALUE="${PREFIX}${TMP_VAL}"
    fi

    # strip old from the regentry, then add new
    TEMPFILE=`GetTemp`
    grep -iv "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} > ${TEMPFILE}
    echo "${TMP_KEY}=${NEW_VALUE}" >> ${TEMPFILE} &&
      sort ${TEMPFILE} > ${FILE} ||
        ErrorMsg "unable to update ${FILE}"
    rm -f ${TEMPFILE}
  done
  return 0
}

##
## Update a file with a key="value comment comment" pair that may contain spaces.
## Format of text:
##  %define ld_service_name_bmu_solproxy              "DSM BMU SOL Proxy"
## Format of pair value:
##  %define ld_service_name_bmu_solproxy_pair     %{ld_service_name_bmu_solproxy_key}="%{ld_service_name_bmu_solproxy}"
## Format of key/value pair passed in:
##   %{ld_service_name_bmu_solproxy_pair},%{ld_service_name_omsa_dcstor_pair}
## adds the pair if it doesn't exist
## replaces the pair if it does exist
## updates the value with the current registry prefix if provided
##
UpdateRegPairText()
{
	FILE="${1}"
	PREFIX="${2}"
	shift
	shift

	# echo "Old Parameters: '${1}'"

	# replace any spaces/blanks with _REPLACINGSPACE_
	# as "for PAIR" in UpdateRegPair parses on spaces
    RPL_SPACE_ENTRY=`echo "${1}" | sed "s/ /$REPLACESPACEWITH/g"`

	# replace any commas with spaces/blanks
    RPL_COMMA_ENTRY=`echo "${RPL_SPACE_ENTRY}" | sed "s/,/ /g"`
	# echo "New Parameters: '${RPL_COMMA_ENTRY}'"

    # call the update registry pair call
	RETURN_VALUE=`UpdateRegPair "${FILE}" "${PREFIX}" "${RPL_COMMA_ENTRY}"`

  	return ${RETURN_VALUE}
}


##
## Remove a service from the list of registry services.
## Example : RemoveRegSvc "/tmp/omreg.cfg" "upgrade.relocation=svc1"
## This removes the service "svc1" from the list of services.
## If no more services exist, the key also will be removed from the registry.
## if the file is empty after the deletion, the file is removed as well.
##
RemoveRegSvc() {
  FILE="${1}"
  shift

  [ ! -f "${FILE}" ] && return 0

  for PAIR in ${*} ;
  do
    TMP_KEY=`GetKey "${PAIR}"`
    TMP_VAL=`GetVal "${PAIR}"`

    # strip old from the regentry
    TEMPFILE=`GetTemp`
    grep -iv "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} > ${TEMPFILE}

    #Strip off $TMP_VAL from the service list
    New_SvcList=`grep -i "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} | awk -F"=" '{print $2}' | \
                       sed -e "s/ $TMP_VAL / /g" -e "s/^$TMP_VAL //g" -e "s/ $TMP_VAL$//g" \
                           -e "s/^$TMP_VAL$//g" -e 's/^[[:space:]]*//;s/[[:space:]]*$//'`

    if [[ -n $New_SvcList ]]; then
       echo "${TMP_KEY}=${New_SvcList}" >> ${TEMPFILE} 
    fi

    sort ${TEMPFILE} > ${FILE} ||
        ErrorMsg "unable to update ${FILE}"
    rm -f ${TEMPFILE}
  done

  # if now empty, remove the file
  [ ! -s "${FILE}" ] && rm -f ${FILE}
  return 0
}

##
## Remove a key from registry
## if a pair is provided, the value is stripped and the key alone is used
## if the file is empty after the deletion, the file is removed as well
##
RemoveRegPair() {
  FILE="${1}"
  shift

  [ ! -f "${FILE}" ] && return 0

  for PAIR in ${*} ;
  do
    TMP_KEY=`GetKey "${PAIR}"`

    # strip old from the regentry
    TEMPFILE=`GetTemp`
    grep -iv "^[[:space:]]*${TMP_KEY}[[:space:]]*=" ${FILE} > ${TEMPFILE}
    sort ${TEMPFILE} > ${FILE} ||
        ErrorMsg "unable to update ${FILE}"
    rm -f ${TEMPFILE}
  done

  # if now empty, remove the file
  [ ! -s "${FILE}" ] && rm -f ${FILE}
  return 0
}

##
## Remove a key from registry
## if a pair is provided, the value is stripped and the key alone is used
## if the file is empty after the deletion, the file is removed as well
##
RemoveRegPairText()
{
	FILE="${1}"
	shift

	[ ! -f "${FILE}" ] && return 0

	# replace any spaces/blanks with _REPLACINGSPACE_
	# as "for PAIR" in UpdateRegPair parses on spaces
    RPL_SPACE_ENTRY=`echo "${1}" | sed "s/ /$REPLACESPACEWITH/g"`

	# replace any commas with spaces/blanks
    RPL_COMMA_ENTRY=`echo "${RPL_SPACE_ENTRY}" | sed "s/,/ /g"`
	# echo "New Parameters: '${RPL_COMMA_ENTRY}'"

    # call the remove registry pair call
	RETURN_VALUE=`RemoveRegPair "${FILE}" "${RPL_COMMA_ENTRY}"`

  	return ${RETURN_VALUE}
}

##
## Add an entry to the config file
## Prevent duplicates
##
AddEntry() {
  LDFILE="${1}"
  shift

  MakeFile "${LDFILE}"

  for ENTRY in ${*} ;
  do
    # add once and ++ ref-count for duplicates
    OMREGKEY=`echo "${ENTRY}" | sed "s#/##g"`
    OMREGKEY="ldconf.path.${OMREGKEY}.rcount"
    OMREGVAL=`GetRegVal "/etc/omreg.cfg" "${OMREGKEY}"`
    if [ -n "${OMREGVAL}" ] && [ `echo ${OMREGVAL} | egrep '^[0-9]+$'` ]; then
      # ++ ref-count
      OMREGVAL=`expr ${OMREGVAL} + 1` &&
      UpdateRegPair "/etc/omreg.cfg" "" "${OMREGKEY}=${OMREGVAL}"
    else
      # add new entry, and sort
      UpdateRegPair "/etc/omreg.cfg" "" "${OMREGKEY}=1"
      PREVCNT=`grep -ic "^[[:space:]]*${ENTRY}[[:space:]]*" ${LDFILE}`
      if [ "$PREVCNT" = "0" ]; then
        TEMPFILE=`GetTemp`
        cat ${LDFILE} > ${TEMPFILE} &&
          echo "${ENTRY}" >> ${TEMPFILE} &&
            sort ${TEMPFILE} > ${LDFILE} ||
              ErrorMsg "unable to update ${LDFILE}"
        rm -f ${TEMPFILE}
      fi
    fi
  done
  ldconfig
  return 0
}


##
## Remove a single instance of a line from a config file
##
RemoveEntry() {
  FILE="${1}"
  shift

  [ ! -f "${FILE}" ] && return 0

  for ENTRY in ${*} ;
  do
    # remove entry or -- ref-count for duplicates
    OMREGKEY=`echo "${ENTRY}" | sed "s#/##g"`
    OMREGKEY="ldconf.path.${OMREGKEY}.rcount"
    OMREGVAL=`GetRegVal "/etc/omreg.cfg" "${OMREGKEY}"`
    if [ -n "${OMREGVAL}" ] && [ `echo ${OMREGVAL} | egrep '^[0-9]+$'` ] && [ ${OMREGVAL} -gt 1 ]; then
      # -- ref-count
      OMREGVAL=`expr ${OMREGVAL} - 1`
      UpdateRegPair "/etc/omreg.cfg" "" "${OMREGKEY}=${OMREGVAL}"
    else
      # strip old from the regentry
      OMREGFILE="/etc/omreg.cfg"
      PREVCNT=`grep -ic "^[[:space:]]*${OMREGKEY}[[:space:]]*" ${OMREGFILE}`
      [ "${PREVCNT}" = "0" ] && continue

      TEMPFILE=`GetTemp`
      grep -iv "^[[:space:]]*${OMREGKEY}[[:space:]]*" ${OMREGFILE} > ${TEMPFILE}
      sort ${TEMPFILE} > ${OMREGFILE} ||
        ErrorMsg "unable to update ${OMREGFILE}"
      rm -f ${TEMPFILE}

      # strip old from the regentry
      PREVCNT=`grep -ic "^[[:space:]]*${ENTRY}[[:space:]]*$" ${FILE}`
      [ "${PREVCNT}" = "0" ] && continue

      TEMPFILE=`GetTemp`
      grep -iv "^[[:space:]]*${ENTRY}[[:space:]]*$" ${FILE} > ${TEMPFILE}
      for (( i=1 ; i<${PREVCNT} ; i++ )) ;
      do
        echo "${ENTRY}" >> ${TEMPFILE}
      done
      sort ${TEMPFILE} > ${FILE} ||
        ErrorMsg "unable to update ${FILE}"
      rm -f ${TEMPFILE}
    fi
  done
  ldconfig
  return 0
}


##
## FixPath updates the contents of the oma/ini/omprvXX.ini file
## with the 32/64 and prefix -adjusted values
##
FixPath() {
  # FixPath %{ld_omreg_file} "oma/ini/omprvARCHTYPE.ini" %{ld_omreg_omacore_install_path_key} %{ld_package_name} "${RPM_INSTALL_PREFIX}" replace_om_hip_path
  FILE="${1}"
  INST_OMPRVINI="${2}"
  OMREG_OMACORE_INSTALL_PATH_KEY="${3}"
  PACKAGE_NAME="${4}"
  PREFIX="${5}"
  TOKEN="${6}"

  # some sanity checking...
  [ -z "${TOKEN}" ] && ErrorMsg "${PACKAGE_NAME}: FixPath missing TOKEN arg"
  [ -n "${8}" ] && ErrorMsg "${PACKAGE_NAME}: FixPath extra arg (${8})"
  for i in $* ; do
    [ -z "${i}" ] && ErrorMsg "${PACKAGE_NAME}: FixPath empty value for arg"
    echo "${i}" | grep -q "ld_" && ErrorMsg "${PACKAGE_NAME}: FixPath passed undefined value ($i)"
  done

  OMACOREINSTALPATH=`GetRegVal ${FILE} ${OMREG_OMACORE_INSTALL_PATH_KEY}`
  OMPRVINIFILE="${OMACOREINSTALPATH}/${INST_OMPRVINI}"

  if [ -z "${OMACOREINSTALPATH}" ];
  then

    ErrorBlock <<EOF
${PACKAGE_NAME}:
Warning: OMACOREINSTALPATH is empty : ${OMACOREINSTALPATH}
         Could not update the install path in ${OMPRVINIFILE}
         ${PACKAGE_NAME} may not work
EOF

  else

    TEMPFILE=`GetTemp`
    sed "s#${TOKEN}#${PREFIX}#g" ${OMPRVINIFILE} > ${TEMPFILE} &&
      cat ${TEMPFILE} > ${OMPRVINIFILE} ||
        ErrorMsg "${PACKAGE_NAME}: unable to update ${OMPRVINIFILE}"
    rm -f ${TEMPFILE}

  fi
}



##
## make a private copy of the omilcore Funcs.sh file that will
## stick around even after omilcore is removed
##
CreateShadowFuncs() {
  PACKAGE_NAME="${1}"
  PREFIX="${2}"
  ORIG_FUNCS="${3}"

  [ ! -f "${ORIG_FUNCS}" ] && ErrorMsg "${PACKAGE_NAME}: file does not exist ${ORIG_FUNCS}" && return 1
  mkdir -p "${PREFIX}/funcs"
  install --owner=root --group=root --mode=755 "${ORIG_FUNCS}" "${PREFIX}/funcs/${PACKAGE_NAME}.sh"
}


##
## remove private copy of Funcs.sh file, also remove parent directories if empty
##
RemoveShadowFuncs() {
  PACKAGE_NAME="${1}"
  PREFIX="${2}"

  rm -f "${PREFIX}/funcs/${PACKAGE_NAME}.sh"
  rmdir --ignore-fail-on-non-empty "${PREFIX}/funcs"
  rmdir --ignore-fail-on-non-empty "${PREFIX}" >/dev/null 2>&1
}


##
## return the path to the private copy of Funcs.sh
##
GetShadowFuncs() {
  PACKAGE_NAME="${1}"
  PREFIX="${2}"

  echo "${PREFIX}/funcs/${PACKAGE_NAME}.sh"
}


##
## Execute the sysid lookup utility
## exit with an error message if this systems sysid is not in the list of supported systems
## attempt to uninstall package if "UNINSTALL" is passed in arg5
## use alternate message if non-null value passed in arg5
##
ExitIfBadSysId() {
    FILE="${1}"
    OMREG_SYSIDCHECKUTIL_KEY="${2}"
    OMREG_SYSLISTFILE_KEY="${3}"
    PACKAGE_NAME="${4}"
    OPERATION="${5}"


    # if SYSID is to be ignored, then return success now!
    [ -n "${OMIIGNORESYSID}" ] && return 0

    [ ! -f "${FILE}" ] && ErrorMsg "${PACKAGE_NAME}: ${FILE} file is not found" && exit 1

    SYSIDCHECKPATH=`GetRegVal "${FILE}" "${OMREG_SYSIDCHECKUTIL_KEY}"`
    [ ! -f "${SYSIDCHECKPATH}" ] && ErrorMsg "${PACKAGE_NAME}: ${SYSIDCHECKPATH} file is not found" && exit 1

    SYSIDFILEPATH=`GetRegVal "${FILE}" "${OMREG_SYSLISTFILE_KEY}"`
    [ ! -f "${SYSIDFILEPATH}" ] && ErrorMsg "${PACKAGE_NAME}: ${SYSIDFILEPATH} file is not found" && exit 1

    # look the sysid up in the syslist file
    SYSID=`${SYSIDCHECKPATH} | grep "System ID" | sed 's#^.*0x##; s#[[:space:]].*$##'`
    VAL=`GetRegVal "${SYSIDFILEPATH}" "${SYSID}"`

    if [ -z "${VAL}" ] ;
    then
      if [ "${OPERATION}" = "UNINSTALL" ] ;
      then
        pid=$$
        while true ; do
          sleep 5s
          if [ ! -d /proc/${pid} ] ; then
            rpm --erase ${PACKAGE_NAME} instsvc-drivers
            exit 1
          fi
        done >/dev/null 2>&1 &
      fi
      ErrorMsg "${PACKAGE_NAME}: not supported on this system (sysid=${SYSID})" && exit 1
    fi

    return 0
}


##
## exit if the abort flag is found
##
ExitIfAbortFlag() {
    FILE="${1}"
    PACKAGE_NAME="${2}"

    [ -f "${FILE}" ] && ErrorMsg "${PACKAGE_NAME}: Install Disabled" && exit 1
    return 0
}


##
## will exit with error message if option is not installed or an exception in the 8G or 9G files
## pass the option (DRAC3/DRAC4/DRAC5) in arg5
##
ExitIfNotInstalled() {
    FILE="${1}"
    OMREG_SYSIDCHECKUTIL_KEY="${2}"
    OMREG_SYSLISTFILE_KEY="${3}"
    PACKAGE_NAME="${4}"
    OPTION="${5}"
    OMREG_8GSYSLISTFILE_KEY="${6}"
    OMREG_9GSYSLISTFILE_KEY="${7}"

    # if SYSID is to be ignored, then return success now!
    [ -n "${OMIIGNORESYSID}" ] && return 0

    SYSIDFILEPATH=`GetRegVal "${FILE}" "${OMREG_SYSLISTFILE_KEY}"`
    [ ! -f "${SYSIDFILEPATH}" ] && ErrorMsg "${PACKAGE_NAME}: file does not exist: ${SYSIDFILEPATH}" && exit 1

    #Allow DRAC3 installs on Non-8G systems, Block DRAC3 installs on 8G and 9G systems
    #Allow DRAC4 installs on 8G systems, Block DRAC4 installs on Non-8G systems
    #Allow DRAC5 installs on 9G systems, Block DRAC5 installs on Non-9G systems
    if [ -n "${OPTION}" ];
    then
        SYSIDFILEPATH8G=`GetRegVal "${FILE}" "${OMREG_8GSYSLISTFILE_KEY}"`
        [ ! -f "${SYSIDFILEPATH8G}" ] && ErrorMsg "${PACKAGE_NAME}: file does not exist: ${SYSIDFILEPATH8G}" && exit 1
        SYSIDFILEPATH9G=`GetRegVal "${FILE}" "${OMREG_9GSYSLISTFILE_KEY}"`
        [ ! -f "${SYSIDFILEPATH9G}" ] && ErrorMsg "${PACKAGE_NAME}: file does not exist: ${SYSIDFILEPATH9G}" && exit 1

        if [ "${OPTION}" == "DRAC3" ];
        then
            #Allow DRAC3 installs on Non-8G systems, Block DRAC3 installs on 8G or 9G systems
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`
            [ -n "${VAL}" ] && ErrorMsg "This system does not support ${OPTION}.  This package cannot be installed." && exit 1
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            [ -n "${VAL}" ] && ErrorMsg "This system does not support ${OPTION}.  This package cannot be installed." && exit 1

        elif [ "${OPTION}" == "DRAC4" ];
        then
            #Allow DRAC4 installs on 8G systems, Block DRAC4 insalls on Non-8G systems
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`
            [ -z "${VAL}" ] && ErrorMsg "This system does not support ${OPTION}.  This package cannot be installed." && exit 1

        elif [ "${OPTION}" == "DRAC5" ];
        then
            #Allow DRAC5 installs on 9G systems, Block DRAC5 insalls on Non-9G systems
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            [ -z "${VAL}" ] && ErrorMsg "This system does not support ${OPTION}.  This package cannot be installed." && exit 1
        fi
    fi
    return 0
}

UpgradeRelocationCheck() {
    UPGRADE_CHECK=`GetRegVal /etc/omreg.cfg upgrade.relocation`
    if [ -n "${UPGRADE_CHECK}" -a "${UPGRADE_CHECK}" == "bad" ]; then
      RemoveRegPair /etc/omreg.cfg "upgrade.relocation=bad"
      exit 1
    fi
}


##
## Relocation Path Check:
## If this package is getting relocated, ensure that it is getting relocated
## to a good destination
##
ExitOnRelocPathCheck() {
    FILE="${1}"
    OMREG_OMILCORE_INSTALL_PATH_KEY="${2}"
    OMILCORE="${3}"
    PREFIX="${4}"
    RELOCATION_PATH="${5}"
    PACKAGE_NAME="${6}"
    IS_UPGRADE="${7}"

    # restrict to absolute paths
    if ! echo "${PREFIX}" | grep -iq "^/" ; then
      # non-absolute
      LErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
Specified directory must be an absolute path.
EOF
      exit 1
    fi

    # restrict relocation path to non-space characters
    if echo "${PREFIX}" | egrep -iq "[[:space:]]" ; then
      # prefix contains non-space characters
      ErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
Path name cannot include spaces.
EOF
      exit 1
    fi

    # restrict relocation path to non-space characters
    if [ -e "${PREFIX}" -a ! -d "${PREFIX}" ] ; then
      # prefix is not a directory, but exists... so it is invalid
      ErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
The path exists, but it is not a directory.
EOF
      exit 1
    fi

    # restrict relocation path to alphanumeric characters
    if ! echo "${PREFIX}" | egrep -iq "^[-/._a-z0-9]+$" ; then
      # prefix contains non alphanumeric characters
      ErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
Path name is restricted to the characters: ., -, _, 0-9, a-z and A-Z
EOF
      exit 1
    fi

    REJECTED_PATHS="/etc /proc /dev /lost\+found"
    for e in ${REJECTED_PATHS} ; do
      if echo "${PREFIX}" | egrep -q "(^${e}/)|(^${e}$)" ; then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
Path name cannot exist under the directory ${e}.
EOF
        exit 1
      fi
    done

    return 0
}

##
## Relocation Check:
## If this package is getting relocated, ensure that it is getting relocated
## to the same path where the rest of the Systems Management packages are.
##
ExitOnRelocCheck() {
    FILE="${1}"
    OMREG_OMILCORE_INSTALL_PATH_KEY="${2}"
    OMILCORE="${3}"
    PREFIX="${4}"
    RELOCATION_PATH="${5}"
    PACKAGE_NAME="${6}"
    IS_UPGRADE="${7}"

    OMILCOREINSTPATH=`GetRegVal ${FILE} ${OMREG_OMILCORE_INSTALL_PATH_KEY}`
    if [ -z "${OMILCOREINSTPATH}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Could not obtain the install path for ${OMILCORE} package : ${OMREG_OMILCORE_INSTALL_PATH_KEY}
This package cannot be installed.
EOF
        exit 1
    else
        if [ "${OMILCOREINSTPATH}" != "${PREFIX}" ]; then
            if [ "${RELOCATION_PATH}" != "${PREFIX}" ]; then
                ErrorBlock <<EOF
${PACKAGE_NAME}:
Invalid --prefix : ${PREFIX}
Systems Management is installed at : ${OMILCOREINSTPATH}
Cannot continue, please change --prefix to : ${OMILCOREINSTPATH}
EOF
            else
                # not relocating, using the default path
                ErrorBlock <<EOF
${PACKAGE_NAME}:
RPM option [--prefix] is required
Systems Management is installed at : ${OMILCOREINSTPATH}
Cannot continue, please reinstall with --prefix ${OMILCOREINSTPATH}
EOF
            fi

            # on an upgrade with bad relocation, prevent previous packages from getting deleted.
            if [ -n "${IS_UPGRADE}" -a "${IS_UPGRADE}" == "upgrade" ]; then
              UpdateRegPair "/etc/omreg.cfg" "" "upgrade.relocation=bad"
            fi
            exit 1
        else
            if [ -n "${IS_UPGRADE}" -a "${IS_UPGRADE}" == "upgrade" ]; then
              RemoveRegPair /etc/omreg.cfg "upgrade.relocation=bad"
            fi
        fi
    fi

    return 0
}


##
## uninstall dep check:
## If this package is getting relocated, ensure that it is getting relocated
## to the same path where the rest of the Systems Management packages are.
##
ExitIfOtherRPMSUninstalled() {
    PACKAGE_NAME="${1}"
    PACKAGE_PREFIX="${2}"
    # because other serveradministrator packages depend on the payload of this package, it must be
    # uninstalled last.  this addresses the behavior of rpm which will uninstall packages in
    # random order despite prereq and depends statements

    if sed "s#${PACKAGE_NAME}##g" /proc/${PPID}/cmdline | grep -q "${PACKAGE_PREFIX}" ;
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Package cannot be uninstalled at the same time with other ${PACKAGE_PREFIX}
packages. Retry the uninstall operation separately.
EOF
        exit 1
    fi

    return 0
}

## Execute the sysreport utility
## return value as '1' if dell 
##
CheckDellServer() 
{
    # if sysid check is disabled, then use the value passed in
    [ -n "${OMIIGNORESYSID}" ] && echo  "${OMIIGNORESYSID}" && return 0

    FILE="/etc/omreg.cfg"
    OMREG_SYSIDCHECKUTIL_KEY="openmanage.sysidchkutlpath"

    [ ! -f "${FILE}" ] && return 1

    #Get path to utility
    SYSIDCHECKPATH=`GetRegVal "${FILE}" "${OMREG_SYSIDCHECKUTIL_KEY}"`
    [ ! -f "${SYSIDCHECKPATH}" ] && return 1

    # execute system id utility
    VENDOR=`${SYSIDCHECKPATH} | grep "Vendor"` 
    RESULT=`echo $VENDOR | grep -i "dell"`

    if [ -n "${RESULT}" ] 
    then
      echo "1" 
     else
      echo "0" 
    fi

    return 0
}

##
## Execute the sysid utility
## return system id
##
GetSysId() {
    # if sysid check is disabled, then use the value passed in
    [ -n "${OMIIGNORESYSID}" ] && echo  "${OMIIGNORESYSID}" && return 0

    FILE="/etc/omreg.cfg"
    OMREG_SYSIDCHECKUTIL_KEY="openmanage.sysidchkutlpath"

    [ ! -f "${FILE}" ] && return 1

    SYSIDCHECKPATH=`GetRegVal "${FILE}" "${OMREG_SYSIDCHECKUTIL_KEY}"`
    [ ! -f "${SYSIDCHECKPATH}" ] && return 1

    # execute system id utility
    SYSID=`${SYSIDCHECKPATH} | grep "System ID" | sed 's#^.*0x##; s#[[:space:]].*$##'`

    [ -n "${SYSID}" ] && echo "${SYSID}" || return 1

    return 0
}

##
## Verify proper command to install an init script exists
##
VerifyInstallInitScript()
{
    PACKAGE_NAME="${1}"

    # check for lsb install
    if [ ! -x /usr/lib/lsb/install_initd ];
    then
        # no lsb install
        # check for Red Hat style install
        if [ ! -x /sbin/chkconfig ];
        then
            # no Red Hat style install
            # this is seriously broken
            ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script installers: /usr/lib/lsb/install_initd or /sbin/chkconfig
could not be found or do not have executable permissions.
This package cannot be installed.
EOF
            exit 1
        fi
    fi

    if [ -n "${OMIDEBUG}" ];
    then
        echo ""
        # check for lsb install
        if [ -x /usr/lib/lsb/install_initd ];
        then
            # this is an lsb install
            echo "Using /usr/lib/lsb/install_initd to install init scripts"
        else
            # this is a Red Hat type install
            echo "Using /sbin/chkconfig to install init scripts"
        fi
        echo ""
    fi

    return 0
}


##
## Execute proper command to install an init script
##
InstallInitScript()
{
    PACKAGE_NAME="${1}"
    INIT_SCRIPT_NAME="${2}"

    # sanity check, was a non-NULL init script name passed
    if [ -z "${INIT_SCRIPT_NAME}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script name passed in, is NULL.
This package cannot be installed.
EOF
        exit 2
    fi

    # sanity check, was a valid init script name passed
    if [ ! -e /etc/init.d/"${INIT_SCRIPT_NAME}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script name passed in, does not exist.
This package cannot be installed.
Verify that /etc/init.d/${INIT_SCRIPT_NAME} exists.
EOF
        exit 3
    fi

    # check for lsb install
    if [ -x /usr/lib/lsb/install_initd ];
    then
        # this is an lsb install
        /usr/lib/lsb/install_initd /etc/init.d/${INIT_SCRIPT_NAME} >/dev/null 2>&1
    elif [ -x /sbin/chkconfig ];
    then
        # this is a Red Hat type install
        /sbin/chkconfig --add ${INIT_SCRIPT_NAME}
    else
        # this is seriously broken
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script installers: /usr/lib/lsb/install_initd or /sbin/chkconfig
could not be found or do not have executable permission.
The /etc/init.d/${INIT_SCRIPT_NAME} cannot be installed.
This package cannot be installed.
EOF
        exit 1
    fi

    return 0
}

##
## Execute proper command to delete an init script
##
UnInstallInitScript()
{
    PACKAGE_NAME="${1}"
    INIT_SCRIPT_NAME="${2}"

    # sanity check, was a non-NULL init script name passed
    if [ -z "${INIT_SCRIPT_NAME}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script name passed in, is NULL.
EOF
        exit 2
    fi

    # sanity check, was a valid init script name passed
    if [ ! -e /etc/init.d/"${INIT_SCRIPT_NAME}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script name passed in, does not exist.
Verify that /etc/init.d/${INIT_SCRIPT_NAME} exists.
EOF
        return 3
    fi

    if [ -n "${OMIDEBUG}" ];
    then
        echo ""
        # check for lsb uninstall
        if [ -x /usr/lib/lsb/remove_initd ];
        then
            # this is an lsb uninstall
            echo "Using /usr/lib/lsb/remove_initd to uninstall init scripts"
        else
            # this is a Red Hat type install
            echo "Using /sbin/chkconfig to uninstall init scripts"
        fi
        echo ""
    fi

    # check for lsb uninstall
    if [ -x /usr/lib/lsb/remove_initd ];
    then
        # this is an lsb uninstall
         /usr/lib/lsb/remove_initd /etc/init.d/${INIT_SCRIPT_NAME} >/dev/null 2>&1
    elif [ -x /sbin/chkconfig ];
    then
        # this is a Red Hat type uninstall
        /sbin/chkconfig --del ${INIT_SCRIPT_NAME}
    else
        # this is seriously broken
        ErrorBlock <<EOF
${PACKAGE_NAME}:
The init script uninstallers: /usr/lib/lsb/remove_initd or /sbin/chkconfig
could not be found or do not have executable permission.
The /etc/init.d/${INIT_SCRIPT_NAME} could not be properly uninstalled.
EOF
        return 1
    fi

    return 0
}

##
## Check if the system that OMI running on, is the passed in generation number
##
IsThisCorrectGeneration()
{
    FILE="${1}"
    OMREG_SYSIDCHECKUTIL_KEY="${2}"
    OMREG_SYSLISTFILE_KEY="${3}"
    PACKAGE_NAME="${4}"
    OPTION="${5}"
    OMREG_8GSYSLISTFILE_KEY="${6}"
    OMREG_9GSYSLISTFILE_KEY="${7}"
    EXIT_ON_FILE_ERROR="${8}"

    # if SYSID is to be ignored, then return success now!
    [ -n "${OMIIGNORESYSID}" ] && return 0

    SYSIDFILEPATH=`GetRegVal "${FILE}" "${OMREG_SYSLISTFILE_KEY}"`
    if [ ! -f "${SYSIDFILEPATH}" ];
    then
         ErrorMsg "${PACKAGE_NAME}: required file entry, ${OMREG_SYSLISTFILE_KEY}, does not exist in ${FILE}"
         if [ 0 = "${EXIT_ON_FILE_ERROR}" ];
         then
            return 1
         else
            exit 1
         fi
    fi

    # Check the option passed in
    if [ -n "${OPTION}" ];
    then
        SYSIDFILEPATH8G=`GetRegVal "${FILE}" "${OMREG_8GSYSLISTFILE_KEY}"`
        if [ ! -f "${SYSIDFILEPATH8G}" ];
        then
             ErrorMsg "${PACKAGE_NAME}: required file entry, ${OMREG_8GSYSLISTFILE_KEY}, does not exist in ${FILE}"
            if [ 0 = "${EXIT_ON_FILE_ERROR}" ];
            then
                return 1
            else
                exit 1
            fi
        fi


        SYSIDFILEPATH9G=`GetRegVal "${FILE}" "${OMREG_9GSYSLISTFILE_KEY}"`
        if [ ! -f "${SYSIDFILEPATH9G}" ];
        then
             ErrorMsg "${PACKAGE_NAME}: required file entry, ${OMREG_9GSYSLISTFILE_KEY}, does not exist in ${FILE}"
            if [ 0 = "${EXIT_ON_FILE_ERROR}" ];
            then
                return 1
            else
                exit 1
            fi
        fi

        if [ "${OPTION}" == "PRE8G" ];
        then
            # check for a pre-8th generation system, return 1 if not a pre-8th generation system
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`
            [ -n "${VAL}" ] && return 1
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            [ -n "${VAL}" ] && return 1

        elif [ "${OPTION}" == "8G" ];
        then
            # check for an 8th generation system, return 1 if not an 8th generation system
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`
            [ -z "${VAL}" ] && return 1

        elif [ "${OPTION}" == "9G" ];
        then
            # check for a 9th generation system, return 1 if not a 9th generation system
            SYSID=`GetSysId` && VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            [ -z "${VAL}" ] && return  1
        fi
    fi
    return 0
}

############################################################
# Verify the version attempting to be installed is the same
# version that is in omreg.cfg. Some Linux RPM utilities
# will exit %pre with an error and retain dependency
# information too long. This causes RPM's being installed
# immediately following, to use wrong dependency information
############################################################
VerifyVersion()
{
    FILE="${1}"
    VERSION_STRING="${2}"
    PACKAGE_NAME="${3}"

    OMREG_VERSION_STRING_KEY="openmanage.version"

    # insure the OM registry file exists, exit if not
    if [ ! -f "${FILE}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Missing file! This rpm, "${PACKAGE_NAME}", depends on
version "${VERSION_STRING}" of "srvadmin-omilcore" being installed.
The file, "${FILE}", does not exist. This file, when present, contains
required installation information needed to install this rpm.
This indicates that the required version, "${VERSION_STRING}" of
"srvadmin-omilcore" is not installed. This rpm has not been installed.
EOF
        exit 1
    fi

    # insure the OM registry file's version entry exists, exit if not
    VERSIONPATH=`GetRegVal "${FILE}" "${OMREG_VERSION_STRING_KEY}"`
    if [ -z "${VERSIONPATH}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Missing file information! This rpm, "${PACKAGE_NAME}", depends on
version "${VERSION_STRING}" of "srvadmin-omilcore" being installed.
The file, "${FILE}", does not contain the required "srvadmin-omilcore"
version information needed to install this rpm.
This indicates that the required version, "${VERSION_STRING}" of
"srvadmin-omilcore" is not installed. This rpm has not been installed.
EOF
        UpdateRegPair "${FILE}" "" "upgrade.relocation=bad"
        exit 1
    fi

    # Verify the versions match, exit if not
    if [ "${VERSIONPATH}" != "${VERSION_STRING}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Dependency failure! This rpm, "${PACKAGE_NAME}", depends on
version "${VERSION_STRING}" of "srvadmin-omilcore" being installed.
The installed "srvadmin-omilcore" version number, "${VERSIONPATH}",
does not match the version number required by this rpm.
This indicates that the required version of "srvadmin-omilcore" is
not installed. This rpm has not been installed.
EOF
        UpdateRegPair "${FILE}" "" "upgrade.relocation=bad"
        exit 1
    fi

    return 0
}

############################################################
# Preamble function call for preuninstall processing.
# In case, any is needed.
############################################################
PreunPreamble()
{
    PACKAGE_NAME="${1}"
    UPGRADE_IF_ONE="${2}"

    return 0
}

############################################################
# Verify the version dependencies
############################################################
VerifyVersionDependencies()
{
    FILE="${1}"
    VERSION_STRING="${2}"
    PACKAGE_NAME="${3}"
    DEPENDENCY_NAME="${4}"
    DEPENDENCY_VERSION="${5}"
    DEPENDENCY_KEY="${6}"

    # insure the OM registry file exists, exit if not
    if [ ! -f "${FILE}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Missing file! This rpm, "${PACKAGE_NAME}", depends on
version "${DEPENDENCY_VERSION}" of "${DEPENDENCY_NAME}" being installed.
The file, "${FILE}", does not exist. This file, when present, contains
required installation information needed to install this rpm.
This indicates that the "srvadmin-omilcore" rpm is not installed.
This rpm has not been installed.
EOF
        exit 1
    fi

    # insure the OM registry file's version entry exists, exit if not
    VERSIONPATH=`GetRegVal "${FILE}" "${DEPENDENCY_KEY}"`
    if [ -z "${VERSIONPATH}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Missing file information! This rpm, "${PACKAGE_NAME}", depends on
version "${DEPENDENCY_VERSION}" of "${DEPENDENCY_NAME}" being installed.
The file, "${FILE}", does not contain the required "${DEPENDENCY_NAME}"
version information needed by this rpm to install.
This indicates that the required version of "${DEPENDENCY_NAME}" is
not installed. This rpm has not been installed.
EOF
        exit 1
    fi

    # Verify the versions match, exit if not
    if [ "${VERSIONPATH}" != "${DEPENDENCY_VERSION}" ];
    then
        ErrorBlock <<EOF
${PACKAGE_NAME}:
Dependency failure! This rpm, "${PACKAGE_NAME}", depends on
version "${DEPENDENCY_VERSION}" of "${DEPENDENCY_NAME}" being installed.
The installed "${DEPENDENCY_NAME}" version number, "${VERSIONPATH}",
does not match the version number required by this rpm.
This indicates that the required version of "${DEPENDENCY_NAME}" is
not installed. This rpm has not been installed.
EOF
        exit 1
    fi

    return 0
}

############################################################
# Preamble function call for preuninstall processing.
# In case, any is needed.
############################################################
PreunPreamble()
{
    PACKAGE_NAME="${1}"
    UPGRADE_IF_ONE="${2}"

    return 0
}

#checks whether a sysid belongs to a blade server
#prints a 1 if blade, else a null
BladeServer () 
{
   OMREG_FILE="/etc/omreg.cfg"
   SYSLIST_KEY="openmanage.syslistfile"   

   SYSID=`GetSysId`
   SYSLISTFILE=`GetRegVal "${OMREG_FILE}" "${SYSLIST_KEY}"`

awk -v found=0 -v pat="$SYSID" -F= '
/\[BladeSystems\]/ {found = 1} 
{ if((found == "1") && ($1 ~ pat)) { print "1";found=0 }}
' $SYSLISTFILE

}

CheckRACInstall() {
    FILE="${1}"
    OMREG_SYSIDCHECKUTIL_KEY="${2}"
    OMREG_SYSLISTFILE_KEY="${3}"
    PACKAGE_NAME="${4}"
    OPTION="${5}"
    OMREG_8GSYSLISTFILE_KEY="${6}"
    OMREG_9GSYSLISTFILE_KEY="${7}"

    # if SYSID is to be ignored, then return success now!
    [ -n "${OMIIGNORESYSID}" ] && return 0

    SYSIDFILEPATH=`GetRegVal "${FILE}" "${OMREG_SYSLISTFILE_KEY}"`

    #Allow DRAC3 installs on Non-8G systems, Block DRAC3 installs on 8G and 9G systems
    #Allow DRAC4 installs on 8G systems, Block DRAC4 installs on Non-8G systems
    #Allow DRAC5 installs on 9G systems, Block DRAC5 installs on Non-9G systems

    if [ -n "${OPTION}" ];
    then
        SYSIDFILEPATH8G=`GetRegVal "${FILE}" "${OMREG_8GSYSLISTFILE_KEY}"`

        SYSIDFILEPATH9G=`GetRegVal "${FILE}" "${OMREG_9GSYSLISTFILE_KEY}"`

        if [ "${OPTION}" == "DRAC3" ];
        then
            #Allow DRAC3 installs on Non-8G systems, Block DRAC3 installs on 8G or 9G systems
            SYSID=`GetSysId` 
            VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`
            if [ -n "${VAL}" ] 
            then return 1
            fi  

            VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            if [ -n "${VAL}" ] 
            then return 1
            fi  
            return 0

        elif [ "${OPTION}" == "DRAC4" ];
        then
            #Allow DRAC4 installs on 8G systems, Block DRAC4 installs on Non-8G systems
            SYSID=`GetSysId` 
            VAL=`GetRegVal "${SYSIDFILEPATH8G}" "${SYSID}"`

            if [ -z "${VAL}" ]; then
               if [ ${SYSID} != "023C" ]; then
                  return 1
               fi
            fi

            return 0

        elif [ "${OPTION}" == "DRAC5" ];
        then
            #Allow DRAC5 installs on 9G systems, Block DRAC5 insalls on Non-9G systems
            SYSID=`GetSysId` 
	    VAL=`GetRegVal "${SYSIDFILEPATH9G}" "${SYSID}"`
            if [ -z "${VAL}" ] 
            then return 1
            fi  
            return 0
        fi
    fi
    return 1
}

##
##
##
##
## test code, pass 'test' as arg1 to execute
##
if [ "${1}" = "test" ] ; then

MYREG=`GetTemp`
MYENT=`GetTemp`

ErrorMsg "here is an error message"

echo "Here is the value for ABC: [`GetRegVal ${MYREG} ABC`]"
echo "Here is the key for ABC=DEF=GHI: [`GetKey ABC=DEF=GHI`]"
echo "Here is the val for ABC=DEF=GHI: [`GetVal ABC=DEF=GHI`]"

UpdateRegPair ${MYREG} "PREF" "DEF=123" "GHI=/somthing/file"
echo "Here is the value for DEF: [`GetRegVal ${MYREG} DEF`]"
echo "Here is the value for GHI: [`GetRegVal ${MYREG} GHI`]"
UpdateRegPair ${MYREG} "PREF" "DEF=999" "GHI=/something/dir"
echo "Here is the value for DEF: [`GetRegVal ${MYREG} DEF`]"
echo "Here is the value for GHI: [`GetRegVal ${MYREG} GHI`]"

RemoveRegPair ${MYREG} "UNK=999"
RemoveRegPair ${MYREG} "DEF=999"
echo "Here is the value for DEF: [`GetRegVal ${MYREG} DEF`]"
echo "Here is the value for GHI: [`GetRegVal ${MYREG} GHI`]"
RemoveRegPair ${MYREG} "DEF=999" "GHI=/something/dir"
echo "Here is the value for DEF: [`GetRegVal ${MYREG} DEF`]"
echo "Here is the value for GHI: [`GetRegVal ${MYREG} GHI`]"
RemoveRegPair ${MYREG} "UNK=999"

echo "======== Current Reg =========="
cat ${MYREG} 2>/dev/null || echo "(no file)"
echo "==============================="

AddEntry ${MYENT} entry1
AddEntry ${MYENT} entry2
AddEntry ${MYENT} entry2
AddEntry ${MYENT} entry3
AddEntry ${MYENT} entry2

echo "======== Current entries  =========="
cat ${MYENT} 2>/dev/null || echo "(no file)"
echo "===================================="

RemoveEntry ${MYENT} entry2
RemoveEntry ${MYENT} entry1
RemoveEntry ${MYENT} entry1
RemoveEntry ${MYENT} entry1

echo "======== Current entries  =========="
cat ${MYENT} 2>/dev/null || echo "(no file)"
echo "===================================="


ExitIfAbortFlag /abort "TestPackage"
echo "result is ${?}"
fi
