##################################
#
# ARCH mapping (from uname -s):
# - HP-UX           = HP-UX
# - Solaris         = SunOS
# - Linux           = Linux
# - SGI IRIX        = IRIX
# - IBM AIX         = AIX
# - SCO UnixWare    = UnixWare
# - SCO OpenServer  = SCO_SV
# - Tru64           = OSF1
# - Mac OS X        = Darwin
#
##################################


get_inet_port()
{
    # determine inet port, 5555 is the default
    # the search order is: newconfig/client/customize/socket, /etc/services, global
    INET_PORT=""
    SOCKET_DAT=/tmp/omni_tmp/socket.dat
    if [ -f ${SOCKET_DAT} ]; then
        INET_PORT=`cat ${SOCKET_DAT}`
    elif [ -f ${NEWCONFIG}/${OMNICONFIGCLIENT}/customize/socket ]; then
        INET_PORT=`awk '{print}' ${NEWCONFIG}/${OMNICONFIGCLIENT}/customize/socket`
    else
        INET_PORT=`awk '{if($1 == "'${SERVICE_NAME}'") {print $2; exit}}' ${SERVICES_FILE} | cut -d'/' -f1`
        if [ -z "$INET_PORT" -a -f ${OMNICONFIG}/options/global ]; then
            INET_PORT=`awk '{if ( $1 ~ /^[pP][oO][rR][tT]/){split($0,a,"=");print a[2]} else{;}}' ${OMNICONFIG}/options/global`
        fi
    fi

    if [ -z "$INET_PORT" ]; then
        INET_PORT=5555
    fi
}


get_sysconfig()
{
    DPSVC_LOG=/tmp/dpsvcsetup.log
    echo > ${DPSVC_LOG}

    ### I. get OS settings
    ARCH=`uname -s`
    if [ "${ARCH}" = "SunOS" ]; then
        MACHINE=`uname -p`
        AWK=nawk
    else
        MACHINE=`uname -m`
        AWK=awk
    fi
    OS_REV=`uname -r`
    if [ "${ARCH}" = "AIX" -o "${ARCH}" = "SCO_SV" ]; then
        IFCONFIG=/etc/ifconfig
    elif [ "${ARCH}" = "IRIX" ]; then
        IFCONFIG=/usr/etc/ifconfig
    elif [ "${ARCH}" = "UnixWare" ]; then
        IFCONFIG=/usr/sbin/ifconfig
    else
        IFCONFIG=/sbin/ifconfig
    fi
    SERVICES_FILE="/etc/services"
    SERVICE_NAME="omni"
    INETD_TYPE=""

    case "${ARCH}" in
        HP-UX|SunOS|Linux)
            OMNIHOME="/opt/omni"
            OMNIINET_PATH="/opt/omni/lbin/inet"
            OMNIINET_LOG_PATH="/var/opt/omni/log/inet.log"
            ;;
        *)
            OMNIHOME="/usr/omni"
            OMNIINET_PATH="/usr/omni/bin/inet"
            OMNIINET_LOG_PATH="/usr/omni/log/inet.log"
            ;;
    esac
    NEWCONFIG=${OMNIHOME}/newconfig
    OMNICONFIGBASE=/etc${OMNIHOME}
    OMNICONFIG=${OMNICONFIGBASE}/server
    OMNICONFIGCLIENT=${OMNICONFIGBASE}/client

    get_inet_port


    ### II. inetd checks
    if [ "${ARCH}" = "Darwin" ]; then
        # 1a. special handling on Mac OS X - uses launchd instead of init/inetd etc.
        INETD_TYPE="launchd"
        INETD_FILE="/System/Library/LaunchDaemons/omni.plist"
    else
        # 1. check for xinetd
        if [ -f /etc/xinetd.conf ]; then
            INETD_TYPE="xinetd"
            INETD_INCDIR=`grep '^includedir' /etc/xinetd.conf | ${AWK} '{print $2}'`
            # 1a. includedir present, use separate file there
            if [ "${INETD_INCDIR}" ]; then
                INETD_FILE="${INETD_INCDIR}/${SERVICE_NAME}"
            # 1b. direct configuration via xinetd.conf
            else
                INETD_FILE="/etc/xinetd.conf"
            fi
        fi

        # 2. check for inetd if xinetd not found
        if [ -z "${INETD_TYPE}" -a -f /etc/inetd.conf ]; then
            INETD_TYPE="inetd"
            INETD_FILE="/etc/inetd.conf"
            if [ "${ARCH}" = "SunOS" ]; then
                OS_MAJVER=`echo ${OS_REV} | cut -f1 -d.`
                OS_MINVER=`echo ${OS_REV} | cut -f2 -d.`
                # Solaris 10+ : convert inetd.conf entries using inetconv + smf
                if [ ${OS_MAJVER} -gt 5 -o ${OS_MINVER} -gt 9 ]; then
                    INETD_TYPE="inetd_solaris_smf"
                    INETD_FILE="/tmp/dp_inetd.conf"
                fi
            fi
        fi
    fi

    if [ -z "${INETD_TYPE}" ]; then
        echo "[ERROR]: No inetd/xinetd configuration found!" >&2
        exit 1
    fi


    ### III. TCP stack check
    case "${ARCH}" in
        Linux|SunOS|AIX|OSF1|IRIX|UnixWare|SCO_SV|Darwin)
            IFCBUF=`${IFCONFIG} -a | grep inet`
            IFCV4=`echo ${IFCBUF} | grep 'inet '`
            IFCV6=`echo ${IFCBUF} | grep inet6`
            ;;
        HP-UX)
            IFCV4=`netstat -i -f inet`
            IFCV6=`netstat -i -f inet6`
            ;;
        *)
            ;;
    esac

    if [ "${IFCV4}" -a "${IFCV6}" ]; then
        TCP_TYPE="dual"
    elif [ "${IFCV6}" ]; then
        TCP_TYPE="ipv6"
    else
        if [ -z "${IFCV4}" ]; then
            echo "[INFO_WARN]: Unknown TCP/IP configuration, using IPv4!" >> ${DPSVC_LOG}
        fi
        TCP_TYPE="ipv4"
    fi
}


restart_inetd()
{
    if [ $MODE = mode_check ]; then
        return
    fi

    if [ "${ARCH}" = "Linux" ]; then
        KILL_BY_PID=0
        if [ "${INETD_TYPE}" = "xinetd" ]; then
            if [ -x /etc/init.d/xinetd ]; then
                /etc/init.d/xinetd restart >> ${DPSVC_LOG}
                for I in 1 2 3 4 5
                do
                if [ "`netstat -a |grep "omni"`" == "" ]; then
                   sleep 1
                fi
                done
                sleep 1
            elif [ -x /etc/rc.d/init.d/xinetd ]; then
                /etc/rc.d/init.d/xinetd restart >> ${DPSVC_LOG}
                for I in 1 2 3 4 5
                do
                if [ "`netstat -a |grep "omni"`" == "" ]; then
                   sleep 1
                fi
                done
                sleep 1
            else
                KILL_BY_PID=1
            fi
        else
            if [ -x /etc/init.d/inetd ]; then
                /etc/init.d/inetd restart >> ${DPSVC_LOG}
            elif [ -x /etc/rc.d/init.d/inetd ]; then
                /etc/rc.d/init.d/inetd restart >> ${DPSVC_LOG}
            else
                KILL_BY_PID=1
            fi
        fi
    else
        KILL_BY_PID=1
    fi

    if [ $KILL_BY_PID = 1 ]; then
        INETDPID=`ps -e | egrep '[/ ]*inetd'| grep -v grep | ${AWK} '{print $1}'`
        if [ "${INETDPID}" ]; then
            echo "[INFO] Restarting inetd" >> ${DPSVC_LOG}
            kill -HUP $INETDPID
        else
            echo "[INFO_WARN] (x)inetd not running!" >> ${DPSVC_LOG}
        fi
    fi
}


omniinet_install()
{
    # 1. update /etc/services:
    if [ $MODE = mode_check ]; then
        OUT_SERVICES_FILE="/dev/tty"    # redirect output to console
        OUT_INETD_FILE="/dev/tty"       # redirect output to console
    else
        OUT_SERVICES_FILE=${SERVICES_FILE}
        OUT_INETD_FILE=${INETD_FILE}
    fi
    SVCFOUND=`grep '^'${SERVICE_NAME}' .*tcp' ${SERVICES_FILE}`

    if [ -z "${SVCFOUND}" ]; then
        echo "[INFO] Updating ${SERVICES_FILE}" >> ${DPSVC_LOG}
        echo "${SERVICE_NAME}             ${INET_PORT}/tcp    # DATA-PROTECTOR" >> ${OUT_SERVICES_FILE}
    else
        echo "[INFO_WARN] ${SERVICE_NAME} found in ${SERVICES_FILE}, skipping..." >> ${DPSVC_LOG}
    fi


    # 2. update/create /etc/inetd.conf or /etc/xinetd.conf or /etc/xinetd.d/<service_name>, depending on UNIX flavour
    echo "[INFO] Updating ${INETD_FILE}" >> ${DPSVC_LOG}
    case "${INETD_TYPE}" in
        xinetd)
            if [ "${TCP_TYPE}" != "ipv4" -o "${TCP_TYPE}" = "dual" ]; then
                SVCFLAGS="flags               = IPV6"
            fi

            echo "service ${SERVICE_NAME}
{
    socket_type         = stream
    protocol            = tcp
    wait                = no
    user                = root
    server              = ${OMNIINET_PATH}
    server_args         = inet -log ${OMNIINET_LOG_PATH}
    disable             = no
    ${SVCFLAGS}
    instances           = UNLIMITED
    per_source          = UNLIMITED
}" >> ${OUT_INETD_FILE}
        ;;
        inetd*)
            # Linux inetd.conf (Debian) needs tcp4 for dualstack, others need tcp
            if [ "${ARCH}" = "Linux" -a "${TCP_TYPE}" = "dual" ]; then
                TCP4_STR="tcp4"
            else
                TCP4_STR="tcp"
            fi

            if [ "${ARCH}" = "UnixWare" ]; then
                TCP6_STR="tcp"
            else
                TCP6_STR="tcp6"
            fi

            if [ "${TCP_TYPE}" = "ipv6" ]; then
                echo "${SERVICE_NAME} stream ${TCP6_STR} nowait root ${OMNIINET_PATH} inet -log ${OMNIINET_LOG_PATH}" >> ${OUT_INETD_FILE}
            elif [ "${TCP_TYPE}" = "dual" ]; then
                # HP-UX, SCO UnixWare, AIX and OSF1/Tru64 require only tcp(6) for either IPv6-only or dual-stack
                # others need two separate entries, one for each IP version
                if [ "${ARCH}" != "HP-UX" -a "${ARCH}" != "UnixWare" -a "${ARCH}" != "AIX" -a "${ARCH}" != "OSF1" ]; then
                    echo "${SERVICE_NAME} stream ${TCP4_STR} nowait root ${OMNIINET_PATH} inet -log ${OMNIINET_LOG_PATH}" >> ${OUT_INETD_FILE}
                fi
                echo "${SERVICE_NAME} stream ${TCP6_STR} nowait root ${OMNIINET_PATH} inet -log ${OMNIINET_LOG_PATH}" >> ${OUT_INETD_FILE}
            else
                echo "${SERVICE_NAME} stream ${TCP4_STR} nowait root ${OMNIINET_PATH} inet -log ${OMNIINET_LOG_PATH}" >> ${OUT_INETD_FILE}
            fi

            if [ "${INETD_TYPE}" = "inetd_solaris_smf" -a $MODE != mode_check ]; then
                # Solaris 10+ : convert inetd.conf entries using inetconv + smf
                echo "[INFO] Running inetconv -f -i ${INETD_FILE}..." >> ${DPSVC_LOG}
                if [ "${OS_MINVER}" -ge "11" ]; then
                    OMNITMP="/tmp/omnitmp"
                    if [ -d ${OMNITMP} ]; then
                       rm -rf ${OMNITMP}
                    fi
                    mkdir ${OMNITMP}
                    inetconv -f -n -i ${INETD_FILE} -o ${OMNITMP} >> ${DPSVC_LOG}
                    for file in ${OMNITMP}/*; do
                      svccfg import  ${file}
                    done
                    rm -rf ${OMNITMP} ${INETD_FILE}
                elif [ "${OS_MINVER}" -eq "10" ]; then
                    inetconv -f -i ${INETD_FILE} >> ${DPSVC_LOG}
                    rm -f ${INETD_FILE} 
                fi 
                # enable the services
                if [ "${TCP_TYPE}" = "ipv6" -o "${TCP_TYPE}" = "dual" ]; then
                    inetadm -e network/${SERVICE_NAME}/tcp6
                fi
                if [ "${TCP_TYPE}" = "ipv4" -o "${TCP_TYPE}" = "dual" ]; then
                    inetadm -e network/${SERVICE_NAME}/tcp
                fi

                # inetd seems to get restarted (sometimes?), but this isn't documented anywhere, so it
                # cannot be relied upon. as a workaround, sleep for a second before restarting it...
                sleep 1
            fi
        ;;
        launchd)
                echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>
<!DOCTYPE plist PUBLIC \"-//Apple Computer//DTD PLIST 1.0//EN\" \"http://www.apple.com/DTDs/PropertyList-1.0.dtd\">
<plist version=\"1.0\">
<dict>
        <key>Disabled</key>
        <false/>
        <key>Label</key>
        <string>com.hp.omni</string>
        <key>ProgramArguments</key>
        <array>
                <string>/usr/omni/bin/inet</string>
                <string>-log</string>
                <string>/usr/omni/log/inet.log</string>
        </array>
        <key>inetdCompatibility</key>
        <dict>
                <key>Wait</key>
                <false/>
        </dict>
        <key>Sockets</key>
        <dict>
                <key>Listeners</key>
                <dict>
                       <key>SockServiceName</key>
                       <string>omni</string>
                       <key>SockType</key>
                       <string>stream</string>
                </dict>
        </dict>
        <key>OnDemand</key>
        <true/>
</dict>
</plist>" > ${OUT_INETD_FILE}
         if [ "${OUT_INETD_FILE}" = "${INETD_FILE}" ]; then       
                launchctl load "${INETD_FILE}" >> ${DPSVC_LOG}
                if [ $? -ne 0 ]; then
                    echo "[ERROR]: Failed to register inet with launchd." >&2
                    exit 1
                fi
         else
              TMPBUF=`launchctl list | grep 'com.hp.omni'`
              if [ -z "$TMPBUF" ]; then
                 echo "[ERROR]: Inet registration with launchd failed." >&2
                 exit 1
              fi

         fi
        ;;
    esac
}


omniinet_remove()
{
    # 1. remove inetd service
    echo "[INFO] Updating ${INETD_TYPE} configuration" >> ${DPSVC_LOG}
    case "${INETD_TYPE}" in
        xinetd)
            if [ "${INETD_FILE}" = "/etc/xinetd.d/${SERVICE_NAME}" ]; then
                # separate config file, just remove it
                if [ -f ${INETD_FILE} ]; then
                    rm -f ${INETD_FILE}
                fi
            else
                # remove all lines between "service ${SERVICE_NAME}" and the "}"
                ${AWK} 'BEGIN {mark=0}
                        /^[ \t]*service '${SERVICE_NAME}'/ {mark=1}
                        /^[ \t]*}/ {if (mark) {mark=0; next}}
                        {if (mark) next; else print}' ${INETD_FILE} > ${INETD_FILE}.dpuninst
                cp ${INETD_FILE}.dpuninst ${INETD_FILE}
                rm ${INETD_FILE}.dpuninst
            fi
            ;;
        inetd)
            grep -v '^'${SERVICE_NAME}' stream' ${INETD_FILE} > ${INETD_FILE}.dpuninst
            cp ${INETD_FILE}.dpuninst ${INETD_FILE}
            rm ${INETD_FILE}.dpuninst
            if [ "${ARCH}" = "HP-UX" ]; then
                # HP-UX special handling, inetd.sec is an optional security file for inetd
                INETDSEC="/var/adm/inetd.sec"
                if [ -r ${INETDSEC} ]; then
                    grep -v '^'${SERVICE_NAME}' stream' ${INETDSEC} > ${INETDSEC}.dpuninst
                    cp ${INETDSEC}.dpuninst ${INETDSEC}
                    rm ${INETDSEC}.dpuninst
                fi
            fi
            ;;
        inetd_solaris_smf)
            # Solaris 10+: disable + delete the service(s), then delete the manifest(s)
            # Warning: don't do this when updating, because any running inet gets kill-ed
            # when a service is disabled!
            if [ $MODE = mode_remove ]; then
                TMPBUF=`inetadm | awk '/network\/'${SERVICE_NAME}'\/tcp6/ {print}'`
                if [ "${TMPBUF}" ]; then
                    svcadm disable network/${SERVICE_NAME}/tcp6 >> ${DPSVC_LOG}
                    svccfg delete network/${SERVICE_NAME}/tcp6 >> ${DPSVC_LOG}
                    rm -f /var/svc/manifest/network/${SERVICE_NAME}-tcp6.xml
                fi

                TMPBUF=`inetadm | awk '/network\/'${SERVICE_NAME}'\/tcp/ {print}'`
                if [ "${TMPBUF}" ]; then
                    svcadm disable network/${SERVICE_NAME}/tcp >> ${DPSVC_LOG}
                    svccfg delete network/${SERVICE_NAME}/tcp >> ${DPSVC_LOG}
                    rm -f /var/svc/manifest/network/${SERVICE_NAME}-tcp.xml
                fi
            fi
            ;;
        launchd)
            # Mac OS X: use launchctl to remove the service
            TMPBUF=`launchctl list | grep 'com.hp.omni'`
            if [ "${TMPBUF}" != "" ]; then
                launchctl unload "${INETD_FILE}" >> ${DPSVC_LOG}
                if [ -f "${INETD_FILE}" ]; then
                    rm -f "${INETD_FILE}"
                fi
            fi
            ;;
    esac
        
    # 2. remove entry in /etc/services
    SVCFOUND=`grep '^[ \t]*'${SERVICE_NAME}' .*tcp' ${SERVICES_FILE}`
    if [ -z "${SVCFOUND}" ]; then
        echo "[INFO] ${SERVICE_NAME} NOT found in ${SERVICES_FILE}, skipping..." >> ${DPSVC_LOG}
    else
        echo "[INFO] Updating ${SERVICES_FILE}" >> ${DPSVC_LOG}
        grep -v '^'${SERVICE_NAME}' .*tcp' ${SERVICES_FILE} > ${SERVICES_FILE}.dpuninst
        cp ${SERVICES_FILE}.dpuninst ${SERVICES_FILE}
        rm ${SERVICES_FILE}.dpuninst
    fi
}


#############
# main
#############
get_sysconfig

echo "=== System configuration ===" >> ${DPSVC_LOG}
echo "  ARCH: ${ARCH}" >> ${DPSVC_LOG}
echo "  MACHINE: ${MACHINE}" >> ${DPSVC_LOG}
echo "  OS_REV: ${OS_REV}" >> ${DPSVC_LOG}
echo "  SERVICES_FILE: ${SERVICES_FILE}" >> ${DPSVC_LOG}
echo "  SERVICE_NAME: ${SERVICE_NAME}" >> ${DPSVC_LOG}
echo "  INET_PORT: ${INET_PORT}" >> ${DPSVC_LOG}
echo "  INETD_TYPE: ${INETD_TYPE}" >> ${DPSVC_LOG}
echo "  INETD_FILE: ${INETD_FILE}" >> ${DPSVC_LOG}
echo "  TCP_TYPE: ${TCP_TYPE}" >> ${DPSVC_LOG}
echo "=== System configuration ===" >> ${DPSVC_LOG}
echo >> ${DPSVC_LOG}


while [ "$1" ]; do
    case "$1" in
        "-check")
            MODE=mode_check
            omniinet_install
            restart_inetd
            echo >> ${DPSVC_LOG}
            ;;
        "-install")
            MODE=mode_install
            omniinet_install
            restart_inetd
            ;;
        "-uninstall")
            MODE=mode_remove
            omniinet_remove
            restart_inetd
            ;;
        "-update")
            MODE=mode_update
            omniinet_remove
            omniinet_install
            restart_inetd
            ;;
        *)
            echo "$1": Incorrect option!
            echo "svcsetup.sh { -install | -uninstall | -update }"
            exit 1
            ;;
    esac
    shift
done



