package NorisOtrsReport::Framework;

use strict;
use warnings;
use utf8;

use DateTime;
use NorisOtrsReport::OTRS_API;
use Carp;
use Data::Dump qw(pp);

use base 'Exporter';
our $VERSION = "0.1";
our @EXPORT_OK = qw(generate run_fixture_conversions load_fixture);
our %EXPORT_TAGS = (
    data => 
        [qw(data_otrs_ticket data_customtext data_customdate 
            make_data_articles data_state_history data_linked_tickets
        )],
    convert => 
        [qw(convert_epoch_to_date_hash convert_isodate_to_date_hash)],
);
Exporter::export_ok_tags('data');
Exporter::export_ok_tags('convert');

=head1 NAME


NorisOtrsReport::Framework - Reporting-Framework für OTRS

=head1 SYNOPSIS

=for example begin

 use NorisOtrsReport::Framework qw(:data :convert generate);

 use NorisOtrsReport::DebugProcessor;

 generate(
        UserID => 1,
        Search => { TicketNumber => 10000902 },
        Conversions => 
            [
                [ CreateTimeUnix => \&convert_epoch_to_date_hash, 'created' ],
                [ RealTillTimeNotUsed => \&convert_epoch_to_date_hash, 'pending-until' ],
                [ CreateTime  => \&convert_isodate_to_date_hash, 'created' ],
            ],
        Collectors => [
                \&data_otrs_ticket,
                \&data_customtext, 
                \&data_customdate,
                \&data_state_history,
                \&data_linked_tickets,
                make_data_articles(
                        qw(note-external-error note-external-reason 
                        note-external-actions note-external-opcall 
                        note-external-progression ),
                    ),
            ],
        LoadFixture => 'filename.json',
        Processors => [NorisOtrsReport::DebugProcessor->new()],
    );

=for example end

=head1 DESCRIPTION

Berichte über OTRS-Tickets wird es bald diverse geben. Ziel dieses
Moduls ist es, daß die Logik der Abarbeitung (die ja bei allen Berichten
identetisch ist) von den spezifischen Bearbeitungsfunktionen den 
Darstellungsfunktionen getrennt bleibt, damit nicht immer wieder das 
Rad neu erfunden werden muß.

Dazu stellt dieses package bereit:
    
=over

=item *

Eine Funktion "generate()", die die gesamte Ablaufsteuerung übernimmt 
und einen Bericht auf stdout ausgibt.

=item *

Einen Satz von "data collectors" C<data_*()>, die Daten aus dem Ticketsystem 
holen und jeweils in ein Hash eintragen. 

=item *

Einen Satz von Konvertierungsfunktionen C<convert_*()>, die die aus Daten aus dem
Ticketsystem in einfacher benutzbare umwandeln.

=back

      
Außerdem gibt es in verwandten packages:
    
=over

=item Prozessor-Klassen,
    
die die Daten erhalten, daraus dann die für den
Bericht benötigten Daten ermitteln und über einen "Renderer" die
Ausgabe erstellen.
      
=item Renderer-Klassen, 

die die finale Ausgabe erstellen.

=back

Wenn man einen neuen Bericht programmiert, muß man im Regelfall "nur" noch 
die Kommadozeilen-Verarbeitung sowie die Berechnungsfunktion des Prozessors 
implementieren - der Rest ist schon vorhanden und wird über die generate()-
Funktion eingebunden.

=cut

# Verwendung der "Freitextfelder" in otrs:
# Name => Nummer
my %TICKET_FREE_TEXT_KEYS = (
        'Info' => 1,
        'externe Servicestelle' => 4,
        'Externe-ID' => 6,
        'Mandant' => 8,
        'Kategorie/Anwendung' => 9,
        'Störung nach Change' => 10,
        'Prio QC/Primondo' => 11,
        'Op-Call' => 12,
        'Standort/MA' => 13,
        'Business-Impact' => 14,
    );

# Verwendung der Freizeitfelder in otrs:
# Name => Nummer
my %TICKET_FREE_DATE_KEYS = (
        'Due-Date' => 1,
        'Incident-Start' => 2,
        'Incident-Resolved' => 3,
        'Dienstleister-Termin' => 4,
    );

=head1 SUBROUTINES

=head2 Generierung eines Berichts

=head4 generate()

generiert einen Bericht.

=for example begin

    my ($UserID, $SearchListRef, $CollectorListRef, 
        $ConversionListRef, $ProcessorListRef);

    generate(
        UserID => $UserID,
        Search => $SearchListRef,
        Collectors => $CollectorListRef,
        Conversions => $ConversionListRef,
        Processors => $ProcessorListRef
    );
    
=for example end

Parameter:
    
=over

=item UserID

Die numerische ID (im otrs!) des otrs-Benutzers, 
nach dem sich die Zugriffsrechte richten.
Eine "1" entspricht Administrator-Zugriff.
      
=item Search

Bestimmt die Tickets, auf die sich der Bericht
bezieht. Hier steht eine Referenz auf eine Liste
von Hashes. Jedes Hash spezifiert eine Suche, wie 
sie in otrs unter Kernel::Ticket::TicketSearch
beschrieben ist. Die verschiedenen Suchhashes werden
einfach hintereinander abgearbeitet.


Der folgende Ausdruck arbeitet bestimmte Ticket-Nummern
ab:
    
    Search => [ { TicketNumber => 423 }, 
                { TicketNumber => 429 } ]
            
=item Collectors

Die hier aufgelisteten Funktionen bestimmen den Umfang an
Daten, der zu dem Ticket aus otrs geholt wird. Das absolute
Minimum ist

    Collectors => [ &data_otrs_ticket ]

Damit werden nur die Daten geholt, die mittels 
Kernel::Ticket::TicketGet verfügbar sind.

Näheres dazu im Abschnitt "Collector-Funktionen".

=item LoadFixture

Der Pfad zu einer Datei, die gespeicherte Ticketdaten enthält (Fixture).
Die Daten werden dann anstelle aus der Datenbank von der Datei geladen.
Die Parameter C<Search> und C<Collectors> müssen leer sein, 
wenn C<LoadFixture> etwas enthält.
            
=item Conversions

Funktionen zur Umrechnung von otrs-Daten in leichter benutzbares.
            
Jeder Listeneintrag ist wieder eine Referenz auf eine Liste mit
folgenden drei Elementen:

=over

=item *

Key des ursprünglichen Eintrages im Daten-Hash

=item *

Anzuwendende Konvertierungsfunktion

=item *

Key des in das Daten-Hash einzufügenden Eintrags
mit den konvertierten Daten

=back

=back

=cut

sub generate {
    my %Param = @_;
    
    my ($Search, $Collectors);
    my $Conversions = $Param{Conversions} || {};    
    my $LoadFixture = $Param{LoadFixture} || '';
    my $FixtureData;
    if ($LoadFixture) {
        die "generate: Die Parameter 'LoadFixture' und 'Collectors' sind inkompatibel!" if ($Param{Collectors});
        die "generate: Die Parameter 'LoadFixture' und 'Search' sind inkompatibel!" if ($Param{Search});
        $Search = [ 1 ];
        $FixtureData = load_fixture($LoadFixture);
        my $FixtureIndex = 0;
        $Collectors = [ 
            sub {
                my ($UserID, $Conversions, $TicketID, $Data) = @_;
                %$Data = %{$FixtureData->[$FixtureIndex++]};
                run_fixture_conversions($Conversions, $Data);
                return $Data;
            }
        ];
    }
    else {
        $Search = $Param{Search} || die "Missing Parameter: Search";
        $Collectors = $Param{Collectors} || [\&data_otrs_ticket];
    }
    my $UserID = $Param{UserID} || 1;
    
    my $Processors = $Param{Processors} || [NorisOtrsReport::DebugProcessor->new()];
    
    for my $Processor (@$Processors) {
        $Processor->start();
    }
    
    for my $CurrentSearch (@$Search) {
        for my $Processor (@$Processors) {
            $Processor->start_search();
        }
        my @TicketIDs;
        if ($LoadFixture) {
            @TicketIDs = map $_->{TicketID}, @$FixtureData
        }
        else {
            @TicketIDs = $NorisOtrsReport::OTRS_API::TicketObject->TicketSearch(
                UserID => $UserID,
                Result => 'ARRAY',
                %$CurrentSearch,
            );
        }

        # fetch data, run conversions, store result in Processors
        for my $TicketID (@TicketIDs) {
            my %Data = ();            
            for my $Collector (@$Collectors) {
                $Collector->($UserID, $Conversions, $TicketID, \%Data);
            }
            for my $Processor (@$Processors) {
                $Processor->item(\%Data);
            }
        }
        for my $Processor (@$Processors) {
            $Processor->finish_search();
        }
    }
    
    for my $Processor (@$Processors) {
            $Processor->finish();
    }
    return 1;
}


=head2 Collectors

Collectors holen Daten zu einem Ticket aus otrs und stecken sie in
einen Hash. Alle haben dasselbe Interface, Parameter werden als Liste
übergeben, als C<($UserID, $Conversions, $TicketID, $Data)>
        
=over

=item C<$UserID>

Wie in generate(), die numerische ID (im otrs!)

=item C<$Conversions>
        
Wie in generate(), eine Referenz auf eine Liste
mit den anzuwendenden Conversions
        
=item C<$TicketID>

Die ID (nicht die Ticket-Nummer!) des Tickets,
dessen Daten gerade gefragt sind.
        
=item C<$Data>

Referenz auf ein Hash, in dem zum einen die
Daten von den vorangegangenen Collectors 
stehen, und in dem auch dieser Collector
seine Ergebnisse ablegt.
        
=item Rückgabewerte

Es gibt keine Rückgabewerte. Im Fehlerfall wird abgebrochen.

=back

Es gibt folgende Colletors:

=head4 data_otrs_ticket()

Holt die Basis-Daten zu einem Ticket, das sind alle, die
in otrs von Kernel::Ticket::TicketGet geliefert werden.
Parameter etc.: Siehe Abschnitt "Collectors".

=cut

sub data_otrs_ticket {
    my ($UserID, $Conversions, $TicketID, $Data) = @_;
    croak "data_otrs_ticket should be the first collector, working on an empty hash" if %$Data;
    %$Data = $NorisOtrsReport::OTRS_API::TicketObject->TicketGet(TicketID => $TicketID, UserID => $UserID);
    apply_conversions($Conversions, $Data);
    return 1;
}

=head4 data_customtext()

Dieser Collector holt seine Daten nicht direkt aus otrs holt, sondern aus
C<%$Data>, soweit es von data_otrs_ticket() befüllt worde. Daher
muß unbedingt data_otrs_ticket() in der Liste der Collectors
vorher stehen!

Die Funktion ergänzt C<%$Data> um Einträge, die die Verarbeitung der
Freitextfelder vereinfachen. Dabei werden nur die Freitextfelder 
berücksichtigt, die in C<TICKET_FREE_TEXT_KEYS> abgelegt sind!

Geliefert werden:
            
=over

=item $Data->{customtext}->{${TicketFreeTextKey$id}}->{id}

Die Nummer des Freitextfelds, also $id.

=item $Data->{customtext}->{${TicketFreeTextKey$id}}->{value}

Der Wert des Eintrags.

=back

Für das Freitextfeld "Mandant", das in otrs im Freitextfeld Nummer 8 abgelegt 
wird, wird also erzeugt:
            
    $Data->{customtext}->{Mandant} = {
        id => 8;
        value => 'Der Mandant';
    };

=begin testing

my $Data = { 
    TicketFreeKey8 => 'Mandant',
    TicketFreeText8 => 'Der Mandant'
};

data_customtext(1, [], 10001234, $Data);
is_deeply $Data->{customtext},
    { Mandant => 
        { id => 8, 
          value => 'Der Mandant',
        }
    };

$Data = { 
    TicketFreeKey8 => 'MandantXXX',
    TicketFreeText8 => 'Der Mandant'
};
is_deeply $Data->{customtext}, undef;

data_customtext(1, [], 10001234, $Data);

=end testing

=cut


sub data_customtext {
    my ($UserID, $Conversions, $TicketID, $Data) = @_;
    my %Customtext = ();
    for my $key (keys %TICKET_FREE_TEXT_KEYS) {
        my $index = $TICKET_FREE_TEXT_KEYS{$key};
        if ($Data->{"TicketFreeKey$index"}
            && $Data->{"TicketFreeKey$index"} eq $key) {
            $Customtext{$key} = {
                id => $index,
                value => $Data->{"TicketFreeText$index"}
            };
        }
        apply_conversions($Conversions, \%Customtext);
    }
    $Data->{customtext} = \%Customtext;
    return 1;
}

=head4 data_customdate()

Vergleichbar mit data_customtext(), nur eben für TicketFreeTime ...

Dieser Collector holt seine Daten nicht direkt aus otrs holt, sondern aus
C<%$Data>, soweit es von data_otrs_ticket() befüllt worde. Daher
muß unbedingt data_otrs_ticket() in der Liste der Collectors
vorher stehen!

Die Funktion ergänzt C<%$Data> um Einträge, die die Verarbeitung der
Freidatum/zeitfelder vereinfachen. Dabei werden nur die Felder 
berücksichtigt, die in C<%TICKET_FREE_DATE_KEYS> abgelegt sind!

Geliefert werden:

=over

=item $Data->{customtime}->{${TicketFreeTime$id}}->{id}

Die Nummer des Freidatum/zeitfelds, also $id.

=item $Data->{customtime}->{${TicketFreeTime$id}}->{date}

Der Wert des Eintrags als Hash-Referenz mit den Einträgen
qw(year month day hour minute second epoch)
            
=back

Für das Freizeit/datumfeld "Incident-Resolved", das in otrs im Freitextfeld Nummer 3 abgelegt 
wird, wird also erzeugt:
            
    $Data->{customdate}->{'Incident-Resolved'} = {
        id => 3,
        date => { year => 2009, month => 2, ... }
    };

=begin testing

my $Data = { 
    TicketFreeTime3 => '2009-04-01 13:31:42',
};

data_customdate(1, [], 10001234, $Data);
is_deeply $Data->{customdate},
    { 'Incident-Resolved' => 
        { id => 3, 
          date => { year => 2009, month => "04", day => "01",
                    hour => 13, minute => 31, second => 42,
                    epoch => convert_isodate_to_date_hash('2009-04-01 13:31:42')->{epoch}
                  }
        }
    };
    
=end testing

=cut

sub data_customdate {
    my ($UserID, $Conversions, $TicketID, $Data) = @_;
    my %Customdate = ();
    for my $Key (keys %TICKET_FREE_DATE_KEYS) {
        my $Index = $TICKET_FREE_DATE_KEYS{$Key};
        if ($Data->{"TicketFreeTime$Index"}) {
            $Customdate{$Key} = { 
                id => $Index,
                date => convert_isodate_to_date_hash($Data->{"TicketFreeTime$Index"}) };
        }
        apply_conversions($Conversions, \%Customdate);
    }
    $Data->{customdate} = \%Customdate;
    return 1;
}

=head4 make_data_articles()

Dies ist kein Collector, sondern eine Funktion, die einen Collector zurück gibt!

Parameter:
            
=over

=item $ArticleTypes

Referenz auf eine Liste von Artikel-Typen (als Namen)
            
=item Rückgabe

Referenz auf einen Collector (Interface wie alle Collectors), 
der folgenden Eintrag erzeugt:

        $Data->{articles} = {
            Alle zum Ticket gehörenden Artikeldaten, 
            die Kernel::Ticket::ArticleGet erfaßt. Dabei
            werden nur Artikel geholt, deren Typname in
            @$ArticleTypes enthalten ist.
        };
        
=back

=cut

sub make_data_articles {
    my ($ArticleTypes) = @_;
    
    return sub {
        my ($UserID, $Conversions, $TicketID, $Data) = @_;
        my @Articles = $NorisOtrsReport::OTRS_API::TicketObject->ArticleGet(
                                TicketID => $TicketID,
                                UserID => $UserID,
                                ArticleType => $ArticleTypes,
                            );
        for my $Item (@Articles) {
            apply_conversions($Conversions, $Item);
        }
        $Data->{articles} = \@Articles;
        return 1;
    }
}

=head4 data_state_history()

Ein Collector, der die History der Status-Änderungen herunterlädt.
Zum Interface siehe die Beschreibung zu Collectors.

Es werden für pro Status-Historie-Eintrag folgende Daten erzeugt:
  
    $Data->{history}->[...] = {
        ... alles, was Kernel::Ticket::HistoryGet holt ...
        OldState => Name des Zustands vorher
        NewState => Name des Zustands danach
    };

=cut

sub data_state_history {
    my ($UserID, $Conversions, $TicketID, $Data) = @_;
    my @History = $NorisOtrsReport::OTRS_API::TicketObject->HistoryGet(
                        TicketID => $TicketID,
                        UserID => $UserID,
                      );
    for my $Item (@History) {
        delete $Item->{UserPw};
        apply_conversions($Conversions, $Item);
        if ($Item->{HistoryType} eq 'StateUpdate'
            && $Item->{Name} =~ /^%%([^%]+)%%([^%]+)%%$/
        ) {
            $Item->{OldState} = $1;
            $Item->{NewState} = $2;
        }
    }
    $Data->{history} = \@History;
}

=head4 data_state_history()

Ein Collector, der die verlinkten Tickets herunterlädt.
Zum Interface siehe die Beschreibung zu Collectors.

Es werden für pro Ticket-Link folgende Einträge erzeugt:
            
    $Data->{'linked-tickets'}->{$LinkType}->{$Direction} = {
            ... Ticketdaten wie in data_otrs_ticket() 
                bzw. Kernel::Ticket::TicketGet ...
        };


Dabei ist:

=over

=item $LinkType

der Name des Typs der Verknüpfung. Das kann sein:

=over

=item 'ParentChild'

damit sind Tickets verknüpft, die gemerged sind

=item 'MainSub'

für Verknüpfung mit Statusüberwachung, z.B. ein Rückfrage-Ticket

=item 'Split'

für Verknüpfungen ohne Statusüberwachung, z.B. ein Spalt-Ticket
        
=item 'Normal'

wird eigentlich nicht benutzt, gibt es aber als Standard in otrs

=back

=item $Direction

ist die Verknüpfungsrichtung. Das kann sein:
            
=over

=item 'Source'

Das andere Ticket ist das untergeordnete.

=item 'Target'

Dieses Ticket (das in $TicketID) ist das untergeordnete.

=back

=back

=cut

sub data_linked_tickets {
    my ($UserID, $Conversions, $TicketID, $Data) = @_;
    my $LinkList = $NorisOtrsReport::OTRS_API::LinkObject->LinkListWithData(
            Object    => 'Ticket',
            Key       => $TicketID,
            Object2   => 'Ticket',
            State     => 'Valid',
            UserID    => $UserID,
        );
    
    $Data->{'linked-tickets'} = $LinkList->{Ticket} if exists $LinkList->{Ticket};
    # LinkTyp -> Source|Target->{ id=>1}
}
    

=head4 load_fixture()

Lädt einen gespeicherten Datensatz (fixture).

Parameter:
            
=over

=item $Filename

Dateiname; in der Datei stehen dann Ticket-Daten als Perl-Code,
und zwar als eine Referenz auf eine Liste. Jeder Listeneintrag
enthält die Daten zu einem Ticket als Hash Referenz.
            
=item Rückgabe

Referenz auf die Ticket-Liste.

=back

=cut

sub load_fixture {
    my ($Filename) = @_;

    my $fh;
    open ($fh, "<:utf8", $Filename) || die "Cannot open $Filename";
    my $FixtureString = join("", <$fh>);
    close $fh;
    my $Result = eval($FixtureString);
    die "Cannot load $Filename, eval complaines: $@" if $@;
    die "Cannot load $Filename, result is not a list ref!" unless ref $Result eq 'ARRAY';
    die "Cannot load $Filename, first element is not a hash ref!" unless ref $Result->[0] eq 'HASH';
    return $Result;
}

sub run_fixture_conversions {
    my ($Conversions, $Data) = @_;
    apply_conversions($Conversions, $Data);
    if (ref $Data->{articles}) {
        for my $article (@{$Data->{articles}}) {
            apply_conversions($Conversions, $article);
        }
    }
    if (ref $Data->{history}) {
        for my $item (@{$Data->{history}}) {
            apply_conversions($Conversions, $item);
        }
    }
    if (defined $Data->{customtext}) {
        for my $d (values %{$Data->{customtext}}) {
            apply_conversions($Conversions, $d);
        }
    }
    if (defined $Data->{customdate}) {
        for my $d (values %{$Data->{customtext}}) {
            apply_conversions($Conversions, $d);
        }
    }
    return $Data;
}


=head2 Conversions

Conversions konvertieren bereits vorhandene Daten aus dem
Daten-Hash in einfacher verwendbare.

Alle haben dasselbe Interface. Sie erhalten den zu konvertierenden
Wert als Parameter und geben eine Hash-Referenz mit dem Ergebnis zurück.

=begin testing

my $Data = { bla => '2009-04-01 13:31:42' };
my $Result = NorisOtrsReport::Framework::apply_conversions(
    [[ 'bla', \&convert_isodate_to_date_hash, 'foo']], 
    $Data);
is_deeply $Data->{foo},
        { year => 2009, month => "04", day => "01",
          hour => 13, minute => 31, second => 42,
          epoch => convert_isodate_to_date_hash('2009-04-01 13:31:42')->{epoch}
        };
is $Data->{bla}, '2009-04-01 13:31:42';

=end testing

=cut

sub apply_conversions {
    my ($Conversions, $Data) = @_;
    my ($Key, $Fn);
    for my $Item (@$Conversions) {
        my ($Key, $Fn, $Target) = @$Item;
        $Data->{$Target} = $Fn->($Data->{$Key}) if exists $Data->{$Key};
    }
}

=head4 convert_epoch_to_date_hash()

Gibt zu einem als Epoch-Wert ausgedrückten Datums/Zeitwert
einen Hash mit den Einträgen
qw(year month day hour minute second epoch) 
zurück.

 use NorisOtrsReport::Framework qw(:convert);
 my $date = convert_epoch_to_date_hash(1233836217);

=begin testing

 use NorisOtrsReport::Framework qw(:convert);
 my $date = convert_epoch_to_date_hash(1233836217);
 is_deeply $date,
    { day=>"5", epoch=>"1233836217", hour=>"13",
      minute=>"16", month=>"2", second=>"57", year=>"2009" 
    };

=end testing


=cut

# Cachen der Zeitzone zur Optimierung der Performance (siehe #10012664)
my $tz_local = DateTime::TimeZone->new( name => 'local' );

sub convert_epoch_to_date_hash {
    my ($epoch) = @_;
    return unless defined $epoch;
    my $date = DateTime->from_epoch( epoch => $epoch, time_zone => $tz_local );
    return {
        epoch => $epoch,
        day => $date->day(),
        month => $date->month(),
        year => $date->year(),
        hour => $date->hour(),
        minute => $date->minute(),
        second => $date->second(),
    }
}

=head4 convert_isodate_to_date_hash()

Gibt zu einem als Iso-Datum (YYYY-MM-DD hh:mm:ss) ausgedrückten
Datums/Zeitwert einen Hash mit den Einträgen
qw(year month day hour minute second epoch) 
zurück.

 use NorisOtrsReport::Framework qw(:convert);
 my $epoch = convert_isodate_to_date_hash('2009-02-05 12:16:57');

=begin testing

 use NorisOtrsReport::Framework qw(:convert);
 my $Result = convert_isodate_to_date_hash('2009-02-05 13:16:57');
 my $Expected = { day=>"05", epoch=>'1233836217', hour=>"13",
      minute=>"16", month=>"02", second=>"57", year=>"2009" 
    };
 is_deeply $Result, $Expected;
 # Und die Zeitzonen stimmen auch noch zwischen den beiden Funktionen überein:
 is convert_epoch_to_date_hash($Result->{epoch})->{hour}, $Expected->{hour};
    

=end testing

=cut


sub convert_isodate_to_date_hash {
    my ($isodate) = @_;
    return {} unless $isodate =~ /^(....)-(..)-(..) (..):(..)(?::(..))?$/;
    my %result = (
        year => $1,
        month => $2,
        day => $3,
        hour => $4,
        minute => $5,
        second => $6 || 0,
    );
    my $date = DateTime->new(%result, time_zone => $tz_local);

    $result{epoch} = $date->epoch();
    return \%result;
}

1;
