/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
 * Application to create the union of two or more rwset files
 */

#include <silk/silk.h>

RCSIDENT("$SiLK: rwsetunion.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/iptree.h>
#include <silk/utils.h>
#include <silk/sksite.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout


/* OPTIONS SETUP */

/*
** typedef enum {
** } appOptionsEnum;
*/

static struct option appOptions[] = {
    {0,0,0,0}           /* sentinel entry */
};

static const char *appHelp[] = {
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                       \
    ("<OUTPUT_IPSET> <INPUT_IPSET1> [<INPUT_IPSET2> ...]\n"             \
     "\trwsetunion is deprecated; use rwsettool instead.\n"             \
     "\tMerge the input binary IPSet files into the output IPSet;\n"    \
     "\tan IP in any input file will be in the output file.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    sksiteCompmethodOptionsUsage(fh);
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         UNUSED(opt_index),
    char        UNUSED(*opt_arg))
{
    return 0;
}


int main(int argc, char **argv)
{
    skIPTree_t *new_tree = NULL;
    skIPTree_t *tmp_tree = NULL;
    const char *file_name;
    int arg_index;
    skstream_t *s;
    /* the compression method to use when writing the file. */
    sk_compmethod_t comp_method;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || sksiteCompmethodOptionsRegister(&comp_method))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* get name of output file and create a stream for it */
    if (arg_index == argc) {
        skAppPrintErr("Missing output and input file names");
        skAppUsage();
    }
    if ((rv = skStreamCreate(&s, SK_IO_WRITE, SK_CONTENT_SILK))
        || (rv = skStreamBind(s, argv[arg_index]))
        || (rv = skStreamSetCompressionMethod(s, comp_method)))
    {
        skStreamPrintLastErr(s, rv, &skAppPrintErr);
        skStreamDestroy(&s);
        return 1;
    }

    /* move to first input file */
    ++arg_index;

    /* Check for at least one input file and read it */
    if (arg_index == argc) {
        skAppPrintErr("Missing input file names");
        skAppUsage();
    }
    file_name = argv[arg_index];
    rv = skIPTreeLoad(&new_tree, argv[arg_index]);
    if (rv) {
        skAppPrintErr("Unable to read IPset from '%s': %s",
                      file_name, skIPTreeStrError(rv));
        return 1;
    }
    ++arg_index;

    /* Open output stream */
    rv = skStreamOpen(s);
    if (rv) {
        skStreamPrintLastErr(s, rv, &skAppPrintErr);
        skStreamDestroy(&s);
        return 1;
    }

    /* Now handle remaining input files */
    for ( ; arg_index < argc; ++arg_index) {
        /* Open file and read IPset */
        file_name = argv[arg_index];
        rv = skIPTreeLoad(&tmp_tree, file_name);
        if (rv) {
            skAppPrintErr("Unable to read IPset from '%s': %s",
                          file_name, skIPTreeStrError(rv));
            return 1;
        }

        /* Do the union */
        skIPTreeUnion(new_tree, tmp_tree);

        skIPTreeDelete(&tmp_tree);
    }

    /* Write new_tree to the output file */
    if (skIPTreeWrite(new_tree, s)) {
        skAppPrintErr("Error writing IPset to file '%s'",
                      skStreamGetPathname(s));
        skStreamDestroy(&s);
        return 1;
    }

    rv = skStreamDestroy(&s);
    if (rv) {
        skStreamPrintLastErr(s, rv, &skAppPrintErr);
        return 1;
    }

    skAppUnregister();

    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
