/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**
**  rwresolve.c
**
**  Mark Thomas
**  December 2005
**
**      A pipe-line filter to read delimited textual input, convert
**      IP(s) to hostname(s), and print the results.  The default
**      field delimiter is '|' in deference to our default.  The
**      default fields are 1,2 (numbering starts at 1) as for rwcut.
**      Changes can be provided via options --ip-fields=<range> and
**      --delimiter=<char>.
**
**      The --ip-fields value is parsed to get a list of fields to
**      change.  The result is an array of "ip_field_type_t", where
**      each "line_part" is either an IP to lookup or a textual part
**      to be displayed as-is.
**
**      For the synchronous DNS resolution:
**
**      A line of text is read, and the ip_field_type_t of each field
**      is checked.  This creates an array of "line_part_t".
**      Contiguous textual fields are grouped into a single
**      "line_part" to speed output.  If the line_part contains an IP
**      address, it is converted to an integer (IPv6 addresses are not
**      supported).
**
**      Another function processes the array of "line_parts" to either
**      resolve the IP address or print the textual parts as-is.  The
**      function uses a hash table to avoid the DNS lookup for IPs it
**      has seen before.  If the hash table completely fills, it is
**      destroyed and re-created.
**
**      For the asynchronous DNS resolution:
**
**      The reading of a single line of input is similar to the above.
**
**      The array of line_part_t objects is stored in a "line_t", and
**      we create a linked-list of "line_t" objects.  For each IP to
**      be resolved in the array, an asynchronous DNS query object is
**      created.  The "line_t" contains an array of these outstanding
**      DNS queries.
**
**      At each iteration, the head item on the linked list of line_t
**      objects is checked to determine if the adns queries for that
**      line_t have resolved.  If so, the names are fetched and the
**      line is printed.  If not, the next line of input is read.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwresolve.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skstringmap.h>
#include <silk/hashlib.h>
#include <silk/skvector.h>
#ifdef SK_HAVE_ADNS_H
#include <adns.h>
#endif


/* LOCAL DEFINES AND TYPEDEFS */

/* where to send usage output */
#define USAGE_FH stdout

/* max fields (columns) we support in each line of the output. */
#define MAX_FIELD_COUNT         1024

/* max length of input line---must be less than UINT16_MAX since we
 * use uint16_t to hold offsets into the line */
#define MAX_LINE_LENGTH         2048

/* absolute max number of outstanding ADNS queries permitted and
 * default maximum.  used to initialize and provide range for
 * max_requests which can be set with --max-requests */
#define RWRESOLVE_REQUESTS_ABS   1 << 15
#define RWRESOLVE_REQUESTS_DEF   128

/* initializer for hash table creation */
#define HASH_INITIAL_SIZE       500000

/* size of a hostname */
#ifdef NI_MAXHOST
#  define RWRESOLVE_MAXHOST     NI_MAXHOST
#else
#  define RWRESOLVE_MAXHOST     1024
#endif

/* DNS names are stored in large character buffers that have this
 * maximum size */
#define NAMEBUF_MAX_SIZE        (1 << 23)  /* 8 MB */

/* we use multiple allocations to get to the maximum buffer size.
 * each allocation has this size */
#define NAMEBUF_STEP_SIZE       (NAMEBUF_MAX_SIZE / 8)

/* the buffers are stored in an sk_vector. values in the vector of
 * buffers are indexed by a 32-bit integer, where part of the integer
 * is which vector entry, and part is an index into the character
 * buffer.  The following create the ID and get the values from the
 * ID */
#define NB_INDEX_CREATE(nbic_vector, nbic_offset)       \
    (((nbic_vector) << 23) | (nbic_offset))

#define NB_INDEX_GET_VECTOR(nbigv_index)        \
    ((nbigv_index) >> 23)

#define NB_INDEX_GET_OFFSET(nbigo_index)        \
    ((nbigo_index) & 0x007FFFFF)

/* since the ID has a max size of 32-bits, the maximum number of
 * buffers that the vector may hold is given here */
#define NAMEBUF_VECTOR_MAX      ((1 << 9) - 1)

/* value to signify that no columnar output should be attempted */
#define RWRESOLVE_NO_COLUMNS    INT32_MAX

/* message denoting an allocation failure */
#define PERROR_MEM_LINE(pml_line)                                       \
    skAppPrintErr("Out of memory at %s:%d", __FILE__, (pml_line))
#define PERROR_MEM  PERROR_MEM_LINE(__LINE__)


/* macros to print one line_part_t to the output */
#define PRINT_PART_TEXT(pp_line, pp_idx, pp_text)                       \
    if ((0 == (pp_line)->part[(pp_idx)].columnar)                       \
        || (RWRESOLVE_NO_COLUMNS == column_width))                      \
    {                                                                   \
        fprintf(outf, "%s%s", (pp_text),                                \
                ((pp_line)->part[(pp_idx)].delim ? delim_str : ""));    \
    } else {                                                            \
        char *cp = (pp_text);                                           \
        while (isspace((int)*cp)) {                                     \
            ++cp;                                                       \
        }                                                               \
        fprintf(outf, "%*s%s", column_width, cp,                        \
                ((pp_line)->part[(pp_idx)].delim ? delim_str : ""));    \
    }

#define PRINT_PART_DEFAULT(pp_line, pp_idx)                             \
    PRINT_PART_TEXT((pp_line), (pp_idx),                                \
                    &((pp_line)->buf[(pp_line)->part[(pp_idx)].offset]))



/* the type of each field on a line of input */
typedef enum ip_field_type_en {
    /* the final field to handle; nothing but text remains */
    RWRES_TEXT_FINAL,
    /* field is surrounded by lookup fields */
    RWRES_TEXT_SINGLE,
    /* field begins a contiguous text fields */
    RWRES_TEXT_OPEN,
    /* field is in middle of contiguous text fields */
    RWRES_TEXT_CONTINUE,
    /* field closes a list of contiguous text fields */
    RWRES_TEXT_CLOSE,
    /* field contains an IP to resolve */
    RWRES_LOOKUP
} ip_field_type_t;


/* each line is split into "parts", where some parts are the IP fields
 * to be converted to names, and other parts are fields (or groups of
 * fields) to leave as-is */
typedef struct line_part_st {
    /* the IP address if this part contains an IP */
    uint32_t        ip;
    /* where in the line->buf does this part begin? */
    uint16_t        offset;
    /* whether there was a delimiter after the field */
    unsigned        delim :1;
    /* whether this part contains an address */
    unsigned        has_addr :1;
    /* whether this part should be printed in a fixed-width column */
    unsigned        columnar :1;
    /* whether the DNS lookup for this part is outstanding */
    unsigned        waiting :1;
} line_part_t;

/* for the synchronous case, there only needs to be one line_t object
 * to hold the current line.  for the asynchronous case, there is a
 * linked-list of them. */
typedef struct line_st line_t;
struct line_st {
    /* the line of text from the input */
    char           *buf;
    /* information about each part of the input */
    line_part_t    *part;
#ifdef SK_HAVE_ADNS_H
    /* next line */
    line_t         *next;
    /* outstanding queries */
    adns_query     *query;
    /* number of queries that we made */
    uint16_t        query_count;
#endif
    /* the number of parts in the line */
    uint16_t        part_count;
    /* the length of the line of input */
    uint16_t        len;
};


/* possible resolve methods to use */
typedef enum resolver_type_en {
    RESOLVE_GETHOSTBYADDR,
    RESOLVE_GETNAMEINFO,
    RESOLVE_ADNS_SUBMIT
} resolver_type_t;

/* state used by getnameinfo() and/or gethostbyaddr() */
typedef struct resolve_state_st {
    char                hostname[RWRESOLVE_MAXHOST];
    struct sockaddr_in  sock;
} resolve_state_t;


/* LOCAL VARIABLES */

/* resolvers: the first of these will be the default */
static const sk_stringmap_entry_t resolver_name[] = {
#ifdef SK_HAVE_ADNS_H
    {"adns",            RESOLVE_ADNS_SUBMIT,    NULL, NULL},
#endif
#ifdef SK_HAVE_GETNAMEINFO
    {"getnameinfo",     RESOLVE_GETNAMEINFO,    NULL, NULL},
#endif
    {"gethostbyaddr",   RESOLVE_GETHOSTBYADDR,  NULL, NULL},
    SK_STRINGMAP_SENTINEL
};

/* which resolver function to use */
static resolver_type_t resolver;

/* what each field of the input contains */
static ip_field_type_t ip_fields[MAX_FIELD_COUNT];

/* number of IP fields to resolve */
static uint16_t ip_field_count = 0;

/* max number of line_part_t's we expect for each line */
static int line_part_count = 0;

/* input stream */
static skstream_t *in_stream = NULL;

/* where to send output */
static FILE *outf;

/* width of IP columns */
static int column_width = RWRESOLVE_NO_COLUMNS;

/* delimiter between the fields */
static char delimiter = '|';

/* a string containing the delimiter */
static char delim_str[2];

/* hash table to cache DNS results */
static HashTable *hash = NULL;

/* value that indicates the cache failed to allocate memory */
#define RWRES_CACHE_FAIL UINT32_MAX

/* value to store in hash to indicate that DNS lookup failed */
#define RWRES_NONAME     (UINT32_MAX - 1u)

#ifdef SK_HAVE_ADNS_H
/* value to store in hash to indicate that DNS lookup is outstanding */
#define RWRES_WAITING    (UINT32_MAX - 2u)

/* maximum number of outstanding ADNS queries allowed */
static uint32_t max_requests = RWRESOLVE_REQUESTS_DEF;
#endif

/* for efficiency of DNS name storage, a vector of character buffers
 * is maintained.  */
static sk_vector_t *namebuf_vec = NULL;

/* size of current buffer and number of bytes that are available */
static uint32_t namebuf_size = 0;
static size_t namebuf_avail = 0;


/* OPTIONS SETUP */

typedef enum {
    OPT_IP_FIELDS, OPT_DELIMITER, OPT_COLUMN_WIDTH, OPT_RESOLVER
#ifdef SK_HAVE_ADNS_H
    , OPT_MAX_REQUESTS
#endif
} appOptionsEnum;


static struct option appOptions[] = {
    {"ip-fields",       REQUIRED_ARG, 0, OPT_IP_FIELDS},
    {"delimiter",       REQUIRED_ARG, 0, OPT_DELIMITER},
    {"column-width",    REQUIRED_ARG, 0, OPT_COLUMN_WIDTH},
    {"resolver",        REQUIRED_ARG, 0, OPT_RESOLVER},
#ifdef SK_HAVE_ADNS_H
    {"max-requests",    REQUIRED_ARG, 0, OPT_MAX_REQUESTS},
#endif
    {0,0,0,0}           /* sentinel entry */
};

static const char *appHelp[] = {
    ("Convert IPs to host names in these input columns.  Column\n"
     "\tnumbers start with 1. Def. 1,2"),
    "Set delimiter between fields to this character. Def. '|'",
    ("Specify the output width of the column(s) specified\n"
     "\tin --fields.  Def. No justification for host names"),
    "Specify IP-to-host mapping function",
#ifdef SK_HAVE_ADNS_H
    ("When adns is used, allow no more than this many pending\n"
     "\tDNS requests"),
#endif
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  parseIPFields(const char *arg);
static int  parseResolverName(const char *res_name);
static void reallocCache(int recreate);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES]\n"                                                           \
     "\tReads delimited text (such as from rwcut) from the standard input\n"  \
     "\tand resolves the IP addresses in the specified columns.  If the\n"    \
     "\t--ip-fields switch is not given, columns 1 and 2 are resolved.\n"     \
     "\tOutput is sent to the standard output.  Beware, this is going\n"      \
     "\tto be slow.\n")

    FILE *fh = USAGE_FH;
    int i;
    int j;

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);
    skOptionsDefaultUsage(fh);

    for (i = 0; appOptions[i].name; ++i) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch (appOptions[i].val) {
          case OPT_RESOLVER:
            fprintf(fh, "%s. Def. %s\n",
                    appHelp[i], resolver_name[0].name);
            fprintf(fh, "\tChoices: %s", resolver_name[0].name);
            for (j = 1; resolver_name[j].name; ++j) {
                fprintf(fh, ", %s", resolver_name[j].name);
            }
            break;

#ifdef SK_HAVE_ADNS_H
          case OPT_MAX_REQUESTS:
            fprintf(fh, ("%s, 1-%" PRIu32 ". Def. %" PRIu32),
                    appHelp[i], RWRESOLVE_REQUESTS_ABS,RWRESOLVE_REQUESTS_DEF);
            break;

#endif  /* SK_HAVE_ADNS_H */
          default:
            fprintf(fh, "%s", appHelp[i]);
            break;
        }
        fprintf(fh, "\n");
    }
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    skStreamDestroy(&in_stream);

    reallocCache(0);

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    resolver = (resolver_type_t)resolver_name[0].id;
    outf = stdout;

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        skAppUsage();             /* never returns */
    }

    /* check for extra arguments */
    if (arg_index != argc) {
        skAppPrintErr("Unexpected argument '%s'", argv[arg_index]);
        skAppUsage();             /* never returns */
    }

#ifdef SK_HAVE_ADNS_H
    /* max number of outstanding requests must be at least as large as
     * the number of IP fields */
    if (max_requests < ip_field_count) {
        max_requests = ip_field_count;
    }
#endif  /* SK_HAVE_ADNS_H */

    /* set the delimiter string */
    snprintf(delim_str, sizeof(delim_str), "%c", delimiter);

    /* set the default fields if none specified */
    if (0 == line_part_count) {
        parseIPFields("1,2");
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    /* create hash table */
    reallocCache(1);

    /* open input */
    if ((rv = skStreamCreate(&in_stream, SK_IO_READ, SK_CONTENT_TEXT))
        || (rv = skStreamBind(in_stream, "stdin"))
        || (rv = skStreamOpen(in_stream)))
    {
        skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    return;                     /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Called by skOptionsParse(), this handles a user-specified switch
 *    that the application has registered, typically by setting global
 *    variables.  Returns 1 if the switch processing failed or 0 if it
 *    succeeded.  Returning a non-zero from from the handler causes
 *    skOptionsParse() to return a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;
    uint32_t tmp32;

    switch ((appOptionsEnum)opt_index) {
      case OPT_IP_FIELDS:
        if (parseIPFields(opt_arg)) {
            return 1;
        }
        break;

      case OPT_DELIMITER:
        delimiter = *opt_arg;
        if ('\0' == delimiter) {
            skAppPrintErr("Invalid %s: Empty string not valid argument",
                          appOptions[opt_index].name);
            return 1;
        }
        break;

      case OPT_COLUMN_WIDTH:
        rv = skStringParseUint32(&tmp32, opt_arg, 0, MAX_LINE_LENGTH);
        if (rv) {
            goto PARSE_ERROR;
        }
        column_width = (int)tmp32;
        break;

      case OPT_RESOLVER:
        if (parseResolverName(opt_arg)) {
            return 1;
        }
        break;

#ifdef SK_HAVE_ADNS_H
      case OPT_MAX_REQUESTS:
        rv = skStringParseUint32(&max_requests, opt_arg,
                                 1, RWRESOLVE_REQUESTS_ABS);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;
#endif  /* SK_HAVE_ADNS_H */
    }

    return 0;                   /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/* comparison function used to sort the user's field list */
static int compr32(const void *a, const void* b)
{
    const uint32_t a32 = *((uint32_t*)a);
    const uint32_t b32 = *((uint32_t*)b);

    if (a32 < b32) {
        return -1;
    }
    if (a32 > b32) {
        return 1;
    }
    return 0;
}


/*
 *  is_ok = parseIPFields(arg);
 *
 *    Given a comma-separated list of numbers and/or ranges, set the
 *    appropriate flags in the 'ip_fields' array.
 */
static int parseIPFields(const char *arg)
{
    uint32_t *list;
    uint32_t count;
    uint32_t i;
    uint32_t j;
    int rv;

    /* have we been here before? */
    if (line_part_count) {
        skAppPrintErr("Invalid %s: Option given multiple times",
                      appOptions[OPT_IP_FIELDS].name);
        return 1;
    }

    /* parse the values into an array */
    rv = skStringParseNumberList(&list, &count, arg, 1, MAX_FIELD_COUNT, 0);
    if (rv) {
        skAppPrintErr("Invalid %s '%s': %s",
                      appOptions[OPT_IP_FIELDS].name, arg,
                      skStringParseStrerror(rv));
        return 1;
    }

    /* the list of fields must be in ascending order and contain no
     * duplicates */
    qsort(list, count, sizeof(uint32_t), &compr32);
    for (i = 0, j = 1; j < count; ++j) {
        if (list[i] < list[j]) {
            ++i;
            if (i != j) {
                list[i] = list[j];
            }
        }
    }
    count = i + 1;

    /* initialize values */
    memset(ip_fields, 0, sizeof(ip_fields));
    i = 0;
    j = 0;

    /* handle the first field */
    if (i + 1 == list[j]) {
        /* this field is an IP to lookup */
        ip_fields[i] = RWRES_LOOKUP;
        ++ip_field_count;
        ++line_part_count;
        ++j;
        if (j == count) {
            ip_fields[i + 1] = RWRES_TEXT_FINAL;
            ++line_part_count;
            goto END;
        }
    } else {
        ip_fields[i] = RWRES_TEXT_OPEN;
        ++line_part_count;
    }
    ++i;

    /* handle all remaining fields */
    for ( ; i < MAX_FIELD_COUNT; ++i) {
        if (i + 1 == list[j]) {
            /* this field is an IP to lookup */

            /* properly close previous field */
            switch (ip_fields[i - 1]) {
              case RWRES_LOOKUP:
                break;

              case RWRES_TEXT_OPEN:
                ip_fields[i - 1] = RWRES_TEXT_SINGLE;
                break;

              case RWRES_TEXT_CONTINUE:
                ip_fields[i - 1] = RWRES_TEXT_CLOSE;
                break;

              default:
                skAbortBadCase(ip_fields[i-1]);
            }

            ip_fields[i] = RWRES_LOOKUP;
            ++ip_field_count;
            ++line_part_count;
            ++j;
            if (j == count) {
                ip_fields[i + 1] = RWRES_TEXT_FINAL;
                ++line_part_count;
                break;
            }
        } else {
            /* this is a text field. set its type based on the
             * previous field */
            switch (ip_fields[i - 1]) {
              case RWRES_LOOKUP:
                ip_fields[i] = RWRES_TEXT_OPEN;
                ++line_part_count;
                break;

              case RWRES_TEXT_OPEN:
              case RWRES_TEXT_CONTINUE:
                ip_fields[i] = RWRES_TEXT_CONTINUE;
                break;

              default:
                skAbortBadCase(ip_fields[i-1]);
            }
        }
    }

  END:
    free(list);
    return 0;                   /* OK */
}


/*
 *  ok = parseResolverName(res_name);
 *
 *    Set the global 'resolver' based on the resolver name specified
 *    in 'res_name'.  Return 0 on success, -1 on failure.
 */
static int parseResolverName(const char *res_name)
{
    sk_stringmap_t *str_map = NULL;
    sk_stringmap_status_t rv_map;
    sk_stringmap_entry_t *found_entry;
    int rv = -1;

    /* create a stringmap of the available resolvers */
    if (SKSTRINGMAP_OK != skStringMapCreate(&str_map)) {
        PERROR_MEM;
        goto END;
    }
    if (skStringMapAddEntries(str_map, -1, resolver_name) != SKSTRINGMAP_OK) {
        goto END;
    }

    /* attempt to match */
    rv_map = skStringMapGetByName(str_map, res_name, &found_entry);
    switch (rv_map) {
      case SKSTRINGMAP_OK:
        resolver = (resolver_type_t)found_entry->id;
        rv = 0;
        break;

      case SKSTRINGMAP_PARSE_AMBIGUOUS:
        skAppPrintErr("%s value '%s' is ambiguous",
                      appOptions[OPT_RESOLVER].name, res_name);
        break;

      case SKSTRINGMAP_PARSE_NO_MATCH:
        skAppPrintErr("%s value '%s' is not recognized",
                      appOptions[OPT_RESOLVER].name, res_name);
        break;

      default:
        skAppPrintErr("Unexpected return value from string-map parser (%d)",
                      rv_map);
        break;
    }

  END:
    if (str_map) {
        skStringMapDestroy(str_map);
    }
    return rv;
}


/*
 *  reallocCache(recreate);
 *
 *    Creates, clears, and/or destroys the hash table and vector used
 *    to cache the DNS names.
 *
 *    If the global hashtable 'hash' is non-NULL, calling this
 *    function destroys the hash and all of its contents.  (There is
 *    no way to clear the hash table.)
 *
 *    If 'recreate' is non-zero, calling this function creates a new
 *    hashtable and stores it in the global 'hash'.
 *
 *    This function also handles the namebuf vector.  When this
 *    function is called, the namebuf vector is emptied if it exists.
 *    If the namebuf vector does not exist and 'recreate' is 1, it is
 *    created.  If 'recreate' is 0 and the namebuf vector exists, it
 *    will be destroyed.
 *
 *    If an allocation fails, the problem terminates.
 */
static void reallocCache(int recreate)
{
    uint32_t hash_no_value = UINT32_MAX;
    size_t i;
    void **p;

    if (namebuf_vec) {
        for (i = 0; i < skVectorGetCount(namebuf_vec); ++i) {
            p = skVectorGetValuePointer(namebuf_vec, i);
            if (p && *p) {
                free(*p);
            }
        }
        if (recreate) {
            /* empty the vector */
            skVectorClear(namebuf_vec);
        } else {
            /* destroy it */
            skVectorDestroy(namebuf_vec);
        }
    } else if (recreate) {
        namebuf_vec = skVectorNew(sizeof(char*));
        if (NULL == namebuf_vec) {
            PERROR_MEM;
            exit(EXIT_FAILURE);
        }
    }
    namebuf_avail = 0;
    namebuf_size = 0;

    if (hash) {
        hashlib_free_table(hash);
        hash = NULL;
    }

    if (recreate) {
        hash = hashlib_create_table(sizeof(uint32_t), sizeof(uint32_t),
                                    HTT_INPLACE, (uint8_t*)&hash_no_value,
                                    NULL, 0,
                                    HASH_INITIAL_SIZE, DEFAULT_LOAD_FACTOR);
        if (NULL == hash) {
            PERROR_MEM;
            exit(EXIT_FAILURE);
        }
    }
}


/*
 *  name = getCachedName(id);
 *
 *    Get the string in the cache that is indexed by 'id'.
 */
static char *getCachedName(uint32_t id)
{
    char **buf_ptr = skVectorGetValuePointer(namebuf_vec,
                                             NB_INDEX_GET_VECTOR(id));
    return ((*buf_ptr) + NB_INDEX_GET_OFFSET(id));
}


/*
 *  id = cacheName(name);
 *
 *    Stores a copy of 'name' in a large character buffer.  Returns a
 *    32bit integer that can be used to get the name.  Returns
 *    UINT32_MAX if growing the buffer fails.
 */
static uint32_t cacheName(const char *name)
{
    static char *namebuf = NULL;
    static uint32_t namebuf_offset = 0;
    static uint32_t vector_idx = 0;
    size_t len = 1 + strlen(name);
    uint32_t rv;

    if (len > namebuf_avail) {
        if ((0 == namebuf_size) || (NAMEBUF_MAX_SIZE == namebuf_size)) {
            /* must create new buffer */
            namebuf = malloc(NAMEBUF_STEP_SIZE);
            if (NULL == namebuf) {
                return RWRES_CACHE_FAIL;
            }

            if (skVectorAppendValue(namebuf_vec, &namebuf)) {
                free(namebuf);
                return RWRES_CACHE_FAIL;
            }
            vector_idx = skVectorGetCount(namebuf_vec) - 1;
            if (NAMEBUF_VECTOR_MAX == vector_idx) {
                /* ran out space in the 32bit ID */
                return RWRES_CACHE_FAIL;
            }
            namebuf_size = NAMEBUF_STEP_SIZE;
            namebuf_avail = namebuf_size;
            namebuf_offset = 0;

        } else {
            namebuf_size += NAMEBUF_STEP_SIZE;
            namebuf = realloc(namebuf, namebuf_size);
            if (NULL == namebuf) {
                /* failed.  restore old values */
                namebuf_size -= NAMEBUF_STEP_SIZE;
                skVectorGetValue(&namebuf, namebuf_vec, vector_idx);
                return RWRES_CACHE_FAIL;
            }
            /* buffer may have moved.  replace in vector */
            skVectorSetValue(namebuf_vec, vector_idx, &namebuf);
            namebuf_avail += NAMEBUF_STEP_SIZE;
        }
    }

    memcpy(namebuf + namebuf_offset, name, len);
    rv = NB_INDEX_CREATE(vector_idx, namebuf_offset);
    namebuf_avail -= len;
    namebuf_offset += len;
    return rv;
}


/*
 *  ok = getLine(&line, linebuf_len);
 *
 *    Gets the next line of input and stores it in the 'buf' element
 *    of the 'line' structure.  The user should provide the size of
 *    that buffer in the 'linebuf_len' parameter.  The 'len' element
 *    of 'line' will be set to the length (strlen()) of the input.
 *
 *    Returns 0 if there was input.  Returns 1 if there is no more
 *    input or -1 if there was an error reading the input.
 *
 *    In addition, the line will be parsed in accordance with the
 *    global 'ip_fields' array, and the line_part_t's of the 'part[]'
 *    array on 'line' will be filled in with the offets into
 *    line->buf.  The number of part[] elements found is stored in
 *    line->part_count.
 *
 *    The function will attempt to parse the IP fields.  If parsing
 *    succeeds, the 'ip' and 'has_addr' members of the appropriate
 *    'line_part_t' are set; otherwise, the 'has_addr' element is set
 *    to 0.
 */
static int getLine(
    line_t     *line,
    size_t      linebuf_len)
{
    skipaddr_t ip;
    int rv;
    char *cp;
    char *ep;
    int field;
    int i;

    /* get next valid line of input */
    while ((rv = skStreamGetLine(in_stream, line->buf, linebuf_len, NULL))
           != SKSTREAM_OK)
    {
        switch (rv) {
          case SKSTREAM_ERR_EOF:
            /* no more input */
            return 1;

          case SKSTREAM_ERR_LONG_LINE:
            /* bad: line was longer than sizeof(line) */
            continue;

          default:
            /* unexpected error */
            skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
            return -1;
        }
    }

    /* store length of the line */
    line->len = strlen(line->buf);

    /* process each field */
    for (cp = line->buf, field = 0, i = 0; *cp; ++field) {

        switch (ip_fields[field]) {
          case RWRES_LOOKUP:
            line->part[i].offset = cp - line->buf;
            line->part[i].columnar = 1;
            ep = strchr(cp, delimiter);
            if (ep) {
                *ep = '\0';
            }
            if (0 == skStringParseIP(&ip, cp) && 0 == skipaddrIsV6(&ip)) {
                line->part[i].has_addr = 1;
                line->part[i].ip = skipaddrGetV4(&ip);
            } else {
                /* parsing failed or address is IPv6 and just treat as
                 * normal text */
                line->part[i].has_addr = 0;
            }
            if (ep) {
                line->part[i].delim = 1;
                cp = ep + 1;
            } else {
                line->part[i].delim = 0;
                cp = &line->buf[line->len];
            }
            ++i;
            break;

          case RWRES_TEXT_FINAL:
            line->part[i].offset = cp - line->buf;
            cp = &line->buf[line->len];
            ++i;
            break;

          case RWRES_TEXT_OPEN:
            line->part[i].offset = cp - line->buf;
            line->part[i].columnar = 0;
            line->part[i].has_addr = 0;
            /* FALLTHROUGH */

          case RWRES_TEXT_CONTINUE:
            ep = strchr(cp, delimiter);
            if (ep) {
                cp = ep + 1;
                if (!*cp) {
                    line->part[i].delim = 0;
                    ++i;
                }
            } else {
                /* unexpected end of input */
                line->part[i].delim = 0;
                cp = &line->buf[line->len];
                ++i;
            }
            break;

          case RWRES_TEXT_SINGLE:
            line->part[i].offset = cp - line->buf;
            line->part[i].columnar = 0;
            line->part[i].has_addr = 0;
            /* FALLTHROUGH */

          case RWRES_TEXT_CLOSE:
            ep = strchr(cp, delimiter);
            if (ep) {
                *ep = '\0';
                line->part[i].delim = 1;
                cp = ep + 1;
            } else {
                line->part[i].delim = 0;
                cp = &line->buf[line->len];
            }
            ++i;
            break;
        }
    }

    line->part_count = i;

    return 0;
}


/*
 *  ok = resolve_gethostbyaddr(ip, name, state);
 *
 *    Attempt to resolve, using gethostbyaddr(), the IP address in
 *    'ip'.  If unsuccessful, return -1.  If successful, attempt to
 *    make a copy of the name and copy it and store the location in
 *    the memory pointed at by 'name'.  If successful, return 0.  If
 *    unsuccessful, copy the name into the buffer in the 'state'
 *    buffer and return 1.
 */
static int resolve_gethostbyaddr(
    uint32_t            ip,
    uint32_t           *cache_id,
    resolve_state_t    *state)
{
    struct hostent *he;
    in_addr_t a;

    a = htonl(ip);
    he = gethostbyaddr((char*)&a, sizeof(in_addr_t), AF_INET);
    if (!he) {
        return -1;
    }

    *cache_id = cacheName(he->h_name);
    if (RWRES_CACHE_FAIL == *cache_id) {
        strncpy(state->hostname, he->h_name, sizeof(state->hostname));
        return 1;
    }
    return 0;
}


#ifdef SK_HAVE_GETNAMEINFO
/*
 *  ok = resolve_getnameinfo(ip, name, state);
 *
 *    Attempt to resolve, using getnameinfo(), the IP address in
 *    'ip'.  If unsuccessful, return -1.  If successful, attempt to
 *    make a copy of the name and copy it and store the location in
 *    the memory pointed at by 'name'.  If successful, return 0.  If
 *    unsuccessful, copy the name into the buffer in the 'state'
 *    buffer and return 1.
 *
 *    This function assumes the 'sock' member of the 'state' attribute
 *    has been properly initialized for an IPv4 address.
 */
static int resolve_getnameinfo(
    uint32_t            ip,
    uint32_t           *cache_id,
    resolve_state_t    *state)
{
    int rv;

    state->sock.sin_addr.s_addr = htonl(ip);
    rv = getnameinfo((struct sockaddr *)&state->sock,
                     sizeof(struct sockaddr_in),
                     state->hostname, sizeof(state->hostname),
                     NULL, 0, NI_NAMEREQD);
    if (0 != rv) {
        return -1;
    }
    *cache_id = cacheName(state->hostname);
    if (RWRES_CACHE_FAIL == *cache_id) {
        return 1;
    }
    return 0;
}
#endif  /* SK_HAVE_GETNAMEINFO */


/*
 *  ok = resolve_synchronous();
 *
 *    Process the input using a synchronous DNS resolver such as
 *    gethostbyaddr() or getnameinfo().
 *
 *    A line of input is read, the IPs on the line are
 *    processed---either fetched from the cache or resolved and then
 *    inserted into the cache---and the line is printed.
 */
static int resolve_synchronous(void)
{
    int (*resolve_fun)(uint32_t, uint32_t *, resolve_state_t *);
    resolve_state_t state;
    char line_buf[MAX_LINE_LENGTH];
    line_t line;
    uint16_t i;
    uint32_t *cache_id;
    int rv;

    /* initialize the state for the resolver function and choose which
     * function to use */
    memset(&state, 0, sizeof(resolve_state_t));

#ifdef SK_HAVE_GETNAMEINFO
    if (resolver == RESOLVE_GETNAMEINFO) {
        resolve_fun = &resolve_getnameinfo;
        /* set up the sockaddr_in structure */
        memset(&state.sock, 0, sizeof(struct sockaddr_in));
        state.sock.sin_family = AF_INET;
#ifdef SK_HAVE_STRUCT_SOCKADDR_IN_SIN_LEN
        state.sock.sin_len = sizeof(struct sockaddr_in);
#endif
    } else
#endif  /* SK_HAVE_GETNAMEINFO */
    {
        resolve_fun = &resolve_gethostbyaddr;
    }

    /* initialize the line_t object.  since we are synchronous, we
     * only need one.  create the array of line_part_t's. */
    memset(&line, 0, sizeof(line_t));
    line.buf = line_buf;
    line.part = calloc(line_part_count, sizeof(line_part_t));
    if (NULL == line.part) {
        PERROR_MEM;
        return 1;
    }

    /* process the input */
    while ((rv = getLine(&line, sizeof(line_buf))) == 0) {

        for (i = 0; i < line.part_count; ++i) {
            if (!line.part[i].has_addr) {
                /* no lookup for this part */
                PRINT_PART_DEFAULT(&line, i);
                continue;
            }

            /* check for the IP */
            rv = hashlib_insert(hash, (uint8_t*)&line.part[i].ip,
                                (uint8_t**)&cache_id);
            switch (rv) {
              case OK_DUPLICATE:
                /* found in cache */
                if (*cache_id == RWRES_NONAME) {
                    /* previous lookup failed */
                    PRINT_PART_DEFAULT(&line, i);
                } else {
                    PRINT_PART_TEXT(&line, i, getCachedName(*cache_id));
                }
                break;

              case ERR_OUTOFMEMORY:
              case ERR_NOMOREBLOCKS:
                reallocCache(1);
                rv = hashlib_insert(hash, (uint8_t*)&line.part[i].ip,
                                    (uint8_t**)&cache_id);
                if (rv != OK) {
                    PERROR_MEM;
                    free(line.part);
                    return 1;
                }
                /* FALLTHROUGH */

              case OK:
                /* new entry; must do the DNS lookup */
                rv = resolve_fun(line.part[i].ip, cache_id, &state);
                if (0 == rv) {
                    PRINT_PART_TEXT(&line, i, getCachedName(*cache_id));
                } else if (1 == rv) {
                    /* lookup ok but allocatation failed. print name
                     * from the state and then destroy the hash */
                    PRINT_PART_TEXT(&line, i, state.hostname);
                    reallocCache(1);
                } else {
                    /* lookup failed */
                    assert(-1 == rv);
                    *cache_id = RWRES_NONAME;
                    PRINT_PART_DEFAULT(&line, i);
                }
                break;
            }
        }

        fprintf(outf, "\n");
    }

    free(line.part);
    return (rv >= 0 ? 0 : 1);
}


#ifdef SK_HAVE_ADNS_H
/*
 *  freeLine(line);
 *
 *    Frees all memory used by a line_t.  Does not cancel any
 *    outstanding DNS queries.
 */
static void freeLine(line_t *line)
{
    if (line) {
        if (line->buf) {
            free(line->buf);
        }
        if (line->query) {
            free(line->query);
        }
        if (line->part) {
            free(line->part);
        }
        free(line);
    }
}


/*
 *  line = allocLine();
 *
 *    Creates space for a new line_t object as well as space for the
 *    maximum number of line_part_t and adns queries.  Returns NULL if
 *    memory allocation failed.
 */
static line_t *allocLine(void)
{
    line_t *line;

    line = calloc(1, sizeof(line_t));
    if (line) {
        line->part = calloc(line_part_count, sizeof(line_part_t));
        line->query = calloc(ip_field_count, sizeof(adns_query));
        if (!line->part || !line->query) {
            freeLine(line);
            line = NULL;
        }
    }
    return line;
}


/*
 *  ok = resolve_adns_submit();
 *
 *    Process the input using the asynchronous DNS resolver library
 *    ADNS.
 *
 *    A line of input is read, and adns_submit() is called on each IP
 *    address to be looked up.  To allow input processing to continue
 *    while waiting for the DNS lookup to complete, a linked-list of
 *    input lines is maintained.  The input line contains an array of
 *    outstanding DNS queries.
 *
 *    At each iteration, the head item on the linked list of line_t
 *    objects is checked to determine if the adns queries for that
 *    line_t have resolved.  If so, the names are fetched and the
 *    line is printed.  If not, the next line of input is read.
 *
 *    If the maximum number of ADNS requets is reached, processing of
 *    input stops until some of the outstanding requests are handled.
 *
 *    If an out of memory condition is detected, all lines in the
 *    linked list are printed, and then the cache of DNS names is
 *    destroyed.  An LRU cache would be better, but the brute force
 *    method is simple and it works.
 */
static int resolve_adns_submit(void)
{
    adns_state adns;
    adns_answer **answers;
    uint16_t answer_count;
    uint32_t num_requests;
    char line_buf[MAX_LINE_LENGTH];
    line_t *line;
    line_t *head;
    line_t *tail;
    line_t *free_list;
    line_t *line_no_mem;
    uint32_t *cache_id;
    char arpa_addr[64];        /* "255.255.255.255.in-addr.arpa", */
    uint16_t i, j;
    int eof;
    int no_mem;
    int rv;

    /* The pointers for maintaining the linked list of lines */
    head = tail = NULL;

    /* To avoid malloc/free, keep a free-list of line_t objects */
    free_list = NULL;

    /* If a memory error occurs when building a line, we keep a
     * pointer to that line and process outstanding requests.  This is
     * a pointer to the postponed line. */
    line_no_mem = NULL;

    /* set to non-zero when all input has been processed */
    eof = 0;

    /* non-zero for a memory error; value will be __LINE__ where error
     * occurred. */
    no_mem = 0;

    /* initialize the ADNS library */
    rv = adns_init(&adns, 0, 0);
    if (rv) {
        PERROR_MEM;
        return 1;
    }

    /* all DSN queries for a line must be answered before we move to
     * the next line, so there are at most 'ip_field_count' answers
     * outstanding. */
    answers = calloc(ip_field_count, sizeof(adns_answer*));
    if (NULL == answers) {
        PERROR_MEM;
        adns_finish(adns);
        return 1;
    }
    answer_count = 0;

    /* number of outstanding requests.  used to ensure no more that
     * max_requests are ever made */
    num_requests = 0;

    /* continue processing as long as there is input or outstanding
     * queries to process */
    while (head || !eof) {

        /* remove as many lines from the linked list as possible */
        while (head) {
            line = head;

            /* check whether we have read all results for this line */
            while (answer_count < line->query_count) {
                /* if adding another line's worth of DNS requests will
                 * put us over max_requests, we need to wait for the
                 * queries to complete */
                if (no_mem || num_requests + ip_field_count > max_requests) {
                    rv = adns_wait(adns, &line->query[answer_count],
                                   &answers[answer_count], NULL);
                } else {
                    rv = adns_check(adns, &line->query[answer_count],
                                    &answers[answer_count], NULL);
                }
                if (EAGAIN == rv) {
                    /* not ready */
                    break;
                }
                if (0 != rv) {
                    if (no_mem || num_requests + ip_field_count > max_requests){
                        skAppPrintSyserror("Error in adns_wait()");
                    } else {
                        skAppPrintSyserror("Error in adns_check()");
                    }
                    exit(EXIT_FAILURE);
                }
                --num_requests;
                ++answer_count;
            }

            if (answer_count != line->query_count) {
                /* queries are still outstanding for this line */
                break;
            }

            /* print the line */
            for (i = 0, j = 0; i < line->part_count; ++i) {
                if (!line->part[i].has_addr) {
                    /* no lookup on this field */
                    PRINT_PART_DEFAULT(line, i);
                    continue;
                }

                rv = hashlib_lookup(hash, (uint8_t*)&line->part[i].ip,
                                    (uint8_t**)&cache_id);
                if (rv != OK) {
                    skAbort();
                }
                if (!line->part[i].waiting) {
                    if (*cache_id == RWRES_NONAME) {
                        /* previous lookup failed */
                        PRINT_PART_DEFAULT(line, i);
                    } else {
                        PRINT_PART_TEXT(line, i, getCachedName(*cache_id));
                    }
                } else {
                    if (answers[j]->status == adns_s_ok) {
                        PRINT_PART_TEXT(line, i, *answers[j]->rrs.str);
                        *cache_id = cacheName(*answers[j]->rrs.str);
                        if (RWRES_CACHE_FAIL == *cache_id) {
                            /* have pending lines that use the address.
                             * just print the IP, and set no_mem so
                             * all pending lines will be processed. */
                            *cache_id = RWRES_NONAME;
                            no_mem = __LINE__;
                        }
                    } else {
                        *cache_id = RWRES_NONAME;
                        PRINT_PART_DEFAULT(line, i);
                    }
                    free(answers[j]);
                    ++j;
                }
            }

            fprintf(outf, "\n");

            answer_count = 0;

            /* go to next line */
            head = line->next;

            /* reset the line and add it to the free list */
            free(line->buf);
            line->buf = NULL;
            line->next = free_list;
            free_list = line;
        }

        /* no line to read if we're at end of input */
        if (eof) {
            continue;
        }

        /* handle no memory condition from elsewhere */
        if (no_mem) {
            if (head) {
                /* we should have called 'adns_wait()' to completely
                 * process all entries in the while() loop.  something
                 * is unhappy. */
                PERROR_MEM_LINE(no_mem);
                skAppPrintErr("Memory condition not handled");
                exit(EXIT_FAILURE);
            }
            /* all lines have been printed, we can safely rebuild the
             * cache */
            reallocCache(1);
            no_mem = 0;
        }

        /* if there is a line that got postponed due to memory issues,
         * handle it now */
        if (line_no_mem) {
            line = line_no_mem;
            line_no_mem = NULL;
        } else {
            /* get space to hold the next line of input */
            if (free_list) {
                line = free_list;
                free_list = free_list->next;
                line->len = 0;
                line->part_count = 0;
                line->query_count = 0;
                line->next = NULL;
            } else {
                /* allocate a line */
                line = allocLine();
                if (!line) {
                    if (head) {
                        /* print all entries and then rebuild the cache */
                        no_mem = __LINE__;
                        continue;
                    }
                    /* no lines are in waiting but we cannot create a new
                     * line.  not much we can do */
                    PERROR_MEM;
                    exit(EXIT_FAILURE);
                }
            }

            /* we use the static buffer to read the line, then copy
             * the line into fresh memory */
            line->buf = line_buf;
            rv = getLine(line, sizeof(line_buf));
            if (1 == rv) {
                /* no more data.  add line to free_list */
                eof = 1;
                line->buf = NULL;
                line->next = free_list;
                free_list = line;
                continue;
            }
        }

        line->buf = malloc(1 + line->len);
        if (NULL == line->buf) {
            /* memory error processing code is at bottom of outermost
             * while()  */
            no_mem = __LINE__;
            goto LINE_NO_MEM;
        }

        memcpy(line->buf, line_buf, 1 + line->len);

        for (i = 0; !no_mem && i < line->part_count; ++i) {
            if (!line->part[i].has_addr) {
                /* no lookup for this part */
                continue;
            }

            /* check for the IP or get space to store it */
            rv = hashlib_insert(hash, (uint8_t*)&line->part[i].ip,
                                (uint8_t**)&cache_id);
            switch (rv) {
              case OK_DUPLICATE:
                line->part[i].waiting = 0;
                /* found in cache. nothing to do with it yet */
                break;

              case ERR_OUTOFMEMORY:
              case ERR_NOMOREBLOCKS:
                /* memory error processing code is at bottom of
                 * outermost while()  */
                no_mem = __LINE__;
                goto LINE_NO_MEM;

              case OK:
                /* new entry; must do the DNS lookup */
                snprintf(arpa_addr, sizeof(arpa_addr),
                         "%d.%d.%d.%d.in-addr.arpa",
                         (line->part[i].ip & 0xFF),
                         ((line->part[i].ip >> 8) & 0xFF),
                         ((line->part[i].ip >> 16) & 0xFF),
                         ((line->part[i].ip >> 24) & 0xFF));
                rv = adns_submit(adns, arpa_addr, adns_r_ptr,
                                 adns_qf_quoteok_cname|adns_qf_cname_loose,
                                 NULL, &line->query[line->query_count]);
                if (0 == rv) {
                    *cache_id = RWRES_WAITING;
                    line->part[i].waiting = 1;
                    ++line->query_count;
                    ++num_requests;
                } else if (ENOMEM == rv) {
                    /* memory error processing code is at bottom of
                     * outermost while()  */
                    no_mem = __LINE__;
                    goto LINE_NO_MEM;
                } else {
                    /* treat as if lookup failed */
                    *cache_id = RWRES_NONAME;
                    line->part[i].waiting = 0;
                }
            }
        }

        /* add the line to the linked list */
        if (NULL == head) {
            head = line;
        } else {
            tail->next = line;
        }
        tail = line;

        /* we're good */
        continue;

        /* error processing code for failure to create a line */
      LINE_NO_MEM:
        /* postpone processing of this line.  free all resources it is
         * using.  loop around so we can print all outstanding lines,
         * rebuild the cache, then try again. */
        for (j = 0; j < line->query_count; ++j) {
            adns_cancel(line->query[j]);
        }
        if (line->buf) {
            free(line->buf);
            line->buf = NULL;
        }
        line_no_mem = line;

        if (!head) {
            /* there are no outstanding lines. try to empty the cache */
            if (0 == hashlib_count_entries(hash)) {
                /* name cache is empty but still could not allocate
                 * memory.  there is not much way to save ourselves */
                PERROR_MEM_LINE(no_mem);
                exit(EXIT_FAILURE);
            }
            reallocCache(1);
        }
    }

    while (free_list) {
        line = free_list;
        free_list = free_list->next;
        freeLine(line);
    }
    freeLine(line_no_mem);

    if (answers) {
        free(answers);
    }
    adns_finish(adns);

    return 0;
}
#endif  /* SK_HAVE_ADNS_H */


int main(int argc, char **argv)
{
    int rv;

    appSetup(argc, argv);                       /* never returns on error */

#ifdef SK_HAVE_ADNS_H
    if (resolver == RESOLVE_ADNS_SUBMIT) {
        rv = resolve_adns_submit();
    } else
#endif
    {
        rv = resolve_synchronous();
    }

    appTeardown();
    return (rv ? 1 : 0);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
