/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwipfix2silk.c
**
**    IPFIX to SiLK translation filter application
**
**    Brian Trammell
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwipfix2silk.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skipfix.h>
#include <silk/sksite.h>
#include <silk/rwrec.h>
#include <silk/utils.h>
#include <silk/skstringmap.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

/* where to write --print-stat output */
#define STATS_FH stderr


/* LOCAL VARIABLE DEFINITIONS */

/* index into argv for looping over input */
static int arg_index;

/* the SiLK flow file to write */
static skstream_t *silk_output = NULL;

/* whether to print statistics */
static int print_statistics = 0;

/* whether to decode vlan values */
static int decode_vlan = 0;

/* the compression method to use when writing the file.
 * sksiteCompmethodOptionsRegister() will set this to the default or
 * to the value the user specifies. */
static sk_compmethod_t comp_method;


/* OPTIONS SETUP */

typedef enum {
    OPT_SILK_OUTPUT,
    OPT_PRINT_STATISTICS,
    OPT_INTERFACE_VALUES
} appOptionsEnum;

static struct option appOptions[] = {
    {"silk-output",             REQUIRED_ARG, 0, OPT_SILK_OUTPUT},
    {"print-statistics",        NO_ARG,       0, OPT_PRINT_STATISTICS},
    {"interface-values",        REQUIRED_ARG, 0, OPT_INTERFACE_VALUES},
    {0,0,0,0}                   /* sentinel entry */
};

static const char *appHelp[] = {
    ("Write the SiLK Flow records to the specified path.\n\tDef. stdout"),
    "Print the count of processed records. Def. No.",
    ("Specify value to store in 'input' and 'output'\n"
     "\tfields.  Def. snmp.  Choices: snmp, vlan"),
    (char *)NULL
};



/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  parseInterfaceValue(const char *if_value_choice);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES] [IPFIX_FILES]\n"                                             \
     "\tReads IPFIX records from files named on the command line or from\n"   \
     "\tthe standard input, converts them to the SiLK format, and writes\n"   \
     "\tthe SiLK records to the named file or to the standard output.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    skOptionsNotesUsage(fh);
    sksiteCompmethodOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;
    int rv;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* close SiLK flow output file */
    if (silk_output) {
        rv = skStreamClose(silk_output);
        if (rv && rv != SKSTREAM_ERR_NOT_OPEN) {
            skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
        }
        skStreamDestroy(&silk_output);
    }

    skOptionsNotesTeardown();
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    sk_file_header_t *silk_hdr;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsNotesRegister(NULL)
        || sksiteCompmethodOptionsRegister(&comp_method))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* arg_index is looking at first file name to process */
    if (arg_index == argc) {
        if (FILEIsATty(stdin)) {
            skAppPrintErr("No input files on command line and"
                          " stdin is connected to a terminal");
            skAppUsage();       /* never returns */
        }
    }

    /* default output is "stdout" */
    if (!silk_output) {
        if ((rv =skStreamCreate(&silk_output,SK_IO_WRITE,SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(silk_output, "stdout")))
        {
            skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
            exit(EXIT_FAILURE);
        }
    }

    /* get the header */
    silk_hdr = skStreamGetSilkHeader(silk_output);

    /* open the output */
    if ((rv = skHeaderSetCompressionMethod(silk_hdr, comp_method))
        || (rv = skOptionsNotesAddToStream(silk_output))
        || (rv = skHeaderAddInvocation(silk_hdr, 1, argc, argv))
        || (rv = skStreamOpen(silk_output))
        || (rv = skStreamWriteSilkHeader(silk_output)))
    {
        skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;  /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_SILK_OUTPUT:
        if (silk_output) {
            skAppPrintErr("--%s specified multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if ((rv =skStreamCreate(&silk_output,SK_IO_WRITE,SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(silk_output, opt_arg)))
        {
            skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
            exit(EXIT_FAILURE);
        }
        break;

      case OPT_PRINT_STATISTICS:
        print_statistics = 1;
        break;

      case OPT_INTERFACE_VALUES:
        if (parseInterfaceValue(opt_arg)) {
            return 1;
        }
        break;
    }

    return 0;  /* OK */
}


/*
 *  ok = parseInterfaceValue(choice);
 *
 *    Parse the user's choice for the --interface-values switch and
 *    set the global variable 'decode_vlan' appropriately.  Return 0
 *    on success, or -1 if the user's choice is not valid.
 */
static int parseInterfaceValue(
    const char *if_value_choice)
{
    const sk_stringmap_entry_t if_values[] = {
        {"snmp", 0, NULL, NULL},
        {"vlan", 1, NULL, NULL},
        SK_STRINGMAP_SENTINEL
    };
    sk_stringmap_t *str_map = NULL;
    sk_stringmap_status_t rv_map;
    sk_stringmap_entry_t *map_entry;
    int rv = -1;

    /* create a stringmap of the available interface values */
    if (SKSTRINGMAP_OK != skStringMapCreate(&str_map)) {
        skAppPrintErr("Unable to create stringmap");
        goto END;
    }
    if (skStringMapAddEntries(str_map, -1, if_values) != SKSTRINGMAP_OK) {
        goto END;
    }

    /* attempt to match */
    rv_map = skStringMapGetByName(str_map, if_value_choice, &map_entry);
    switch (rv_map) {
      case SKSTRINGMAP_OK:
        decode_vlan = map_entry->id;
        rv = 0;
        break;

      case SKSTRINGMAP_PARSE_AMBIGUOUS:
        skAppPrintErr("Invalid %s: Ambiguous value '%s'",
                      appOptions[OPT_INTERFACE_VALUES].name, if_value_choice);
        break;

      case SKSTRINGMAP_PARSE_NO_MATCH:
        skAppPrintErr(("Invalid %s: Unrecognized value '%s'"),
                      appOptions[OPT_INTERFACE_VALUES].name, if_value_choice);
        break;

      default:
        skAppPrintErr("Unexpected return value from string-map parser (%d)",
                      rv_map);
        break;
    }

  END:
    if (str_map) {
        skStringMapDestroy(str_map);
    }
    return rv;
}


/*
 *  ok = appNextInput(argc, argv, &fpath, &fp);
 *
 *    Open the next input file from the command line or the standard
 *    input if no files were given on the command line.  Return the
 *    name of the file in 'fpath' and the opened file pointer in 'fp'.
 *
 *    Return 0 on success, 1 if no more files, and -1 on error.
 */
static int appNextInput(int argc, char **argv, const char **fpath, FILE **fp)
{
    static int initialized = 0;
    const char *fname = NULL;

    if (arg_index < argc) {
        /* get current file and prepare to get next */
        fname = argv[arg_index];
        ++arg_index;
    } else if (initialized) {
        /* no more input */
        return 1;
    } else {
        /* input is from stdin */
        fname = "stdin";
    }

    initialized = 1;

    /* check for special names */
    if (0 == strcmp(fname, "stdin") || 0 == strcmp(fname, "-")) {
        *fpath = fname;
        *fp = stdin;
        return 0;
    }

    /* open file */
    *fp = fopen(fname, "rb");
    if (*fp == NULL)
    {
        skAppPrintSyserror("Unable to open file '%s'", fname);
        return -1;
    }

    *fpath = fname;
    return 0;
}


int main(int argc, char **argv)
{
    fBuf_t *fbuf = NULL;
    GError *err = NULL;
    const char *ipfix_path;
    FILE *ipfix_in;
    rwRec rec, revRec;
    uint64_t fwd_count = 0;
    uint64_t rev_count = 0;
    int rv;

    appSetup(argc, argv);                       /* never returns on error */

    /* initialize the rwrec */
    RWREC_CLEAR(&rec);
    RWREC_CLEAR(&revRec);

    /* loop over all input */
    while ((rv = appNextInput(argc, argv, &ipfix_path, &ipfix_in)) == 0) {
        /* Create IPFIX buffer for reading */
        if (!(fbuf = skiCreateReadBufferForFP(ipfix_in, &err))) {
            skAppPrintErr("Could not open %s for IPFIX: %s",
                          ipfix_path, err->message);
            exit(EXIT_FAILURE);
        }

        /* read IPFIX and write RW */
        while (skiRwNextRecord(fbuf, &rec, &revRec, &err, decode_vlan)) {
            rv = skStreamWriteRecord(silk_output, &rec);
            if (rv) {
                skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
                exit(EXIT_FAILURE);
            }
            ++fwd_count;
            if (rwRecGetBytes(&revRec) && rwRecGetPkts(&revRec)) {
                rv = skStreamWriteRecord(silk_output, &revRec);
                if (rv) {
                    skStreamPrintLastErr(silk_output, rv, &skAppPrintErr);
                    exit(EXIT_FAILURE);
                }
                ++rev_count;
            }
        }

        /* Handle EOF condition---a normal error */
        if ( !g_error_matches(err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            skAppPrintErr("Could not read IPFIX record from %s: %s",
                          ipfix_path, err->message);
            exit(EXIT_FAILURE);
        }

        g_clear_error(&err);

        fBufFree(fbuf);
        fclose(ipfix_in);
    }
    if (rv == -1) {
        exit(EXIT_FAILURE);
    }

    if (print_statistics) {
        fprintf(STATS_FH, ("%s: Wrote %" PRIu64 " records"
                           " (%" PRIu64 " forward %" PRIu64 " reverse)"
                           " to '%s'\n"),
                skAppName(), (fwd_count + rev_count),
                fwd_count, rev_count, skStreamGetPathname(silk_output));
    }

    appTeardown();
    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
