#! /usr/bin/perl -w
# ERR_MD5: varies
# TEST: ./rwflowpack ----sensor-conf=sk-teststmp-sensor.conf --verify-sensor 2>&1"

use strict;
use SiLKTests;

srand(627389505);

my $rwflowpack = $ENV{RWFLOWPACK} || './rwflowpack';

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or exit 77;


my $sensor_conf = "./$SiLKTests::SILK_TESTS_PREFIX-sensor.conf";

# clean up when we're done
END {
    if (!$ENV{SK_TESTS_SAVEOUTPUT}) {
        # remove files
        if (defined $sensor_conf) {
            unlink $sensor_conf;
        }
    }
}

# list of probe types
my @probe_types = qw(netflow-v5 ipfix silk);

# ways to get data into SiLK
my @sources_valid =
    ("listen-on-port 9900\n    protocol udp",
     "listen-on-port 9900\n    protocol tcp",
     "poll-directory /tmp/incoming-nf",
     "read-from-file /tmp/incoming-nf",
    );
# the following are invalid/unsupported sources
my @sources_bad = (
     "listen-on-unix-socket /tmp/incoming-nf",
     "listen-on-port 9900",
     "listen-on-port 9900,9901\n    protocol tcp",
    );

my @probe_extras =
    ("listen-as-host 192.168.22.22",
     "accept-from-host 192.168.22.22",
     ("listen-as-host 192.168.22.22\n    accept-from-host 192.168.22.22"),
     "log-flags all",
     "log-flags none",
     "log-flags bad",
     "log-flags missing",
     "log-flags junk",
     "log-flags bad, missing",
     "log-flags all none",
     "priority 25",
     "priority 125",
     "interface-values snmp",
     "interface-values vlan",
     "interface-values junk",
    );

my @networks = qw(internal external null);

my @interfaces =
    ('7',
     '7,8',
     'remainder');

my @ipblocks =
    ('172.16.22.22/31',
     '172.16.22.22, 172.16.22.23',
     '2001:172:16::22:22/127',
     '2001:172:16::22:22 2001:172:16::22:23',
     'remainder');


unlink $sensor_conf;
open SENSOR, ">$sensor_conf"
    or die "ERROR: Cannot open $sensor_conf: $!\n";

my $probe = 0;
my $group = 0;
my $sensor = 0;
my @sensor_list = ();

for my $pt (@probe_types) {
    for my $src (@sources_valid, @sources_bad) {
        my $name = sprintf("PROBE-%04d", ++$probe);

        # make certain file/directory names are unique
        my $uniq_src = $src;
        $uniq_src =~ s,(incoming-nf),$1-$probe,;

        print SENSOR <<EOF;
probe $name $pt
    $uniq_src
end probe
EOF
    }
}

# add various statements to valid NetFlow v5 probes
for my $src (@sources_valid) {
    next if $src =~ /tcp/;
    for my $xtra (@probe_extras) {
        my $name = sprintf("PROBE-%04d", ++$probe);

        # make certain file/directory names are unique
        my $uniq_src = $src;
        $uniq_src =~ s,(incoming-nf),$1-$probe,;

        print SENSOR <<EOF;
probe $name $probe_types[0]
    $uniq_src
    $xtra
end probe
EOF
    }
}


# create groups
print SENSOR "\n\n";

for my $i (@interfaces) {
    next if $i eq 'remainder';

    my $name = sprintf("GROUP-%04d", ++$group);
    if ($i eq $interfaces[0]) {
        push @interfaces, '@'.$name;
    }

    print SENSOR <<EOF;
group $name
    interfaces $i
end group
EOF
}

for my $i (@ipblocks) {
    next if $i eq 'remainder';

    my $name = sprintf("GROUP-%04d", ++$group);
    if ($i eq $ipblocks[0]) {
        push @ipblocks, '@'.$name;
    }

    print SENSOR <<EOF;
group $name
    ipblocks $i
end group
EOF
}

for my $g ($interfaces[-1], $ipblocks[-1], '@NOT_EXIST') {
    for my $i ('interfaces 0', 'ipblocks 64.0.0.0/8') {

        my $name = sprintf("GROUP-%04d", ++$group);

        print SENSOR <<EOF;
group $name
    $i
    $i, $g
end group
EOF
    }
}


# create sensors
print SENSOR "\n\n";

# these should all be valid
for my $pt (@probe_types) {
    for my $src (@networks) {
        for my $dst (@networks) {

            my $name = sprintf("SENSOR-%04d", ++$sensor);
            push @sensor_list, $name;

            print SENSOR <<EOF;
sensor $name
    $pt-probes PROBE-$pt
    source-network $src
    destination-network $dst
end sensor
EOF
        }
    }
}

# some of these are valid and some are not
for my $nw (@networks) {
    my ($src_nw, $dst_nw) = grep !/$nw/, @networks;
    my @end_sensor;
    if ($dst_nw eq 'null') {
        @end_sensor = ("end sensor");
    }
    else {
        @end_sensor = ("end sensor", "    null-interface 0\nend sensor");
    }

    for my $if_type (qw(interfaces ipblocks)) {
        my ($src_if_list, $dst_if_list);
        if ($if_type eq 'interfaces') {
            $src_if_list = \@interfaces;
            $dst_if_list = \@interfaces;
        }
        else {
            $src_if_list = \@ipblocks;
            $dst_if_list = \@ipblocks;
        }

        for my $src_if (@$src_if_list) {
            for my $dst_if (@$dst_if_list) {
                for my $end (@end_sensor) {

                    # discard-when/discard-unless
                    my $discard = "";

                    for my $dis_phrase ('source-ipblocks 10.0.0.0/8',
                                        'destination-ipblocks 11.0.0.0/8',
                                        'any-ipblocks '.$ipblocks[-1],
                                        'source-interfaces 10',
                                        'destination-interfaces 11',
                                        'any-interfaces '.$interfaces[-1])
                    {
                        my $chance = rand 10;
                        if ($chance < 2) {
                            $discard .= "\n    discard-when $dis_phrase";
                        }
                        elsif ($chance < 4) {
                            $discard .= "\n    discard-unless $dis_phrase";
                        }
                        # else do nothing
                    }

                    my $name = sprintf("SENSOR-%04d", ++$sensor);
                    push @sensor_list, $name;

                    print SENSOR <<EOF;
sensor $name
    $probe_types[0]-probes PROBE-$probe_types[0]$discard
    $src_nw-$if_type $src_if
    $dst_nw-$if_type $dst_if
$end
EOF
                }
            }
        }
    }
}

# check handling of duplicate source/destination
for my $sd (qw/source destination/) {
    my $name = sprintf("SENSOR-%04d", ++$sensor);
    push @sensor_list, $name;

    print SENSOR <<EOF;
sensor $name
    $probe_types[0]-probes PROBE-$probe_types[0]
    source-network internal
    destination-network external
    $sd-network internal
end sensor
EOF
}

# check handling of duplicate network
for my $nw (@networks) {
    my $name = sprintf("SENSOR-%04d", ++$sensor);
    push @sensor_list, $name;

    print SENSOR <<EOF;
sensor $name
    $probe_types[0]-probes PROBE-$probe_types[0]
    internal-interfaces 4
    external-interfaces 5
    null-interfaces 6
    $nw-interfaces 7
end sensor
EOF
}

close SENSOR
    or die "ERROR: Cannot close $sensor_conf: $!\n";

my $silk_conf = make_tempname('silk');
open SILK, ">$silk_conf"
    or die "ERROR: Cannot open $silk_conf: $!\n";

for (my $i = 0, my $j = 1; $i < @sensor_list; ++$i, ++$j) {
    my $s = $sensor_list[$i];
    print SILK "sensor $j $s\n";
}

print SILK "group my-sensors\n";
for my $s (@sensor_list) {
    print SILK "    sensors $s\n";
}

print SILK <<EOF;
end group
class all
    sensors \@my-sensors
end class
class all
    type  0 in      in
    type  1 out     out
    type  2 inweb   iw
    type  3 outweb  ow
    type  4 innull  innull
    type  5 outnull outnull
    type  6 int2int int2int
    type  7 ext2ext ext2ext
    type  8 inicmp  inicmp
    type  9 outicmp outicmp
    type 10 other   other
    default-types in inweb inicmp
end class
packing-logic "packlogic-twoway.so"
EOF

close SILK
    or die "ERROR: Cannot close $silk_conf: $!\n";

my $cmd = ("$rwflowpack --site-conf=$silk_conf --sensor-conf=$sensor_conf"
           ." --verify-sensor 2>&1");
my $md5;
if ($SiLKTests::SK_ENABLE_IPV6 && $SiLKTests::SK_ENABLE_IPFIX) {
    # * IPv6 support:                     yes
    # * IPFIX collection support:         yes
    $md5 = 'ecc2082647cdd74fa410ffe4d6cc6a15';
}
elsif ($SiLKTests::SK_ENABLE_IPV6) {
    # * IPv6 support:                     yes
    # * IPFIX collection support:         no
    $md5 = '72888decc4bc6ed761e6663142ab7656';
}
elsif ($SiLKTests::SK_ENABLE_IPFIX) {
    # * IPv6 support:                     no
    # * IPFIX collection support:         yes
    $md5 = 'd5a7b209221a32d3b3c70404477adec0';
}
else {
    # * IPv6 support:                     no
    # * IPFIX collection support:         no
    $md5 = '092d6d10eee23f785fab871f38aaf880';
}

exit check_md5_output($md5, $cmd, 1);
