#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-send.pl 72e0a2f73883 2010-09-09 21:30:31Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = $ENV{RWFLOWPACK} || './rwflowpack';

# find the apps we need.  this will exit 77 if they're not available
my $rwuniq = check_silk_app('rwuniq');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or exit 77;

# create our tempdir.  clean up unless the appropriate environment
# variable is set.
my $tmpdir = File::Temp::tempdir(CLEANUP => !$ENV{SK_TESTS_SAVEOUTPUT});
exit 77 unless -d $tmpdir;

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'silk', 'polldir');

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     #"--log-level=debug",
                     "--sensor-conf=$sensor_conf",
                     "--output-mode=sending",
                     "--copy $file{data}:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--flush-timeout=5",
                     "--",
                     "--polling-interval=5",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);


# the following directories should be empty
my @empty_dirs = qw(error incoming incremental root);
for my $d (map {"$tmpdir/$_"} @empty_dirs) {
    next unless -d $d;
    unless (opendir D, $d) {
        warn "ERROR: Cannot open directory '$d': $!\n";
        next;
    }
    while (my $f = readdir(D)) {
        next if $f =~ /^\.\.?$/;
        die "ERROR: Found unexpected file '$d/$f'\n";
    }
}

# In sending mode, files are named with a mkstemp()-type extension.
# We must find all files having the same name minus this extension,
# and combine them before checking the output.

# path to the data directory
my $sender_dir = "$tmpdir/sender";
die "ERROR: Missing data directory '$sender_dir'\n"
    unless -d $sender_dir;

# number of files representing a unique {flowtype,sensor,hour} tuple
# to find; i.e., where we have mapped out-S0_20090212.00.LELcDc and
# out-S0_20090212.00.PzEA5g into one entry
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every {flowtype,sensor,hour} we expect to find.
# Read these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

my %file_list = ();

# find the files in the data directory
File::Find::find({wanted => \&find_files, no_chdir => 1}, $sender_dir);

# combine the files and check the output
while (my ($basename, $files) = each %file_list) {
    $cmd = ("$rwuniq --fields=1-5,type --ipv6-policy=ignore --sort-output"
            ." --values=records,packets,sTime-Earliest,eTime-Latest"
            .$files);
    check_md5_output($md5_map{$basename}, $cmd);
    ++$file_count;
}

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files types; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub find_files
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename, stripping the mkstemp
    # extension
    s,^.*/,,;
    s,([^.]+\.\d\d)\..+$,$1,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};

    $file_list{$_} .= " $path";
}


__DATA__
ef699b8e35f975e6ad72d5017bf844ff  in-S0_20090212.00
e4b67ebff2cb0c16fdaa2663b3fa8a2e  in-S0_20090212.01
de12ec57b9335963609f2528d29c4c55  in-S0_20090212.02
2f6dfe30529d5ddc5d24adec1012ef98  in-S0_20090212.03
841323ef7d998900fa5ef63f387df526  in-S0_20090212.04
53c8c0f0fe3e60f81d1ad141896ccfc1  in-S0_20090212.05
8636a7193e3f8e062a37a53daa88d7d7  in-S0_20090212.06
dfb139bab142130f3f6d89a527ace733  in-S0_20090212.07
e69fbeef2221cb49c054e9f10f06181d  in-S0_20090212.08
e00f8290aabf7818a607eed219e7ff13  in-S0_20090212.09
fbd88c5a982d4cf59b98899e0c14cd6e  in-S0_20090212.10
e85c8e87bea6ed036b8b91360e5b014a  in-S0_20090212.11
684fda8efde061c21a9c8ea039e4e6e4  in-S0_20090212.12
14664647147b7a97a5913a48353f4756  in-S0_20090212.13
9ea8839043322d32026e93a61d857c96  in-S0_20090212.14
6fd1cdafa888114302d96ea7ca7e5ae7  in-S0_20090212.15
5b89c54856c326f9be16303d88f3f02f  in-S0_20090212.16
ab555c09a4a995845eda536d7f90d1cd  in-S0_20090212.17
7258f2be8525bac4843f236d92689e30  in-S0_20090212.18
2f4a87f6ca2872c666ceb8b4f2c16252  in-S0_20090212.19
5c1c568bf5924ea987480bc1637734fe  in-S0_20090212.20
ec891b8936673c08244bea6ad4c8d2d3  in-S0_20090212.21
f18f8cb5720b3516950e44c6b9bc6810  in-S0_20090212.22
f7c91c1d9253f05b02e274c14c043c95  in-S0_20090212.23
f30878f5ccf6b594d1674c15d422920a  in-S0_20090213.00
fc7a58aa58bd8244fcf7128e4e0fb8ef  in-S0_20090213.01
8cd91ad313b9eaa513c86ffc8e37523b  in-S0_20090213.02
5044ccf2e6202bf9d64c756b2cc9b9c1  in-S0_20090213.03
014251d388cef9c3f684d76bddb638cf  in-S0_20090213.04
610101e2564b000e6504161bcde2f0ba  in-S0_20090213.05
1b5cf39da997c126468e104b79ac7dfa  in-S0_20090213.06
15b1df2b59d5706b9c079f5d250f8924  in-S0_20090213.07
95af1af84201bd0d106f6a62dde555c8  in-S0_20090213.08
02c247842146144c759d5d31f1662c78  in-S0_20090213.09
33bb0a7f18f65ddc7de0a0ffff987f14  in-S0_20090213.10
74ee57c98752554ab3de761d50571947  in-S0_20090213.11
2f9abbfdb644821d1461d1c9c9f26d34  in-S0_20090213.12
c52be7ad15babfb329e2181228d1a652  in-S0_20090213.13
3ee486147d8f9b2a11b13fcf4969935c  in-S0_20090213.14
dda2d12d471d9d4dea34bbc1d1a1b65a  in-S0_20090213.15
6d69300e544595c4546ba30806313c9d  in-S0_20090213.16
20ba7eb04567ecc52705ce1939da0eff  in-S0_20090213.17
7290b6ea0143e67f3c728f49f1120ad0  in-S0_20090213.18
96d6bc9234e2d2e1f34c4f169bc0e355  in-S0_20090213.19
e410782d6b6f5250d86beaf447239b12  in-S0_20090213.20
61245d4c7fc541b8f8bcf8bbe6679aa2  in-S0_20090213.21
7fd37f979727f0ba975884caab779a47  in-S0_20090213.22
c997c24c659a54e95f4ecb017d1b90e5  in-S0_20090213.23
4b51e3c60bdb8d0953cc38c6ef7e9d14  in-S0_20090214.00
977a6c5fdaf29799c666222e268c3ae5  in-S0_20090214.01
e451f8a3c242835765e0f173a085fd3b  in-S0_20090214.02
1415931c89617c994d1755baa3917b55  in-S0_20090214.03
b4cac13a9d61f16a09256c233355aa6e  in-S0_20090214.04
95f49097b541788bc8298c1ec77af05f  in-S0_20090214.05
1306531de9f663cbd662f636e12ff7d2  in-S0_20090214.06
66ded577f610fb1621912351952de4f4  in-S0_20090214.07
ad27736ab60c5aa1599115b38a77aa20  in-S0_20090214.08
dc59ad2e6e0b4948058cd2ab55c853f5  in-S0_20090214.09
d42f104f1799f23a2ef10008eedf11c9  in-S0_20090214.10
dcc5be2f1b8f414b9af05c2a57b03314  in-S0_20090214.11
fd9f13735c2f4ce0ae3e3ea41807ed25  in-S0_20090214.12
ad4c9b21844be2aa2e74c3363d21c3d7  in-S0_20090214.13
9131b3e8a6c5835521d3e7cc9db54006  in-S0_20090214.14
c152627acde67c09a00a0f8a4110a525  in-S0_20090214.15
22c65070c60e4f059dc9aa571d235496  in-S0_20090214.16
020f6ed4844f586819378c445c7fcb2e  in-S0_20090214.17
d737df80fccf61604cd83196b64b6bd6  in-S0_20090214.18
e1957d3fe89d9b05d30d88003e94df92  in-S0_20090214.19
010da11384820ed951e4c4714c8ac140  in-S0_20090214.20
fa89f582ccf01f8835f68172c3e0c1c0  in-S0_20090214.21
9b2028f903ad4970e8bd01f1129f1d7c  in-S0_20090214.22
c4ceb2e8da8f63f771c8f799fa9aeedf  in-S0_20090214.23
8a23e2e42ef7fdcf0dfc7472f775d9e2  iw-S0_20090212.00
9807adcf8e682e34ea070c40d0bc38d1  iw-S0_20090212.01
cfb82782475b40e16a0474636946abb5  iw-S0_20090212.02
0446b9825fce1afd30288e8bcdc3cb00  iw-S0_20090212.03
6474a08226a95263a987675836da30ac  iw-S0_20090212.04
5a1cb7222af35d0437a19f489e575f93  iw-S0_20090212.05
1f1baa05809c8f30a90d1a2ad71f5e48  iw-S0_20090212.06
179d1c43fd0f5bacd84ea06eed48c17e  iw-S0_20090212.07
40304e913a29f9f528625f2878f53a4a  iw-S0_20090212.08
029ee38739c6f704b25f29d8589105b7  iw-S0_20090212.09
6c21faf9d6b83bf48ef04a56f41e343c  iw-S0_20090212.10
61bfdf000ad96031e630f57b7acafeb5  iw-S0_20090212.11
2e59c864f3557df8f7df9758bf9da921  iw-S0_20090212.12
e802c6e903ea155aa80f3a1466c0a4a0  iw-S0_20090212.13
98442c4f31a5a9c71290be9b7350287d  iw-S0_20090212.14
90e8ba20bacd5c8a72e82cf048d3eab8  iw-S0_20090212.15
dcf3067b9ccb14a55bec542f7e6f0eca  iw-S0_20090212.16
2c2b657dfe2c3230bfab34cf69b4753a  iw-S0_20090212.17
04cb249984a3186d673e42bd7399f5eb  iw-S0_20090212.18
6b4b45ff204b7ab6adff5258de0d4dce  iw-S0_20090212.19
38d41a1b2bb1d1a873c1be79685cb4f1  iw-S0_20090212.20
dbb10e3c364cfce720c376a09ef5dca2  iw-S0_20090212.21
9f49d4d95aa06e6ccd048bb3a2e7717d  iw-S0_20090212.22
d9603cd9242e85c1e3bfaef25b80be26  iw-S0_20090212.23
9f45a92ddf15e7f93bc75b04491af9ab  iw-S0_20090213.00
a6859845de0545dd0977e383c6a83876  iw-S0_20090213.01
21cac0da71976e79577e404381207f74  iw-S0_20090213.02
d9971929dfe0ea9324ba40fc470a9769  iw-S0_20090213.03
fdaf0f2bafb2b6ce8999560f556cb0e0  iw-S0_20090213.04
fae82f9c75651d0728daec11ed14923a  iw-S0_20090213.05
fb66851c75a2fb348bdbcea5c845db24  iw-S0_20090213.06
b49c6afb399f23a97ce8a49504eca0f3  iw-S0_20090213.07
f47e73672baaeed7587cdfe1d68b242f  iw-S0_20090213.08
064983eeea031adf08cddbdf4449fb63  iw-S0_20090213.09
9c7ada3aed9df66f1230261b0c696c80  iw-S0_20090213.10
102d4e0743495816b2e9a85c92b03943  iw-S0_20090213.11
1e285b862ad69fc8bc989775810a1b31  iw-S0_20090213.12
fe3fb61347e12c2e254031028e536785  iw-S0_20090213.13
c13dbc937cbc502ec61dcb6ecab8d6a3  iw-S0_20090213.14
8421b43ae77f885f42ef66922be2955e  iw-S0_20090213.15
f0180f7c8a244cbc9f20ddc6dab8cf06  iw-S0_20090213.16
cf74c6d184c48c209541b7f2278fc679  iw-S0_20090213.17
e5ffd27bfb598c169fff37b0457b5f8f  iw-S0_20090213.18
f92b57b7cb6ef22bc848ffc11ee37cb2  iw-S0_20090213.19
417826bba59c330160c48363c2880fbe  iw-S0_20090213.20
ba57fde9ec252385e59ba3cb6572ffb1  iw-S0_20090213.21
4c543bd123848d8e906427a5ad88558e  iw-S0_20090213.22
931fba62408c715c32c093d03d30a03c  iw-S0_20090213.23
112b697702a1bfe73c981de3ffceab02  iw-S0_20090214.00
862c6fe05e520b5d1574ccb7ff531a73  iw-S0_20090214.01
2b7a64b82d7786f7c62c4dc5a649b877  iw-S0_20090214.02
a9590370e4760473aae9f17d940a766b  iw-S0_20090214.03
8db63d592c58b4fc792ef8191cc68f99  iw-S0_20090214.04
f0aa296eb49c623c99e808124c465063  iw-S0_20090214.05
7153c800dc46c6b80170bc070e0c5b0b  iw-S0_20090214.06
d0f421d197d4a274149d736657ff1748  iw-S0_20090214.07
76b9263d7d93a553c164c6ab8f96db7b  iw-S0_20090214.08
dc6277b35f42485feff47d61d4147ed5  iw-S0_20090214.09
8ac6e247709c9826c10430e7435ebe4a  iw-S0_20090214.10
a2b2a21bc6efcd805843ea889945901b  iw-S0_20090214.11
4cdb066baffa4f1693891dd818f5caf5  iw-S0_20090214.12
298cb48f87ffc05cbe7d10811b221ca1  iw-S0_20090214.13
4001d447582c4779518e276a70ee2f15  iw-S0_20090214.14
cb3901f5427790dea70a43aaa74b21b7  iw-S0_20090214.15
70dbe38512425677a1306efc26ef4930  iw-S0_20090214.16
3797c2eb4aa4a1933ad435f9f22eb652  iw-S0_20090214.17
4a0d628dfd7967e194ad58290b6fe298  iw-S0_20090214.18
383c2ce4a54f72661716fa1bb58a9894  iw-S0_20090214.19
f00cc6c65ecb59d6d85ae43a040cff02  iw-S0_20090214.20
e9f0eb46381f86e803c83406391d9709  iw-S0_20090214.21
6f48c3fe4b7e046b28472350067a1788  iw-S0_20090214.22
a69c1aa3a0ee86724626d38661021d29  iw-S0_20090214.23
498f4d07a3b2cfb405b8534a052ec77a  other-S0_20090212.00
8fb61c046c01d642c4c5f0fc469be9c8  other-S0_20090212.01
a5bb5eb3a22d080a509fef7239648b54  other-S0_20090212.02
35b6f4e0a00c2b396374307b1cd1f6d5  other-S0_20090212.03
2eb02b579d771ece17068d63143d83c0  other-S0_20090212.04
51de3572690dfe4965920b671fb432f2  other-S0_20090212.05
a5b7ab751dff561e9ea8d319fda87412  other-S0_20090212.06
757f0d761f477e225dfe9807f45264e5  other-S0_20090212.07
268006d9ead56ea88ff0bf9d55d9be84  other-S0_20090212.08
7fbd74517107e6031bcf58114feaf682  other-S0_20090212.09
95c4d0f85cb67e98fc32ff3b9e3c119d  other-S0_20090212.10
b8252ea9bbc1d678b2c40947bedf28af  other-S0_20090212.11
7d8719ca78144bd9608b76b0c9a35eaf  other-S0_20090212.12
97047c592370c19401e9e3be6bc884d2  other-S0_20090212.13
020d92c6f8a41d513a6759fe2ecb35f1  other-S0_20090212.14
8052771b895274cac671b426623c46ce  other-S0_20090212.15
778e956aeae7efb29dc65696cee3801f  other-S0_20090212.16
8324ef7a5972452480af6f7aa1d9f16f  other-S0_20090212.17
a08f229b37cf8e5f6e3fb594b7701ad8  other-S0_20090212.18
b9f44a3c0ffd0da18344c9a68925ff7e  other-S0_20090212.19
e8dc691ded77613a34e7b399f7b76291  other-S0_20090212.20
db25dfda3b7231c9b1a009c6037cc3ae  other-S0_20090212.21
5007bda1c0e32a8a38bed3ee62985c91  other-S0_20090212.22
7d564e96c706a4e0012085836bbb89fd  other-S0_20090212.23
c2d616cd61ed0fce60752bdf97ef6955  other-S0_20090213.00
62f662fae7f890d03636c43b8eb0ba20  other-S0_20090213.01
b71c14c2985deb7910303080eb912e51  other-S0_20090213.02
0b84007070c4b3f40c3faea449343c0b  other-S0_20090213.03
ccf58679feb4fad9d4bdff69642cb18c  other-S0_20090213.04
e156728e32e4f0f6d19a95fdf34b3f15  other-S0_20090213.05
0fb1784e9dda380ccd6382f72e6ed08d  other-S0_20090213.06
7eb152527e56bb4525fb6d4808fa375f  other-S0_20090213.07
66b9921d1e816c391c37ef5b31b46024  other-S0_20090213.08
5ed4f3680aa3842e39abe2625f331e67  other-S0_20090213.09
ffd1428a7f13855e703558cf6f5758e1  other-S0_20090213.10
183a29fdde6670967b323d40490a7003  other-S0_20090213.11
3f132d8b87aa505f983cd08ce562073b  other-S0_20090213.12
e7f7889755f6ce95ed40113ba32d63da  other-S0_20090213.13
55a559d26f756c495f88a24b7871d134  other-S0_20090213.14
bcefa24bc77f39d4f274eb762da28431  other-S0_20090213.15
719da291dac29c7396e3faf417492a82  other-S0_20090213.16
b472df7bf61a3ab55fefca2573db3d38  other-S0_20090213.17
f994f480860dfbe7e6215d162f614597  other-S0_20090213.18
29cedf38969bdfc61e0b077b4eb235a5  other-S0_20090213.19
23b58159dd645ccc5ba04f38cbef9127  other-S0_20090213.20
4dd185fb0bab3ad5711ca598de9bccec  other-S0_20090213.21
ae570941af3815bd29f6cf457e89e6af  other-S0_20090213.22
dc21aa2b0aa3391f451b75d73cfe0472  other-S0_20090213.23
4484dda94ed42188e7d0426894346c2b  other-S0_20090214.00
3e0c9490dfcd26a9ea2f16f9f3ad2a7e  other-S0_20090214.01
8ddc7b79b00342af76833f22172cb804  other-S0_20090214.02
3b4c298d29fc1c9caa242ca076de11d4  other-S0_20090214.03
58147beedba77c8633f37f0566838652  other-S0_20090214.04
72745c0f25703a5c39f132d609e0931e  other-S0_20090214.05
766f79db92fd67236b8c0a005fd431b8  other-S0_20090214.06
096a9010605a15d7ae93828438461330  other-S0_20090214.07
7df0b5b808f8b3a9c69b325f9a9e7a46  other-S0_20090214.08
e5747f574dad736f7d0f154f30eecfe8  other-S0_20090214.09
f11b604350b4cc45b5201272a3497d55  other-S0_20090214.10
b2ee4beac36cdaa3be58b0b639e6fd74  other-S0_20090214.11
ade60d6a1f803d0ad1a2f6bdf7c42549  other-S0_20090214.12
c87fbba37154aeb2088c2071c6175275  other-S0_20090214.13
0ed03072e522a1aa2881aaf6841bcdb6  other-S0_20090214.14
34e6770bd14e6b246a071f3d9dcb6c84  other-S0_20090214.15
871eb79fd9632db36b0430f40dbd6932  other-S0_20090214.16
0331966e3b9d52446cd1894673087b56  other-S0_20090214.17
72f8ba9ddaf74b586486bc46a0c97751  other-S0_20090214.18
4869e3f4b5b19b9c9bd7e7553e348fa0  other-S0_20090214.19
f17b5e64ffeecf0e8a1efcd7872f7d97  other-S0_20090214.20
ca1eec05f2a78f91af8abdd466aecc52  other-S0_20090214.21
603cde408b86ca4dc0bc6a0facf4bd4c  other-S0_20090214.22
339b37a25cdeb10a3f2f817b1f2e360d  other-S0_20090214.23
3c59c981abab6c972a4af827737da4a6  out-S0_20090212.00
bd5f025f0377429ac0f9879f5c854817  out-S0_20090212.01
752e70d877e1acb9f332c910e1d98085  out-S0_20090212.02
8b1258a32a83a016bbc8be8715996530  out-S0_20090212.03
cf8dc2355b24c3cbc29cc7b3ccb02986  out-S0_20090212.04
e9244fb64c5a20f441ca5422ae0dd83e  out-S0_20090212.05
840f658e4c04e993198bb0c6dcccafba  out-S0_20090212.06
c4fb55c121e517cec6e9ef8981a585c9  out-S0_20090212.07
bf408f184713d1d08b7124c048839785  out-S0_20090212.08
c027ed0a11bd8ae24870fa616951711e  out-S0_20090212.09
10af1babf925d7e6c880c1df5dbada46  out-S0_20090212.10
b76439905d6aafa8f023b42f581af5b9  out-S0_20090212.11
9e96aa15d76e4b1c0f1083d80d9e4d35  out-S0_20090212.12
fea25ed674e840c672e192ff393af94e  out-S0_20090212.13
d7c689619ff4d705c1fac49f17e1abaf  out-S0_20090212.14
47dd8d363edbd7fc881916ddf6daaedc  out-S0_20090212.15
e1cb2354b80934d0545a53d6424fcf38  out-S0_20090212.16
bd5c105f971aec3885f103b16e056098  out-S0_20090212.17
fb9532ea1d7b8e7eeede8ce09d82bc86  out-S0_20090212.18
06e94aba6b88d06474fde38ef7beb271  out-S0_20090212.19
072bda8b6e7354a4e1edeccca20477fe  out-S0_20090212.20
263f1c71d3ed2be8b9d21650c61dce8f  out-S0_20090212.21
d3e4d3747c30f085179fc7a7b58bd16e  out-S0_20090212.22
1f7c9bffb0ea3e9b5ae679334677439b  out-S0_20090212.23
f2a4a163145536e55535e3010b43680c  out-S0_20090213.00
c61a8e9d8df6e2308c88315025f8462a  out-S0_20090213.01
4062def20a87dfef8397b4d3dd1aeb5f  out-S0_20090213.02
4f0a6fd354eab52c39f00cc92bc299c5  out-S0_20090213.03
92fa58445f32ae2c5e043eedcfbff148  out-S0_20090213.04
622bc8dbdf012390e0aba00eee9c1704  out-S0_20090213.05
3b95f8a0d34211a7e94c3a437ef97133  out-S0_20090213.06
e819a4e0f0b0431f1dcb2e4f3cdb6104  out-S0_20090213.07
8f2d456a61d86355f16fdd7b5f672f48  out-S0_20090213.08
e06441b60b7ed9ac0945750f4558e7c3  out-S0_20090213.09
a8cb7a4c57b412e8957eeea74f1043ce  out-S0_20090213.10
672527a4d93595118b68c82c123a73af  out-S0_20090213.11
1bac206c03480d842678cc11b818d9c0  out-S0_20090213.12
1f5563f6d488be8be2a13e5c02ac97ad  out-S0_20090213.13
35e430e74111f0802f7e4645c0cde1ca  out-S0_20090213.14
89b030b7587892687449994c12995292  out-S0_20090213.15
79b311548f1dce480304a3840b3efcd6  out-S0_20090213.16
24cfeca4702c1a03e235a27846b49a34  out-S0_20090213.17
934f3e22e0cdef633cab56cefb9c0a92  out-S0_20090213.18
afad9b1848c9997057c766e301a1e940  out-S0_20090213.19
3c0f04212083e484682bcdc03eb3f7da  out-S0_20090213.20
369a0da5c850abc6b2688ec9f48e4854  out-S0_20090213.21
9ce705b31805c77a682069cc9efdcbeb  out-S0_20090213.22
26f06311e1af4b5f95d19a4456cb2966  out-S0_20090213.23
15420a7963b88426b9da61238565f117  out-S0_20090214.00
5945f5b27ca8003312d04ac334a5c2f3  out-S0_20090214.01
1a27a6dc6eeaa90c9eaf14ba695b230b  out-S0_20090214.02
c15d57f2a4dcfe8f56ac760a9b347497  out-S0_20090214.03
0a6bb2661c007186ab7616d72826ed48  out-S0_20090214.04
811ed250c0573327d7bd75c8365edd5d  out-S0_20090214.05
e57e86e65c64a7cf44d3c0a2954183ce  out-S0_20090214.06
ad8d9125c16e8e78c2cb0201dee6887d  out-S0_20090214.07
97708605413f1cf4c89f8622dfa12526  out-S0_20090214.08
0879486f3422b9f8887e9fce75cf0b98  out-S0_20090214.09
fe73547fed98a2f3e4c72f5e1087017c  out-S0_20090214.10
491a7144e3f8e2865818a366c586d223  out-S0_20090214.11
2e878514029e426070d433672f453020  out-S0_20090214.12
3896fa44d92e85ac0a9b0f297cb4b8be  out-S0_20090214.13
6d891832751a24c9b37c5123a305e30e  out-S0_20090214.14
9a53a8f3ab08894ccc22550f80fe67d1  out-S0_20090214.15
2cfeb037e058bda17bb1637bf1a74f6f  out-S0_20090214.16
1b425b256051d062c3e30e716c7d3864  out-S0_20090214.17
2674b1ceb74c5c366982d1d3e8a3bb97  out-S0_20090214.18
b17ea9a5aed6ab2e1095ca2a7c5445f9  out-S0_20090214.19
3fdcc032839ca965f48525baac81f077  out-S0_20090214.20
1faa00382f5043e2bef637dc0deb23cd  out-S0_20090214.21
b75bff7f3e412a6a78ffc39f886fd38f  out-S0_20090214.22
9d8c655d82aa3da1c9377b1bb6ca34b5  out-S0_20090214.23
2ea22c19be876e950f5a61ee823c403d  outnull-S0_20090212.00
bb005ac047689d5f68924a2c6a54f6a0  outnull-S0_20090212.01
4617ba84582a4ad8bc05e48a823d162f  outnull-S0_20090212.02
102279d95028d5dcaec58954bb8f33d9  outnull-S0_20090212.03
dd3a4caae92054bc8844b8cc694fd654  outnull-S0_20090212.04
d8398b0bd9dc3373b4ec1ca03893ed19  outnull-S0_20090212.05
ff9c126bf21643a87c37f12ce58c10d2  outnull-S0_20090212.06
df7520cfe69a4d747d48c3b9419b2729  outnull-S0_20090212.07
22613eeaf845149dc52543fc9103dd25  outnull-S0_20090212.08
9af8e44d2e9a00f6057b11322924007c  outnull-S0_20090212.09
8f2c4b7d5dddf4ad2014d1e773b6b091  outnull-S0_20090212.10
523deca8a8a1cde669b3a6929fc832cc  outnull-S0_20090212.11
21eecd617e3830ed1152b376c2435cf1  outnull-S0_20090212.12
1c1aeed181d06205d345e693b31c2cb2  outnull-S0_20090212.13
801d831187d6e6a7ebefddb43f387b38  outnull-S0_20090212.14
ef68e9922e87e94a767a9689c0124b57  outnull-S0_20090212.15
23875b4d4be58abfbf996ae64faf350e  outnull-S0_20090212.16
3c805fafb925f41029d30f6a079d3f37  outnull-S0_20090212.17
468dc0bbd45decb9539602a08fc20952  outnull-S0_20090212.18
dfcadba51d1cdd276295a5581ffeecd4  outnull-S0_20090212.19
6933207acedca446e193203f78a7b3ef  outnull-S0_20090212.20
935bc8dd4b72ed0d4befb930bd06e7c5  outnull-S0_20090212.21
f4e27eeed385f669dce3cae3cfcda407  outnull-S0_20090212.22
efac0a0032f2a3c9b526da0bc8080fe4  outnull-S0_20090212.23
fe361c92d01d129931ca2d6b4f33a282  outnull-S0_20090213.00
d6f20d121e77b515ec10e37489fb5db8  outnull-S0_20090213.01
bdd663f77afa0046b7f0a0d60d4d11de  outnull-S0_20090213.02
462432301d5dee82754ab8749e05e8e3  outnull-S0_20090213.03
35a99bf850949aae6a56c1574e23f054  outnull-S0_20090213.04
c9268691b141e9a9c8eb6697c5eb736c  outnull-S0_20090213.05
f20473cdee28802f2cc2bda2b4f83645  outnull-S0_20090213.06
8da0ec6a152bdd59b0b83c5e0cd16e3f  outnull-S0_20090213.07
8d4b6be7aa2504d7c718aec6252504cc  outnull-S0_20090213.08
b7c5a34d109fbe44da6856adb903d4bc  outnull-S0_20090213.09
4beb3d1981db4c1a998c497d497f5af7  outnull-S0_20090213.10
09e6dc7f051666f00dc81b1da45ebf2b  outnull-S0_20090213.11
e52db972aa22e3915d8c484c353a5cd9  outnull-S0_20090213.12
ffd26c03f1d33fd5e01c586c70c46120  outnull-S0_20090213.13
364634eea374cf86337b7106a7df6149  outnull-S0_20090213.14
f0ae712c5f66e709162d571863ff07be  outnull-S0_20090213.15
e8ad36da357980c930f338be8daf8695  outnull-S0_20090213.16
fb517efb1bdc1c3e2632efc1684d165f  outnull-S0_20090213.17
b2fde7ef522bc3c25d74bf0f78be1719  outnull-S0_20090213.18
ac83e2b688bdd1139f1f4a0f73b220e5  outnull-S0_20090213.19
892ca56c0cf018dd96bacade9a674272  outnull-S0_20090213.20
58a2116f24cdb41cae27f9596faf1837  outnull-S0_20090213.21
da548ef180c3e01afafe799ea68365e6  outnull-S0_20090213.22
5442788189f77f83d1acda18208968b7  outnull-S0_20090213.23
81a941d523addb846ee56f10c84907b9  outnull-S0_20090214.00
bd28d9aa663397cbaf7d044463a9c40f  outnull-S0_20090214.01
826b7cd491853e7b94df084593cf577e  outnull-S0_20090214.02
3feccd56321e130968c01e76e3ed30c7  outnull-S0_20090214.03
7c69f824b60b048371afed6425eeda2a  outnull-S0_20090214.04
bbd7bf589a3aec34c4a3b3f02cbc2b30  outnull-S0_20090214.05
c025d2e2bfc3febd051385d09220734e  outnull-S0_20090214.06
8394ee2c873ad5a592cff1fe9a603769  outnull-S0_20090214.07
0212c0038e71aa8cc2010e64c30399d9  outnull-S0_20090214.08
b978ed9cb42001790b44779090406477  outnull-S0_20090214.09
15e688850033edc932aad94c4687299f  outnull-S0_20090214.10
b13b0ad04a54f24ca08dff61674ae584  outnull-S0_20090214.11
bd263aac5cbab4560e7fe227a05f4d07  outnull-S0_20090214.12
7cc28291154fc2f1f339842bf1c453c5  outnull-S0_20090214.13
c588f4eab1d9e3deb71eed27dab94b10  outnull-S0_20090214.14
48fe52fd5b0efbf8102e34338d80a394  outnull-S0_20090214.15
30d7c746e3d59eb69ca2cbd3ddbc7b58  outnull-S0_20090214.16
4966fed41615b98f35a1701d7baf4366  outnull-S0_20090214.17
51ed4112e901faf45ad5758d46e95497  outnull-S0_20090214.18
0e6d0e5c838ff3edd4312dc030463eef  outnull-S0_20090214.19
db982854b7d3818d4f2b9f83f9211911  outnull-S0_20090214.20
b0aba88f1bf354811bd4708329e0f694  outnull-S0_20090214.21
8d49f3aee39d067103cbdd001f3f3c19  outnull-S0_20090214.22
1bb4f5b11b6c19e13f2f75b494afda78  outnull-S0_20090214.23
50aa265b9bb1629e216de8eb1522f7ee  ow-S0_20090212.00
7628dd33a05ce7bf405522464147c485  ow-S0_20090212.01
51d120385cb77c3cd5b05f539c458b6c  ow-S0_20090212.02
76f974c18e8dec67cba44e3ff10b473e  ow-S0_20090212.03
bd1c0dce00c1d5a7f6deedf60ce32ce5  ow-S0_20090212.04
133e214308caf84e80c54cf1f531bca4  ow-S0_20090212.05
573be1db4e4df0a05c54ca3df68367ce  ow-S0_20090212.06
95ab095dffc910f730c56f6f1daaca6f  ow-S0_20090212.07
ca318fa33c8fe75a657ba7fa2bada3cf  ow-S0_20090212.08
fb4cffb32e415eeaa04323dcb7a0dfd7  ow-S0_20090212.09
3e8e690259be3720dcb191e0902660ab  ow-S0_20090212.10
97c98e28160bb453cab5b69303909576  ow-S0_20090212.11
990989c8fa52d144dd6af18330c65e6f  ow-S0_20090212.12
079ab19065086d7e1b3d5629dd3d9c8d  ow-S0_20090212.13
49e516ea3c446d83dd9635f6a783ec5b  ow-S0_20090212.14
90e1d1cb94b32c480adda40c6e7257e0  ow-S0_20090212.15
f011606161edc815eb29b1fc75c943a3  ow-S0_20090212.16
c4c696e275c33b3e1c1ce8c5a87d6697  ow-S0_20090212.17
e406c4d74ffcd60ae7991aabe8cdade8  ow-S0_20090212.18
1ba344d7b5d01137c5d28c62c3e9209e  ow-S0_20090212.19
98dbe31f8d662323fb77ed51bf5ffd18  ow-S0_20090212.20
4331f0266ef7ee288c3938077c937133  ow-S0_20090212.21
037b6544d00b20fabb42e9171f763d73  ow-S0_20090212.22
b734e28863d1d49b9f2e64929be33d50  ow-S0_20090212.23
52a00fd4210a5f7b0080964a9191efda  ow-S0_20090213.00
a62c6307d33ccd6d8bb8bf1529ff2b79  ow-S0_20090213.01
d21e8765a8542ae1abe7c3c85647ea99  ow-S0_20090213.02
8de5749add1b59da0a11da1292843beb  ow-S0_20090213.03
8c80a916df2070cbc40aa3ad66ba3ad1  ow-S0_20090213.04
12599f2d00544e1684e6406c1a5f61b8  ow-S0_20090213.05
999ec9e8a17669797b383722ecea40b0  ow-S0_20090213.06
37535b27fae0a393d003c1646674437e  ow-S0_20090213.07
9a8613c25f2245f3dab6f6fdd35f86d8  ow-S0_20090213.08
af7481a6af4eb0449d3bf89181c905f5  ow-S0_20090213.09
ebda599893f422ef54fe632715d5f9e6  ow-S0_20090213.10
770d766b155e868442951b2276694ad8  ow-S0_20090213.11
10a120507c40558d870f7f323a9f4db0  ow-S0_20090213.12
846245975f056241836abc1f5aca6b1b  ow-S0_20090213.13
63a10eaee4f3fa1e1d231f801c1cd4b6  ow-S0_20090213.14
0752260753c8f29270479cd016c82de9  ow-S0_20090213.15
2b7775adfd40768d38ed9409c8220619  ow-S0_20090213.16
fb499b5d224aa3aa24bca626ef5d5da6  ow-S0_20090213.17
0d9b7e9022496563bca9246e6c545b03  ow-S0_20090213.18
017094f39b493d5d17ab5c0d3105b9a7  ow-S0_20090213.19
abe73c55632f1c558b54823ef9ba55a8  ow-S0_20090213.20
db063b34cf6d10c8df44e6c35e57e6f4  ow-S0_20090213.21
61f7f2f9aec7fdea9d902f84c8537e88  ow-S0_20090213.22
cfc3ba970a73dd2be4e306f5a5061d0b  ow-S0_20090213.23
6d8a4d95feb13004ebfdb16ce5ba0a1c  ow-S0_20090214.00
a804263db1b32fbfe7239fbbf25f53a0  ow-S0_20090214.01
77b8f61df8df6811f8c933268a7e6a42  ow-S0_20090214.02
ac0144ec621272bf763df838b5dec694  ow-S0_20090214.03
e91dc8e3a385bd8305f9db4c03bd1b76  ow-S0_20090214.04
4e1d1499c3b0a9ee3ac963c714729803  ow-S0_20090214.05
b102f7263e880977a35871d26373fe6c  ow-S0_20090214.06
4dab38a190b0985cba206d0eb466778e  ow-S0_20090214.07
586b762b35a430f2ae89de13a24bebba  ow-S0_20090214.08
70705bc974300f27a181f5f352dd1e7b  ow-S0_20090214.09
6d6dcd6acb650adc9a3ecf1aba9380ff  ow-S0_20090214.10
049fd4988c246eb5d6e511cd636a2c29  ow-S0_20090214.11
b7347b7862b04dae4c2f5a2e3d2c1e71  ow-S0_20090214.12
a460c36d7bae847d12f3484ab26accd3  ow-S0_20090214.13
e8f6cec696dba59c34d21cc469ae228b  ow-S0_20090214.14
f65ad4d31428b7d4721c34691edfa044  ow-S0_20090214.15
7ad015d6be6dd5a51f55179115d11c43  ow-S0_20090214.16
5ea6a80096948c53a498e4188c1112c3  ow-S0_20090214.17
265aa4ef0518a3854b9b439a00702bfd  ow-S0_20090214.18
49023c9160ca9b7bfeed4ef30dd8fa8d  ow-S0_20090214.19
387256f21ec78974e58fa9fc7f58c223  ow-S0_20090214.20
dd35f1e4643e1c99de74ca5d8e08055a  ow-S0_20090214.21
8b579e91f8e311e301c5169b009a6130  ow-S0_20090214.22
2e4aa6f8e14b558f968d7efcd18d0370  ow-S0_20090214.23
