/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _RWFLOWPACK_PRIV_H
#define _RWFLOWPACK_PRIV_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_RWFLOWPACK_PRIV_H, "$SiLK: rwflowpack_priv.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

/*
**  rwflowpack_priv.h
**
**    Private definitions and prototypes for rwflowpack.
*/

#include <silk/utils.h>
#include <silk/libflowsource.h>
#include <silk/probeconf.h>
#include <silk/sksite.h>
#include <silk/sklog.h>
#include <silk/rwflowpack.h>
#include "rwflow_utils.h"


/* Message to print when out of memory */
#define FMT_MEM_FAILURE  "Out of memory at %s:%d", __FILE__, __LINE__

/*
 *  These values indicate whether the flow reader operates as a
 *  daemon.  When acting as a daemon, rwflowpack will fork().
 */
typedef enum {
    FP_DAEMON_OFF, FP_DAEMON_ON
} fp_daemon_mode_t;


/*
 * These values give information back to the caller of the
 * get_record_fn as to whether the function worked, and whether it is
 * safe to stop pulling data.
 */
typedef enum {
    FP_FATAL_ERROR = -2,    /* A critical error occured */
    FP_GET_ERROR = -1,      /* An error occured */
    FP_BREAK_POINT = 0,     /* You may stop before processing the record */
    FP_FILE_BREAK,          /* You may stop before processing the record */
    FP_RECORD,              /* You may NOT stop after processing record */
    FP_END_STREAM           /* You must stop.  There is no record. */
} fp_get_record_result_t;


/*
 *  A structure to pass options between the rwflowpack and the
 *  reader_types.
 */
typedef union reader_options_un {
    struct pdu_file_st {
        const char *netflow_file;
    } pdu_file;
#if 0
    struct silk_file_st {
        const char *pathname;
    } silk_file;
#endif
    struct fcfiles_st {
        /* The directory flowcap files mode will poll for new flowcap
         * files to process */
        const char *incoming_directory;

        /* Polling interval (in seconds) for the incoming directory */
        uint32_t polling_interval;
    } fcfiles;

    struct respool_st {
        /* The directory respool mode will poll for new silk files to
         * process */
        const char *incoming_directory;

        /* Polling interval (in seconds) for the incoming directory */
        uint32_t polling_interval;
    } respool;

    struct stream_polldir_st {
        /* Polling interval (in seconds) for PDU/SiLK files. */
        uint32_t polling_interval;
    } stream_polldir;
} reader_options_t;


/*
 *  This is the structure that the main rwflowpack uses to communicate
 *  with the flow readers.  rwflowpack will have an array of these,
 *  with only the initialize_fn filled in---when we move to using
 *  dlopen(), this may be handled by the shared object's _init()
 *  function.  This struct gets passed into the initialize_fn which
 *  will fill in other values.  Since the struct is self-referencing,
 *  we must pre-declare it.
 */
typedef struct reader_type_st reader_type_t;

typedef struct flow_proc_st flow_proc_t;

/*
 *  Flow-reader initialization function type
 */
typedef int (*fr_initialize_fn_t)(reader_type_t *reader_type);


struct reader_type_st {
    /* Name of this reader, for printing in --help output */
    const char *reader_name;

    /* Probes assigned to this reader_type */
    sk_vector_t *probes;

    /* Function for flow reader to tear itself down */
    void      (*teardown_fn)(void);

    /* Function to say whether this reader_type can read data from the
     * specified probe.  0=No, 1=Yes.
     */
    int       (*want_probe_fn)(skpc_probe_t *probe);

    /* Function to run to complete reader setup after the user has
     * chosen this reader.  The function set any other state to
     * prepare for the main processing loop in rwflowpack to call this
     * reader's 'get_record_fn'.  'reader_type' is this structure.
     * 'termination_flag' is a pointer to the variable that that turns
     * false when it is time to terminate the program.  This
     * 'setup_fn' should set the 'is_daemon' flag depending on whether
     * the main processing loop should daemon-ize itself.
     */
    int       (*setup_fn)(fp_daemon_mode_t     *is_daemon,
                          const sk_vector_t    *probe_vec,
                          reader_options_t     *options);

    /* Function to start any necessary background record collection.
     * Returns 0 on success, -1 on failure. */
    int       (*start_fn)(flow_proc_t *fproc);

    /* Function to stop the previous background record collection.
     * Must also be sure to unblock a currently blocking call to
     * get_record_fn, if blocked. */
    int       (*stop_fn)(flow_proc_t *fproc);

    /* Retrieves the next record and the probe where the flow was
     * collected.  Return value indicates what actions are safe after
     * retrieving the value.  (See the enum definition for
     * fp_get_record_result_t above) */
    fp_get_record_result_t (*get_record_fn)(rwRec               *out_rec,
                                            const skpc_probe_t **out_probe,
                                            flow_proc_t         *fproc);

    /* Logs information about the current reader */
    void      (*print_stats_fn)(flow_proc_t *fproc);
};


struct flow_proc_st {
    /* Total number of records processed */
    uint64_t            rec_count_total;

    /* Number of bad records processed */
    uint64_t            rec_count_bad;

    /* The class of this processor */
    reader_type_t      *reader_type;

    /* The probe and flow_src are where the processor gets its data.
     * The flow_src is created based on settings in the probe. */
    const skpc_probe_t *probe;
    void               *flow_src;

    /* A flow processor is associated with a single thread. */
    pthread_t           thread;
};


/* Functions in rwflowpack.c to limit the number of incoming files the
 * directory polling threads can have open */
int flowpackAcquireFileHandle(void);
void flowpackReleaseFileHandle(void);


/*
 * The initialization functions for each flow reader.  Probably should
 * be in a header file for each reader, but seems overkill to have a
 * header for a single function declaration.
 */
extern int  pduReaderInitialize(reader_type_t *reader_type);
extern int  pduFileReaderInitialize(reader_type_t *reader_type);
extern int  dirReaderInitialize(reader_type_t *reader_type);
/* extern int  silkFileReaderInitialize(reader_type_t *reader_type); */
extern int  fcFilesReaderInitialize(reader_type_t *reader_type);
extern int  respoolReaderInitialize(reader_type_t *reader_type);
#if SK_ENABLE_IPFIX
extern int  ipfixReaderInitialize(reader_type_t *reader_type);
#endif

/* Declare function to initialize rwflowpack for respooling */
int packLogicRespoolInitialize(packlogic_plugin_t *);

#endif /* _RWFLOWPACK_PRIV_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
