/*
** Copyright (C) 2006-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Daemon that appends rw files to hourly files.
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwflowappend.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skpolldir.h>
#include <silk/sksite.h>
#include <silk/sklog.h>
#include <silk/skdaemon.h>
#include "rwflow_utils.h"


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

/* Default number of seconds between which to poll for new files */
#define DEFAULT_POLLING_INTERVAL 15


/* LOCAL VARIABLES */

/* how often to poll the directory for new incremental files */
static uint32_t polling_interval = DEFAULT_POLLING_INTERVAL;

/* directory to watch for new incremental files */
static const char *incoming_dir = NULL;

/* directory in which to write hourly files */
static const char *root_dir = NULL;

/* command, supplied by user, to run whenever a new hourly file is
 * created */
static const char *hour_file_command = NULL;

/* oldest file (in hours) that is considered acceptable.  incremental
 * files older than this will be put into the error directory. */
static int64_t reject_hours_past = INT64_MAX;

/* how far into the future are incremental files accepted. */
static int64_t reject_hours_future = INT64_MAX;

/* whether reject_hours_past and/or reject_hours_future differ from
 * their default values--meaning we need to run the tests */
static int check_time_window = 0;

/* byte order of the files we generate; default is to use the
 * order of the files we receive */
static silk_endian_t byte_order = SILK_ENDIAN_ANY;

/* compression method of the files we generate; default is to use
 * the method of the files we receive */
static sk_compmethod_t comp_method;

/* whether we are in shut-down mode */
static volatile int shuttingdown = 0;

/* objects that handle polling of the 'incoming_dir' */
static skPollDir_t *polldir = NULL;

/* the thread that handles incremental files */
static pthread_t filehandler;

/* whether that thread has been started */
static int filehandler_started = 0;



/* OPTIONS SETUP */

typedef enum {
    OPT_INCOMING_DIRECTORY, OPT_ROOT_DIRECTORY, OPT_ERROR_DIRECTORY,
    OPT_ARCHIVE_DIRECTORY, OPT_FLAT_ARCHIVE,
    OPT_POST_COMMAND, OPT_HOUR_FILE_COMMAND,
    OPT_REJECT_HOURS_PAST, OPT_REJECT_HOURS_FUTURE,
    OPT_POLLING_INTERVAL,
    OPT_BYTE_ORDER, OPT_PAD_HEADER
} appOptionsEnum;

static struct option appOptions[] = {
    {"incoming-directory",      REQUIRED_ARG, 0, OPT_INCOMING_DIRECTORY},
    {"root-directory",          REQUIRED_ARG, 0, OPT_ROOT_DIRECTORY},
    {"error-directory",         REQUIRED_ARG, 0, OPT_ERROR_DIRECTORY},
    {"archive-directory",       REQUIRED_ARG, 0, OPT_ARCHIVE_DIRECTORY},
    {"flat-archive",            NO_ARG,       0, OPT_FLAT_ARCHIVE},
    {"post-command",            REQUIRED_ARG, 0, OPT_POST_COMMAND},
    {"hour-file-command",       REQUIRED_ARG, 0, OPT_HOUR_FILE_COMMAND},
    {"reject-hours-past",       REQUIRED_ARG, 0, OPT_REJECT_HOURS_PAST},
    {"reject-hours-future",     REQUIRED_ARG, 0, OPT_REJECT_HOURS_FUTURE},
    {"polling-interval",        REQUIRED_ARG, 0, OPT_POLLING_INTERVAL},
    {"byte-order",              REQUIRED_ARG, 0, OPT_BYTE_ORDER},
    {"pad-header",              NO_ARG,       0, OPT_PAD_HEADER},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    ("Watch this directory for new incremental files to\n"
     "\tappend to hourly files"),
    ("Append to/Create hourly files in this directory tree"),
    ("Store in this directory incremental files that were\n"
     "\tNOT successfully appended to an hourly file"),
    ("Archive into this directory tree incremental files\n"
     "\tthat were successfully appended to an hourly file.  If not given,\n"
     "\tincremental files are deleted after appending. Def. No archive"),
    ("Store incremental files in the root of the archive\n"
     "\tdirectory.  When not given, incremental files are stored in\n"
     "\tsubdirectories of the archive-directory. Def. Use subdirectories"),
    ("Run this command on each incremental file after\n"
     "\tsuccessfully appending it and moving it to the archive-directory.\n"
     "\tDef. None.  Each \"%s\" in the command will be substituted by the\n"
     "\tarchived file's path.  Requires use of --archive-directory."),
    ("Run this command on new hourly files upon their\n"
     "\tcreation.  Def. None.  \"%s\" in the command is substituted by the\n"
     "\tfull path to the hourly file."),
    ("Reject incremental files containing records whose\n"
     "\tstart times occur more than this number of hours in the past.  The\n"
     "\tfiles are moved into the error directory.  Def. Accept all files"),
    ("Reject incremental files containing records whose\n"
     "\tstart times occur more than this number of hours in the future.  The\n"
     "\tfiles are moved into the error directory.  Def. Accept all files"),
    ("Check the incoming-directory this often for new\n"
     "\tincremental files (in seconds)"),
    ("Create new hourly files in this byte order. Def. 'as-is'.\n"
     "\tChoices: 'as-is'=same as incremental file, 'native', 'little', 'big'"),
    ("Ignored.  For backward compatibility only"),
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  byteOrderParse(const char *endian_string);


/* FUNCTION DEFINITONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                          \
    ("<SWITCHES>\n"                                                        \
     "\tWatches a directory for files containing small numbers of SiLK\n"  \
     "\tflow records (incremental files) and appends those records to\n"   \
     "\thourly files stored in a directory tree creating subdirectories\n" \
     "\tand new hourly files as required.\n")

    FILE *fh = USAGE_FH;
    int i;

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);
    fprintf(fh, "\nSWITCHES:\n");
    skOptionsDefaultUsage(fh);

    for (i = 0; appOptions[i].name; ++i) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch ((appOptionsEnum)appOptions[i].val) {
          case OPT_POLLING_INTERVAL:
            fprintf(fh, "%s. Def. %d", appHelp[i], DEFAULT_POLLING_INTERVAL);
            break;
          default:
            fprintf(fh, "%s", appHelp[i]);
            break;
        }
        fprintf(fh, "\n");
    }
    sksiteCompmethodOptionsUsage(fh);
    sksiteOptionsUsage(fh);

    fprintf(fh, "\nLogging and daemon switches:\n");
    skdaemonOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;
    shuttingdown = 1;

    INFOMSG("Shutting down...");

    if (polldir) {
        skPollDirStop(polldir);
    }

    if (filehandler_started) {
        pthread_join(filehandler, NULL);
    }

    if (polldir) {
        skPollDirDestroy(polldir);
    }

    INFOMSG("Finished shutting down.");

    skdaemonTeardown();
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    int error_count = 0;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* Add the --compression-method switch.  This call will cause us
     * to use the compression method set at compile time if the user
     * doesn't provide the switch.  Since we want to default to using
     * the compression on the incremental files, reset the comp_method
     * variable to "invalid". */
    if (sksiteCompmethodOptionsRegister(&(comp_method))) {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }
    comp_method = SK_INVALID_COMPMETHOD;

    /* rwflowappend runs as a daemon; use the threaded logger */
    if (skdaemonSetup((SKLOG_FEATURE_LEGACY | SKLOG_FEATURE_SYSLOG),
                      argc, argv)
        || sklogEnableThreadedLogging())
    {
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* Check directories */
    if (NULL == incoming_dir) {
        skAppPrintErr("The --%s switch is required",
                      appOptions[OPT_INCOMING_DIRECTORY].name);
        ++error_count;
    }
    if (NULL == root_dir) {
        skAppPrintErr("The --%s switch is required",
                      appOptions[OPT_ROOT_DIRECTORY].name);
        ++error_count;
    }
    if (!errorDirectoryIsSet()) {
        skAppPrintErr("The --%s switch is required",
                      appOptions[OPT_ERROR_DIRECTORY].name);
        ++error_count;
    }

    /* When post-command is given, verify that archive_dir is set. */
    if (archiveDirectoryIsSet() == -1) {
        skAppPrintErr("The --%s switch is required when using --%s",
                      appOptions[OPT_ARCHIVE_DIRECTORY].name,
                      appOptions[OPT_POST_COMMAND].name);
        ++error_count;
    }

    /* verify the required options for logging */
    if (skdaemonOptionsVerify()) {
        ++error_count;
    }

    /* check for extraneous arguments */
    if (arg_index != argc) {
        skAppPrintErr("Too many arguments or unrecognized switch '%s'",
                      argv[arg_index]);
        ++error_count;
    }

    /* set root directory */
    if (sksiteSetRootDir(root_dir)) {
        exit(EXIT_FAILURE);
    }

    /* Ensure the site config file is available */
    if (sksiteConfigure(1)) {
        ++error_count;
    }

    if (error_count) {
        skAppUsage();             /* never returns */
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;  /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    uint32_t tmp32;
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_INCOMING_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        incoming_dir = opt_arg;
        break;

      case OPT_ROOT_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        root_dir = opt_arg;
        break;

      case OPT_ARCHIVE_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        archiveDirectorySetPath(opt_arg);
        break;

      case OPT_FLAT_ARCHIVE:
        archiveDirectorySetFlat();
        break;

      case OPT_ERROR_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        errorDirectorySetPath(opt_arg);
        break;

      case OPT_REJECT_HOURS_PAST:
        rv = skStringParseUint32(&tmp32, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        reject_hours_past = (int64_t)tmp32;
        check_time_window = 1;
        break;

      case OPT_REJECT_HOURS_FUTURE:
        rv = skStringParseUint32(&tmp32, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        reject_hours_future = (int64_t)tmp32;
        check_time_window = 1;
        break;

      case OPT_POLLING_INTERVAL:
        rv = skStringParseUint32(&polling_interval, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_PAD_HEADER:
        break;

      case OPT_BYTE_ORDER:
        if (byteOrderParse(opt_arg)) {
            return 1;
        }
        break;

      case OPT_POST_COMMAND:
        archiveDirectorySetPostCommand(opt_arg);
        break;

      case OPT_HOUR_FILE_COMMAND:
        hour_file_command = opt_arg;
        break;
    }

    return 0;  /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 *  ok = byteOrderParse(argument)
 *
 *    parse the argument to the --byte-order switch
 */
static int byteOrderParse(const char *endian_string)
{
    static int option_seen = 0;
    int i;
    size_t len;
    /* Options for byte-order switch */
    struct {
        const char     *name;
        silk_endian_t   value;
    } byte_order_opts[] = {
        {"as-is",  SILK_ENDIAN_ANY},
        {"native", SILK_ENDIAN_NATIVE},
        {"little", SILK_ENDIAN_LITTLE},
        {"big",    SILK_ENDIAN_BIG},
        {NULL,     SILK_ENDIAN_ANY} /* sentinel */
    };

    /* only process option one time */
    if (option_seen != 0) {
        skAppPrintErr("Option %s given multiple times",
                      appOptions[OPT_BYTE_ORDER].name);
        return 1;
    }
    option_seen = 1;

    len = strlen(endian_string);
    if (len == 0) {
        skAppPrintErr("Invalid %s: Empty string",
                      appOptions[OPT_BYTE_ORDER].name);
        return 1;
    }

    /* parse user's input */
    for (i = 0; byte_order_opts[i].name; ++i) {
        if ((len <= strlen(byte_order_opts[i].name))
            && (0 == strncmp(byte_order_opts[i].name, endian_string, len)))
        {
            byte_order = byte_order_opts[i].value;
            option_seen = 2;
            break;
        }
    }

    if (option_seen != 2) {
        skAppPrintErr("Invalid %s '%s': Unrecognized value",
                      appOptions[OPT_BYTE_ORDER].name, endian_string);
        return 1;
    }

    if (byte_order == SILK_ENDIAN_NATIVE) {
#if SK_LITTLE_ENDIAN
        byte_order = SILK_ENDIAN_LITTLE;
#else
        byte_order = SILK_ENDIAN_BIG;
#endif
    }

    return 0;
}


/*
 *  status = openOutputStream(out_ios, out_pos, hourly_file, in_hdr)
 *
 *    Given the SiLK Flow stream connected to an incremental file
 *    whose SiLK header is in 'in_hdr', either open an existing hourly
 *    file or create a new hourly file at the location specified by
 *    'hourly_file' of the same type and version (RWSPLIT, etc) to
 *    hold the data in the incremental file.  The handle to the opened
 *    stream is put into the value pointed at by 'out_ios'.  'out_pos'
 *    is set to 0 if the file is newly created, or to the current size
 *    of the file.
 *
 *    The endianness of the new file is determined by the global
 *    'byte_order' variable.  The compression method of the new file
 *    is determined by the 'comp_method' global if that value is set
 *    to a valid compression method.
 *
 *    Returns 0 on success.  On error, prints a message to the log and
 *    returns -1.  When an error occurs, the value of 'out_pos' is
 *    indeterminate.
 */
static int openOutputStream(
    skstream_t            **out_ios,
    off_t                  *out_pos,
    const char             *hourly_file,
    const sk_file_header_t *in_hdr)
{
    int rv = SKSTREAM_OK;
    silk_endian_t out_endian;
    sk_compmethod_t out_compmethod;
    sk_file_header_t *out_hdr = NULL;

    /* Open or create a new file as necessary, protecting against a
     * zero-length file in the data store.  These zero-length files
     * should be a rare occurrence.  Note that skFileSize() also
     * returns 0 for non-existent files. */
    if (skFileSize(hourly_file)) {
        /* Open existing output file */
        DEBUGMSG("Opening existing output file '%s'", hourly_file);

        if ((rv = skStreamCreate(out_ios, SK_IO_APPEND, SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(*out_ios, hourly_file))
            || (rv = skStreamOpen(*out_ios))
            || (rv = skStreamReadSilkHeader(*out_ios, &out_hdr)))
        {
            goto ERROR;
        }
        *out_pos = skStreamTell(*out_ios);
#if 0
        {
            /* Sanity check that the times in the headers of the two
             * files are the same. */
            union h_un {
                sk_header_entry_t          *he;
                sk_hentry_packedfile_t     *pf;
            } h_in, h_out;
            sktime_t in_time;
            sktime_t out_time;
            char inbuf[SKTIMESTAMP_STRLEN];
            char outbuf[SKTIMESTAMP_STRLEN];

            h_in.he = skHeaderGetFirstMatch(in_hdr, SK_HENTRY_PACKEDFILE_ID);
            h_out.he = skHeaderGetFirstMatch(out_hdr, SK_HENTRY_PACKEDFILE_ID);
            if (h_in.he && h_out.he) {
                in_time = skHentryPackedfileGetStartTime(h_in.pf);
                out_time = skHentryPackedfileGetStartTime(h_out.pf);
                if (in_time != out_time) {
                    NOTICEMSG(("Incremental StartTime (%s) does not match"
                               " Hourly StartTime (%s) in file '%s'"),
                              sktimestamp_r(inbuf, in_time, 0),
                              sktimestamp_r(outbuf, out_time, 0), hourly_file);
                    goto ERROR;
                }
            }
        }
#endif  /* 0 */
    } else {
        *out_pos = 0;

        if (skFileExists(hourly_file)) {
            /* Remove the zero-length file noisily */
            NOTICEMSG("Removing zero length file '%s'", hourly_file);
            if (unlink(hourly_file) == -1) {
                WARNINGMSG("Cannot remove '%s': %s",
                           hourly_file, strerror(errno));
                *out_ios = NULL;
                goto ERROR;
            }
        }

        DEBUGMSG("Opening new output file '%s'", hourly_file);

        /* Determine the byte order for the new file */
        if (byte_order == SILK_ENDIAN_ANY) {
            out_endian = skHeaderGetByteOrder(in_hdr);
        } else {
            assert(byte_order != SILK_ENDIAN_NATIVE);
            out_endian = byte_order;
        }

        if (comp_method == SK_INVALID_COMPMETHOD) {
            /* if user didn't set the value, use that from the file we
             * received */
            out_compmethod = skHeaderGetCompressionMethod(in_hdr);
        } else {
            /* use the value the user specified */
            out_compmethod = comp_method;
        }

        /* Create the output stream object */
        if ((rv = skStreamCreate(out_ios, SK_IO_WRITE, SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(*out_ios, hourly_file))
            || (rv = skStreamMakeDirectory(*out_ios)))
        {
            goto ERROR;
        }

        /* Set the output stream's attributes */
        out_hdr = skStreamGetSilkHeader(*out_ios);
        if ((rv = skHeaderCopy(out_hdr, in_hdr,
                               (SKHDR_CP_ALL
                                & ~(SKHDR_CP_ENDIAN | SKHDR_CP_COMPMETHOD))))
            || (rv = skHeaderSetCompressionMethod(out_hdr, out_compmethod))
            || (rv = skHeaderSetByteOrder(out_hdr, out_endian)))
        {
            goto ERROR;
        }

        /* Open the file and write its header to avoid zero-length
         * files in the data store. */
        if ((rv = skStreamOpen(*out_ios))
            || (rv = skStreamWriteSilkHeader(*out_ios)))
        {
            /* error opening the file or writing to it.  remove it. */
            unlink(hourly_file);
            goto ERROR;
        }
    }

    /* Success! */
    return 0;

  ERROR:
    if (rv) {
        skStreamPrintLastErr(*out_ios, rv, &WARNINGMSG);
    }
    skStreamDestroy(out_ios);
    return -1;
}


/*
 *  status = appendIncremental(in_path, out_path, &out_pos);
 *
 *    Given the names of an incremental source file, 'in_path', and an
 *    hourly destination file 'out_path', append the data in the
 *    incremental file to the destination file, creating a new
 *    destination file if it does not exist.
 *
 *    The function will set 'out_pos' to the position in the hourly
 *    file where thew new incremental data begins (i.e., the size of
 *    the file before the append).  This value is 0 for a newly
 *    created output file.  It will also be zero if an error occurs
 *    either opening the input file or opening/creating the output
 *    file.
 *
 *    If all goes well, return 0.  If an error occurs, write an error
 *    to the log buffer and return non-zero, where a negative return
 *    value indicates a write error, and a positive value indicates a
 *    read error.  Specifically, a return value of -1 indicates either
 *    an error opening an output file or an error writing records to
 *    the file.  A return value of 1 indicates there was a problem
 *    opening 'in_path' or reading the first record from it.  A return
 *    value of 2 indicates some records were written to the output
 *    (appended at 'out_pos'), and then a read error occurred.
 */
static int appendIncremental(
    const char     *in_path,
    const char     *out_path,
    off_t          *out_pos)
{
    uint32_t long_duration = 0;
    skstream_t *in_ios = NULL;
    skstream_t *out_ios = NULL;
    sk_file_header_t *in_hdr;
    int ret_val = 0;
    int rv;
    rwRec rwrec;

    *out_pos = 0;

    /* Open input file */
    if ((rv = skStreamCreate(&in_ios, SK_IO_READ, SK_CONTENT_SILK_FLOW))
        || (rv = skStreamBind(in_ios, in_path))
        || (rv = skStreamOpen(in_ios))
        || (rv = skStreamReadSilkHeader(in_ios, &in_hdr)))
    {
        skStreamPrintLastErr(in_ios, rv, &WARNINGMSG);
        ret_val = 1;
        goto END;
    }

    /* read the first record from the input */
    rv = skStreamReadRecord(in_ios, &rwrec);
    switch (rv) {
      case SKSTREAM_OK:
        break;

      case SKSTREAM_ERR_EOF:
        INFOMSG("Incremental file '%s' has no records", in_path);
        goto END;

      default:
        NOTICEMSG("Could not read first record from '%s'", in_path);
        skStreamPrintLastErr(in_ios, rv, &WARNINGMSG);
        ret_val = 1;
        goto END;
    }

    /* check for incremental files outside of the time window */
    if (check_time_window) {
        int64_t diff;
        time_t t = time(NULL);

        diff = ((int64_t)t / 3600) - (rwRecGetStartSeconds(&rwrec) / 3600);
        if (diff > reject_hours_past) {
            INFOMSG("Incremental file '%s' is %" PRId64 " hours old",
                    in_path, diff);
            ret_val = 1;
            goto END;
        }
        if (-diff > reject_hours_future) {
            INFOMSG(("Incremental file '%s' occurs %" PRId64
                     " hours in the future"),
                    in_path, -diff);
            ret_val = 1;
            goto END;
        }
    }

    /* Consider using the stream cache here to avoid open/close */

    /* Open output */
    rv = openOutputStream(&out_ios, out_pos, out_path, in_hdr);
    if (rv) {
        /* no records were written; make certain *out_pos is 0. */
        *out_pos = 0;
        ret_val = -1;
        goto END;
    }

    do {
        rv = skStreamWriteRecord(out_ios, &rwrec);
        if (rv != SKSTREAM_OK) {
            if (SKSTREAM_ERROR_IS_FATAL(rv)) {
                skStreamPrintLastErr(out_ios, rv, &WARNINGMSG);
                ret_val = -1;
                goto END;
            }
            if ((rv == SKSTREAM_ERR_ELPSD_OVRFLO)
                && (rwRecGetElapsed(&rwrec) > (1000 * 0x03FF)))
            {
                /* try to handle case of an overflow in the elapsed
                 * time---our formats changed from 11 bits to 12 bits,
                 * but we may be trying to write new 12b into old 11b.
                 */
                ++long_duration;
                rwRecSetElapsed(&rwrec, 1000 * 0x03FF);
                rv = skStreamWriteRecord(out_ios, &rwrec);
            }
            if (rv != SKSTREAM_OK) {
                /* either error wasn't elapsed overflow, or the above
                 * fix didn't help */
                skStreamPrintLastErr(out_ios, rv, &WARNINGMSG);
            }
        }
    } while (SKSTREAM_OK == (rv = skStreamReadRecord(in_ios, &rwrec)));

    /* log about truncating the elapsed time field */
    if (long_duration > 0) {
        INFOMSG("Truncated elapsed time for %" PRIu32 " records in file %s",
                long_duration, skStreamGetPathname(in_ios));
    }

    /* warn if we didn't get to the end of the input file */
    if (SKSTREAM_ERR_EOF != rv) {
        skStreamPrintLastErr(in_ios, rv, &NOTICEMSG);
        ret_val = 2;
    }

    /* close input */
    rv = skStreamClose(in_ios);
    if (rv) {
        skStreamPrintLastErr(in_ios, rv, &NOTICEMSG);
    }

    /* close the output file */
    rv = skStreamClose(out_ios);
    if (rv) {
        skStreamPrintLastErr(out_ios, rv, &WARNINGMSG);
        if (SKSTREAM_ERROR_IS_FATAL(rv)) {
            ret_val = -1;
            goto END;
        }
    }

  END:
    (void)skStreamDestroy(&out_ios);
    (void)skStreamDestroy(&in_ios);

    return ret_val;
}


/*
 *  THREAD ENTRY POINT
 *
 *  This is the entry point for the filehandler thread.
 *
 *  This function waits for files to appear on the valid file deque
 *  (validQ).  When files appear, they are popped off the queue and
 *  processed.
 */
static void *handleFileThread(void UNUSED(*dummy))
{
    sigset_t sigs;
    char in_path[PATH_MAX];
    char out_path[PATH_MAX];
    char *in_basename;
    char *out_basename;
    char *relative_dir;
    skPollDirErr_t pderr;
    off_t pos;
    int rv;

    sigfillset(&sigs);
    pthread_sigmask(SIG_SETMASK, &sigs, NULL);

    INFOMSG("Starting file handling thread.");

    while (!shuttingdown) {

        /* Get the next incremental file name from the polling
         * directory; this is the full path to the file. */
        pderr = skPollDirGetNextFile(polldir, in_path, &in_basename);
        if (pderr != PDERR_NONE) {
            if (pderr == PDERR_STOPPED) {
                assert(shuttingdown);
                continue;
            }
            ERRMSG("Fatal directory polling error: %s",
                   ((pderr == PDERR_SYSTEM)
                    ? strerror(errno)
                    : skPollDirStrError(pderr)));
            exit(EXIT_FAILURE);
        }

        /* Break the pathname of the incremental file into pieces and
         * determine the pathname of the hourly file to which the
         * incremental file will be appended. */
        if ( !sksiteParseGeneratePath(out_path, sizeof(out_path),
                                      in_basename, "", /* no suffix */
                                      &relative_dir,
                                      &out_basename))
        {
            WARNINGMSG("File '%s' does not match SiLK file naming convention",
                       in_path);
            errorDirectoryInsertFile(in_path);
            continue;
        }

        /* Append incremental file to hourly file. */
        rv = appendIncremental(in_path, out_path, &pos);
        if (0 == rv) {
            /* Success */
            INFOMSG(("APPEND OK %s to %s @ %" PRId64),
                    in_basename, out_path, (int64_t)pos);

        } else if (-1 == rv) {
            /* Output error.  Shutdown. */
            if (pos == 0) {
                /* Error opening output file. */
                ERRMSG("APPEND FAILED '%s' to '%s' -- nothing written",
                       in_basename, out_path);
            } else {
                /* Error writing.  Move partially appended file to
                 * error directory. */
                ERRMSG(("APPEND FAILED '%s' to '%s' @ %" PRId64),
                       in_basename, out_path, (int64_t)pos);

                /* Move to error directory */
                NOTICEMSG("Moving incremental file '%s' to error directory",
                          in_basename);
                errorDirectoryInsertFile(in_path);
            }
            CRITMSG("Aborting due to append error.");
            exit(EXIT_FAILURE);

        } else if (1 == rv) {
            /* Problem with input file.  Move to error directory. */
            WARNINGMSG(("Error initializing incremental file '%s' --"
                        " nothing written.  Moving to error directory."),
                       in_basename);
            errorDirectoryInsertFile(in_path);

            /* output file was not opened; go to next input file */
            continue;

        } else {
            /* Success; though unexpected error on read.  Currently
             * treat this as successful, but should we move to the
             * error_directory instead? */
            NOTICEMSG(("Unexpected error reading incremental file '%s'"
                       " -- treating as successful"),
                      in_basename);
            INFOMSG(("APPEND OK %s to %s @ %" PRId64),
                    in_basename, out_path, (int64_t)pos);
        }

        /* Run command if this is a new hourly file */
        if (pos == 0 && hour_file_command) {
            runCommand(hour_file_command, out_path);
        }

        /* we need to pass the relative-directory to the archive
         * function.  Modify out_path so it terminates just before the
         * basename which is just after the relative directory. */
        *(out_basename - 1) = '\0';

        /* archive or remove the incremental file.  this also invokes
         * the post-command if that was specified. */
        archiveDirectoryInsertOrRemove(in_path, relative_dir);

    } /* while (!shuttingdown) */

    INFOMSG("Exiting file handling thread.");

    return NULL;
}


int main(int argc, char **argv)
{
    appSetup(argc, argv);       /* never returns on error */

    /* start the logger and become a daemon */
    if (skdaemonize(&shuttingdown, NULL) == -1) {
        exit(EXIT_FAILURE);
    }

    /* Set up directory polling */
    polldir = skPollDirCreate(incoming_dir, polling_interval);
    if (NULL == polldir) {
        ERRMSG("Could not initiate polling on %s", incoming_dir);
        exit(EXIT_FAILURE);
    }

    /* Start the appending thread. */
    if (pthread_create(&filehandler, NULL, handleFileThread, NULL)) {
        ERRMSG("Failed to start file handling thread.");
        exit(EXIT_FAILURE);
    }
    filehandler_started = 1;

    while (!shuttingdown) {
        pause();
    }

    /* done */
    appTeardown();
    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
