/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
** rwcutsetup.c
**      utility routines in support of rwcut.
** Suresh Konda
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwcutsetup.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skprefixmap.h>
#include <silk/skcountry.h>
#include "rwcut.h"


/* TYPEDEFS AND MACROS */

/* where to write --help output */
#define USAGE_FH stdout

/* The last field printed by default. */
#define RWCUT_LAST_DEFAULT_FIELD  RWREC_FIELD_SID


/* LOCAL VARIABLES */

/* start and end record number */
static uint64_t start_rec_num = 0;
static uint64_t end_rec_num = 0;
/* num_recs is a global */

/* name of program to run to page output */
static char *pager = NULL;

/* are we using the pager? */
static int using_pager = 0;

/* the text the user entered for the --fields switch */
static char *fields_arg = NULL;

/* whether the --all-fields switch was given */
static int all_fields = 0;

/* available fields */
static sk_stringmap_t *key_field_map;

/* fields that get defined just like plugins */
static const struct app_static_plugins_st {
    char               *name;
    skplugin_setup_fn_t setup_fn;
} app_static_plugins[] = {
    {"addrtype",        skAddressTypesAddFields},
    {"ccfilter",        skCountryAddFields},
    {"pmapfilter",      skPrefixMapAddFields},
    {NULL, NULL}        /* sentinel */
};

/* List of plugins to attempt to open at startup */
static const char *app_plugin_names[] = {
    SK_PLUGIN_ADD_SUFFIX("silkpython"),
    NULL /* sentinel */
};



/* OPTIONS SETUP */

typedef enum {
    /* Keep this list in sync with appOptions[] */
    OPT_FIELDS,
    OPT_ALL_FIELDS,
    OPT_NUM_RECS,
    OPT_START_REC_NUM,
    OPT_END_REC_NUM,
    OPT_DRY_RUN,
    OPT_PLUGIN,
    OPT_ICMP_TYPE_AND_CODE,
    OPT_EPOCH_TIME,
    OPT_INTEGER_IPS,
    OPT_ZERO_PAD_IPS,
    OPT_INTEGER_SENSORS,
    OPT_INTEGER_TCP_FLAGS,
    OPT_NO_TITLES,
    OPT_NO_COLUMNS,
    OPT_COLUMN_SEPARATOR,
    OPT_NO_FINAL_DELIMITER,
    OPT_DELIMITED,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT,
    OPT_OUTPUT_PATH,
    OPT_PAGER,
    OPT_LEGACY_TIMESTAMPS
} appOptionsEnum;

static struct option appOptions[] = {
    {"fields",              REQUIRED_ARG, 0, OPT_FIELDS},
    {"all-fields",          NO_ARG,       0, OPT_ALL_FIELDS},
    {"num-recs",            REQUIRED_ARG, 0, OPT_NUM_RECS},
    {"start-rec-num",       REQUIRED_ARG, 0, OPT_START_REC_NUM},
    {"end-rec-num",         REQUIRED_ARG, 0, OPT_END_REC_NUM},
    {"dry-run",             NO_ARG,       0, OPT_DRY_RUN},
    {"plugin",              REQUIRED_ARG, 0, OPT_PLUGIN},
    {"icmp-type-and-code",  NO_ARG,       0, OPT_ICMP_TYPE_AND_CODE},
    {"epoch-time",          NO_ARG,       0, OPT_EPOCH_TIME},
    {"integer-ips",         NO_ARG,       0, OPT_INTEGER_IPS},
    {"zero-pad-ips",        NO_ARG,       0, OPT_ZERO_PAD_IPS},
    {"integer-sensors",     NO_ARG,       0, OPT_INTEGER_SENSORS},
    {"integer-tcp-flags",   NO_ARG,       0, OPT_INTEGER_TCP_FLAGS},
    {"no-titles",           NO_ARG,       0, OPT_NO_TITLES},
    {"no-columns",          NO_ARG,       0, OPT_NO_COLUMNS},
    {"column-separator",    REQUIRED_ARG, 0, OPT_COLUMN_SEPARATOR},
    {"no-final-delimiter",  NO_ARG,       0, OPT_NO_FINAL_DELIMITER},
    {"delimited",           OPTIONAL_ARG, 0, OPT_DELIMITED},
    {"print-filenames",     NO_ARG,       0, OPT_PRINT_FILENAMES},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {"output-path",         REQUIRED_ARG, 0, OPT_OUTPUT_PATH},
    {"pager",               REQUIRED_ARG, 0, OPT_PAGER},
    {"legacy-timestamps",   OPTIONAL_ARG, 0, OPT_LEGACY_TIMESTAMPS},
    {"dynamic-library",     REQUIRED_ARG, 0, OPT_PLUGIN},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    NULL, /* generated dynamically */
    "Print all known fields to the output",
    "Number of records to print, or 0 to print all. Def. 0",
    "Starting record number. Def. 1",
    "Ending record number. Def. None",
    "Parse options and print column titles only. Def. No",
    ("Load given plug-in to add fields. Switch may be repeated to\n"
     "\t load multiple plug-ins. Def. None"),
    "Print ICMP type/code in sPort/dPort fields. Def. No",
    "Print times in UNIX epoch seconds. Def. No",
    "Print IP numbers as integers. Def. Canonical form",
    "Print IP numbers in zero-padded canonical form. Def. No",
    "Print sensor as an integer. Def. Sensor name",
    "Print TCP Flags as an integer. Def. No",
    "Do not print column headers. Def. Print titles.",
    "Disable fixed-width columnar output. Def. Columnar",
    "Use specified character between columns. Def. '|'",
    "Suppress column delimiter at end of line. Def. No",
    "Shortcut for --no-columns --no-final-del --column-sep=CHAR",
    "Print names of input files as they are opened. Def. No",
    "Copy all input SiLK Flows to given pipe or file. Def. No",
    "Send output to given file path. Def. stdout",
    "Program to invoke to page output. Def. $SILK_PAGER or $PAGER",
    ("Timestamp format. Choices:\n"
     "\t0==(new)\"YYYY/MM/DDThh:mm:ss.sss\";"
     " 1==(legacy)\"MM/DD/YYYY hh:mm:ss\".\n"
     "\tDef. 0 when switch not provided; 1 when switch provided with no value"),
    "Deprecated.  Alias for --plugin",
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static void usageFields(FILE *fh);
static int  createStringmaps(void);
static int  selectFieldsDefault(void);
static int  selectFieldsAll(void);
static int  parseFields(const char* opt_arg);
static int  appAddPluginField(const sk_stringmap_entry_t *map_entry);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES] [FILES]\n"                                                   \
     "\tPrint SiLK Flow records in a |-delimited, columnar, human-readable\n" \
     "\tformat.  Use --fields to select columns to print. When no files are\n"\
     "\tgiven on the command line, flows are read from the standard input.\n")

    FILE *fh = USAGE_FH;
    int i;

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);
    fprintf(fh, "\nSWITCHES:\n");
    skOptionsDefaultUsage(fh);
    sksiteOptionsUsage(fh);

    for (i = 0; appOptions[i].name; i++ ) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch (appOptions[i].val) {
          case OPT_FIELDS:
            /* Dynamically build the help */
            usageFields(fh);

            /* Print the IPv6 usage next */
            skIPv6PolicyUsage(fh);
            break;
          default:
            /* Simple static help text from the appHelp array */
            fprintf(fh, "%s\n", appHelp[i]);
            break;
        }
    }

    skPluginOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
void appTeardown(void)
{
    static uint8_t teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* Plugin teardown */
    skPluginRunCleanup(SKPLUGIN_APP_CUT);
    skPluginTeardown();

    /* close the pager */
    if (using_pager) {
        skClosePager(stream_out, pager);
    }

    /* close the output-path */
    if (ioISP->passCount && ioISP->passFD[0] && ioISP->passFD[0] != stdout) {
        if (ioISP->passIsPipe[0]) {
            if (-1 == pclose(ioISP->passFD[0])) {
                skAppPrintErr("Error closing output pipe %s",
                              ioISP->passFPath[0]);
            }
        } else {
            if (EOF == fclose(ioISP->passFD[0])) {
                skAppPrintSyserror("Error closing output file %s",
                                   ioISP->passFPath[0]);
            }
        }
        ioISP->passFD[0] = NULL;
    }

    /* destroy output */
    rwAsciiStreamDestroy(&ascii_str);

    /* destroy field map */
    if (key_field_map != NULL) {
        skStringMapDestroy(key_field_map);
        key_field_map = NULL;
    }

    iochecksTeardown(ioISP);
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
void appSetup(int argc, char **argv)
{
    int j;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    memset(&cut_opts, 0, sizeof(cut_opts));
    delimiter = '|';
    stream_out = stdout;

    /* Initialize plugin library */
    skPluginSetup(1, SKPLUGIN_APP_CUT);

    /* do not allow ANY output files other than stdout */
    ioISP = iochecksSetup(1, 0, argc, argv);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE)
        || skIPv6PolicyOptionsRegister(&ipv6_policy))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* try to load hard-coded plugins */
    for (j = 0; app_static_plugins[j].name; ++j) {
        skPluginAddAsPlugin(app_static_plugins[j].name,
                            app_static_plugins[j].setup_fn);
    }
    for (j = 0; app_plugin_names[j]; ++j) {
        skPluginLoadPlugin(app_plugin_names[j], 0);
    }

    /* parse options */
    ioISP->firstFile = skOptionsParse(argc, argv);
    if (ioISP->firstFile < 0) {
        skAppUsage();         /* never returns */
    }

    /* Not having site config is allowed */
    sksiteConfigure(0);

    /* Create the --fields */
    if (createStringmaps()) {
        exit(EXIT_FAILURE);
    }

    /* Create the ascii stream */
    if (rwAsciiStreamCreate(&ascii_str)) {
        skAppPrintErr("Unable to create ascii stream");
        exit(EXIT_FAILURE);
    }

    /* Parse the --fields or --all-fields argument, or use the default
     * fields */
    if (fields_arg != NULL) {
        if (parseFields(fields_arg)) {
            exit(EXIT_FAILURE);
        }
    } else if (all_fields) {
        if (selectFieldsAll()) {
            exit(EXIT_FAILURE);
        }
    } else if (selectFieldsDefault()) {
        skAppPrintErr("Cannot set default output fields");
        exit(EXIT_FAILURE);
    }

    /* check limits; main loop uses skip_recs and num_recs */
    if (start_rec_num) {
        skip_recs = start_rec_num - 1;
    }
    if (end_rec_num) {
        if (end_rec_num < start_rec_num) {
            skAppPrintErr(("The %s is less than the %s: "
                           "%" PRIu64 " < %" PRIu64),
                          appOptions[OPT_END_REC_NUM].name,
                          appOptions[OPT_START_REC_NUM].name,
                          end_rec_num, start_rec_num);
            exit(EXIT_FAILURE);
        }
        if (start_rec_num) {
            /* unconditionally set num_recs to their difference */
            num_recs = end_rec_num - skip_recs;
        } else if ((num_recs > 0) && (num_recs < end_rec_num)) {
            skip_recs = end_rec_num - num_recs;
        } else {
            num_recs = end_rec_num;
        }
    }

    /* if no destination was specified, use stdout */
    if ((0 == ioISP->passCount)
        && iochecksPassDestinations(ioISP, "stdout", 1))
    {
        exit(EXIT_FAILURE);
    }
    stream_out = ioISP->passFD[0];

    /* Invoke the pager */
    using_pager = skOpenPagerWhenStdoutTty(&stream_out, &pager);
    if (using_pager < 0) {
        exit(EXIT_FAILURE);
    }

    /* set properties on the ascii-stream */
    rwAsciiSetDelimiter(ascii_str, delimiter);
    rwAsciiSetOutputHandle(ascii_str, stream_out);
    rwAsciiSetIPv6Policy(ascii_str, ipv6_policy);
    rwAsciiSetTimestampFlags(ascii_str, time_flags);

    if (cut_opts.no_titles) {
        rwAsciiSetNoTitles(ascii_str);
    }
    if (cut_opts.no_columns) {
        rwAsciiSetNoColumns(ascii_str);
    }
    if (cut_opts.integer_sensors) {
        rwAsciiSetIntegerSensors(ascii_str);
    }
    if (cut_opts.integer_tcp_flags) {
        rwAsciiSetIntegerTcpFlags(ascii_str);
    }
    if (cut_opts.no_final_delimiter) {
        rwAsciiSetNoFinalDelimiter(ascii_str);
    }
    if (cut_opts.icmp_type_and_code) {
        rwAsciiSetIcmpTypeCode(ascii_str);
    }

    /* conflicting properties */
    if (cut_opts.zero_pad_ips && !cut_opts.integer_ips) {
        rwAsciiSetZeroPadIps(ascii_str);
    }
    if (cut_opts.integer_ips) {
        if (cut_opts.zero_pad_ips) {
            skAppPrintErr("--integer-ips option overrides --zero-pad-ips\n"
                          "\tWill print IPs as integers.");
        }
        rwAsciiSetIntegerIps(ascii_str);
    }

    /*
     * if dry-run, print the column titles and exit
     */
    if (cut_opts.dry_run) {
        rwAsciiPrintTitles(ascii_str);
        appTeardown();
        exit(EXIT_SUCCESS);
    }

    /* Use STDIN as an input stream if it is not a TTY; make certain
     * we have some input and we are either reading from STDIN or
     * using files listed the command line, but not both. */
    if (iochecksAcceptFromStdin(ioISP) || iochecksInputs(ioISP, 0)) {
        skAppUsage();
    }

    /* looks good, open the --copy-input destination */
    if (iochecksOpenCopyDest(ioISP)) {
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;                       /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_FIELDS:
        if (fields_arg) {
            skAppPrintErr("Invalid %s: Switch specified multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if (all_fields) {
            skAppPrintErr("Invalid %s: The --%s switch was already given",
                          appOptions[opt_index].name,
                          appOptions[OPT_ALL_FIELDS].name);
            return 1;
        }
        fields_arg = opt_arg;
        break;

      case OPT_ALL_FIELDS:
        if (fields_arg) {
            skAppPrintErr("Invalid %s: The --%s switch was already given",
                          appOptions[opt_index].name,
                          appOptions[OPT_FIELDS].name);
            return 1;
        }
        all_fields = 1;
        break;

      case OPT_INTEGER_IPS:
        cut_opts.integer_ips = 1;
        break;

      case OPT_ZERO_PAD_IPS:
        cut_opts.zero_pad_ips = 1;
        break;

      case OPT_NUM_RECS:
        rv = skStringParseUint64(&num_recs, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_START_REC_NUM:
        rv = skStringParseUint64(&start_rec_num, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_END_REC_NUM:
        rv = skStringParseUint64(&end_rec_num, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_NO_TITLES:
        cut_opts.no_titles = 1;
        break;

      case OPT_NO_COLUMNS:
        cut_opts.no_columns = 1;
        break;

      case OPT_NO_FINAL_DELIMITER:
        cut_opts.no_final_delimiter = 1;
        break;

      case OPT_COLUMN_SEPARATOR:
        delimiter = opt_arg[0];
        break;

      case OPT_DELIMITED:
        cut_opts.no_columns = 1;
        cut_opts.no_final_delimiter = 1;
        if (opt_arg) {
            delimiter = opt_arg[0];
        }
        break;

      case OPT_EPOCH_TIME:
        /* disable legacy time and enable epoch time, but leave NOMSEC
         * flag as-is */
        time_flags = (time_flags & ~SKTIMESTAMP_MMDDYYYY) | SKTIMESTAMP_EPOCH;
        break;

      case OPT_ICMP_TYPE_AND_CODE:
        cut_opts.icmp_type_and_code = 1;
        break;

      case OPT_PLUGIN:
        if (skPluginLoadPluginOrDynlib(opt_arg, DYNLIB_CUT, 1) != 0) {
            skAppPrintErr("Unable to load %s as a plugin", opt_arg);
            return 1;
        }
        break;

      case OPT_INTEGER_SENSORS:
        cut_opts.integer_sensors = 1;
        break;

      case OPT_INTEGER_TCP_FLAGS:
        cut_opts.integer_tcp_flags = 1;
        break;

      case OPT_DRY_RUN:
        cut_opts.dry_run = 1;
        break;

      case OPT_PRINT_FILENAMES:
        cut_opts.print_filenames = 1;
        break;

      case OPT_COPY_INPUT:
        if (iochecksAllDestinations(ioISP, opt_arg)) {
            return 1;
        }
        break;

      case OPT_OUTPUT_PATH:
        if (iochecksPassDestinations(ioISP, opt_arg, 1)) {
            return 1;
        }
        break;

      case OPT_PAGER:
        pager = opt_arg;
        break;

      case OPT_LEGACY_TIMESTAMPS:
        if ((opt_arg == NULL) || (opt_arg[0] == '\0') || (opt_arg[0] == '1')) {
            if (time_flags & SKTIMESTAMP_EPOCH) {
                /* epoch time alredy set; just enable NOMSEC */
                time_flags |= SKTIMESTAMP_NOMSEC;
            } else {
                time_flags = SKTIMESTAMP_MMDDYYYY | SKTIMESTAMP_NOMSEC;
            }
        } else if (opt_arg[0] != '0') {
            skAppPrintErr("Invalid %s '%s': Select 0 or 1",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;

    } /* switch */

    return 0;                     /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 *  status = selectFieldsDefault();
 *
 *    Set the global ascii-stream to print rwcut's default columns.
 *    Return 0 on success or -1 on memory error.
 */
static int selectFieldsDefault(void)
{
    uint32_t default_fields[1+RWCUT_LAST_DEFAULT_FIELD];
    const uint32_t count = 1+RWCUT_LAST_DEFAULT_FIELD;
    uint32_t i;

    /* set up the default fields for rwcut */
    for (i = 0; i < count; ++i) {
        default_fields[i] = i;
    }

    rwAsciiAppendFields(ascii_str, default_fields, count);

    return 0;
}


/*
 *  status = selectFieldsAll();
 *
 *    Set the global ascii-stream to print all known fields---both
 *    built-in and from plug-ins.  Return 0 on success or -1 on memory
 *    error.
 */
static int selectFieldsAll(void)
{
    int rv = -1;
    uint32_t i;
    FILE *old_errs = NULL;
    sk_bitmap_t *field_seen = NULL;
    sk_dll_iter_t node;
    sk_stringmap_entry_t *entry;

    /* create a bitmap to keep track of the fields we've added */
    if (skBitmapCreate(&field_seen, 65536)) {
        goto END;
    }

    /* add all built-in fields to the ascii-stream */
    for (i = 0; i < RWREC_PRINTABLE_FIELD_COUNT; ++i) {
        if (rwAsciiAppendOneField(ascii_str, i)) {
            skAppPrintErr(("Cannot add field %" PRIu32 " to stream"), i);
            goto END;
        }
    }

    /* disable error output to avoid seeing warnings from plug-ins */
    old_errs = skAppSetErrStream(NULL);

    /* add the fields from every plug-in */
    skDLLAssignIter(&node, key_field_map);
    while (skDLLIterForward(&node, (void **)&entry) == 0) {
        if ((NULL != entry->userdata)
            && !skBitmapGetBit(field_seen, entry->id))
        {
            /* ignore errors */
            (void)appAddPluginField(entry);
            skBitmapSetBit(field_seen, entry->id);
        }
    }

    /* re-enable errors */
    skAppSetErrStream(old_errs);

    /* successful */
    rv = 0;

  END:
    if (field_seen) {
        skBitmapDestroy(&field_seen);
    }

    return rv;
}


/*
 *  status = parseFields(fields_string);
 *
 *    Parse the user's option for the --fields switch and set up the
 *    rwAsciiStream.  Return 0 on success; 1 on failure.
 */
static int parseFields(const char *field_string)
{
    sk_vector_t            *parsed_vec = NULL;
    sk_stringmap_entry_t  **entry;
    char *errmsg;
    uint32_t i;
    int rv = 1;

    if (field_string == NULL || field_string[0] == '\0') {
        skAppPrintErr("Missing --fields value");
        return 1;
    }

    /* vector to hold the parsed fields */
    parsed_vec = skVectorNew(sizeof(sk_stringmap_entry_t*));
    if (NULL == parsed_vec) {
        skAppPrintErr("Out of memory");
        goto END;
    }

    /* parse the field-list */
    if (skStringMapParse(key_field_map, field_string, SKSTRINGMAP_DUPES_KEEP,
                         parsed_vec, &errmsg))
    {
        skAppPrintErr("Invalid %s: %s",
                      appOptions[OPT_FIELDS].name, errmsg);
        return 1;
    }

    for (i = 0; NULL != (entry = skVectorGetValuePointer(parsed_vec, i)); ++i){
        if ((*entry)->userdata == NULL) {
            /* field is built-in */
            if (rwAsciiAppendOneField(ascii_str, (*entry)->id)) {
                skAppPrintErr("Cannot add field %s to stream",
                              (*entry)->name);
                goto END;
            }
        } else {
            /* field comes from a plug-in */
            assert((*entry)->id >= RWREC_PRINTABLE_FIELD_COUNT);

            if (appAddPluginField(*entry)) {
                skAppPrintErr("Cannot add field %s from plugin",
                              (*entry)->name);
                goto END;
            }
        }
    }

    /* successful */
    rv = 0;

  END:
    if (parsed_vec != NULL) {
        skVectorDestroy(parsed_vec);
    }

    return rv;
}


/*
 *  usageFields(fh);
 *
 *    Print the usage (help) message for --fields to the 'fh' file pointer
 */
static void usageFields(FILE *fh)
{
    const char *entry_name;
    uint32_t i;

    /* Create the string map for --fields */
    createStringmaps();

    fprintf(fh, "Field(s) to print. List columns separated by commas:\n");

    skStringMapPrintUsage(key_field_map, fh, 4);

    /* Print default fields */
    fprintf(fh, "\tDef: ");

    for (i = 0; i <= RWCUT_LAST_DEFAULT_FIELD; ++i) {
        entry_name = skStringMapGetFirstName(key_field_map, i);
        if (i > 0) {
            fprintf(fh, ",");
        }
        fprintf(fh, "%s", entry_name);
    }
    fprintf(fh, "\n");
}


/*
 *  ok = createStringmaps();
 *
 *    Create the string-map to assist in parsing the --fields switch.
 */
static int createStringmaps(void)
{
    skplugin_field_iter_t  iter;
    skplugin_err_t         err;
    skplugin_field_t      *field_handle;
    sk_stringmap_status_t  map_err;
    sk_stringmap_entry_t   entry;
    const char           **field_names;
    const char           **name;
    uint32_t               max_id;

    /* initialize string-map of field identifiers */
    if (rwAsciiFieldMapAddDefaultFields(&key_field_map)) {
        skAppPrintErr("Unable to setup fields stringmap");
        return -1;
    }
    max_id = RWREC_PRINTABLE_FIELD_COUNT - 1;

    /* add --fields from plug-ins */
    err = skPluginFieldIteratorBind(&iter, SKPLUGIN_FN_REC_TO_TEXT, 1);
    assert(err == SKPLUGIN_OK);

    while (skPluginFieldIteratorNext(&iter, &field_handle)) {
        err = skPluginFieldName(field_handle, &field_names);
        assert(err == SKPLUGIN_OK);
        ++max_id;

        /* Add fields to the key_field_map */
        for (name = field_names; *name; name++) {
            memset(&entry, 0, sizeof(entry));
            entry.name = *name;
            entry.id = max_id;
            entry.userdata = field_handle;
            map_err = skStringMapAddEntries(key_field_map, 1, &entry);
            if (map_err != SKSTRINGMAP_OK) {
                const char *plugin_name;
                skPluginFieldGetPluginName(field_handle, &plugin_name);
                skAppPrintErr(("Plug-in cannot add field named '%s': %s."
                               " Plug-in file: %s"),
                              *name, skStringMapStrerror(map_err),plugin_name);
                return -1;
            }
        }
    }

    return 0;
}


static void appPluginGetTitle(
    char        *text_buf,
    size_t       text_buf_size,
    void        *cb_data)
{
    skplugin_field_t *field_handle = (skplugin_field_t*)cb_data;
    skplugin_err_t  err;
    const char *title;

    err = skPluginFieldTitle(field_handle, &title);
    assert(SKPLUGIN_OK == err);
    strncpy(text_buf, title, text_buf_size);
    text_buf[text_buf_size-1] = '\0';
}


static int appPluginGetValue(
    const rwRec *rwrec,
    char        *text_buf,
    size_t       text_buf_size,
    void        *cb_data)
{
    skplugin_field_t *field_handle = (skplugin_field_t*)cb_data;
    skplugin_err_t err;

    err = skPluginFieldRunRecToTextFn(field_handle, text_buf, text_buf_size,
                                      rwrec, NULL);
    if (err != SKPLUGIN_OK) {
        const char **name;
        skPluginFieldName(field_handle, &name);
        skAppPrintErr(("Plugin-based field %s failed converting to text "
                       "with error code %d"), name[0], err);
        exit(EXIT_FAILURE);
    }
    return 0;
}


/*
 *  status = appAddPluginField(map_entry);
 *
 *    Add callbacks to the global 'ascii_str' to print a field that
 *    comes from a plug-in.
 *
 *    Returns 0 on success, or -1 for a memory allocation error or if
 *    the plug-in that provides the field numbered 'field_id' cannot
 *    be found.
 */
static int appAddPluginField(const sk_stringmap_entry_t *map_entry)
{
    skplugin_field_t *field_handle;
    size_t            text_width;
    skplugin_err_t    err;

    field_handle = (skplugin_field_t*)map_entry->userdata;

    /* Activate the plugin (so cleanup knows about it) */
    err = skPluginFieldActivate(field_handle);
    if (err != SKPLUGIN_OK) {
        return -1;
    }

    /* Initialize this field */
    err = skPluginFieldRunInitialize(field_handle);
    if (err != SKPLUGIN_OK) {
        return -1;
    }

    /* get the text width for this field */
    err = skPluginFieldGetLenText(field_handle, &text_width);
    if (err != SKPLUGIN_OK) {
        return -1;
    }
    if (text_width == 0 || text_width > RWCUT_MAX_COLUMN_WIDTH) {
        text_width = RWCUT_MAX_COLUMN_WIDTH;
    }

    return rwAsciiAppendCallbackField(ascii_str, &appPluginGetTitle,
                                      &appPluginGetValue, field_handle,
                                      text_width);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
