/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
** rwcut.c
**
**      cut fields/records from the given input file(s) using field
**      specifications from here, record filter specifications from
**      module libfilter
**
** 1/15/2002
**
** Suresh L. Konda
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwcut.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "rwcut.h"


/* TYPEDEFS AND MACROS */

/* Where to write filenames if --print-file specified */
#define PRINT_FILENAMES_FH  stderr

/* When --copy-input is active but the required 'num_recs' records
 * have been printed, skStreamSkipRecords() is used to read data from
 * all remaining input streams.  This specifies the record-count
 * parameter to pass to that function. */
#define CUT_SKIP_COUNT 65536

/* EXPORTED VARIABLES */

/* The object to convert the record to text */
rwAsciiStream_t *ascii_str = NULL;

/* The output stream: where to print the records */
FILE *stream_out;

/* interface to the iochecks library */
iochecksInfoStruct_t *ioISP;

/* number records to print */
uint64_t num_recs = 0;

/* number of records to skip before printing */
uint64_t skip_recs = 0;

/* user's options */
cut_opt_flags_t cut_opts;

/* delimiter between columns */
char delimiter;

/* how to handle IPv6 flows */
sk_ipv6policy_t ipv6_policy = SK_IPV6POLICY_MIX;

/* how to print timesamps */
uint32_t time_flags = 0;


/* FUNCTION DEFINITIONS */

/*
 *  status = cutFile(path);
 *
 *    Open the SiLK Flow file at 'path', then read and print its
 *    records.  Return -1 on error, 1 if all requested records have
 *    been printed, and 0 if processing should continue.
 */
static int cutFile(const char *path)
{
    static int copy_input_only = 0;
    static int print_titles = 1;
    skstream_t *rwios;
    rwRec rwrec;
    int rv = SKSTREAM_OK;
    size_t num_skipped;
    int ret_val = 0;

    /* open input file */
    rv = skStreamOpenSilkFlow(&rwios, path, SK_IO_READ);
    if (rv) {
        skStreamPrintLastErr(rwios, rv, &skAppPrintErr);
        skStreamDestroy(&rwios);
        return -1;
    }
    (void)skStreamSetCopyInput(rwios, ioISP->inputCopyFD);
    if (cut_opts.print_filenames) {
        fprintf(PRINT_FILENAMES_FH, "%s\n", skStreamGetPathname(rwios));
    }

    skStreamSetIPv6Policy(rwios, ipv6_policy);

    /* handle case where all requested records have been printed, but
     * we need to write all records to the --copy-input stream. */
    if (copy_input_only) {
        while ((rv = skStreamSkipRecords(rwios, CUT_SKIP_COUNT, NULL))
               == SKSTREAM_OK)
            ;  /* empty */

        if (SKSTREAM_ERR_EOF != rv) {
            ret_val = -1;
        }
        goto END;
    }

    /* print title if we haven't yet */
    if (print_titles) {
        print_titles = 0;
        rwAsciiPrintTitles(ascii_str);
    }

    /* skip any leading records */
    if (skip_recs) {
        rv = skStreamSkipRecords(rwios, skip_recs, &num_skipped);
        switch (rv) {
          case SKSTREAM_OK:
            skip_recs -= num_skipped;
            break;
          case SKSTREAM_ERR_EOF:
            skip_recs -= num_skipped;
            goto END;
          default:
            ret_val = -1;
            goto END;
        }
    }

    if (0 == num_recs) {
        /* print all records */
        while ((rv = skStreamReadRecord(rwios, &rwrec)) == SKSTREAM_OK) {
            rwAsciiPrintRec(ascii_str, &rwrec);
        }
        if (SKSTREAM_ERR_EOF != rv) {
            ret_val = -1;
        }
    } else {
        while (num_recs
               && ((rv = skStreamReadRecord(rwios, &rwrec)) == SKSTREAM_OK))
        {
            rwAsciiPrintRec(ascii_str, &rwrec);
            --num_recs;
        }
        switch (rv) {
          case SKSTREAM_OK:
          case SKSTREAM_ERR_EOF:
            break;
          default:
            ret_val = -1;
            goto END;
        }
        if (0 == num_recs) {
            if (NULL == ioISP->inputCopyFD) {
                /* we're done */
                ret_val = 1;
            } else {
                /* send all remaining records to copy-input */
                copy_input_only = 1;
                while ((rv = skStreamSkipRecords(rwios, CUT_SKIP_COUNT, NULL))
                       == SKSTREAM_OK)
                    ;  /* empty */
                if (SKSTREAM_ERR_EOF != rv) {
                    ret_val = -1;
                }
            }
        }
    }

  END:
    if (-1 == ret_val) {
        skStreamPrintLastErr(rwios, rv, &skAppPrintErr);
    }
    skStreamDestroy(&rwios);
    return ret_val;
}


int main(int argc, char **argv)
{
    int rv = 0;
    int i;

    appSetup(argc, argv);                 /* never returns on error */

    /* Process the files from command line or stdin */
    for (i = ioISP->firstFile; rv == 0 && i < ioISP->fileCount; ++i) {
        rv = cutFile(ioISP->fnArray[i]);
        if (-1 == rv) {
            exit(EXIT_FAILURE);
        }
    }

    /* done */
    appTeardown();

    return 0;
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
