/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwcountutils.c
**
**  utility routines for rwcount.c
**
**  Michael P. Collins
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwcountsetup.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "rwcount.h"


/* LOCAL DEFINES AND TYPEDEFS */

/* where to send --help output */
#define USAGE_FH stdout


/* LOCAL VARIABLES */

/* start and end time strings that user entered */
static const char *start_epoch;
static const char *end_epoch;


/* OPTIONS SETUP */

typedef enum {
    OPT_BIN_SIZE, OPT_LOAD_SCHEME, OPT_SKIP_ZEROES,
    OPT_EPOCH_SLOTS, OPT_BIN_SLOTS,
    OPT_START_EPOCH, OPT_END_EPOCH,
    OPT_NO_TITLES, OPT_NO_COLUMNS,
    OPT_COLUMN_SEPARATOR, OPT_NO_FINAL_DELIMITER, OPT_DELIMITED,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT, OPT_OUTPUT_PATH, OPT_PAGER, OPT_LEGACY_TIMESTAMPS
} appOptionsEnum;

static struct option appOptions[] = {
    /* Size of the bins in seconds */
    {"bin-size",            REQUIRED_ARG, 0, OPT_BIN_SIZE},
    /* Bin splitting scheme */
    {"load-scheme",         REQUIRED_ARG, 0, OPT_LOAD_SCHEME},
    {"skip-zeroes",         NO_ARG,       0, OPT_SKIP_ZEROES},
    {"epoch-slots",         NO_ARG,       0, OPT_EPOCH_SLOTS},
    {"bin-slots",           NO_ARG,       0, OPT_BIN_SLOTS},
    {"start-epoch",         REQUIRED_ARG, 0, OPT_START_EPOCH},
    {"end-epoch",           REQUIRED_ARG, 0, OPT_END_EPOCH},
    {"no-titles",           NO_ARG,       0, OPT_NO_TITLES},
    {"no-columns",          NO_ARG,       0, OPT_NO_COLUMNS},
    {"column-separator",    REQUIRED_ARG, 0, OPT_COLUMN_SEPARATOR},
    {"no-final-delimiter",  NO_ARG,       0, OPT_NO_FINAL_DELIMITER},
    {"delimited",           OPTIONAL_ARG, 0, OPT_DELIMITED},
    {"print-filenames",     NO_ARG,       0, OPT_PRINT_FILENAMES},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {"output-path",         REQUIRED_ARG, 0, OPT_OUTPUT_PATH},
    {"pager",               REQUIRED_ARG, 0, OPT_PAGER},
    {"legacy-timestamps",   OPTIONAL_ARG, 0, OPT_LEGACY_TIMESTAMPS},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    "Size of bins in seconds; may be fractional. Def. 30.000",
    ("Defines bin loading, options are:\n"
     "\t0: split volume evenly across the BINS the record covers\n"
     "\t1: fill FIRST appropriate bin with complete volume\n"
     "\t2: fill LAST appropriate bin with complete volume\n"
     "\t3: fill CENTERMOST bin with complete volume\n"
     "\t4: split volume into bins proportional to time ACTIVE in bin [Def]"),
    "Do not print bins that have no flows. Def. Print all",
    "Print bin labels using epoch time. Def. Human readable",
    "Print bin labels using the internal bin index. Def. No",
    "Print bins from this time forward. Def. First nonzero bin",
    "Print bins until this time. Def. Last nonzero bin",
    "Do not print column titles. Def. Print titles",
    "Disable fixed-width columnar output. Def. Columnar",
    "Use specified character between columns. Def. '|'",
    "Suppress column delimiter at end of line. Def. No",
    "Shortcut for --no-columns --no-final-del --column-sep=CHAR",
    "Print names of input files as they are opened. Def. No",
    "Copy all input SiLK Flows to given pipe or file. Def. No",
    "Send output to given file path. Def. stdout",
    "Program to invoke to page output. Def. $SILK_PAGER or $PAGER",
    ("Timestamp format. Choices:\n"
     "\t0==(new)\"YYYY/MM/DDThh:mm:ss\"; 1==(legacy)\"MM/DD/YYYY hh:mm:ss\".\n"
     "\tDef. 0 when switch not provided; 1 when switch provided with no value"),
    (char *)NULL
};



/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES] [FILES]\n"                                                   \
     "\tSummarize SiLK Flow records across time, producing textual output\n"  \
     "\twith counts of bytes, packets, and flow records for each time bin.\n" \
     "\tWhen no files given on command line, flows are read from STDIN.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    sksiteOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
void appTeardown(void)
{
    static uint8_t teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* close the output-path */
    if (ioISP->passCount && ioISP->passFD[0] && ioISP->passFD[0] != stdout) {
        if (ioISP->passIsPipe[0]) {
            if (-1 == pclose(ioISP->passFD[0])) {
                skAppPrintErr("Error closing output pipe %s",
                              ioISP->passFPath[0]);
            }
        } else {
            if (EOF == fclose(ioISP->passFD[0])) {
                skAppPrintSyserror("Error closing output file %s",
                                   ioISP->passFPath[0]);
            }
        }
        ioISP->passFD[0] = NULL;
    }
    iochecksTeardown(ioISP);

    /* free our memory */
    if (bins.data) {
        free(bins.data);
    }

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
void appSetup(int argc, char **argv)
{
    sktime_t end_time;
    int start_precision = -1;
    int end_precision = -1;
    int64_t bin_count;
    int rv;

    /* make sure count of option's declarations and help-strings match */
    assert((sizeof(appOptions)/sizeof(struct option)) ==
           (sizeof(appHelp)/sizeof(char *)));

    /* initialize globals */
    memset(&flags, 0, sizeof(flags));
    flags.delimiter = '|';
    flags.load_scheme = DEFAULT_LOAD_SCHEME;

    memset(&bins, 0, sizeof(bins));
    bins.start_epoch = RWCO_UNINIT_START;
    bins.end_epoch = RWCO_UNINIT_END;
    bins.size = DEFAULT_BINSIZE;

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    ioISP = iochecksSetup(1, 0, argc, argv);

    /* parse options; print usage if error */
    ioISP->firstFile = skOptionsParse(argc, argv);
    if (ioISP->firstFile < 0) {
        skAppUsage();
    }

    /* try to load site config file; if it fails, we will not be able
     * to resolve flowtype and sensor from input file names */
    sksiteConfigure(0);

    /* parse the epoch times */
    if (start_epoch) {
        rv = skStringParseDatetime(&(bins.start_epoch), start_epoch,
                                   &start_precision);
        if (rv) {
            skAppPrintErr("Invalid %s '%s': %s",
                          appOptions[OPT_START_EPOCH].name, start_epoch,
                          skStringParseStrerror(rv));
            skAppUsage();
        }
    }

    if (end_epoch) {
        rv = skStringParseDatetime(&end_time, end_epoch, &end_precision);
        if (rv) {
            skAppPrintErr("Invalid %s '%s': %s",
                          appOptions[OPT_END_EPOCH].name, end_epoch,
                          skStringParseStrerror(rv));
            skAppUsage();
        }

        if (start_epoch) {
            /* move end-time to its ceiling */
            if (start_precision > end_precision) {
                end_precision = start_precision;
            }
            skDatetimeCeiling(&end_time, &end_time, end_precision);
            ++end_time;

            /* verify times */
            if (end_time <= bins.start_epoch) {
                char buf_s[SKTIMESTAMP_STRLEN];
                char buf_e[SKTIMESTAMP_STRLEN];
                skAppPrintErr("The %s is less than %s: %s < %s",
                              appOptions[OPT_END_EPOCH].name,
                              appOptions[OPT_START_EPOCH].name,
                              sktimestamp_r(buf_e, end_time, SKTIMESTAMP_NOMSEC),
                              sktimestamp_r(buf_s, bins.start_epoch,
                                            SKTIMESTAMP_NOMSEC));
                exit(EXIT_FAILURE);
            }

            /* make certain end_epoch fails on a boundary */
            bin_count = ((end_time - bins.start_epoch) / bins.size);
            if (end_time > (bins.start_epoch + bins.size * bin_count)) {
                /* make end_epoch fail on a bin boundary */
                ++bin_count;
                end_time = bins.start_epoch + bins.size * bin_count;
            }
            bins.end_epoch = end_time;
        } else {
            /* when only end_time is given, create bins up to its
             * ceiling value */
            bins.end_epoch = end_time;
            skDatetimeCeiling(&end_time, &end_time, end_precision);
            ++end_time;
            bin_count = ((end_time - bins.end_epoch) / bins.size);
            bins.end_epoch += bin_count * bins.size;
        }
    }

    /* Use STDIN as an input stream if it is not a TTY; make certain
     * we have some input and we are either reading from STDIN or
     * using files listed the command line, but not both. */
    if (iochecksAcceptFromStdin(ioISP) || iochecksInputs(ioISP, 0)) {
        skAppUsage();
    }

    /* if no destination was specified, use stdout */
    if ((0 == ioISP->passCount)
        && iochecksPassDestinations(ioISP, "stdout", 1))
    {
        exit(EXIT_FAILURE);
    }

    /* looks good, open the --copy-input destination */
    if (iochecksOpenCopyDest(ioISP)) {
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;                     /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Called by skOptionsParse(), this handles a user-specified switch
 *    that the application has registered, typically by setting global
 *    variables.  Returns 1 if the switch processing failed or 0 if it
 *    succeeded.  Returning a non-zero from from the handler causes
 *    skOptionsParse() to return a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int  appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    uint32_t opt_u32;
    double opt_double;
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_LOAD_SCHEME:
        rv = skStringParseUint32(&opt_u32, opt_arg, 0, MAX_LOAD_SCHEME);
        if (rv) {
            skAppPrintErr("Invalid %s '%s': %s",
                          appOptions[opt_index].name, opt_arg,
                          skStringParseStrerror(rv));
            return 1;
        }
        flags.load_scheme = opt_u32;
        break;

      case OPT_BIN_SIZE:
        rv = skStringParseDouble(&opt_double, opt_arg, 0.001, INT32_MAX);
        if (rv) {
            skAppPrintErr("Invalid %s '%s': %s",
                          appOptions[opt_index].name, opt_arg,
                          skStringParseStrerror(rv));
            return 1;
        }
        bins.size = (sktime_t)(1000.0 * opt_double);
        break;

      case OPT_EPOCH_SLOTS:
        flags.timeflags = SKTIMESTAMP_EPOCH;
        break;

      case OPT_BIN_SLOTS:
        flags.label_index = 1;
        break;

      case OPT_START_EPOCH:
        if (start_epoch != NULL) {
            skAppPrintErr("The --%s switch was given multiple times",
                          appOptions[opt_index].name);
        }
        start_epoch = opt_arg;
        break;

      case OPT_END_EPOCH:
        if (end_epoch != NULL) {
            skAppPrintErr("The --%s switch was given multiple times",
                          appOptions[opt_index].name);
        }
        end_epoch = opt_arg;
        break;

      case OPT_SKIP_ZEROES:
        flags.skip_zeroes = 1;
        break;

      case OPT_NO_TITLES:
        flags.no_titles = 1;
        break;

      case OPT_NO_COLUMNS:
        flags.no_columns = 1;
        break;

      case OPT_NO_FINAL_DELIMITER:
        flags.no_final_delimiter = 1;
        break;

      case OPT_COLUMN_SEPARATOR:
        flags.delimiter = opt_arg[0];
        break;

      case OPT_DELIMITED:
        flags.no_columns = 1;
        flags.no_final_delimiter = 1;
        if (opt_arg) {
            flags.delimiter = opt_arg[0];
        }
        break;

      case OPT_PRINT_FILENAMES:
        flags.print_filenames = 1;
        break;

      case OPT_COPY_INPUT:
        if (iochecksAllDestinations(ioISP, opt_arg)) {
            return 1;
        }
        break;

      case OPT_OUTPUT_PATH:
        if (iochecksPassDestinations(ioISP, opt_arg, 1)) {
            return 1;
        }
        break;

      case OPT_PAGER:
        pager = opt_arg;
        break;

      case OPT_LEGACY_TIMESTAMPS:
        if ((opt_arg == NULL) || (opt_arg[0] == '\0') || (opt_arg[0] == '1')) {
            flags.timeflags = SKTIMESTAMP_MMDDYYYY;
        } else if (opt_arg[0] != '0') {
            skAppPrintErr("Invalid %s '%s': Select 0 or 1",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;

    }

    return 0;                     /* OK */
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
