/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
 * addrcount.c
 *
 * this is the last major rwset tool I want to write, it takes in two streams
 * of data, an rwset and an rwfilter stream.  From this data, it then generates
 * a result - one of three outputs:
 *            totals (default) - outputs to screen a table containing the
 *                               ip address, bytes, packets, records
 *            print-ips        - outputs to screen the ip addresses
 *            set-file         - outputs to screen the set data.
 *
 * So the reason for the second two is because I'm including three thresholds
 * here - bytes, packets & records.
 *
 * 12/2 notes.  Often, the best is the enemy of the good, I am implementing
 * a simple version of this application for now with the long-term plan being
 * that I am going to write a better, faster, set-friendly version later.
 */

#include <silk/silk.h>

RCSIDENT("$SiLK: rwaddrcount.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/iptree.h>
#include <silk/skstream.h>
#include <silk/utils.h>
#include <silk/iochecks.h>
#include <silk/sksite.h>
#include "rwaddrcount.h"


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write output from --help */
#define USAGE_FH stdout

/* Where to write filenames if --print-file specified */
#define PRINT_FILENAMES_FH  stderr

/*
 * Get the IP address from the record 'r' to use as the key.  Uses the
 * global variable 'use_dest'
 */
#define GETIP(r)                                                \
    ((use_dest) ? rwRecGetDIPv4(r) : rwRecGetSIPv4(r))

/*
 * One of the problems I've had recently is a tendency to focus on
 * doing the perfect hash.  The perfect hash is nice, but what we need
 * RIGHT HERE RIGHT NOW is a tool that'll actually do the job.  Enough
 * mathematical wanking.
 *
 * So, this is a hash whose collision compensation algorithm is linear
 * chaining.  I'm not happy about it, but it'll do for now.
 */
#define HASHFUNC(value)                                         \
    ((value ^ (value >> 7) ^ (value << 23)) % RWAC_ARRAYSIZE)

/*
 * CMPFNC: return TRUE if IP on the rwRec 'r' matches the IP stored in
 * the countRecord_t 'b'.
 */
#define CMPFNC(r, cr)                           \
    ((cr)->cr_key == GETIP(r))

/*
 * when generating ouput, this macro will evaluate to TRUE if the
 * record is within the limits given by the user and should be
 * printed/counted/used-to-genrate-output.  This macro uses the global
 * limit variables.
 */
#define IS_RECORD_WITHIN_LIMITS(count_rec)      \
    ((count_rec)->cr_bytes   >= min_bytes &&    \
     (count_rec)->cr_packets >= min_packets &&  \
     (count_rec)->cr_records >= min_records &&  \
     (count_rec)->cr_bytes   <= max_bytes &&    \
     (count_rec)->cr_packets <= max_packets &&  \
     (count_rec)->cr_records <= max_records)


/* formats for printing records */
#define FMT_REC_VALUE                                                   \
    "%*s%c%*" PRIu64 "%c%*" PRIu32 "%c%*" PRIu32 "%c%*s%c%*s%s\n"
#define FMT_REC_TITLE  "%*s%c%*s%c%*s%c%*s%c%*s%c%*s%s\n"
#define FMT_REC_WIDTH  {15, 20, 10, 10, 20, 20}

/* formats for printing statistics */
#define FMT_STAT_VALUE                                                  \
    "%*s%c%*" PRIu32 "%c%*" PRIu64 "%c%*" PRIu64 "%c%*" PRIu64 "%s\n"
#define FMT_STAT_TITLE "%*s%c%*s%c%*s%c%*s%c%*s%s\n"
#define FMT_STAT_WIDTH {10, 10, 20, 15, 15}


/* LOCAL VARIABLES */

static iochecksInfoStruct_t *ioISP;

/* output mode */
static rwac_print_mode_t print_mode = RWAC_PMODE_NONE;

static uint64_t min_bytes = 0;
static uint64_t max_bytes = UINT64_MAX;
static uint32_t min_packets = 0;
static uint32_t max_packets = UINT32_MAX;
static uint32_t min_records = 0;
static uint32_t max_records = UINT32_MAX;
static countRecord_t *hash_bins[RWAC_ARRAYSIZE];
static uint32_t hash_bin_count = 0;
static const char *ipset_file = NULL;

/* whether to use the source(==0) or destination(==1) IPs */
static uint8_t use_dest = 0;

/* output mode for IPs */
static uint8_t integer_ips_flag = 0;
static uint8_t zero_pad_ips_flag = 0;
static uint8_t sort_ips_flag = 0;

/* whether to suppress column titles; default no (i.e. print titles) */
static uint8_t no_titles = 0;

/* whether to suppress columnar output; default no (i.e. columnar) */
static uint8_t no_columns = 0;

/* whether to suppress the final delimiter; default no (i.e. end with '|') */
static uint8_t no_final_delimiter = 0;

/* column separator */
static char delimiter = '|';

/* what to print at the end of the line */
static char final_delim[] = {'\0', '\0'};

/* flags to pass to sktimestamp_r() */
static uint32_t time_flags = SKTIMESTAMP_NOMSEC;

/* whether to print the names of files as they are opened */
static int print_filenames = 0;

/* name of program to run to page output */
static char *pager = NULL;


/* OPTIONS */

/* Names of options; keep the order in sync with appOptions[] */
typedef enum {
    OPT_PRINT_RECORDS,
    OPT_PRINT_STAT,
    OPT_PRINT_IPS,
    OPT_USE_DEST,
    OPT_MIN_BYTES,
    OPT_MIN_PACKETS,
    OPT_MIN_RECORDS,
    OPT_MAX_BYTES,
    OPT_MAX_PACKETS,
    OPT_MAX_RECORDS,
    OPT_SET_FILE,
    OPT_INTEGER_IPS,
    OPT_ZERO_PAD_IPS,
    OPT_SORT_IPS,
    OPT_NO_TITLES,
    OPT_NO_COLUMNS,
    OPT_COLUMN_SEPARATOR,
    OPT_NO_FINAL_DELIMITER,
    OPT_DELIMITED,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT,
    OPT_OUTPUT_PATH,
    OPT_PAGER,
    OPT_LEGACY_TIMESTAMPS
} appOptionsEnum;

static struct option appOptions[] = {
    {"print-recs",          NO_ARG,       0, OPT_PRINT_RECORDS},
    {"print-stat",          NO_ARG,       0, OPT_PRINT_STAT},
    {"print-ips",           NO_ARG,       0, OPT_PRINT_IPS},
    {"use-dest",            NO_ARG,       0, OPT_USE_DEST},
    {"min-bytes",           REQUIRED_ARG, 0, OPT_MIN_BYTES},
    {"min-packets",         REQUIRED_ARG, 0, OPT_MIN_PACKETS},
    {"min-records",         REQUIRED_ARG, 0, OPT_MIN_RECORDS},
    {"max-bytes",           REQUIRED_ARG, 0, OPT_MAX_BYTES},
    {"max-packets",         REQUIRED_ARG, 0, OPT_MAX_PACKETS},
    {"max-records",         REQUIRED_ARG, 0, OPT_MAX_RECORDS},
    {"set-file",            REQUIRED_ARG, 0, OPT_SET_FILE},
    {"integer-ips",         NO_ARG,       0, OPT_INTEGER_IPS},
    {"zero-pad-ips",        NO_ARG,       0, OPT_ZERO_PAD_IPS},
    {"sort-ips",            NO_ARG,       0, OPT_SORT_IPS},
    {"no-titles",           NO_ARG,       0, OPT_NO_TITLES},
    {"no-columns",          NO_ARG,       0, OPT_NO_COLUMNS},
    {"column-separator",    REQUIRED_ARG, 0, OPT_COLUMN_SEPARATOR},
    {"no-final-delimiter",  NO_ARG,       0, OPT_NO_FINAL_DELIMITER},
    {"delimited",           OPTIONAL_ARG, 0, OPT_DELIMITED},
    {"print-filenames",     NO_ARG,       0, OPT_PRINT_FILENAMES},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {"output-path",         REQUIRED_ARG, 0, OPT_OUTPUT_PATH},
    {"pager",               REQUIRED_ARG, 0, OPT_PAGER},
    {"legacy-timestamps",   OPTIONAL_ARG, 0, OPT_LEGACY_TIMESTAMPS},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    "Print summary byte, packet, flow counts per IP bin",
    "Print statistics (total bytes, packets, flows, unique IPs)",
    "Print IP addresses only to stdout",
    "Use destination IP address as key. Def. Source address",
    ("Do not print IPs when sum has less than this many total\n"
     "\tbytes. Def. 1"),
    ("Do not print IPs when sum has less than this many total\n"
     "\tpackets. Def. 1"),
    ("Do not print IPs when sum has less than this many total\n"
     "\trecords. Def. 1"),
    ("Do not print IPs when sum has more than this many total\n"
     "\tbytes. Def. 18446744073709551615"),
    ("Do not print IPs when sum has more than this many total\n"
     "\tpackets. Def. 4294967295"),
    ("Do not print IPs when sum has more than this many total\n"
     "\trecords. Def. 4294967295"),
    "Write IPs to specified binary IPset file. Def. No",
    "Print IP numbers as integers. Def. No",
    "Print IP numbers as zero-padded dotted-decimal. Def. No",
    "When printing results, sort by IP address. Def. No",
    "Do not print column titles. Def. Print titles",
    "Disable fixed-width columnar output. Def. Columnar",
    "Use specified character between columns. Def. '|'",
    "Suppress column delimiter at end of line. Def. No",
    "Shortcut for --no-columns --no-final-del --column-sep=CHAR",
    "Print names of input files as they are opened. Def. No",
    "Copy all input SiLK Flows to given pipe or file. Def. No",
    "Send output to given file path. Def. stdout",
    "Program to invoke to page output. Def. $SILK_PAGER or $PAGER",
    ("Timestamp format. Choices:\n"
     "\t0==(new)\"YYYY/MM/DDThh:mm:ss\"; 1==(legacy)\"MM/DD/YYYY hh:mm:ss\".\n"
     "\tDef. 0 when switch not provided; 1 when switch provided with no value"),
    (char *)NULL
};

static struct option legacyOptions[] = {
    {"byte-min",            REQUIRED_ARG, 0, OPT_MIN_BYTES},
    {"packet-min",          REQUIRED_ARG, 0, OPT_MIN_PACKETS},
    {"rec-min",             REQUIRED_ARG, 0, OPT_MIN_RECORDS},
    {"byte-max",            REQUIRED_ARG, 0, OPT_MAX_BYTES},
    {"packet-max",          REQUIRED_ARG, 0, OPT_MAX_PACKETS},
    {"rec-max",             REQUIRED_ARG, 0, OPT_MAX_RECORDS},
    {0,0,0,0}               /* sentinel entry */
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
void appUsageLong(void)
{
#define USAGE_MSG                                                            \
    ("{--print-recs|--print-stat|--print-ips} [SWITCHES] [FILES]\n"          \
     "\tSummarize SiLK Flow records by source or destination IP; with\n"     \
     "\tthe --print-recs option will produce textual ouput with counts of\n" \
     "\tbytes, packets, and flow records for each IP, and the time range\n"  \
     "\twhen the IP was active.  When no files are given on command line,\n" \
     "\tflows are read from STDIN.\n")

    FILE *fh = USAGE_FH;
    int i;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    sksiteOptionsUsage(fh);

    fprintf(fh, "\nDEPRECATED SWITCHES:\n");
    for (i = 0; legacyOptions[i].name; ++i) {
        fprintf(fh, "--%s %s. Deprecated alias for --%s\n",
                legacyOptions[i].name, SK_OPTION_HAS_ARG(legacyOptions[i]),
                appOptions[legacyOptions[i].val].name);
    }
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;
    countRecord_t *bin;
    countRecord_t *next;
    uint32_t i;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* free our memory */
    for (i = 0; i < RWAC_ARRAYSIZE; ++i) {
        if (hash_bins[i] != NULL) {
            bin = hash_bins[i];
            do {
                next = bin->cr_next;
                free(bin);
                bin = next;
            } while (bin != hash_bins[i]);
        }
    }

    /* close the outputs */
    if (ioISP->passCount && ioISP->passFD[0] && ioISP->passFD[0] != stdout) {
        if (ioISP->passIsPipe[0]) {
            if (-1 == pclose(ioISP->passFD[0])) {
                skAppPrintErr("Error closing output pipe %s",
                              ioISP->passFPath[0]);
            }
        } else {
            if (EOF == fclose(ioISP->passFD[0])) {
                skAppPrintSyserror("Error closing output file %s",
                                   ioISP->passFPath[0]);
            }
        }
        ioISP->passFD[0] = NULL;
    }
    iochecksTeardown(ioISP);
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    use_dest = 0;

    /* create the input/output checker */
    ioISP = iochecksSetup(1, 0, argc, argv);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsRegister(legacyOptions, &appOptionsHandler, NULL)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse options */
    ioISP->firstFile = skOptionsParse(argc, argv);
    if (ioISP->firstFile < 0) {
        skAppUsage();/* never returns */
    }

    /* handle the final delimiter */
    if (!no_final_delimiter) {
        final_delim[0] = delimiter;
    }

    /* try to load site config file; if it fails, we will not be able
     * to resolve flowtype and sensor from input file names */
    sksiteConfigure(0);

    /* Use STDIN as an input stream if it is not a TTY; make certain
     * we have some input and we are either reading from STDIN or
     * using files listed the command line, but not both. */
    if (iochecksAcceptFromStdin(ioISP) || iochecksInputs(ioISP, 0)) {
        skAppUsage();
    }

    if (print_mode == RWAC_PMODE_NONE) {
        skAppPrintErr("Must specify --%s, --%s, --%s, or --%s",
                      appOptions[OPT_PRINT_RECORDS].name,
                      appOptions[OPT_PRINT_STAT].name,
                      appOptions[OPT_PRINT_IPS].name,
                      appOptions[OPT_SET_FILE].name);
        skAppUsage();
    }

    /* verify that the bounds make sense */
    if (min_bytes > max_bytes) {
        skAppPrintErr(("The %s value is greater than %s:"
                       " %" PRIu64 " > %" PRIu64),
                      appOptions[OPT_MIN_BYTES].name,
                      appOptions[OPT_MAX_BYTES].name,
                      min_bytes, max_bytes);
        exit(EXIT_FAILURE);
    }
    if (min_packets > max_packets) {
        skAppPrintErr(("The %s value is greater than %s:"
                       " %" PRIu32 " > %" PRIu32),
                      appOptions[OPT_MIN_PACKETS].name,
                      appOptions[OPT_MAX_PACKETS].name,
                      min_packets, max_packets);
        exit(EXIT_FAILURE);
    }
    if (min_records > max_records) {
        skAppPrintErr(("The %s value is greater than %s:"
                       " %" PRIu32 " > %" PRIu32),
                      appOptions[OPT_MIN_RECORDS].name,
                      appOptions[OPT_MAX_RECORDS].name,
                      min_records, max_records);
        exit(EXIT_FAILURE);
    }

    /* Do they want the IPs in sorted order? */
    if (sort_ips_flag) {
        switch (print_mode) {
          case RWAC_PMODE_IPS:
            print_mode = RWAC_PMODE_SORTED_IPS;
            break;
          case RWAC_PMODE_RECORDS:
            print_mode = RWAC_PMODE_SORTED_RECORDS;
            break;
          default:
            /* skAppPrintErr("--sort-ips switch ignored"); */
            break;
        }
    }

    /* Using both the integer-ips and zero-pad-ips options is
     * inconsistent.  Let integer-ips override zero-pad-ips and print
     * warning */
    if (integer_ips_flag && zero_pad_ips_flag) {
        skAppPrintErr("Will print IPs as integers: --%s overrides --%s",
                      appOptions[OPT_INTEGER_IPS].name,
                      appOptions[OPT_ZERO_PAD_IPS].name);
        zero_pad_ips_flag = 0;
    }

    /* if no destination was specified, use stdout */
    if ((0 == ioISP->passCount)
        && iochecksPassDestinations(ioISP, "stdout", 1))
    {
        exit(EXIT_FAILURE);
    }

    /* looks good, open the --copy-input destination */
    if (iochecksOpenCopyDest(ioISP)) {
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;                       /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Called by skOptionsParse(), this handles a user-specified switch
 *    that the application has registered, typically by setting global
 *    variables.  Returns 1 if the switch processing failed or 0 if it
 *    succeeded.  Returning a non-zero from from the handler causes
 *    skOptionsParse() to return a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int  appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_USE_DEST:
        use_dest = 1;
        break;

      case OPT_MIN_BYTES:
        rv = skStringParseUint64(&min_bytes, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MAX_BYTES:
        rv = skStringParseUint64(&max_bytes, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MIN_PACKETS:
        rv = skStringParseUint32(&min_packets, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MAX_PACKETS:
        rv = skStringParseUint32(&max_packets, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MIN_RECORDS:
        rv = skStringParseUint32(&min_records, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MAX_RECORDS:
        rv = skStringParseUint32(&max_records, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_PRINT_STAT:
        print_mode = RWAC_PMODE_STAT;
        break;

      case OPT_PRINT_IPS:
        print_mode = RWAC_PMODE_IPS;
        break;

      case OPT_PRINT_RECORDS:
        print_mode = RWAC_PMODE_RECORDS;
        break;

      case OPT_SET_FILE:
        print_mode = RWAC_PMODE_IPSETFILE;
        ipset_file = opt_arg;
        break;

      case OPT_INTEGER_IPS:
        integer_ips_flag = 1;
        break;

      case OPT_ZERO_PAD_IPS:
        zero_pad_ips_flag = 1;
        break;

      case OPT_SORT_IPS:
        sort_ips_flag = 1;
        break;

      case OPT_NO_TITLES:
        no_titles = 1;
        break;

      case OPT_NO_COLUMNS:
        no_columns = 1;
        break;

      case OPT_COLUMN_SEPARATOR:
        delimiter = opt_arg[0];
        break;

      case OPT_NO_FINAL_DELIMITER:
        no_final_delimiter = 1;
        break;

      case OPT_DELIMITED:
        no_columns = 1;
        no_final_delimiter = 1;
        if (opt_arg) {
            delimiter = opt_arg[0];
        }
        break;

      case OPT_PRINT_FILENAMES:
        print_filenames = 1;
        break;

      case OPT_COPY_INPUT:
        if (iochecksAllDestinations(ioISP, opt_arg)) {
            return 1;
        }
        break;

      case OPT_OUTPUT_PATH:
        if (iochecksPassDestinations(ioISP, opt_arg, 1)) {
            return 1;
        }
        break;

      case OPT_PAGER:
        pager = opt_arg;
        break;

      case OPT_LEGACY_TIMESTAMPS:
        if ((opt_arg == NULL) || (opt_arg[0] == '\0') || (opt_arg[0] == '1')) {
            time_flags |= SKTIMESTAMP_MMDDYYYY;
        } else if (opt_arg[0] != '0') {
            skAppPrintErr("Invalid %s '%s': Select 0 or 1",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;

    }

    return 0;                     /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 * void addToBin(countRecord_t *bin, rwRec *rwrec)
 *
 * Adds the contents of a record to the values stored in a bin.
 */
static void addToBin(countRecord_t *bin, const rwRec *rwrec)
{
    assert(bin->cr_key == GETIP(rwrec));
    bin->cr_bytes += rwRecGetBytes(rwrec);
    bin->cr_packets += rwRecGetPkts(rwrec);
    ++bin->cr_records;
    if (rwRecGetStartSeconds(rwrec) < bin->cr_start) {
        bin->cr_start = rwRecGetStartSeconds(rwrec);
    }
    if (bin->cr_end < rwRecGetEndSeconds(rwrec)) {
        bin->cr_end = rwRecGetEndSeconds(rwrec);
    }
}


/*
 * int newBin(rwRec *rwrec)
 *
 * Creates a new countRecord and initializes it with the values from
 * the record.
 */
static countRecord_t *newBin(const rwRec *rwrec)
{
    countRecord_t *bin;

    bin = malloc(sizeof(countRecord_t));
    if (bin == NULL) {
        skAppPrintErr("Error allocating memory for bin");
        exit(EXIT_FAILURE);
    }
    hash_bin_count++;

    bin->cr_bytes = rwRecGetBytes(rwrec);
    bin->cr_packets = rwRecGetPkts(rwrec);
    bin->cr_records = 1;
    bin->cr_key = GETIP(rwrec);
    bin->cr_start = rwRecGetStartSeconds(rwrec);
    bin->cr_end = rwRecGetEndSeconds(rwrec);
    bin->cr_next = NULL;

    return bin;
}



/*
 * SECTION: Dumping
 *
 * All the output routines are in this section of the text.
 */


/*
 *  hashToIPTree(&iptree);
 *
 *    Fills in the 'iptree' with all the IPs in global hash_bins array.
 */
static void hashToIPTree(skIPTree_t **ip_tree)
{
    countRecord_t *bin;
    uint32_t i;

    skIPTreeCreate(ip_tree);

    for (i = 0; i < RWAC_ARRAYSIZE; i++) {
        bin = hash_bins[i];
        while (bin != NULL) {
            if (IS_RECORD_WITHIN_LIMITS(bin)) {
                skIPTreeAddAddress((*ip_tree), (bin->cr_key));
            }
            if (bin->cr_next == hash_bins[i]) {
                break;
            }
            bin = bin->cr_next;
        }
    }
}


/*
 *  int dumpRecords(outfp)
 *
 *    Dumps the addrcount contents as a record of bytes, packets,
 *    times &c to 'outfp'
 *
 *    This is the typical text output from addrcount.
 *
 */
static int dumpRecords(FILE *outfp)
{
    int w[] = FMT_REC_WIDTH;
    uint32_t i;
    countRecord_t *bin;
    char ip_st[SK_NUM2DOT_STRLEN];
    char start_st[SKTIMESTAMP_STRLEN];
    char end_st[SKTIMESTAMP_STRLEN];

    if (no_columns) {
        memset(w, 0, sizeof(w));
    }

    if ( !no_titles) {
        fprintf(outfp, FMT_REC_TITLE,
                w[0], (use_dest ? "dIP" : "sIP"), delimiter,
                w[1], "Bytes",      delimiter,
                w[2], "Packets",    delimiter,
                w[3], "Records",    delimiter,
                w[4], "Start_Time", delimiter,
                w[5], "End_Time",   final_delim);
    }

    for (i = 0; i < RWAC_ARRAYSIZE; i++) {
        bin = hash_bins[i];
        while (bin != NULL) {
            if (IS_RECORD_WITHIN_LIMITS(bin)) {
                /* print IP in appropriate form */
                if (integer_ips_flag) {
                    snprintf(ip_st, sizeof(ip_st), "%lu",
                             (unsigned long)bin->cr_key);
                } else if (zero_pad_ips_flag) {
                    num2dot0_r(bin->cr_key, ip_st);
                } else {
                    num2dot_r(bin->cr_key, ip_st);
                }
                /* print "volumes" and time */
                fprintf(outfp, FMT_REC_VALUE,
                        w[0], ip_st,           delimiter,
                        w[1], bin->cr_bytes,   delimiter,
                        w[2], bin->cr_packets, delimiter,
                        w[3], bin->cr_records, delimiter,
                        w[4], sktimestamp_r(start_st,
                                            sktimeCreate(bin->cr_start, 0),
                                            time_flags),
                        delimiter,
                        w[5], sktimestamp_r(end_st,
                                            sktimeCreate(bin->cr_end, 0),
                                            time_flags),
                        final_delim);
            }
            if (bin->cr_next == hash_bins[i]) {
                break;
            }
            bin = bin->cr_next;
        }
    }
    return 0;
}


/*
 *  int dumpRecordsSorted(outfp)
 *
 *    Dumps the addrcount contents as a record of bytes, packets,
 *    times &c to 'outfp', sorted by the IP address.
 *
 */
static int dumpRecordsSorted(FILE *outfp)
{
    int w[] = FMT_REC_WIDTH;
    uint32_t ip, key;
    countRecord_t *bin;
    skIPTree_t *ipset;
    skIPTreeIterator_t iter;
    char ip_st[SK_NUM2DOT_STRLEN];
    char start_st[SKTIMESTAMP_STRLEN];
    char end_st[SKTIMESTAMP_STRLEN];

    if (no_columns) {
        memset(w, 0, sizeof(w));
    }

    hashToIPTree(&ipset);
    if (skIPTreeIteratorBind(&iter, ipset)) {
        skAppPrintErr("Unable to bind IPTree iterator");
        return 1;
    }

    if ( !no_titles) {
        fprintf(outfp, FMT_REC_TITLE,
                w[0], (use_dest ? "dIP" : "sIP"), delimiter,
                w[1], "Bytes",      delimiter,
                w[2], "Packets",    delimiter,
                w[3], "Records",    delimiter,
                w[4], "Start_Time", delimiter,
                w[5], "End_Time",   final_delim);
    }

    while (skIPTreeIteratorNext(&ip, &iter) == SK_ITERATOR_OK) {

        /* find the ip's entry in the hash table */
        key = HASHFUNC(ip);

        /* loop through the list of records at this hash table entry
         * until we find the one that has the IP we want */
        bin = hash_bins[key];
        while (bin != NULL) {
            if (ip != bin->cr_key) {
                /* not the ip we wanted; goto next */
                bin = bin->cr_next;
                /* if we've looped all the around, we missed the
                 * IP we wanted, and things are horked */
                assert(bin != hash_bins[key]);
                continue;
            }

            /* print IP in appropriate form */
            if (integer_ips_flag) {
                sprintf(ip_st, "%lu", (unsigned long)bin->cr_key);
            } else if (zero_pad_ips_flag) {
                num2dot0_r(bin->cr_key, ip_st);
            } else {
                num2dot_r(bin->cr_key, ip_st);
            }
            /* print "volumes" and time */
            fprintf(outfp, FMT_REC_VALUE,
                    w[0], ip_st,                  delimiter,
                    w[1], bin->cr_bytes,   delimiter,
                    w[2], bin->cr_packets, delimiter,
                    w[3], bin->cr_records, delimiter,
                    w[4], sktimestamp_r(start_st,
                                        sktimeCreate(bin->cr_start, 0),
                                        time_flags),
                    delimiter,
                    w[5], sktimestamp_r(end_st,
                                        sktimeCreate(bin->cr_end, 0),
                                        time_flags),
                    final_delim);
            break;
        }
    }

    return 0;
}



/*
 *  int dumpIPs(outfp)
 *
 *    writes IP addresses to the stream 'outfp' in hash-table order
 *    (unsorted).
 *
 *    Returns 0 on success.
 */
static int dumpIPs(FILE *outfp)
{
    uint32_t i;
    countRecord_t *bin;

    if ( !no_titles) {
        fprintf(outfp, "%15s\n", (use_dest ? "dIP" : "sIP"));
    }

    for (i = 0; i < RWAC_ARRAYSIZE; i++) {
        bin = hash_bins[i];
        while (bin != NULL) {
            if (IS_RECORD_WITHIN_LIMITS(bin)) {
                if (integer_ips_flag) {
                    fprintf(outfp, "%15lu\n",
                            (unsigned long)bin->cr_key);
                } else if (zero_pad_ips_flag) {
                    fprintf(outfp, "%15s\n",
                            num2dot0(bin->cr_key));
                } else {
                    fprintf(outfp, "%15s\n",
                            num2dot(bin->cr_key));
                }
            }
            if (bin->cr_next == hash_bins[i]) {
                break;
            }
            bin = bin->cr_next;
        }
    }
    return 0;
}



/*
 *  int dumpIPsSorted(outfp)
 *
 *    Writes the IPs to the stream 'outfp' in sorted order.
 *
 */
static int dumpIPsSorted(FILE *outfp)
{
    uint32_t ip;
    skIPTree_t *ipset;
    skIPTreeIterator_t iter;

    hashToIPTree(&ipset);
    if (skIPTreeIteratorBind(&iter, ipset)) {
        skAppPrintErr("Unable to create IPTree iterator");
        return 1;
    }

    if ( !no_titles) {
        fprintf(outfp, "%15s\n", (use_dest ? "dIP" : "sIP"));
    }

    while (skIPTreeIteratorNext(&ip, &iter) == SK_ITERATOR_OK) {
        if (integer_ips_flag) {
            fprintf(outfp, "%15lu\n",
                    (unsigned long)ip);
        } else if (zero_pad_ips_flag) {
            fprintf(outfp, "%15s\n",
                    num2dot0(ip));
        } else {
            fprintf(outfp, "%15s\n",
                    num2dot(ip));
        }
    }

    return 0;
}



/*
 *  int dumpStats(outfp)
 *
 *    dumps a text string describing rwaddrcont results to the stream
 *    'outfp'.
 */
static int dumpStats(FILE *outfp)
{
    int fmt_width[] = FMT_STAT_WIDTH;
    uint32_t i;
    uint32_t qual_ips;
    uint64_t qual_bytes, qual_packets, qual_records;
    uint64_t tot_bytes,  tot_packets,  tot_records;
    countRecord_t *bin;

    qual_ips = 0;
    qual_bytes = qual_packets = qual_records = 0;
    tot_bytes  = tot_packets  = tot_records  = 0;

    if (no_columns) {
        memset(fmt_width, 0, sizeof(fmt_width));
    }
    for (i = 0; i < RWAC_ARRAYSIZE; i++) {
        bin = hash_bins[i];
        while (bin != NULL) {
            tot_bytes   += bin->cr_bytes;
            tot_packets += bin->cr_packets;
            tot_records += bin->cr_records;

            if (IS_RECORD_WITHIN_LIMITS(bin)) {
                ++qual_ips;
                qual_bytes   += bin->cr_bytes;
                qual_packets += bin->cr_packets;
                qual_records += bin->cr_records;
            }
            if (bin->cr_next == hash_bins[i]) {
                break;
            }
            bin = bin->cr_next;
        }
    }

    /* title */
    if ( !no_titles) {
        fprintf(outfp, FMT_STAT_TITLE,
                fmt_width[0], "", delimiter,
                fmt_width[1], (use_dest ? "dIP_Uniq" : "sIP_Uniq"), delimiter,
                fmt_width[2], "Bytes",   delimiter,
                fmt_width[3], "Packets", delimiter,
                fmt_width[4], "Records", final_delim);
    }

    fprintf(outfp, FMT_STAT_VALUE,
            fmt_width[0], "Total",          delimiter,
            fmt_width[1], hash_bin_count,   delimiter,
            fmt_width[2], tot_bytes,        delimiter,
            fmt_width[3], tot_packets,      delimiter,
            fmt_width[4], tot_records,      final_delim);

    /* print qualifying records if limits were given */
    if (0 < min_bytes || max_bytes < UINT64_MAX
        || 0 < min_packets || max_packets < UINT32_MAX
        || 0 < min_records || max_records < UINT32_MAX)
    {
        fprintf(outfp, FMT_STAT_VALUE,
                fmt_width[0], "Qualifying", delimiter,
                fmt_width[1], qual_ips,     delimiter,
                fmt_width[2], qual_bytes,   delimiter,
                fmt_width[3], qual_packets, delimiter,
                fmt_width[4], qual_records, final_delim);
    }

    return 0;
}



/*
 *  int dumpIPSet (char *path)
 *
 *    Dumps the ip addresses counted during normal operation to disk
 *    in IPSet format.
 */
static int dumpIPSet(const char *path)
{
    skIPTree_t *ipset;

    /* Create the ip tree */
    hashToIPTree(&ipset);

    /*
     * Okay, now we write to disk.
     */
    if (skIPTreeSave(ipset, path)) {
        exit(EXIT_FAILURE);
    };
    return 0;
}


/*
 *  void processInput(void)
 *
 *    Reads the flow records from all input streams and fills the hash
 *    table with countRecord_t's.
 */
static void processInput(void)
{
    int file_idx;
    uint32_t hash_idx;
    countRecord_t *bin;
    skstream_t *rwIOS;
    rwRec rwrec;
    int rv;

    /* Read in records from all input files */
    for (file_idx = ioISP->firstFile; file_idx < ioISP->fileCount; ++file_idx){
        /* Open file */
        rv = skStreamOpenSilkFlow(&rwIOS, ioISP->fnArray[file_idx],SK_IO_READ);
        if (rv) {
            skStreamPrintLastErr(rwIOS, rv, &skAppPrintErr);
            skStreamDestroy(&rwIOS);
            exit(EXIT_FAILURE);
        }
        (void)skStreamSetCopyInput(rwIOS, ioISP->inputCopyFD);
        if (print_filenames) {
            fprintf(PRINT_FILENAMES_FH, "%s\n", skStreamGetPathname(rwIOS));
        }
        skStreamSetIPv6Policy(rwIOS, SK_IPV6POLICY_ASV4);

        /* Read records */
        while ((rv = skStreamReadRecord(rwIOS, &rwrec)) == SKSTREAM_OK) {
            hash_idx = HASHFUNC(GETIP(&rwrec));

            /*
             * standard hash foo - check to see if we've got a value.
             * If not, create and stuff.  If so, check to see if they
             * match - if so, stuff.  If not, move down until you do -
             * if you find nothing, stuff.
             */
            if (hash_bins[hash_idx] == NULL) {
                /* new bin */
                hash_bins[hash_idx] = newBin(&rwrec);
                hash_bins[hash_idx]->cr_next = hash_bins[hash_idx];
            } else {
                /* hash collision */
                bin = hash_bins[hash_idx];
                while ((bin->cr_next != hash_bins[hash_idx]) &&
                       !CMPFNC(&rwrec, bin))
                {
                    bin = bin->cr_next;
                }
                /*
                 * Alright, we've either hit the end of the linked
                 * list or we've found the value (or both).  Check if
                 * we found the value. */
                if (CMPFNC(&rwrec, bin)) {
                    addToBin(bin, &rwrec);
                } else {
                    assert(bin->cr_next == hash_bins[hash_idx]);
                    bin->cr_next = newBin(&rwrec);
                    bin = bin->cr_next;
                    bin->cr_next = hash_bins[hash_idx]; /* Restore the loop */
                }
                hash_bins[hash_idx] = bin;
            }
        }
        if (rv != SKSTREAM_ERR_EOF) {
            skStreamPrintLastErr(rwIOS, rv, &skAppPrintErr);
        }
        skStreamDestroy(&rwIOS);
    }
}



/*
 * main invocation
 */
int main(int argc, char **argv)
{
    int using_pager = 0;
    FILE *stream_out;

    appSetup(argc, argv);                 /* never returns on error */

    processInput();

    stream_out = ioISP->passFD[0];

    /* Invoke the pager when appropriate. */
    switch (print_mode) {
      case RWAC_PMODE_STAT:
      case RWAC_PMODE_IPSETFILE:
        break;
      case RWAC_PMODE_RECORDS:
      case RWAC_PMODE_IPS:
      case RWAC_PMODE_SORTED_RECORDS:
      case RWAC_PMODE_SORTED_IPS:
        using_pager = skOpenPagerWhenStdoutTty(&stream_out, &pager);
        if (using_pager < 0) {
            exit(EXIT_FAILURE);
        }
        break;
      case RWAC_PMODE_NONE:
      case RWAC_PMODE_RWACFILE:
        skAbortBadCase(print_mode);
    }

    /* Produce the output */
    switch (print_mode) {
      case RWAC_PMODE_STAT:
        dumpStats(stream_out);
        break;
      case RWAC_PMODE_IPSETFILE:
        dumpIPSet(ipset_file);
        break;
      case RWAC_PMODE_RECORDS:
        dumpRecords(stream_out);
        break;
      case RWAC_PMODE_IPS:
        dumpIPs(stream_out);
        break;
      case RWAC_PMODE_SORTED_RECORDS:
        dumpRecordsSorted(stream_out);
        break;
      case RWAC_PMODE_SORTED_IPS:
        dumpIPsSorted(stream_out);
        break;
      case RWAC_PMODE_NONE:
      case RWAC_PMODE_RWACFILE:
        skAbortBadCase(print_mode);
    }

    /* close the pager  */
    if (using_pager) {
        skClosePager(stream_out, pager);
    }

    /* output */
    appTeardown();

    return (0);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
