#######################################################################
# Copyright (C) 2007-2012 by Carnegie Mellon University.
#
# @OPENSOURCE_HEADER_START@
#
# Use of the SILK system and related source code is subject to the terms
# of the following licenses:
#
# GNU Public License (GPL) Rights pursuant to Version 2, June 1991
# Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
#
# NO WARRANTY
#
# ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
# PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
# PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
# "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
# LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
# MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
# OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
# SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
# TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
# WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
# LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
# CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
# CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
# DELIVERABLES UNDER THIS LICENSE.
#
# Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
# Mellon University, its trustees, officers, employees, and agents from
# all claims or demands made against them (and any related losses,
# expenses, or attorney's fees) arising out of, or relating to Licensee's
# and/or its sub licensees' negligent use or willful misuse of or
# negligent conduct or willful misconduct regarding the Software,
# facilities, or other rights or assistance granted by Carnegie Mellon
# University under this License, including, but not limited to, any
# claims of product liability, personal injury, death, damage to
# property, or violation of any laws or regulations.
#
# Carnegie Mellon University Software Engineering Institute authored
# documents are sponsored by the U.S. Department of Defense under
# Contract FA8721-05-C-0003. Carnegie Mellon University retains
# copyrights in all material produced under this contract. The U.S.
# Government retains a non-exclusive, royalty-free license to publish or
# reproduce these documents, or allow others to do so, for U.S.
# Government purposes only pursuant to the copyright license under the
# contract clause at 252.227.7013.
#
# @OPENSOURCE_HEADER_END@
#
#######################################################################

#######################################################################
# $SiLK: site.py 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $
#######################################################################

try:
    from silk.pysilk_nl import *
except ImportError:
    from silk.pysilk import *

import copy

__all__ = ['sensors', 'classes', 'classtypes', 'types', 'default_types', 
           'class_sensors', 'sensor_classes', 'sensor_id', 
           'sensor_from_id', 'init_site', 'have_site_config']

_initialized = False

_sensor_map = {}
_class_map = {}
_flowtype_map = {}

def _init_maps():
    global _initialized
    if _initialized:
        return
    si = sensor_info()
    ci = class_info()
    fi = flowtype_info()
    for item in si.itervalues():
        newitem = copy.deepcopy(item)
        newitem['classes'] = tuple(ci[x]['name'] for x in item['classes'])
        _sensor_map[item['name']] = newitem
    for item in ci.itervalues():
        newitem = copy.deepcopy(item)
        newitem['sensors'] = tuple(si[x]['name'] for x in item['sensors'])
        newitem['flowtypes'] = tuple(fi[x]['name'] for x in item['flowtypes'])
        newitem['default_flowtypes'] = tuple(fi[x]['name'] for x in
                                             item['default_flowtypes'])
        _class_map[item['name']] = newitem
    for item in fi.itervalues():
        newitem = copy.deepcopy(item)
        newitem['class'] = ci[item['class']]['name']
        _flowtype_map[item['name']] = newitem

def sensors():
    "sensors() -> tuple of sensors"
    _init_maps()
    return tuple(_sensor_map)

def class_sensors(cls):
    "class_sensors(class) -> tuple of sensors in class"
    _init_maps()
    return _class_map[cls]['sensors']

def classes(): 
    "classes() -> tuple of classes"
    _init_maps()
    return tuple(_class_map)

def sensor_classes(sensor):
    "sensor_classes(sensor) -> tuple of classes associated with sensor"
    return _sensor_map[sensor]['classes']

def classtypes():
    "classtypes() -> tuple of (class, type) pairs"
    _init_maps()
    clinfo = class_info()
    return tuple((clinfo[x['class']]['name'], x['type'])
                 for x in flowtype_info().itervalues())

def types(cls):
    "types(class) -> tuple of types in class"
    _init_maps()
    return tuple(_flowtype_map[x]['type'] 
                 for x in _class_map[cls]['flowtypes'])

def default_types(cls):
    "default_types(class) -> tuple of default types in class"
    _init_maps()
    return tuple(_flowtype_map[x]['type'] 
                 for x in _class_map[cls]['default_flowtypes'])

def sensor_id(sensor):
    "sensor_id(sensor) -> numeric id of sensor"
    _init_maps()
    return _sensor_map[sensor]['id']

def sensor_from_id(id):
    "sensor_from_id(id) -> sensor from numeric id"
    _init_maps()
    sinfo = sensor_info()
    return sinfo[id]['name']
