/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwuniqsetup.c
**
**  Application setup for rwuniq.  See rwuniq.c for a description.
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwuniqsetup2.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/sksite.h>
#include <silk/iochecks.h>
#include <silk/skprefixmap.h>
#include <silk/skcountry.h>
#include "skunique.h"
#include "rwuniq2.h"


/* TYPEDEFS AND DEFINES */

/* file handle for --help usage message */
#define USAGE_FH stdout

/* where to write filenames if --print-file specified */
#define PRINT_FILENAMES_FH  stderr


/* LOCAL VARIABLES */

/* available key fields */
static sk_stringmap_t *key_field_map = NULL;

/* available aggregate value fields */
static sk_stringmap_t *value_field_map = NULL;

/* the text the user entered for the --fields switch */
static char *fields_arg = NULL;

/* the text the user entered for the --values switch */
static char *values_arg = NULL;

/* name of program to run to page output */
static char *pager;

/* temporary directory */
static const char *temp_directory = NULL;

/* the floor of the sTime and/or eTime */
static sktime_t time_bin_size = 0;

/* when time-binning is active and all time fields---sTime, eTime,
 * elapased---are requested, we must adjust 'elapsed' so that it is
 * equal to eTime-sTime.  this is non-0 if we must adjust */
static int adjust_elapsed = 0;

/* input checker */
static iochecksInfoStruct_t *ioISP = NULL;

/* fields that get defined just like plugins */
static const struct app_static_plugins_st {
    char               *name;
    skplugin_setup_fn_t setup_fn;
} app_static_plugins[] = {
    {"addrtype",        skAddressTypesAddFields},
    {"ccfilter",        skCountryAddFields},
    {"pmapfilter",      skPrefixMapAddFields},
    {NULL, NULL}        /* sentinel */
};

/* plug-ins to attempt to load at startup */
static const char *app_plugin_names[] = {
    SK_PLUGIN_ADD_SUFFIX("silkpython"),
    NULL /* sentinel */
};

/* non-zero if we are shutting down due to a signal; controls whether
 * errors are printed in appTeardown(). */
static int caught_signal = 0;


/* OPTIONS */

typedef enum {
    OPT_FIELDS,
    OPT_VALUES,
    OPT_PLUGIN,
    OPT_ALL_COUNTS,
    /* OPT_BYTES...OPT_DIP_DISTINCT must be contiguous and appear in
     * same order as in builtin_values[] */
    OPT_BYTES,
    OPT_PACKETS,
    OPT_FLOWS,
    OPT_STIME,
    OPT_ETIME,
    OPT_SIP_DISTINCT,
    OPT_DIP_DISTINCT,
    OPT_PRESORTED_INPUT,
    OPT_SORT_OUTPUT,
    OPT_BIN_TIME,
    OPT_EPOCH_TIME,
    OPT_INTEGER_IPS,
    OPT_ZERO_PAD_IPS,
    OPT_INTEGER_SENSORS,
    OPT_NO_TITLES,
    OPT_NO_COLUMNS,
    OPT_COLUMN_SEPARATOR,
    OPT_NO_FINAL_DELIMITER,
    OPT_DELIMITED,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT,
    OPT_OUTPUT_PATH,
    OPT_PAGER,
    OPT_LEGACY_TIMESTAMPS
} appOptionsEnum;


static struct option appOptions[] = {
    {"fields",              REQUIRED_ARG, 0, OPT_FIELDS},
    {"values",              REQUIRED_ARG, 0, OPT_VALUES},
    {"plugin",              REQUIRED_ARG, 0, OPT_PLUGIN},
    {"all-counts",          NO_ARG,       0, OPT_ALL_COUNTS},
    {"bytes",               OPTIONAL_ARG, 0, OPT_BYTES},
    {"packets",             OPTIONAL_ARG, 0, OPT_PACKETS},
    {"flows",               OPTIONAL_ARG, 0, OPT_FLOWS},
    {"stime",               NO_ARG,       0, OPT_STIME},
    {"etime",               NO_ARG,       0, OPT_ETIME},
    {"sip-distinct",        OPTIONAL_ARG, 0, OPT_SIP_DISTINCT},
    {"dip-distinct",        OPTIONAL_ARG, 0, OPT_DIP_DISTINCT},
    {"presorted-input",     NO_ARG,       0, OPT_PRESORTED_INPUT},
    {"sort-output",         NO_ARG,       0, OPT_SORT_OUTPUT},
    {"bin-time",            OPTIONAL_ARG, 0, OPT_BIN_TIME},
    {"epoch-time",          NO_ARG,       0, OPT_EPOCH_TIME},
    {"integer-ips",         NO_ARG,       0, OPT_INTEGER_IPS},
    {"zero-pad-ips",        NO_ARG,       0, OPT_ZERO_PAD_IPS},
    {"integer-sensors",     NO_ARG,       0, OPT_INTEGER_SENSORS},
    {"no-titles",           NO_ARG,       0, OPT_NO_TITLES},
    {"no-columns",          NO_ARG,       0, OPT_NO_COLUMNS},
    {"column-separator",    REQUIRED_ARG, 0, OPT_COLUMN_SEPARATOR},
    {"no-final-delimiter",  NO_ARG,       0, OPT_NO_FINAL_DELIMITER},
    {"delimited",           OPTIONAL_ARG, 0, OPT_DELIMITED},
    {"print-filenames",     NO_ARG,       0, OPT_PRINT_FILENAMES},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {"output-path",         REQUIRED_ARG, 0, OPT_OUTPUT_PATH},
    {"pager",               REQUIRED_ARG, 0, OPT_PAGER},
    {"legacy-timestamps",   OPTIONAL_ARG, 0, OPT_LEGACY_TIMESTAMPS},
    {"dynamic-library",     REQUIRED_ARG, 0, OPT_PLUGIN},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    NULL, /* generated dynamically */
    NULL, /* generated dynamically */
    ("Load given plug-in to add fields and/or values. Switch may\n"
     "\tbe repeated to load multiple plug-ins. Def. None"),
    ("Enable the next five switches--count everything.  If no\n"
     "\tcount is specified, flows are counted.  Def. No"),
    ("Sum bytes in each bin; optionally choose to print\n"
     "\tbins whose total is in given range; range is MIN or MIN-MAX. Def. No"),
    ("Sum packets in each bin; optionally choose to print\n"
     "\tbins whose total is in given range; range is MIN or MIN-MAX. Def. No"),
    ("Count flow records in each bin; optionally choose to print\n"
     "\tbins whose count is in given range; range is MIN or MIN-MAX. Def. No"),
    "Print earliest time flow was seen in each bin. Def. No",
    "Print latest time flow was seen  in each bin. Def. No",
    ("Count distinct sIPs in each bin; optionally choose to\n"
     "\tprint bins whose count is in range; range is MIN or MIN-MAX. Def. No"),
    ("Count distinct dIPs in each bin; optionally choose to\n"
     "\tprint bins whose count is in range; range is MIN or MIN-MAX. Def. No"),
    ("Assume input has been presorted using\n"
     "\trwsort invoked with the exact same --fields value. Def. No"),
    ("Present the output in sorted order. Def. No"),
    ("When using 'sTime' or 'eTime' as a key, adjust time(s) to\n"
     "\tto appear in N-second bins (floor of time is used). Def. No, "),
    "Print times in UNIX epoch seconds. Def. No",
    "Print IP numbers as integers. Def. Canonical form",
    "Print IP numbers in zero-padded canonical form. Def. No",
    "Print sensor as an integer. Def. Sensor name",
    "Do not print column titles. Def. Print titles",
    "Disable fixed-width columnar output. Def. Columnar",
    "Use specified character between columns. Def. '|'",
    "Suppress column delimiter at end of line. Def. No",
    "Shortcut for --no-columns --no-final-del --column-sep=CHAR",
    "Print names of input files as they are opened. Def. No",
    "Copy all input SiLK Flows to given pipe or file. Def. No",
    "Send output to given file path. Def. stdout",
    "Program to invoke to page output. Def. $SILK_PAGER or $PAGER",
    ("Timestamp format. Choices:\n"
     "\t0==(new)\"YYYY/MM/DDThh:mm:ss\"; 1==(legacy)\"MM/DD/YYYY hh:mm:ss\".\n"
     "\tDef. 0 when switch not provided; 1 when switch provided with no value"),
    "Deprecated.  Alias for --plugin",
    (char *)NULL
};



/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static void appHandleSignal(int sig);

static int createStringmaps(void);
static int parseKeyFields(const char *field_string);
static int parseValueFields(const char *value_string);
static int appAddPlugin(
    skplugin_field_t   *pi_field,
    int                 key_or_value);
static int prepareFileForRead(skstream_t *rwios);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
    FILE *fh = USAGE_FH;
    int i;

#define USAGE_MSG                                                             \
    ("--fields=N [SWITCHES] [FILES]\n"                                        \
     "\tSummarize SiLK Flow records into user-defined keyed bins specified\n" \
     "\twith the --fields switch.  For each keyed bin, print byte, packet,\n" \
     "\tand/or flow counts and/or the time window when key was active.\n"     \
     "\tWhen no files are given on command line, flows are read from STDIN.\n")

    /* Create the string maps for --fields and --values */
    createStringmaps();

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);

    fprintf(fh, "\nSWITCHES:\n");
    skOptionsDefaultUsage(fh);
    for (i = 0; appOptions[i].name; i++) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch ((appOptionsEnum)i) {
          case OPT_FIELDS:
            /* Dynamically build the help */
            fprintf(fh, "Field(s) to use as key:\n");
            skStringMapPrintUsage(key_field_map, fh, 4);
            break;
          case OPT_VALUES:
            fprintf(fh, "Value(s) to compute:\n");
            skStringMapPrintUsage(value_field_map, fh, 4);
            break;
          case OPT_BIN_TIME:
            fprintf(fh, "%s%d\n", appHelp[i], DEFAULT_TIME_BIN);
            break;
          default:
            /* Simple help text from the appHelp array */
            fprintf(fh, "%s\n", appHelp[i]);
            break;
        }
    }

    skIPv6PolicyUsage(fh);
    skOptionsTempDirUsage(fh);
    sksiteOptionsUsage(fh);
    skPluginOptionsUsage(fh);
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
void appSetup(int argc, char **argv)
{
    int using_pager = 0;
    int j;
    int i;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    output.pathname = "stdout";
    output.fp = stdout;
    memset(&app_flags, 0, sizeof(app_flags));

    /* initialize plugin library */
    skPluginSetup(2, SKPLUGIN_APP_UNIQ_FIELD, SKPLUGIN_APP_UNIQ_VALUE);

    /* allow for 0 input and 0 output pipes */
    ioISP = iochecksSetup(1, 0, argc, argv);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsTempDirRegister(&temp_directory)
        || skIPv6PolicyOptionsRegister(&ipv6_policy)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        appExit(EXIT_FAILURE);
    }

    /* try to load hard-coded plugins */
    for (j = 0; app_static_plugins[j].name; ++j) {
        skPluginAddAsPlugin(app_static_plugins[j].name,
                            app_static_plugins[j].setup_fn);
    }
    for (j = 0; app_plugin_names[j]; ++j) {
        skPluginLoadPlugin(app_plugin_names[j], 0);
    }

    /* parse options */
    ioISP->firstFile = skOptionsParse(argc, argv);
    if (ioISP->firstFile < 0) {
        skAppUsage();/* never returns */
    }

    /* try to load site config file; if it fails, we will not be able
     * to resolve flowtype and sensor from input file names, but we
     * should not consider it a complete failure */
    sksiteConfigure(0);

    /* create the ascii stream and set its properties */
    if (rwAsciiStreamCreate(&ascii_str)) {
        skAppPrintErr("Unable to create ascii stream");
        appExit(EXIT_FAILURE);
    }
    rwAsciiSetDelimiter(ascii_str, delimiter);
    rwAsciiSetIPv6Policy(ascii_str, ipv6_policy);
    rwAsciiSetTimestampFlags(ascii_str, time_flags);
    if (app_flags.no_final_delimiter) {
        rwAsciiSetNoFinalDelimiter(ascii_str);
    }
    if (app_flags.no_titles) {
        rwAsciiSetNoTitles(ascii_str);
    }
    if (app_flags.no_columns) {
        rwAsciiSetNoColumns(ascii_str);
    }
    if (app_flags.integer_sensors) {
        rwAsciiSetIntegerSensors(ascii_str);
    }

    /* conflicting properties */
    if (app_flags.zero_pad_ips && !app_flags.integer_ips) {
        rwAsciiSetZeroPadIps(ascii_str);
    }
    if (app_flags.integer_ips) {
        if (app_flags.zero_pad_ips) {
            skAppPrintErr("--integer-ips option overrides"
                          " --zero-pad-ips\n"
                          "\tWill print IPs as integers.");
        }
        rwAsciiSetIntegerIps(ascii_str);
    }

    /* set up the key_field_map and value_field_map */
    if (createStringmaps()) {
        appExit(EXIT_FAILURE);
    }

    /* make sure the user specified at least one key field */
    if (fields_arg == NULL || fields_arg[0] == '\0') {
        skAppPrintErr("The --%s switch is required",
                      appOptions[OPT_FIELDS].name);
        skAppUsage();         /* never returns */
    }

    /* parse the --fields and --values switches */
    if (parseKeyFields(fields_arg)) {
        appExit(EXIT_FAILURE);
    }
    if (parseValueFields(values_arg)) {
        appExit(EXIT_FAILURE);
    }

    /* Use STDIN as an input stream if it is not a TTY; make certain
     * we have some input and we are either reading from STDIN or
     * using files listed the command line, but not both. */
    if (iochecksAcceptFromStdin(ioISP)
        || iochecksInputs(ioISP, 0))
    {
        skAppUsage();
    }

    /* create and initialize the uniq object */
    if (app_flags.presorted_input) {
        if (skPresortedUniqueCreate(&ps_uniq)) {
            appExit(EXIT_FAILURE);
        }

        skPresortedUniqueSetTempDirectory(ps_uniq, temp_directory);

        if (skPresortedUniqueSetFields(ps_uniq, key_fields, distinct_fields,
                                       value_fields))
        {
            skAppPrintErr("Unable to set fields");
            appExit(EXIT_FAILURE);
        }

        for (i = ioISP->firstFile; i < ioISP->fileCount; ++i) {
            skPresortedUniqueAddInputFile(ps_uniq, ioISP->fnArray[i]);
        }

        skPresortedUniqueSetPostOpenFn(ps_uniq, prepareFileForRead);
        if (time_bin_size > 1) {
            skPresortedUniqueSetReadFn(ps_uniq, readRecord);
        }

    } else {
        if (skUniqueCreate(&uniq)) {
            appExit(EXIT_FAILURE);
        }
        if (app_flags.sort_output) {
            skUniqueSetSortedOutput(uniq);
        }

        skUniqueSetTempDirectory(uniq, temp_directory);

        if (skUniqueSetFields(uniq, key_fields, distinct_fields, value_fields)
            || skUniquePrepareForInput(uniq, skAppPrintErr))
        {
            skAppPrintErr("Unable to set fields");
            appExit(EXIT_FAILURE);
        }
    }

    /* final check. See if stdout is being used for both --copy-input
     * and as the destination for the uniq data.  */
    if (ioISP->stdoutUsed && output.fp == stdout) {
        skAppPrintErr("stdout used for both --copy-input and ascii output");
        appExit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appExit(EXIT_FAILURE);
    }

    /* invoke the pager */
    using_pager = skOpenPagerWhenStdoutTty(&(output.fp), &pager);
    if (using_pager < 0) {
        appExit(EXIT_FAILURE);
    }
    if (using_pager) {
        output.pathname = pager;
        output.ispipe = 1;
    }

    /* bind the Ascii Stream to the output */
    rwAsciiSetOutputHandle(ascii_str, output.fp);

    /* looks good, open the --copy-input destination */
    if (iochecksOpenCopyDest(ioISP)) {
        appExit(EXIT_FAILURE);
    }

    /* set signal handler to clean up temp files on SIGINT, SIGTERM, etc */
    if (skAppSetSignalHandler(&appHandleSignal)) {
        appExit(EXIT_FAILURE);
    }

    return;                       /* OK */
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    skUniqueDestroy(&uniq);
    skPresortedUniqueDestroy(&ps_uniq);

    /* destroy field lists */
    skFieldListDestroy(&key_fields);
    skFieldListDestroy(&distinct_fields);
    skFieldListDestroy(&value_fields);

    iochecksTeardown(ioISP);

    /* plugin teardown */
    skPluginRunCleanup(SKPLUGIN_FN_ANY);
    skPluginTeardown();

    /* destroy output */
    rwAsciiStreamDestroy(&ascii_str);

    /* close output */
    if (output.fp != stdout) {
        if (output.ispipe) {
            pclose(output.fp);
        } else {
            if (EOF == fclose(output.fp) && !caught_signal) {
                skAppPrintErr("Error closing output file '%s': %s",
                              output.pathname, strerror(errno));
            }
        }
    }

    /* destroy string maps for keys and values */
    if (key_field_map) {
        skStringMapDestroy(key_field_map);
        key_field_map = NULL;
    }
    if (value_field_map) {
        skStringMapDestroy(value_field_map);
        value_field_map = NULL;
    }

    skAppUnregister();
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Called by skOptionsParse(), this handles a user-specified switch
 *    that the application has registered, typically by setting global
 *    variables.  Returns 1 if the switch processing failed or 0 if it
 *    succeeded.  Returning a non-zero from from the handler causes
 *    skOptionsParse() to return a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    uint32_t val32;
    size_t i;
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_FIELDS:
        if (fields_arg) {
            skAppPrintErr("The --%s switch was specified multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        fields_arg = opt_arg;
        break;

      case OPT_VALUES:
        if (values_arg) {
            skAppPrintErr("The --%s switch was specified multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        values_arg = opt_arg;
        break;

      case OPT_ALL_COUNTS:
        for (i = 0; i < num_builtin_values; ++i) {
            if (builtin_values[i].bf_all_counts) {
                builtin_values[i].bf_switched_on = 1;
            }
        }
        break;

      case OPT_BYTES:
      case OPT_PACKETS:
      case OPT_FLOWS:
      case OPT_STIME:
      case OPT_ETIME:
      case OPT_SIP_DISTINCT:
      case OPT_DIP_DISTINCT:
        i = opt_index - OPT_BYTES;
        builtin_values[i].bf_switched_on = 1;
        if (opt_arg) {
            rv = skStringParseRange64(&builtin_values[i].bf_min,
                                      &builtin_values[i].bf_max,
                                      opt_arg, 0, 0,
                                      SKUTILS_RANGE_SINGLE_OPEN);
            if (rv) {
                goto PARSE_ERROR;
            }
            /* treat a single value as having no max, not as a range
             * of a single value */
            if ((builtin_values[i].bf_min == builtin_values[i].bf_max)
                && !strchr(opt_arg, '-'))
            {
                builtin_values[i].bf_max = UINT64_MAX;
            }
            app_flags.check_limits = 1;
        }
        break;

      case OPT_PLUGIN:
        if (skPluginLoadPluginOrDynlib(opt_arg, DYNLIB_UNIQ, 1) != 0) {
            skAppPrintErr("Unable to load %s as a plugin", opt_arg);
            return 1;
        }
        break;

      case OPT_BIN_TIME:
        if (opt_arg == NULL || opt_arg[0] == '\0') {
            /* no time given; use default */
            time_bin_size = sktimeCreate(DEFAULT_TIME_BIN, 0);
        } else {
            /* parse user's time */
            rv = skStringParseUint32(&val32, opt_arg, 1, 0);
            if (rv) {
                goto PARSE_ERROR;
            }
            time_bin_size = sktimeCreate(val32, 0);
        }
        break;

      case OPT_PRESORTED_INPUT:
        app_flags.presorted_input = 1;
        break;

      case OPT_SORT_OUTPUT:
        app_flags.sort_output = 1;
        break;

      case OPT_EPOCH_TIME:
        if (time_flags & SKTIMESTAMP_MMDDYYYY) {
            skAppPrintErr("Both --%s and --%s specified. Using --%s",
                          appOptions[OPT_EPOCH_TIME].name,
                          appOptions[OPT_LEGACY_TIMESTAMPS].name,
                          appOptions[OPT_EPOCH_TIME].name);
        }
        time_flags = (time_flags & ~SKTIMESTAMP_MMDDYYYY) | SKTIMESTAMP_EPOCH;

        /* Reduce width of the textual columns for the MIN_STARTTIME
         * and MAX_ENDTIME fields. */
        for (i = 0; i < num_builtin_values; ++i) {
            if ((builtin_values[i].bf_id == SK_FIELD_MIN_STARTTIME)
                || (builtin_values[i].bf_id == SK_FIELD_MAX_ENDTIME))
            {
                builtin_values[i].bf_text_len = 10;
            }
        }
        break;

      case OPT_INTEGER_IPS:
        app_flags.integer_ips = 1;
        break;

      case OPT_ZERO_PAD_IPS:
        app_flags.zero_pad_ips = 1;
        break;

      case OPT_INTEGER_SENSORS:
        app_flags.integer_sensors = 1;
        break;

      case OPT_NO_TITLES:
        app_flags.no_titles = 1;
        break;

      case OPT_NO_COLUMNS:
        app_flags.no_columns = 1;
        break;

      case OPT_NO_FINAL_DELIMITER:
        app_flags.no_final_delimiter = 1;
        break;

      case OPT_COLUMN_SEPARATOR:
        delimiter = opt_arg[0];
        break;

      case OPT_DELIMITED:
        app_flags.no_columns = 1;
        app_flags.no_final_delimiter = 1;
        if (opt_arg) {
            delimiter = opt_arg[0];
        }
        break;

      case OPT_PRINT_FILENAMES:
        app_flags.print_filenames = 1;
        break;

      case OPT_COPY_INPUT:
        if (iochecksAllDestinations(ioISP, opt_arg)) {
            return 1;
        }
        break;

      case OPT_OUTPUT_PATH:
        output.pathname = opt_arg;
        if (skOpenFile(output.pathname, 1 /* write */, &output.fp,
                       &output.ispipe))
        {
            return 1;
        }
        break;

      case OPT_PAGER:
        pager = opt_arg;
        break;

      case OPT_LEGACY_TIMESTAMPS:
        if ((opt_arg == NULL) || (opt_arg[0] == '\0') || (opt_arg[0] == '1')) {
            if (time_flags & SKTIMESTAMP_EPOCH) {
                skAppPrintErr("Both --%s and --%s specified; using --%s",
                              appOptions[OPT_EPOCH_TIME].name,
                              appOptions[OPT_LEGACY_TIMESTAMPS].name,
                              appOptions[OPT_EPOCH_TIME].name);
                /* no change */
            } else {
                time_flags |= SKTIMESTAMP_MMDDYYYY;
            }
        } else if (opt_arg[0] != '0') {
            skAppPrintErr("Invalid %s '%s': Select 0 or 1",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;
    }

    return 0;                     /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 *  appExit(status)
 *
 *  Exit the application with the given status.
 */
void appExit(int status)
{
    appTeardown();
    exit(status);
}


/*
 *  appHandleSignal(signal_value)
 *
 *    Call appExit() to exit the program.  If signal_value is SIGPIPE,
 *    close cleanly; otherwise print a message that we've caught the
 *    signal and exit with EXIT_FAILURE.
 */
static void appHandleSignal(int sig)
{
    caught_signal = 1;

    if (sig == SIGPIPE) {
        /* we get SIGPIPE if something downstream, like rwcut, exits
         * early, so don't bother to print a warning, and exit
         * successfully */
        appExit(EXIT_SUCCESS);
    } else {
        skAppPrintErr("Caught signal..cleaning up and exiting");
        appExit(EXIT_FAILURE);
    }
}


/*
 *  builtin_get_title(buf, bufsize, field_entry);
 *
 *    Invoked by rwAsciiPrintTitles() to get the title for a built-in
 *    aggregate value or distinct field.
 *
 *    Fill 'buf' with the title for the column represented by the
 *    field list entry 'field_entry'.  This function should write no
 *    more than 'bufsize' characters to 'buf'.
 */
static void builtin_get_title(
    char        *text_buf,
    size_t       text_buf_size,
    void        *v_fl_entry)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
    builtin_field_t *bf;

    bf = (builtin_field_t*)skFieldListEntryGetContext(fl_entry);
    strncpy(text_buf, bf->bf_title, text_buf_size);
}

/*
 *  builtin_value_to_ascii(rwrec, buf, bufsize, field_entry, extra);
 *
 *    Invoked by rwAsciiPrintRecExtra() to get the value for a
 *    built-in aggregate value field.
 *
 *    Fill 'buf' with the value for the column represented by the
 *    aggregate value field list entry 'field_entry'.  'rwrec' is
 *    ignored; 'extra' is an array[3] that contains the buffers for
 *    the key, aggregate value, and distinct field-lists.  This
 *    function should write no more than 'bufsize' characters to
 *    'buf'.
 */
static int builtin_value_to_ascii(
    const rwRec UNUSED(*rwrec),
    char               *text_buf,
    size_t              text_buf_size,
    void               *v_fl_entry,
    void               *v_outbuf)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
    uint64_t val64;
    uint32_t val32;

    switch (skFieldListEntryGetId(fl_entry)) {
      case SK_FIELD_SUM_BYTES:
        skFieldListExtractFromBuffer(value_fields, ((uint8_t**)v_outbuf)[1],
                                     fl_entry, (uint8_t*)&val64);
        snprintf(text_buf, text_buf_size, ("%" PRIu64), val64);
        break;

      case SK_FIELD_RECORDS:
      case SK_FIELD_SUM_PACKETS:
      case SK_FIELD_SUM_ELAPSED:
        skFieldListExtractFromBuffer(value_fields, ((uint8_t**)v_outbuf)[1],
                                     fl_entry, (uint8_t*)&val32);
        snprintf(text_buf, text_buf_size, ("%" PRIu32), val32);
        break;

      case SK_FIELD_MIN_STARTTIME:
      case SK_FIELD_MAX_ENDTIME:
        skFieldListExtractFromBuffer(value_fields, ((uint8_t**)v_outbuf)[1],
                                     fl_entry, (uint8_t*)&val32);
        assert(text_buf_size > SKTIMESTAMP_STRLEN);
        sktimestamp_r(text_buf, sktimeCreate(val32, 0), time_flags);
        break;

      default:
        skAbortBadCase(skFieldListEntryGetId(fl_entry));
    }

    return 0;
}

/*
 *  builtin_distinct_to_ascii(rwrec, buf, bufsize, field_entry, extra);
 *
 *    Invoked by rwAsciiPrintRecExtra() to get the value for a
 *    built-in distinct field.
 *
 *    Like builtin_value_to_ascii(), but works on a distinct field.
 */
static int builtin_distinct_to_ascii(
    const rwRec UNUSED(*rwrec),
    char               *text_buf,
    size_t              text_buf_size,
    void               *v_fl_entry,
    void               *v_outbuf)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
#if SK_ENABLE_IPV6
    ipv6_distinct_t val_ip6;
#endif
    uint32_t val32;

    switch (skFieldListEntryGetId(fl_entry)) {
#if SK_ENABLE_IPV6
      case SK_FIELD_SIPv6:
      case SK_FIELD_DIPv6:
        skFieldListExtractFromBuffer(distinct_fields, ((uint8_t**)v_outbuf)[2],
                                     fl_entry, val_ip6.ip);
        snprintf(text_buf, text_buf_size, ("%" PRIu64), val_ip6.count);
        break;
#endif  /* SK_ENABLE_IPV6 */

      case SK_FIELD_SIPv4:
      case SK_FIELD_DIPv4:
        skFieldListExtractFromBuffer(distinct_fields, ((uint8_t**)v_outbuf)[2],
                                     fl_entry, (uint8_t*)&val32);
        snprintf(text_buf, text_buf_size, ("%" PRIu32), val32);
        break;

      default:
        skAbortBadCase(skFieldListEntryGetId(fl_entry));
    }

    return 0;
}

/*
 *  plugin_get_title(buf, buf_size, field_entry);
 *
 *    Invoked by rwAsciiPrintTitles() to get the title for a key or
 *    aggregate value field defined by a plug-in.
 *
 *    Fill 'buf' with the title for the column represented by the
 *    distinct field list entry 'field_entry'.  'rwrec' is ignored;
 *    'extra' is an array[3] that contains the buffers for the key,
 *    aggregate value, and distinct field-lists.  This function should
 *    write no more than 'bufsize' characters to 'buf'.
 */
static void plugin_get_title(
    char        *text_buf,
    size_t       text_buf_size,
    void        *v_fl_entry)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
    skplugin_err_t err;
    const char *title;

    err = skPluginFieldTitle(skFieldListEntryGetContext(fl_entry), &title);
    assert(SKPLUGIN_OK == err);
    strncpy(text_buf, title, text_buf_size);
    text_buf[text_buf_size-1] = '\0';
}

/*
 *  plugin_key_to_ascii(rwrec, buf, buf_size, keyfield, extra);
 *
 *    Invoked by rwAsciiPrintRecExtra() to get the value for a key
 *    field that is defined by a plug-in.
 *
 *    Fill 'buf' with the title for the column represented by the
 *    distinct field list entry 'field_entry'.  'rwrec' is ignored;
 *    'extra' is an array[3] that contains the buffers for the key,
 *    aggregate value, and distinct field-lists.  This function should
 *    write no more than 'bufsize' characters to 'buf'.
 */
static int plugin_key_to_ascii(
    const rwRec UNUSED(*rwrec),
    char               *text_buf,
    size_t              text_buf_size,
    void               *v_fl_entry,
    void               *v_outbuf)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
    uint8_t bin_buf[HASHLIB_MAX_KEY_WIDTH];

    /* get the binary value from the field-list */
    skFieldListExtractFromBuffer(key_fields, ((uint8_t**)v_outbuf)[0],
                                 fl_entry, bin_buf);

    /* call the plug-in to convert from binary to text */
    skPluginFieldRunBinToTextFn(skFieldListEntryGetContext(fl_entry),
                                text_buf, text_buf_size, bin_buf);

    return 0;
}

/*
 *  plugin_value_to_ascii(rwrec, buf, buf_size, keyfield, extra);
 *
 *    Invoked by rwAsciiPrintRecExtra() to get the value for an
 *    aggregate value field that is defined by a plug-in.
 *
 *    Like plugin_key_to_ascii(), but works on an aggregate value
 *    field.
 */
static int plugin_value_to_ascii(
    const rwRec UNUSED(*rwrec),
    char               *text_buf,
    size_t              text_buf_size,
    void               *v_fl_entry,
    void               *v_outbuf)
{
    sk_fieldentry_t *fl_entry = (sk_fieldentry_t*)v_fl_entry;
    uint8_t bin_buf[HASHLIB_MAX_VALUE_WIDTH];

    /* get the binary value from the field-list */
    skFieldListExtractFromBuffer(value_fields, ((uint8_t**)v_outbuf)[1],
                                 fl_entry, bin_buf);

    /* call the plug-in to convert from binary to text */
    skPluginFieldRunBinToTextFn(skFieldListEntryGetContext(fl_entry),
                                text_buf, text_buf_size, bin_buf);

    return 0;
}

/*
 *  plugin_rec_to_bin(rwrec, out_buf, plugin_field);
 *
 *    Invoked by skFieldListRecToBinary() to get the binary value,
 *    based on the given 'rwrec', for a key field that is defined by a
 *    plug-in.
 *
 *    The size of 'out_buf' was specified when the field was added to
 *    the field-list.
 */
static void plugin_rec_to_bin(
    const rwRec    *rwrec,
    uint8_t        *out_buf,
    void           *v_pi_field)
{
    skPluginFieldRunRecToBinFn((skplugin_field_t*)v_pi_field,
                               out_buf, rwrec, NULL);
}

/*
 *  plugin_rec_to_bin(rwrec, in_out_buf, plugin_field);
 *
 *    Invoked by skFieldListAddRecToBinary() to get the binary value,
 *    based on the given 'rwrec' and merge that with the current
 *    binary value for a key field that is defined by a plug-in.
 *
 *    The size of 'out_buf' was specified when the field was added to
 *    the field-list.
 */
static void plugin_add_rec_to_bin(
    const rwRec    *rwrec,
    uint8_t        *in_out_buf,
    void           *v_pi_field)
{
    /* FIXME---should be AddRecToBin */
    skPluginFieldRunRecToBinFn((skplugin_field_t*)v_pi_field,
                               in_out_buf, rwrec, NULL);
}

/*
 *  plugin_bin_compare(buf1, buf2, plugin_field);
 *
 *    Invoked by skFieldListCompareBuffers() to compare current value
 *    of the key or aggregate value fields specified by 'buf1' and
 *    'buf2'.
 *
 *    The size of 'buf1' and 'buf2' were specified when the field was
 *    added to the field-list.
 */
static int  plugin_bin_compare(
    const uint8_t  *buf1,
    const uint8_t  *buf2,
    void           *v_pi_field)
{
    int val = 0;
    skPluginFieldRunBinCompareFn((skplugin_field_t*)v_pi_field,
                                 &val, buf1, buf2);
    return val;
}

/*
 *  plugin_bin_merge(in_out_buf, in_buf, plugin_field);
 *
 *    Invoked by skFieldListMergeBuffers() to merge the current values
 *    of the key or aggregate value fields specified by 'in_out_buf'
 *    and 'in_buf'.  The merged value should be placed into
 *    'in_out_buf'.
 *
 *    The size of 'in_out_buf' and 'in_buf' were specified when the
 *    field was added to the field-list.
 */
static void plugin_bin_merge(
    uint8_t        *in_out_buf,
    const uint8_t  *in_buf,
    void           *v_pi_field)
{
    skPluginFieldRunBinMergeFn((skplugin_field_t*)v_pi_field,
                               in_out_buf, in_buf);
}


/*
 *  ok = createStringmaps();
 *
 *    Create the string-maps to assist in parsing the --fields and
 *    --values switches.
 */
static int createStringmaps(void)
{
    skplugin_field_iter_t  iter;
    skplugin_err_t         err;
    skplugin_field_t      *pi_field;
    sk_stringmap_status_t  map_err;
    sk_stringmap_entry_t   entry;
    const char           **field_names;
    const char           **name;
    uint32_t               max_id;
    size_t                 i;

    /* initialize string-map of field identifiers: add default fields,
     * then remove millisec fields, since unique-ing over them makes
     * little sense.
     *
     * Note that although we remove the MSEC fields from the available
     * fields here, the remainder of the code still supports MSEC
     * fields---which are mapped onto the non-MSEC versions of the
     * fields. */
    if (rwAsciiFieldMapAddDefaultFields(&key_field_map)) {
        skAppPrintErr("Unable to setup fields stringmap");
        return -1;
    }
    (void)skStringMapRemoveByID(key_field_map, RWREC_FIELD_STIME_MSEC);
    (void)skStringMapRemoveByID(key_field_map, RWREC_FIELD_ETIME_MSEC);
    (void)skStringMapRemoveByID(key_field_map, RWREC_FIELD_ELAPSED_MSEC);
    max_id = RWREC_PRINTABLE_FIELD_COUNT - 1;

    /* add --fields from the plug-ins */
    err = skPluginFieldIteratorBind(&iter, SKPLUGIN_APP_UNIQ_FIELD, 1);
    assert(err == SKPLUGIN_OK);

    while (skPluginFieldIteratorNext(&iter, &pi_field)) {
        err = skPluginFieldName(pi_field, &field_names);
        assert(err == SKPLUGIN_OK);
        ++max_id;

        /* Add keys to the key_field_map */
        for (name = field_names; *name; name++) {
            memset(&entry, 0, sizeof(entry));
            entry.name = *name;
            entry.id = max_id;
            entry.userdata = pi_field;
            map_err = skStringMapAddEntries(key_field_map, 1, &entry);
            if (map_err != SKSTRINGMAP_OK) {
                const char *plugin_name;
                skPluginFieldGetPluginName(pi_field, &plugin_name);
                skAppPrintErr(("Plug-in cannot add field named '%s': %s."
                               " Plug-in file: %s"),
                              *name, skStringMapStrerror(map_err),plugin_name);
                return -1;
            }
        }
    }


    max_id = 0;

    /* create the string-map for value field identifiers */
    if (skStringMapCreate(&value_field_map)) {
        skAppPrintErr("Unable to create map for values");
        return -1;
    }

    /* add the built-in names */
    for (i = 0; i < num_builtin_values; ++i) {
        memset(&entry, 0, sizeof(sk_stringmap_entry_t));
        entry.name = builtin_values[i].bf_title;
        entry.id = i;
        map_err = skStringMapAddEntries(value_field_map, 1, &entry);
        if (map_err) {
            skAppPrintErr("Unable to add value field named '%s': %s",
                          entry.name, skStringMapStrerror(map_err));
            return -1;
        }
        if (entry.id > max_id) {
            max_id = entry.id;
        }
    }

    /* add the value fields from the plugins */
    err = skPluginFieldIteratorBind(&iter, SKPLUGIN_APP_UNIQ_VALUE, 1);
    assert(err == SKPLUGIN_OK);

    while (skPluginFieldIteratorNext(&iter, &pi_field)) {
        err = skPluginFieldName(pi_field, &field_names);
        assert(err == SKPLUGIN_OK);
        ++max_id;

        /* Add value names to the field_map */
        for (name = field_names; *name; ++name) {
            memset(&entry, 0, sizeof(entry));
            entry.name = *name;
            entry.id = max_id;
            entry.userdata = pi_field;
            map_err = skStringMapAddEntries(value_field_map, 1, &entry);
            if (map_err != SKSTRINGMAP_OK) {
                const char *plugin_name;
                skPluginFieldGetPluginName(pi_field, &plugin_name);
                skAppPrintErr(("Plug-in cannot add value named '%s': %s."
                               " Plug-in file: %s"),
                              *name, skStringMapStrerror(map_err),plugin_name);
                return -1;
            }
        }
    }

    return 0;
}


/*
 *  status = parseKeyFields(field_string);
 *
 *    Parse the string that represents the key fields the user wishes
 *    to "uniq" over, create and fill in the global sk_fieldlist_t
 *    'key_fields', and add columns to the rwAsciiStream.  Return 0 on
 *    success or non-zero on error.
 */
static int parseKeyFields(const char *field_string)
{
    sk_vector_t *parsed_vec = NULL;
    sk_stringmap_entry_t **map_entry;
    sk_fieldentry_t *fl_entry;
    uint32_t field_count;
    uint32_t i;

    /* return value; assume failure */
    int rv = -1;

    /* error message generated when parsing fields */
    char *errmsg;

    /* the field IDs for SIP, DIP, and NHIP depend on the ipv6 policy */
    sk_fieldid_t ip_fields[3] = {
        SK_FIELD_SIPv4, SK_FIELD_DIPv4, SK_FIELD_NHIPv4
    };

#if SK_ENABLE_IPV6
    if (ipv6_policy >= SK_IPV6POLICY_MIX) {
        ip_fields[0] = SK_FIELD_SIPv6;
        ip_fields[1] = SK_FIELD_DIPv6;
        ip_fields[2] = SK_FIELD_NHIPv6;
    }
#endif  /* SK_ENABLE_IPV6 */

    /* vector to hold the parsed fields */
    parsed_vec = skVectorNew(sizeof(sk_stringmap_entry_t*));
    if (NULL == parsed_vec) {
        skAppPrintErr("Out of memory");
        goto END;
    }

    /* convert the --fields argument to a vector */
    if (skStringMapParse(key_field_map, field_string, SKSTRINGMAP_DUPES_ERROR,
                         parsed_vec, &errmsg))
    {
        skAppPrintErr("Invalid %s: %s",
                      appOptions[OPT_FIELDS].name, errmsg);
        goto END;
    }

    field_count = skVectorGetCount(parsed_vec);

    /* create the field-list */
    if (skFieldListCreate(&key_fields)) {
        skAppPrintErr("Unable to create key field list");
        goto END;
    }

    /* see what time fields are requested */
    for (i = 0; i < field_count; ++i) {
        map_entry = skVectorGetValuePointer(parsed_vec, i);
        switch ((*map_entry)->id) {
          case RWREC_FIELD_STIME:
          case RWREC_FIELD_STIME_MSEC:
            time_fields |= PARSE_KEY_STIME;
            break;
          case RWREC_FIELD_ELAPSED:
          case RWREC_FIELD_ELAPSED_MSEC:
            time_fields |= PARSE_KEY_ELAPSED;
            break;
          case RWREC_FIELD_ETIME:
          case RWREC_FIELD_ETIME_MSEC:
            time_fields |= PARSE_KEY_ETIME;
            break;
          default:
            break;
        }
    }

    /* determine how to handle time when we are binning */
    if (time_bin_size != 0) {
        if (time_fields < 2) {
            if (FILEIsATty(stderr)) {
                skAppPrintErr("No time fields in key; %s switch ignored",
                              appOptions[OPT_BIN_TIME].name);
            }
        } else if (time_fields == 7) {
            /* must adjust elapsed to be eTime-sTime */
            adjust_elapsed = 1;
            if (FILEIsATty(stderr)) {
                skAppPrintErr("Warning: modifying duration field "
                              "to be eTime-sTime");
            }
        }
    }

    /* add the key fields to the field-list and to the ascii stream. */
    for (i = 0; i < field_count; ++i) {
        map_entry = skVectorGetValuePointer(parsed_vec, i);
        if ((*map_entry)->userdata) {
            if (appAddPlugin((skplugin_field_t*)((*map_entry)->userdata), 0)) {
                skAppPrintErr("Cannot add key field %s from plugin",
                              (*map_entry)->name);
                goto END;
            }
            continue;
        }
        assert((*map_entry)->id < RWREC_PRINTABLE_FIELD_COUNT);
        switch ((rwrec_printable_fields_t)(*map_entry)->id) {
          case RWREC_FIELD_SIP:
            fl_entry = skFieldListAddKnownField(key_fields, ip_fields[0],NULL);
            break;
          case RWREC_FIELD_DIP:
            fl_entry = skFieldListAddKnownField(key_fields, ip_fields[1],NULL);
            break;
          case RWREC_FIELD_NHIP:
            fl_entry = skFieldListAddKnownField(key_fields, ip_fields[2],NULL);
            break;
          default:
            fl_entry = skFieldListAddKnownField(key_fields, (*map_entry)->id,
                                                NULL);
            break;
        }
        if (NULL == fl_entry) {
            skAppPrintErr("Cannot add key field %s to field list",
                          (*map_entry)->name);
            goto END;
        }
        if (rwAsciiAppendOneField(ascii_str, (*map_entry)->id)) {
            skAppPrintErr("Cannot add key field %s to stream",
                          (*map_entry)->name);
            goto END;
        }
    }

    /* successful */
    rv = 0;

  END:
    if (rv != 0) {
        /* something went wrong.  clean up */
        if (key_fields) {
            skFieldListDestroy(&key_fields);
            key_fields = NULL;
        }
    }
    /* do standard clean-up */
    if (parsed_vec != NULL) {
        skVectorDestroy(parsed_vec);
    }

    return rv;
}


/*
 *  ok = parseValueFields(field_string);
 *
 *    Parse the string that represents the aggregate value and
 *    distinct fields the user wishes to compute, create and fill in
 *    the global sk_fieldlist_t 'value_fields' and 'distinct_fields',
 *    and add columns to the rwAsciiStream.  Return 0 on success or
 *    non-zero on error.
 *
 *    Returns 0 on success, or non-zero on error.
 */
static int parseValueFields(const char *value_string)
{
    sk_vector_t *parsed_vec = NULL;
    sk_stringmap_entry_t **map_entry;
    sk_stringmap_entry_t *one_entry;
    uint32_t field_count;
    size_t i;

    /* return value; assume failure */
    int rv = -1;

    /* error message generated when parsing fields */
    char *errmsg;

    builtin_field_t *bf;
    sk_fieldentry_t *fl_entry;

#if SK_ENABLE_IPV6
    if (ipv6_policy >= SK_IPV6POLICY_MIX) {
        /* change the field id of the distinct fields */
        for (i = 0, bf = builtin_values; i < num_builtin_values; ++i, ++bf) {
            switch (bf->bf_id) {
              case SK_FIELD_SIPv4:
                bf->bf_id = SK_FIELD_SIPv6;
                break;
              case SK_FIELD_DIPv4:
                bf->bf_id = SK_FIELD_DIPv6;
                break;
              default:
                break;
            }
        }
    }
#endif  /* SK_ENABLE_IPV6 */

    /* vector to hold the parsed fields */
    parsed_vec = skVectorNew(sizeof(sk_stringmap_entry_t*));
    if (NULL == parsed_vec) {
        skAppPrintErr("Out of memory");
        goto END;
    }

    /* parse the --values field list if given */
    if (value_string) {
        if (skStringMapParse(value_field_map, value_string,
                             SKSTRINGMAP_DUPES_ERROR, parsed_vec, &errmsg))
        {
            skAppPrintErr("Invalid %s: %s",
                          appOptions[OPT_VALUES].name, errmsg);
            goto END;
        }
    }

    /* turn off the --bytes,--packets,etc switches if they also appear
     * in the --values switch (i.e., they are already in parsed_vec) */
    for (i = 0;
         (map_entry = skVectorGetValuePointer(parsed_vec, i)) != NULL;
         ++i)
    {
        if ((*map_entry)->id < num_builtin_values) {
            builtin_values[(*map_entry)->id].bf_switched_on = 0;
        }
    }

    /* add active --bytes,--packets,etc switches to the vector */
    for (i = 0, bf = builtin_values; i < num_builtin_values; ++i, ++bf) {
        if (bf->bf_switched_on) {
            skStringMapGetByName(value_field_map, bf->bf_title, &one_entry);
            skVectorAppendValue(parsed_vec, &one_entry);
        }
    }

    field_count = skVectorGetCount(parsed_vec);

    /* if the field_count is empty, count records */
    if (0 == field_count) {
        for (i = 0, bf = builtin_values; i < num_builtin_values; ++i, ++bf) {
            if (SK_FIELD_RECORDS == bf->bf_id) {
                skStringMapGetByName(value_field_map, bf->bf_title,&one_entry);
                skVectorAppendValue(parsed_vec, &one_entry);
                field_count = skVectorGetCount(parsed_vec);
                break;
            }
        }
    }

    /* create the field-lists */
    if (skFieldListCreate(&value_fields)) {
        skAppPrintErr("Unable to create value field list");
        goto END;
    }
    if (skFieldListCreate(&distinct_fields)) {
        skAppPrintErr("Unable to create distinct field list");
        goto END;
    }

    /* loop over the selected values */
    for (i = 0; i < field_count; ++i) {
        map_entry = skVectorGetValuePointer(parsed_vec, i);
        if ((*map_entry)->userdata) {
            /* field comes from a plug-in */
            if (appAddPlugin((skplugin_field_t*)((*map_entry)->userdata), 1)) {
                skAppPrintErr("Cannot add value field %s from plugin",
                              (*map_entry)->name);
                goto END;
            }
            continue;
        }
        assert((*map_entry)->id < num_builtin_values);
        bf = &builtin_values[(*map_entry)->id];
        if (bf->bf_is_distinct) {
            fl_entry = skFieldListAddKnownField(distinct_fields, bf->bf_id,bf);
            if (NULL == fl_entry) {
                skAppPrintErr("Cannot add distinct field %s to field list",
                              (*map_entry)->name);
                goto END;
            }
            if (rwAsciiAppendCallbackFieldExtra(ascii_str, &builtin_get_title,
                                                &builtin_distinct_to_ascii,
                                                fl_entry, bf->bf_text_len))
            {
                skAppPrintErr("Cannot add distinct field %s to stream",
                              (*map_entry)->name);
                goto END;
            }
        } else {
            fl_entry = skFieldListAddKnownField(value_fields, bf->bf_id, bf);
            if (NULL == fl_entry) {
                skAppPrintErr("Cannot add value field %s to field list",
                              (*map_entry)->name);
                goto END;
            }
            if (rwAsciiAppendCallbackFieldExtra(ascii_str, &builtin_get_title,
                                                &builtin_value_to_ascii,
                                                fl_entry, bf->bf_text_len))
            {
                skAppPrintErr("Cannot add value field %s to stream",
                              (*map_entry)->name);
                goto END;
            }
        }
    }

    rv = 0;

  END:
    /* do standard clean-up */
    if (parsed_vec != NULL) {
        skVectorDestroy(parsed_vec);
    }
    if (rv != 0) {
        /* something went wrong. do additional clean-up */
        if (value_fields) {
            skFieldListDestroy(&value_fields);
            value_fields = NULL;
        }
        if (distinct_fields) {
            skFieldListDestroy(&distinct_fields);
            distinct_fields = NULL;
        }
    }

    return rv;
}


/*
 *  status = appAddPlugin(plugin_field, key_or_value);
 *
 *    Given a key or an aggregate value field defined in a plug-in,
 *    activate that field and get the information from the field that
 *    rwuniq requires.  'key_or_value' indicates whether the field
 *    represents a key field (0) or an aggregate value field (1).
 *
 *    The function adds the field to the approprirate sk_fieldlist_t
 *    ('key_fields' or 'value_fields') and to the rwAsciiStream.
 */
static int appAddPlugin(
    skplugin_field_t   *pi_field,
    int                 key_or_value)
{
    uint8_t bin_buf[HASHLIB_MAX_VALUE_WIDTH];
    rwAsciiStreamGetValueExtra_t bin_to_ascii;
    sk_fieldlist_entrydata_t regdata;
    sk_fieldentry_t *fl_entry;
    size_t text_width;
    skplugin_err_t err;

    /* set the regdata for the sk_fieldlist_t */
    memset(&regdata, 0, sizeof(regdata));
    regdata.bin_compare = plugin_bin_compare;
    regdata.add_rec_to_bin = plugin_add_rec_to_bin;
    regdata.bin_merge = plugin_bin_merge;
    /* regdata.bin_output; */

    if (0 == key_or_value) {
        /* only set this for keys */
        regdata.rec_to_bin = plugin_rec_to_bin;
    }

    /* set a variable holding the rwAsciiStream callback function */
    if (0 == key_or_value) {
        bin_to_ascii = &plugin_key_to_ascii;
    } else {
        bin_to_ascii = &plugin_value_to_ascii;
    }

    /* activate the field (so cleanup knows about it) */
    err = skPluginFieldActivate(pi_field);
    if (err != SKPLUGIN_OK) {
        return -1;
    }

    /* initialize this field */
    err = skPluginFieldRunInitialize(pi_field);
    if (err != SKPLUGIN_OK) {
        return -1;
    }

    /* get the required (textual) width of the column */
    err = skPluginFieldGetLenText(pi_field, &text_width);
    if (err != SKPLUGIN_OK) {
        return -1;
    }

    /* get the bin width for this field */
    err = skPluginFieldGetLenBin(pi_field, &regdata.bin_octets);
    if (err != SKPLUGIN_OK) {
        return -1;
    }
    if (regdata.bin_octets > HASHLIB_MAX_VALUE_WIDTH) {
        return -1;
    }

    memset(bin_buf, 0, sizeof(bin_buf));
    err = skPluginFieldGetInitialValue(pi_field, bin_buf);
    if (err != SKPLUGIN_OK) {
        return -1;
    }
    regdata.initial_value = bin_buf;

    if (0 == key_or_value) {
        fl_entry = skFieldListAddField(key_fields, &regdata, (void*)pi_field);
    } else {
        fl_entry = skFieldListAddField(value_fields, &regdata,(void*)pi_field);
    }
    if (NULL == fl_entry) {
        skAppPrintErr("Unable to add field to field list");
        return -1;
    }

    return rwAsciiAppendCallbackFieldExtra(ascii_str, &plugin_get_title,
                                           bin_to_ascii, fl_entry,
                                           text_width);
}


static int prepareFileForRead(skstream_t *rwios)
{
    if (app_flags.print_filenames) {
        fprintf(PRINT_FILENAMES_FH, "%s\n", skStreamGetPathname(rwios));
    }
    skStreamSetCopyInput(rwios, ioISP->inputCopyFD);
    skStreamSetIPv6Policy(rwios, ipv6_policy);

    return 0;
}


/*
 *  status = readRecord(stream, rwrec);
 *
 *    Fill 'rwrec' with a SiLK Flow record read from 'stream'.  Modify
 *    the times on the record if the user has requested time binning.
 *
 *    Return the status of reading the record.
 */
int readRecord(skstream_t *rwios, rwRec *rwrec)
{
    sktime_t sTime;
    sktime_t sTime_mod;
    uint32_t elapsed;
    int rv;

    rv = skStreamReadRecord(rwios, rwrec);
    if (SKSTREAM_OK == rv) {
        if (time_bin_size > 1) {
            sTime = rwRecGetStartTime(rwrec);
            sTime_mod = sTime % time_bin_size;
            rwRecSetStartTime(rwrec, (sTime - sTime_mod));
            if (adjust_elapsed) {
                /*
                 * the following sets elapsed to:
                 * ((eTime - (eTime % bin_size))
                 *    - (sTime - (sTime % bin_size)))
                 */
                elapsed = rwRecGetElapsed(rwrec);
                elapsed = (elapsed + sTime_mod
                           - ((sTime + elapsed) % time_bin_size));
                rwRecSetElapsed(rwrec, elapsed);
            }
        }
    }

    return rv;
}


/*
 *  int = appNextInput(&rwios);
 *
 *    Fill 'rwios' with the next input file to read.  Return 0 if
 *    'rwios' was successfully opened, 1 if there are no more input
 *    files, or -1 if an error was encountered.
 *
 *    When an input file cannot be opened, the return value is
 *    dependent on the error.  If the error is due to being out of
 *    file handles or memory (EMFILE or ENOMEM), return -2; otherwise
 *    return -1.
 */
int appNextInput(skstream_t **rwios)
{
    static int counter = -1;
    int rv;

    if (counter < 0) {
        counter = ioISP->firstFile;
    } else {
        ++counter;
        if (counter == ioISP->fileCount) {
            /* no more files */
            return 1;
        }
    }

    rv = skStreamOpenSilkFlow(rwios, ioISP->fnArray[counter], SK_IO_READ);
    if (rv) {
        skStreamPrintLastErr(*rwios, rv, &skAppPrintErr);
        skStreamDestroy(rwios);
        return -1;
    }

    (void)prepareFileForRead(*rwios);
    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
