/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Miscellaneous functions for dealing with IP addresses.
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: sku-ips.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>


const uint8_t sk_ipv6_zero[SK_IPV6_ZERO_LEN] =
    {0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0};
const uint8_t sk_ipv6_v4inv6[SK_IPV6_V4INV6_LEN] =
    {0,0,0,0, 0,0,0,0, 0,0,0xFF,0xFF};


#define SILK_IPV6_POLICY_ENVAR  "SILK_IPV6_POLICY"


/* LOCAL VARIABLES */

/* masks of various sizes used when computing CIDR blocks */
static const uint32_t bitmask[] = {
    /*  0- 3 */ 0xffffffff, 0x7fffffff, 0x3fffffff, 0x1fffffff,
    /*  4- 7 */  0xfffffff,  0x7ffffff,  0x3ffffff,  0x1ffffff,
    /*  8-11 */   0xffffff,   0x7fffff,   0x3fffff,   0x1fffff,
    /* 12-15 */    0xfffff,    0x7ffff,    0x3ffff,    0x1ffff,
    /* 16-19 */     0xffff,     0x7fff,     0x3fff,     0x1fff,
    /* 20-23 */      0xfff,      0x7ff,      0x3ff,      0x1ff,
    /* 24-27 */       0xff,       0x7f,       0x3f,       0x1f,
    /* 28-31 */        0xf,        0x7,        0x3,        0x1,
    /* 32    */        0x0
};


/* this is used to find the log-base-2 of the CIDR block difference.
 * http://graphics.stanford.edu/~seander/bithacks.html */
static const uint8_t log_table_256[256] =
{
    /*   0- 15 */  0, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
    /*  16- 31 */  4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    /*  32- 47 */  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    /*  48- 63 */  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    /*  64- 79 */  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    /*  80- 95 */  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    /*  96-111 */  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    /* 112-127 */  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    /* 128-143 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 144-159 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 160-175 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 176-191 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 192-207 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 208-223 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 224-239 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    /* 240-255 */  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
};



/* FUNCTION DEFINITIONS */


/* compute the log2() of 'value' */
int skIntegerLog2(uint64_t value)
{
    uint64_t tmp1;
    uint64_t tmp2;

    /* The comments use 63 for most significant bit and 0 for LSB */

    if ((tmp1 = value >> 32)) {          /* MSB is in bits 63-32 */
        if ((tmp2 = tmp1 >> 16)) {       /* MSB is in bits 63-48 */
            if ((tmp1 = tmp2 >> 8)) {    /* MSB is in bits 63-56 */
                return 56 + log_table_256[tmp1];
            } else {                     /* MSB is in bits 55-48 */
                return 48 + log_table_256[tmp2];
            }
        } else {                         /* MSB is in bits 47-32 */
            if ((tmp2 = tmp1 >> 8)) {    /* MSB is in bits 47-40 */
                return 40 + log_table_256[tmp2];
            } else {                     /* MSB is in bits 39-32 */
                return 32 + log_table_256[tmp1];
            }
        }
    } else {                             /* MSB is in bits 31- 0 */
        if ((tmp1 = value >> 16)) {    /* MSB is in bits 31-16 */
            if ((tmp2 = tmp1 >> 8)) {    /* MSB is in bits 31-24 */
                return 24 + log_table_256[tmp2];
            } else {                     /* MSB is in bits 23-16 */
                return 16 + log_table_256[tmp1];
            }
        } else {                         /* MSB is in bits 15- 0 */
            if ((tmp1 = value >> 8)) { /* MSB is in bits 15- 8 */
                return 8 + log_table_256[tmp1];
            } else {                     /* MSB is in bits  7- 0 */
                return log_table_256[value];
            }
        }
    }
}


/* compute a CIDR block */
int skComputeCIDR(
    uint32_t    start_ip,
    uint32_t    end_ip,
    uint32_t   *new_start_ip)
{
    skipaddr_t start_addr;
    skipaddr_t end_addr;
    skipaddr_t new_start_addr;
    int prefix;

    skipaddrSetV4(&start_addr, &start_ip);
    skipaddrSetV4(&end_addr, &end_ip);
    if (new_start_ip) {
        prefix = skCIDRComputePrefix(&start_addr, &end_addr, &new_start_addr);
        *new_start_ip = skipaddrGetV4(&new_start_addr);
    } else {
        prefix = skCIDRComputePrefix(&start_addr, &end_addr, NULL);
    }

    return prefix;
}


/* compute the prefix to hold 'start_addr' through 'end_addr' */
int skCIDRComputePrefix(
    const skipaddr_t   *start_addr,
    const skipaddr_t   *end_addr,
    skipaddr_t         *new_start_addr)
{
    int prefix = -1;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(start_addr) || skipaddrIsV6(end_addr)) {
        uint8_t start_ip6[16];
        uint8_t end_ip6[16];
        uint8_t range_start[16];
        int i;
        int tmp1;
        uint8_t tmp2 = 0;

        tmp1 = skipaddrCompare(start_addr, end_addr);
        if (tmp1 > 0) {
            /* bad range, start_addr > end_addr */
            return -1;
        }

        /* handle a range that contains a single IP */
        if (tmp1 == 0) {
            if (new_start_addr) {
                skipaddrClear(new_start_addr);
            }
            return 128;
        }

        skipaddrGetAsV6(start_addr, start_ip6);
        skipaddrGetAsV6(end_addr, end_ip6);

        /* handle an odd start_addr */
        if (start_ip6[15] & 0x1) {
            if (new_start_addr) {
                skipaddrCopy(new_start_addr, start_addr);
                skipaddrIncrement(new_start_addr);
            }
            return 128;
        }

        memset(range_start, 0, sizeof(range_start));

        /* find the most significant bit where the values differ */
        for (i = 0; i < 16; ++i) {
            range_start[i] = start_ip6[i];
            if (start_ip6[i] != end_ip6[i]) {
                tmp1 = 1 + end_ip6[i] - start_ip6[i];
                if (256 == tmp1) {
                    tmp2 = 8;
                    prefix = 8 * i;
                } else {
                    tmp2 = log_table_256[tmp1];
                    prefix = 8 * (i + 1) - tmp2;
                    range_start[i] &= (0xFF << tmp2);
                }
                break;
            }
        }

        while (memcmp(range_start, start_ip6, sizeof(range_start)) < 0) {
            ++prefix;
            if (tmp2 != 0) {
                --tmp2;
            } else {
                ++i;
                tmp2 = 7;
            }
            range_start[i] = start_ip6[i] & (0xFF << tmp2);
        }

        if (new_start_addr) {
            /* compute the start of the next CIDR block, which is the
             * IP after the block we just finished.  In the case of
             * roll-over, the 'else' clause will be invoked, and we
             * will return 0. */
            if (tmp2 & 0x07) {
                range_start[i] |= ~(0xFF << tmp2);
            }
            for (++i; i < 16; ++i) {
                range_start[i] = 0xFF;
            }
            if (0 == memcmp(range_start, end_ip6, sizeof(range_start))) {
                skipaddrClear(new_start_addr);
            } else {
                skipaddrSetV6(new_start_addr, range_start);
                skipaddrIncrement(new_start_addr);
            }
        }

        return prefix;

    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        uint32_t start_ip4 = skipaddrGetV4(start_addr);
        uint32_t end_ip4 = skipaddrGetV4(end_addr);
        uint32_t range_start;

        if (end_ip4 < start_ip4) {
            return -1;
        }

        /* handle a range that contains a single IP */
        if (end_ip4 == start_ip4) {
            if (new_start_addr) {
                skipaddrClear(new_start_addr);
            }
            return 32;
        }
        /* handle an odd start_addr */
        if (start_ip4 & 0x1) {
            if (new_start_addr) {
                ++start_ip4;
                skipaddrSetV4(new_start_addr, &start_ip4);
            }
            return 32;
        }

        /* compute the log-base-2 (position of most significant bit)
         * of the number of IPs in the range, and subtract that from
         * 32 to get the widest possible CIDR block */
        prefix = 32 - skIntegerLog2(UINT64_C(1) + end_ip4 - start_ip4);

        /* tighten the range if we need to, in case the IPs don't fall
         * into a single CIDR block (e.g., 10.0.0.6--10.0.0.9) */
        while ((range_start = (start_ip4 & ~(bitmask[prefix]))) < start_ip4) {
            ++prefix;
        }

        /* assert that the CIDR block is within the limits */
        assert(range_start == start_ip4);
        assert((range_start | (bitmask[prefix])) <= end_ip4);

        if (new_start_addr) {
            /* compute the start of the next CIDR block, which is the
             * IP after the block we just finished.  In the case of
             * roll-over, the 'else' clause will be invoked, and we
             * will return 0. */
            start_ip4 = 1 + (range_start | (bitmask[prefix]));
            if (start_ip4 > end_ip4) {
                skipaddrClear(new_start_addr);
            } else {
                skipaddrSetV4(new_start_addr, &start_ip4);
            }
        }
    }

    return prefix;
}


int skCIDR2IPRange(
    const skipaddr_t   *ipaddr,
    uint32_t            cidr,
    skipaddr_t         *min_ip,
    skipaddr_t         *max_ip)
{
    uint32_t ip4;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        uint8_t ip6[16];
        uint32_t i;

        if (cidr == 128) {
            /* don't use skipaddrCopy() in case caller supplied
             * pointers are the same */
            skipaddrGetV6(ipaddr, &ip6);
            skipaddrSetV6(min_ip, &ip6);
            skipaddrSetV6(max_ip, &ip6);
            return 0;
        }

        if (cidr > 128) {
            return -1;
        }

        skipaddrGetV6(ipaddr, &ip6);

        /* first byte of address where there is any effect */
        i = cidr >> 3;

        /* handle max: apply mask to this byte, remaining bits all 1 */
        ip6[i] |= (0xFF >> (cidr & 0x07));
        memset(&ip6[i+1], 0xFF, (15 - i));
        skipaddrSetV6(max_ip, &ip6);

        /* handle min: apply mask, remaining bits all 0 */
        ip6[i] &= ~(0xFF >> (cidr & 0x07));
        memset(&ip6[i+1], 0, (15 - i));
        skipaddrSetV6(min_ip, &ip6);

        return 0;
    }
#endif /* SK_ENABLE_IPV6 */

    if (cidr == 32) {
        ip4 = skipaddrGetV4(ipaddr);
        skipaddrSetV4(min_ip, &ip4);
        skipaddrSetV4(max_ip, &ip4);
        return 0;
    }

    if (cidr > 32) {
        return -1;
    }

    /* handle max IP */
    ip4 = (UINT32_MAX >> cidr) | skipaddrGetV4(ipaddr);
    skipaddrSetV4(max_ip, &ip4);

    /* handle min IP */
    ip4 &= ~(UINT32_MAX >> cidr);
    skipaddrSetV4(min_ip, &ip4);

    return 0;
}


int skCIDRComputeStart(
    const skipaddr_t   *ipaddr,
    uint32_t            cidr,
    skipaddr_t         *min_ip)
{
    uint32_t ip4;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        uint8_t ip6[16];
        uint32_t i;

        if (cidr == 128) {
            if (ipaddr != min_ip) {
                skipaddrCopy(min_ip, ipaddr);
            }
            return 0;
        }

        if (cidr > 128) {
            return -1;
        }

        skipaddrGetV6(ipaddr, &ip6);

        /* first byte of address where there is any effect */
        i = cidr >> 3;

        /* handle min: apply mask to this byte, remaining bits all 0 */
        ip6[i] &= ~(0xFF >> (cidr & 0x07));
        memset(&ip6[i+1], 0, (15 - i));
        skipaddrSetV6(min_ip, &ip6);

        return 0;
    }
#endif /* SK_ENABLE_IPV6 */

    if (cidr == 32) {
        ip4 = skipaddrGetV4(ipaddr);
        skipaddrSetV4(min_ip, &ip4);
        return 0;
    }

    if (cidr > 32) {
        return -1;
    }

    /* handle min IP */
    ip4 = ~(UINT32_MAX >> cidr) & skipaddrGetV4(ipaddr);
    skipaddrSetV4(min_ip, &ip4);

    return 0;
}


int skCIDRComputeEnd(
    const skipaddr_t   *ipaddr,
    uint32_t            cidr,
    skipaddr_t         *max_ip)
{
    uint32_t ip4;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        uint8_t ip6[16];
        uint32_t i;

        if (cidr == 128) {
            if (ipaddr != max_ip) {
                skipaddrCopy(max_ip, ipaddr);
            }
            return 0;
        }

        if (cidr > 128) {
            return -1;
        }

        skipaddrGetV6(ipaddr, &ip6);

        /* first byte of address where there is any effect */
        i = cidr >> 3;

        /* handle max: apply mask to this byte, remaining bits all 1 */
        ip6[i] |= (0xFF >> (cidr & 0x07));
        memset(&ip6[i+1], 0xFF, (15 - i));
        skipaddrSetV6(max_ip, &ip6);

        return 0;
    }
#endif /* SK_ENABLE_IPV6 */

    if (cidr == 32) {
        ip4 = skipaddrGetV4(ipaddr);
        skipaddrSetV4(max_ip, &ip4);
        return 0;
    }

    if (cidr > 32) {
        return -1;
    }

    /* handle max IP */
    ip4 = (UINT32_MAX >> cidr) | skipaddrGetV4(ipaddr);
    skipaddrSetV4(max_ip, &ip4);

    return 0;
}


#if SK_ENABLE_IPV6
int skipaddrV6toV4(const skipaddr_t *srcaddr, skipaddr_t *dstaddr)
{
    uint32_t ipv4;

    if (!SK_IPV6_IS_V4INV6(srcaddr->ip_ip.ipu_ipv6)) {
        return -1;
    }
    memcpy(&ipv4, &(srcaddr->ip_ip.ipu_ipv6[12]), 4);
    ipv4 = ntohl(ipv4);
    skipaddrSetV4(dstaddr, &ipv4);
    return 0;
}


int skipaddrGetAsV4(const skipaddr_t *ipaddr, uint32_t *ipv4)
{
    if (skipaddrIsV6(ipaddr)) {
        if (!SK_IPV6_IS_V4INV6(ipaddr->ip_ip.ipu_ipv6)) {
            return -1;
        }
        memcpy(ipv4, &(ipaddr->ip_ip.ipu_ipv6[12]), 4);
        *ipv4 = ntohl(*ipv4);
    } else {
        *ipv4 = skipaddrGetV4(ipaddr);
    }
    return 0;
}
#endif  /* SK_ENABLE_IPV6 */


char *skipaddrString(char *outbuf, const skipaddr_t *ip, uint32_t ip_flags)
{
#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ip)) {
        switch (ip_flags) {
          case SKIPADDR_DECIMAL:
          case SKIPADDR_CANONICAL:
#ifdef SK_HAVE_INET_NTOP
#  if    SK_NUM2DOT_STRLEN < INET6_ADDRSTRLEN
#    error "SK_NUM2DOT_STRLEN is not big enough"
#  endif
            if (NULL == inet_ntop(AF_INET6, &(ip->ip_ip.ipu_ipv6), outbuf,
                                  SK_NUM2DOT_STRLEN))
            {
                outbuf[0] = '\0';
            }
            break;
#endif /* SK_HAVE_INET_NTOP */

          case SKIPADDR_ZEROPAD:
            /* Convert integer 0 to string "000.000.000.000" */
            snprintf(outbuf, SK_NUM2DOT_STRLEN,
                     ("%02x%02x:%02x%02x:%02x%02x:%02x%02x"
                      ":%02x%02x:%02x%02x:%02x%02x:%02x%02x"),
                     ip->ip_ip.ipu_ipv6[ 0], ip->ip_ip.ipu_ipv6[ 1],
                     ip->ip_ip.ipu_ipv6[ 2], ip->ip_ip.ipu_ipv6[ 3],
                     ip->ip_ip.ipu_ipv6[ 4], ip->ip_ip.ipu_ipv6[ 5],
                     ip->ip_ip.ipu_ipv6[ 6], ip->ip_ip.ipu_ipv6[ 7],
                     ip->ip_ip.ipu_ipv6[ 8], ip->ip_ip.ipu_ipv6[ 9],
                     ip->ip_ip.ipu_ipv6[10], ip->ip_ip.ipu_ipv6[11],
                     ip->ip_ip.ipu_ipv6[12], ip->ip_ip.ipu_ipv6[13],
                     ip->ip_ip.ipu_ipv6[14], ip->ip_ip.ipu_ipv6[15]);
            break;
        }
    } else
#endif
    {
        switch (ip_flags) {
          case SKIPADDR_CANONICAL:
            /* Convert integer 0 to string "0.0.0.0" */
            snprintf(outbuf, SK_NUM2DOT_STRLEN, "%lu.%lu.%lu.%lu",
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 24) & 0xFF),
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 16) & 0xFF),
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 8) & 0xFF),
                     (unsigned long)(ip->ip_ip.ipu_ipv4 & 0xFF));
            break;

          case SKIPADDR_ZEROPAD:
            /* Convert integer 0 to string "000.000.000.000" */
            snprintf(outbuf, SK_NUM2DOT_STRLEN, "%03lu.%03lu.%03lu.%03lu",
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 24) & 0xFF),
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 16) & 0xFF),
                     (unsigned long)((ip->ip_ip.ipu_ipv4 >> 8) & 0xFF),
                     (unsigned long)(ip->ip_ip.ipu_ipv4 & 0xFF));
            break;

          case SKIPADDR_DECIMAL:
            snprintf(outbuf, SK_NUM2DOT_STRLEN, ("%" PRIu32),
                     ip->ip_ip.ipu_ipv4);
            break;
        }
    }

    outbuf[SK_NUM2DOT_STRLEN-1] = '\0';
    return outbuf;
}


static struct policies_st {
    sk_ipv6policy_t     policy;
    const char         *name;
    const char         *description;
} policies[] = {
    {SK_IPV6POLICY_IGNORE, "ignore",
     "Completely ignore IPv6 flows"},
    {SK_IPV6POLICY_ASV4,   "asv4",
     "Convert IPv6 flows to IPv4 if possible, else ignore"},
    {SK_IPV6POLICY_MIX,    "mix",
     "Process a mixture of IPv4 and IPv6 flows"},
    {SK_IPV6POLICY_FORCE,  "force",
     "Force IPv4 flows to be converted to IPv6"},
    {SK_IPV6POLICY_ONLY,   "only",
     "Only process flows that were marked as IPv6"}
};


/* Parse an IPv6 policy.  Return 0 if it is valid, -1 otherwise. */
int skIPv6PolicyParse(
    sk_ipv6policy_t    *ipv6_policy,
    const char         *policy_name,
    const char         *option_name)
{
    size_t len = strlen(policy_name);
    size_t i;

    for (i = 0; i < sizeof(policies)/sizeof(struct policies_st); ++i) {
        if (len < strlen(policies[i].name)) {
            if (0 == strncmp(policies[i].name, policy_name, len)) {
                *ipv6_policy = policies[i].policy;
                return 0;
            }
        } else if (0 == strcmp(policies[i].name, policy_name)) {
            *ipv6_policy = policies[i].policy;
            return 0;
        }
    }

    if (option_name) {
        skAppPrintErr("Invalid %s '%s'", option_name, policy_name);
    }
    return -1;
}


/* store the default policy from the application */
static sk_ipv6policy_t ipv6_default;

/* support for the option */
#define OPT_IPV6_POLICY  0

static struct option ipv6_policy_options[] = {
    {"ipv6-policy",         REQUIRED_ARG, 0, OPT_IPV6_POLICY},
    {0,0,0,0}               /* sentinel */
};

/* handler for the option */
static int ipv6PolicyHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg)
{
    sk_ipv6policy_t *ipv6_policy = (sk_ipv6policy_t*)cData;

    switch (opt_index) {
      case OPT_IPV6_POLICY:
        if (skIPv6PolicyParse(ipv6_policy, opt_arg,
                              ipv6_policy_options[opt_index].name))
        {
            return 1;
        }
        break;

      default:
        skAbortBadCase(opt_index);
    }

    return 0;
}


int skIPv6PolicyOptionsRegister(sk_ipv6policy_t *ipv6_policy)
{
    sk_ipv6policy_t tmp_policy;
    char *env;

    assert(ipv6_policy);

    /* store the default policy wanted by the application */
    ipv6_default = *ipv6_policy;

    /* get the default from the environment */
    env = getenv(SILK_IPV6_POLICY_ENVAR);
    if (env) {
        if (skIPv6PolicyParse(&tmp_policy, env, SILK_IPV6_POLICY_ENVAR) == 0) {
            *ipv6_policy = tmp_policy;
        }
    }

#if !SK_ENABLE_IPV6
    /* Force an IPv4-only SiLK to ignore any IPv6 flows */
    ipv6_default = SK_IPV6POLICY_IGNORE;
    *ipv6_policy = ipv6_default;

    /* Register the option for compatibility with an IPv6-enabled
     * silk, but pass 'ipv6_default' as the clientData, so the user's
     * value does not modify the value the application uses. */
    return skOptionsRegister(ipv6_policy_options, &ipv6PolicyHandler,
                             (clientData)&ipv6_default);
#else
    /* add the option */
    return skOptionsRegister(ipv6_policy_options, &ipv6PolicyHandler,
                             (clientData)ipv6_policy);
#endif  /* SK_ENABLE_IPV6 */
}


void skIPv6PolicyUsage(FILE *fh)
{
    size_t i;

    fprintf(fh, "--%s %s. ",
            ipv6_policy_options[OPT_IPV6_POLICY].name,
            SK_OPTION_HAS_ARG(ipv6_policy_options[OPT_IPV6_POLICY]));
#if !SK_ENABLE_IPV6
    fprintf(fh, ("No IPv6 support available; IPv6 flows are always ignored\n"
                 "\tregardless of the value passed to this switch."
                 " Legal values:\n"));
#else
    fprintf(fh, "Set policy for IPv4/IPv6 flows. ");
    for (i = 0; i < sizeof(policies)/sizeof(struct policies_st); ++i) {
        if (ipv6_default != policies[i].policy) {
            continue;
        }
        fprintf(fh, "Def. %s. ", policies[i].name);
        break;
    }
    fprintf(fh, "Choices:\n");
#endif
    for (i = 0; i < sizeof(policies)/sizeof(struct policies_st); ++i) {
        fprintf(fh, "\t%-6s  - %s\n",
                policies[i].name, policies[i].description);
    }
}


#if SK_ENABLE_IPV6
int skipaddrCompare(const skipaddr_t *addr1, const skipaddr_t *addr2)
{
    skipaddr_t tmp;

    if (addr1->ip_is_v6) {
        if (addr2->ip_is_v6) {
            return memcmp(addr1->ip_ip.ipu_ipv6, addr2->ip_ip.ipu_ipv6, 16);
        }
        skipaddrV4toV6(addr2, &tmp);
        return memcmp(addr1->ip_ip.ipu_ipv6, tmp.ip_ip.ipu_ipv6, 16);
    }
    if (addr2->ip_is_v6) {
        skipaddrV4toV6(addr1, &tmp);
        return memcmp(tmp.ip_ip.ipu_ipv6, addr2->ip_ip.ipu_ipv6, 16);
    }
    /* both addresses are IPv4 */
    if (addr1->ip_ip.ipu_ipv4 < addr2->ip_ip.ipu_ipv4) {
        return -1;
    }
    if (addr1->ip_ip.ipu_ipv4 > addr2->ip_ip.ipu_ipv4) {
        return 1;
    }
    return 0;
}


/* apply the bit-mask in 'mask_ip' to 'ipaddr' */
void skipaddrMask(skipaddr_t *ipaddr, const skipaddr_t *mask_ip)
{
    skipaddr_t tmp;

    if (ipaddr->ip_is_v6) {
        if (mask_ip->ip_is_v6) {
            /* both addresses are IPv6 */
            ipUnionApplyMaskV6(&ipaddr->ip_ip, mask_ip->ip_ip.ipu_ipv6);
            return;
        }
        /* convert mask to IPv6. This result will be strange */
        skipaddrV4toV6(mask_ip, &tmp);
        ipUnionApplyMaskV6(&ipaddr->ip_ip, tmp.ip_ip.ipu_ipv6);
        return;
    }
    if (mask_ip->ip_is_v6) {
        /* 'ipaddr' is IPv4 and 'mask_ip' is IPv6. if bytes 10 and 11
         * of 'mask_ip' are 0xFFFF, then an IPv4 address will result;
         * otherwise, we'll get something strange */
        if (0 == memcmp(&mask_ip->ip_ip.ipu_ipv6[10], &sk_ipv6_v4inv6[10], 2)){
            uint32_t mask_v4;
            memcpy(&mask_v4, &mask_ip->ip_ip.ipu_ipv6[12], 4);
            ipUnionApplyMaskV4(&ipaddr->ip_ip, ntohl(mask_v4));
            return;
        }
        skipaddrV4toV6(ipaddr, ipaddr);
        ipUnionApplyMaskV6(&ipaddr->ip_ip, mask_ip->ip_ip.ipu_ipv6);
        return;
    }
    /* both addresses are IPv4 */
    ipUnionApplyMaskV4(&ipaddr->ip_ip, ipUnionGetV4(&mask_ip->ip_ip));
    return;
}

#endif /* SK_ENABLE_IPV6 */


/* *************    IP WILDCARDS   ******************* */


void skIPWildcardClear(skIPWildcard_t *ipwild)
{
    assert(ipwild);
    memset(ipwild, 0, sizeof(skIPWildcard_t));
    memset(ipwild->m_min, 0xFF, sizeof(ipwild->m_min));
}



#if SK_ENABLE_IPV6
int skIPWildcardCheckIp(const skIPWildcard_t *ipwild, const skipaddr_t *ipaddr)
{
    assert(ipwild);
    assert(ipaddr);

    if (skIPWildcardIsV6(ipwild)) {
        uint8_t ip6[16];
        int i;

        if (skipaddrIsV6(ipaddr)) {
            skipaddrGetV6(ipaddr, ip6);
        } else {
            skipaddr_t tmpip;
            skipaddrV4toV6(ipaddr, &tmpip);
            skipaddrGetV6(&tmpip, ip6);
        }

        for (i = 0; i < 8; ++i) {
            if (!_IPWILD_BLOCK_IS_SET(ipwild, i, ip6[2*i] << 8 | ip6[2*i+1])) {
                return 0;
            }
        }

        return 1;
    }

    if (skipaddrIsV6(ipaddr)) {
        skipaddr_t tmpip;
        const uint8_t ip4in6[12] = {0x0, 0x0, 0x0, 0x0,  0x0,  0x0,
                                    0x0, 0x0, 0x0, 0x0, 0xFF, 0xFF};
        uint8_t ip6[16];
        uint32_t ip4;

        skipaddrGetV6(ipaddr, ip6);
        if (memcmp(ip6, ip4in6, sizeof(ip4in6)) != 0) {
            return 0;
        }

        memcpy(&ip4, &ip6[12], 4);
        ip4 = ntohl(ip4);
        skipaddrSetV4(&tmpip, &ip4);
        return _IPWILD_IPv4_IS_SET(ipwild, &tmpip);
    }

    return _IPWILD_IPv4_IS_SET(ipwild, ipaddr);
}


/* Bind iterator to an ipwildcard, forcing IPv6 addresses */
int skIPWildcardIteratorBindV6(
    skIPWildcardIterator_t *out_iter,
    const skIPWildcard_t   *ipwild)
{
    if (skIPWildcardIteratorBind(out_iter, ipwild)) {
        return -1;
    }
    out_iter->force_ipv6 = 1;
    return 0;
}
#endif /* SK_ENABLE_IPV6 */


/* Bind iterator to an ipwildcard */
int skIPWildcardIteratorBind(
    skIPWildcardIterator_t *out_iter,
    const skIPWildcard_t   *ipwild)
{
    assert(out_iter);
    if (ipwild == NULL) {
        return -1;
    }

    out_iter->ipwild = ipwild;
    out_iter->force_ipv6 = 0;
    skIPWildcardIteratorReset(out_iter);

    return 0;
}


/* Get next entry in tree */
skIteratorStatus_t skIPWildcardIteratorNext(
    skIPWildcardIterator_t *iter,
    skipaddr_t             *ipaddr)
{
    int i;

    assert(ipaddr);

    /* the stopping condition */
    if (iter->no_more_entries) {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }

    /* iterator is looking at current value */
#if SK_ENABLE_IPV6
    if (skIPWildcardIsV6(iter->ipwild)) {
        uint8_t ip6[16];
        for (i = 0; i < 8; ++i) {
            ip6[2*i] = 0xFF & (iter->i_block[i] >> 8);
            ip6[2*i + 1] = 0xFF & iter->i_block[i];
        }
        skipaddrSetV6(ipaddr, ip6);
    } else if (iter->force_ipv6) {
        uint8_t ip6[16] = {0,0,0,0, 0,0,0,0, 0,0,0xFF,0xFF, 0,0,0,0};
        for (i = 0; i < 4; ++i) {
            ip6[12+i] = (uint8_t)(iter->i_block[i]);
        }
        skipaddrSetV6(ipaddr, ip6);
    } else
#endif /* SK_ENABLE_IPV6 */
    {
        uint32_t ip4 = ((iter->i_block[0] << 24) |
                        (iter->i_block[1] << 16) |
                        (iter->i_block[2] <<  8) |
                        (iter->i_block[3]));
        skipaddrSetV4(ipaddr, &ip4);
    }

    /* find the next value */
    for (i = iter->ipwild->num_blocks - 1; i >= 0; --i) {
        /* can we increment the i'th block? */
        if (iter->i_block[i] < iter->ipwild->m_max[i]) {
            /* yes */
            break;
        }
        /* no; reset it and try the block to the left */
        iter->i_block[i] = iter->ipwild->m_min[i];
    }

    /* did we run out of blocks? */
    if (i == -1) {
        iter->no_more_entries = 1;
    } else {
        assert(_IPWILD_BLOCK_IS_SET(iter->ipwild, i, iter->ipwild->m_max[i]));
        do {
            ++iter->i_block[i];
        } while ( !_IPWILD_BLOCK_IS_SET(iter->ipwild, i, iter->i_block[i]));
    }

    return SK_ITERATOR_OK;
}


/* Reset iterator */
void skIPWildcardIteratorReset(skIPWildcardIterator_t *iter)
{
    int i;

    assert(iter);
    iter->no_more_entries = 0;
    for (i = 0; i < iter->ipwild->num_blocks; ++i) {
        iter->i_block[i] = iter->ipwild->m_min[i];
    }
}


/* ******************************************************************** */
/* skcidr_t                                                             */
/* ******************************************************************** */


#if SK_ENABLE_IPV6
/* check whether 'ipaddr' is in 'cidr' */
int skcidrCheckIP(
    const skcidr_t     *cidr,
    const skipaddr_t   *ipaddr)
{
    uint8_t  ipv6[16];
    uint32_t ipv4;

    if (skcidrIsV6(cidr)) {
        skipaddrGetAsV6(ipaddr, ipv6);

        return (0 == memcmp(cidr->v6.ip, ipv6, cidr->v6.byte_length)
                && ((0 == cidr->v6.mask)
                    || ((cidr->v6.mask & ipv6[cidr->v6.byte_length])
                        == cidr->v6.ip[cidr->v6.byte_length])));
    }
    if (0 == skipaddrGetAsV4(ipaddr, &ipv4)) {
        return ((ipv4 & cidr->v4.mask) == cidr->v4.ip);
    }
    return 0;
}
#endif  /* SK_ENABLE_IPV6 */

/* Fill 'ipaddr' with the IP address in 'cidr' */
void skcidrGetIPAddr(
    const skcidr_t     *cidr,
    skipaddr_t         *ipaddr)
{
#if SK_ENABLE_IPV6
    if (skcidrIsV6(cidr)) {
        skipaddrSetV6(ipaddr, cidr->v6.ip);
        return;
    }
#endif
    skipaddrSetV4(ipaddr, &(cidr->v4.ip));
}


/* Fill 'cidr' using the 'ipaddr' and 'cidr_len' */
int skcidrSetFromIPAddr(
    skcidr_t           *cidr,
    const skipaddr_t   *ipaddr,
    uint32_t            cidr_len)
{
#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        uint8_t tmp_ip[16];
        skipaddrGetV6(ipaddr, tmp_ip);
        return skcidrSetV6(cidr, tmp_ip, cidr_len);
    }
#endif
    return skcidrSetV4(cidr, skipaddrGetV4(ipaddr), cidr_len);
}


/* Fill 'cidr' using the 'ipv4' and 'cidr_len' */
int skcidrSetV4(
    skcidr_t           *cidr,
    uint32_t            ipv4,
    uint32_t            cidr_len)
{
    if (cidr_len > 32) {
        return -1;
    }

    skcidrClear(cidr);
    cidr->v4.cidr_length = cidr_len;
    cidr->v4.mask = ((cidr_len == 32)
                     ? UINT32_MAX
                     : ~(UINT32_MAX >> cidr_len));
    cidr->v4.ip = ipv4 & cidr->v4.mask;
    return 0;
}


#if SK_ENABLE_IPV6
/* Fill 'cidr' using the 'ipv6' and 'cidr_len' */
int skcidrSetV6(
    skcidr_t           *cidr,
    const uint8_t      *ipv6,
    uint32_t            cidr_len)
{
    if (cidr_len > 128) {
        return -1;
    }

    skcidrClear(cidr);
    cidr->v6.is_ipv6 = 1;
    cidr->v6.cidr_length = cidr_len;
    cidr->v6.byte_length = cidr_len >> 3;
    cidr->v6.mask = 0xFF & ~(0xFF >> (cidr_len & 0x7));
    memcpy(cidr->v6.ip, ipv6, cidr->v6.byte_length);
    if (cidr->v6.mask) {
        cidr->v6.ip[cidr->v6.byte_length]
            = ipv6[cidr->v6.byte_length] & cidr->v6.mask;
    }
    return 0;
}
#endif  /* SK_ENABLE_IPV6 */


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
