/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Functions to print error message for skstream
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: skstream-err.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "skstream_priv.h"


/* FUNCTION DEFINITIONS */


void skStreamPrintLastErr(
    const skstream_t   *stream,
    int                 errcode,
    sk_msg_fn_t         errfn)
{
    const char *msg;
    char t_stamp1[SKTIMESTAMP_STRLEN];
    char t_stamp2[SKTIMESTAMP_STRLEN];
    char format_name[SK_MAX_STRLEN_FILE_FORMAT+1];
    sktime_t t;

    if (errfn == NULL) {
        errfn = &skAppPrintErr;
    }
#ifdef TEST_PRINTF_FORMATS
#define errfn printf
#endif

    /* macros for common error messages */
#define FILENAME_MSG(m)                                 \
    if (!stream) { errfn("%s", m); }                    \
    else { errfn("%s '%s'", m, stream->pathname); }

#define STRERROR_MSG(m)                                         \
    if (!stream) { errfn("%s", m); }                            \
    else if (stream->errnum == 0) {                             \
        errfn("%s '%s'", m, stream->pathname); }                \
    else {                                                      \
        errfn("%s '%s': %s",                                    \
              m, stream->pathname, strerror(stream->errnum));   \
    }

    switch (errcode) {
      case SKSTREAM_OK:
        FILENAME_MSG("Command completed successfully");
        return;

      case SKSTREAM_ERR_UNSUPPORT_FORMAT:
        msg = "Cannot process file given its format";
        if (!stream) {
            errfn("%s", msg);
        } else {
            sksiteFileformatGetName(format_name, sizeof(format_name),
                                    skHeaderGetFileFormat(stream->silk_hdr));
            errfn("%s: '%s' has format %s (0x%02x)", msg,
                  stream->pathname, format_name,
                  skHeaderGetFileFormat(stream->silk_hdr));
        }
        return;

      case SKSTREAM_ERR_REQUIRE_SILK_FLOW:
        msg = "File does not contain SiLK Flow data";
        if (!stream) {
            errfn("%s", msg);
        } else {
            sksiteFileformatGetName(format_name, sizeof(format_name),
                                    skHeaderGetFileFormat(stream->silk_hdr));
            errfn("%s: '%s' has format %s (0x%02x)", msg,
                  stream->pathname, format_name,
                  skHeaderGetFileFormat(stream->silk_hdr));
        }
        return;

      case SKSTREAM_ERR_UNSUPPORT_VERSION:
        msg = "This SiLK release does not support";
        if (!stream) {
            errfn("%s the format and version of the file", msg);
        } else {
            sk_file_header_t *hdr = stream->silk_hdr;
            sksiteFileformatGetName(format_name, sizeof(format_name),
                                    skHeaderGetFileFormat(hdr));
            errfn("%s %s(0x%02x) v%u records in the v%u file '%s'", msg,
                  format_name, skHeaderGetFileFormat(hdr),
                  skHeaderGetRecordVersion(hdr), skHeaderGetFileVersion(hdr),
                  stream->pathname);
        }
        return;

      case SKSTREAM_ERR_READ_SHORT: /* RWIO_ERR_READ_SHORT */
        msg = "Read incomplete record";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s (%lu of %lu bytes) from %s",
                  msg, (unsigned long)(stream->errobj.num),
                  (unsigned long)stream->recLen, stream->pathname);
        }
        return;

      case SKSTREAM_ERR_STIME_UNDRFLO:
        msg = "Record's start time less than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            sktimestamp_r(t_stamp1, rwRecGetStartTime(stream->errobj.rec), 0);
            sktimestamp_r(t_stamp2, stream->hdr_starttime, 0);

            errfn("%s '%s': %s < %s", msg,
                  stream->pathname, t_stamp1, t_stamp2);
        }
        return;

      case SKSTREAM_ERR_STIME_OVRFLO:
        msg = "Record's start time greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            sktimestamp_r(t_stamp1, rwRecGetStartTime(stream->errobj.rec), 0);
            t = sktimeCreate(MAX_START_TIME, stream->hdr_starttime);
            sktimestamp_r(t_stamp2, t, 0);

            errfn("%s '%s': %s >= %s", msg,
                  stream->pathname, t_stamp1, t_stamp2);
        }
        return;

      case SKSTREAM_ERR_ELPSD_OVRFLO:
        msg = "Record's duration greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s': %u >= %u", msg,
                  stream->pathname,
                  rwRecGetElapsedSeconds(stream->errobj.rec),
                  MAX_ELAPSED_TIME);
        }
        return;

      case SKSTREAM_ERR_PKTS_OVRFLO:
        msg = "Record's packet count greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s': %u >= %u", msg,
                  stream->pathname, rwRecGetPkts(stream->errobj.rec),
                  (MAX_PKTS * PKTS_DIVISOR));
        }
        return;

      case SKSTREAM_ERR_PKTS_ZERO:
        FILENAME_MSG("Record's packet count is zero while writing to file");
        return;

      case SKSTREAM_ERR_BPP_OVRFLO:
        msg = "Record's byte-per-pkt ratio greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s': %u >= %u", msg,
                  stream->pathname, (rwRecGetBytes(stream->errobj.rec)
                                         / rwRecGetPkts(stream->errobj.rec)),
                  MASKARRAY_14);
        }
        return;

      case SKSTREAM_ERR_SNMP_OVRFLO:
        msg = "Record's SNMP index greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else if (rwRecGetInput(stream->errobj.rec) > 255) {
            errfn("%s '%s': input %u > 255", msg,
                  stream->pathname,
                  (unsigned int)rwRecGetInput(stream->errobj.rec));
        } else {
            errfn("%s '%s': output %u > 255", msg,
                  stream->pathname,
                  (unsigned int)rwRecGetOutput(stream->errobj.rec));
        }
        return;

      case SKSTREAM_ERR_SENSORID_OVRFLO:
        msg = "Record's Sensor ID greater than that allowed in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s': %u > 255", msg,
                  stream->pathname,
                  (unsigned int)rwRecGetSensor(stream->errobj.rec));
        }
        return;

      case SKSTREAM_ERR_PROTO_MISMATCH:
        msg = "Record's IP-protocol is not supported in file";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s': %u", msg,
                  stream->pathname,
                  (unsigned int)rwRecGetProto(stream->errobj.rec));
        }
        return;

      case SKSTREAM_ERR_PKTS_GT_BYTES:
        msg = "Record's 'pkts' value is greater than its 'bytes' value";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s in file '%s': %u > %u", msg,
                  stream->pathname,
                  (unsigned int)rwRecGetPkts(stream->errobj.rec),
                  (unsigned int)rwRecGetBytes(stream->errobj.rec));
        }
        return;

      case SKSTREAM_ERR_UNSUPPORT_IPV6:
        FILENAME_MSG("Record has an unsupported IPv6 address");
        return;

      case SKSTREAM_ERR_ALLOC:
        errfn("Memory allocation failed");
        return;

      case SKSTREAM_ERR_PREV_DATA:
        FILENAME_MSG("Initial data has already been read or written");
        return;

      case SKSTREAM_ERR_BAD_MAGIC:
        FILENAME_MSG("File does not appear to be a SiLK data file");
        return;

      case SKSTREAM_ERR_CLOSED:
        FILENAME_MSG("Cannot modify a stream once it is closed");
        return;

      case SKSTREAM_ERR_EOF:    /* RWIO_ERR_READ_EOF */
        FILENAME_MSG("Reached end of file");
        return;

      case SKSTREAM_ERR_FILE_EXISTS:
        STRERROR_MSG("Will not create new file over existing file");
        return;

      case SKSTREAM_ERR_INVALID_INPUT:
        errfn("Argument's value is invalid");
        return;

      case SKSTREAM_ERR_IOBUF:
        if (!stream) {
            errfn("Error reading/writing iobuffer");
        } else {
            errfn("Error %s iobuffer for '%s': %s",
                  ((stream->io_mode == SK_IO_READ) ? "reading" : "writing"),
                  stream->pathname, skIOBufStrError(stream->iobuf));
        }
        return;

      case SKSTREAM_ERR_ISTERMINAL:
        FILENAME_MSG("Will not read/write binary data on a terminal");
        return;

      case SKSTREAM_ERR_LONG_LINE:
        errfn("Input string is too long");
        return;

      case SKSTREAM_ERR_NOPAGER:
        msg = "Unable to invoke pager";
        if (!stream) {
            errfn("%s", msg);
        } else {
            errfn("%s '%s'",
                  msg, stream->pager);
        }
        return;

      case SKSTREAM_ERR_NOT_BOUND:
        errfn("Stream is not bound to a file");
        return;

      case SKSTREAM_ERR_NOT_OPEN:
        FILENAME_MSG("Cannot read/write/close an unopened stream");
        return;

      case SKSTREAM_ERR_NOT_SEEKABLE:
        FILENAME_MSG("Unsupported operation---cannot seek on stream");
        return;

      case SKSTREAM_ERR_NULL_ARGUMENT:
        errfn("Unexpected NULL or empty argument");
        return;

      case SKSTREAM_ERR_PREV_BOUND:
        errfn("Cannot bind stream because it is already bound");
        return;

      case SKSTREAM_ERR_PREV_OPEN:
        FILENAME_MSG("Stream is already open");
        return;

      case SKSTREAM_ERR_PREV_COPYINPUT:
        FILENAME_MSG("Only one copy stream is supported per input stream");
        return;

      case SKSTREAM_ERR_READ:
        STRERROR_MSG("Error reading from stream");
        return;

      case SKSTREAM_ERR_RLOCK:
        STRERROR_MSG("Cannot get read lock on file");
        return;

      case SKSTREAM_ERR_SYS_FDOPEN:
        STRERROR_MSG("Cannot convert file descriptor");
        return;

      case SKSTREAM_ERR_SYS_FORK:
        skAppPrintErr("Cannot fork");
        return;

      case SKSTREAM_ERR_SYS_LSEEK:
        STRERROR_MSG("Cannot seek on stream");
        return;

      case SKSTREAM_ERR_SYS_MKSTEMP:
        STRERROR_MSG("Cannot create temporary file");
        return;

      case SKSTREAM_ERR_SYS_OPEN:
        STRERROR_MSG("Error opening file");
        return;

      case SKSTREAM_ERR_SYS_PIPE:
        STRERROR_MSG("Cannot create pipe");
        return;

      case SKSTREAM_ERR_SYS_MKDIR:
        STRERROR_MSG("Cannot create directory component to file");
        return;

      case SKSTREAM_ERR_SYS_FTRUNCATE:
        STRERROR_MSG("Cannot set length of file");
        return;

      case SKSTREAM_ERR_COMPRESS_INVALID:
        FILENAME_MSG("Specified compression method is not valid");
        return;

      case SKSTREAM_ERR_COMPRESS_UNAVAILABLE:
        FILENAME_MSG("Specified compression method is not available");
        return;

      case SKSTREAM_ERR_UNSUPPORT_CONTENT:
        msg = "Action not supported on stream's content type";
        if (!stream) {
            errfn("%s", msg);
        } else {
            const char *ct = "";
            switch (stream->content_type) {
              case SK_CONTENT_SILK:
              case SK_CONTENT_SILK_FLOW:
                ct = " is SiLK";
                break;
              case SK_CONTENT_TEXT:
                ct = " is text";
                break;
              case SK_CONTENT_OTHERBINARY:
                ct = " is binary";
                break;
            }
            errfn("%s '%s'%s",
                  msg, stream->pathname, ct);
        }
        return;

      case SKSTREAM_ERR_UNSUPPORT_IOMODE:
        msg = "Action not permitted on stream";
        if (!stream) {
            errfn("%s", msg);
        } else {
            const char *io = "";
            switch (stream->io_mode) {
              case SK_IO_READ:
                io = ": read from";
                break;
              case SK_IO_WRITE:
                io = ": write to";
                break;
              case SK_IO_APPEND:
                io = ": append to";
                break;
            }
            errfn("%s%s '%s'",
                  msg, io, stream->pathname);
        }
        return;

      case SKSTREAM_ERR_WLOCK:
        STRERROR_MSG("Cannot get write lock on file");
        return;

      case SKSTREAM_ERR_WRITE:
        STRERROR_MSG("Error writing to stream");
        return;

      case SKSTREAM_ERR_ZLIB:
        msg = "Error in zlib library";
        if (!stream) {
            errfn("%s", msg);
        } else {
            int zerr = 0;
            const char *zerr_msg = NULL;
#if SK_ENABLE_ZLIB
            if (stream->gz) {
                zerr_msg = gzerror(stream->gz, &zerr);
            } else
#endif  /* SK_ENABLE_ZLIB */
            {
                zerr = stream->errnum;
            }
            if (zerr_msg) {
                errfn("%s for '%s': %s",
                      msg, stream->pathname, zerr_msg);
            } else {
                errfn("%s for '%s': [%d]",
                      msg, stream->pathname, zerr);
            }
        }
        return;

      case SKSTREAM_ERR_IO:     /* -1 */
        if (!stream) {
            errfn("Bad read/write");
        } else if (stream->err_info == SKSTREAM_ERR_IO) {
            /* avoid infinite loop */
            FILENAME_MSG("Bad read/write");
        } else {
            skStreamPrintLastErr(stream, stream->err_info, errfn);
        }
        return;
    }

    if (errcode > 0) {
        /* Pass it off to skheaders */
        msg = "Error processing headers";
        if (!stream) {
            errfn("%s: %s", msg, skHeaderStrerror(errcode));
        } else {
            errfn("%s on file '%s': %s",
                  msg, stream->pathname, skHeaderStrerror(errcode));
        }
        return;
    }

    errfn("Unrecognized error code %d", errcode);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
