/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
** skprefixmap.c
**
** John McClary Prevost
** December 2nd, 2004
**
**   Data structure for binding values to IP address prefixes (for
**   example, mapping every value under 128.2/16 to a single value.)
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skprefixmap.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skprefixmap.h>
#include <silk/redblack.h>
#include <silk/skmempool.h>


/* DEFINES AND TYPEDEFS */

typedef struct skPrefixMapRecord_st {
    uint32_t left;
    uint32_t right;
} skPrefixMapRecord_t;

struct skPrefixMap_st {
    /* the nodes that make up the tree */
    skPrefixMapRecord_t    *tree;
    /* the name of the map */
    char                   *mapname;
    /* all terms in dictionary joined by '\0', or NULL for vers 1 */
    char                   *dict_buf;
    /* Pointers into 'dict_buf', one for each word, or NULL for vers 1 */
    char                  **dict_words;
    /* number of nodes in the tree that are in use */
    uint32_t                tree_used;
    /* number of nodes in the tree */
    uint32_t                tree_size;
    /* size of dictionary as it would be written to disk, or 0 for vers 1 */
    uint32_t                dict_buf_used;
    /* number of bytes in dictionary that are in use, or 0 for vers 1 */
    uint32_t                dict_buf_end;
    /* number of bytes allocated to dictionary, or 0 for vers 1 */
    uint32_t                dict_buf_size;
    /* number of words in dictionary, or 0 for vers 1 */
    uint32_t                dict_words_used;
    /* number of possible words in dictionary, or 0 for vers 1 */
    uint32_t                dict_words_size;
    /* max len of word in dictionary, or 0 for vers 1 */
    uint32_t                dict_max_wordlen;
    /* map of dictionary words to values */
    struct rbtree          *word_map;
    /* Memory pool for word map entries */
    sk_mempool_t           *word_map_pool;
    /* type of data in the map */
    skPrefixMapContent_t    content_type;
};

/* Used as the nodes in the word map dictionary */
typedef struct skPrefixMapDictNode_st {
    const char *word;
    uint32_t    value;
} skPrefixMapDictNode_t;

#define SKPREFIXMAP_IS_NODE(x)      (!(x & 0x80000000u))
#define SKPREFIXMAP_IS_LEAF(x)      (x & 0x80000000u)
#define SKPREFIXMAP_LEAF_VALUE(x)   (x & (~0x80000000u))
#define SKPREFIXMAP_MAKE_LEAF(val)  ((val) | 0x80000000u)


/* The initial sizes of and how quickly to grow the (1)tree of nodes,
 * the (2)dictionary buffer, and the (3)dictionary words index */
#define SKPREFIXMAP_TREE_SIZE_INIT      (1 << 14)
#define SKPREFIXMAP_TREE_SIZE_GROW      SKPREFIXMAP_TREE_SIZE_INIT

#define SKPREFIXMAP_DICT_BUF_INIT       (1 << 16)
#define SKPREFIXMAP_DICT_BUF_GROW       SKPREFIXMAP_DICT_BUF_INIT

#define SKPREFIXMAP_WORDS_COUNT_INIT    8192
#define SKPREFIXMAP_WORDS_COUNT_GROW    2048



/* LOCAL FUNCTION PROTOTYPES */

static skPrefixMapErr_t prefixMapGrowDictionaryBuff(
    skPrefixMap_t  *map,
    size_t          min_bytes);
static skPrefixMapErr_t prefixMapGrowDictionaryWords(
    skPrefixMap_t  *map,
    size_t          min_entries);
static skPrefixMapErr_t prefixMapGrowTree(
    skPrefixMap_t  *map);
static int prefixMapWordCompare(
    const void *va,
    const void *vb,
    const void *ctx);


/* FUNCTION DEFINITIONS */

/*
 *  err = prefixMapAdd(map, low, high, dict_val, key, bit);
 *
 *    Add the range 'low' to 'high' to the prefix map 'map' with a
 *    value of 'dict_val'.  This function is recursive.
 */
static skPrefixMapErr_t prefixMapAdd(
    skPrefixMap_t  *map,
    uint32_t        low_val,
    uint32_t        high_val,
    uint32_t        dict_val,
    uint32_t        key,
    uint32_t        bit)
{
    uint32_t masked;
    int rv;

    assert(low_val <= high_val);
    assert(bit < 32);

    /* Does the left subtree overlap [low_val,high_val] ? */
    if (0 == GET_MASKED_BITS(low_val, bit, 1)) {
        /* Yes, left tree overlaps */

        /*
         * Is the left subtree completely contained in
         * [low_bits,high_bits] ?
         *
         * That is, is the low end the low end of the left subtree?
         * And, is the high end either the high end of the left
         * subtree or somewhere in the right subtree?)
         */

        if ((0 == GET_MASKED_BITS(low_val, 0, bit))
            && ((1 == GET_MASKED_BITS(high_val, bit, 1))
                || (GET_MASKED_BITS(high_val, 0, bit) == _BITMASK(bit))))
        {
            /* Left subtree completely contained; set left subtree to
             * the dictionary value. */
            map->tree[key].left = dict_val;
        } else {

            /* Left subtree overlaps but is not completely contained. */

            /* If left subtree is a leaf, we need to break it into two
             * subrecords */
            if (SKPREFIXMAP_IS_LEAF(map->tree[key].left)) {
                if (map->tree_used == map->tree_size) {
                    if (prefixMapGrowTree(map)) {
                        return SKPREFIXMAP_ERR_MEMORY;
                    }
                }
                map->tree[map->tree_used].left = map->tree[key].left;
                map->tree[map->tree_used].right = map->tree[key].left;
                map->tree[key].left = map->tree_used;
                ++map->tree_used;
            }

            /* Recurse down the tree.  */
            if (1 == GET_MASKED_BITS(high_val, bit, 1)) {
                /* set least significant 'bit' bits to 1 */
                masked = low_val;
                SET_MASKED_BITS(masked, UINT32_MAX, 0, bit);
                rv = prefixMapAdd(map, low_val, masked, dict_val,
                                  map->tree[key].left, bit-1);
            } else {
                rv = prefixMapAdd(map, low_val, high_val, dict_val,
                                  map->tree[key].left, bit-1);
            }
            if (rv != SKPREFIXMAP_OK) {
                return rv;
            }
        }
    }


    /* NOW, handle the right-hand side, a mirror image */

    /* Does the right subtree overlap [low_val,high_val] ? */
    if (1 == GET_MASKED_BITS(high_val, bit, 1)) {
        /* Yes, right tree overlaps */

        /*
         * Is the right subtree completely contained in
         * [low_bits,high_bits] ?
         *
         * That is, is the high end the high end of the right subtree?
         * And, is the low end either the low end of the right
         * subtree or somewhere in the left subtree?)
         */

        if ((GET_MASKED_BITS(high_val, 0, bit) == _BITMASK(bit))
            && ((0 == GET_MASKED_BITS(low_val, bit, 1))
                || (0 == GET_MASKED_BITS(low_val, 0, bit))))
        {
            /* Right subtree completely contained; set right subtree
             * to the dictionary value. */
            map->tree[key].right = dict_val;
        } else {

            /* Right subtree overlaps but is not completely contained. */

            /* If right subtree is a leaf, we need to break it into two
             * subrecords */
            if (SKPREFIXMAP_IS_LEAF(map->tree[key].right)) {
                if (map->tree_used == map->tree_size) {
                    if (prefixMapGrowTree(map)) {
                        return SKPREFIXMAP_ERR_MEMORY;
                    }
                }
                map->tree[map->tree_used].left = map->tree[key].right;
                map->tree[map->tree_used].right = map->tree[key].right;
                map->tree[key].right = map->tree_used;
                ++map->tree_used;
            }

            /* Recurse down the tree.  */
            if (0 == GET_MASKED_BITS(low_val, bit, 1)) {
                /* set least significant 'bit' bits to 0 */
                masked = high_val;
                SET_MASKED_BITS(masked, 0, 0, bit);
                rv = prefixMapAdd(map, masked, high_val, dict_val,
                                  map->tree[key].right, bit-1);
            } else {
                rv = prefixMapAdd(map, low_val, high_val, dict_val,
                                  map->tree[key].right, bit-1);
            }
            if (rv != SKPREFIXMAP_OK) {
                return rv;
            }
        }
    }

    return SKPREFIXMAP_OK;
}


/* Insert 'word' as the value in pdict_val.  If word already exists
 * with a different value than the one in pdict_val, place that value
 * in pdict_val and return SKPREFIXMAP_ERR_DUPLICATE.
 *
 * This helper function is used by skPrefixMapDictionaryInsert() and
 * skPrefixMapDictionarySearch().  */
static skPrefixMapErr_t prefixMapDictionaryInsertHelper(
    skPrefixMap_t  *map,
    uint32_t       *pdict_val,
    const char     *word)
{
    skPrefixMapDictNode_t *word_node;
    const skPrefixMapDictNode_t *found;
    uint32_t dict_val;
    uint32_t val;
    size_t len;
    int rv;

    assert(pdict_val);
    dict_val = *pdict_val;

    if (map == NULL || word == NULL) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    len = strlen(word);
    if (len == 0) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    if (dict_val > SKPREFIXMAP_MAX_VALUE) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    val = skPrefixMapDictionaryLookup(map, word);
    if (val == dict_val) {
        return SKPREFIXMAP_OK;
    }
    if (val != SKPREFIXMAP_NOT_FOUND) {
        *pdict_val = val;
        return SKPREFIXMAP_ERR_DUPLICATE;
    }

    if (dict_val < map->dict_words_used) {
        if (map->dict_words[dict_val]) {
            /* error: new word is not same as value at this entry,
             * else we would have caught it above with
             * skPrefixMapDictionaryLookup() */
            return SKPREFIXMAP_ERR_DUPLICATE;
        }
        /* Subtract out the interior \0 that we will skip during save */
        map->dict_buf_used -= 1;
    } else {
        /* Add in the new interior \0s that we will be adding on save */
        map->dict_buf_used += dict_val - map->dict_words_used;
    }

    /* We add this entry to the end of the dictionary, perhaps
     * far beyond the final value */

    /* account for NUL byte */
    ++len;

    /* grow the dictionary */
    if ((map->dict_buf_size - map->dict_buf_end) < len) {
        if (prefixMapGrowDictionaryBuff(map, len)) {
            return SKPREFIXMAP_ERR_MEMORY;
        }
    }

    /* grow the dict_words */
    if (map->dict_words_size < (1 + dict_val)) {
        if (prefixMapGrowDictionaryWords(map,
                                         1 + dict_val - map->dict_words_size))
        {
            return SKPREFIXMAP_ERR_MEMORY;
        }
    }

    map->dict_words[dict_val] = &(map->dict_buf[map->dict_buf_end]);
    strncpy(map->dict_words[dict_val], word, len);

    map->dict_buf_end += len;
    map->dict_buf_used += len;
    if (dict_val >= map->dict_words_used) {
        map->dict_words_used = 1 + dict_val;
    }

    /* Make sure we have a memory pool */
    if (map->word_map_pool == NULL) {
        rv = skMemoryPoolCreate(&map->word_map_pool,
                                sizeof(skPrefixMapDictNode_t),
                                SKPREFIXMAP_WORDS_COUNT_GROW);
        if (rv != 0) {
            return SKPREFIXMAP_ERR_MEMORY;
        }
    }

    /* Add the word to the word map */
    word_node = skMemPoolElementNew(map->word_map_pool);
    if (word_node == NULL) {
        return SKPREFIXMAP_ERR_MEMORY;
    }
    word_node->value = dict_val;
    word_node->word = map->dict_words[dict_val];
    if (map->word_map == NULL) {
        map->word_map = rbinit(prefixMapWordCompare, NULL);
        if (map->word_map == NULL) {
            return SKPREFIXMAP_ERR_MEMORY;
        }
    }
    found = rbsearch(word_node, map->word_map);
    if (found == NULL) {
        return SKPREFIXMAP_ERR_MEMORY;
    }
    assert(found == word_node);

    *pdict_val = dict_val;

    return SKPREFIXMAP_OK;
}


/* Return the dict_val for the given key, or SKPREFIXMAP_NOT_FOUND */
static uint32_t prefixMapGet(
    const skPrefixMap_t *map,
    uint32_t             key,
    int                 *depth)
{
    uint32_t node = 0;          /* Start at the root node */

    *depth = 32;                /* Start at the leftmost bit */

    while ( SKPREFIXMAP_IS_NODE(node) ) {
        --*depth;
        if ( *depth < 0 ) {
            /* This should be caught when the map is loaded. */
            skAppPrintErr("Corrupt prefix map.  No result found in 32 bits.");
            return SKPREFIXMAP_NOT_FOUND;
        }
        if ( key & (1 << *depth) ) {
            /* Go right if the bit is 1. */
            node = map->tree[node].right;
        } else {
            /* Go left if the bit is 0. */
            node = map->tree[node].left;
        }
    }
    return SKPREFIXMAP_LEAF_VALUE(node);
}


/*
 *  status = prefixMapGrowDictionaryBuff(map, bytes);
 *
 *    Grow the dictionary buffer at least enough to hold an entry of
 *    size 'bytes'.  Adjust the 'word' pointers into this buffer as
 *    well.
 */
static skPrefixMapErr_t prefixMapGrowDictionaryBuff(
    skPrefixMap_t  *map,
    size_t          min_bytes)
{
    char *old_ptr = map->dict_buf;
    size_t grow;

    /* use initial size if this is first time */
    if (map->dict_buf_size == 0) {
        grow = SKPREFIXMAP_DICT_BUF_INIT;
    } else {
        grow = SKPREFIXMAP_DICT_BUF_GROW;
    }

    /* make certain it is big enough */
    while (grow < min_bytes) {
        grow += SKPREFIXMAP_DICT_BUF_GROW;
    }

    /* compute total size to allocate */
    grow += map->dict_buf_size;

    map->dict_buf = realloc(old_ptr, grow * sizeof(char));
    if (map->dict_buf == NULL) {
        map->dict_buf = old_ptr;
        return SKPREFIXMAP_ERR_MEMORY;
    }

    map->dict_buf_size = grow;

    /* move the dict_words pointers if they are no longer valid
     * because realloc() moved the memory */
    if (map->word_map && map->dict_buf != old_ptr) {
        RBLIST *iter;
        skPrefixMapDictNode_t *node;

        /* Amount to adjust pointers into the buffer by */
        ptrdiff_t diff = map->dict_buf - old_ptr;

        iter = rbopenlist(map->word_map);
        if (iter == NULL) {
            return SKPREFIXMAP_ERR_MEMORY;
        }
        while ((node = (skPrefixMapDictNode_t*)rbreadlist(iter)) != NULL) {
            node->word += diff;
            map->dict_words[node->value] += diff;
        }
        rbcloselist(iter);
    }

    return SKPREFIXMAP_OK;
}


/*
 *  status = prefixMapGrowDictionaryWords(map, entries);
 *
 *    Grow the dictionary words list at least enough to hold 'entries'
 *    new entriese.
 */
static skPrefixMapErr_t prefixMapGrowDictionaryWords(
    skPrefixMap_t  *map,
    size_t          min_entries)
{
    void *old_ptr = map->dict_words;
    size_t grow;

    /* use initial size if this is first time */
    if (map->dict_words_size == 0) {
        grow = SKPREFIXMAP_WORDS_COUNT_INIT;
    } else {
        grow = SKPREFIXMAP_WORDS_COUNT_GROW;
    }

    /* make certain it is big enough */
    while (grow < min_entries) {
        grow += SKPREFIXMAP_WORDS_COUNT_GROW;
    }

    /* compute total size to allocate */
    grow += map->dict_words_size;

    map->dict_words = realloc(old_ptr, grow * sizeof(char*));
    if (map->dict_words == NULL) {
        map->dict_words = old_ptr;
        return SKPREFIXMAP_ERR_MEMORY;
    }
    /* clear new memory */
    memset((map->dict_words + map->dict_words_size), 0,
           ((grow - map->dict_words_size) * sizeof(void *)));

    map->dict_words_size = grow;
    return SKPREFIXMAP_OK;
}


/*
 *  status = prefixMapGrowTree(map);
 *
 *    Grow the number of nodes in the tree for the prefix map 'map'.
 */
static skPrefixMapErr_t prefixMapGrowTree(
    skPrefixMap_t  *map)
{
    void *old_ptr = map->tree;
    uint32_t grow = map->tree_size + SKPREFIXMAP_TREE_SIZE_GROW;

    map->tree = realloc(map->tree, grow * sizeof(skPrefixMapRecord_t));
    if (map->tree == NULL) {
        map->tree = old_ptr;
        return SKPREFIXMAP_ERR_MEMORY;
    }

    map->tree_size = grow;
    return SKPREFIXMAP_OK;
}


/* Compare two skPrefixMapDictNode_t's.  Used by the rbtree. */
static int prefixMapWordCompare(
    const void *va,
    const void *vb,
    const void *UNUSED(ctx))
{
    const skPrefixMapDictNode_t *a = va;
    const skPrefixMapDictNode_t *b = vb;
    return strcasecmp(a->word, b->word);
}


/* Set entries from 'low_val' to 'high_val' inclusive to 'dict_val' in 'map' */
skPrefixMapErr_t skPrefixMapAddRange(
    skPrefixMap_t  *map,
    uint32_t        low_val,
    uint32_t        high_val,
    uint32_t        dict_val)
{
    if (dict_val > SKPREFIXMAP_MAX_VALUE) {
        return -1;
    }

    if (high_val < low_val) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    return prefixMapAdd(map, low_val, high_val,
                        SKPREFIXMAP_MAKE_LEAF(dict_val), 0, 31);
}


/* Create a new prefix map at memory pointed at by 'map' */
skPrefixMapErr_t skPrefixMapCreate(
    skPrefixMap_t **map)
{
    if (NULL == map) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    *map = calloc(1, sizeof(skPrefixMap_t));
    if (NULL == *map) {
        return SKPREFIXMAP_ERR_MEMORY;
    }

    (*map)->tree_size = SKPREFIXMAP_TREE_SIZE_INIT;
    (*map)->tree = calloc((*map)->tree_size, sizeof(skPrefixMapRecord_t));
    if (NULL == (*map)->tree) {
        free(*map);
        return SKPREFIXMAP_ERR_MEMORY;
    }
    (*map)->tree[0].left = SKPREFIXMAP_MAKE_LEAF(SKPREFIXMAP_MAX_VALUE);
    (*map)->tree[0].right = SKPREFIXMAP_MAKE_LEAF(SKPREFIXMAP_MAX_VALUE);
    (*map)->tree_used = 1;

    return SKPREFIXMAP_OK;
}


/* destroy the prefix map */
void skPrefixMapDelete(
    skPrefixMap_t  *map)
{
    if (NULL == map) {
        return;
    }
    if ( map->tree != NULL ) {
        if (map->mapname) {
            free(map->mapname);
        }
        if (map->dict_buf) {
            free(map->dict_buf);
        }
        if (map->dict_words) {
            free(map->dict_words);
        }
        if (map->tree) {
            free(map->tree);
        }
        if (map->word_map) {
            rbdestroy(map->word_map);
        }
        skMemoryPoolDestroy(&map->word_map_pool);
        memset(map, 0, sizeof(skPrefixMap_t));
    }
    free(map);
}


/* Fill 'out_buf' with dictionary entry given the value 'dict_val' */
int skPrefixMapDictionaryGetEntry(
    const skPrefixMap_t    *map,
    uint32_t                dict_val,
    char                   *out_buf,
    size_t                  bufsize)
{
    if ((map->dict_buf_size == 0) || (dict_val >= map->dict_words_used)) {
        if (dict_val == SKPREFIXMAP_NOT_FOUND
            || dict_val == SKPREFIXMAP_MAX_VALUE)
        {
            return snprintf(out_buf, bufsize, "UNKNOWN");
        }
        return snprintf(out_buf, bufsize, ("%" PRIu32), dict_val);
    } else if (map->dict_words[dict_val] != NULL) {
        return snprintf(out_buf, bufsize, "%s", map->dict_words[dict_val]);
    } else {
        if (bufsize) {
            out_buf[0] = '\0';
        }
        return 0;
    }
}


/* Returns the max length of any word in the dictionary. */
uint32_t skPrefixMapDictionaryGetMaxWordSize(
    const skPrefixMap_t    *map)
{
    if (map->dict_words_used) {
        return map->dict_max_wordlen;
    }
    /* size necessary to hold string representation of UINT32_MAX. */
    return 10;
}


/* Returns the number of words in the prefix map's dictionary. */
uint32_t skPrefixMapDictionaryGetWordCount(
    const skPrefixMap_t    *map)
{
    return map->dict_words_used;
}


/* Create a new dictionary entry mapping 'dict_val' to 'label' */
skPrefixMapErr_t skPrefixMapDictionaryInsert(
    skPrefixMap_t  *map,
    uint32_t        dict_val,
    const char     *word)
{
    return prefixMapDictionaryInsertHelper(map, &dict_val, word);
}


/* Returns the dict_val for a given dictionary word
 * (case-insensitive), or SKPREFIXMAP_NOT_FOUND */
uint32_t skPrefixMapDictionaryLookup(
    const skPrefixMap_t    *map,
    const char             *word)
{
    skPrefixMapDictNode_t target;
    const skPrefixMapDictNode_t *found;

    if (map->word_map == NULL) {
        return SKPREFIXMAP_NOT_FOUND;
    }

    target.word = word;
    found = rbfind(&target, map->word_map);
    if (found) {
        return found->value;
    }
    return SKPREFIXMAP_NOT_FOUND;
}


/* Create a new dictionary entry mapping 'out_dict_val' to 'label' */
skPrefixMapErr_t skPrefixMapDictionarySearch(
    skPrefixMap_t  *map,
    const char     *word,
    uint32_t       *out_dict_val)
{
    skPrefixMapErr_t rv;

    if (map == NULL || word == NULL || out_dict_val == NULL) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    /* Try to insert the word using a new unused value */
    *out_dict_val = map->dict_words_used;
    rv = prefixMapDictionaryInsertHelper(map, out_dict_val, word);
    if (rv == SKPREFIXMAP_ERR_DUPLICATE) {
        /* If the word already existed with a different value, that
         * value was returned in out_dict_val.  This is fine. */
        return SKPREFIXMAP_OK;
    }

    return rv;
}


/* Return the dict_val for the given key, or SKPREFIXMAP_NOT_FOUND */
uint32_t skPrefixMapGet(const skPrefixMap_t *map, uint32_t key)
{
    int depth;
    return prefixMapGet(map, key, &depth);
}


/* Return a textual representation of the content type. */
const char *skPrefixMapGetContentName(
    int             content_id)
{
    static char buf[256];

    switch ((skPrefixMapContent_t)content_id) {
      case SKPREFIXMAP_CONT_ADDR:
        return "address";
      case SKPREFIXMAP_CONT_PROTO_PORT:
        return "proto-port";
    }

    snprintf(buf, sizeof(buf), "Unrecognized prefix map content type id %d",
             content_id);
    return buf;
}


/* Return the prefix map's name */
const char *skPrefixMapGetMapName(
    const skPrefixMap_t    *map)
{
    return map->mapname;
}


/* Returns the content type for a given map */
skPrefixMapContent_t skPrefixMapGetContentType(
    const skPrefixMap_t    *map)
{
    return map->content_type;
}


/* Fill 'out_buf' with dictionary entry name given the key 'key' */
int skPrefixMapGetString(
    const skPrefixMap_t    *map,
    uint32_t                key,
    char                   *out_buf,
    size_t                  bufsize)
{
    int depth;
    return skPrefixMapDictionaryGetEntry(map, prefixMapGet(map, key, &depth),
                                         out_buf, bufsize);
}


/* Bind an iterator to a prefix map */
int skPrefixMapIteratorBind(
    skPrefixMapIterator_t  *iter,
    const skPrefixMap_t    *map)
{
    if (iter == NULL || map == NULL) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    iter->map = map;
    skPrefixMapIteratorReset(iter);
    return SKPREFIXMAP_OK;
}


/* Create an iterator */
int skPrefixMapIteratorCreate(
    skPrefixMapIterator_t **out_iter,
    const skPrefixMap_t    *map)
{
    assert(out_iter);

    *out_iter = malloc(sizeof(skPrefixMapIterator_t));
    if (*out_iter == NULL) {
        return SKPREFIXMAP_ERR_MEMORY;
    }

    if (skPrefixMapIteratorBind(*out_iter, map)) {
        skPrefixMapIteratorDestroy(out_iter);
        return SKPREFIXMAP_ERR_ARGS;
    }

    return SKPREFIXMAP_OK;
}


/* Destroy the iterator */
void skPrefixMapIteratorDestroy(
    skPrefixMapIterator_t **out_iter)
{
    if (*out_iter) {
        free(*out_iter);
        *out_iter = NULL;
    }
}


/* Fill 'start' and 'end' with next range */
skIteratorStatus_t skPrefixMapIteratorNext(
    skPrefixMapIterator_t  *iter,
    uint32_t               *out_key_start,
    uint32_t               *out_key_end,
    uint32_t               *out_value)
{
    uint32_t key;
    int depth;
    uint32_t val;
    uint32_t old_val = SKPREFIXMAP_NOT_FOUND;

    assert(out_key_start);
    assert(out_key_end);
    assert(out_value);

    /* check for the stoping condition */
    if (iter->end == UINT32_MAX) {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }

    /* check for the starting condition */
    if (iter->end < iter->start) {
        iter->start = 0;
    } else {
        /* move key to start of next range */
        iter->start = iter->end + 1;
    }

    key = iter->start;
    for (;;) {
        val = prefixMapGet(iter->map, key, &depth);
        if (old_val == SKPREFIXMAP_NOT_FOUND) {
            old_val = val;
        }
        if (old_val == val) {
            /* grow current range */
            key += (1 << depth);
            if (key == 0) {
                iter->end = UINT32_MAX;
                break;
            }
        } else {
            iter->end = key - 1;
            break;
        }
    }

    /* set the output values to our current location */
    *out_key_start = iter->start;
    *out_key_end = iter->end;
    *out_value = old_val;
    return SK_ITERATOR_OK;
}


/* Reset iterator */
void skPrefixMapIteratorReset(skPrefixMapIterator_t *iter)
{
    assert(iter);
    iter->end = 0;
    iter->start = 1;
}


/* Allocate new prefix map; open and read contents from 'path' */
skPrefixMapErr_t skPrefixMapLoad(skPrefixMap_t **map, const char *path)
{
    skstream_t *in;
    int rv = SKPREFIXMAP_OK;

    /* Check arguments for sanity */
    if ( NULL == map ) {
        skAppPrintErr("No place was provided to store new prefix map.");
        return SKPREFIXMAP_ERR_ARGS;
    }
    if ( NULL == path ) {
        skAppPrintErr("No input file provided from which to read prefix map.");
        return SKPREFIXMAP_ERR_ARGS;
    }

    if ((rv = skStreamCreate(&in, SK_IO_READ, SK_CONTENT_SILK))
        || (rv = skStreamBind(in, path))
        || (rv = skStreamOpen(in)))
    {
        skStreamPrintLastErr(in, rv, &skAppPrintErr);
        rv = SKPREFIXMAP_ERR_IO;
        goto END;
    }

    rv = skPrefixMapRead(map, in);

  END:
    skStreamDestroy(&in);
    return rv;
}


/* Allocate new prefix map and read contents from 'in' */
skPrefixMapErr_t skPrefixMapRead(skPrefixMap_t **map, skstream_t *in)
{
    union h_un {
        sk_header_entry_t      *he;
        sk_hentry_prefixmap_t  *pm;
    } h;
    sk_file_header_t *hdr;
    fileVersion_t vers;
    uint32_t record_count;
    uint32_t max_key_used;
    uint32_t swapFlag;
    uint32_t i;
    char *current;
    char *start;
    char *end;
    int rv;

    /* Check arguments for sanity */
    if ( NULL == map ) {
        skAppPrintErr("No place was provided to store new prefix map.");
        return SKPREFIXMAP_ERR_ARGS;
    }
    if ( NULL == in ) {
        skAppPrintErr("No input stream provided from which to read prefix map");
        return SKPREFIXMAP_ERR_ARGS;
    }

    *map = NULL;

    rv = skStreamReadSilkHeader(in, &hdr);
    if (rv) {
        skStreamPrintLastErr(in, rv, &skAppPrintErr);
        return SKPREFIXMAP_ERR_IO;
    }

    if (skStreamCheckSilkHeader(in, FT_PREFIXMAP, 1, 3, &skAppPrintErr)){
        return SKPREFIXMAP_ERR_IO;
    }
    vers = skHeaderGetRecordVersion(hdr);

    if (SK_COMPMETHOD_NONE != skHeaderGetCompressionMethod(hdr) ) {
        skAppPrintErr("Unrecognized prefix map compression method.");
        return SKPREFIXMAP_ERR_IO;
    }

    swapFlag = !skHeaderIsNativeByteOrder(hdr);

    /* Read record count */
    if (skStreamRead(in, &(record_count), sizeof(record_count))
        != sizeof(record_count))
    {
        skAppPrintErr("Error reading header from input file (short read).");
        return SKPREFIXMAP_ERR_IO;
    }

    if ( swapFlag ) {
        record_count = BSWAP32(record_count);
    }

    if ( record_count < 1 ) {
        skAppPrintErr("Input file contains invalid prefix map (no data).");
        return SKPREFIXMAP_ERR_IO;
    }

    /* Allocate a prefix map, and a storage buffer */
    *map = calloc(1, sizeof(skPrefixMap_t));
    if (NULL == *map) {
        skAppPrintErr("Failed to allocate memory for prefix map.");
        return SKPREFIXMAP_ERR_MEMORY;
    }

    if (3 == vers) {
        (*map)->content_type = SKPREFIXMAP_CONT_PROTO_PORT;
    } else {
        (*map)->content_type = SKPREFIXMAP_CONT_ADDR;
    }

    (*map)->tree = malloc(record_count * sizeof(skPrefixMapRecord_t));
    if (NULL == (*map)->tree) {
        skAppPrintErr("Failed to allocate memory for prefix map data.");
        rv = SKPREFIXMAP_ERR_MEMORY;
        goto ERROR;
    }
    (*map)->tree_size = record_count;
    (*map)->tree_used = record_count;

    /* Get the mapname from the header if it was specified and if the
     * header-entry version is 1. */
    h.he = skHeaderGetFirstMatch(hdr, SK_HENTRY_PREFIXMAP_ID);
    if ((h.he) && (1 == skHentryPrefixmapGetVersion(h.pm))) {
        (*map)->mapname = strdup(skHentryPrefixmapGetMapmame(h.pm));
        if (NULL == (*map)->mapname) {
            skAppPrintErr("Failed to allocate memory for prefix map name.");
            rv = SKPREFIXMAP_ERR_MEMORY;
            goto ERROR;
        }
    }

    /* Allocation completed successfully, read in the records. */
    if (skStreamRead(in, (*map)->tree,
                     (record_count * sizeof(skPrefixMapRecord_t)))
        != (ssize_t)(record_count * sizeof(skPrefixMapRecord_t)))
    {
        skAppPrintErr("Failed to read all records from input file.");
        rv = SKPREFIXMAP_ERR_IO;
        goto ERROR;
    }

    /* Swap the byte order of the data if needed. */
    if ( swapFlag ) {
        for ( i = 0; i < record_count; i++ ) {
            (*map)->tree[i].left = BSWAP32((*map)->tree[i].left);
            (*map)->tree[i].right = BSWAP32((*map)->tree[i].right);
        }
    }

    /* If version is 2, allocate and read dictionary. */
    if ( vers >= 2 ) {

        if (skStreamRead(in, &((*map)->dict_buf_size), sizeof(uint32_t))
            != (ssize_t)sizeof(uint32_t))
        {
            skAppPrintErr("Error reading dictionary from input file.");
            rv = SKPREFIXMAP_ERR_IO;
            goto ERROR;
        }

        if (swapFlag) {
            (*map)->dict_buf_size = BSWAP32((*map)->dict_buf_size);
        }

        (*map)->dict_buf = malloc((*map)->dict_buf_size * sizeof(char));
        if ( NULL == (*map)->dict_buf ) {
            skAppPrintErr("Failed to allocate memory for prefix map dict.");
            rv = SKPREFIXMAP_ERR_MEMORY;
            goto ERROR;
        }

        /* Allocation on the dictionary is done, read in data. */
        if (skStreamRead(in, (*map)->dict_buf, (*map)->dict_buf_size)
            != (ssize_t)(*map)->dict_buf_size)
        {
            skAppPrintErr("Failed to read dictionary from input file.");
            rv = SKPREFIXMAP_ERR_IO;
            goto ERROR;
        }
        (*map)->dict_buf_end = (*map)->dict_buf_size;
        (*map)->dict_buf_used = (*map)->dict_buf_size;

        /* Index the dictionary data */

        /* First pass: count the words */
        start = (*map)->dict_buf;
        end = (*map)->dict_buf + (*map)->dict_buf_end;
        (*map)->dict_words_used = 0;
        for ( current = start; current < end; current++ ) {
            if ( (*current) == '\0' ) {
                (*map)->dict_words_used++;
            }
        }
        (*map)->dict_words_size = (*map)->dict_words_used;

        /* Allocate space for index */
        (*map)->dict_words = malloc((*map)->dict_words_size * sizeof(char*));
        if ((*map)->dict_words == NULL) {
            skAppPrintErr("Failed to allocated memory for prefix map index.");
            rv = SKPREFIXMAP_ERR_MEMORY;
            goto ERROR;
        }
        rv = skMemoryPoolCreate(&(*map)->word_map_pool,
                                sizeof(skPrefixMapDictNode_t),
                                SKPREFIXMAP_WORDS_COUNT_GROW);
        if (rv != 0) {
            skAppPrintErr("Failed to allocated memory for prefix map "
                          "index memory pool.");
            rv = SKPREFIXMAP_ERR_MEMORY;
            goto ERROR;
        }
        (*map)->word_map = rbinit(prefixMapWordCompare, NULL);
        if ((*map)->word_map == NULL) {
            skAppPrintErr("Failed to allocated memory for prefix map "
                          "index tree.");
            rv = SKPREFIXMAP_ERR_MEMORY;
            goto ERROR;
        }

        /* Now build index */
        current = (*map)->dict_buf;
        start = current;
        end = current + (*map)->dict_buf_used;
        for ( i = 0; i < (*map)->dict_words_used; i++ ) {
            while ( (*current != '\0') && (current < end) ) {
                current++;
            }
            (*map)->dict_words[i] = ((current == start) ? NULL : start);
            if (current != start) {
                skPrefixMapDictNode_t *node;
                const skPrefixMapDictNode_t *found;

                node = skMemPoolElementNew((*map)->word_map_pool);
                if (node == NULL) {
                    rv = SKPREFIXMAP_ERR_MEMORY;
                    goto ERROR;
                }
                node->word = start;
                node->value = i;
                found = rbsearch(node, (*map)->word_map);
                if (found == NULL) {
                    rv = SKPREFIXMAP_ERR_MEMORY;
                    goto ERROR;
                }
                if (found != node) {
                    rv = SKPREFIXMAP_ERR_DUPLICATE;
                    goto ERROR;
                }
            }

            if ((uint32_t)(current - start) > (*map)->dict_max_wordlen) {
                (*map)->dict_max_wordlen = (current - start);
            }

            current++;
            start = current;
        }
    }

    /*
     * Check the invariants that ensure that the prefixmap is a total
     * mapping (every input maps to a single output.)
     *
     * In theory, the invariants are:
     *
     * 1) No node may point to a non-existent child (>= record_count).
     * See note below.
     *
     * 2) No node may point to a node earlier than itself (prevents
     * cycles)
     *
     *
     * The first invariant as written is too strict as there is
     * sometimes extra data at the end of the map.  Instead, we make
     * certain that no node points to locatation beyond the first node
     * that contains an invalid child.  To determine this, visit the
     * nodes until we find one that points beyond the end of the tree,
     * while keeping track of the highest key in use.  If the highest
     * key points beyond the invalid node, the tree is invalid.
     *
     * The second invariant assumes the nodes were created in a
     * top-down manner, which may not be true, so the rule that
     * implements the second invariant is commented out below.
     */
    max_key_used = 0;
    for ( i = 0; i < record_count; ++i ) {
        uint32_t L = (*map)->tree[i].left;
        uint32_t R = (*map)->tree[i].right;
        if (SKPREFIXMAP_IS_NODE(L)) {
            if (L >= record_count /* || L <= i */) {
                break;
            }
            if (L > max_key_used) {
                max_key_used = L;
            }
        }
        if (SKPREFIXMAP_IS_NODE(R)) {
            if (R >= record_count /* || R <= i */) {
                break;
            }
            if (R > max_key_used) {
                max_key_used = R;
            }
        }
    }
    if (i < record_count) {
        if (max_key_used >= i) {
            skAppPrintErr("Prefix map is malformed (contains invalid child).");
            free((*map)->tree);
            free(*map);
            *map = NULL;
            return SKPREFIXMAP_ERR_IO;
        }
        (*map)->tree_used = i;
    }

    /* Hrm.  We should also try to look for chains longer than 32
     * steps.  What's a good way to do that?
     */

    return SKPREFIXMAP_OK;

  ERROR:
    if (*map) {
        skPrefixMapDelete(*map);
        *map = NULL;
    }
    return rv;
}


/* Write 'map' to the file specified by 'pathname'. Wraps skPrefixMapWrite. */
skPrefixMapErr_t skPrefixMapSave(
    skPrefixMap_t  *map,
    const char     *pathname)
{
    skstream_t *stream = NULL;
    int rv;

    if (pathname == NULL || map == NULL) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    if ((rv = skStreamCreate(&stream, SK_IO_WRITE, SK_CONTENT_SILK))
        || (rv = skStreamBind(stream, pathname))
        || (rv = skStreamOpen(stream)))
    {
        skStreamPrintLastErr(stream, rv, &skAppPrintErr);
        rv = SKPREFIXMAP_ERR_IO;
        goto END;
    }

    rv = skPrefixMapWrite(map, stream);
    if (rv) {
        goto END;
    }

    rv = skStreamClose(stream);
    if (rv) {
        skStreamPrintLastErr(stream, rv, &skAppPrintErr);
        rv = SKPREFIXMAP_ERR_IO;
        goto END;
    }

  END:
    skStreamDestroy(&stream);
    return (skPrefixMapErr_t)rv;
}


skPrefixMapErr_t skPrefixMapSetContentType(
    skPrefixMap_t          *map,
    skPrefixMapContent_t    content_type)
{
    map->content_type = content_type;
    return SKPREFIXMAP_OK;

}

/* Set default value in 'map' to 'dict_val'.  'map' must be empty. */
skPrefixMapErr_t skPrefixMapSetDefaultVal(
    skPrefixMap_t  *map,
    uint32_t        dict_val)
{
    if (dict_val > SKPREFIXMAP_MAX_VALUE) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    /* ensure no entries have been added to the tree and default has
     * not been set.  This does not detect if default is set to
     * 0x7fffffff or if 0.0.0.0/1 or 128.0.0.0/1 has been set to
     * 0x7fffffff, but those are extremely rare possibilities. */
    if ((map->tree_used > 1)
        || (map->tree[0].right != SKPREFIXMAP_MAKE_LEAF(SKPREFIXMAP_MAX_VALUE))
        || (map->tree[0].left != SKPREFIXMAP_MAKE_LEAF(SKPREFIXMAP_MAX_VALUE)))
    {
        return SKPREFIXMAP_ERR_NOTEMPTY;
    }

    map->tree[0].left = SKPREFIXMAP_MAKE_LEAF(dict_val);
    map->tree[0].right = SKPREFIXMAP_MAKE_LEAF(dict_val);
    return SKPREFIXMAP_OK;
}


/* Set the prefix map's name */
skPrefixMapErr_t skPrefixMapSetMapName(skPrefixMap_t *map, const char *name)
{
    char *duplicate = NULL;

    if (name) {
        duplicate = strdup(name);
        if (duplicate == NULL) {
            return SKPREFIXMAP_ERR_MEMORY;
        }
    }

    if (map->mapname) {
        free(map->mapname);
    }
    map->mapname = duplicate;
    return SKPREFIXMAP_OK;
}


/* Return a textual representation of the specified error code. */
const char *skPrefixMapStrerror(int error_code)
{
    static char buf[256];

    switch ((skPrefixMapErr_t)error_code) {
      case SKPREFIXMAP_OK:
        return "Success";
      case SKPREFIXMAP_ERR_ARGS:
        return "Invalid arguments";
      case SKPREFIXMAP_ERR_MEMORY:
        return "Out of memory";
      case SKPREFIXMAP_ERR_IO:
        return "I/O error";
      case SKPREFIXMAP_ERR_DUPLICATE:
        return "Duplicate dictionary ID or word";
      case SKPREFIXMAP_ERR_NOTEMPTY:
        return "Cannot set default in non-empty map";
    }

    snprintf(buf, sizeof(buf), "Unrecognized prefix map error code %d",
             error_code);
    return buf;
}


/* Write the binary prefix map 'map' to the specified stream */
skPrefixMapErr_t skPrefixMapWrite(
    skPrefixMap_t  *map,
    skstream_t     *stream)
{
    sk_file_header_t *hdr;
    int rv;
    fileVersion_t vers;

    if (stream == NULL || map == NULL) {
        return SKPREFIXMAP_ERR_ARGS;
    }

    if (map->content_type == SKPREFIXMAP_CONT_PROTO_PORT) {
        vers = 3;
    } else if (map->content_type == SKPREFIXMAP_CONT_ADDR) {
        if (map->dict_buf == NULL) {
            vers = 1;
        } else {
            vers = 2;
        }
    } else {
        return SKPREFIXMAP_ERR_ARGS;
    }

    /* create the header */
    hdr = skStreamGetSilkHeader(stream);
    skHeaderSetFileFormat(hdr, FT_PREFIXMAP);
    skHeaderSetRecordVersion(hdr, vers);
    skHeaderSetCompressionMethod(hdr, SK_COMPMETHOD_NONE);
    skHeaderSetRecordLength(hdr, 1);

    /* add the prefixmap header if a mapname was given */
    if (map->mapname) {
        rv = skHeaderAddPrefixmap(hdr, map->mapname);
        if (rv) {
            skAppPrintErr("%s", skHeaderStrerror(rv));
            return SKPREFIXMAP_ERR_IO;
        }
    }

    /* write the header */
    rv = skStreamWriteSilkHeader(stream);
    if (rv) {
        skStreamPrintLastErr(stream, rv, &skAppPrintErr);
        return SKPREFIXMAP_ERR_IO;
    }

    /* write the number of records */
    rv = skStreamWrite(stream, &map->tree_used, sizeof(uint32_t));
    if (rv == -1) {
        goto ERROR;
    }

    /* write the records */
    rv = skStreamWrite(stream, map->tree,
                       map->tree_used * sizeof(skPrefixMapRecord_t));
    if (rv == -1) {
        goto ERROR;
    }

    if (map->dict_buf) {
        uint32_t i;

        /* write the number of characters in the dictionary */
        rv = skStreamWrite(stream, &map->dict_buf_used, sizeof(uint32_t));
        if (rv == -1) {
            goto ERROR;
        }

        /* write the dictionary entries */
        for (i = 0; i < map->dict_words_used; i++) {
            static const char zero = '\0';
            if (map->dict_words[i] == NULL) {
                rv = skStreamWrite(stream, &zero, sizeof(char));
            } else {
                rv = skStreamWrite(stream, map->dict_words[i],
                                   (sizeof(char)
                                    * (strlen(map->dict_words[i]) + 1)));
            }
            if (rv == -1) {
                goto ERROR;
            }
        }
    }

    /* Success */
    return SKPREFIXMAP_OK;

  ERROR:
    skStreamPrintLastErr(stream, rv, &skAppPrintErr);
    return SKPREFIXMAP_ERR_IO;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
