/*
** Copyright (C) 2008-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  SiLK plugin implementation
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skplugin-dynlib.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/dynlib.h>
#include "skplugin_priv.h"
#include <silk/skstringmap.h>

/* LOCAL DEFINES AND TYPEDEFS */

typedef int (*filterf_t)(rwRec *rwrec);

typedef int (*cutf_t)(
    unsigned int    id,
    char           *text_value,
    size_t          text_len,
    rwRec          *rwrec);

typedef int (*uniqf_t)(
    unsigned int    id,
    uint8_t        *bin_value,
    char           *text_value,
    size_t          text_len,
    rwRec          *rwrec);

typedef int (*sortf_t)(
    unsigned int    id,
    uint8_t        *bin_value,
    rwRec          *rwrec);

typedef int (*ptoflowf_t)(
    rwRec *rwrec,
    void *pktsrc);

typedef struct dynlib_field_st {
    dynlibInfoStruct_t *dlISP;
    sk_stringmap_id_t   field_id;
    sk_dllist_t        *names;
    skp_field_t        *field;
} dynlib_field_t;


/* LOCAL VARIABLE DEFINITIONS */

/* The legacy dynlib handles */
static sk_dllist_t *skp_dynlib_list = NULL;

/* The number of dynlibs that haven't had their cleanup code called */
static int skp_dynlib_unclean = 0;


/* LOCAL FUNCTION PROTOTYPES */

static skplugin_err_t skp_load_legacy(
    const char     *name,
    dynlibSymbolId  dynlib_id);
static skplugin_err_t skp_dynlib_init(void *cbdata);
static skplugin_err_t skp_dynlib_field_init(void *cbdata);
static skplugin_err_t skp_dynlib_cut_init(void *cbdata);
static skplugin_err_t skp_dynlib_uniq_init(void *cbdata);
static skplugin_err_t skp_dynlib_sort_init(void *cbdata);
static skplugin_err_t skp_dynlib_filter(
    const rwRec *rec,
    void *data,
    void **extra);
static skplugin_err_t skp_dynlib_ptoflow(
    rwRec *rec,
    void *data,
    void **extra);

static skplugin_err_t skp_dynlib_field_cleanup(void *cbdata);
static sk_dllist_t *skp_dynlib_field_from_stringmap(
    const sk_stringmap_entry_t *entries);
static skplugin_err_t skp_dynlib_register_cut(dynlibInfoStruct_t *dlISP);
static skplugin_err_t skp_dynlib_register_sort(dynlibInfoStruct_t *dlISP);
static skplugin_err_t skp_dynlib_register_uniq(dynlibInfoStruct_t *dlISP);

static skplugin_err_t skp_dynlib_field_cut(
    const rwRec *rec,
    char *dest,
    size_t width,
    void *data,
    void **extra);

static skplugin_err_t skp_dynlib_field_sort(
    const rwRec *rec,
    uint8_t *dest,
    void *data,
    void **extra);

static skplugin_err_t skp_dynlib_field_uniq_bin(
    const rwRec *rec,
    uint8_t *dest,
    void *data,
    void **extra);

static skplugin_err_t skp_dynlib_field_uniq_bin_to_text(
    const uint8_t *bin,
    char *dest,
    size_t width,
    void *data);

static void skp_dynlib_cleanup(void);


/* Loads the plugin represented by the filename 'name'.  Will handle
 * legacy dynlib-style plugins. */
skplugin_err_t skPluginLoadPluginOrDynlib(
    const char     *name,
    dynlibSymbolId  dynlib_id,
    int             complain_on_error)
{
    skplugin_err_t err;

    assert(skp_initialized);
    assert(!skp_in_plugin_init);

    /* Call skPluginLoadPlugin without error complaints*/
    err = skPluginLoadPlugin(name, 0);

    /* If we succeeded, return */
    if (err == SKPLUGIN_OK) {
        return err;
    }

    if (err == SKPLUGIN_ERR) {
        /* If we fail, attempt to load an old-style dynlib */
        err = skp_load_legacy(name, dynlib_id);

        /* If we succeeded, return */
        if (err == SKPLUGIN_OK) {
            return err;
        }
    }

    /* If that fails and we're being verbose, re-call
     * skPluginLoadPlugin with complaints */
    if (complain_on_error) {
        err = skPluginLoadPlugin(name, 1);
    }

    return err;
}


/* Clean up memory on cleanup */
static void skp_dynlib_cleanup(void)
{
    skp_dynlib_unclean -= 1;
    if (skp_dynlib_unclean == 0) {
        skDLListDestroy(skp_dynlib_list);
        skp_dynlib_list = NULL;
    }
}


/* Run usage callbacks */
void skp_dynlib_usage(FILE *fh)
{
    sk_dll_iter_t       iter;
    dynlibInfoStruct_t *dlISP;

    if (NULL == skp_dynlib_list) {
        return;
    }

    skDLLAssignIter(&iter, skp_dynlib_list);
    while (skDLLIterForward(&iter, (void **)&dlISP) == 0) {
        dynlibOptionsUsage(dlISP, fh);
    }
}


/* Attempt to load the plugin as a legacy dynlib library */
static skplugin_err_t skp_load_legacy(
    const char     *name,
    dynlibSymbolId  dynlib_id)
{
#define PTOFLOW_ARG "ptoflow"
    static const char    *ptoflow_extra[] = {PTOFLOW_ARG, NULL};
    dynlibInfoStruct_t   *dlISP;
    skplugin_err_t        err;
    int                   rv;
    skplugin_callbacks_t  regdata;

    if (skp_debug) {
        skAppPrintErr(SKPLUGIN_DEBUG_ENVAR ": attempting legacy load of '%s'",
                      name);
    }

    switch (dynlib_id) {
      case DYNLIB_EXCL_FILTER:
      case DYNLIB_SHAR_FILTER:
        if (!HANDLE_TYPE(SKPLUGIN_APP_FILTER)) {
            return SKPLUGIN_ERR;
        }
        break;
      case DYNLIB_CUT:
        if (!HANDLE_TYPE(SKPLUGIN_FN_REC_TO_TEXT)) {
            return SKPLUGIN_ERR;
        }
        break;
      case DYNLIB_SORT:
        if (!HANDLE_TYPE(SKPLUGIN_FN_REC_TO_BIN)) {
            return SKPLUGIN_ERR;
        }
        break;
      case DYNLIB_UNIQ:
        if (!HANDLE_TYPE(SKPLUGIN_FN_REC_TO_BIN |
                         SKPLUGIN_FN_ADD_REC_TO_BIN))
        {
            return SKPLUGIN_ERR;
        }
        break;
      case DYNLIB_PTOFLOW:
        if (!HANDLE_TYPE(SKPLUGIN_FN_TRANSFORM) ||
            (skp_arg_location(PTOFLOW_ARG, skp_app_support_extra_args) == -1))
        {
            return SKPLUGIN_ERR;
        }
        break;
      default:
        break;
    }

    dlISP = dynlibCreate(dynlib_id);
    CHECK_MEM(dlISP);

    rv = dynlibLoad(dlISP, name);
    if (rv != 0) {
        dynlibTeardown(dlISP);
        return SKPLUGIN_ERR;
    }

    skp_in_plugin_init = 1;
    skp_current_plugin_name = strdup(dynlibGetPath(dlISP));
    CHECK_MEM(skp_current_plugin_name);
    rv = skDLListPushTail(skp_plugin_names,
                          (void *)skp_current_plugin_name);
    CHECK_MEM(rv == 0);

    switch (dynlib_id) {
      case DYNLIB_EXCL_FILTER:
      case DYNLIB_SHAR_FILTER:
        memset(&regdata, 0, sizeof(regdata));
        regdata.init = skp_dynlib_init;
        regdata.filter = skp_dynlib_filter;
        err = skpinRegFilter(NULL, &regdata, dlISP);
        break;
      case DYNLIB_PTOFLOW:
        memset(&regdata, 0, sizeof(regdata));
        regdata.init = skp_dynlib_init;
        regdata.transform = skp_dynlib_ptoflow;
        regdata.extra = ptoflow_extra;
        err = skpinRegTransformer(NULL, &regdata, dlISP);
        break;
      case DYNLIB_CUT:
        err = skp_dynlib_register_cut(dlISP);
        break;
      case DYNLIB_SORT:
        err = skp_dynlib_register_sort(dlISP);
        break;
      case DYNLIB_UNIQ:
        err = skp_dynlib_register_uniq(dlISP);
        break;
      default:
        skAbortBadCase(dynlib_id);
    }
    if (!dynlibSupportsThreads(dlISP)) {
        skpinSetThreadNonSafe();
    }

    if (skp_dynlib_list == NULL) {
        skp_dynlib_list = skDLListCreate((sk_dll_free_fn_t)dynlibTeardown);
        CHECK_MEM(skp_dynlib_list);
    }

    rv = skDLListPushTail(skp_dynlib_list, dlISP);
    CHECK_MEM(rv == 0);
    skp_dynlib_unclean += 1;
    skpinRegCleanup(skp_dynlib_cleanup);

    skp_in_plugin_init = 0;

    return SKPLUGIN_OK;
}

/* Call a dynlib init function */
static skplugin_err_t skp_dynlib_init(
    void *cbdata)
{
    dynlibInfoStruct_t *dlISP = (dynlibInfoStruct_t *)cbdata;
    dynlibSymbolId id = dynlibGetAppType(dlISP);

    if (id == DYNLIB_CUT ||
        id == DYNLIB_SORT ||
        id == DYNLIB_UNIQ ||
        id == DYNLIB_PTOFLOW ||
        dynlibCheckActive(dlISP))
    {
        if (dynlibInitialize(dlISP)) {
            return SKPLUGIN_ERR;
        }
    } else {
        /* Remove this dynlib function */
        return SKPLUGIN_FILTER_IGNORE;
    }

    return SKPLUGIN_OK;
}

/* Call a dynlib init function for a field */
static skplugin_err_t skp_dynlib_field_init(
    void *cbdata)
{
    dynlib_field_t *field = (dynlib_field_t *)cbdata;

    return skp_dynlib_init(field->dlISP);
}


/* Call a dynlib filter function */
static skplugin_err_t skp_dynlib_filter(
    const rwRec  *rec,
    void         *data,
    void        **UNUSED(extra))
{
    dynlibInfoStruct_t *dlISP = (dynlibInfoStruct_t *)data;
    filterf_t           filt  = (filterf_t)dynlibGetRWProcessor(dlISP);

    if (filt((rwRec *)rec)) {
        return SKPLUGIN_FILTER_FAIL;
    }
    return SKPLUGIN_FILTER_PASS;
}

/* Call a dynlib ptoflow function */
static skplugin_err_t skp_dynlib_ptoflow(
    rwRec  *rec,
    void   *data,
    void  **extra)
{
    dynlibInfoStruct_t *dlISP   = (dynlibInfoStruct_t *)data;
    ptoflowf_t          ptof = (ptoflowf_t)dynlibGetRWProcessor(dlISP);

    assert(extra[0]);

    switch (ptof(rec, extra[0])) {
      case 0:
        return SKPLUGIN_FILTER_PASS;
      case 1:
        return SKPLUGIN_FILTER_PASS_NOW;
      case 2:
        return SKPLUGIN_FILTER_FAIL;
      case 3:
        return SKPLUGIN_FILTER_IGNORE;
    }

    return SKPLUGIN_ERR;
}

/* Free the extra data associated with dynlib fields */
static skplugin_err_t skp_dynlib_field_cleanup(
    void *cbdata)
{
    dynlib_field_t *field = (dynlib_field_t *)cbdata;
    if (field->names != NULL) {
        skDLListDestroy(field->names);
    }
    free(field);
    return SKPLUGIN_OK;
}


/* Call a dynlib sort function */
static skplugin_err_t skp_dynlib_field_cut(
    const rwRec  *rec,
    char         *dest,
    size_t        width,
    void         *data,
    void        **UNUSED(extra))
{
    dynlib_field_t *field_data;
    cutf_t          cutf;
    int             rv;

    assert(rec);
    assert(dest);
    assert(data);

    field_data = (dynlib_field_t *)data;
    cutf       = (cutf_t)dynlibGetRWProcessor(field_data->dlISP);

    rv = cutf(field_data->field_id, dest, width, (rwRec *)rec);
    if (rv < 0) {
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}



/* Register a dynlib cut function */
static skplugin_err_t skp_dynlib_register_cut(
    dynlibInfoStruct_t *dlISP)
{
    skplugin_err_t        err;
    sk_stringmap_entry_t *entries = NULL;
    cutf_t                cutf    = (cutf_t)dynlibGetRWProcessor(dlISP);
    int                   title_width;
    skplugin_callbacks_t  regdata;

    if (dynlibConfigure(dlISP, (void *)&entries) != 0) {
        return SKPLUGIN_ERR;
    }

    memset(&regdata, 0, sizeof(regdata));
    regdata.init = skp_dynlib_cut_init;
    regdata.rec_to_text = skp_dynlib_field_cut;

    if (entries != NULL) {
        /* the dynlibConfigure() routine gave us the fields */
        dynlib_field_t *field;
        sk_dllist_t    *field_list;
        sk_dll_iter_t   field_iter;


        field_list = skp_dynlib_field_from_stringmap(entries);
        assert(field_list);

        skDLLAssignIter(&field_iter, field_list);
        while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
            char          *title;
            char          *name;
            sk_dll_iter_t  name_iter;

            field->dlISP = dlISP;

            title_width = cutf(field->field_id, NULL, 0, NULL);
            title = (char *)malloc(title_width);
            CHECK_MEM(title);
            cutf(field->field_id, title, title_width, NULL);

            err = skpinRegField(&field->field, title, NULL, &regdata, field);

            free(title);

            if (err != SKPLUGIN_OK) {
                skp_dynlib_field_cleanup(field);
                while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
                    skp_dynlib_field_cleanup(field);
                }
                skDLListDestroy(field_list);
                return err;
            }

            field->field->common.cbfree = skp_dynlib_field_cleanup;

            skDLLAssignIter(&name_iter, field->names);
            while (skDLLIterForward(&name_iter, (void **)&name) == 0) {
                skpinAddFieldAlias(field->field, name);
            }
            skDLListDestroy(field->names);
            field->names = NULL;
        }

        skDLListDestroy(field_list);
    } else {
        /* the dynlibConfigure() routine did not give us the fields,
         * so fall-back to the old way of getting them */
        int i;
        int count = cutf(0, NULL, 0, NULL);

        for (i = 1; i <= count; i++) {
            dynlib_field_t *field;
            char           *title;

            field = (dynlib_field_t *)calloc(1, sizeof(*field));
            CHECK_MEM(field);

            field->field_id = i;
            field->dlISP = dlISP;

            title_width = cutf(i, NULL, 0, NULL);
            title = (char *)malloc(title_width);
            CHECK_MEM(title);
            cutf(i, title, title_width, NULL);

            err = skpinRegField(&field->field, title, NULL, &regdata, field);

            free(title);

            if (err != SKPLUGIN_OK) {
                skp_dynlib_field_cleanup(field);
                return err;
            }

            field->field->common.cbfree = skp_dynlib_field_cleanup;
        }
    }

    return SKPLUGIN_OK;
}

/* Init code for dynlib cut  */
static skplugin_err_t skp_dynlib_cut_init(
    void *cbdata)
{
    static rwRec        dummy;
    dynlib_field_t     *field = (dynlib_field_t *)cbdata;
    cutf_t              cutf;
    int                 field_width;
    skplugin_err_t      err;

    err =  skp_dynlib_field_init(cbdata);
    if (err == SKPLUGIN_OK) {
        cutf        = (cutf_t)dynlibGetRWProcessor(field->dlISP);
        field_width = cutf(field->field_id, NULL, 0, &dummy);
        field->field->field_width_text = field_width - 1;
    }

    return err;
}


/* Init code for dynlib sort  */
static skplugin_err_t skp_dynlib_sort_init(
    void *cbdata)
{
    dynlib_field_t     *field = (dynlib_field_t *)cbdata;
    sortf_t             sortf;
    int                 field_width;
    skplugin_err_t      err;

    err =  skp_dynlib_field_init(cbdata);
    if (err == SKPLUGIN_OK) {
        sortf        = (sortf_t)dynlibGetRWProcessor(field->dlISP);
        field_width = sortf(field->field_id, NULL, NULL);
        field->field->field_width_bin = field_width;
    }

    return err;
}


/* Init code for dynlib uniq  */
static skplugin_err_t skp_dynlib_uniq_init(
    void *cbdata)
{
    static rwRec    dummy_rec;
    static uint8_t  dummy_bin;
    dynlib_field_t *field = (dynlib_field_t *)cbdata;
    uniqf_t         uniqf;
    int             text_width;
    int             bin_width;
    skplugin_err_t  err;

    err =  skp_dynlib_field_init(cbdata);
    if (err == SKPLUGIN_OK) {
        uniqf      = (uniqf_t)dynlibGetRWProcessor(field->dlISP);
        bin_width  = uniqf(field->field_id, &dummy_bin, NULL, 0, &dummy_rec);
        text_width = uniqf(field->field_id, NULL, NULL, 0, &dummy_rec);
        field->field->field_width_bin = bin_width;
        field->field->field_width_text = text_width - 1;
    }

    return err;
}


/* Call a dynlib sort function */
static skplugin_err_t skp_dynlib_field_sort(
    const rwRec *rec,
    uint8_t     *dest,
    void        *data,
    void       **UNUSED(extra))
{
    dynlib_field_t *field_data;
    sortf_t         sortf;
    int             rv;

    assert(rec);
    assert(dest);
    assert(data);

    field_data = (dynlib_field_t *)data;
    sortf      = (sortf_t)dynlibGetRWProcessor(field_data->dlISP);

    rv = sortf(field_data->field_id, dest, (rwRec *)rec);
    if (rv < 0) {
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}

/* Register a dynlib sort function */
static skplugin_err_t skp_dynlib_register_sort(
    dynlibInfoStruct_t *dlISP)
{
    skplugin_err_t        err;
    sk_stringmap_entry_t *entries = NULL;

    if (dynlibConfigure(dlISP, (void *)&entries) != 0) {
        return SKPLUGIN_ERR;
    }

    if (entries != NULL) {
        /* the dynlibConfigure() routine gave us the fields */
        dynlib_field_t      *field;
        sk_dllist_t         *field_list;
        sk_dll_iter_t        field_iter;
        skplugin_callbacks_t regdata;

        memset(&regdata, 0, sizeof(regdata));
        regdata.init = skp_dynlib_sort_init;
        regdata.rec_to_bin = skp_dynlib_field_sort;

        field_list = skp_dynlib_field_from_stringmap(entries);
        assert(field_list);

        skDLLAssignIter(&field_iter, field_list);
        while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
            char                               *name;
            sk_dll_iter_t                       name_iter;
            int                                 rv;

            field->dlISP = dlISP;

            skDLLAssignIter(&name_iter, field->names);
            rv = skDLLIterForward(&name_iter, (void **)&name);
            if (rv != 0) {
                err = SKPLUGIN_ERR;
            } else {
                err = skpinRegField(&field->field, name, NULL,
                                    &regdata, field);
            }

            if (err != SKPLUGIN_OK) {
                skp_dynlib_field_cleanup(field);
                while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
                    skp_dynlib_field_cleanup(field);
                }
                skDLListDestroy(field_list);
                return err;
            }

            field->field->common.cbfree = skp_dynlib_field_cleanup;

            while (skDLLIterForward(&name_iter, (void **)&name) == 0) {
                skpinAddFieldAlias(field->field, name);
            }
            skDLListDestroy(field->names);
            field->names = NULL;
        }

        skDLListDestroy(field_list);
    } else {
        /* Sort requires an entry map */
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}

/* Call a dynlib uniq bin function */
static skplugin_err_t skp_dynlib_field_uniq_bin(
    const rwRec *rec,
    uint8_t     *dest,
    void        *data,
    void       **UNUSED(extra))
{
    dynlib_field_t *field_data;
    uniqf_t         uniqf;
    int             rv;

    assert(rec);
    assert(dest);
    assert(data);

    field_data = (dynlib_field_t *)data;
    uniqf      = (uniqf_t)dynlibGetRWProcessor(field_data->dlISP);

    rv = uniqf(field_data->field_id, dest, NULL, 0, (rwRec *)rec);
    if (rv < 0) {
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}

/* Call a dynlib uniq bin_to_text function */
static skplugin_err_t skp_dynlib_field_uniq_bin_to_text(
    const uint8_t *bin,
    char          *dest,
    size_t         width,
    void          *data)
{
    dynlib_field_t *field_data;
    uniqf_t         uniqf;
    int             rv;

    assert(bin);
    assert(dest);
    assert(data);

    field_data = (dynlib_field_t *)data;
    uniqf      = (uniqf_t)dynlibGetRWProcessor(field_data->dlISP);

    rv = uniqf(field_data->field_id, (uint8_t *)bin, dest, width, NULL);
    if (rv < 0) {
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}

/* Register a dynlib uniq function */
static skplugin_err_t skp_dynlib_register_uniq(
    dynlibInfoStruct_t *dlISP)
{
    skplugin_err_t        err;
    sk_stringmap_entry_t *entries = NULL;
    uniqf_t               uniqf  = (uniqf_t)dynlibGetRWProcessor(dlISP);

    if (dynlibConfigure(dlISP, (void *)&entries) != 0) {
        return SKPLUGIN_ERR;
    }

    if (entries != NULL) {
        /* the dynlibConfigure() routine gave us the fields */
        dynlib_field_t      *field;
        sk_dllist_t         *field_list;
        sk_dll_iter_t        field_iter;
        skplugin_callbacks_t regdata;

        memset(&regdata, 0, sizeof(regdata));
        regdata.init = skp_dynlib_uniq_init;
        regdata.rec_to_bin = skp_dynlib_field_uniq_bin;
        regdata.bin_to_text = skp_dynlib_field_uniq_bin_to_text;

        field_list = skp_dynlib_field_from_stringmap(entries);
        assert(field_list);

        skDLLAssignIter(&field_iter, field_list);
        while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
            int            title_width;
            char          *title;
            char          *name;
            sk_dll_iter_t  name_iter;

            field->dlISP = dlISP;

            title_width = uniqf(field->field_id, NULL, NULL, 0, NULL);
            title = (char *)malloc(title_width);
            CHECK_MEM(title);
            uniqf(field->field_id, NULL, title, title_width, NULL);

            err = skpinRegField(&field->field, title, NULL, &regdata, field);

            free(title);

            if (err != SKPLUGIN_OK) {
                skp_dynlib_field_cleanup(field);
                while (skDLLIterForward(&field_iter, (void **)&field) == 0) {
                    skp_dynlib_field_cleanup(field);
                }
                skDLListDestroy(field_list);
                return err;
            }

            field->field->common.cbfree = skp_dynlib_field_cleanup;

            skDLLAssignIter(&name_iter, field->names);
            while (skDLLIterForward(&name_iter, (void **)&name) == 0) {
                skpinAddFieldAlias(field->field, name);
            }
            skDLListDestroy(field->names);
            field->names = NULL;
        }

        skDLListDestroy(field_list);
    } else {
        /* Sort requires an entry map */
        return SKPLUGIN_ERR;
    }

    return SKPLUGIN_OK;
}

/* Parse a stringmap entry */
static sk_dllist_t *skp_dynlib_field_from_stringmap(
    const sk_stringmap_entry_t *entries)
{
    int             rv;
    int             ecount;
    sk_dllist_t    *retval;
    sk_dll_iter_t   iter;
    char           *name_copy;
    dynlib_field_t *field;
    dynlib_field_t *target;

    retval = skDLListCreate(NULL);
    CHECK_MEM(retval);

    /* Loop through the entries */
    for (ecount = 0; entries[ecount].name; ecount++) {
        if (entries[ecount].id == 0) {
            /* Reject for legacy compatibility reasons */
            goto END;
        }

        field = NULL;
        skDLLAssignIter(&iter, retval);
        /* Find the field id in the current retval list */
        while (skDLLIterForward(&iter, (void **)&target) == 0) {
            if (target->field_id == entries[ecount].id) {
                field = target;
                break;
            }
        }
        /* If not found, create a new entry */
        if (field == NULL) {
            field = (dynlib_field_t *)calloc(1, sizeof(*field));
            CHECK_MEM(field);
            field->field_id = entries[ecount].id;
            field->names = skDLListCreate(free);
            CHECK_MEM(field->names);
            rv = skDLListPushTail(retval, field);
            CHECK_MEM(rv == 0);
        }

        /* Add the name to this entry */
        name_copy = strdup(entries[ecount].name);
        CHECK_MEM(name_copy);
        rv = skDLListPushTail(field->names, name_copy);
        CHECK_MEM(rv == 0);
    }

  END:

    return retval;
}
