/*
** Copyright (C) 2010-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Test program for the sktimer module
**  Michael Welsh Duggan - 2010.07.21
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: sktimer-test.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/sktimer.h>

/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

/* LOCAL VARIABLE DEFINITIONS */

/* OPTIONS SETUP */
typedef enum {
    OPT_INTERVAL,
    OPT_START,
    OPT_COUNT,
    OPT_PROCESS_TIME
} appOptionsEnum;

static struct option appOptions[] = {
    {"interval",        REQUIRED_ARG, 0, OPT_INTERVAL},
    {"start",           REQUIRED_ARG, 0, OPT_START},
    {"count",           REQUIRED_ARG, 0, OPT_COUNT},
    {"process-time",    REQUIRED_ARG, 0, OPT_PROCESS_TIME},
    {0,0,0,0}           /* sentinel entry */
};

static const char *appHelp[] = {
    "Interval between timer firings (in seconds)",
    "Date/time when timer firing should commence (Def: now)",
    "Number of times to call the timer callback (Def: unlimited)",
    ("Comma-separated list of seconds of processing time after\n"
     "\ttimer triggers (Def: 0)"),
    (char *)NULL
};

static int quit;

static sktime_t start_time;
static uint32_t interval;
static uint32_t timer_count;
static uint32_t *proc_times;
static uint32_t num_proc_times;

static uint32_t callback_count;

static pthread_mutex_t mutex;
static pthread_cond_t  cond;

/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static void timer_signal_handler(int signal_num);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                       \
    ("--interval <SECS> [SWITCHES]\n"                                   \
     "\tOutputs the time every SECS seconds\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;
    quit = 1;

    pthread_mutex_destroy(&mutex);
    pthread_cond_destroy(&cond);

    if (proc_times) {
        free(proc_times);
    }

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    start_time = (sktime_t)(-1);
    interval = 0;
    timer_count = 0;
    callback_count = 0;
    proc_times = NULL;
    num_proc_times = 0;
    quit = 0;

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)) {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* check for extraneous arguments */
    if (arg_index != argc) {
        skAppPrintErr("Too many arguments or unrecognized switch '%s'",
                      argv[arg_index]);
        skAppUsage();           /* never returns */
    }

    if (interval == 0) {
        skAppPrintErr("The --%s switch is required",
                      appOptions[OPT_INTERVAL].name);
        skAppUsage();
    }

    rv = skAppSetSignalHandler(timer_signal_handler);
    if (rv != 0) {
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    pthread_mutex_init(&mutex, NULL);
    pthread_cond_init(&cond, NULL);

    return;  /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;
    uint32_t value;

    switch ((appOptionsEnum)opt_index) {
      case OPT_INTERVAL:
        rv = skStringParseUint32(&value, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        interval = value;
        break;


      case OPT_COUNT:
        rv = skStringParseUint32(&value, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        timer_count = value;
        break;

      case OPT_START:
        rv = skStringParseDatetime(&start_time, opt_arg, NULL);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_PROCESS_TIME:
        rv = skStringParseNumberList(&proc_times, &num_proc_times, opt_arg,
                                     0, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;
    }

    return 0;  /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}

static void timer_signal_handler(int signal_num)
{
    skAppPrintErr("Stopping due to SIG%s", skSignalToName(signal_num));
    quit = 1;
    pthread_cond_broadcast(&cond);
}

static skTimerRepeat_t timer_callback(void *UNUSED(v))
{
    skTimerRepeat_t retval = SK_TIMER_REPEAT;

    skAppPrintErr("Timer called: %s", sktimestamp(sktimeNow(), 0));

    pthread_mutex_lock(&mutex);

    if (proc_times) {
        struct timespec w;
        struct timeval now;
        int rv;

        gettimeofday(&now, NULL);
        w.tv_nsec = now.tv_usec * 1000;
        w.tv_sec = now.tv_sec + proc_times[callback_count % num_proc_times];
        do {
            rv = pthread_cond_timedwait(&cond, &mutex, &w);
        } while (rv == EINTR);

        if (quit) {
            retval = SK_TIMER_END;
        }
    }

    callback_count++;
    if (timer_count && timer_count == callback_count) {
        quit = 1;
        retval = SK_TIMER_END;
        pthread_cond_signal(&cond);
    }
    pthread_mutex_unlock(&mutex);

    return retval;
}


int main(int argc, char **argv)
{
    skTimer_t timer;
    int rv;

    appSetup(argc, argv);                       /* never returns on error */

    if (start_time == (sktime_t)(-1)) {
        rv = skTimerCreate(&timer, interval, timer_callback, NULL);
    } else {
        rv = skTimerCreateAtTime(&timer, interval, start_time,
                                 timer_callback, NULL);
    }
    if (rv != 0) {
        skAppPrintErr("Timer creation failed");
        appTeardown();
        return EXIT_FAILURE;
    }

    pthread_mutex_lock(&mutex);
    while (!quit && (timer_count == 0 || timer_count != callback_count)) {
        rv = pthread_cond_wait(&cond, &mutex);
    }
    pthread_mutex_unlock(&mutex);

    appTeardown();

    return EXIT_SUCCESS;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
